/*
 * Copyright (C) 2001, John Leuner.
 *
 * This file is part of the kissme project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2,
 * or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#ifdef NATIVE_THREADS

#include <jni.h>
#include <interp.h>
#include <pthread.h>

#ifdef KISSME_LINUX_USER
#include <stdio.h>
#include <stdlib.h>
#endif

#include <assert.h>
#include <garbage.h>

#include <sys/time.h>
#include <unistd.h>
#include <errno.h>

#include "threads_native.h"
#include "threadinfo.h"
#include "interp.h"
#include "interp_methods.h"
#include "classfile_methods.h"
#include "methodstacks.h"


extern tAllocatorHeap* system_heap;
extern tClassLoaderTuple* pstClassType; // this can be globally used to get the type for class

/* Stuff to manipulate the thread objects */

tThreadNode* pstThreadList = NULL; //List of threads
static pthread_mutex_t threadListMutex;// = PTHREAD_MUTEX_INITIALIZER;

static pthread_mutex_t threadLockListMutex = PTHREAD_MUTEX_INITIALIZER;

volatile uint32 iLiveThreads = 0; //Number of live threads left s
static pthread_cond_t cond = PTHREAD_COND_INITIALIZER;

pthread_mutex_t* THREAD_getThreadListMutex()
{
  return &threadListMutex;
}

int pthread_mutexattr_setkind_np (pthread_mutexattr_t* kind, int tkind);

//makes threadListMutex recursive
void THREAD_Init()
{
  pthread_mutexattr_t kind;
  pthread_mutexattr_init (&kind);
  pthread_mutexattr_setkind_np (&kind,
				PTHREAD_MUTEX_RECURSIVE_NP);
  pthread_mutex_init (&threadListMutex, &kind);
}

tThreadNode* THREAD_getThreadList()
{
  return pstThreadList;
}

void THREAD_setThreadList(tThreadNode* nlist)
{
  pstThreadList = nlist;
  return;
}

tThreadNode* THREAD_lockAndGetThreadList()
{
  pthread_mutex_lock( & threadListMutex );
  return pstThreadList;
}

int THREAD_unlockThreadList()
{
  pthread_mutex_unlock( & threadListMutex );
  return 0;
}

int THREAD_GetNumberThreads()
{
  return iLiveThreads;
}

//XXX I must clean this up -- jewel
// This work is already done in teaseme

//tJNIData* pstCurrJNIData = NULL;
/*
 * @doc TYPE
 * @struct
 * This structure contains offsets of all of the relevant Java Thread object's
 * fields. It is used so that the offsets only have to be lookup up once and
 * can be used often.
 *
 */

struct tthreadoffsets
{
	int iStateOffset;    	   /* @field What the thread is currently up to. */
	int iPriorityOffset;     /* @field Thread priority. */
	int iFlagsOffset;			   /* @field Interupted/daemon. If the interupted flag
                                     is set, this means that the thread has
                                     been interupted by another thread since
                                     its last time-slice.	As soon as it is
                                     running again, it will process	the
                                     interupt, rather than continue from its
                                     execution point. */
	int iGroupOffset;				 /* @field The thread group to which this thread
                                     belongs */
	int iTargetOffset;			 /* @field The Runnable target of this thread, if
                                     there is one; null otherwise */
	int iNameOffset;				 /* @field The name of the thread. */
	int iSPOffset;					 /* @field The current stack frame pointer. */
	int iPCOffset;					 /* @field The current program counter. */
  int iTOSOffset;		    	 /* @field The current top-of-operand-stack. */
	int iTimeUpOffset;			 /* @field Only valid if/when the thread is sleeping.
													           This is the time at which the sleep is
													           over, or else a count-down to zero,
													           depending on the timer support. */
	int iThrownObjectOffset; /* @field Only valid if the thread has been
                                     interupted or stopped. This is the object
                                     thrown explicitly in these cases, or null
                                     otherwise */
	int iNextOffset;				 /* @field Pointer to the next thread of the same
                                     priority. Threads are linked circularly
                                     for round-robin scheduling. */
	int iSiblingOffset;			 /* @field Pointer to the next thread/thread group in
                                     the ThreadGroup. */
  int iWaitNext;           /* @field Pointer to next thread in wait set */
  int iWantToLock;         /* @field Pointer to object on which this thread
                                     needs a lock */
  int iOldLockCount;       /* @field Number of locks this thread had on an
                                     object before it called wait() */
  int iJNIData;            /* @field Pointer to this thread's JNIData struct */
};

/*
 * @doc TYPE
 * @type tThreadOffsets | typdef of tthreadoffsets struct
 *
 */

typedef struct tthreadoffsets tThreadOffsets;

/*
 * Structure containing all of the offsets we need.
 */

static tThreadOffsets stThreadOffsets;

#define SETGROUP(obj, i)        (((DEREF(obj))->pi32Vars[stThreadOffsets.iGroupOffset]) = (int32) (i))
#define SETTARGET(obj, i)       (((DEREF(obj))->pi32Vars[stThreadOffsets.iTargetOffset]) = (int32) (i))
#define SETNAME(obj, i)         (((DEREF(obj))->pi32Vars[stThreadOffsets.iNameOffset]) = (int32) (i))

#define GETGROUP(obj)        ((tOBREF)     ((DEREF(obj))->pi32Vars[stThreadOffsets.iGroupOffset]))
#define GETTARGET(obj)       ((tOBREF)     ((DEREF(obj))->pi32Vars[stThreadOffsets.iTargetOffset]))
#define GETNAME(obj)         ((tOBREF)     ((DEREF(obj))->pi32Vars[stThreadOffsets.iNameOffset]))





/* Stuff to keep the garbage collector happy */

tJNIData* THREAD_GetCurrentJNIData( void )
{
 return JNI_getJNIData( pthread_self() );
}




/* Stuff to keep java.lang.Object happy */

/* This is a linked list of locks that have been obtained by various threads
   But we need a mutex to protect this structure!

 */

struct native_thread_lock* THREAD_locks = NULL;



/* Routine to dump the locks 
 *
 * Assumes that threadLockListMutex is locked
 *
 */

void dumpLocks()
{
    tThreadLock* iter = THREAD_locks;
    fprintf(stderr, "Dumping locks from thread %x --------------> \n", (int) pthread_self());
    while(iter != NULL)
	{
	    //	    fprintf(stderr, "Object %p was locked by thread_id %x, mutex is %x\n", iter->object, (int) (iter->thread_id), (int) (iter->mutex));
	    iter = iter->next;
	}
    fprintf(stderr, "<--------------- End Dumping locks \n");
}



/* Stuff called from INTERP */


tStackFrame* THREAD_GetNewStackFrame
(
  tStackFrame* pstOldFrame  /* @parm Environment frame of old thread */
)
{
    assert(  6 == 7);
  return NULL;
}


/*
 * @doc FUNC
 * @func
 * Destroy a thread. This is invoked when a thread has run to completion, or if
 * the destroy() method is invoked in Java.
 *
 */

void THREAD_Destroy
(
  tOBREF objThread  /* @parm Thread to destroy */
)
{
  tThreadNode* iter, *prev;
   
  if(objThread == NULL)
	return;
  pthread_mutex_lock( & threadListMutex );

 prev = iter =  pstThreadList;

  assert(pstThreadList);
  while(iter != NULL)
      {
	  if(iter->object == objThread)
	      {
		prev->next = iter->next;
		    pthread_mutex_unlock( & threadListMutex );    
		    //fprintf(stderr, "nativeCurrentThread Returning the object %p\n", iter->object);
		    return ;
	      }
	  prev = iter;
	  iter = iter->next;
      }
  pthread_mutex_unlock( & threadListMutex );    
  fprintf(stderr, "------ FAILED to DESTROY thread object %p %p\n", objThread, *objThread);
}

jobject java_lang_Thread_nativeCurrentThread(JNIEnv* env, jclass thisclass)
{
  tThreadNode* iter;
  pthread_mutex_lock( & threadListMutex );

  iter = pstThreadList;

  if((env == NULL) && (thisclass == NULL)) {
    pthread_mutex_unlock( & threadListMutex );
    return NULL;
  }
  assert(pstThreadList);
  while(iter != NULL)
      {
	  if(iter->thread_id == (pthread_t) pthread_self())
	      {
		    pthread_mutex_unlock( & threadListMutex );    
//		    eprintf("nativeCurrentThread Returning the object %p\n", iter->object);
		    return iter->object;
	      }
	  iter = iter->next;
      }
  pthread_mutex_unlock( & threadListMutex );    
// eprintf( "nativeCurrentThread Returning NULL\n");
  return NULL;
}


void THREAD_InitMainThread
(
  tOBREF       hThread,  /* @parm Thread to initialise */
  int          iPriority,  /* @parm Priority to set thread to */
  tStackFrame* pstFrame,    /* @parm Environment frame to give to thread */
  tJNIData* pstJNIData, /* A JNIData structure that has been created by __createJNIata */
  void* stackBottom /* An optional argument indicating where the start of the C stack is */
)
{
  assert( 1 == 3);
  return;
#ifdef EVICTIONS
  /* flag thread so that garbage collector doesn't do anything with it */
  SETFLAG(DEREF(hThread), GARBAGE_LEAVEALONE);
  /* add thread to thread stack list */

#ifdef DEBUG
  assert(pstThreadStackList == NULL);
#endif
#endif

//  THREAD_addThreadToList( pstFrame, pthread_self(), hThread , stackBottom);

  pthread_mutex_lock( & threadListMutex );
  iLiveThreads = 1;
  pthread_mutex_unlock( & threadListMutex );
  return ;
}






#ifdef BOGUS
int THREAD_free_lock(  tOBREF pstObject         /* @parm Object for which we can release locks */ )
{
  struct native_thread_lock* iter;
  struct native_thread_lock* prev;

  int iFound = 0;
  assert(pstObject);
  pthread_mutex_lock(&threadLockListMutex);
  iter = prev = THREAD_locks;

  while( iter != NULL)
    {
      if( iter->object == pstObject)
	{
	    int success;

            prev->next = iter->next;
	    if(iter == THREAD_locks)
		THREAD_locks = iter->next;
            sys_free(iter);
            break;
	}
      prev = iter;
      iter = iter->next;
    }
  
/*  if(iFound)
    {
      //We got the lock, that's cool
    }
  else
    {
    }*/
    pthread_mutex_unlock(&threadLockListMutex);
}
#endif

/* The java.lang.Thread native methods */ 
void java_lang_Thread__initC
(
  JNIEnv* env,
  jclass clazz
)
{
  tField* fieldoffset;
  tClassLoaderTuple *pstTempClass = CLASSFILE_FindOrLoad(env, "java/lang/Thread", pstClassType); //XXX THREAD?
  struct
  {
    char *pszField;
    int *pui32Offset;
  }
  astFields[] =
  {
    {"_group",				&(stThreadOffsets.iGroupOffset)},
    {"toRun",				&(stThreadOffsets.iTargetOffset)},
    {"_name",				  &(stThreadOffsets.iNameOffset)},
  };
  int iField;
  
#ifdef DEBUG
  assert(pstTempClass);
#endif
  
  /* find the offsets of the instance variables */
  for(iField = 0; iField < sizeof(astFields) / sizeof(astFields[0]); iField++)
    {
      fieldoffset = CLASSFILE_InstFieldOffset(env, pstTempClass, astFields[iField].pszField);
#ifdef DEBUG
      assert(fieldoffset);
#endif
      *(astFields[iField].pui32Offset) = fieldoffset->u16Offset;
    }
  
  /* initialise random number generator... used by notify */
  srand((unsigned) time(NULL));
}


void java_lang_Thread_yield
(
 JNIEnv* env,
 jobject obj
 )
{
}

/* Joins the current thread to the specified thread */
void java_lang_Thread_join
(
 JNIEnv* env,
 jobject obj,
 jlong millis
 )
{
  (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/InternalError", "Thread.join not supported in kissme yet"));
  return;
}


void java_lang_Thread_resume
(
 JNIEnv* env,
 jobject obj
 )
{
  assert( 7 == 8);
}



void java_lang_Thread_destroy
(
 JNIEnv* env,
 jobject obj
 )
{
  //We should remove it from the thread list
      
      // THREAD_Destroy(obj);
}


void java_lang_Thread_sleep
(
 JNIEnv* env,
 jclass clazz,
 jlong millis,
 jint nanos
 )
{
  /* ### We ignore everything except the low milli's */
  //int32 i32HiMillis = millis.hi;
  //int32 i32LoMillis = millis.lo;
  
  
  //usleep( i32LoMillis * 1000 + i32HiMillis * 1000 * 1000);
  usleep( millis * 1000 );
  //	fprintf(stderr, "Sleep over !!\n");
  
  

  
}


jboolean java_lang_Thread_isAlive
(
 JNIEnv* env,
 jobject obj
 )
{
  return JNI_TRUE;
}





void java_lang_Thread_suspend
(
 JNIEnv* env,
 jobject obj
)
{
  assert (4 == 5);
}


/* Just a little helper method 
 *
 * It is the C method which runs a new Java thread
 *
 *
 */

void Interpret_MultiThread(void* arg)
{
    int i ; 
    pthread_t newThread;
    tThreadArgs* targs;
    tThreadNode* nnode;
    JNIEnv* env;

    THREADINFO_IncrementThreadCount();
    newThread = pthread_self();
    
    targs = (tThreadArgs*) arg;

    env = targs->env;
    java_lang_Thread__createJNIData( NULL, targs->threadObject); 
    assert( targs->threadObject);
    
    /* We store the bottom (although high mem) of the stack for this thread
       Normally each stack is 2 megabytes, which we search for the GC
       */
    
    targs->returnStackBottom = (void*) &i;
    
    /* Now put this new thread in the global list of threads */

    nnode = THREADINFO_CreateNewNodeForThread(sys_current_thread(), &i);
    nnode->stackTop = nnode->stackBottom;
    THREADINFO_addThreadNodeToList(nnode);
    THREADINFO_SetNodeInfoForCurrentThread(targs->threadObject, &i);


#ifdef SHOWTHREADS
    eprintf( "Interpret_MultiThread is starting %x\n", (int) pthread_self());
#endif

    assert(THREADINFO_addInitialStackFrameToNode( THREAD_FindThread(), targs->newFrame ) == 0);
    INTERP_Interpret(env, targs->newFrame, 1);
    assert(THREADINFO_removeInitialStackFrameFromNode( THREAD_FindThread(), targs->newFrame ) == 0);
   
    //Maybe we should check for a stack trace here ?
    //let's free any locks on our waiting list 
    THREAD_FreeLocks();

    //indicate that the GC shouldn't scan this thread stack any longer
    nnode->isTerminated = 1;

#ifdef SHOWTHREADS
    eprintf( "Interpret_MultiThread has finished %x\n", (int) pthread_self());
#endif
    {
      pthread_mutex_lock( &threadListMutex );
      
      if(THREADINFO_DecrementThreadCount() == 1)
	{
	  //		  fprintf(stderr, "We're the last thread left!\n");
	}
      pthread_cond_broadcast(&cond);
      pthread_mutex_unlock( &threadListMutex );
    }
    free(targs);
i = 77; //prevent optimising away
    pthread_exit(NULL);
}

extern int INTERP_DebugOn;
//tThreadArgs* targs;

void java_lang_Thread_start
(
  JNIEnv* env,
  jobject obj
)
{
  tStackFrame* pstNewFrame;
  tOBREF pstObject;
  tClassLoaderTuple* pstThreadClass;
  tClassLoaderTuple* toRunClass;
  tMethod* pstMethod;

  tThreadArgs* targs;

  jboolean isThread = JNI_FALSE;

  pthread_t newThread = 0;
  int result;

  /* get pointer to thread-inherited object */
  pstObject = obj;

  /* get pointer to thread class */
  pstThreadClass = CLASSFILE_FindOrLoad(env, "java/lang/Thread", pstClassType); //XXX System class ? -- THREAD?

  /* get pointer to the object's class */
  toRunClass = ODEREF(pstObject)->pstType;

  /* Make sure this object is an instance of thread or runnable */
  if( INTERP_CheckCast(env, pstThreadClass, toRunClass) == 1)
   {
    isThread = JNI_TRUE;
   }

  if(isThread )
   {
  /* get pointer to _run method */
  //  pstMethod = CLASSFILE_FindMethod(pstThreadClass, "run", "()V");


  pstMethod = CLASSFILE_FindMethodInVT(env, toRunClass, "run", "()V");

  if(pstMethod == NULL)
  {
   eprintf("Couldn't find method run() in class %s\n", toRunClass->uidName);
  }
  assert(pstMethod); 

  /* make a stack frame */
  pstNewFrame = InitialStackFrame(pstMethod, pstObject, (int32*) &pstObject, system_heap);

#ifdef SHOWTHREADS
  eprintf("Starting native thread, thread object %x\n", (int) obj);
#endif

  assert(obj);

  targs = sys_malloc( sizeof(tThreadArgs));
 
  assert(targs); 

  targs->newFrame = pstNewFrame;
  targs->threadObject  = obj;
  targs->returnStackBottom = (void*) 0;
  targs->env = env;

  result = pthread_create(&newThread, NULL, (void* (*)(void*)) &Interpret_MultiThread, (void*) targs);

  assert( result == 0);
 
  /* Spin until we know the new thread has been added */
  while(1)
    {
      int found = 0;
      if( THREAD_FindThreadByObject(obj) )
	{
	  found = 1;
	  break;
	}
      else
	{
	  //eprintf("Waiting for thread %p\n", obj);
	}
      usleep(100);
    }
  }
  else
  {
   //not a Thread
   eprintf("%s is not an instance of java/lang/Thread\n", toRunClass->uidName);
  }
}


/* This is called from interp.c's main routine. It "joins" all the other running non-daemon threads.
 */

int THREAD_WaitForNativeThreads()
{
  int broke = 0;
	  while(1)
	      {
		 pthread_mutex_lock( THREAD_getThreadListMutex());
#ifdef SHOWTHREADS
		 eprintf("iLiveThreads is %i\n", (int) iLiveThreads);
#endif
		 if(iLiveThreads <= 0)
		   {
		     broke = 1;
#ifdef GARBAGE_VERBOSE
		     eprintf("Live thread is %i\n", iLiveThreads);
#endif
		     break;
		   }
		  pthread_cond_wait(&cond, THREAD_getThreadListMutex());
		 pthread_mutex_unlock(  THREAD_getThreadListMutex());
		 usleep(5000);
	      }
	  if(broke == 1)
	    pthread_mutex_unlock(&threadListMutex);
	 return 0;
}


void java_lang_Thread__createJNIData
(
  JNIEnv* env,
  jobject obj
)
{
  tJNIData* pstJNIData;

  /* create this thread's JNIData structure */
  pstJNIData = (tJNIData*) sys_malloc(sizeof(tJNIData));
  assert(pstJNIData);
  pstJNIData->functions = JNI_pstFunctionTable;
  pstJNIData->pstException = NULL;
  pstJNIData->ppstLocalRefs = NULL;
  pstJNIData->u16NumLocalRefs = 0;
  pstJNIData->u16NumLocalRefsUsed = 0;
  pstJNIData->pstTopFrame = NULL;
  #ifndef TEASEME
  pstJNIData->pstHeap = system_heap;
  #endif
  JNI_setJNIData( pstJNIData, pthread_self() );

}


tThreadNode* THREAD_FindThreadByID(int id)
{
  tThreadNode* iter;
  pthread_mutex_lock( & threadListMutex );

  if(pstThreadList == NULL)
      {
	pthread_mutex_unlock( & threadListMutex );
	return NULL;
      }

  iter = pstThreadList; 
  while(iter != NULL)
    {
      if( iter->thread_id == id)
	{
	  pthread_mutex_unlock( & threadListMutex );
	  return iter;
	}
      else
	{
//	  eprintf("Couldn't match %x against %x %p\n", id, iter->thread_id, iter->object);
	}
      iter = iter->next;
    }
  pthread_mutex_unlock( & threadListMutex );
#ifdef SHOWTHREADS
  eprintf("Couldn't find thread by ID ...\n");
#endif
  return NULL;
}

tThreadNode* THREAD_FindThread()
{
  tThreadNode* iter;
  pthread_t comp = pthread_self();

  pthread_mutex_lock( & threadListMutex );

  if(pstThreadList == NULL)
      {
	pthread_mutex_unlock( & threadListMutex );
	return NULL;
      }
  
  iter = pstThreadList; 
  while(iter != NULL)
    {
      if( iter->thread_id == comp)
	{
	  pthread_mutex_unlock( & threadListMutex );
	  return iter;
	}
      iter = iter->next;
    }
  pthread_mutex_unlock( & threadListMutex );
  //eprintf("Couldn't find thread ...\n");
  return NULL;
}

tThreadNode* THREAD_FindThreadByObject(tOBREF pstObject)
{
  tThreadNode* iter;

  pthread_mutex_lock( & threadListMutex );

  if(pstThreadList == NULL)
      {
	pthread_mutex_unlock( & threadListMutex );
	return NULL;
      }
  
  iter = pstThreadList; 
  while(iter != NULL)
    {
      if( iter->object == pstObject)
	{
	  pthread_mutex_unlock( & threadListMutex );
	  return iter;
	}
      iter = iter->next;
    }
  pthread_mutex_unlock( & threadListMutex );
#ifdef SHOWTHREADS
  eprintf("Couldn't find thread ...\n");
#endif
  return NULL;
}




/* We don't use this anymore with the new garbage scheme */
#ifndef GARBAGE2

#include <signal.h>



int THREAD_StopAllThreads()
{
tThreadNode* iter;
fprintf(stderr, "Stopping all threads ..\n");
assert( 1 == 7);
pthread_mutex_lock( & threadListMutex );

//  assert(pstThreadList); it can be NULL if there are no threads started yet
						     iter = pstThreadList; 
while(iter != NULL)
  {
#ifdef SHOWTHREADS
    fprintf(stderr, "%x %x\n", (int) iter->thread_id, (int) pthread_self());
#endif
    if( iter->thread_id != pthread_self())
      pthread_kill( &iter->thread_id, SIGSTOP); //here too ???????
    iter = iter->next;
  }
pthread_mutex_unlock( & threadListMutex );
#ifdef SHOWTHREADS
fprintf(stderr, "Stopped all threads\n");
#endif
return 0;
}

int THREAD_WakeAllThreads()
{
  tThreadNode* iter;
#ifdef SHOWTHREADS
    fprintf(stderr, "Waking all threads ..\n");
#endif
    assert( 2 == 7);
    pthread_mutex_lock( & threadListMutex );
    
    assert(pstThreadList);
    iter = pstThreadList; 
    while(iter != NULL)
      {
	pthread_kill( (pthread_t) &iter->thread_id, SIGCONT); //Hmm is this &iter->thread_id ?
	iter = iter->next;
      }
    pthread_mutex_unlock( & threadListMutex );
#ifdef SHOWTHREADS
    fprintf(stderr, "Woken all threads\n");
#endif
    return 0;
}

#endif

//Daemon methods

jboolean java_lang_Thread_nativeIsDaemon(JNIEnv* env, jobject thisthread)
{
  tThreadNode* node = THREAD_FindThreadByObject(thisthread)  ;
    if(node)
    {
      return node->isDaemon;
    }
  return JNI_FALSE;
}

void java_lang_Thread_nativeSetDaemon(JNIEnv* env, jobject thisthread, jboolean on)
{
    tThreadNode* node = THREAD_FindThreadByObject(thisthread)  ;
    if(node)
    {
      //Now we need to determine if this changes the status, if it does, we have to decrease the number of threads that have to stop before the VM exits
      int changedStatus = (on == node->isDaemon);							     
      node->isDaemon = on;
#ifdef SHOWTHREADS
      eprintf("Daemon status is %x\n", on);
#endif
      if(changedStatus && on)
	{
	  pthread_mutex_lock( & threadListMutex );
	  iLiveThreads--;
	  pthread_mutex_unlock( & threadListMutex );
	}
      else if(changedStatus)
	{
	  assert( 2 == 3); //can't make a deamon thread non-deamon
	}
    }
  return;
}
#endif // #ifdef NATIVE_THREADS








