#include <config.h>

/* This file contains 'helper' methods which are called by optimised code

   They do things which are too difficult to write in assembler, like invoking methods, getting object fields etc
 */


#include "svetlana.h"
#include "../interp.h"
#include "../jni.h"
#include "../interp_methods.h"
#include "../classfile_methods.h"
#include "../newobject.h"

#include <stdio.h>

#include <assert.h>

/* Takes a pointer to an array and returns the element at index

   The return result is put at the 'result' pointer, which is just a position on the optimised method's stack

   If an exception needs to be thrown we return an exception object.

   Otherwise we return 0 to indicate success.
 */

tOBREF SVETLANA_iaload( int* result, JNIEnv* env, int loadType, int index, jarray array)
{
    //    fprintf(stderr," Args are %x %x %x\n", result, index , array);
    if(array != NULL) 
	{
	    int i32Number = ADEREF(array)->i32Number;
	    if( (index < 0) || (index >= i32Number))
		{ 
		    return INTERP_NewObjectFromName(env,"java/lang/ArrayIndexOutOfBoundsException");
		}
	    else
		{
		  if(loadType == 0)
		    *result = ((int*) (ADEREF(array)->pvElements))[index];
		  else if(loadType == 1)
		    *result = ((uint16*) (ADEREF(array)->pvElements))[index];
		  else if(loadType == 2)
		    {
		      int temp = ((byte*) (ADEREF(array)->pvElements))[index];
		      if(temp & 0x80)
			*result = 0xffffff00 | temp;
		    }
		    return (tOBREF) 0;
		}
	}
    else
	{
	  tOBREF ret = INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/NullPointerException", "optimised method accessed null array reference");
	  fprintf(stderr, "Returning exp %p from jni_methods.SVETLANA_iaload\n", ret);
	  return  ret;
	}
}

/* The byte version */


tOBREF SVETLANA_bastore(JNIEnv* env, int value, int index, jarray array)
{
    if(array)
	{
	    int i32Number = ADEREF(array)->i32Number;
	    if( (index < 0) || (index >= i32Number))
	      {
		  return INTERP_NewObjectFromName(env, "java/lang/ArrayIndexOutOfBoundsException");
		}
	    else
	      {
		  ((byte*) (ADEREF(array)->pvElements))[index] = (value); 
		    return (tOBREF) 0;
		}
	}
    else
      {
	tOBREF ret = INTERP_ExceptionObjectFromNameAndMessage(env,  "java/lang/NullPointerException", "optimised method accessed null array reference");
	//fprintf(stderr, "Returning exp %p from jni_methods.SVETLANA_iaload\n", ret);
	return  ret;
      }
}

/* The byte version */

tOBREF SVETLANA_baload(int* result, JNIEnv* env, int index, jarray array)
{
    if(array)
	{
	    int i32Number = ADEREF(array)->i32Number;
	    if( (index < 0) || (index >= i32Number))
		{
		    return INTERP_NewObjectFromName(env, "java/lang/ArrayIndexOutOfBoundsException");
		}
	    else
		{
		    //Warning, you can only change *result once!!
		    int temp = ((byte*) (ADEREF(array)->pvElements))[index]; 
		    if(temp & 0x80)
			*result = 0xffffff00 | temp;
		    return (tOBREF) 0;
		}
	}
    else
	{
	    tOBREF ret = INTERP_ExceptionObjectFromNameAndMessage(env,  "java/lang/NullPointerException", "optimised method accessed null array reference");
	    //fprintf(stderr, "Returning exp %p from jni_methods.SVETLANA_iaload\n", ret);
	    return  ret;
	}
}

/*
tOBREF SVETLANA_laload(int* result2, int* result1, int index, jarray array)
{
    if(ADEREF(array))
	{
	    int i32Number = ADEREF(array)->i32Number;
	    if( (index < 0) || (index >= i32Number))
		{
		    return INTERP_NewObjectFromName("java/lang/ArrayIndexOutOfBoundsException");
		}
	    else
		{
		    *result1 = ((int*) (ADEREF(array)->pvElements))[index * 2];
		    *result2 = ((int*) (ADEREF(array)->pvElements))[index * 2 + 1];
		    return (tOBREF) 0;
		}
	}
    else
	return INTERP_NewObjectFromName("java/lang/NullPointerException");
}
*/
tOBREF SVETLANA_iastore(JNIEnv* env, int value, int index, jarray array)
{
    //    fprintf(stderr," Args are %x %x %x\n", value, index , array);
    if(ADEREF(array))
	{
	    int i32Number = ADEREF(array)->i32Number;
	    if( (index < 0) || (index >= i32Number))
		{
		    return INTERP_NewObjectFromName(env, "java/lang/ArrayIndexOutOfBoundsException");
		}
	    else
		{
		    ((int*) (ADEREF(array)->pvElements))[index] = value;
		    return (tOBREF) 0;
		}
	}
    else
	return INTERP_NewObjectFromName(env, "java/lang/NullPointerException");
}


tOBREF SVETLANA_arraylength(int* result, JNIEnv* env, jarray array)
{
        if (ADEREF(array))
        {
	    *result =  ADEREF( array )->i32Number;
	    return 0;
        }
        else
        {
	    return  INTERP_NewObjectFromName(env, "java/lang/NullPointerException");
        }
}


tOBREF SVETLANA_newarray(int* result, JNIEnv* env, int eltype, int i32Number)
{
    //    fprintf(stderr,"newarray %x %x %x\n", result, eltype, i32Number);
  if (i32Number >= 0)
        {
          *result = (int32) INTERP_NewArray(env, eltype, i32Number, NULL, NULL);
	  return 0;
        }
        else
        {
	    return INTERP_NewObjectFromName(env, "java/lang/NegativeArraySizeException");
	}
}

int SVETLANA_getfield_quick(int* result, JNIEnv* env, jobject object, int i32Temp)
{
    //    fprintf(stderr, "Get field quick %p %x %x\n", result, object, i32Temp);
    if(object)
	{
	*result = (int) (ODEREF(object))->pi32Vars[i32Temp];
	return 0;
	}
    else
	return (int) INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/NullPointerException", "Optimised method executing getfield_quick encountered null reference");
}


/* Just puts a value into an object field 

   The oldStackPointer argument is a pointer to the value of the stack pointer when the method was called.

   We may take 1 or 2 words from that address depending on the size of the slot we are -putting- into.

   We change the address of the stack pointer to reflect whether we took 1 or 2 words
*/


tOBREF SVETLANA_putfield(JNIEnv* env, tClassLoaderTuple* pstCurrClass, int i32Temp,  int32* nativeStackPointer) 
{
    Uid           uidBytes;
    fprintf(stderr, "here we are in putfield %p %x %p\n", pstCurrClass, i32Temp, nativeStackPointer);
    //    assert( 1 == 2);
    uidBytes = CONSTGET_UidFieldSig(env, pstCurrClass, i32Temp);

        switch (uidBytes[0])
        {
	    //We have removed the PERSIST code here 
	  case 'L':
	  case '[':
          case 'B':
          case 'C':
          case 'F':
          case 'I':
          case 'S':
          case 'Z':
          {
	    tOBREF object = (tOBREF) nativeStackPointer[1];
            if (object != NULL)
            {
               DEREF((tOBREF) object)->pi32Vars[CONSTGET_InstFieldOffset(env, pstCurrClass, i32Temp)] = nativeStackPointer[0];
	       nativeStackPointer[0] = 8 + 4; //We tell the native code to pop 3 words
	       return 0; //success
            }
            else
            {
		return INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/NullPointerException", "Optimised method executing putfield encountered null reference");
            }
            break;
          }
          // 64 bits - put in consecutive stores 
          case 'D':
          case 'J':
          {
	    tOBREF object = (tOBREF) nativeStackPointer[2];
            if (object != NULL)
            {
               DEREF((tOBREF) object)->pi32Vars[CONSTGET_InstFieldOffset(env, pstCurrClass, i32Temp)] = nativeStackPointer[1];
               DEREF((tOBREF) object)->pi32Vars[CONSTGET_InstFieldOffset(env, pstCurrClass, i32Temp) + 1] = nativeStackPointer[0];
	       nativeStackPointer[0] = 16; //We tell the native code to pop 4 words (the two long parts and 
	       return 0;
            }
            else
            {
		return INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/NullPointerException", "Optimised method executing putfield encountered null reference");
            }
            break;
          }
          default:
          {
	      
            fprintf(stderr, "Interpreter error in native getfield instruction. \n");
	    exit(2); //We do NOT want to be here
		return INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/NullPointerException", "Optimised method executing putfield encountered internal error");
          }
        }


	return NULL;
}


/* We must fix this to return an exception if necessary */

int SVETLANA_getfield(JNIEnv* env, jobject object, tClassLoaderTuple* pstCurrClass, int i32Temp)
{
    Uid           uidBytes;


    uidBytes = CONSTGET_UidFieldSig(env, pstCurrClass, i32Temp);

        switch (uidBytes[0])
        {
	    //We have removed the PERSIST code here 
	  case 'L':
	  case '[':
          case 'B':
          case 'C':
          case 'F':
          case 'I':
          case 'S':
          case 'Z':
          {
            if (DEREF(object))
            {
              return DEREF((tOBREF) object)->pi32Vars[CONSTGET_InstFieldOffset(env, pstCurrClass, i32Temp)];
            }
            else
            {
		return -1;
		//              SAVEENV();
		//              pstCurrFrame = ThrowNamedException(pstCurrFrame, "java/lang/NullPointerException");
		//              LOADENV();
            }
            break;
          }
          // 64 bits - put in consecutive stores 
  				case 'D':
          case 'J':
          {
            if (object)
            {
              return DEREF((tOBREF) (object))->pi32Vars[CONSTGET_InstFieldOffset(env, pstCurrClass, i32Temp) + 1];
	      //              *(optop - 1) = DEREF((tOBREF) *(optop - 1))->pi32Vars[CONSTGET_InstFieldOffset(pstCurrClass, i16Temp)];
            }
            else
            {
		//              SAVEENV();
		//              pstCurrFrame = ThrowNamedException(pstCurrFrame, "java/lang/NullPointerException");
		//              LOADENV();
            }
            break;
          }
          default:
          {
            fprintf(stderr, "Interpreter error in native getfield instruction. \n");
	    return -1;
          }
        }


	return 0;
}


/* XXXXXXXXXXXXXXXXXXXXXXXXXXX warning, result points to itself, once you've modified *result, leave it alone */

tOBREF SVETLANA_ldc(JNIEnv* env, tClass* pstCurrClass, int i32Temp, int32* nativestack)
{
				Uid           uidBytes;

				fprintf(stderr,"Calling SVETLANA_ldc %x %x %x\n", (int) nativestack, (int) pstCurrClass, i32Temp);

				switch (CONSTTAG(pstCurrClass, i32Temp))
				{
					case CONSTANT_Integer:
					{
					    nativestack[0] = 1;
					    nativestack[1] = CONSTGET_Int(pstCurrClass, i32Temp);
					    fprintf(stderr, "returning from _ldc INT with 1 words\n");
					    return 0;
					}
				case CONSTANT_Long: 
					{
					    assert( 3 == 4); //This should be ldc2
					    nativestack[0] = 2;
					    nativestack[1] = CONSTGET_Int(pstCurrClass, i32Temp);
					    nativestack[2] = CONSTGET_Int(pstCurrClass, i32Temp + 1);
					    fprintf(stderr, "returning from _ldc LONG with 2 words\n");
					    return 0;
					}
					case CONSTANT_Float:
					{
#ifdef DEBUG_FLOAT
				fprintf(stderr, "Processing ldc %x, index is %i\n", *(pbPC - 2), i16Temp);
#endif
#ifdef DEBUG_FLOAT
					    fprintf(stderr, "Getting float %G\n", CONSTGET_Float(pstCurrClass, i16Temp));
					    fprintf(stderr, "Class is %s, index %i %G\n", pstCurrClass->uidName, i16Temp, ((float*)pstCurrClass->pu32ConstPool)[i16Temp]);
#endif
					    nativestack[0] = 1;
					    *((float*) (nativestack + 1)) =  CONSTGET_Float(pstCurrClass, i32Temp);
					    fprintf(stderr, "returning from _ldc FLOAT with 1 words\n");
					    return 0;
				        }
				        case CONSTANT_String:
				        {
					    //				assert(2 == 3);
					    //all strings are stored as Asciz strings in UTF8 format 
					    nativestack[0] = 1;
					    
					uidBytes = CONSTGET_UidAsciz(pstCurrClass, CONSTGET_String_Index(pstCurrClass, i32Temp));
					nativestack[1] = (int32) INTERP_NewStringFromAsciz(env, uidBytes);
					    fprintf(stderr, "returning from _ldc STRING with 1 words\n");
					return 0;
					//					*(++optop) = (int32) INTERP_NewStringFromAsciz(uidBytes);
					}
					default:
					{
					fprintf(stderr, "SVETLANA Interpreter error in ldc1 instruction. Tag type was %i\n", CONSTTAG(pstCurrClass, i32Temp));
					return (tOBREF) 1;
					}
				}
}


tOBREF SVETLANA_new(JNIEnv* env, int i32Temp, tClassLoaderTuple* pstCurrClass)
{
				tClassLoaderTuple*       pstClassTemp;
				tOBREF      pstObjectTemp;

				//				i16Temp = (*pbPC++) * 256;
				//				i16Temp += (*pbPC++);

				#ifdef DEBUG
				assert(CONSTTAG(pstCurrClass->pstClass, i32Temp) == CONSTANT_Class);
				#endif

				pstClassTemp = CONSTGET_Class(env, pstCurrClass, i32Temp);
				if (pstClassTemp)
				{

				    pstObjectTemp = INTERP_NewObject(env, pstClassTemp);
				    if (pstObjectTemp)
					{
					    return pstObjectTemp;
					    //					    *(++optop) = (int32) pstObjectTemp;
					}
				    else
					{
					    return NULL;
					}
				}
				else
				    {
					return NULL;
				    }
}


/* I have realised that there is no crafty solution here. The calling code must do a comparison to see whether it should put 0,1, or 2 words on the native stack.

We should put the number of words returned in nativestack[0].

The optional low word goes in nativestack[1] //Most common case
The optional high word goes in nativestack[2] 

*/

tOBREF SVETLANA_invoke_virtual_quick(JNIEnv* env, int32* optop, int i32VTIndex, int i32ArgSize, int32* nativestack)
{
				tClassLoaderTuple*       pstClassTemp;
				tMethod*      pstMethodTemp;

				//This native stack grows downwards
				if ((tOBREF) *(optop - 1 + i32ArgSize) == NULL)
				    {
				    fprintf(stderr, "invokevirtual_quick is returning NP exception\n");
					return INTERP_NewObjectFromName(env, "java/lang/NullPointerException");
				    }


				pstClassTemp = (DEREF((tOBREF) (*(optop - 1 + i32ArgSize))))->pstType;
				pstMethodTemp = pstClassTemp->pstClass->ppstVT[i32VTIndex];

				if(pstMethodTemp->u16AccessFlags & ACC_NATIVE)
				  {
				      int i;
				      tClassLoaderTuple* theClass = pstMethodTemp->pstClass;
				      int iFound;
				      void* pFunction = NULL;
				      assert((pstMethodTemp->u16AccessFlags & ACC_STATIC) == 0);
				      assert(1 == 2);
				      for (i = 0; i < theClass->pstClass->u16NumNatives; i++)
					  {
					      if (theClass->pstClass->pstNativeMethods[i].fnPtr)
						  {
	      //      fprintf(stderr, "%i Comping %s and %s\n", i, pstMethod->pstClass->pstNativeMethods[i].name , pstMethodTemp->uidName);
						      if (theClass->pstClass->pstNativeMethods[i].name == pstMethodTemp->uidName)
							  {
							      if (theClass->pstClass->pstNativeMethods[i].signature == pstMethodTemp->uidSignature)
								  {
								      iFound = 1;
								      pFunction = theClass->pstClass->pstNativeMethods[i].fnPtr;
								      break;
								  }
							  }
						  }
					  }
				      assert(pFunction);
				      fprintf(stderr, "******* Cannot invoke a native method here %s\n", pstMethodTemp->uidName);
				       *optop = (int) pFunction;
				    return (tOBREF) 1;
				  }

				fprintf(stderr,"Doing invoke_v_quick for method %s.%s (%s) args %i\n", pstClassTemp->uidName, pstMethodTemp->uidName, pstMethodTemp->uidSignature, i32ArgSize);



				//We only return an exception object. Otherwise the result goes on the optop stack
				{
				    int iWords;
				    tOBREF ret;
				    ret = INTERP_RunVirtualMethodFromPtr_with_native_stack(env, pstMethodTemp, (int32*) optop);



				    //Alright now we need to find out how many words this method returns
				    iWords = pstMethodTemp->u16RetSize;
				    if(iWords == 0)
					{
					    nativestack[0] = 0;
					    nativestack[1] = 0;
					    nativestack[2] = 0;
					    //					    fprintf(stderr, "Set nativestack[0] to %p\n", nativestack[0]);
					}
				    else if(iWords == 1)
					{
					    fprintf(stderr, "Setting nativestack[0] to 1 %p\n", nativestack);
				          nativestack[0] = 1;
					    fprintf(stderr, "Set nativestack[0] to 1 %p\n", nativestack);
					  nativestack[1] = (int) *(optop);
					    fprintf(stderr, "Set nativestack[1] to %x %p\n", (int) *(optop),nativestack);
					    nativestack[2] = 0;

					    //					    for(i = -4; i < 3;i++)
					    //						fprintf(stderr, "Stack %i: %p %x\n", i, nativestack + i, *(nativestack + i));
					}
				    else if(iWords == 2)
					{
					    nativestack[0] = 2;
					    nativestack[1] = *(optop + 1); //Which way round?
					    nativestack[2] = *(optop);

					    //				    fprintf(stderr, "Called virtual method, words to pop is %x, ret is %x, *optop (%x)is %x, *(optop + 1) is %x\n", nativestack[0], ret, optop, *optop, *(optop + 1));
					}

				    fprintf(stderr, "invokevirtual_quick is returning %p\n", ret);
				    return ret;
				    }
}

tOBREF SVETLANA_invoke_virtual(JNIEnv* env, int32* optop, tClassLoaderTuple* pstCurrClass, int i32VTIndex)
{
				tMethod*      pstMethodTemp;

				pstMethodTemp = CONSTGET_MethodNonVirtual(env, pstCurrClass, i32VTIndex);

				//This native stack grows downwards
				if ((tOBREF) *(optop -1 + pstMethodTemp->u16ArgSize ) == NULL)
				    {
					return INTERP_NewObjectFromName(env, "java/lang/NullPointerException");
				    }

				pstMethodTemp = DEREF((tOBREF) *(optop - pstMethodTemp->u16ArgSize + 1))->pstType->pstClass->ppstVT[pstMethodTemp->u16VTIndex];


				if(pstMethodTemp->u16AccessFlags & ACC_NATIVE)
				  {
				      int i;
				      tClassLoaderTuple* theClass = pstMethodTemp->pstClass;
				      int iFound;
				      void* pFunction = NULL;

				      for (i = 0; i < theClass->pstClass->u16NumNatives; i++)
					  {
					      if (theClass->pstClass->pstNativeMethods[i].fnPtr)
						  {
	      //      fprintf(stderr, "%i Comping %s and %s\n", i, pstMethod->pstClass->pstNativeMethods[i].name , pstMethodTemp->uidName);
						      if (theClass->pstClass->pstNativeMethods[i].name == pstMethodTemp->uidName)
							  {
							      if (theClass->pstClass->pstNativeMethods[i].signature == pstMethodTemp->uidSignature)
								  {
								      iFound = 1;
								      pFunction = theClass->pstClass->pstNativeMethods[i].fnPtr;
								      break;
								  }
							  }
						  }
					  }
				      assert(pFunction);
				      fprintf(stderr, "******* Cannot invoke a native (non quick )method here %s\n", pstMethodTemp->uidName);
				       *optop = (int) pFunction;
				    return (tOBREF) 1;
				  }

				//We only return an exception object. Otherwise the result goes on the optop stack
				{
				    tOBREF ret;
				    fprintf(stderr, "Calling Virtual method %p %p\n", pstMethodTemp, optop);
				ret = INTERP_RunVirtualMethodFromPtr_with_native_stack(env, pstMethodTemp, optop);
				fprintf(stderr, "Called virtual method, ret is %p, *optop (%p)is %x\n", ret, optop, (int) *optop);
				return ret;
				}
}


tOBREF SVETLANA_invokestatic(JNIEnv* env, int32* optop, tClassLoaderTuple* pstCurrClass, int i32MethodIndex) 
{
				tMethod*      pstMethodTemp;

				pstMethodTemp = CONSTGET_MethodStatic(env, pstCurrClass, i32MethodIndex);
				
				if(pstMethodTemp->u16AccessFlags & ACC_NATIVE)
				  {
				      int i;
				      tClassLoaderTuple* theClass = pstMethodTemp->pstClass;
				      int iFound;
				      void* pFunction = NULL;

				      assert( 3 == 4);
				      for (i = 0; i < theClass->pstClass->u16NumNatives; i++)
					  {
					      if (theClass->pstClass->pstNativeMethods[i].fnPtr)
						  {
	      //      fprintf(stderr, "%i Comping %s and %s\n", i, pstMethod->pstClass->pstNativeMethods[i].name , pstMethodTemp->uidName);
						      if (theClass->pstClass->pstNativeMethods[i].name == pstMethodTemp->uidName)
							  {
							      if (theClass->pstClass->pstNativeMethods[i].signature == pstMethodTemp->uidSignature)
								  {
								      iFound = 1;
								      pFunction = theClass->pstClass->pstNativeMethods[i].fnPtr;
								      break;
								  }
							  }
						  }
					  }
				      assert(pFunction);
				      fprintf(stderr, "******* Cannot invoke a static native method here %s\n", pstMethodTemp->uidName);
				       *optop = (int) pFunction;
				    return (tOBREF) 1;
				  }

				assert( pstMethodTemp->u16ArgSize > 0);
				//We only return an exception object. Otherwise the result goes on the optop stack
				{
				    tOBREF ret;
				    ret = INTERP_RunStaticMethodFromPtr_with_native_stack(env, pstMethodTemp, optop);
				    //				    fprintf(stderr, "Called virtual method, ret is %x, *optop (%x)is %x\n", ret, optop, *optop);
				    return ret;
				}
}


tOBREF SVETLANA_invokespecial(JNIEnv* env, int32* optop, tClassLoaderTuple* pstCurrClass, int isQuick, int i32MethodIndex) 
{
  tMethod*      pstMethodTemp;
  
if(isQuick)
  {
  eprintf("Invoking special_quick, pstCurrClass %p, i32MethodIndex %i\n", pstCurrClass, i32MethodIndex);
  pstMethodTemp = (tMethod*) CONSTGET(pstCurrClass->pstClass, i32MethodIndex);
  }
else
  pstMethodTemp = CONSTGET_MethodNonVirtual(env, pstCurrClass, i32MethodIndex);
  
  if ((tOBREF) *(optop - pstMethodTemp->u16ArgSize + 1) == NULL)
    {
      return INTERP_NewObjectFromName(env, "java/lang/NullPointerException");
    }
  
  
  if(pstMethodTemp->u16AccessFlags & ACC_NATIVE)
    {
      assert( 3 == 4);
      return (tOBREF) 1;
    }
  
  assert( pstMethodTemp->u16ArgSize > 0);
  //We only return an exception object. Otherwise the result goes on the optop stack
    {
      tOBREF ret;
      ret = INTERP_RunSpecialMethodFromPtr_with_native_stack(env, pstMethodTemp, optop);
//      fprintf(stderr, "Called special method, ret is %x, *optop (%x)is %x\n", (int) ret, (int) optop, (int) *optop);
      return ret;
    }
}

void Java_debugprint(int num)
{
    fprintf(stderr, "Debug print %x\n", num);
}


//1

jint Java_getDebugprint_methodpointer(  JNIEnv* env, jobject obj)
{
    void (*func) (int) =  Java_debugprint;
    return (int) (func);
}
//2
jint Java_getGetfield_quick_methodpointer(  JNIEnv* env, jobject obj)
{
   int (*func) (int*, jobject, int) =  SVETLANA_getfield_quick;
   return (int) (func);
}
//3
jint Java_getGetfield_methodpointer(  JNIEnv* env, jobject obj)
{
   int (*func) (jobject, tClassLoaderTuple*, int) =  SVETLANA_getfield;
   return (int) (func);
}
//4
//tOBREF SVETLANA_putfield( tClassLoaderTuple* pstCurrClass, int i32Temp,  int32* nativeStackPointer) 
jint Java_getPutfield_methodpointer(  JNIEnv* env, jobject obj)
{
   tOBREF (*func) ( tClassLoaderTuple*, int, int32*) =  SVETLANA_putfield;
   return (int) (func);
}

//5
jint Java_getInvokevirtual_quick_methodpointer(  JNIEnv* env, jobject obj)
{
   tOBREF (*func) (int32*, int, int, int32*) =  SVETLANA_invoke_virtual_quick;
   return (int) (func);
}

//6
//tOBREF SVETLANA_invoke_virtual(int32* optop, tClassLoaderTuple* pstCurrClass, int i32VTIndex)
jint Java_getInvokevirtual_methodpointer(  JNIEnv* env, jobject obj)
{
   tOBREF (*func) (int32*, tClassLoaderTuple*, int) =  SVETLANA_invoke_virtual;
   return (int) (func);
}

//7
jint Java_getInvokestatic_methodpointer(  JNIEnv* env, jobject obj)
{
   tOBREF (*func) (int32*, tClassLoaderTuple*,  int) =  SVETLANA_invokestatic;
   return (int) (func);
}

//8
jint Java_getInvokespecial_methodpointer(  JNIEnv* env, jobject obj)
{
  tOBREF (*func) (int32*, tClassLoaderTuple*, int, int) =  SVETLANA_invokespecial;
  return (int) (func);
}

//tOBREF SVETLANA_invokespecial(int32* optop, tClassLoaderTuple* pstCurrClass, int i32MethodIndex) 

//9

jint Java_getLdc_methodpointer(  JNIEnv* env, jobject obj)
{
    tOBREF (*func) (tClassLoaderTuple*, int, int32*) =  SVETLANA_ldc;
    return (int) (func);
}

//int SVETLANA_ldc(int* result, tClassLoaderTuple* pstCurrClass, int i32Temp)

//10

jint Java_getNew_methodpointer(  JNIEnv* env, jobject obj)
{
   tOBREF (*func) (int,tClassLoaderTuple*) =  SVETLANA_new;
   return (int) (func);
}

//11
jint Java_getNewarray_methodpointer(  JNIEnv* env, jobject obj)
{
    tOBREF (*func) (int*, int ,int) =  SVETLANA_newarray;
    return (int) (func);
}

//12

jint Java_getIaload_methodpointer(  JNIEnv* env, jobject obj)
{
    tOBREF (*func) (int*, int, int, jarray) =  SVETLANA_iaload;
    return (int) (func);
}
//13
jint Java_getBaload_methodpointer(  JNIEnv* env, jobject obj)
{
   tOBREF (*func) (int*, int, jarray) =  SVETLANA_baload;
   return (int) (func);
}

//14
//tOBREF SVETLANA_bastore(int value, int index, jarray array)
jint Java_getBastore_methodpointer(  JNIEnv* env, jobject obj)
{
 tOBREF (*func) (int, int, jarray) =  SVETLANA_bastore;
 return (int) (func);
}

//15
jint Java_getLaload_methodpointer(  JNIEnv* env, jobject obj)
{
  assert(3 == 4);
    //    tOBREF (*func) (int*, int*, int, jarray) =  SVETLANA_laload;
    //    return (int) (func);
}

//16
jint Java_getIastore_methodpointer(  JNIEnv* env, jobject obj)
{
    tOBREF (*func) (int, int, jarray) =  SVETLANA_iastore;
    return (int) (func);
}

//17


jint Java_getArray_length_methodpointer(  JNIEnv* env, jobject obj)
{
    tOBREF (*func) (int*,jarray) =  SVETLANA_arraylength;
    return (int) (func);
}

int reference()
{
    int bobo = 0x12345678;
    if(bobo & 0x80)
	bobo = bobo | 0xffffff00;

    return 0;
}


