#include <config.h>
/*
 * store.c
 *
 * Very simple store. PIDs are the (negated) offset of the object in the
 * data file. The size of an object is stored preceeding the actual
 * object data in the store. Transfer to and from disk is done with fread
 * and fwrite, so the store can only be used on one platform after it's
 * created.
 *
 */

#ifdef DEBUG
#include <assert.h>
#endif

#include <stdio.h>
#include <stdlib.h>

#include "store.h"


#ifndef TEASEME
static FILE* datafile;
static int firstavail;
#endif

/*
 * Opens an existing persistent store.
 *
 */

int STORE_OpenStore
(
  const char* storename
)
{
  #ifdef TEASEME
  return 1;
  #else
  datafile = fopen(storename, "rb+");
  if (datafile)
  {
    fread(&firstavail, sizeof(firstavail), 1, datafile);
    return 1;
  }
  else
  {
    return 0;
  }
#endif
}

/*
 * Opens a new persistent store.
 *
 */

int STORE_NewStore
(
  const char* storename
)
{

  #ifdef TEASEME
  return 0;
#else
  fprintf(stderr, "store.c: NewStore with %s\n", storename);
  /* try opening file with that name */
  datafile = fopen(storename, "rb");
  if (datafile)
  {
    /* store already exists, so close it and return 0 */
    fclose(datafile);
    return 0;
  }
  else
  {
    /* store does not exist, so create it */


     fprintf(stderr, "Creating store %s...\n", storename);


    datafile = fopen(storename, "wb+");
    /* firstavail must point to after itself and the root PID */
    firstavail = sizeof(firstavail) + sizeof(PID);
    fwrite(&firstavail, sizeof(firstavail), 1, datafile);
    STORE_SetRootPID(0);
    return 1;
  }
#endif
}

/*
 * Closes an open store.
 *
 */

void STORE_CloseStore
(
  void
)
{
#ifdef TEASEME
  return;
#else

  #ifdef DEBUG
  fprintf(stderr, "STORE_CloseStore Closing store\n");
  #endif
  fseek(datafile, 0, SEEK_SET);
  fwrite(&firstavail, sizeof(firstavail), 1, datafile);
  fclose(datafile);
  datafile = NULL;
  firstavail = 0;
  #ifdef DEBUG
  fprintf(stderr, "STORE_CloseStore Closed store\n");
  #endif
#endif
}

/*
 * Gets the root object's PID
 *
 */

PID STORE_GetRootPID
(

  void
)
{
  #ifdef TEASEME
  return 0;
#else

  PID pid;
#ifdef TRACESTORE
  fprintf(stderr, " In STORE_GetRootPID ,seeking to %i\n", sizeof(firstavail));
#endif

  fseek(datafile, sizeof(firstavail), SEEK_SET);

#ifdef TRACESTORE
  fprintf(stderr, " In STORE_GetRootPID ,reading %i bytes\n", sizeof(pid));
#endif

  fread(&pid, sizeof(pid), 1, datafile);
#ifdef TRACESTORE
  fprintf(stderr, " In STORE_GetRootPID ,read pid: %i\n", pid);
#endif
  return pid;
#endif
}

/*
 * Set the root object's PID: NEVER USE! (except where needed)
 *
 */

void STORE_SetRootPID
(
  PID pid
)
{
  #ifdef TEASEME
  return;
  #else
#ifdef TRACESTORE
  fprintf(stderr, " In STORE_SetRootPID ,seeking to %i\n", sizeof(firstavail));
#endif
  fseek(datafile, sizeof(firstavail), SEEK_SET);

  fwrite(&pid, sizeof(pid), 1, datafile);

#ifdef TRACESTORE
  fprintf(stderr, " In STORE_SetRootPID ,wrote pid %i\n", pid);
#endif
  #endif
}

/*
 * Gets the size of an object in the store.
 *
 */

int STORE_GetObjectSize
(
  PID pid
)
{
  #ifdef TEASEME
  return 0;
  #else
  int objsize;

  #ifdef DEBUG
  if ((int32*) pid == NULL)
  {
    fprintf(stderr, "Tried to get size from NULL PID\n");
    assert( (int32*) pid != NULL);
  }
  #endif

  fseek(datafile, -pid, SEEK_SET);
  fread(&objsize, sizeof(int), 1, datafile);
  return objsize;
  #endif
}

/*
 * Loads an object from the store.
 *
 */

void STORE_LoadObject
(
  PID pid,
  char* obj
)
{
#ifdef TEASEME
return;
#else
  int objsize;
  int result;

  //  #ifdef DEBUG
  if (-pid >= firstavail)
  {
    fprintf(stderr, "Loading past end of store!!!");
  }
  if ((int32*) pid == NULL)
  {
    fprintf(stderr, "Tried to load from NULL PID\n");
  }
  //  #endif

  #ifdef TRACESTORE
   fprintf(stderr, "STORE_LoadObject %x %x\n", pid, obj);
#endif
  fseek(datafile, -pid, SEEK_SET);
  fread(&objsize, sizeof(int), 1, datafile);
  result = fread(obj, objsize, 1, datafile);
  if(result != 1)
      {
	  perror("didn't read whole object\n");
      }

  #ifdef TRACESTORE
   fprintf(stderr, "STORE_LoadObject loaded read %i, objsize %i, pid %x obj %x\n", result, objsize, pid, obj);
#endif
#endif
}

/*
 * Updates an existing object in the store with new data given.
 *
 */


#include "classfil.h"

void STORE_UpdateObject
(
  PID pid,
  void* obj
)
{
#ifdef TEASEME

#else
  int objsize;
  tObject* dummy;


 #ifdef TRACESTORE
  fprintf(stderr, "STORE_UpdateObject %x %x\n", pid, obj);
 #endif

  #ifdef DEBUG
  if( (int32*) pid == NULL)
  {
    fprintf(stderr, "Tried to update NULL PID\n");
  }
  #endif

  dummy = obj;
  //  printf("Type %x %s\n", dummy, dummy->pstType->uidName);
  
  //assert( dummy->pstType != NULL);

  fseek(datafile, -pid, SEEK_SET);
  fread(&objsize, sizeof(int), 1, datafile);
  fseek(datafile, 0, SEEK_CUR);
  fwrite(obj, objsize, 1, datafile);
 #ifdef TRACESTORE
  fprintf(stderr, "STORE_UpdateObject fin %x %x\n", pid, obj);
 #endif
#endif
}

/*
 * Allocates a new object in the store - doesn't write it though... just
 * gives it a pid and writes its size... use STORE_UpdateObject to write
 * it once you have its pid.
 *
 */

PID STORE_NewObject
(
  int objsize
)
{
#ifdef TEASEME
return -1;
#else

  PID pid;

#ifdef TRACESTORE
  fprintf(stderr, "Store.c STORE_NewObject firstAvail %i, objsize %i\n", firstavail, objsize);
#endif

  /* seek to first available space */
  fseek(datafile, firstavail, SEEK_SET);
  /* write new object's size */
  fwrite(&objsize, sizeof(objsize), 1, datafile);

  /* write new object */ /* don't write... we just want to allocate */
/*  fwrite(obj, objsize, 1, datafile);*/
  /* new object's PID is its location negated */
  pid = -firstavail;
  /* first available space is now after this object (and its size) */
  firstavail += (objsize + sizeof(objsize));

#ifdef TRACESTORE
  fprintf(stderr, "firstavail is now %i\n", firstavail);
#endif

  /* seek to beginning and update record of first available space */
  /* don't write it yet - much more efficient just to write when closing */
/*  fseek(datafile, 0, SEEK_SET);
  fwrite(&firstavail, sizeof(firstavail), 1, datafile);*/
  /* return new object's PID */
  return pid;
#endif
}



















































/*  -  ----------------------- Old PJSL stuff ------- --------------- */














#ifdef PJSLCOMPAT

pjsl_t* STORE_Store = NULL;

#ifdef OLDPJSLCOMPAT

pjsl_t* pjsl_new
(
  void
)
{
  pjsl_t* p;

  p = (pjsl_t*) malloc(sizeof(pjsl_t));
  p->dataFile = NULL;
  p->firstAvail = 0;
  return p;
}

pjsl_err_t pjsl_create
(
  pjsl_t* pjsl,
  char* name,
  uint bufferSize,
  uint maxParts,
  uint maxSize,
  char* appVersion
)
{
  /* try opening file with that name */
  pjsl->dataFile = fopen(name, "rb");
  if (pjsl->dataFile)
  {
    /* store already exists, so close it and return 0 */
    fclose(pjsl->dataFile);
    return pjsl_error;
  }
  else
  {
    /* store does not exist, so create it */
    pjsl->dataFile = fopen(name, "wb+");
    /* firstavail must point to after itself and the root PID */
    pjsl->firstAvail = sizeof(pjsl->firstAvail) + sizeof(pjsl_pid_t);
    fwrite(&(pjsl->firstAvail), sizeof(pjsl->firstAvail), 1, pjsl->dataFile);
    return pjsl_ok;
  }
}

pjsl_err_t pjsl_open
(
  pjsl_t* pjsl,
  char* name,
  uint bufferSize
)
{
  pjsl->dataFile = fopen(name, "rb+");
  if (pjsl->dataFile)
  {
    fread(&(pjsl->firstAvail), sizeof(pjsl->firstAvail), 1, pjsl->dataFile);
    return pjsl_ok;
  }
  else
  {
    return pjsl_error;
  }
}

pjsl_err_t pjsl_close
(
  pjsl_t* pjsl
)
{
  fseek(pjsl->dataFile, 0, SEEK_SET);
  fwrite(&(pjsl->firstAvail), sizeof(pjsl->firstAvail), 1, pjsl->dataFile);
  fclose(pjsl->dataFile);
  pjsl->dataFile = NULL;
  pjsl->firstAvail = 0;
  return pjsl_ok;
}

pjsl_err_t pjsl_objInspect
(
  pjsl_t* pjsl,
  pjsl_pid_t pid,
  pjsl_kind_t* kind,
  pjsl_regime_t* reg,
  pjsl_size_t* size
)
{
  int32 objsize;

  fseek(pjsl->dataFile, -pid, SEEK_SET);
  fread(&objsize, sizeof(int), 1, pjsl->dataFile);
  *size = objsize;
  return pjsl_ok;
}

pjsl_err_t pjsl_setPersistentRoot
(
  pjsl_t* pjsl,
  pjsl_pid_t pid
)
{
  fseek(pjsl->dataFile, sizeof(pjsl->firstAvail), SEEK_SET);
  fwrite(&pid, sizeof(pjsl_pid_t), 1, pjsl->dataFile);
  return pjsl_ok;
}

pjsl_err_t pjsl_getPersistentRoot
(
  pjsl_t* pjsl,
  pjsl_pid_t* pid
)
{
  fseek(pjsl->dataFile, sizeof(pjsl->firstAvail), SEEK_SET);
  fread(pid, sizeof(pjsl_pid_t), 1, pjsl->dataFile);
  return pjsl_ok;
}

#endif



#endif


