/*
 * Copyright (C) 2001, John Leuner.
 *
 * This file is part of the kissme/teaseme project, which in turn is part of the JOS project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2,
 * or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

 #include <config.h>

/* 
   May 2000		      - Modified GC to work with native threads

   Changes
   April / May  2000	      - Wrote code that checks in the invokevirtual_quick, invokevirtual_quick_optimised instructions if it can run an optimised version of the java code.
			      - This involves running an analyser on the code to see if it can be optimised (SVETLANA_CanOptimiseMethod) and creating a compiled version if it can (SVETLANA_createOptimisedMethod). This involves setting the flags ACC_OPTIMISED and ACC_CANNOT_OPTIMISE in the pstMethod->u16AccessFlags. 
			      - This is all disabled if the ENABLE_SVETLANA define is not defined.
			      
Tue Jan 25 04:51:32 SAST 2000 - Started implementation of native threads - Jewel
			        Created threads_native.c for a GNU/Linux pthreads implementation
				Retained thread.c for the old "green" threads emulated threads
				Currently can run programs with green, including gc
				native threads doesn't function for more than a single thread
				gc doesn't work for native threads at all

				Changes include interp.c, garbage.c, jni.c
				Implemented THREAD_SynchroniseEnter|Exit in threads_native.c

Mon Jan 24 23:59:35 SAST 2000 - Removed all references to __WIN32__ #ifdefs - Jewel
				Removed all references to __PC_TARGET__ #ifdefs, also in global.h  - Jewel
				Removed all references to LONGNAMES #ifdefs  - Jewel
				Removed all references to CL_OPTIONS #ifdefs, also in global.c and global.h  - Jewel
 */


/*
% * @doc MODULE
 * @module interp.c |
 *
 * This file contains the core of the virtual machine.  It contains the
 * Interpret function that executes the bytecode.  It also contains the
 * functions for manipulating the environment stack.
 *
 */

/*
 * @doc TOPIC
 * @topic Interpreter |
 *
 * The interpreter is what executes the bytecode. Functions in other modules
 * are called as the interpreter requires them.
 *
 * Before the Interpret() function is called, all the classes that are
 * required should be loaded with calls to LoadClass(). An environment stack
 * frame must be created in order to pass to Interpret(). This environment
 * frame needs to be created by InitialStackFrame(). The method to be executed
 * has to be found before InitialStackFrame() is called.
 *
 * The interpreter has to be told whether to execute in single-threaded or
 * multi-threaded mode. Multi-threaded mode is the usual mode of execution
 * but single-threaded mode is needed for class initialisations and for
 * Java methods that are called from native methods.
 *
 * @ex Example of interpreter invocation. This example shows the use for
 *     multi-threaded mode. A java.lang.Thread object has to be created and
 *     initialised for the main method. |
 *
 * tClass*      pstClass;
 * tMethod*     pstMethod;
 * tStackFrame* pstFrame;
 * tOBREF     pstMainThread;
 * int32*       i32ArgV;
 *
 * / * Load classes * /
 * LoadClass("classes/java/lang/Object");
 * LoadClass("classes/java/lang/Thread");
 * LoadClass("classes/java/lang/String");
 * LoadClass("classes/mine/MyClass");
 *
 * / * Find main class * /
 * pstClass =  CPLIST_Find("MyClass");
 * / * Find main method in main class * /
 * pstMethod = CLASSFILE_FIndMethod(pstClass, "main", "([Ljava/lang/String;)V");
 * / * Create an environment frame for this method * /
 * pstFrame = InitialStackFrame(pstMethod, NULL, i32ArgV);
 * / * Create a thread for the main method * /
 * pstMainThread = INTERP_NewObjectFromName("java/lang/Thread");
 * / * Initialise the thread for use with this frame * /
 * THREAD_InitMainThread(pstMainThread, 5, pstFrame);
 * / * Call the interpreter in multi-threaded mode * /
 * Interpret(pstFrame, 1);
 * / * Dispose of stack frame afterwards * /
 * PopStackFrame(pstFrame);
 *
 */


#ifdef KISSME_LINUX_USER
#include <assert.h>
#include <math.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <signal.h>
#endif

#ifdef TIMING
#include <time.h>
#endif


#include "wrappers.h"
#include "uid.h"

#include "interp.h"
#include "classfil.h"
#include "jni.h"
#include "interp_methods.h"
#include "classfile_methods.h"
#include "opcodes.h"
#include "stdtypes.h"
#include "disass.h"

#include "cplist.h"
#include "arith64.h"
#include "jutils.h"

#include "threadex.h"
#include "threads_native.h"
#include "garbage2.h"
#include "newobject.h"
#include "methodstacks.h"

#include "locks.h"
#include "lib/indigenous/java.lang/Class.h"
#include "lib/indigenous/java.lang/Class_Reflection.h"

#ifdef ENABLE_SVETLANA
#include "svetlana/analyse.h"
#include "svetlana/assemble.h"
#endif

#ifdef VERIFIER
#include "verifier/verifier.h"
#endif

#ifdef GARBAGE
#include "garbage.h"
#endif

#include "global.h"

#ifdef PERSIST
#include "rpot.h"
#endif

#ifdef TEASEME
#include "../lib/indigenous/jos.system/processes.h"
#endif

/* local prototypes ********************************************************/

#ifdef DEBUG_TRACE
void StackInspect(tStackFrame*, int32*);
#endif

/* global variables ********************************************************/

#ifdef DEBUG
 int INTERP_DebugOn=0;
#endif

int virtualcalls = 0; 
extern int canStartOptimising;
extern tThreadNode* pstThreadList;

extern tOBREF OOMExceptionObject;
extern tAllocatorHeap* allocHeap; //just for debugging

extern tClassLoaderTuple* pstSystemType;

#ifdef DEBUG_CLASS_FOR_GC
int instructions_since_gc = 0; //for debugging, records the number of interpreter iterations since last GC
#endif

/* This function prints a back trace from the given Java frame. Used for debugging.
 *
 *
 *
 */

void INTERP_printBackTrace(tStackFrame* pstCurrFrame)
{
  tStackFrame* iter = pstCurrFrame;

  while(iter->pstPrevFrame != NULL)
    iter = iter->pstPrevFrame;

  if(strcmp(iter->pstCurrMethod->uidName, "intern") == 0)
    return ;
  if(strcmp(iter->pstCurrMethod->uidName, "get") == 0)
    return ;
  if(strcmp(iter->pstCurrMethod->uidName, "hashCode") == 0)
    return ;
  if(strcmp(iter->pstCurrMethod->uidName, "equals") == 0)
    return ;


  eprintf("/////////////////// Stack Trace (%p)\n", pstCurrFrame);
  while(iter != NULL)
    {
      eprintf(" %s.%s(%s) %i\n",  iter->pstCurrMethod->pstClass->uidName, iter->pstCurrMethod->uidName, iter->pstCurrMethod->uidSignature, INTERP_FigureOutLineNumber(iter));
      iter = iter->pstChildFrame;
    }
  eprintf("\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\ \n");
}

//#ifdef DEBUG_TRACE
/*
 * @doc FUNC
 * @func
 * This function is used to display the contents of the operand stack for
 * debugging purposes.  It is only compiled when DEBUG_TRACE is defined. A
 * pointer to the top of the operand stack has to be passed to it since the
 * operand stack pointer in the environment frame that is passed is unlikely
 * to be up to date unless SAVEENV() has been called recently.
 * 
 */

void StackInspect
(
  tStackFrame* pstCurrFrame, /* @parm Pointer to current environment frame */
  int32 *optop               /* @parm Pointer to top of operand stack */
)
{
	int32* temp;
	JNIEnv* env;
#ifdef DEBUG_FRAMES
  tStackFrame* checkFrame = pstCurrFrame;
  while(checkFrame)
    {
      assert(checkFrame);
      assert(checkFrame->pstCurrClass);
      assert(checkFrame->pstCurrMethod);
      assert(checkFrame->pstCurrClass->pstClass->pu32ConstPool);
      assert(checkFrame->pstCurrMethod->uidName);
      assert(checkFrame->pstHeap);
      checkFrame = checkFrame->pstPrevFrame;
    }
#endif

	env = & JNI_getJNIData( sys_current_thread() )->functions;
	assert( pstCurrFrame->pi32OpTop >= pstCurrFrame->pi32OpBot);
	eprintf( "\t\t-- Stack: -- ");
	eprintf( "\"%s.", pstCurrFrame->pstCurrClass->uidName);
	eprintf( "%s", pstCurrFrame->pstCurrMethod->uidName);
	eprintf( "%s\"\n", pstCurrFrame->pstCurrMethod->uidSignature);

	eprintf( "Curr stack size %i words, max stack %i, vars %i \n", optop - pstCurrFrame->pi32OpBot, pstCurrFrame->pstCurrMethod->pstCode->u16MaxStack + 1, pstCurrFrame->pstCurrMethod->pstCode->u16MaxLocals);
	
	    {
		int lv = 0;
	if(pstCurrFrame->pi32Vars)
		for(;lv < (pstCurrFrame->pstCurrMethod->pstCode->u16MaxLocals);lv++)
		    {
			if(lv < pstCurrFrame->pstCurrMethod->u16ArgSize)
			    eprintf("A");
			else
			    eprintf( "V");

			if( GARBAGE_InHeap(env, (tOBREF) pstCurrFrame->pi32Vars[lv]) == 1)
			    {

			      eprintf("\t\t%x %i = %x (%s)\n", (int) pstCurrFrame->pi32Vars + lv,  (int) pstCurrFrame->pi32Vars[lv], (int) pstCurrFrame->pi32Vars[lv], ODEREF((tOBREF) pstCurrFrame->pi32Vars[lv])->pstType->uidName);
			    }
			else
			  eprintf("\t\t%x %i = %x\n", (int) pstCurrFrame->pi32Vars + lv,  (int) pstCurrFrame->pi32Vars[lv], (int) pstCurrFrame->pi32Vars[lv]);

		    }
		       
for(temp = (pstCurrFrame->pi32OpBot + 1); temp != (optop + 1); temp++)
  {
    eprintf("O\t\t%p %i = %x\n", temp, (int) (*temp), (int) (*temp));
  }
  eprintf("\t\t\t------------\n");

	    }
}
//#endif

/* the interpreter procedure ***********************************************/

/*
 * @doc FUNC
 * @func
 * This is the procedure that executes the Java bytecode.  It takes the
 * method to execute and all other environment information from the environment
 * stack frame pointer that is passed to it.  It also needs to be told whether
 * to execute in multi-threaded mode or not.  Single-thread mode is used when
 * initialising classes or when the interpreter is called back from a native
 * function.
 *
 * The environment stack frame that is passed to Interpret() must have been
 * created beforehand by a call to InitialStackFrame().
 *
 *
 *
 * @ex This example shows how Interpret() would be called. i32ArgV will need
 *     to be set to point to the parameter (in this case of main, an array
 *     of strings) before passing it |
 *
 * @rdesc Returns NULL for normal exit, or a pointer to a an exception object
 *        if it exits due to an uncaught exception.
 *
 * tClass*      pstClass;
 * tMethod*     pstMethod;
 * tStackFrame* pstFrame;
 * int32*       i32ArgV;
 *
 * pstClass = CPLIST_Find("MyClass");
 * pstMethod = CLASSFILE_FIndMethod(pstClass, "main", "([Ljava/lang/String;)V");
 * pstFrame = InitialStackFrame(pstMethod, NULL, i32ArgV);
 * Interpret(pstFrame, 0);
 * PopStackFrame(pstFrame);
 *
 */

/* This hack helps us with calling interpret from a thread , Why though?, I think Interpret was static */
tOBREF INTERP_Interpret(JNIEnv* env, void* arg1, int arg2)
{
  return Interpret(env, arg1, arg2);
}

tOBREF Interpret
(
 JNIEnv* env,
 tStackFrame* pstInitFrame,  /* @parm Initial environment frame */
 int          iMultiThread   /* @parm 1 for multithread, 0 for single thread */
)
{
  /* the registers */
  register byte* 		pbPC;  //Pointer to the bytecode currently being interpreted
  register int32*		optop; //Pointer to the top of the java stack
  register int32*		pi32Vars; //Pointer to local variables for this method
  register tClassLoaderTuple*  pstCurrClass;	  //Pointer to the class structure (whose method we are executing)
					      
  tStackFrame* pstCurrFrame = pstInitFrame;
  assert(iMultiThread);
  assert( pstCurrFrame->pstCurrClass->pstClass->persIndex == 0);

  /* initialise the registers */
  LOADENV();

  /* if the first method is native then run it */
  if (pstCurrFrame->pstCurrMethod->u16AccessFlags & ACC_NATIVE)
  {
    int32 i32NativeRet;
    tMethod* pstMethodTemp;

    pstMethodTemp = pstCurrFrame->pstCurrMethod;

    if (!(pstCurrFrame->pstCurrMethod->u16AccessFlags & ACC_STATIC))
      {
	if ((tOBREF) *(optop - pstMethodTemp->u16ArgSize) == NULL) //minus a + 1 for the native case
	  {
	    return INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/NullPointerException", "null pointer encountered when invoking native method");
	  }
	pstMethodTemp = DEREF((tOBREF) *(optop - pstMethodTemp->u16ArgSize))->pstType->pstClass->ppstVT[pstMethodTemp->u16VTIndex];
	
	SAVEENV();
	pstCurrFrame->pi32OpTop -= pstMethodTemp->u16ArgSize; //XXX check that this works for static native methods and for methods with arguments, jewel, July 2001
	
	i32NativeRet = JNI_CallNativeMethod(pstCurrFrame, pstMethodTemp);
	if (i32NativeRet == -1)
	  {
	    return pstCurrFrame->pstException;
	  }
	else
	  {
	    LOADENV();
	    return NULL;
	  }
      }
    else
      {
	    SAVEENV();
	    i32NativeRet = JNI_CallNativeMethod(pstCurrFrame, pstMethodTemp);
	    if (i32NativeRet == -1)
		{
		  return pstCurrFrame->pstException;
		}
	    LOADENV();
	    return NULL;
      }
  }

  LabelLoopTop:
  while (1)
  {

#ifdef DEBUG_FRAMES
  assert(pstCurrFrame);
  assert(pstCurrFrame->pstCurrClass);
  assert(pstCurrFrame->pstCurrMethod);
  assert(pstCurrFrame->pstCurrClass->pstClass->pu32ConstPool);
  assert(pstCurrFrame->pstCurrMethod->uidName);
  assert((((pstCurrFrame->pstCurrClass->uidName[0] >= 'a') && (pstCurrFrame->pstCurrClass->uidName[0] <= 'z')) || ((pstCurrFrame->pstCurrClass->uidName[0] >= 'A') && (pstCurrFrame->pstCurrClass->uidName[0] <= 'Z'))  || (pstCurrFrame->pstCurrClass->uidName[0] == '[')));
  assert( pstCurrFrame->pstCurrClass->pstClass->persIndex == 0);
#endif
#ifdef DEBUG_CLASS_FOR_GC
  assert(((byte*)pstCurrFrame->pstCurrClass)   >= CLASSFILE_getClassStructHeap());
  assert( ((byte*) pstCurrFrame->pstCurrClass) - CLASSFILE_getClassStructHeap() <= CLASSFILE_getBytesUsedForClassHeaders());
  assert(pstCurrFrame->pstCurrClass->magic1 == MAGIC1);
  assert(pstCurrFrame->pstCurrClass->magic2 == MAGIC2);
  assert(pstCurrFrame->pstCurrClass->magic3 == MAGIC3);
  instructions_since_gc++;

#endif

   #ifdef DEBUG_TRACE
   if (INTERP_DebugOn)
   {
   eprintf("%s.%s %i: ", pstCurrFrame->pstCurrClass->uidName, pstCurrFrame->pstCurrMethod->uidName, (int) (pbPC - pstCurrFrame->pbCode));
   DISASS_PrintInstruction(pbPC, pstCurrFrame->pbCode);
   }
    #endif

   //We take action depending on bytecode instruction
  switch (*pbPC++)
    {
      case nop:
      {
        break;
      }
      #include "constants.h" //I decided to put this in a separate file just to make it more manageable

      case bipush:
	{
		/* #### better to do this by casting. */
        if ((*pbPC) & 0x80)   /* to see if it's negative */
        {
          *(++optop) = ((int) (*pbPC++)) - 256;
        }
        else
        {
          *(++optop) = (*pbPC++);
        }
        break;
      }
      case sipush:
      {
        if ((*pbPC) & 0x80)
        {
          *(++optop) = (*pbPC++) - 256;
        }
        else
        {
          *(++optop) = (*pbPC++);
        }
        *optop *= 256;
        *optop += (*pbPC++);
        break;
      }
    case ldc1:
      {
	int16        i16Temp;
	Uid          uidBytes;
	
	i16Temp = (*pbPC++);
	
	switch (CONSTTAG(pstCurrClass->pstClass, i16Temp))
	  {
	  case CONSTANT_Integer:
	    {
	      *(++optop) = CONSTGET_Int(pstCurrClass->pstClass, i16Temp);
	      break;
	    }
	  case CONSTANT_Float:
	    {
#ifdef DEBUG_FLOAT
				fprintf(stderr, "Processing ldc %x, index is %i\n", *(pbPC - 2), i16Temp);
#endif
#ifdef DEBUG_FLOAT
				fprintf(stderr, "Getting float %G\n", CONSTGET_Float(pstCurrClass, i16Temp));
				fprintf(stderr, "Class is %s, index %i %G\n", pstCurrClass->uidName, i16Temp, ((float*)pstCurrClass->pu32ConstPool)[i16Temp]);
#endif
				*((float*) (++optop)) = CONSTGET_Float(pstCurrClass->pstClass, i16Temp);
				break;
	    }
	  case CONSTANT_String:
	    {
	      /* all strings are stored as Asciz strings in UTF8 format */
	      uidBytes = CONSTGET_UidAsciz(pstCurrClass->pstClass, CONSTGET_String_Index(pstCurrClass->pstClass, i16Temp));
	      *(++optop) = (int32) INTERP_NewStringFromAsciz(env, uidBytes);
	      if(*optop == 0)
		{
		  pstCurrFrame = ThrowNamedException(env, pstCurrFrame, "java/lang/NullPointerException");
		}
	      break;
	    }
	  default:
	    {
	      eprintf("Interpreter error in ldc1 instruction. pbPC = %u\n", pbPC - 1 - pstCurrFrame->pstCurrMethod->pstCode->pbCode);
	      exit(1);
	    }
	  }
	break;
      }
    case ldc2:
      {
	int16        i16Temp;
	Uid           uidBytes;
	
	i16Temp = (*pbPC++) * 256;
	i16Temp += (*pbPC++);
#ifdef DEBUG_FLOAT
	fprintf(stderr, "Processing ldc %x, index is %i\n", *(pbPC - 2), i16Temp);				
#endif
	switch (CONSTTAG(pstCurrClass->pstClass, i16Temp))
	  {
	  case CONSTANT_Integer:
	    {
	      *(++optop) = CONSTGET_Int(pstCurrClass->pstClass, i16Temp);
	      break;
	    }
	  case CONSTANT_Float:
	    {
	      *(++optop) = CONSTGET_Float(pstCurrClass->pstClass, i16Temp);
	      break;
	    }
	  case CONSTANT_String:
	    {
	      /* all strings are stored as Asciz strings in UTF8 format */
	      uidBytes = CONSTGET_UidAsciz(pstCurrClass->pstClass, CONSTGET_String_Index(pstCurrClass->pstClass, i16Temp));
	      *(++optop) = (int32) INTERP_NewStringFromAsciz(env, uidBytes);
	      break;
	    }
	  default:
	    {
	      eprintf( "Interpreter error in ldc2 instruction. pbPC = %u\n", pbPC - 2 - pstCurrFrame->pstCurrMethod->pstCode->pbCode);
	      exit(1);
	    }
	  }
	break;
      } 
    case ldc2w:
      {
	int16        i16Temp;
	
	i16Temp = (*pbPC++) * 256;
	i16Temp += (*pbPC++);
	
	switch (CONSTTAG(pstCurrClass->pstClass, i16Temp))
	  {
	  case CONSTANT_Long:
	    {
//      eprintf("Loading long constant: hi %i lo %i (index %i) \n", CONSTGET_Long_High(pstCurrClass, i16Temp), CONSTGET_Long_Low(pstCurrClass, i16Temp), i16Temp);
	      *(++optop) = CONSTGET_Long_High(pstCurrClass->pstClass, i16Temp);
	      *(++optop) = CONSTGET_Long_Low(pstCurrClass->pstClass, i16Temp);
//      eprintf("Loaded long constant: %lli \n", *((long long*) (optop - 1)));
	      break;
	    }
	  case CONSTANT_Double:
	    {
	      *(++optop) = CONSTGET_Double_High(pstCurrClass->pstClass, i16Temp);
	      *(++optop) = CONSTGET_Double_Low(pstCurrClass->pstClass, i16Temp);
//      eprintf("Loaded double constant: %G \n", *((double*) (optop - 1)));
	      break;
	    }
	  default:
	    {
		eprintf( "Interpreter error in ldc2w instruction. pbPC = %u\n", pbPC - 2 - pstCurrFrame->pstCurrMethod->pstCode->pbCode);
		return NULL;
	    }
	  }
	  break;
      }



      #include "load_and_store.h" //All the load and store instructions are here
      //Also the array load and store instructions

      /* Misc stack manipulation */
		case pop:
		{
		  optop--;
		  break;
		}
		case pop2:
		{
		  optop -= 2;
		  break;
		}
		case dup_:
		{
		  optop++;
		  *optop = *(optop - 1);
		  break;
		}
		case dup_x1:
		{
		  optop++;
		  *optop = *(optop - 1);
		  *(optop - 1) = *(optop - 2);
		  *(optop - 2) = *optop;
		  break;
		}
		case dup_x2:
		{
		  optop++;
		  *optop = *(optop - 1);
		  *(optop - 1) = *(optop - 2);
		  *(optop - 2) = *(optop - 3);
		  *(optop - 3) = *optop;
		  break;
		}
		case dup2_:
	      {
				optop += 2;
		*optop = *(optop - 2);
	        *(optop - 1) = *(optop - 3);
	        break;
	      }
      case dup2_x1:
      {
        optop += 2;
        *optop = *(optop - 2);
        *(optop - 1) = *(optop - 3);
        *(optop - 2) = *(optop - 4);
        *(optop - 3) = *optop;
        *(optop - 4) = *(optop - 1);
        break;
      }
      case dup2_x2:
      {
        optop += 2;
        *optop = *(optop - 2);
        *(optop - 1) = *(optop - 3);
        *(optop - 2) = *(optop - 4);
        *(optop - 3) = *(optop - 5);
        *(optop - 4) = *optop;
        *(optop - 5) = *(optop - 1);
        break;
      }
      case swap:
      {
	int32         i32Temp;
	i32Temp = *optop;
        *optop = *(optop - 1);
        *(optop - 1) = i32Temp;
	break;
      }
#include "arithmetic_and_logic.h"
#include "primitive_conversion.h"

#include "comparison_and_branches.h"

		case lookupswitch:
		{
			int32 i32Temp;
			int32 i;
			int32 i32DefaultOff;
			int32 i32NoPairs;
      byte* pbStartPC;

      /* remember PC at beginning of instruction */
      pbStartPC = pbPC - 1;

      /* skip 0 to 3 bytes of padding */
			while ((pbPC-pstCurrFrame->pbCode) % 4)
			{
			  pbPC++;
			}

      i32DefaultOff = (((int32)pbPC[0]) << 24) | (((int32)pbPC[1]) << 16) | (((int32)pbPC[2]) << 8) | pbPC[3];
      i32NoPairs = (((int32)pbPC[4]) << 24) | (((int32)pbPC[5]) << 16) | (((int32)pbPC[6]) << 8) | pbPC[7];
      pbPC += 8;

			for (i = 0; i < i32NoPairs; i++)
			{
			 /* read in number to compare */
       i32Temp = (((int32)pbPC[0]) << 24) | (((int32)pbPC[1]) << 16) | (((int32)pbPC[2]) << 8) | pbPC[3];
       pbPC += 4;

       /* compare number to key */
       if (i32Temp == *optop)
	 {
	   /* re-use i32Temp and set it to the offset */
	   i32Temp = (((int32)pbPC[0]) << 24) | (((int32)pbPC[1]) << 16) | (((int32)pbPC[2]) << 8) | pbPC[3];
	   optop--;
	   pbPC = pbStartPC + i32Temp;
	   goto labelLookupEnd;
	 }
       else
	 {
	   pbPC += 4;
	 }
			}
			
			/* will get here if there was no match */
			optop--;
			pbPC = pbStartPC + i32DefaultOff;
		labelLookupEnd:
			break;
		}
#include "method_return.h"
#include "get_and_put.h"
#include "method_invocation.h"
#include "optimised_method_invocation.h"

    case newfromname:
      {
	tClassLoaderTuple*       pstClassTemp;
	tOBREF      pstObjectTemp;
	char*         pszAsciz;
	
	if (*optop)
	  {
	    pszAsciz = INTERP_AscizFromString(env, (tOBREF) *optop);
	    
#ifdef SHOWLOADING 
	    eprintf("newfromname %s\n", pszAsciz);
#endif
	    
	    pstClassTemp = CLASSFILE_FindOrLoad(env, pszAsciz, pstCurrFrame->pstCurrClass);
	    sys_free(pszAsciz);
	    if (pstClassTemp)
	      {
		pstObjectTemp = INTERP_NewObject(env, pstClassTemp);
		if (pstObjectTemp)
		  {
		    *(++optop) = (int32) pstObjectTemp;
		  }
		else
		  {
		    SAVEENV();
		    pstCurrFrame = ThrowNamedException(env, pstCurrFrame, "java/lang/OutOfMemoryError");
		    LOADENV();
		  }
	      }
	    else
	      {
		SAVEENV();
		pstCurrFrame = ThrowNamedException(env, pstCurrFrame, "java/lang/ClassNotFoundException");
		LOADENV();
	      }
	  }
	else
	  {
	    SAVEENV();
	    pstCurrFrame = ThrowNamedException(env, pstCurrFrame, "java/lang/NullPointerException");
	    LOADENV();
	  }
        break;
      }
    
    case new_:
      {
	int16        i16Temp;
	tClassLoaderTuple*       pstClassTemp;
	tOBREF      pstObjectTemp;
	
	i16Temp = (*pbPC++) * 256;
	i16Temp += (*pbPC++);
	
#ifdef DEBUG
	assert(CONSTTAG(pstCurrClass->pstClass, i16Temp) == CONSTANT_Class);
#endif
	
        pstClassTemp = CONSTGET_Class(env,pstCurrClass, i16Temp);
	if (pstClassTemp)
	  {
#ifdef RESURRECT_QUICK
	    /* overwrite operation with new_quick */
	    pbPC[-3] = new_quick;
	    /* constant pool item already overwritten with resolved version */
	    /*          CONSTSET(env,pstCurrClass, i16Temp, pstClassTemp);*/
#endif
	    
	    pstObjectTemp = INTERP_NewObject(env, pstClassTemp);
	    if (pstObjectTemp)
	      {
		*(++optop) = (int32) pstObjectTemp;
	      }
	    else
	      {
		SAVEENV();
		pstCurrFrame = ThrowOOMException(env,pstCurrFrame);
		LOADENV();
	      }
	  }
	else
	  {
	    SAVEENV();
	    pstCurrFrame = ThrowNamedException(env, pstCurrFrame, "java/lang/ClassNotFoundException");
	    LOADENV();
	  }
	break;
      }
    
    //      #ifdef QUICK
	       case new_quick:
		 {
				int16        i16Temp;
				tClassLoaderTuple*       pstClassTemp;
				tOBREF      pstObjectTemp;

				i16Temp = (*pbPC++) * 256;
				i16Temp += (*pbPC++);

				#ifdef DEBUG
				assert(CONSTTAG(pstCurrClass->pstClass, i16Temp) == CONSTANT_Class);
				#endif

        pstClassTemp = (tClassLoaderTuple*) CONSTGET(pstCurrClass->pstClass, i16Temp);
        pstObjectTemp = INTERP_NewObject(env, pstClassTemp);
        if (pstObjectTemp)
        {
          *(++optop) = (int32) pstObjectTemp;
        }
        else
        {
          SAVEENV();
          pstCurrFrame = ThrowOOMException(env,pstCurrFrame);
          LOADENV();
        }
        break;
      }
      //      #endif /* QUICK */

      case newarray:
      {
	  //Stack holds number of elements to allocate
        if (*optop >= 0)
	    { //extra byte holds type of element
          *optop = (int32) INTERP_NewArray(env, *pbPC++, *optop, NULL, pstCurrFrame->pstCurrClass);
        }
        else
        {
	  pbPC++; //just to make sure
          SAVEENV();
          pstCurrFrame = ThrowNamedException(env, pstCurrFrame, "java/lang/NegativeArraySizeException");
          LOADENV();
        }
        break;
      }
			case anewarray: /* what if size = 0? */
			{
				int16         i16Temp;
				tClassLoaderTuple*       pstClassTemp;

				i16Temp = (*pbPC++) * 256;
				i16Temp += (*pbPC++);

				#ifdef DEBUG
				assert(CONSTTAG(pstCurrClass->pstClass, i16Temp) == CONSTANT_Class);
				#endif

				if (*optop >= 0)
				{
					pstClassTemp = CONSTGET_Class(env,pstCurrClass, i16Temp);
          if (pstClassTemp)
          {

            #ifdef QUICK
            /* overwrite operation with new_quick */
            pbPC[-3] = anewarray_quick;
            /* constant pool item already overwitten with resolved version */
/*            CONSTSET(pstCurrClass, i16Temp, pstClassTemp);*/
            #endif

            *optop = (int32) INTERP_NewArray(env, T_OBJECT, *optop, pstClassTemp->uidName, pstCurrFrame->pstCurrClass);
          }
					else
          {
            SAVEENV();
            pstCurrFrame = ThrowNamedException(env, pstCurrFrame, "java/lang/ClassNotFoundException");
            LOADENV();
          }
        }
        else
        {
          SAVEENV();
          pstCurrFrame = ThrowNamedException(env, pstCurrFrame, "java/lang/NegativeArraySizeException");
          LOADENV();
				}
        break;
      }

      #ifdef QUICK
			case anewarray_quick: /* what if size = 0? */
			{
				int16        i16Temp;
				tClassLoaderTuple*       pstClassTemp;

				i16Temp = (*pbPC++) * 256;
				i16Temp += (*pbPC++);

				#ifdef DEBUG
				assert(CONSTTAG(pstCurrClass->pstClass, i16Temp) == CONSTANT_Class);
				#endif

				if (*optop >= 0)
				{
          pstClassTemp = (tClassLoaderTuple*) CONSTGET(env,pstCurrClass, i16Temp);
          if (pstClassTemp)
          {
            *optop = (int32) INTERP_NewArray(env, T_OBJECT, *optop,  pstClassTemp->uidName);
          }
					else
          {
            SAVEENV();
            pstCurrFrame = ThrowNamedException(env, pstCurrFrame, "java/lang/ClassNotFoundException");
            LOADENV();
          }
        }
        else
        {
          SAVEENV();
          pstCurrFrame = ThrowNamedException(env, pstCurrFrame, "java/lang/NegativeArraySizeException");
          LOADENV();
				}

        break;
      }
      #endif /* QUICK */

      case arraylength:
      {
        if (*optop)
        {
          *optop = ADEREF((tARREF) *optop)->i32Number;
	  assert(*optop >= 0);
        }
        else
        {
          SAVEENV();
          pstCurrFrame = ThrowNamedException(env, pstCurrFrame, "java/lang/NullPointerException");
          LOADENV();
        }
        break;
      }
      case athrow:
      {
        if (pstCurrFrame->pstException || pstCurrFrame->bIsOrphan)
        {
          #ifdef DEBUG
          assert (pstCurrFrame->pstPrevFrame == NULL);
          #endif
	  if(pstCurrFrame->bIsOrphan)
	      return (tOBREF) *optop; //The exception is on the stack?

          return pstCurrFrame->pstException;
        }
        if (*optop)
	  {
	    tOBREF thrownObject = (tOBREF) *optop;
	    tClassLoaderTuple* pstThrowable = CLASSFILE_FindOrLoad(env, "java/lang/Throwable", NULL);
	    //Check that it is Throwable
	    if(INTERP_CheckCast(env, pstThrowable, ODEREF(thrownObject)->pstType) == 1)
	      {
		SAVEENV();
		pstCurrFrame = CatchException(env,pstCurrFrame);
		LOADENV();
	      }
	    else
	      {
		SAVEENV();
		pstCurrFrame = ThrowNamedException(env, pstCurrFrame, "java/lang/RuntimeException");
		LOADENV();
	      }
	  }
        else
	  {
	    SAVEENV();
	    pstCurrFrame = ThrowNamedException(env, pstCurrFrame, "java/lang/NullPointerException");
	    LOADENV();
	  }
        break;
      }
    case checkcast:
      {
	int16         i16Temp;
	tClassLoaderTuple*       pstClassTemp;
	tClassLoaderTuple*       pstTempClass1;
	
        /* check if optop is NULL... a NULL class can be cast to anything */
        if (*optop)
	  {
	    i16Temp = (*pbPC++) * 256;
	    i16Temp += (*pbPC++);
	    
#ifdef DEBUG
	    assert(CONSTTAG(pstCurrClass->pstClass, i16Temp) == CONSTANT_Class);
#endif
	    
	    pstTempClass1 = CONSTGET_Class(env,pstCurrClass, i16Temp);
	    
#ifdef QUICK
	    /* overwrite operation with checkcast_quick */
	    pbPC[-3] = checkcast_quick;
	    /* constant pool item is already overwritten with resolved version */
	    /*          CONSTSET(pstCurrClass, i16Temp, pstTempClass1);*/
#endif
	    
	    pstClassTemp = DEREF((tOBREF) *optop)->pstType;
	    
	    if (INTERP_CheckCast(env, pstTempClass1,pstClassTemp) == 0)
	      {
		char* errMessage;
		optop--;
		SAVEENV();
		errMessage = sys_malloc(1024 + strlen(pstClassTemp->uidName) + strlen(pstTempClass1->uidName));
		if(errMessage == NULL) { pstCurrFrame = CatchExternalException(env, pstCurrFrame, OOMExceptionObject); break; }

		sprintf(errMessage, "Could not cast object to type %s from type %s\n", pstTempClass1->uidName, pstClassTemp->uidName);
#ifdef DEBUG
		eprintf( "Throwing ClassCastException in checkcast. to %s from %s\n", pstTempClass1->uidName, pstClassTemp->uidName);
#endif
		pstCurrFrame = ThrowNamedExceptionWithMessage(env, pstCurrFrame, "java/lang/ClassCastException", errMessage);
		sys_free(errMessage);
		LOADENV();
	      }
	  }
	else
	  {
	    pbPC += 2;  /* skip the two index bytes */
	  }
	break;
      }

      #ifdef QUICK
      case checkcast_quick:
      {
				int16        i16Temp;
				tClassLoaderTuple*       pstClassTemp;
				tClassLoaderTuple*       pstTempClass1;

        if (*optop)
				{
					i16Temp = (*pbPC++) * 256;
				        i16Temp += (*pbPC++);

				       #ifdef DEBUG
					assert(CONSTTAG(pstCurrClass->pstClass, i16Temp) == CONSTANT_Class);
					#endif

					pstTempClass1 = (tClass*) CONSTGET(env,pstCurrClass, i16Temp);

					//Jewel sep
					pstClassTemp = DEREF(((tOBREF) *optop))->pstType;
					if (INTERP_CheckCast(pstTempClass1,pstClassTemp) == 0)
					{
							optop--;
					              SAVEENV();
					              pstCurrFrame = ThrowNamedException(pstCurrFrame, "java/lang/ClassCastException");
					              LOADENV();
					}
				}
				else
				{
					pbPC += 2;  /* skip the two index bytes */
				}
				break;
      }
      #endif /* QUICK */

      case instanceof:
      {
	int16         i16Temp;
	uint16 u16Temp;
	tClassLoaderTuple*       pstClassTemp;
        tClassLoaderTuple*       pstTempClass1;

        /* check if optop is NULL... a NULL object CANNOT be an instance of anything */
	if (*optop)
	{
	  i16Temp = (*pbPC++) * 256;
          i16Temp += (*pbPC++);

          pstTempClass1 = CONSTGET_Class(env,pstCurrClass, i16Temp);

  if(strcmp(pstTempClass1->uidName, "de/fub/bytecode/generic/ReferenceType") == 0)
{
  u16Temp = 2;
}
//I deleted the quick overwrite

	  pstClassTemp = DEREF((tOBREF) *optop)->pstType;
          /* check if class can be cast */
	  if (INTERP_CheckCast(env, pstTempClass1, pstClassTemp))
	    {
	      *optop = 1;
	    }
	  else
	    {
	      *optop = 0;
	    }
        }
        else
        {
	  //A null object is NOT an instance of ANYTHING!
	  *optop = 0;
          pbPC += 2;  /* skip the two index bytes */
        }
        break;
      }

//I deleted the instanceof_quick
//jewel April 2001
      case monitorenter:
      {
        if (*optop)
        {
          THREAD_SynchroniseEnter((tOBREF) *(optop--));
        }
        else
        {
          ThrowNamedException(env, pstCurrFrame, "java/lang/NullPointerException");
        }
        break;
      }
      case monitorexit:
      {
        if (*optop)
        {
          THREAD_SynchroniseExit((tOBREF) *(optop--));
        }
        else
        {
          ThrowNamedException(env, pstCurrFrame, "java/lang/NullPointerException");
        }
        break;
      }
    case wide:
      {
	switch(pbPC[0])
	  {
	    int16 constant;
	    int offset;
	  case iinc:

	    //we get a 16-bit unsigned offset
	    offset = (pbPC[1] << 8) | (pbPC[2]);
	    //and a signed 16-bit constant

	    if( pbPC[3] & 0x80)
	      constant = pbPC[3] - 256;
	    else
	      constant = pbPC[3];
	    constant *= 256;
	    constant += pbPC[4];
	    
	    pi32Vars[ offset ] += constant;
	    pbPC += 5;
	    break;
	  default:
	    {
	    eprintf("Unknown opcode %i modified by wide opcode. The interpreter does not handle this yet. Please contact the author of the JVM\n", (int) pbPC[0]);
	    pstCurrFrame = ThrowNamedException(env, pstCurrFrame, "java/lang/InternalError");
	    }
	  }
	break;
      }
      case multianewarray:   
	{
	  int16        i16Temp;
	  int32         i32Temp;
	  Uid           uidBytes;
	  int popoffstack = 0;
				//The type of reference
	  i16Temp = (*pbPC++) * 256;
	  i16Temp += (*pbPC++);
	  
	  
				//How many dimensions
	  i32Temp = (*pbPC++);
	  

				popoffstack = i32Temp  - 1;
				optop -= popoffstack; 

				uidBytes = CONSTGET_Class(env,pstCurrClass, i16Temp)->uidName;
				*(optop) = (int32) MultiNewArray(env, optop, 0, i32Temp, uidBytes, pstCurrFrame->pstCurrClass);
				break;
	}

      /* we get here if we haven't found a valid match for the instruction */
      default:
      {
        eprintf("Unknown bytecode instruction encountered\n");
        eprintf("Instruction number: %d\n", pbPC[-1]);
        eprintf("Current method: %s.%s%s\n", pstCurrClass->uidName, pstCurrFrame->pstCurrMethod->uidName, pstCurrFrame->pstCurrMethod->uidSignature);
        eprintf("Program counter: %u\n", pbPC - pstCurrFrame->pstCurrMethod->pstCode->pbCode - 1);
	assert( 1 == 3);
	StackInspect(NULL, NULL); //this line is just here to get rid of a compiler warning
	return NULL;
      }
    
      }

   #ifdef DEBUG_TRACE
   if (INTERP_DebugOn)
   {
     eprintf( "Optop is at %x vars at %x\n", (int) optop, (int) pi32Vars);
     StackInspect(pstCurrFrame, optop);
   }
   #endif
     }
  }

