/*
 * Copyright (C) 2001, John Leuner.
 *
 * This file is part of the kissme/teaseme project, which in turn is part of the JOS project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2,
 * or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <string.h>

#include "jni.h"
#include "interp.h"
#include "interp_methods.h"
#include "classfile_methods.h"
#include "newobject.h"

#include <assert.h>

extern tOBREF OOMExceptionObject;

jint java_lang_VMSystem_identityHashCode
(
  JNIEnv* env,
  jclass class,
  jobject obj
)
{
  return (jint) obj; /* just use pointer as hash code */
}

void java_lang_VMSystem_kprint(JNIEnv* env, jclass class, jstring str)
{
  char* pszString = INTERP_AscizFromString(env,str); 
  eprintf("%s\n", pszString);
  sys_free(pszString);
}


	/** Get the system properties.
	 ** <dl>
	 ** <dt>java.version         <dd>Java version number
	 ** <dt>java.vendor          <dd>Java vendor specific string
	 ** <dt>java.vendor.url      <dd>Java vendor URL
	 ** <dt>java.home            <dd>Java installation directory
	 ** <dt>java.class.version   <dd>Java class version number
	 ** <dt>java.class.path      <dd>Java classpath
	 ** <dt>os.name              <dd>Operating System Name
	 ** <dt>os.arch              <dd>Operating System Architecture
	 ** <dt>os.version           <dd>Operating System Version
	 ** <dt>file.separator       <dd>File separator ("/" on Unix)
	 ** <dt>path.separator       <dd>Path separator (":" on Unix)
	 ** <dt>line.separator       <dd>Line separator ("\n" on Unix)
	 ** <dt>user.name            <dd>User account name
	 ** <dt>user.home            <dd>User home directory
	 ** <dt>user.dir             <dd>User's current working directory
	 ** </dl>
	 ** It will also define the java.compiler if env(JAVA_COMPILER) is defined.<P>
	 ** 
	 ** <STRONG>Copyright Note:</STRONG> The above text was taken from
	 ** Japhar, by the Hungry Programmers (http://www.japhar.org).
	 **
	 ** @param p the Properties object to insert the system
	 **        properties into.
	 **/

//	static native void insertSystemProperties(Properties p);

static char** command_line_args = NULL;
static int command_line_num_args = 0;

void java_lang_VMSystem_setArgsPointer(char** argv, int argc)
{
  command_line_args = argv;
  command_line_num_args = argc;
}

#include <unistd.h>

void java_lang_VMSystem_insertClassPath(
  JNIEnv* env,
  jclass class,
  jobject prop
)

{
  int32 args[4];
  int i = 0;
  char* strbuf;
  tOBREF Properties = prop;
  tMethod* method;

  char* pathstr = (char*) sys_malloc(4096);

  if(pathstr == NULL)
    {
      (*env)->Throw(env, OOMExceptionObject);
      return ;
    }
  strcpy(pathstr, "java.util.Hashtable");
  method = (*env)->GetMethodID(env, (*env)->FindClass(env, pathstr), "put", "(Ljava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;");

  if(method == NULL)
    {
      (*env)->Throw(env, OOMExceptionObject);
      return ;
    }

  args[0] = (int32)Properties;
  args[1] = (int32)INTERP_NewStringFromAsciz(env, "java.class.path");

  strcpy(pathstr, "");
  for(i = 0; i <  CLASSFILE_u16NumClassPaths; i++)
      {
	  strcat(pathstr, CLASSFILE_ppszClassPaths[i]);
	  if(i != (CLASSFILE_u16NumClassPaths - 1))
	      strcat(pathstr, ":");
      }
  args[2] = (int32)INTERP_NewStringFromAsciz(env, pathstr);

  strbuf = (char*) sys_malloc(4096);
  if(strbuf == NULL)
    {
      (*env)->Throw(env, OOMExceptionObject);
      return ;
    }

  strcpy(strbuf, "java.util.Hashtable");

  INTERP_RunVirtualMethodFromPtr(env,method,args);

  /* We also need to put the current wd */
  #ifdef KISSME_LINUX_USER
  getcwd(pathstr, 4096);
  #else
  strcpy(pathstr, "/usr/share/teaseme/");
  #endif

  args[0] = (int32)Properties;
  args[1] = (int32)INTERP_NewStringFromAsciz(env,"user.dir");
  args[2] = (int32)INTERP_NewStringFromAsciz(env, pathstr);
  strcpy(strbuf, "java.util.Hashtable");

  INTERP_RunVirtualMethodFromPtr(env,method,args);

  //  printf("Buf is now %s\n", strbuf); 

  //Insert the properties passed on the command line with a -D option.
  for(i = 0; i < command_line_num_args;i++)
    {
      if(strncmp(command_line_args[i], "-D", 2) == 0)
	{
	  char* equalIndex = strstr(command_line_args[i], "=");
	    if(equalIndex)
	      {
		*equalIndex = 0;
		args[0] = (int32)Properties;
		args[1] = (int32)INTERP_NewStringFromAsciz(env,command_line_args[i] + 2);
		args[2] = (int32)INTERP_NewStringFromAsciz( env,equalIndex + 1);
		strcpy(strbuf, "java.util.Hashtable");
		INTERP_RunVirtualMethodFromPtr(env,method,args);
	      }
	}
    }
  sys_free(strbuf);
  sys_free(pathstr);
  return;
}

/*
 * @doc NATFUNC
 * @func
 * Implementation of: public static native void arraycopy(Object src,
 *                    int src_position, Object dst, int dst_position,
 *                    int length);
 *
 * @rdesc Returns one of:
 *
 * @flag 0 | Normal return
 * @flag -1 | Return with exception thrown
 *
 */

#include "../../../vm/garbage.h"
#include "Class.h"
#include "Class_Reflection.h"

void java_lang_VMSystem_arraycopy
(
  JNIEnv* env,
  jclass thisclass,
  jarray src,
  jint src_pos,
  jarray dst,
  jint dst_pos,
  jint length
)
{
  tArray* pstA;
  tArray* pstB;
  int iSize;

  if((src == NULL) || (dst == NULL)) {
              (*env)->Throw(env, INTERP_NewObjectFromName(env,"java/lang/NullPointerException"));
              return;
  }

  pstA = ADEREF((tARREF) src);
  pstB = ADEREF((tARREF) dst);

  //Check if they are really arrays

  if( ((pstA->i32Flags & GARBAGE_TYPEMASK) != GARBAGE_ARRAY) ||
      ((pstB->i32Flags & GARBAGE_TYPEMASK) != GARBAGE_ARRAY))
      {
//	eprintf("Arraycopy called with types %s and %s\n", pstA->pstType->uidName, pstB->pstType->uidName);
	      (*env)->Throw(env, INTERP_NewObjectFromName(env,"java/lang/ArrayStoreException"));
	      return;
      }

  //Check their types are the same
  if( pstA->enType != pstB->enType)
      {
	if(!((pstA->enType == T_OBJECT) && (pstB->enType == T_ARRAY)))
	  {
//	eprintf("Arraycopy called with types %s and %s, enType %i and enType %i\n", pstA->pstType->uidName, pstB->pstType->uidName, pstA->enType, pstB->enType);
	  (*env)->Throw(env, INTERP_NewObjectFromName(env,"java/lang/ArrayStoreException"));
	  return;
	  }
      }

  if( (dst_pos < 0) || (src_pos < 0) || (dst_pos > pstB->i32Number) || (src_pos > pstA->i32Number) || (length < 0))
      {
	  (*env)->Throw(env, INTERP_NewObjectFromName(env,"java/lang/ArrayIndexOutOfBoundsException"));
	  return;
      }

  if( ((dst_pos + length) > pstB->i32Number) || ((src_pos + length) > pstA->i32Number))
      {
	  (*env)->Throw(env, INTERP_NewObjectFromName(env,"java/lang/ArrayIndexOutOfBoundsException"));
	  return;
      }

  switch (pstA->enType)
  {
    case T_BYTE:
    case T_BOOLEAN:
    {
      iSize = 1;
      break;
    }
    case T_CHAR:
    case T_SHORT:
    {
      iSize = 2;
      break;
    }
    case T_ARRAY:
    case T_OBJECT:
    case T_FLOAT:
    case T_INT:
    {
      iSize = 4;
      break;
    }
    case T_DOUBLE:
    case T_LONG:
    {
      iSize = 8;
      break;
    }
  default:
      {
      eprintf("Invalid array passed to arraycopy\n");
      return;
      }
  }

  //  eprintf("Doing VMSystem.arraycopy (from %i (max %i) to %i (max %i)) for %i bytes, from %p to %p\n", src_pos, pstA->i32Number, dst_pos, pstB->i32Number, length * iSize, (byte*) pstA->pvElements + iSize * src_pos, (byte*) pstB->pvElements + iSize * dst_pos);

  if(pstA != pstB)
    {
      if((pstA->enType == T_ARRAY) || (pstA->enType == T_OBJECT))
	{
	  int i;
	  tClassLoaderTuple* targetType;
	  jclass classClass;
	  jclass destArrayClass;
	  jclass destArrayComponentClass;
	  jmethodID componentMethod;

	  //If these are references we should actually be checking the 'castability' of each item we copy 
	  
	  destArrayClass = CLASS_MakeClassObject(env, pstB->pstType);
	  classClass = (*env)->GetObjectClass(env, destArrayClass);

	  componentMethod = (*env)->GetMethodID(env, classClass, "getComponentType", "()Ljava/lang/Class;");
	  assert(componentMethod);

	  destArrayComponentClass = (*env)->CallObjectMethod(env, destArrayClass, componentMethod);
	  if((*env)->ExceptionOccurred(env) || (destArrayComponentClass == NULL))
	    {
	      (*env)->Throw(env, INTERP_NewObjectFromName(env,"java/lang/ArrayStoreException"));
	      return;		      
	    }
	  targetType= CLASS_GetClassStruct(env, destArrayComponentClass);

	  assert(targetType);

	  for(i = 0; i < length;i++)
	    {
	      tOBREF item = ((tOBREF*) pstA->pvElements)[src_pos + i];
	      if(item != NULL)
		{
		  if( INTERP_CheckCast(env, targetType, DEREF(item)->pstType) == 1)
		    {
		      ((tOBREF*)pstB->pvElements)[dst_pos + i] = item;
		    }
		  else
		    {
		      (*env)->Throw(env, INTERP_NewObjectFromName(env,"java/lang/ArrayStoreException"));
		      return;		      
		    }
		}
	      else
		{
		  ((tOBREF*)pstB->pvElements)[dst_pos + i] = NULL;
		}
	    }
	}
      else
	{
	  memcpy((byte*) pstB->pvElements + iSize * dst_pos, (byte*) pstA->pvElements + iSize * src_pos, length * iSize);
	}
    }
  else
    {
      #ifndef KISSME_LINUX_USER
      eprintf("///// Using untested array copy *****************\n");
      assert(1 == 2);
      #else
      memmove((byte*) pstB->pvElements + iSize * dst_pos, (byte*) pstA->pvElements + iSize * src_pos, length * iSize);
      #endif
    }
}


#ifdef DEBUG_TRACE
extern int INTERP_DebugOn;
#endif

void java_lang_System_setTraceStatus(JNIEnv* env, jclass class, jboolean bool)
{
  #ifdef DEBUG_TRACE
    if( bool )
	INTERP_DebugOn = 1;
    else
	INTERP_DebugOn = 0;
  #endif
}

