/* java.util.TimeZone
   Copyright (C) 1998, 1999, 2000, 2001 Free Software Foundation, Inc.

This file is part of GNU Classpath.

GNU Classpath is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU Classpath is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU Classpath; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
02111-1307 USA.

Linking this library statically or dynamically with other modules is
making a combined work based on this library.  Thus, the terms and
conditions of the GNU General Public License cover the whole
combination.

As a special exception, the copyright holders of this library give you
permission to link this library with independent modules to produce an
executable, regardless of the license terms of these independent
modules, and to copy and distribute the resulting executable under
terms of your choice, provided that you also meet, for each linked
independent module, the terms and conditions of the license of that
module.  An independent module is a module which is not derived from
or based on this library.  If you modify this library, you may extend
this exception to your version of the library, but you are not
obligated to do so.  If you do not wish to do so, delete this
exception statement from your version. */


package java.util;
import java.text.DateFormatSymbols;
import gnu.classpath.Configuration;

/**
 * This class represents a time zone offset and handles daylight savings.
 * 
 * You can get the default time zone with <code>getDefault</code>.
 * This represents the time zone where program is running.
 *
 * Another way to create a time zone is <code>getTimeZone</code>, where
 * you can give an identifier as parameter.  For instance, the identifier
 * of the Central European Time zone is "CET".
 *
 * With the <code>getAvailableIDs</code> method, you can get all the
 * supported time zone identifiers.
 *
 * @see Calendar
 * @see SimpleTimeZone
 * @author Jochen Hoenicke
 */
public abstract class TimeZone implements java.io.Serializable, Cloneable
{

  /**
   * Constant used to indicate that a short timezone abbreviation should
   * be returned, such as "EST"
   */
  public static final int SHORT = 0;

  /**
   * Constant used to indicate that a long timezone name should be
   * returned, such as "Eastern Standard Time".
   */
  public static final int LONG = 1;

  /**
   * The time zone identifier, e.g. PST.
   */
  private String ID;

  /**
   * The default time zone, as returned by getDefault.
   */
  private static TimeZone defaultZone;

  private static final long serialVersionUID = 3581463369166924961L;

  /**
   * Hashtable for timezones by ID.  
   */
  private static final Hashtable timezones = new Hashtable();

  static
  {
    TimeZone tz;
    // Automatically generated by scripts/timezones.pl
    // XXX - Should we read this data from a file?
    tz = new SimpleTimeZone(-11000 * 3600, "MIT");
    timezones.put("MIT", tz);
    timezones.put("Pacific/Apia", tz);
    timezones.put("Pacific/Midway", tz);
    timezones.put("Pacific/Niue", tz);
    timezones.put("Pacific/Pago_Pago", tz);
    tz = new SimpleTimeZone
      (-10000 * 3600, "America/Adak",
       Calendar.APRIL, 1, Calendar.SUNDAY, 2000 * 3600,
       Calendar.OCTOBER, -1, Calendar.SUNDAY, 2000 * 3600);
    timezones.put("America/Adak", tz);
    tz = new SimpleTimeZone(-10000 * 3600, "HST");
    timezones.put("HST", tz);
    timezones.put("Pacific/Fakaofo", tz);
    timezones.put("Pacific/Honolulu", tz);
    timezones.put("Pacific/Johnston", tz);
    timezones.put("Pacific/Rarotonga", tz);
    timezones.put("Pacific/Tahiti", tz);
    tz = new SimpleTimeZone(-9500 * 3600, "Pacific/Marquesas");
    timezones.put("Pacific/Marquesas", tz);
    tz = new SimpleTimeZone
      (-9000 * 3600, "AST",
       Calendar.APRIL, 1, Calendar.SUNDAY, 2000 * 3600,
       Calendar.OCTOBER, -1, Calendar.SUNDAY, 2000 * 3600);
    timezones.put("AST", tz);
    timezones.put("America/Anchorage", tz);
    timezones.put("America/Juneau", tz);
    timezones.put("America/Nome", tz);
    timezones.put("America/Yakutat", tz);
    tz = new SimpleTimeZone(-9000 * 3600, "Pacific/Gambier");
    timezones.put("Pacific/Gambier", tz);
    tz = new SimpleTimeZone
      (-8000 * 3600, "PST",
       Calendar.APRIL, 1, Calendar.SUNDAY, 2000 * 3600,
       Calendar.OCTOBER, -1, Calendar.SUNDAY, 2000 * 3600);
    timezones.put("PST", tz);
    timezones.put("PST8PDT", tz);
    timezones.put("America/Dawson", tz);
    timezones.put("America/Los_Angeles", tz);
    timezones.put("America/Tijuana", tz);
    timezones.put("America/Vancouver", tz);
    timezones.put("America/Whitehorse", tz);
    timezones.put("US/Pacific-New", tz);
    tz = new SimpleTimeZone(-8000 * 3600, "Pacific/Pitcairn");
    timezones.put("Pacific/Pitcairn", tz);
    tz = new SimpleTimeZone
      (-7000 * 3600, "MST",
       Calendar.APRIL, 1, Calendar.SUNDAY, 2000 * 3600,
       Calendar.OCTOBER, -1, Calendar.SUNDAY, 2000 * 3600);
    timezones.put("MST", tz);
    timezones.put("MST7MDT", tz);
    timezones.put("America/Boise", tz);
    timezones.put("America/Chihuahua", tz);
    timezones.put("America/Denver", tz);
    timezones.put("America/Edmonton", tz);
    timezones.put("America/Inuvik", tz);
    timezones.put("America/Mazatlan", tz);
    timezones.put("America/Shiprock", tz);
    timezones.put("America/Yellowknife", tz);
    tz = new SimpleTimeZone(-7000 * 3600, "MST7");
    timezones.put("MST7", tz);
    timezones.put("PNT", tz);
    timezones.put("America/Dawson_Creek", tz);
    timezones.put("America/Hermosillo", tz);
    timezones.put("America/Phoenix", tz);
    tz = new SimpleTimeZone
      (-6000 * 3600, "CST",
       Calendar.APRIL, 1, Calendar.SUNDAY, 2000 * 3600,
       Calendar.OCTOBER, -1, Calendar.SUNDAY, 2000 * 3600);
    timezones.put("CST", tz);
    timezones.put("CST6CDT", tz);
    timezones.put("America/Cambridge_Bay", tz);
    timezones.put("America/Cancun", tz);
    timezones.put("America/Chicago", tz);
    timezones.put("America/Menominee", tz);
    timezones.put("America/Merida", tz);
    timezones.put("America/Mexico_City", tz);
    timezones.put("America/Monterrey", tz);
    timezones.put("America/Rainy_River", tz);
    timezones.put("America/Winnipeg", tz);
    tz = new SimpleTimeZone(-6000 * 3600, "America/Belize");
    timezones.put("America/Belize", tz);
    timezones.put("America/Costa_Rica", tz);
    timezones.put("America/El_Salvador", tz);
    timezones.put("America/Guatemala", tz);
    timezones.put("America/Managua", tz);
    timezones.put("America/Regina", tz);
    timezones.put("America/Swift_Current", tz);
    timezones.put("America/Tegucigalpa", tz);
    timezones.put("Pacific/Galapagos", tz);
    tz = new SimpleTimeZone
      (-6000 * 3600, "Pacific/Easter",
       Calendar.OCTOBER, 9, -Calendar.SUNDAY, 0 * 3600,
       Calendar.MARCH, 9, -Calendar.SUNDAY, 0 * 3600);
    timezones.put("Pacific/Easter", tz);
    tz = new SimpleTimeZone
      (-5000 * 3600, "America/Grand_Turk",
       Calendar.APRIL, 1, Calendar.SUNDAY, 0 * 3600,
       Calendar.OCTOBER, -1, Calendar.SUNDAY, 0 * 3600);
    timezones.put("America/Grand_Turk", tz);
    timezones.put("America/Havana", tz);
    tz = new SimpleTimeZone(-5000 * 3600, "EST5");
    timezones.put("EST5", tz);
    timezones.put("IET", tz);
    timezones.put("America/Bogota", tz);
    timezones.put("America/Cayman", tz);
    timezones.put("America/Eirunepe", tz);
    timezones.put("America/Guayaquil", tz);
    timezones.put("America/Indiana/Indianapolis", tz);
    timezones.put("America/Indiana/Knox", tz);
    timezones.put("America/Indiana/Marengo", tz);
    timezones.put("America/Indiana/Vevay", tz);
    timezones.put("America/Indianapolis", tz);
    timezones.put("America/Iqaluit", tz);
    timezones.put("America/Jamaica", tz);
    timezones.put("America/Lima", tz);
    timezones.put("America/Panama", tz);
    timezones.put("America/Pangnirtung", tz);
    timezones.put("America/Port-au-Prince", tz);
    timezones.put("America/Porto_Acre", tz);
    timezones.put("America/Rankin_Inlet", tz);
    tz = new SimpleTimeZone
      (-5000 * 3600, "EST",
       Calendar.APRIL, 1, Calendar.SUNDAY, 2000 * 3600,
       Calendar.OCTOBER, -1, Calendar.SUNDAY, 2000 * 3600);
    timezones.put("EST", tz);
    timezones.put("EST5EDT", tz);
    timezones.put("America/Detroit", tz);
    timezones.put("America/Kentucky/Louisville", tz);
    timezones.put("America/Kentucky/Monticello", tz);
    timezones.put("America/Louisville", tz);
    timezones.put("America/Montreal", tz);
    timezones.put("America/Nassau", tz);
    timezones.put("America/New_York", tz);
    timezones.put("America/Nipigon", tz);
    timezones.put("America/Thunder_Bay", tz);
    tz = new SimpleTimeZone(-4000 * 3600, "PRT");
    timezones.put("PRT", tz);
    timezones.put("America/Anguilla", tz);
    timezones.put("America/Antigua", tz);
    timezones.put("America/Aruba", tz);
    timezones.put("America/Barbados", tz);
    timezones.put("America/Boa_Vista", tz);
    timezones.put("America/Caracas", tz);
    timezones.put("America/Curacao", tz);
    timezones.put("America/Dominica", tz);
    timezones.put("America/Grenada", tz);
    timezones.put("America/Guadeloupe", tz);
    timezones.put("America/Guyana", tz);
    timezones.put("America/La_Paz", tz);
    timezones.put("America/Manaus", tz);
    timezones.put("America/Martinique", tz);
    timezones.put("America/Montserrat", tz);
    timezones.put("America/Port_of_Spain", tz);
    timezones.put("America/Porto_Velho", tz);
    timezones.put("America/Puerto_Rico", tz);
    timezones.put("America/Santo_Domingo", tz);
    timezones.put("America/St_Kitts", tz);
    timezones.put("America/St_Lucia", tz);
    timezones.put("America/St_Thomas", tz);
    timezones.put("America/St_Vincent", tz);
    timezones.put("America/Tortola", tz);
    tz = new SimpleTimeZone
      (-4000 * 3600, "America/Asuncion",
       Calendar.OCTOBER, 1, Calendar.SUNDAY, 0 * 3600,
       Calendar.FEBRUARY, -1, Calendar.SUNDAY, 0 * 3600);
    timezones.put("America/Asuncion", tz);
    tz = new SimpleTimeZone
      (-4000 * 3600, "America/Cuiaba",
       Calendar.OCTOBER, 2, Calendar.SUNDAY, 0 * 3600,
       Calendar.FEBRUARY, 3, Calendar.SUNDAY, 0 * 3600);
    timezones.put("America/Cuiaba", tz);
    tz = new SimpleTimeZone
      (-4000 * 3600, "America/Goose_Bay",
       Calendar.APRIL, 1, Calendar.SUNDAY, 60000,
       Calendar.OCTOBER, -1, Calendar.SUNDAY, 60000);
    timezones.put("America/Goose_Bay", tz);
    tz = new SimpleTimeZone
      (-4000 * 3600, "America/Glace_Bay",
       Calendar.APRIL, 1, Calendar.SUNDAY, 2000 * 3600,
       Calendar.OCTOBER, -1, Calendar.SUNDAY, 2000 * 3600);
    timezones.put("America/Glace_Bay", tz);
    timezones.put("America/Halifax", tz);
    timezones.put("America/Thule", tz);
    timezones.put("Atlantic/Bermuda", tz);
    tz = new SimpleTimeZone
      (-4000 * 3600, "America/Santiago",
       Calendar.OCTOBER, 9, -Calendar.SUNDAY, 0 * 3600,
       Calendar.MARCH, 9, -Calendar.SUNDAY, 0 * 3600);
    timezones.put("America/Santiago", tz);
    timezones.put("Antarctica/Palmer", tz);
    tz = new SimpleTimeZone
      (-4000 * 3600, "Atlantic/Stanley",
       Calendar.SEPTEMBER, 2, Calendar.SUNDAY, 0 * 3600,
       Calendar.APRIL, 16, -Calendar.SUNDAY, 0 * 3600);
    timezones.put("Atlantic/Stanley", tz);
    tz = new SimpleTimeZone
      (-3500 * 3600, "CNT",
       Calendar.APRIL, 1, Calendar.SUNDAY, 60000,
       Calendar.OCTOBER, -1, Calendar.SUNDAY, 60000);
    timezones.put("CNT", tz);
    timezones.put("America/St_Johns", tz);
    tz = new SimpleTimeZone
      (-3000 * 3600, "America/Araguaina",
       Calendar.OCTOBER, 2, Calendar.SUNDAY, 0 * 3600,
       Calendar.FEBRUARY, 3, Calendar.SUNDAY, 0 * 3600);
    timezones.put("America/Araguaina", tz);
    timezones.put("America/Sao_Paulo", tz);
    tz = new SimpleTimeZone(-3000 * 3600, "AGT");
    timezones.put("AGT", tz);
    timezones.put("America/Belem", tz);
    timezones.put("America/Buenos_Aires", tz);
    timezones.put("America/Catamarca", tz);
    timezones.put("America/Cayenne", tz);
    timezones.put("America/Cordoba", tz);
    timezones.put("America/Fortaleza", tz);
    timezones.put("America/Jujuy", tz);
    timezones.put("America/Maceio", tz);
    timezones.put("America/Mendoza", tz);
    timezones.put("America/Montevideo", tz);
    timezones.put("America/Paramaribo", tz);
    timezones.put("America/Recife", tz);
    timezones.put("America/Rosario", tz);
    tz = new SimpleTimeZone
      (-3000 * 3600, "America/Godthab",
       Calendar.MARCH, 30, -Calendar.SATURDAY, 22000 * 3600,
       Calendar.OCTOBER, 30, -Calendar.SATURDAY, 22000 * 3600);
    timezones.put("America/Godthab", tz);
    tz = new SimpleTimeZone
      (-3000 * 3600, "America/Miquelon",
       Calendar.APRIL, 1, Calendar.SUNDAY, 2000 * 3600,
       Calendar.OCTOBER, -1, Calendar.SUNDAY, 2000 * 3600);
    timezones.put("America/Miquelon", tz);
    tz = new SimpleTimeZone(-2000 * 3600, "America/Noronha");
    timezones.put("America/Noronha", tz);
    timezones.put("Atlantic/South_Georgia", tz);
    tz = new SimpleTimeZone
      (-1000 * 3600, "America/Scoresbysund",
       Calendar.MARCH, -1, Calendar.SUNDAY, 0 * 3600,
       Calendar.OCTOBER, -1, Calendar.SUNDAY, 0 * 3600);
    timezones.put("America/Scoresbysund", tz);
    timezones.put("Atlantic/Azores", tz);
    tz = new SimpleTimeZone(-1000 * 3600, "Atlantic/Cape_Verde");
    timezones.put("Atlantic/Cape_Verde", tz);
    timezones.put("Atlantic/Jan_Mayen", tz);
    tz = new SimpleTimeZone(0 * 3600, "GMT");
    timezones.put("GMT", tz);
    timezones.put("UTC", tz);
    timezones.put("Africa/Abidjan", tz);
    timezones.put("Africa/Accra", tz);
    timezones.put("Africa/Bamako", tz);
    timezones.put("Africa/Banjul", tz);
    timezones.put("Africa/Bissau", tz);
    timezones.put("Africa/Casablanca", tz);
    timezones.put("Africa/Conakry", tz);
    timezones.put("Africa/Dakar", tz);
    timezones.put("Africa/El_Aaiun", tz);
    timezones.put("Africa/Freetown", tz);
    timezones.put("Africa/Lome", tz);
    timezones.put("Africa/Monrovia", tz);
    timezones.put("Africa/Nouakchott", tz);
    timezones.put("Africa/Ouagadougou", tz);
    timezones.put("Africa/Sao_Tome", tz);
    timezones.put("Africa/Timbuktu", tz);
    timezones.put("Atlantic/Reykjavik", tz);
    timezones.put("Atlantic/St_Helena", tz);
    timezones.put("Europe/Belfast", tz);
    timezones.put("Europe/Dublin", tz);
    timezones.put("Europe/London", tz);
    tz = new SimpleTimeZone
      (0 * 3600, "WET",
       Calendar.MARCH, -1, Calendar.SUNDAY, 1000 * 3600,
       Calendar.OCTOBER, -1, Calendar.SUNDAY, 1000 * 3600);
    timezones.put("WET", tz);
    timezones.put("Atlantic/Canary", tz);
    timezones.put("Atlantic/Faeroe", tz);
    timezones.put("Atlantic/Madeira", tz);
    timezones.put("Europe/Lisbon", tz);
    tz = new SimpleTimeZone(1000 * 3600, "Africa/Algiers");
    timezones.put("Africa/Algiers", tz);
    timezones.put("Africa/Bangui", tz);
    timezones.put("Africa/Brazzaville", tz);
    timezones.put("Africa/Douala", tz);
    timezones.put("Africa/Kinshasa", tz);
    timezones.put("Africa/Lagos", tz);
    timezones.put("Africa/Libreville", tz);
    timezones.put("Africa/Luanda", tz);
    timezones.put("Africa/Malabo", tz);
    timezones.put("Africa/Ndjamena", tz);
    timezones.put("Africa/Niamey", tz);
    timezones.put("Africa/Porto-Novo", tz);
    timezones.put("Africa/Tunis", tz);
    tz = new SimpleTimeZone
      (1000 * 3600, "Africa/Windhoek",
       Calendar.SEPTEMBER, 1, Calendar.SUNDAY, 2000 * 3600,
       Calendar.APRIL, 1, Calendar.SUNDAY, 2000 * 3600);
    timezones.put("Africa/Windhoek", tz);
    tz = new SimpleTimeZone
      (1000 * 3600, "CET",
       Calendar.MARCH, -1, Calendar.SUNDAY, 2000 * 3600,
       Calendar.OCTOBER, -1, Calendar.SUNDAY, 2000 * 3600);
    timezones.put("CET", tz);
    timezones.put("ECT", tz);
    timezones.put("MET", tz);
    timezones.put("Africa/Ceuta", tz);
    timezones.put("Arctic/Longyearbyen", tz);
    timezones.put("Europe/Amsterdam", tz);
    timezones.put("Europe/Andorra", tz);
    timezones.put("Europe/Belgrade", tz);
    timezones.put("Europe/Berlin", tz);
    timezones.put("Europe/Bratislava", tz);
    timezones.put("Europe/Brussels", tz);
    timezones.put("Europe/Budapest", tz);
    timezones.put("Europe/Copenhagen", tz);
    timezones.put("Europe/Gibraltar", tz);
    timezones.put("Europe/Ljubljana", tz);
    timezones.put("Europe/Luxembourg", tz);
    timezones.put("Europe/Madrid", tz);
    timezones.put("Europe/Malta", tz);
    timezones.put("Europe/Monaco", tz);
    timezones.put("Europe/Oslo", tz);
    timezones.put("Europe/Paris", tz);
    timezones.put("Europe/Prague", tz);
    timezones.put("Europe/Rome", tz);
    timezones.put("Europe/San_Marino", tz);
    timezones.put("Europe/Sarajevo", tz);
    timezones.put("Europe/Skopje", tz);
    timezones.put("Europe/Stockholm", tz);
    timezones.put("Europe/Tirane", tz);
    timezones.put("Europe/Vaduz", tz);
    timezones.put("Europe/Vatican", tz);
    timezones.put("Europe/Vienna", tz);
    timezones.put("Europe/Warsaw", tz);
    timezones.put("Europe/Zagreb", tz);
    timezones.put("Europe/Zurich", tz);
    tz = new SimpleTimeZone
      (2000 * 3600, "ART",
       Calendar.APRIL, -1, Calendar.FRIDAY, 0 * 3600,
       Calendar.SEPTEMBER, -1, Calendar.THURSDAY, 23000 * 3600);
    timezones.put("ART", tz);
    timezones.put("Africa/Cairo", tz);
    tz = new SimpleTimeZone(2000 * 3600, "CAT");
    timezones.put("CAT", tz);
    timezones.put("Africa/Blantyre", tz);
    timezones.put("Africa/Bujumbura", tz);
    timezones.put("Africa/Gaborone", tz);
    timezones.put("Africa/Harare", tz);
    timezones.put("Africa/Johannesburg", tz);
    timezones.put("Africa/Kigali", tz);
    timezones.put("Africa/Lubumbashi", tz);
    timezones.put("Africa/Lusaka", tz);
    timezones.put("Africa/Maputo", tz);
    timezones.put("Africa/Maseru", tz);
    timezones.put("Africa/Mbabane", tz);
    timezones.put("Africa/Tripoli", tz);
    timezones.put("Europe/Riga", tz);
    timezones.put("Europe/Tallinn", tz);
    timezones.put("Europe/Vilnius", tz);
    tz = new SimpleTimeZone
      (2000 * 3600, "Asia/Amman",
       Calendar.MARCH, -1, Calendar.THURSDAY, 0 * 3600,
       Calendar.SEPTEMBER, -1, Calendar.THURSDAY, 0 * 3600);
    timezones.put("Asia/Amman", tz);
    tz = new SimpleTimeZone
      (2000 * 3600, "Asia/Beirut",
       Calendar.MARCH, -1, Calendar.SUNDAY, 0 * 3600,
       Calendar.OCTOBER, -1, Calendar.SUNDAY, 0 * 3600);
    timezones.put("Asia/Beirut", tz);
    tz = new SimpleTimeZone
      (2000 * 3600, "Asia/Damascus",
       Calendar.APRIL, 1, 0, 0 * 3600,
       Calendar.OCTOBER, 1, 0, 0 * 3600);
    timezones.put("Asia/Damascus", tz);
    tz = new SimpleTimeZone
      (2000 * 3600, "Asia/Gaza",
       Calendar.APRIL, 3, Calendar.FRIDAY, 0 * 3600,
       Calendar.OCTOBER, 3, Calendar.FRIDAY, 0 * 3600);
    timezones.put("Asia/Gaza", tz);
    tz = new SimpleTimeZone
      (2000 * 3600, "Asia/Jerusalem",
       Calendar.APRIL, 1, 0, 1000 * 3600,
       Calendar.OCTOBER, 1, 0, 1000 * 3600);
    timezones.put("Asia/Jerusalem", tz);
    tz = new SimpleTimeZone
      (2000 * 3600, "EET",
       Calendar.MARCH, -1, Calendar.SUNDAY, 3000 * 3600,
       Calendar.OCTOBER, -1, Calendar.SUNDAY, 3000 * 3600);
    timezones.put("EET", tz);
    timezones.put("Asia/Istanbul", tz);
    timezones.put("Asia/Nicosia", tz);
    timezones.put("Europe/Athens", tz);
    timezones.put("Europe/Bucharest", tz);
    timezones.put("Europe/Chisinau", tz);
    timezones.put("Europe/Helsinki", tz);
    timezones.put("Europe/Istanbul", tz);
    timezones.put("Europe/Kiev", tz);
    timezones.put("Europe/Nicosia", tz);
    timezones.put("Europe/Simferopol", tz);
    timezones.put("Europe/Sofia", tz);
    timezones.put("Europe/Uzhgorod", tz);
    timezones.put("Europe/Zaporozhye", tz);
    tz = new SimpleTimeZone
      (2000 * 3600, "Europe/Kaliningrad",
       Calendar.MARCH, -1, Calendar.SUNDAY, 2000 * 3600,
       Calendar.OCTOBER, -1, Calendar.SUNDAY, 2000 * 3600);
    timezones.put("Europe/Kaliningrad", tz);
    timezones.put("Europe/Minsk", tz);
    tz = new SimpleTimeZone
      (3000 * 3600, "Asia/Baghdad",
       Calendar.APRIL, 1, 0, 3000 * 3600,
       Calendar.OCTOBER, 1, 0, 3000 * 3600);
    timezones.put("Asia/Baghdad", tz);
    tz = new SimpleTimeZone
      (3000 * 3600, "Europe/Moscow",
       Calendar.MARCH, -1, Calendar.SUNDAY, 2000 * 3600,
       Calendar.OCTOBER, -1, Calendar.SUNDAY, 2000 * 3600);
    timezones.put("Europe/Moscow", tz);
    timezones.put("Europe/Tiraspol", tz);
    tz = new SimpleTimeZone(3000 * 3600, "EAT");
    timezones.put("EAT", tz);
    timezones.put("Africa/Addis_Ababa", tz);
    timezones.put("Africa/Asmera", tz);
    timezones.put("Africa/Dar_es_Salaam", tz);
    timezones.put("Africa/Djibouti", tz);
    timezones.put("Africa/Kampala", tz);
    timezones.put("Africa/Khartoum", tz);
    timezones.put("Africa/Mogadishu", tz);
    timezones.put("Africa/Nairobi", tz);
    timezones.put("Antarctica/Syowa", tz);
    timezones.put("Asia/Aden", tz);
    timezones.put("Asia/Bahrain", tz);
    timezones.put("Asia/Kuwait", tz);
    timezones.put("Asia/Qatar", tz);
    timezones.put("Asia/Riyadh", tz);
    timezones.put("Indian/Antananarivo", tz);
    timezones.put("Indian/Comoro", tz);
    timezones.put("Indian/Mayotte", tz);
    tz = new SimpleTimeZone(3500 * 3600, "Asia/Tehran");
    timezones.put("Asia/Tehran", tz);
    tz = new SimpleTimeZone
      (4000 * 3600, "Asia/Baku",
       Calendar.MARCH, -1, Calendar.SUNDAY, 1000 * 3600,
       Calendar.OCTOBER, -1, Calendar.SUNDAY, 1000 * 3600);
    timezones.put("Asia/Baku", tz);
    tz = new SimpleTimeZone
      (4000 * 3600, "Asia/Aqtau",
       Calendar.MARCH, -1, Calendar.SUNDAY, 0 * 3600,
       Calendar.OCTOBER, -1, Calendar.SUNDAY, 0 * 3600);
    timezones.put("Asia/Aqtau", tz);
    timezones.put("Asia/Tbilisi", tz);
    tz = new SimpleTimeZone
      (4000 * 3600, "Asia/Yerevan",
       Calendar.MARCH, -1, Calendar.SUNDAY, 2000 * 3600,
       Calendar.OCTOBER, -1, Calendar.SUNDAY, 2000 * 3600);
    timezones.put("Asia/Yerevan", tz);
    timezones.put("Europe/Samara", tz);
    tz = new SimpleTimeZone(4000 * 3600, "NET");
    timezones.put("NET", tz);
    timezones.put("Asia/Dubai", tz);
    timezones.put("Asia/Muscat", tz);
    timezones.put("Indian/Mahe", tz);
    timezones.put("Indian/Mauritius", tz);
    timezones.put("Indian/Reunion", tz);
    tz = new SimpleTimeZone(4500 * 3600, "Asia/Kabul");
    timezones.put("Asia/Kabul", tz);
    tz = new SimpleTimeZone
      (5000 * 3600, "Asia/Aqtobe",
       Calendar.MARCH, -1, Calendar.SUNDAY, 0 * 3600,
       Calendar.OCTOBER, -1, Calendar.SUNDAY, 0 * 3600);
    timezones.put("Asia/Aqtobe", tz);
    tz = new SimpleTimeZone
      (5000 * 3600, "Asia/Bishkek",
       Calendar.MARCH, -1, Calendar.SUNDAY, 2500 * 3600,
       Calendar.OCTOBER, -1, Calendar.SUNDAY, 2500 * 3600);
    timezones.put("Asia/Bishkek", tz);
    tz = new SimpleTimeZone
      (5000 * 3600, "Asia/Yekaterinburg",
       Calendar.MARCH, -1, Calendar.SUNDAY, 2000 * 3600,
       Calendar.OCTOBER, -1, Calendar.SUNDAY, 2000 * 3600);
    timezones.put("Asia/Yekaterinburg", tz);
    tz = new SimpleTimeZone(5000 * 3600, "PLT");
    timezones.put("PLT", tz);
    timezones.put("Asia/Ashgabat", tz);
    timezones.put("Asia/Dushanbe", tz);
    timezones.put("Asia/Karachi", tz);
    timezones.put("Asia/Samarkand", tz);
    timezones.put("Asia/Tashkent", tz);
    timezones.put("Indian/Chagos", tz);
    timezones.put("Indian/Kerguelen", tz);
    timezones.put("Indian/Maldives", tz);
    tz = new SimpleTimeZone(5500 * 3600, "IST");
    timezones.put("IST", tz);
    timezones.put("Asia/Calcutta", tz);
    tz = new SimpleTimeZone(5750 * 3600, "Asia/Katmandu");
    timezones.put("Asia/Katmandu", tz);
    tz = new SimpleTimeZone(6000 * 3600, "BST");
    timezones.put("BST", tz);
    timezones.put("Antarctica/Mawson", tz);
    timezones.put("Asia/Colombo", tz);
    timezones.put("Asia/Dhaka", tz);
    timezones.put("Asia/Thimphu", tz);
    tz = new SimpleTimeZone
      (6000 * 3600, "Asia/Almaty",
       Calendar.MARCH, -1, Calendar.SUNDAY, 0 * 3600,
       Calendar.OCTOBER, -1, Calendar.SUNDAY, 0 * 3600);
    timezones.put("Asia/Almaty", tz);
    tz = new SimpleTimeZone
      (6000 * 3600, "Asia/Novosibirsk",
       Calendar.MARCH, -1, Calendar.SUNDAY, 2000 * 3600,
       Calendar.OCTOBER, -1, Calendar.SUNDAY, 2000 * 3600);
    timezones.put("Asia/Novosibirsk", tz);
    timezones.put("Asia/Omsk", tz);
    tz = new SimpleTimeZone(6500 * 3600, "Asia/Rangoon");
    timezones.put("Asia/Rangoon", tz);
    timezones.put("Indian/Cocos", tz);
    tz = new SimpleTimeZone(7000 * 3600, "VST");
    timezones.put("VST", tz);
    timezones.put("Antarctica/Davis", tz);
    timezones.put("Asia/Bangkok", tz);
    timezones.put("Asia/Hovd", tz);
    timezones.put("Asia/Jakarta", tz);
    timezones.put("Asia/Phnom_Penh", tz);
    timezones.put("Asia/Saigon", tz);
    timezones.put("Asia/Vientiane", tz);
    timezones.put("Indian/Christmas", tz);
    tz = new SimpleTimeZone
      (7000 * 3600, "Asia/Krasnoyarsk",
       Calendar.MARCH, -1, Calendar.SUNDAY, 2000 * 3600,
       Calendar.OCTOBER, -1, Calendar.SUNDAY, 2000 * 3600);
    timezones.put("Asia/Krasnoyarsk", tz);
    tz = new SimpleTimeZone(8000 * 3600, "CTT");
    timezones.put("CTT", tz);
    timezones.put("Antarctica/Casey", tz);
    timezones.put("Asia/Brunei", tz);
    timezones.put("Asia/Chungking", tz);
    timezones.put("Asia/Harbin", tz);
    timezones.put("Asia/Hong_Kong", tz);
    timezones.put("Asia/Kashgar", tz);
    timezones.put("Asia/Kuala_Lumpur", tz);
    timezones.put("Asia/Kuching", tz);
    timezones.put("Asia/Macao", tz);
    timezones.put("Asia/Manila", tz);
    timezones.put("Asia/Shanghai", tz);
    timezones.put("Asia/Singapore", tz);
    timezones.put("Asia/Taipei", tz);
    timezones.put("Asia/Ujung_Pandang", tz);
    timezones.put("Asia/Ulaanbaatar", tz);
    timezones.put("Asia/Urumqi", tz);
    timezones.put("Australia/Perth", tz);
    tz = new SimpleTimeZone
      (8000 * 3600, "Asia/Irkutsk",
       Calendar.MARCH, -1, Calendar.SUNDAY, 2000 * 3600,
       Calendar.OCTOBER, -1, Calendar.SUNDAY, 2000 * 3600);
    timezones.put("Asia/Irkutsk", tz);
    tz = new SimpleTimeZone(9000 * 3600, "JST");
    timezones.put("JST", tz);
    timezones.put("Asia/Dili", tz);
    timezones.put("Asia/Jayapura", tz);
    timezones.put("Asia/Pyongyang", tz);
    timezones.put("Asia/Seoul", tz);
    timezones.put("Asia/Tokyo", tz);
    timezones.put("Pacific/Palau", tz);
    tz = new SimpleTimeZone
      (9000 * 3600, "Asia/Yakutsk",
       Calendar.MARCH, -1, Calendar.SUNDAY, 2000 * 3600,
       Calendar.OCTOBER, -1, Calendar.SUNDAY, 2000 * 3600);
    timezones.put("Asia/Yakutsk", tz);
    tz = new SimpleTimeZone
      (9500 * 3600, "Australia/Adelaide",
       Calendar.OCTOBER, -1, Calendar.SUNDAY, 2000 * 3600,
       Calendar.MARCH, -1, Calendar.SUNDAY, 2000 * 3600);
    timezones.put("Australia/Adelaide", tz);
    timezones.put("Australia/Broken_Hill", tz);
    tz = new SimpleTimeZone(9500 * 3600, "ACT");
    timezones.put("ACT", tz);
    timezones.put("Australia/Darwin", tz);
    tz = new SimpleTimeZone(10000 * 3600, "Antarctica/DumontDUrville");
    timezones.put("Antarctica/DumontDUrville", tz);
    timezones.put("Australia/Brisbane", tz);
    timezones.put("Australia/Lindeman", tz);
    timezones.put("Pacific/Guam", tz);
    timezones.put("Pacific/Port_Moresby", tz);
    timezones.put("Pacific/Saipan", tz);
    timezones.put("Pacific/Truk", tz);
    timezones.put("Pacific/Yap", tz);
    tz = new SimpleTimeZone
      (10000 * 3600, "Asia/Vladivostok",
       Calendar.MARCH, -1, Calendar.SUNDAY, 2000 * 3600,
       Calendar.OCTOBER, -1, Calendar.SUNDAY, 2000 * 3600);
    timezones.put("Asia/Vladivostok", tz);
    tz = new SimpleTimeZone
      (10000 * 3600, "Australia/Hobart",
       Calendar.OCTOBER, 1, Calendar.SUNDAY, 2000 * 3600,
       Calendar.MARCH, -1, Calendar.SUNDAY, 2000 * 3600);
    timezones.put("Australia/Hobart", tz);
    tz = new SimpleTimeZone
      (10000 * 3600, "AET",
       Calendar.OCTOBER, -1, Calendar.SUNDAY, 2000 * 3600,
       Calendar.MARCH, -1, Calendar.SUNDAY, 2000 * 3600);
    timezones.put("AET", tz);
    timezones.put("Australia/Melbourne", tz);
    timezones.put("Australia/Sydney", tz);
    tz = new SimpleTimeZone
      (10500 * 3600, "Australia/Lord_Howe",
       Calendar.OCTOBER, -1, Calendar.SUNDAY, 2000 * 3600,
       Calendar.MARCH, -1, Calendar.SUNDAY, 2000 * 3600, 500 * 3600);
    timezones.put("Australia/Lord_Howe", tz);
    tz = new SimpleTimeZone
      (11000 * 3600, "Asia/Magadan",
       Calendar.MARCH, -1, Calendar.SUNDAY, 2000 * 3600,
       Calendar.OCTOBER, -1, Calendar.SUNDAY, 2000 * 3600);
    timezones.put("Asia/Magadan", tz);
    tz = new SimpleTimeZone(11000 * 3600, "SST");
    timezones.put("SST", tz);
    timezones.put("Pacific/Efate", tz);
    timezones.put("Pacific/Guadalcanal", tz);
    timezones.put("Pacific/Kosrae", tz);
    timezones.put("Pacific/Noumea", tz);
    timezones.put("Pacific/Ponape", tz);
    tz = new SimpleTimeZone(11500 * 3600, "Pacific/Norfolk");
    timezones.put("Pacific/Norfolk", tz);
    tz = new SimpleTimeZone
      (12000 * 3600, "NST",
       Calendar.OCTOBER, 1, Calendar.SUNDAY, 2000 * 3600,
       Calendar.MARCH, 3, Calendar.SUNDAY, 2000 * 3600);
    timezones.put("NST", tz);
    timezones.put("Antarctica/McMurdo", tz);
    timezones.put("Antarctica/South_Pole", tz);
    timezones.put("Pacific/Auckland", tz);
    tz = new SimpleTimeZone
      (12000 * 3600, "Asia/Anadyr",
       Calendar.MARCH, -1, Calendar.SUNDAY, 2000 * 3600,
       Calendar.OCTOBER, -1, Calendar.SUNDAY, 2000 * 3600);
    timezones.put("Asia/Anadyr", tz);
    timezones.put("Asia/Kamchatka", tz);
    tz = new SimpleTimeZone(12000 * 3600, "Pacific/Fiji");
    timezones.put("Pacific/Fiji", tz);
    timezones.put("Pacific/Funafuti", tz);
    timezones.put("Pacific/Kwajalein", tz);
    timezones.put("Pacific/Majuro", tz);
    timezones.put("Pacific/Nauru", tz);
    timezones.put("Pacific/Tarawa", tz);
    timezones.put("Pacific/Wake", tz);
    timezones.put("Pacific/Wallis", tz);
    tz = new SimpleTimeZone
      (12750 * 3600, "Pacific/Chatham",
       Calendar.OCTOBER, 1, Calendar.SUNDAY, 2750 * 3600,
       Calendar.MARCH, 3, Calendar.SUNDAY, 2750 * 3600);
    timezones.put("Pacific/Chatham", tz);
    tz = new SimpleTimeZone(13000 * 3600, "Pacific/Enderbury");
    timezones.put("Pacific/Enderbury", tz);
    timezones.put("Pacific/Tongatapu", tz);
    tz = new SimpleTimeZone(14000 * 3600, "Pacific/Kiritimati");
    timezones.put("Pacific/Kiritimati", tz);
  }


  /* Look up default timezone */
  static
  {
    if (Configuration.INIT_LOAD_LIBRARY)
      {
	System.loadLibrary("javautil");
      }
    String tzid = System.getProperty("user.timezone");

    // XXX: Remove call to getDefaultTimeZoneId after java.lang.System
    // is merged with libgcj.
    if (tzid == null)
      tzid = getDefaultTimeZoneId();

    if (tzid == null)
      tzid = "GMT";

    defaultZone = getTimeZone(tzid);
  }

  /* This method returns us a time zone id string which is in the
     form <standard zone name><GMT offset><daylight time zone name>.
     The GMT offset is in seconds, except where it is evenly divisible
     by 3600, then it is in hours.  If the zone does not observe
     daylight time, then the daylight zone name is omitted.  Examples:
     in Chicago, the timezone would be CST6CDT.  In Indianapolis 
     (which does not have Daylight Savings Time) the string would
     be EST5
   */
  // XXX: Remove getDefaultTimeZoneId after java.lang.System
  // is merged with libgcj.
  private static native String getDefaultTimeZoneId();

  /**
   * Gets the time zone offset, for current date, modified in case of 
   * daylight savings.  This is the offset to add to UTC to get the local
   * time.
   * @param era the era of the given date
   * @param year the year of the given date
   * @param month the month of the given date, 0 for January.
   * @param day the day of month
   * @param dayOfWeek the day of week
   * @param milliseconds the millis in the day (in local standard time)
   * @return the time zone offset in milliseconds.
   */
  public abstract int getOffset(int era, int year, int month,
				int day, int dayOfWeek, int milliseconds);

  /**
   * Gets the time zone offset, ignoring daylight savings.  This is
   * the offset to add to UTC to get the local time.
   * @return the time zone offset in milliseconds.  
   */
  public abstract int getRawOffset();

  /**
   * Sets the time zone offset, ignoring daylight savings.  This is
   * the offset to add to UTC to get the local time.
   * @param offsetMillis the time zone offset to GMT.
   */
  public abstract void setRawOffset(int offsetMillis);

  /**
   * Gets the identifier of this time zone. For instance, PST for
   * Pacific Standard Time.
   * @returns the ID of this time zone.  
   */
  public String getID()
  {
    return ID;
  }

  /**
   * Sets the identifier of this time zone. For instance, PST for
   * Pacific Standard Time.
   * @param id the new time zone ID.
   */
  public void setID(String id)
  {
    this.ID = id;
  }

  /**
   * This method returns a string name of the time zone suitable
   * for displaying to the user.  The string returned will be the long
   * description of the timezone in the current locale.  The name
   * displayed will assume daylight savings time is not in effect.
   *
   * @return The name of the time zone.
   */
  public final String getDisplayName()
  {
    return (getDisplayName(false, LONG, Locale.getDefault()));
  }

  /**
   * This method returns a string name of the time zone suitable
   * for displaying to the user.  The string returned will be the long
   * description of the timezone in the specified locale. The name
   * displayed will assume daylight savings time is not in effect.
   *
   * @param locale The locale for this timezone name.
   *
   * @return The name of the time zone.
   */
  public final String getDisplayName(Locale locale)
  {
    return (getDisplayName(false, LONG, locale));
  }

  /**
   * This method returns a string name of the time zone suitable
   * for displaying to the user.  The string returned will be of the
   * specified type in the current locale. 
   *
   * @param dst Whether or not daylight savings time is in effect.
   * @param style <code>LONG</code> for a long name, <code>SHORT</code> for
   * a short abbreviation.
   *
   * @return The name of the time zone.
   */
  public final String getDisplayName(boolean dst, int style)
  {
    return (getDisplayName(dst, style, Locale.getDefault()));
  }


  /**
   * This method returns a string name of the time zone suitable
   * for displaying to the user.  The string returned will be of the
   * specified type in the specified locale. 
   *
   * @param dst Whether or not daylight savings time is in effect.
   * @param style <code>LONG</code> for a long name, <code>SHORT</code> for
   * a short abbreviation.
   * @param locale The locale for this timezone name.
   *
   * @return The name of the time zone.
   */
  public String getDisplayName(boolean dst, int style, Locale locale)
  {
    DateFormatSymbols dfs;
    try
      {
	dfs = new DateFormatSymbols(locale);

	// The format of the value returned is defined by us.
	String[][]zoneinfo = dfs.getZoneStrings();
	for (int i = 0; i < zoneinfo.length; i++)
	  {
	    if (zoneinfo[i][0].equals(getID()))
	      {
		if (!dst)
		  {
		    if (style == SHORT)
		      return (zoneinfo[i][2]);
		    else
		      return (zoneinfo[i][1]);
		  }
		else
		  {
		    if (style == SHORT)
		      return (zoneinfo[i][4]);
		    else
		      return (zoneinfo[i][3]);
		  }
	      }
	  }
      }
    catch (MissingResourceException e)
      {
      }

    return getDefaultDisplayName(dst);
  }

  private String getDefaultDisplayName(boolean dst)
  {
    int offset = getRawOffset();
    if (dst && this instanceof SimpleTimeZone)
      {
	// ugly, but this is a design failure of the API:
	// getDisplayName takes a dst parameter even though
	// TimeZone knows nothing about daylight saving offsets.
	offset += ((SimpleTimeZone) this).getDSTSavings();
      }

    StringBuffer sb = new StringBuffer(9);
    sb.append("GMT");
    sb.append(offset >= 0 ? '+' : '-');

    offset = Math.abs(offset) / (1000 * 60);
    int hours = offset / 60;
    int minutes = offset % 60;

    sb.append((char) ('0' + hours / 10)).append((char) ('0' + hours % 10));
    sb.append(':');
    sb.append((char) ('0' + minutes / 10)).append((char) ('0' + minutes % 10));
    return sb.toString();
  }

  /** 
   * Returns true, if this time zone uses Daylight Savings Time.
   */
  public abstract boolean useDaylightTime();

  /**
   * Returns true, if the given date is in Daylight Savings Time in this
   * time zone.
   * @param date the given Date.
   */
  public abstract boolean inDaylightTime(Date date);

  /**
   * Gets the TimeZone for the given ID.
   * @param ID the time zone identifier.
   * @return The time zone for the identifier or GMT, if no such time
   * zone exists.
   */
  public static TimeZone getTimeZone(String ID)
  {
    // First check timezones hash
    TimeZone tz = (TimeZone) timezones.get(ID);
    if (tz != null)
      {
	if (tz.getID().equals(ID))
	  return tz;

	// We always return a timezone with the requested ID.
	// This is the same behaviour as with JDK1.2.
	tz = (TimeZone) tz.clone();
	tz.setID(ID);
	// We also save the alias, so that we return the same
	// object again if getTimeZone is called with the same
	// alias.
	timezones.put(ID, tz);
	return tz;
      }

    // See if the ID is really a GMT offset form.
    // Note that GMT is in the table so we know it is different.
    if (ID.startsWith("GMT"))
      {
	int pos = 3;
	int offset_direction = 1;

	if (ID.charAt(pos) == '-')
	  {
	    offset_direction = -1;
	    pos++;
	  }
	else if (ID.charAt(pos) == '+')
	  {
	    pos++;
	  }

	try
	  {
	    int hour, minute;

	    String offset_str = ID.substring(pos);
	    int idx = offset_str.indexOf(":");
	    if (idx != -1)
	      {
		hour = Integer.parseInt(offset_str.substring(0, idx));
		minute = Integer.parseInt(offset_str.substring(idx + 1));
	      }
	    else
	      {
		int offset_length = offset_str.length();
		if (offset_length <= 2)
		  {
		    // Only hour
		    hour = Integer.parseInt(offset_str);
		    minute = 0;
		  }
		else
		  {
		    // hour and minute, not separated by colon
		    hour = Integer.parseInt
		      (offset_str.substring(0, offset_length - 2));
		    minute = Integer.parseInt
		      (offset_str.substring(offset_length - 2));
		  }
	      }

	    return new SimpleTimeZone((hour * (60 * 60 * 1000) +
				       minute * (60 * 1000))
				      * offset_direction, ID);
	  }
	catch (NumberFormatException e)
	  {
	  }
      }

    // Finally, return GMT per spec
    return getTimeZone("GMT");
  }

  /**
   * Gets the available IDs according to the given time zone
   * offset.  
   * @param rawOffset the given time zone GMT offset.
   * @return An array of IDs, where the time zone has the specified GMT
   * offset. For example <code>{"Phoenix", "Denver"}</code>, since both have
   * GMT-07:00, but differ in daylight savings behaviour.
   */
  public static String[] getAvailableIDs(int rawOffset)
  {
    int count = 0;
    Iterator iter = timezones.entrySet().iterator();
    while (iter.hasNext())
      {
	// Don't iterate the values, since we want to count 
	// doubled values (aliases)
	Map.Entry entry = (Map.Entry) iter.next();
	if (((TimeZone) entry.getValue()).getRawOffset() == rawOffset)
	  count++;
      }

    String[] ids = new String[count];
    count = 0;
    iter = timezones.entrySet().iterator();
    while (iter.hasNext())
      {
	Map.Entry entry = (Map.Entry) iter.next();
	if (((TimeZone) entry.getValue()).getRawOffset() == rawOffset)
	  ids[count++] = (String) entry.getKey();
      }
    return ids;
  }

  /**
   * Gets all available IDs.
   * @return An array of all supported IDs.
   */
  public static String[] getAvailableIDs()
  {
    return (String[])
      timezones.keySet().toArray(new String[timezones.size()]);
  }

  /**
   * Returns the time zone under which the host is running.  This
   * can be changed with setDefault.
   * @return the time zone for this host.
   * @see #setDefault
   */
  public static TimeZone getDefault()
  {
    return defaultZone;
  }

  public static void setDefault(TimeZone zone)
  {
    defaultZone = zone;
  }

  /**
   * Test if the other time zone uses the same rule and only
   * possibly differs in ID.  This implementation for this particular
   * class will return true if the raw offsets are identical.  Subclasses
   * should override this method if they use daylight savings.
   * @return true if this zone has the same raw offset
   */
  public boolean hasSameRules(TimeZone other)
  {
    return other.getRawOffset() == getRawOffset();
  }

  /**
   * Returns a clone of this object.  I can't imagine, why this is
   * useful for a time zone.
   */
  public Object clone()
  {
    try
      {
	return super.clone();
      }
    catch (CloneNotSupportedException ex)
      {
	return null;
      }
  }
}
