#ifndef __PACKET_H__
#define __PACKET_H__

#include <ctype.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <asm/types.h>
#include <netinet/in.h>
#include <linux/netlink.h>
#include <sys/socket.h>
#include <string.h>
#include <signal.h>
#include <unistd.h>

#include "config.h"

// These won't include right if they're not in the C namespace
extern "C" {
#ifdef HAVE_LIBPCAP
#ifndef HAVE_PCAPPCAP_H
#include <pcap.h>
#else
#include <pcap/pcap.h>
#endif
#endif

#ifdef HAVE_LIBWIRETAP
#include <wtap.h>
#endif
}

#define DEVNAME_LEN 16
#define MAX_PACKET_LEN 8192

#define SSID_SIZE 32
#define MAC_LEN 6

const uint8_t CISCO_SIGNATURE[] = {0xAA, 0xAA, 0x03, 0x00, 0x00, 0x0C, 0x20, 0x00};
const int CISCO_OFFSET = 0;

const uint8_t ARP_SIGNATURE[] = {0x08, 0x06};
const int ARP_OFFSET = 6;

//const uint8_t DHCPD_SIGNATURE[] = {0x00, 0x43, 0x00, 0x44};
//const int DHCPD_OFFSET = 28;

const uint8_t DHCPD_SIGNATURE[] = {0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x43, 0x00, 0x44};
const int DHCPD_OFFSET = 24;

const uint8_t UDP_SIGNATURE[] = {0x11};
const int UDP_OFFSET = 17;

// Very similar to pcap_pkthdr and wtap_pkthdr.  This is our
// common packet header that we convert everything to.
typedef struct {
    unsigned int len;
    unsigned int caplen;
    timeval ts;
} pkthdr;

// And 802.11 packet frame header
typedef struct {
    unsigned short version : 2;
    unsigned short type : 2;
    unsigned short subtype : 4;
    unsigned short to_ds : 1;
    unsigned short from_ds : 1;
    unsigned short more_fragments : 1;
    unsigned short retry : 1;
    unsigned short power_management : 1;
    unsigned short more_data : 1;
    unsigned short wep : 1;
    unsigned short order : 1;
} frame_control;

enum protocol_info_type {
    proto_unknown,
    proto_udp, proto_misc_tcp,
    proto_arp, proto_dhcp_server, proto_smb_announce, proto_ripv1_announce,
    proto_cdp
};

// CDP
// Cisco Discovery Protocol
// This spews a tremendous amount of revealing information about the
// internals of a network, if they have cisco equipment.
typedef struct {
    unsigned int : 8;
    unsigned int : 8;
    unsigned int : 8;
    unsigned int : 1;
    unsigned int level1 : 1;
    unsigned int igmp_forward : 1;
    unsigned int nlp : 1;
    unsigned int level2_switching : 1;
    unsigned int level2_sourceroute : 1;
    unsigned int level2_transparent : 1;
    unsigned int level3 : 1;
} cdp_capabilities;

typedef struct {
    char dev_id[128];
    uint8_t ip[4];
    char interface[128];
    cdp_capabilities cap;
    char software[512];
    char platform[128];
} cdp_packet;

typedef struct {
    unsigned int : 8;
    unsigned int type : 8;
    unsigned int : 8;
    unsigned int length : 8;
    char data;
} cdp_element;

typedef struct {
    unsigned int type : 8;
    unsigned int length : 8;
    unsigned int proto : 8;
    unsigned int : 8;
    unsigned int proto_length : 8;
    char addr;
} cdp_proto_element;

// Info about a protocol
typedef struct {
    protocol_info_type type;

    uint8_t source_ip[4];
    uint8_t dest_ip[4];

    uint8_t misc_ip[4];

    uint8_t mask[4];

    uint8_t gate_ip[4];

    cdp_packet cdp;

} proto_info;


// ------------------------
// packet conversion and extraction utilities
// Packet types
enum packet_info_type {
    packet_unknown, packet_beacon, packet_probe_req, packet_data,
    packet_data_broadcast, packet_adhoc, packet_control,
    packet_noise, packet_probe_response
};

// Info about a packet
typedef struct {
    // Packet info type

    packet_info_type type;
    // SSID
    char ssid[SSID_SIZE];
    // Is wep enabled?
    int wep;
    // Is this an AP or a adhoc?
    int ap;
    // What channel?
    int channel;
    // Is this encrypted?
    int encrypted;
    // Is it weak crypto?
    int interesting;
    // MAC source of packet
    uint8_t source_mac[MAC_LEN];
    // MAC dest of packet
    uint8_t dest_mac[MAC_LEN];
    // BSSID MAC this packet belongs to
    uint8_t bssid_mac[MAC_LEN];

    // Offset of the header
    int header_offset;

    proto_info proto;

} packet_info;

// ----------------------------------
// Info extraction functions
int GetTagOffset(int init_offset, int tagnum, const pkthdr *header, const u_char *data);
packet_info GetPacketInfo(const pkthdr *header, const u_char *data);

proto_info GetProtoInfo(const packet_info *in_info, const pkthdr *header, const u_char *in_data);

#endif

