#ifndef KINO_PLUGIN_UTILITY_H
#define KINO_PLUGIN_UTILITY_H

#include <cassert>
#include <deque>
#include <vector>

namespace kino
{

/// Clamps the input value to the range [MinVal, MaxVal]
template<typename ArgType>
ArgType clamp(const ArgType A, const ArgType MinVal, const ArgType MaxVal)
{
        return std::min(std::max(A, MinVal), MaxVal);
}

/// Linear interpolation between two values
template<typename ArgType>
inline ArgType lerp(const ArgType A, const ArgType B, const double Mix)
{
        return static_cast<ArgType>(A * (1.0 - Mix) + B * Mix);
}

inline double step(const double Edge, const double A)
{
        return A < Edge ? 0.0 : 1.0;
}

inline double linearstep(const double Edge1, const double Edge2, const double A)
{
        if(A < Edge1)
                return 0.0;
                
        if(A >= Edge2)
                return 1.0;
                
        return (A - Edge1) / (Edge2 - Edge1);
}

inline double smoothstep(const double Edge1, const double Edge2, const double A)
{
        if(A < Edge1)
                return 0.0;
                
        if(A >= Edge2)
                return 1.0;
                
        double a = (A - Edge1) / (Edge2 - Edge1);
        
        return (a * a * (3 - 2 * a));
}

inline double pulse(const double Edge1, const double Edge2, const double A)
{
        return step(Edge1, A) - step(Edge2, A);
}

inline double factorial(const unsigned int X)
{
	double result = 1;
	
	for(unsigned int i = 2; i <= X; ++i)
		result *= i;
		
	return result;
}

/// Attempts to bring Resource Acquisition Is Initialization to legacy "C" constructs
template<typename ResourceType>
class raii
{
public:
	raii(ResourceType* const Resource, void (*ReleaseMethod)(ResourceType*)) :
		m_resource(Resource),
		m_release_method(ReleaseMethod)
	{
		// Sanity checks ...
		assert(m_release_method);
	}
	
	~raii()
	{
		if(m_resource)
			(*m_release_method)(m_resource);
	}

	ResourceType* get() const
	{
		return m_resource;
	}
			
	ResourceType* operator->() const
	{
		return m_resource;
	}
	
	ResourceType& operator*() const
	{
		return *m_resource;
	}

private:
	ResourceType* const m_resource;
	void (*m_release_method)(ResourceType*);
};

} // namespace kino

#endif // !KINO_PLUGIN_UTILITY_H

