/*
 * $Id: sonyjog.c,v 1.1 2001/07/30 03:57:45 ddennedy Exp $
 *
 *  Copyright (c) 2001 Tomoaki Hayasaka <hayasaka@postman.riken.go.jp>
 *
 *  SONY USB Jog Controller support
 *
 */

/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Should you need to contact me, the author, you can do so either by
 * e-mail - mail your message to <vojtech@suse.cz>, or by paper mail:
 * Vojtech Pavlik, Ucitelska 1576, Prague 8, 182 00 Czech Republic
 */

#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/input.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/usb.h>

/*
 * Version Information
 */
#define DRIVER_VERSION "v0.1"
#define DRIVER_AUTHOR "Tomoaki Hayasaka <hayasaka@postman.riken.go.jp>"
#define DRIVER_DESC "SONY USB Jog Controller driver"

MODULE_AUTHOR( DRIVER_AUTHOR );
MODULE_DESCRIPTION( DRIVER_DESC );

/*
 * SONY USB Jog Controller packet has similar format to USB keyboard's:
 *
 * byte 0: shift (00 or 02)
 * byte 1: unused?
 * byte 2-7: other keys
 *
 * scan codes:
 *   00: none
 *
 *   04: shuttle +1 (neutral)
 *   05: shuttle +2
 *   06: shuttle +3
 *   07: shuttle +4
 *   08: shuttle +5
 *   09: shuttle +6
 *   0a: shuttle +7
 *   0b: shuttle +8
 *   0c: shuttle -1
 *   0d: shuttle -2
 *   0e: shuttle -3
 *   0f: shuttle -4
 *   10: shuttle -5
 *   2e: shuttle -6
 *   30: shuttle -7
 *   31: shuttle -8
 *
 *   1e: jog left
 *   1f: jog right
 *
 *   15: stick button
 *
 *   11: F1
 *   12: F2
 *   13: F3
 *   14: F4
 *   1c: C1
 *   19: C2
 *   1b: C3
 *   16: C4
 *   1d: C5
 *   1a: C6
 *   17: E1
 *   18: E2
 *
 * button locations:
 *    F1 F2 F3 F4
 *        C4      shift  stick
 *     C3    C5
 *        C1              jog
 *     C2    C6            &
 *                      shuttle
 *      E1  E2
 */

static unsigned char sonyjog_keycode[256] = {
	  0,  0,  0,  0, 30, 48, 46, 32, 18, 33, 34, 35, 23, 36, 37, 38,
	 50, 49, 24, 25, 16, 19, 31, 20, 22, 47, 17, 45, 21, 44,  2,  3,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 13,  0,
	 27, 43,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
};

static signed char sonyjog_shuttle[256] = {
	  0,  0,  0,  0,  1,  2,  3,  4,  5,  6,  7,  8, -1, -2, -3, -4,
	 -5,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -6,  0,
	 -7, -8,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0, 42,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
};

#define USB_VENDOR_ID_SONY	0x054c
#define USB_PRODUCT_ID_JOG	0x000e

struct sonyjog_features {
	char *name;
	int pktlen;
	void (*irq)(struct urb *urb);
};

struct sonyjog {
	unsigned char newdata[8];
	unsigned char olddata[8];
	struct input_dev dev;
	struct usb_device *usbdev;
	struct urb irq;
	struct sonyjog_features *features;
	int open;
};

static void sonyjog_irq(struct urb *urb)
{
	struct sonyjog *sonyjog = urb->context;
	unsigned char *newdata = sonyjog->newdata;
	unsigned char *olddata = sonyjog->olddata;
	struct input_dev *dev = &sonyjog->dev;
	int i;

	if (urb->status)
		return;

#if 0
	printk(KERN_INFO "received: %02x %02x %02x %02x %02x %02x %02x %02x\n",
	    newdata[0], newdata[1], newdata[2], newdata[3],
	    newdata[4], newdata[5], newdata[6], newdata[7]);
#endif

	for (i = 1; i < 2; i++)
		input_report_key(dev, sonyjog_keycode[i + 224], (newdata[0] >> i) & 1);

	for (i = 2; i < 8; i++) {
		if (olddata[i] > 3 && memscan(newdata + 2, olddata[i], 6) == newdata + 8) {
			if (sonyjog_keycode[olddata[i]])
				input_report_key(dev, sonyjog_keycode[olddata[i]], 0);
			else
				info("Unknown key (scancode %#x) released.", olddata[i]);
		}

		if (newdata[i] > 3 && memscan(olddata + 2, newdata[i], 6) == olddata + 8) {
			if (sonyjog_keycode[newdata[i]])
				input_report_key(dev, sonyjog_keycode[newdata[i]], 1);
			else
				info("Unknown key (scancode %#x) pressed.", newdata[i]);
		}

		if (sonyjog_shuttle[newdata[i]] != 0)
			input_report_abs(dev, ABS_MISC, sonyjog_shuttle[newdata[i]]);

		if (newdata[i] == 0x1e)
			input_report_rel(dev, REL_DIAL, -1);
		if (newdata[i] == 0x1f)
			input_report_rel(dev, REL_DIAL, 1);
	}

	memcpy(olddata, newdata, 8);
}

struct sonyjog_features sonyjog_features[] = {
	{ "SONY USB Jog Controller", 8, sonyjog_irq },
	{ NULL , 0 }
};

struct usb_device_id sonyjog_ids[] = {
	{ USB_DEVICE(USB_VENDOR_ID_SONY, USB_PRODUCT_ID_JOG) },
	{ }
};

MODULE_DEVICE_TABLE(usb, sonyjog_ids);

static int sonyjog_open(struct input_dev *dev)
{
	struct sonyjog *sonyjog = dev->private;

	if (sonyjog->open++)
		return 0;

	sonyjog->irq.dev = sonyjog->usbdev;
	if (usb_submit_urb(&sonyjog->irq))
		return -EIO;

	return 0;
}

static void sonyjog_close(struct input_dev *dev)
{
	struct sonyjog *sonyjog = dev->private;

	if (!--sonyjog->open)
		usb_unlink_urb(&sonyjog->irq);
}

static void *sonyjog_probe(struct usb_device *dev, unsigned int ifnum, const struct usb_device_id *id)
{
	struct usb_endpoint_descriptor *endpoint;
	struct sonyjog *sonyjog;
	int i;

	if (!(sonyjog = kmalloc(sizeof(struct sonyjog), GFP_KERNEL)))
		return NULL;
	memset(sonyjog, 0, sizeof(struct sonyjog));

	sonyjog->features = sonyjog_features;

	sonyjog->dev.evbit[0] |= BIT(EV_KEY) | BIT(EV_ABS) | BIT(EV_REL);
	sonyjog->dev.absbit[0] |= BIT(ABS_MISC);
	sonyjog->dev.relbit[0] |= BIT(REL_DIAL);
	for (i = 0; i < 255; i++)
		set_bit(sonyjog_keycode[i], sonyjog->dev.keybit);
	clear_bit(0, sonyjog->dev.keybit);

	sonyjog->dev.absmin[ABS_MISC] = -8;
	sonyjog->dev.absmax[ABS_MISC] = 7;
	sonyjog->dev.absfuzz[ABS_MISC] = 0;

	sonyjog->dev.private = sonyjog;
	sonyjog->dev.open = sonyjog_open;
	sonyjog->dev.close = sonyjog_close;

	sonyjog->dev.name = sonyjog->features->name;
	sonyjog->dev.idbus = BUS_USB;
	sonyjog->dev.idvendor = dev->descriptor.idVendor;
	sonyjog->dev.idproduct = dev->descriptor.idProduct;
	sonyjog->dev.idversion = dev->descriptor.bcdDevice;
	sonyjog->usbdev = dev;

	endpoint = dev->config[0].interface[ifnum].altsetting[0].endpoint + 0;

	FILL_INT_URB(&sonyjog->irq, dev, usb_rcvintpipe(dev, endpoint->bEndpointAddress),
		     sonyjog->newdata, sonyjog->features->pktlen, sonyjog->features->irq, sonyjog, endpoint->bInterval);

	input_register_device(&sonyjog->dev);

	printk(KERN_INFO "input%d: %s on usb%d:%d.%d\n",
		 sonyjog->dev.number, sonyjog->features->name, dev->bus->busnum, dev->devnum, ifnum);

	return sonyjog;
}

static void sonyjog_disconnect(struct usb_device *dev, void *ptr)
{
	struct sonyjog *sonyjog = ptr;
	usb_unlink_urb(&sonyjog->irq);
	input_unregister_device(&sonyjog->dev);
	kfree(sonyjog);
}

static struct usb_driver sonyjog_driver = {
	name:		"sonyjog",
	probe:		sonyjog_probe,
	disconnect:	sonyjog_disconnect,
	id_table:	sonyjog_ids,
};

static int __init sonyjog_init(void)
{
	usb_register(&sonyjog_driver);
	info(DRIVER_VERSION ":" DRIVER_DESC);
	return 0;
}

static void __exit sonyjog_exit(void)
{
	usb_deregister(&sonyjog_driver);
}

module_init(sonyjog_init);
module_exit(sonyjog_exit);
