/***************************************************************************
                              pluginsimp.cpp
                             -------------------
    begin                : Tuesday Nov 4 2001
    copyright            : (C) 2001 Benjamin Meyer
    email                : benjamin@csh.rit.edu
 ***************************************************************************/


#include "pluginsimp.h"
#include <qpushbutton.h>
#include <qlabel.h>
#include <qstring.h>
#include <qlistbox.h>
#include <qdir.h>
#include <qgroupbox.h>
#include <qwidgetstack.h>
#include "klibloader.h"
#include "kfiledialog.h"
#include "kinkattaplugin.h"
#include "pluginpriorityimp.h"
#include "preferences.h"
#include "application.h"

#include "buddylistwindow.h"
#include "taim.h"

/**
 * Connects all of the buttons to the slots.  Sets up initial values.
 *  Creates the plugin setting dir.
 * @param parent the parent widget
 * @param name the name of this widget
 */
PluginsImp::PluginsImp(QWidget *parent, const char *name) : Plugins( parent,name,false ){
  connect(buttonLoad, SIGNAL(clicked()), this, SLOT(loadPluginButtonClicked()));
  connect(buttonUnload, SIGNAL(clicked()), this, SLOT(unloadPlugin()));
  connect(buttonReload, SIGNAL(clicked()), this, SLOT(reloadPlugin()));
  connect(buttonPrioritize, SIGNAL(clicked()), this, SLOT(priorityClicked()));
  connect(buttonOk, SIGNAL(clicked()), this, SLOT(okClicked()));
  connect(buttonApply, SIGNAL(clicked()), this, SLOT(buttonApplyClicked()));
  connect(pluginList, SIGNAL(highlighted(int)), this, SLOT(itemSelected(int)));
  currentConfigure = NULL;
  connection = NULL;
  mainWindow = NULL;
  priority = NULL;

  QString DirStr = QDir::homeDirPath() + "/.kinkatta/" ;
  QDir pluginDir( DirStr );
  pluginDir.mkdir( DirStr );
  pluginDir.mkdir( ( DirStr + "plugins/" ) );

}

/**
 * Deletes any plugins that are still here.
 */
PluginsImp::~PluginsImp(){
  // Unload any plugins that are currently loaded.
  KinkattaPlugin *plugin = plugins.first();
  while(plugin != NULL){
    plugins.remove(plugin);
    QString location = plugin->physicalLocation;
    delete plugin;
    KLibLoader *loader = KLibLoader::self(); 
    if( !loader ) return;
      loader->unloadLibrary( location.latin1());
    plugin = plugins.next(); 
  }
}

/**
 * Set our pointers and then load any plugins in our settings.
 * @param cMainWindow pointer to the mainwindow.
 * @param cConnection pointer to the network connection.
 */
void PluginsImp::setPointers(BuddyListWindow *cMainWindow, TAim *cConnection){
  mainWindow = cMainWindow;
  connection = cConnection;
  setUpConnections();

  // Read Settings.
  QString DirStr = QDir::homeDirPath() + "/.kinkatta/plugins/plugins.settings" ;
  Preferences prefs(DirStr, "Kinkatta plugins settings", "1.0");

  prefs.setGroup("plugins");
  int pluginCount = prefs.getNumber("plugin_count");

  for(int i = 0; i < pluginCount; i++){
    QString pluginLocation = prefs.getString((QString("plugin_%1_").arg(i) + "location"), "" );
    loadPlugin(pluginLocation);
    //qDebug(pluginLocation.latin1());
  }
}

/**
 * Hide this window.
 */
void PluginsImp::okClicked(){
  this->hide();
}

/**
 * Open the priority dialog box so that they can be sorted by priority.
 */
void PluginsImp::priorityClicked(){
  if( pluginList->count() < 1 )
    return;
  if( pluginList->currentItem() == -1 )
    return;

  // Bring up priority box.
  if( priority == NULL ){
     priority = new PluginPriorityImp(this, "Priority Box");
     connect(priority, SIGNAL(setPluginList(QList<KinkattaPlugin>)),
             this, SLOT(updatePriority(QList<KinkattaPlugin> )));
     priority->setIcon( *(this->icon()) );
  }
  priority->setPlugins(plugins);
  priority->show();
}

/**
 * Change the list of plugins with this new list, hide the priory dialog.
 * @param plugins_new the new ordered list of plugins.
 */
void PluginsImp::updatePriority( QList<KinkattaPlugin> plugins_new ){
  plugins = plugins_new;
  priority->hide();
}

/**
 * Load a new plugin. Bring up the file dialog.
 */
void PluginsImp::loadPluginButtonClicked(){
  QString libFileName( KFileDialog::getOpenFileName( DATADIR "/plugins" ,QString("*.*"),this, "Libaries.") );
  
  if ( libFileName.isEmpty() == true) return;
  //qDebug("We have a filename.");
  loadPlugin( libFileName );
  saveSettings();
}

/**
 * Attempt to load a plugin.
 * @param libFileName file to try to load.
 */  
void PluginsImp::loadPlugin( QString libFileName ){
  KLibLoader *loader = KLibLoader::self(); 
  if( !loader ) return;
  //qDebug("We have a loader.");
  
  KLibrary *lib = loader->library(libFileName.latin1());
  if( !lib ) return;
  //qDebug("We have a library.");

  void *cplugin = lib->symbol("create_plugin");
  if( !cplugin ) return;
  //qDebug("We have a plugin.");

  KinkattaPlugin* (*plugInStart)();
  plugInStart= (KinkattaPlugin* (*)()) cplugin;
  KinkattaPlugin* plugin = plugInStart();
  // We now have a plugin.
  plugin->physicalLocation = libFileName;
  plugin->setPointers(mainWindow, connection);
  plugin->kinkattaVersionNumber = release_number;
  plugin->initPlugin();

  plugins.append(plugin);
  pluginList->insertItem(plugin->getName());
  labelName->setText( QString("Name: ") + plugin->getName() );
  labelVersion->setText( QString("Version: ") + plugin->getVersion() );
  labelAuthor->setText( QString("Author: ") + plugin->getAuthor() );
  //qDebug("Setting up connections.");
  setUpConnections();
  //qDebug("All done setting up connections.");
  if (pluginList->count() > 0 )
    pluginList->setCurrentItem(pluginList->topItem());

  //delete lib;
  loader->unloadLibrary(libFileName.latin1());
}

/**
 * Unload the current plugin.
 */
void PluginsImp::unloadPlugin(){
  unloadPlugin( false );
  saveSettings();
}

/**
 * unload a plugin.
 * @param reload If this plugin should be reloaded or not.
 */
void PluginsImp::unloadPlugin( bool reload ){
  if( pluginList->count() < 1 )
    return;
  if( pluginList->currentItem() == -1 )
    return;

  QString name = pluginList->currentText();  
  int currentItem = pluginList->currentItem();

  KinkattaPlugin *plugin = plugins.first();
  while(plugin != NULL){
    if ( plugin->getName() == name )
      break;
    plugin = plugins.next();
  }
  if( plugin == NULL ) return;
  
  // Hide any configure information
  QWidget *conf =  plugin->configure( disc );
  if( conf == currentConfigure ){
    labelName->setText( QString("Name: Not Set") );
    labelVersion->setText( QString("Version: 0.00") );
    labelAuthor->setText( QString("Author: Not Set") );
    conf->hide();
  }

  QString location = plugin->physicalLocation;
  plugins.remove(plugin);
  delete plugin;
  pluginList->removeItem(currentItem);

  KLibLoader *loader = KLibLoader::self(); 
  if( !loader ) return;
  //loader->unloadLibrary(location.latin1());
  //delete the library;
  
  setUpConnections();

  if( pluginList->count() < 1 ){
    disc->setEnabled( false );
    buttonUnload->setEnabled( false );
    buttonReload->setEnabled( false );
    buttonPrioritize->setEnabled( false );
  }

  if ( reload ){
    loadPlugin( location );
  }
  else{
    if (pluginList->count() == 1 )
      pluginList->setCurrentItem(pluginList->topItem());
  }
}

/** 
 * Reload the currently selected plugin.
 */
void PluginsImp::reloadPlugin(){
  unloadPlugin( true );
}

/**
 * Tells the current plugin to apply it's settings.
 */
void PluginsImp::buttonApplyClicked(){
  if( pluginList->count() < 1 )
    return;
  if( pluginList->currentItem() == -1 )
    return;

  QString name = pluginList->currentText();  
  // Find the plugin and call the apply function.
  KinkattaPlugin *plugin = plugins.first();
  while(plugin != NULL){
    if ( plugin->getName() == name )
      break;
    plugin = plugins.next();
  }
  if( plugin != NULL ){
    plugin->applyConfigure();
    plugin->saveSettings();
  }
  else
    qDebug("There should be a plugin to apply the settings too...");

  saveSettings();
}

/**
 * Based upon a plugin priority the plugins are connected through the system.
 */
void PluginsImp::setUpConnections(){
  // First thing we want to do is disconnect all of the signals.
  disconnect( mainWindow, SIGNAL(sendMessageChat(QString, QString, bool, bool)),0,0);
  disconnect( connection, SIGNAL(IMReceived(QString, QString, bool)),0,0);
  KinkattaPlugin *plugin = plugins.first();
  while(plugin){
    disconnect( plugin, SIGNAL(goingOutChatMessage(QString, QString, bool, bool)),0,0);
    disconnect( plugin, SIGNAL(goingInChatMessage(QString, QString, bool)),0,0);
    plugin = plugins.next();
  }

#define getNextPlugin(a, b) \
  { \
    if( a == false ) \
      b = plugins.next(); \
    else \
      b = plugins.first(); \
    plugin_info info; \
    if( b == NULL) \
      qDebug("ERROR #1 in setting up the plugins"); \
    else \
      info = b->getInformation(); \
    while( info.messagePlugin == false ){ \
      b = plugins.next(); \
      if( b == NULL) \
        qDebug("ERROR #2 in setting up the plugins"); \
      else \
        info = b->getInformation(); \
    } \
  }
  
  // Count the plugins
  int pluginCount = plugins.count();
  //qDebug(QString("%1").arg(pluginCount).latin1());
  if( pluginCount > 0 ) {
    int realCount = 0;
    plugin = plugins.first();
    for( int i = 0; i < pluginCount; i++ ) {
      plugin_info info = plugin->getInformation();
      if( info.messagePlugin == true )
        realCount++;
      plugin = plugins.next();
    }
    pluginCount = realCount;
  }
  //qDebug(QString("%1").arg(pluginCount).latin1());
  // There are 0 plugins.
  if (pluginCount == 0){
    connect(mainWindow, SIGNAL(sendMessageChat(QString, QString, bool, bool)), connection, SLOT(sendIM(QString, QString, bool, bool)));
    connect(connection, SIGNAL(IMReceived(QString, QString, bool)), mainWindow, SLOT(receivedIM(QString, QString, bool)));
    return;
  }

  getNextPlugin( true, plugin );
  connect(mainWindow, SIGNAL(sendMessageChat(QString, QString, bool, bool)), plugin, SLOT(outChatMessage(QString, QString, bool, bool)));
  connect(connection, SIGNAL(IMReceived(QString, QString, bool)), plugin, SLOT(inChatMessage(QString, QString, bool)));
  
  // There is only 1 plugin.
  if (pluginCount == 1){
    connect(plugin, SIGNAL(goingOutChatMessage(QString, QString, bool, bool)), connection, SLOT(sendIM(QString, QString, bool, bool)));
    connect(plugin, SIGNAL(goingInChatMessage(QString, QString, bool)), mainWindow, SLOT(receivedIM(QString, QString, bool)));
    return;
  }
  
  KinkattaPlugin *nextPlugin;
  getNextPlugin( false , nextPlugin ); 
  for(int i = 0; i < pluginCount-1; i++){
    connect(plugin, SIGNAL(goingOutChatMessage(QString, QString, bool, bool)), nextPlugin, SLOT(outChatMessage(QString, QString, bool, bool)));
    connect(plugin, SIGNAL(goingInChatMessage(QString, QString, bool)), nextPlugin, SLOT(inChatMessage(QString, QString, bool)));
    plugin = nextPlugin;
    if( i < pluginCount-2 )
      getNextPlugin( false , nextPlugin );
  }
  
  connect(plugin, SIGNAL(goingOutChatMessage(QString, QString, bool, bool)), connection, SLOT(sendIM(QString, QString, bool, bool)));
  connect(plugin, SIGNAL(goingInChatMessage(QString, QString, bool)), mainWindow, SLOT(receivedIM(QString, QString, bool)));
}

/**
 * Pulls the plugins configure to the top.
 * @param int the current plugin selected.
 */ 
void PluginsImp::itemSelected( int ){
  if( pluginList->count() > 0 ){
    disc->setEnabled( true );
    buttonUnload->setEnabled( true );
    buttonReload->setEnabled( true );
    buttonPrioritize->setEnabled( true );
  }
  // Get configure widget stuff
  QString name = pluginList->currentText();  

  KinkattaPlugin *plugin = plugins.first();
  while(plugin != NULL){
    if ( plugin->getName() == name )
      break;
    plugin = plugins.next();
  }
  if( plugin == NULL ) return;
 
  QWidget *conf =  plugin->configure( disc );
  conf->show();
  configureWidget->addWidget(conf, -1);
  configureWidget->raiseWidget(conf);
  labelName->setText( QString("Name: ") + plugin->getName() );
  labelVersion->setText( QString("Version: ") + plugin->getVersion() );
  labelAuthor->setText( QString("Author: ") + plugin->getAuthor() );
}

/**
 * Tell each plugin to save its settings.
 */
void PluginsImp::saveSettings(){
  //qDebug("Saving settings.");
  KinkattaPlugin *plugin = plugins.first();
  while(plugin){
    plugin->saveSettings();
    plugin = plugins.next();
  }

  // Save the current set of loaded plugins.
  QString DirStr = QDir::homeDirPath() + "/.kinkatta/plugins/plugins.settings";
  QFile::remove(DirStr);
  Preferences prefs(DirStr, "Kinkatta plugins settings", "1.0");
  prefs.setGroup("plugins");
  plugin = plugins.first();
  int i = 0;
  while(plugin){
    prefs.setString(QString("plugin_%1_").arg(i) + "location",plugin->physicalLocation);
    i++;
    plugin = plugins.next();
  }
  prefs.setNumber("plugin_count", i);
  prefs.flush();
}

// PluginsImp.cpp

