/*
  Copyright Mission Critical Linux, 2000

  Kimberlite is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2, or (at your option) any
  later version.

  Kimberlite is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with Kimberlite; see the file COPYING.  If not, write to the
  Free Software Foundation, Inc.,  675 Mass Ave, Cambridge, 
  MA 02139, USA.

 * Author: Jeff Moyer <moyer@missioncriticallinux.com>
 * $Revision: 1.8 $
 *
 */

#ifndef __MCL_MSG_H
#define __MCL_MSG_H
#ifdef __cplusplus
extern "C" {
#endif

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <unistd.h>
#include <clusterdefs.h>
#include <time.h>
#include <sys/types.h>

typedef int msg_handle_t;
typedef unsigned int msg_addr_t;

#define MSGSVC_VERSION           1   /* Consistency check */
#define MAX_MSGBACKLOG          15   /* Maximum backlog of connections.   */
#define MSGSVC_CONNECT_TIMEOUT   5   /* Connect timeout */

#define MAX_DAEMON_NAMELEN 255  /* Used to convert procid to daemon name */
/*
 * Method for identifying a process for the message subsystem.
 * NOTE: these identifiers correspond to the elements in the proc_id_array.
 * If you change the order, or add an element, be sure to update this array
 * in msg.c.
 */
#define PROCID_HEARTBEAT  0
#define PROCID_SVCMGR     1
#define PROCID_QUORUMD	  2
#define PROCID_POWERD     3

#define MAX_PROCID        3

#define PROCID_NONE       (MAX_PROCID+1)

/*
 *  Service specific messages.
 */

/*
 * Heartbeat
 */
#define HB_QUERY_NETUP          1
#define HB_TERM                 2

/*
 * Quorum Daemon
 */
#define DISK_NODE_STATECHANGE	1  /* From quorumd to SM     */
#define DISK_SM_ALIVE		3  /* From SM to quorumd     */
#define DISK_SM_EXITING		4  /* From SM to quorumd     */
#define DISK_NODE_STATES	5  /* From monitor to quorumd */
#define DISK_POWERD_ALIVE	6  /* From powerd to quorumd */
#define DISK_CONFIG_CHANGE      7  /* From config utility to quorumd */

/*
 * Power Switch Daemon
 */
#define PSWITCH_QUERY           1  /* From quorumd to powerd */
#define PSWITCH_RESET           2  /* From quorumd to powerd */
#define PSWITCH_TERM            3
#define PSWITCH_SET_LOGLEVEL    4  /* Set the current logging level */
#define PSWITCH_ACK             5

/* Power switch states */
#define PSWITCH_OK              1
#define PSWITCH_ERR             2

/*
 * Data structures
 */
#define GENERIC_HDR_MAGIC 0xf4982734 /* arbitrarily chosen  */

typedef struct {
    unsigned long magic; 
    unsigned int  command;
    ssize_t length;
} generic_msg_hdr;
/*
 * Message structure passed between quorumd & SM
 * XXX - may be better off in a different header.
 */
/*
 * Queries the state of an individual node.
 */
typedef struct {
    int nodeNumber;
    int nodeStatus;
} NodeStatusMsg;
/*
 * This structure provides a means of returning a query for the
 * current node states for all nodes in the cluster.
 * The values are in clusterdefs.h
 */
typedef struct {
        int states[MAX_NODES];
} SharedDiskNodeStates;   


/* Currently only used to set loglevel */
typedef struct { int arg; } PswitchCmdMsg;

typedef struct {
    int    status;    /* DISK_PSWITCH_X */
    time_t timestamp;
} PswitchStatusMsg;

typedef struct {
    generic_msg_hdr hdr;
    union {
	SharedDiskNodeStates nodeStates;
	int              nodeNumber;
	NodeStatusMsg    statusMsg;
	pid_t		 daemonPid;
    } data;
} DiskMessageSt;
#define DISK_MESSAGE_SIZE (sizeof(DiskMessageSt))

/*
 * API For manipulating file descriptor based communications channels.
 */
/*
 * msg_handle_t msg_open(msg_addr_t dest, int nodeid)
 *
 * DESCRIPTION
 *   Open a communication path to the daemon associated with dest on
 *   node 'nodeid.'
 *
 * ARGUMENTS
 *   dest  Address to which the connection is made.  Currently, one of:
 *
 *         PROCID_HEARTBEAT
 *         PROCID_SVCMGR
 *         PROCID_QUORUMD
 *         PROCID_ADMIND
 *
 *   nodeid  Node where the daemon you wish to contact lives.  This value
 *           corresponds to that returned from get_clu_cfg(), in the lid field.
 *
 * RETURN VALUES
 *   Upon success, a valid file descriptor is returned.  For failure, -1 is
 *   returned.
 */
msg_handle_t msg_open(msg_addr_t dest, int nodeid);

/*
 * void msg_close(msg_handle_t handle)
 *
 * DESCRIPTION
 *   Close an open msg_handle_t.  It is _required_ that you call this, as
 *   the msg svc keeps internal tables of file descriptors and associated
 *   states.
 *
 * ARGUMENTS
 *   handle  Handle returned by msg_open or msg_accept which you will no
 *           longer use.
 *
 * RETURN VALUES
 *   none
 */
void msg_close(msg_handle_t handle);

/*
 * msg_handle_t msg_listen(msg_addr_t my_proc_id)
 *
 * DESCRIPTION
 *   Create a msg_handle_t which is ready to accept incoming connections.
 *
 * ARGUMENTS
 *   my_proc_id  Address on which to listen.  PROCID_XXX corresponds to a
 *               TCP/IP port on which this daemon will listen.
 *
 * RETURN VALUES
 *   On success, a valid file descriptor is returned.  On error, -1 is 
 *   returned.
 */
msg_handle_t msg_listen(msg_addr_t my_proc_id);

/*
 * msg_handle_t msg_accept(msg_handle_t handle)
 *
 * DESCRIPTION
 *   Call accept on a file descriptor returned from msg_listen.
 *
 * ARGUMENTS
 *   handle  Valid handle returned from msg_listen.
 *
 * RETURN VALUES
 *   On success, a valid file descriptor is returned which describes the new
 *   communication channel.  On error, -1 is returned.
 */
msg_handle_t msg_accept(msg_handle_t handle);

/*
 * msg_handle_t msg_accept_timeout(msg_handle_t handle, int timeout)
 *
 * DESCRIPTION
 *   Call accept on a file descriptor.  If no connections are pending within
 *   timeout seconds, the function returns.
 *
 * ARGUMENTS
 *   handle   valid handle returned by msg_listen.
 *   timeout  time in seconds to wait for a connection.
 *
 * RETURN VALUES
 *   If a connection is pending, a valid file descriptor for that connection
 *   is returned.  If no connections are pending, 0 is returned.  On error,
 *   -1 is returned.
 */
msg_handle_t msg_accept_timeout(msg_handle_t handle, int timeout);

/*
 * int msg_send(msg_handle_t handle, void *buf, ssize_t count)
 *
 * DESCRIPTION
 *   Send a message over the communications channel described by handle.
 *
 * ARGUMENTS
 *   handle  Valid handle returned from msg_open or msg_accept[_timeout].
 *   buf     Pointer to data to be sent.
 *   count   Number of bytes to send.
 *
 * RETURN VALUES
 *   On success, the number of bytes successfully written is returned.  On
 *   error, -1 is returned, and errno is set according to write(2).
 */
int msg_send(msg_handle_t handle, void *buf, ssize_t count);

/*
 * int __msg_send(msg_handle_t handle, void *buf, ssize_t count)
 *
 * DESCRIPTION
 *   Send a message over the communications channel described by handle.  This
 *   call differs from msg_send in that it does no sanity checking of internal
 *   file descriptor tables.  Use this call if you hand craft a connection,
 *   and would like the message service to take care of the communications.
 *
 * ARGUMENTS
 *   handle  Valid file descriptor.
 *   buf     Pointer to data to be sent.
 *   count   Number of bytes to send.
 *
 * RETURN VALUES
 *   On success, the number of bytes successfully written is returned.  On
 *   error, -1 is returned, and errno is set according to write(2).
 */
int __msg_send(msg_handle_t handle, void *buf, ssize_t count);

/*
 * ssize_t msg_receive(msg_handle_t handle, void *buf, ssize_t count)
 *
 * DESCRIPTION
 *   Receive a message off of the communications channel described by handle.
 *
 * ARGUMENTS
 *   handle  Valid handle returned by a call to msg_accept or msg_open.
 *   buf     Buffer into which the received data is copied.
 *   count   Number of bytes to read from msg_handle_t.
 *
 * RETURN VALUES
 *   On success, the number of bytes successfully read is returned.  On error, 
 *   -1 is returned, and errno is set according to read(2).
 */
ssize_t msg_receive(msg_handle_t handle, void *buf, ssize_t count, int *auth);

/*
 * ssize_t __msg_receive(msg_handle_t handle, void *buf, ssize_t count)
 *
 * DESCRIPTION
 *   Receive a message off of the communications channel described by handle.
 *   This call differs from msg_receive in that it does no sanity checking
 *   of internal file descriptor tables.  Use this call if you hand craft a 
 *   connection, and would like the message service to take care of
 *   communications.
 *
 * ARGUMENTS
 *   handle  Valid file descriptor.
 *   buf     Buffer into which the received data is copied.
 *   count   Number of bytes to read from file descriptor.
 *
 * RETURN VALUES
 *   On success, the number of bytes successfully read is returned.  On error, 
 *   -1 is returned, and errno is set according to read(2).
 */
ssize_t __msg_receive(msg_handle_t handle, void *buf, ssize_t count, int *auth);

/*
 * ssize_t msg_receive_timeout(msg_handle_t handle, void *buf,
 *			   ssize_t count, unsigned int timeout);
 *
 * DESCRIPTION
 *   Receive a message on a given communications channel.  If no message
 *   is available within timeout seconds, the call returns.
 *
 * ARGUMENTS
 *   handle  Valid file descriptor returned from a call to msg_open or
 *           msg_accept[_timeout].
 *   buf     Buffer into which the received data is copied.
 *   count   Number of bytes to read from handle.
 *   timeout Time in seconds to wait for a message to arrive.
 *
 * RETURN VALUES
 *   If there is data to be read within timeout seconds, the number
 *   of bytes read is returned.  If the timeout expires before data
 *   is ready, 0 is returned.  On error, -1 is returned and errno is
 *   set according to one of select(2) or read(2).
 */
ssize_t msg_receive_timeout(msg_handle_t handle, void *buf,
			   ssize_t count, int *auth, unsigned int timeout);

/*
 * ssize_t msg_peek(msg_handle_t handle, void *buf, ssize_t count)
 *
 * DESCRIPTION
 *   Check to see if there is data to read from the socket, but do not
 *   retrieve the data.  For a more detailed explanation, see man 2 recv and
 *   search for MSG_PEEK.
 *
 * ARGUMENTS
 *   handle  Handle returned by msg_open or msg_accept.
 *   buf     Buffer into which the available data is copied
 *   count   Number of bytes to read.
 *
 * RETURN VALUES
 *   On success, the number of bytes available for reading is returned, and
 *   buf is filled with that number of bytes from the connection.  0 is 
 *   returned if there is nothing available for reading.  On error, -1 is
 *   returned.
 */
ssize_t msg_peek(msg_handle_t handle, void *buf, ssize_t count);

/*
 * ssize_t __msg_peek(msg_handle_t handle, void *buf, ssize_t count)
 *
 * DESCRIPTION
 *   Check to see if there is data to read from the socket, but do not
 *   retrieve the data.  For a more detailed explanation, see man 2 recv and
 *   search for MSG_PEEK.  This call differs from msg_peek in that it does
 *   no sanity checks on the input parameters.
 *
 * ARGUMENTS
 *   handle  Handle returned by msg_open or msg_accept.
 *   buf     Buffer into which the available data is copied
 *   count   Number of bytes to read.
 *
 * RETURN VALUES
 *   On success, the number of bytes available for reading is returned, and
 *   buf is filled with that number of bytes from the connection.  0 is 
 *   returned if there is nothing available for reading.  On error, -1 is
 *   returned.
 */
ssize_t __msg_peek(msg_handle_t handle, void *buf, ssize_t count);


#ifdef __cplusplus
}
#endif
#endif /* __MCL_MSG_H */
