/***************************************************************************
 *   Copyright (C) 2004 by Emil Stoyanov                                   *
 *   emosto@users.sourceforge.net                                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "kiaxprefs.h"

KiaxPrefs::KiaxPrefs(QObject *parent, const char *name)
    : QObject(parent, name)
{
  setInputDevice(0);
  setOutputDevice(0);
  setRingDevice(0);
  setFilterFlag(IAXC_FILTER_AAGC|IAXC_FILTER_ECHO|IAXC_FILTER_DENOISE|IAXC_FILTER_CN);
  silenceThreshold = -99; // default
}

KiaxPrefs::~KiaxPrefs()
{}

void KiaxPrefs::loadSettings()
{
  loadAccounts();
  // load device settings
  loadDeviceSettings();
  // load filter settings
  loadFilterSettings();
  // load miscellaneous settings
  loadMiscSettings();
  // load session settings
  loadSessionSettings();
}

void KiaxPrefs::loadAccounts()
{
  QSettings * accSettings = getSettings();
  // read stored account names
  QStringList accs = accSettings->subkeyList("/kiax/accounts");
  defaultAccountId = accSettings->readEntry("/kiax/accounts/defaultAccountId","0");
  delete accSettings;
  
  // prepare the accounts vector
  accounts.setAutoDelete(true);
  accounts.clear();
  for (QStringList::Iterator it = accs.begin();it!=accs.end(); ++it) {
    // get the name of the next acount
    QString nextAcc = *it;
    // create an instance with that name
    KiaxAccount * account = new KiaxAccount(nextAcc);
    // set the values
    account->load();
    // and append the account to the list of accounts
    accounts.append(account);
  }
}

void KiaxPrefs::saveSettings()
{
  // save all accounts
  for (uint i=0;i<accounts.count();i++) saveAccount(i);
  // save the defaultAccountId
  QSettings * accSettings = getSettings();
  accSettings->writeEntry("/kiax/accounts/defaultAccountId",defaultAccountId);
  delete accSettings;
  // save device settings
  saveDeviceSettings();
  // save filter settings
  saveFilterSettings();
  // save miscellaneous settings
  saveMiscSettings();
}

KiaxAccount* KiaxPrefs::getAccount(int i)
{
  return accounts.at(i);
}

QPtrList<KiaxAccount> KiaxPrefs:: getAccounts()
{
  return accounts;
}

int KiaxPrefs::createNewAccount(QString accName)
{
  KiaxAccount* newAcc = new KiaxAccount(accName);
  addAccount(newAcc);
  return accounts.count()-1;
}

void KiaxPrefs::addAccount(KiaxAccount* acc)
{
  accounts.append(acc);
}

void KiaxPrefs::removeAccount(int accNumber)
{
  KiaxAccount* acc = accounts.at(accNumber);
  acc->remove();
  accounts.remove(accNumber);
}

void KiaxPrefs::saveAccount(int accNumber)
{
  KiaxAccount* acc = accounts.at(accNumber);
  acc->save();

}

QString KiaxPrefs::getDefaultAccountId()
{
  return defaultAccountId;
}

void KiaxPrefs::setDefaultAccountId(QString df)
{
  defaultAccountId = df;
}

void KiaxPrefs::detectDevices()
{

  struct iaxc_audio_device *devices;
  int devNumber,input,output,ring;
  long devCapabilities;
  QString * devName;
  int result = iaxc_audio_devices_get(&devices, &devNumber, &input, &output, &ring);
  fprintf(stderr, "iaxc_aduio_devices_get = %d\n", result);

  inputDeviceNames.clear();
  outputDeviceNames.clear();
  ringDeviceNames.clear();

  for(int i=0; i<devNumber; i++)
  {
    devCapabilities = devices->capabilities;
    devName = new QString(devices->name);
    if(devCapabilities & IAXC_AD_INPUT)  inputDeviceNames.append(devName);
    if(devCapabilities & IAXC_AD_OUTPUT) outputDeviceNames.append(devName);
    if(devCapabilities & IAXC_AD_RING) ringDeviceNames.append(devName);
    devices++;
  }

}

void KiaxPrefs::updateDevices()
{
  iaxc_audio_devices_set(inputDevice,outputDevice,ringDevice);
}

void KiaxPrefs::loadDeviceSettings()
{
  QSettings * settings = getSettings();
  inputDevice = settings->readNumEntry("/kiax/devices/inputDevice",0);
  outputDevice = settings->readNumEntry("/kiax/devices/outputDevice",0);
  ringDevice = settings->readNumEntry("/kiax/devices/ringDevice",0);
  delete settings;
}

void KiaxPrefs::saveDeviceSettings()
{
  QSettings * settings = getSettings();
  settings->writeEntry("/kiax/devices/inputDevice", inputDevice);
  settings->writeEntry("/kiax/devices/outputDevice", outputDevice);
  settings->writeEntry("/kiax/devices/ringDevice", ringDevice);
  delete settings;
}

void KiaxPrefs::saveMiscSettings()
{
  QSettings * settings = getSettings();
  settings->writeEntry("/kiax/misc/docked", docked);
  settings->writeEntry("/kiax/misc/hiddenOnStartup", hiddenOnStartup);
  settings->writeEntry("/kiax/misc/connectedOnStartup", connectedOnStartup);
  settings->writeEntry("/kiax/misc/raiseWindow", raiseWindow);
  settings->writeEntry("/kiax/misc/maxCalls", maxCallsNumber);
  settings->writeEntry("/kiax/misc/ExecuteCommandOnRing", execCommandOnRing);
  settings->writeEntry("/kiax/misc/CommandOnRing", commandOnRing);
  delete settings;
}

void KiaxPrefs::loadMiscSettings()
{
  QSettings * settings = getSettings();
  setDocked(settings->readBoolEntry("/kiax/misc/docked",false));
  setHiddenOnStartup(settings->readBoolEntry("/kiax/misc/hiddenOnStartup",false));
  setConnectedOnStartup(settings->readBoolEntry("/kiax/misc/connectedOnStartup",true));
  setRaiseWindow(settings->readBoolEntry("/kiax/misc/raiseWindow",true));
  setMaxCalls(settings->readNumEntry("/kiax/misc/maxCalls",2));
  setExecCommandOnRing(settings->readBoolEntry("/kiax/misc/ExecuteCommandOnRing", false));
  setCommandOnRing(settings->readEntry("/kiax/misc/CommandOnRing", ""));
  delete settings;
}

void KiaxPrefs::loadFilterSettings()
{
  QSettings * settings = getSettings();
  filterFlag = 0;
  if (settings->readBoolEntry("/kiax/filters/NoiseReduction",true)) filterFlag |= IAXC_FILTER_DENOISE;
  if (settings->readBoolEntry("/kiax/filters/AGC",true)) filterFlag |= IAXC_FILTER_AGC;
  if (settings->readBoolEntry("/kiax/filters/EchoCancelation",false)) filterFlag |= IAXC_FILTER_ECHO;
  if (settings->readBoolEntry("/kiax/filters/CNFrames",true)) filterFlag |= IAXC_FILTER_CN;
  if (settings->readBoolEntry("/kiax/filters/AAGC",true)) filterFlag |= IAXC_FILTER_AAGC;
  incomingRingMute = settings->readBoolEntry("/kiax/filters/MuteIncomingRing",false);
  outgoingRingMute = settings->readBoolEntry("/kiax/filters/MuteOutgoingRing",false);
  silenceThreshold = settings->readNumEntry("/kiax/filters/SilenceThreshold",-99);
  delete settings;
}

void KiaxPrefs::saveFilterSettings()
{
  QSettings * settings = getSettings();
  settings->writeEntry("/kiax/filters/NoiseReduction", (filterFlag & IAXC_FILTER_DENOISE)==IAXC_FILTER_DENOISE);
  settings->writeEntry("/kiax/filters/AGC",(filterFlag & IAXC_FILTER_AGC)==IAXC_FILTER_AGC);
  settings->writeEntry("/kiax/filters/EchoCancelation",(filterFlag & IAXC_FILTER_ECHO)==IAXC_FILTER_ECHO);
  settings->writeEntry("/kiax/filters/CNFrames", (filterFlag & IAXC_FILTER_CN)==IAXC_FILTER_CN);
  settings->writeEntry("/kiax/filters/AAGC", (filterFlag & IAXC_FILTER_AAGC)==IAXC_FILTER_AAGC);
  settings->writeEntry("/kiax/filters/SilenceThreshold", silenceThreshold);
  settings->writeEntry("/kiax/filters/MuteIncomingRing", incomingRingMute);
  settings->writeEntry("/kiax/filters/MuteOutgoingRing", outgoingRingMute);
  delete settings;
}

void KiaxPrefs::loadSessionSettings() {
  QSettings * settings = getSettings();
  showDialHangupButtons = settings->readBoolEntry("/kiax/session/showDialHangupButtons",true);
  showAccountNumberCombos = settings->readBoolEntry("/kiax/session/showAccountNumberCombos",true);
  showDialPad = settings->readBoolEntry("/kiax/session/showDialPad",true);
  showToolbar = settings->readBoolEntry("/kiax/session/showToolbar",true);
  showCallRegister = settings->readBoolEntry("/kiax/session/showCallRegister",true);
  preferredLanguage = settings->readEntry("/kiax/session/preferredLanguage",QString(QTextCodec::locale()));
  
  int winPosX = settings->readNumEntry("/kiax/session/winPosX", 20);
  int winPosY = settings->readNumEntry("/kiax/session/winPosY", 20);
  int widgetTop = settings->readNumEntry("/kiax/session/widgetGeometryTop", 20);
  int widgetBottom = settings->readNumEntry("/kiax/session/widgetGeometryBottom", 520);
  int widgetLeft = settings->readNumEntry("/kiax/session/widgetGeometryLeft", 20);
  int widgetRight = settings->readNumEntry("/kiax/session/widgetGeometryRight", 280);
  
  windowPosition.setX(winPosX);
  windowPosition.setY(winPosY);
  widgetGeometry.setTop(widgetTop);
  widgetGeometry.setBottom(widgetBottom);
  widgetGeometry.setLeft(widgetLeft);
  widgetGeometry.setRight(widgetRight);
  
  delete settings;
}

void KiaxPrefs::saveSessionSettings() {
  
  int winPosX = windowPosition.x();
  int winPosY = windowPosition.y();
  int wLeft = widgetGeometry.left();
  int wRight = widgetGeometry.right();
  int wTop = widgetGeometry.top();
  int wBottom = widgetGeometry.bottom();
  
  QSettings * settings = getSettings();
  settings->writeEntry("/kiax/session/showDialHangupButtons", showDialHangupButtons);
  settings->writeEntry("/kiax/session/showAccountNumberCombos", showAccountNumberCombos);
  settings->writeEntry("/kiax/session/showDialPad", showDialPad);
  settings->writeEntry("/kiax/session/showToolbar", showToolbar);
  settings->writeEntry("/kiax/session/showCallRegister", showCallRegister);
  settings->writeEntry("/kiax/session/preferredLanguage", preferredLanguage);
  settings->writeEntry("/kiax/session/winPosX", winPosX);
  settings->writeEntry("/kiax/session/winPosY", winPosY);
  settings->writeEntry("/kiax/session/widgetGeometryLeft", wLeft);
  settings->writeEntry("/kiax/session/widgetGeometryRight", wRight);
  settings->writeEntry("/kiax/session/widgetGeometryTop", wTop);
  settings->writeEntry("/kiax/session/widgetGeometryBottom", wBottom);
  
  delete settings;
}
 
KiaxAccount * KiaxPrefs::findAccountById(QString id) {
  KiaxAccount* account;
  for (account=accounts.first(); account; account=accounts.next())
    if (account->accId == id) return account;
  return NULL;
}

QPtrList<QString> KiaxPrefs::getInputDeviceNames()
{
  return inputDeviceNames;
}

QPtrList<QString> KiaxPrefs::getOutputDeviceNames()
{
  return outputDeviceNames;
}

QPtrList<QString> KiaxPrefs::getRingDeviceNames()
{
  return ringDeviceNames;
}

int KiaxPrefs::getInputDevice()
{
  return inputDevice;
}

int KiaxPrefs::getOutputDevice()
{
  return outputDevice;
}

int KiaxPrefs::getRingDevice()
{
  return ringDevice;
}

void KiaxPrefs::setInputDevice(int input)
{
  inputDevice = input;
}

void KiaxPrefs::setOutputDevice(int output)
{

  outputDevice = output;
}

void KiaxPrefs::setRingDevice(int ring)
{
  ringDevice = ring;
}

int KiaxPrefs::getFilterFlag()
{
  return filterFlag;
}

void KiaxPrefs::setFilterFlag(int flag)
{
  filterFlag = flag;
}

QSettings * KiaxPrefs::getSettings()
{
  QSettings * settings = new QSettings();
  settings->setPath("kiax.org","kiax", QSettings::User);// userspace
  return settings;
}

void KiaxPrefs::setSilenceThreshold(int level)
{
  silenceThreshold = level;
}

int KiaxPrefs::getSilenceThreshold()
{
  return silenceThreshold;
}

bool KiaxPrefs::isDocked() {
  return docked;
}

bool KiaxPrefs::isHiddenOnStartup(){
  return hiddenOnStartup;
}

bool KiaxPrefs::isConnectedOnStartup() {
  return connectedOnStartup;
}

void KiaxPrefs::setDocked(bool d){
  docked = d;
}

void KiaxPrefs::setHiddenOnStartup(bool h){
  hiddenOnStartup = h;
}

void KiaxPrefs::setConnectedOnStartup(bool c) {
  connectedOnStartup = c;
}

void KiaxPrefs::setShowAccountNumberCombos(bool c) {
  showAccountNumberCombos = c;
}

void KiaxPrefs::setShowDialHangupButtons(bool c) {
  showDialHangupButtons = c;
}

void KiaxPrefs::setShowCallRegister(bool c) {
  showCallRegister = c;
}

void KiaxPrefs::setShowDialPad(bool c) {
  showDialPad = c;
}

bool KiaxPrefs::getShowAccountNumberCombos() {
  return showAccountNumberCombos;
}

bool KiaxPrefs::getShowDialHangupButtons() {
  return showDialHangupButtons;
}

bool KiaxPrefs::getShowCallRegister() {
  return showCallRegister;
}

bool KiaxPrefs::getShowDialPad() {
  return showDialPad;
}

void KiaxPrefs::setPreferredLanguage(QString lang) {
  preferredLanguage=lang;
}

QString KiaxPrefs::getPreferredLanguage() {
  return preferredLanguage;
}

QPoint KiaxPrefs::getWindowPosition() {
  return windowPosition;
}

void KiaxPrefs::setWindowPosition(QPoint wPosition) {
  windowPosition = wPosition;
}

QRect KiaxPrefs::getWidgetGeometry() {
  return widgetGeometry;
}

void KiaxPrefs::setWidgetGeometry(QRect wGeometry) {
  widgetGeometry = wGeometry;
}

int KiaxPrefs::getMaxCalls() {
  return maxCallsNumber;
}
void KiaxPrefs::setMaxCalls(int max) {
  maxCallsNumber = max;
}
void KiaxPrefs::setRaiseWindow(bool raise) {
  raiseWindow = raise;
}

bool KiaxPrefs::getRaiseWindow() {
  return raiseWindow;
}

void KiaxPrefs::setShowToolbar(bool flag) {
  showToolbar = flag;
}

void KiaxPrefs::setExecCommandOnRing(bool flag) {
  execCommandOnRing = flag;
}

void KiaxPrefs::setIncomingRingMute(bool flag) {
  incomingRingMute = flag;
}

void KiaxPrefs::setOutgoingRingMute(bool flag) {
  outgoingRingMute = flag;
}

void KiaxPrefs::setCommandOnRing(QString command) {
  commandOnRing = command;
}

bool KiaxPrefs::getShowToolbar() {
  return showToolbar;
}

bool KiaxPrefs::getExecCommandOnRing() {
  return execCommandOnRing;
}

bool KiaxPrefs::getIncomingRingMute() {
  return incomingRingMute;
}

bool KiaxPrefs::getOutgoingRingMute() {
  return outgoingRingMute;
}

QString KiaxPrefs::getCommandOnRing() {
  return commandOnRing;
}

