\ lufact      Does a LU factorization of a matrix

\ Forth Scientific Library Algorithm #33

\ lufact ( 'a{{ 'lu -- )
\     Factors matrix a{{ into the LUMATRIX stucture lu such that
\     a{{ = P L U   (P is a permutation matrix implied by the pivots)

\     The LUMATRIX data structure is initialized in one of two ways:
\        * Using lumatrix-init ( 'lu 'mat{{ 'piv{ n -- )
\          to build the LUMATRIX data structure in lu from existing
\          matrix mat{{ and integer array piv{ using size n.
\        * Using lu-malloc ( 'lu n -- )
\          to dynamically allocate the LUMATRIX data structure internals
\          for the structure 'dlu.  (In this case the space should
\          eventually be released with a call to lu-free ( 'lu -- ) ).

\     The routines, 'lu->l', 'lu->u' and 'lu->p' are provided to unpack the
\     appropriate component of the LU structure:
\ lu->l ( 'lu 'l{{ -- )
\     Fills the matrix l{{ with the L part of the LU structure.
\ lu->u ( 'lu 'u{{ -- )
\     Fills the matrix u{{ with the U part of the LU structure.
\ lu->p ( 'lu 'p{{ -- )
\     Fills the matrix p{{ with the P part of the LU structure.


\ This is an ANS Forth program requiring:
\      1. The Floating-Point word set  *** This is the integrated data/fp stack version ***
\      XXX 2. Uses words 'Private:', 'Public:' and 'Reset_Search_Order'
\         to control the visibility of internal code.
\      3. Uses the words 'FLOAT' and ARRAY to create floating point arrays.
\      4. The word '}' to dereference a one-dimensional array.
\      5. Uses the words 'DARRAY' and '&!' to set array pointers.
\      6. Uses the FSL utility word '}}fcopy' to copy one (float) array to another
\      7. FSL data structures as given in structs.fth
\      8. The FSL dynamic allocation words '}malloc' and '}free' are needed if
\         the data structures are dynamically allocated with 'lu-malloc'
\      9. The compilation of the test code is controlled by VALUE TEST-CODE?
\         and the conditional compilation words in the Programming-Tools
\         wordset
\     10. The test code uses the FSL routine HILBERT to generate test matrices.


\  (c) Copyright 1994 Everett F. Carter.  Permission is granted by the
\  author to use this software for any application provided this
\  copyright notice is preserved.

\ =====================================================================
\ The is the kForth version, which requires the following files:
\
\  ans-words.4th
\  fsl-util.4th
\  dynmem.4th
\  struct.4th       \ The structures package by A. Ertl is used
\                   \   in place of the FSL structures package.
\ Revisions:
\
\   2005-08-28  ported to kForth  K. Myneni
\   2005-09-06  define LUMATRIX  KM
\ ======================================================================

\   ( replaced obsolescent words      22 Jan 2005   cgm )

CR .( LUFACT            V1.6c         06 Sep      2005   EFC, KM )


\ a data structure for LU factored matrices
struct
      cell% field  ->matrix{{
      cell% field  ->pivot{
      cell% field  ->N                  \ the size of the matrix
      cell% field  ->status             \ = 0 if Ok
end-struct lumatrix%

: LUMATRIX  create lumatrix% %allot drop ;  \ defining word for data structure

\ pointer to users LU data structure
VARIABLE LU

: LU@    LU a@ ;


INTEGER DARRAY  t-piv       \ temporaries used for the dynamic allocation
FLOAT   DMATRIX  t-mat      \ of the LU data structure

FLOAT   DMATRIX a{{         \ pointer to matrix to factor

FLOAT   DMATRIX matrix{{    \ pointer to LU ->matrix{{  for faster dereferencing
INTEGER DARRAY pivot{       \ pointer to LU ->pivot{



\ For aliasing the internal structures to pre-exisiting arrays
: lumatrix-init ( 'lu 'mat 'piv n -- )
        3 PICK >R    
        R@ ->N !             \ store N
        R@ ->pivot{ !        \ store pointer to pivot array
        R> ->matrix{{ !      \ store pointer to matrix
	DROP ;

\ For dynamically allocating the whole structure
: lu-malloc ( 'lu n -- )

     & t-piv OVER }malloc
     malloc-fail? ABORT" LU-MALLOC failure (1) "

     & t-mat OVER DUP }}malloc
     malloc-fail? ABORT" LU-MALLOC failure (2) "

     t-mat t-piv ROT lumatrix-init
;

\ for freeing a dynamically allocated structure
: lu-free ( 'lu -- )
     DUP ->matrix{{ a@   & t-mat &!  & t-mat }}free
     DUP ->pivot{   a@   & t-piv &!  & t-piv }free
     0 SWAP ->N !
;


: lufact-init ( 'a 'lu -- n )

    LU !
    & a{{ &!

    LU@  ->N @
    0 LU@  ->status !

    LU@ ->pivot{   a@  & pivot{    &!
    LU@ ->matrix{{ a@  & matrix{{  &!

    DUP
    
    a{{ matrix{{ ROT DUP }}fcopy

;

: lufact-finish ( n -- flag )

     1- DUP
     pivot{ OVER } !

     matrix{{ OVER DUP }} F@ F0= IF LU@ ->status !
                                 ELSE
                                    DROP
                                 THEN


    LU@ ->status @

;

FVARIABLE fpivot

: partial-pivot ( n k  -- n l )

        \ ." partial pivot k = " DUP . CR
        \ over DUP matrix{{ }}fprint

        0.0E0  fpivot F!
        DUP DUP
        3 PICK SWAP  DO
                       matrix{{ I 3 PICK  }} F@ FABS
                       fpivot F@ FOVER F< IF
                                         fpivot F! 
                                         DROP I
				ELSE FDROP
                                THEN
                     LOOP

         DUP ROT pivot{ SWAP } !
 \ ." l = " DUP . CR
;

: singular-pivot ( k -- )
     LU@ ->status ! ;

FVARIABLE ftemp

: interchange ( n l k -- n )

        \ OVER OVER ." interchanging k = " . ." l = " . CR

        2 PICK 0 DO
           matrix{{ 2 PICK I }} DUP F@  ftemp F!  
           matrix{{ 2 PICK I }} DUP F@  2>R  
           SWAP 2R> ROT F!  ftemp F@ ROT F!
       LOOP
       2DROP
;

FVARIABLE fscale1
FVARIABLE fscale2 

: scale-row ( n k -- n )

       \ ." scale-row, k = " DUP . CR

       matrix{{ OVER DUP }} F@ 1.0E0 FSWAP F/ fscale1 F!
       
       2DUP 1+ DO
               matrix{{ I 2 PICK }} DUP F@ fscale1 F@ F*
               FDUP  fscale2 F! ROT F!

               OVER OVER 1+ DO
                 matrix{{ OVER I }} F@ fscale2 F@ F* FNEGATE ftemp F!
                 matrix{{ J I }} DUP F@ ftemp F@ F+
                 ROT F!
               LOOP
              
       LOOP

     DROP
;


: build-identity ( 'p n -- 'p )         \ make an NxN identity matrix
             0 DO
                  I 1+ 0 DO
                            I J = IF   DUP I J }} 1.0E0 ROT F!
                                  ELSE
                                       DUP J I }} 0.0E0 ROT F!
                                       DUP I J }} 0.0E0 ROT F!
                                  THEN
                         LOOP
                 LOOP
;

: column_swap ( 'p n k1 k2 -- 'p n )

           2 PICK 0 DO
                        3 PICK I 3 PICK }} DUP F@ ftemp F!
                        4 PICK I 3 PICK }} DUP F@ 2>R
                        SWAP 2R> ROT F! ftemp F@ ROT F!
                    LOOP
           2DROP
;



: lufact ( 'a 'lu --  )

     lufact-init

     DUP 2 < IF
               lufact-finish
               ABORT" lufact failed! (1) "
             THEN


     DUP 1- 0 DO
                  I partial-pivot

                  matrix{{ OVER I }} F@ F0= IF
                                           DROP I singular-pivot
                                        ELSE
                                           I OVER = IF
                                                       DROP
                                                     ELSE
                                                       I interchange
                                                     THEN

                                           I scale-row
                                            
                                        THEN


              LOOP

              
     lufact-finish

     ABORT" lufact failed! (2) "
     
;

: LU->L  ( 'lu 'l{{ -- )         \ unpack L matrix from LU struture

         >R
         DUP  ->matrix{{ a@    \ get base address of matrix
         >R
         ->N @                 \ get N
         R> R>

         ROT 0 DO
                  I 1+ 0 DO
                            I J = IF
                                    DUP I J }} 1.0E0 ROT F!
                                  ELSE
                                    DUP I J }} 0.0E0 ROT F!
                                    OVER  J I }} F@  ftemp F! DUP J I }} ftemp F@ ROT F!
                                  THEN
                         LOOP
                 LOOP

         2DROP
;

: LU->U  ( 'lu 'u{{ -- )         \ unpack U matrix from LU struture

         >R
         DUP  ->matrix{{ a@    \ get base address of matrix
         >R
         ->N @                 \ get N
         R> R>

         ROT 0 DO
                  I 1+ 0 DO
                            DUP J I }} 0.0E0 ROT F!
                            OVER  I J }} F@  ftemp F! DUP I J }} ftemp F@ ROT F!
                         LOOP
                 LOOP

         2DROP
;

: LU->P  ( 'lu 'p{{ -- )         \ extract P matrix from LU struture

         >R
         DUP  ->pivot{    \ get base address of pivot
         >R
         ->N @            \ get N
         R> R>

         ROT DUP >R          ( 'pivot 'p n )
         build-identity     \ build identity matrix first
        
         R>

         \ now swap the appropriate columns
         DUP 0 DO
                  2 PICK I } @ I OVER OVER =
                  IF  2DROP   ELSE  column_swap   THEN
               LOOP

         DROP 2DROP
;


Reset_Search_Order


TEST-CODE? [IF]   \ test code ==============================================

4 4 FLOAT matrix mat{{
4 4 FLOAT matrix lmat{{
4 INTEGER array piv{

: HILBERT ( 'h n -- )     \ stores in array with address 'h
     DUP 0 DO
            DUP 0 DO
                    OVER 1.0E0 I J + 1+ S>F F/
                    ROT I J }} F!
            LOOP
     LOOP
     2DROP
;

LUMATRIX lu4 

: actual-4 ( -- )

    CR ." actual LUFACTORED 4x4 matrix: " CR
       ."  1.0       0.5         0.3333      0.25 " CR 
       ."  0.3333  0.083333      0.088889    0.0833333 " CR 
       ."  0.5       1.0        -0.005556   -0.008333" CR 
       ."  0.25      0.90       -0.6         0.00357" CR
        

;

: actual-3 ( -- )

    CR ." actual LUFACTORED 3x3 matrix: " CR

       ."  4.0      4.0      -3.0 " CR
       ." -0.5      5.0      -2.5 " CR
       ."  0.5      0.2       1.0 " CR 

;


: matrix-3 ( -- )

	2.0e0 mat{{ 0 0 }} F!   3.0e0 mat{{ 0 1 }} F!   -1.0e0 mat{{ 0 2 }} F!
        4.0e0 mat{{ 1 0 }} F!   4.0e0 mat{{ 1 1 }} F!   -3.0e0 mat{{ 1 2 }} F!
       -2.0e0 mat{{ 2 0 }} F!   3.0e0 mat{{ 2 1 }} F!   -1.0e0 mat{{ 2 2 }} F!
;



: lufact-test  ( -- )           \ uses aliased arrays

        lu4 lmat{{ piv{ 4 LUMATRIX-INIT    \ aliasing lu to pre-existing arrays

        mat{{ 4 hilbert

        CR ." Original matrix:" CR

        4 4 mat{{ }}fprint

        mat{{ lu4 lufact

        CR ." Factored matrix:" CR
        4 4 lu4 ->matrix{{ a@ }}fprint      \ 4 4 lmat{{ }}fprint   will also work
	CR ." PIVOTS: " 4 piv{ }iprint CR

        actual-4
;

: lufact-test2    ( -- )              \ uses dynamically allocated space

        lu4 4 LU-MALLOC

        mat{{ 4 hilbert

        CR ." Original matrix:" CR
        4 4 mat{{ }}fprint

        mat{{ lu4 lufact

        CR ." Factored matrix:" CR
        4 4 lu4 ->matrix{{ a@ }}fprint   \ must use ->MATRIX{{ to get to the matrix
	CR ." PIVOTS: " 4 piv{ }iprint CR

        actual-4

        lu4 LU-FREE        
;

: lufact-test3  ( -- )           \ uses aliased arrays

        lu4 lmat{{ piv{ 3 LUMATRIX-INIT    \ aliasing lu to pre-existing arrays

        matrix-3

        CR ." Original matrix:" CR

        3 3 mat{{ }}fprint

        mat{{ lu4 lufact

        CR ." Factored matrix:" CR
        3 3 lu4 ->matrix{{ a@ }}fprint
        CR ." PIVOTS: " 3 piv{ }iprint CR

	actual-3

;

[THEN]

