/*
 * ******************************************************************
 * $Id: properties.c,v 1.7 2000/12/02 01:17:32 orjana Exp $
 * Filename:      properties.c
 * Description:   A property window for configuration of the applet.
 * Status:        Experimental, do not distribute.
 * Author:        rjan Nygaard Austvold <austvold@acm.org>
 * Created at:    Mon Nov 27 16:05:39 2000
 * Modified at:   Fri Dec  1 16:05:15 2000
 * Modified by:   rjan Nygaard Austvold <austvold@acm.org>
 *                
 * Copyright (c) 2000 rjan Nygaard Austvold, All Rights Reserved.
 *                
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 *
 * Revision History:
 *
 *   01-Dec-00 23:21:01   rjan Nygaard Austvold <austvold@acm.org>
 *     Changed default maximum size to 16 pixels so that it would look
 *     better on panels without size-adjustment.
 *
 *   01-Dec-00 16:04:23   rjan Nygaard Austvold <austvold@acm.org>
 *     Added call to gdk_window_set_icon() to display a window icon on
 *     the properties dialog.
 *
 * *****************************************************************
 */


#include "properties.h"

static gchar *str_label[20] = {
  N_("Num Lock"),
  N_("Caps Lock"),
  N_("Scroll Lock"),
  N_("Use default settings?"),
  N_("Minimum width:"),
  N_("Maximum width:"),
  N_("Optimize space-usage on panel?"),
  N_("Keyboard Led Settings"),  
  N_("General"),
  N_("Images"),
  N_("Icon Size"),
  N_("Display"),
  N_("Aspect ratio (width/height):"),
  N_("Select pixmap file"),
  N_("Num Lock off:"),
  N_("Num Lock on:"),
  N_("Caps Lock off:"),
  N_("Caps Lock on:"),
  N_("Scroll Lock off:"),
  N_("Scroll Lock on:") };

static GtkWidget *dialog_property = NULL;
static GtkObject *min_spinbutton_adj;
static GtkWidget *min_spinbutton;
static GtkObject *max_spinbutton_adj;
static GtkWidget *max_spinbutton;
static GtkObject *aspect_spinbutton_adj;
static GtkWidget *aspect_spinbutton;
static GtkWidget *checkbtn[7];
static GtkWidget *pixmapentry[6];

struct _settings settings, settings_tmp;
struct _settings settings_default = {
  TRUE, TRUE, FALSE, TRUE, {TRUE, TRUE, TRUE},
  {PACKAGE_ICON_DIR"/numlock-off-46x46.png",
   PACKAGE_ICON_DIR"/numlock-on-46x46.png",
   PACKAGE_ICON_DIR"/capslock-off-46x46.png",
   PACKAGE_ICON_DIR"/capslock-on-46x46.png",
   PACKAGE_ICON_DIR"/scrollock-off-46x46.png",
   PACKAGE_ICON_DIR"/scrollock-on-46x46.png" },
  12, 16, 1.0 };


extern GdkPixmap *icon_pixmap;
extern GdkBitmap *icon_mask;
extern void reshape_applet (gboolean old_images);



/*
 * Function callback_properties (applet, data)
 *
 *    Create a property window.
 *
 */
void callback_properties (AppletWidget *applet, gpointer data)
{
  static GtkWidget *general_label = NULL;
  static GtkWidget *images_label = NULL;
  static GtkWidget *generalpage = NULL;
  static GtkWidget *imagespage = NULL;

  if (dialog_property) {
    gdk_window_raise (dialog_property->window);
    return;
  }

  dialog_property = gnome_property_box_new();
  gtk_widget_set (dialog_property,
		  "title", _(str_label[PROPERTY_DIALOG_TITLE]),
		  "signal::apply", property_apply_callback, NULL,
		  "signal::destroy", property_destroy_callback, NULL,
		  "signal::help", property_help_callback, NULL,
		  NULL);

  general_label = gtk_label_new (_(str_label[PROPERTY_GENERAL_PAGE]));
  images_label = gtk_label_new (_(str_label[PROPERTY_IMAGES_PAGE]));
  gtk_widget_ref (general_label);
  gtk_widget_ref (images_label);
  
  generalpage = general_page();
  imagespage = images_page();

  (void) gnome_property_box_append_page (GNOME_PROPERTY_BOX (dialog_property),
					 generalpage, general_label);
  (void) gnome_property_box_append_page (GNOME_PROPERTY_BOX (dialog_property),
					 imagespage, images_label);
  configure_property_window();
  
  gtk_widget_show_all (dialog_property);
  gdk_window_raise (dialog_property->window);
  gdk_window_set_icon (dialog_property->window, NULL, icon_pixmap, icon_mask);
}



/*
 * Function general_page ()
 *
 *    Description and creation of the general-page.
 *
 */
GtkWidget * general_page (void)
{
  GtkWidget *general_vbox;
  GtkWidget *size_frame;
  GtkWidget *size_vbox;
  GtkWidget *minmax_hbox;
  GtkWidget *aspect_hbox;
  GtkWidget *aspect_label;
  GtkWidget *display_frame;
  GtkWidget *display_hbox;
  gint i;

  /* vertical box with 3 compartments */
  general_vbox = gtk_vbox_new (FALSE, 0);
  gtk_widget_ref (general_vbox);

  /* check buttons: num-, caps-, and scroll-lock, default settings,
     min. and max., and optimize */
  for (i = 0; i < 7; i++) {
    checkbtn[i] = gtk_check_button_new_with_label (_(str_label[i]));
    gtk_widget_ref (checkbtn[i]);
  }

  /* min-, max-, and aspectratio-spinbuttons */
  min_spinbutton_adj =
    gtk_adjustment_new(1, MIN_MIN, MIN_MAX, MIN_SINC, MIN_PINC, MIN_PSZ);
  max_spinbutton_adj =
    gtk_adjustment_new(1, MAX_MIN, MAX_MAX, MAX_SINC, MAX_PINC, MAX_PSZ);
  aspect_spinbutton_adj =
    gtk_adjustment_new(1, ASP_MIN, ASP_MAX, ASP_SINC, ASP_PINC, ASP_PSZ);
  min_spinbutton =
    gtk_spin_button_new (GTK_ADJUSTMENT (min_spinbutton_adj),    1, 0);
  max_spinbutton =
    gtk_spin_button_new (GTK_ADJUSTMENT (max_spinbutton_adj),    1, 0);
  aspect_spinbutton =
    gtk_spin_button_new (GTK_ADJUSTMENT (aspect_spinbutton_adj), 1, 2);
  gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (min_spinbutton),
				     GTK_UPDATE_IF_VALID);
  gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (max_spinbutton),
				     GTK_UPDATE_IF_VALID);
  gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (aspect_spinbutton),
				     GTK_UPDATE_IF_VALID);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (min_spinbutton), TRUE);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (max_spinbutton), TRUE);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (aspect_spinbutton), TRUE);

  gtk_widget_set_usize (aspect_spinbutton, 69, -2);

  /* aspect-ratio label */
  aspect_label = gtk_label_new (_(str_label[PROPERTY_ASPECT_RATIO]));

  /* default settings checkbutton on the 1st compartment */
  gtk_box_pack_start (GTK_BOX (general_vbox),
		      checkbtn[DEFAULT_CHK],  FALSE, FALSE, 0);

  /* size-settings frame in the 2nd compartment */
  size_frame = gtk_frame_new (_(str_label[PROPERTY_SIZE_FRAME]));
  gtk_widget_ref (size_frame);
  gtk_box_pack_start (GTK_BOX (general_vbox), size_frame, FALSE, TRUE, 0);

  /* size-settings vertical box with 3 compartments in the
     size-settings frame */
  size_vbox = gtk_vbox_new (FALSE, 7);
  gtk_widget_ref (size_vbox);
  gtk_container_add (GTK_CONTAINER (size_frame), size_vbox);
  gtk_container_set_border_width (GTK_CONTAINER (size_vbox), 3);

  /* min-max horizontal box with 4 compartments in the 1st compartment in
     the vertical box of the size-settings frame */
  minmax_hbox = gtk_hbox_new (FALSE, 0);
  gtk_widget_ref (minmax_hbox);
  gtk_box_pack_start (GTK_BOX (size_vbox), minmax_hbox, FALSE, FALSE, 0);

  /* aspect-ratio horizontal box with 2 compartments in the 2nd compartment in
     the vertical box of the size-settings frame */
  aspect_hbox = gtk_hbox_new (FALSE, 0);
  gtk_widget_ref (aspect_hbox);
  gtk_box_pack_start (GTK_BOX (size_vbox), aspect_hbox, TRUE, TRUE, 0);

  /* display-frame in the the 3rd comapartment of the general-page box */
  display_frame = gtk_frame_new (_(str_label[PROPERTY_DISPLAY_FRAME]));
  gtk_widget_ref (display_frame);
  gtk_box_pack_start (GTK_BOX (general_vbox), display_frame, FALSE, TRUE, 0);

  /* display-settings in the display-frame */
  display_hbox = gtk_hbox_new (FALSE, 0);
  gtk_widget_ref (display_hbox);
  gtk_container_add (GTK_CONTAINER (display_frame), display_hbox);
  gtk_container_set_border_width (GTK_CONTAINER (display_hbox), 3);

  /* place the various controls in their right boxes */
  gtk_box_pack_start (GTK_BOX (aspect_hbox), aspect_label, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (aspect_hbox),
		      aspect_spinbutton, FALSE, TRUE,0);
  gtk_box_pack_start (GTK_BOX (display_hbox)
		      , checkbtn[NUM_LOCK],   FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (display_hbox),
		      checkbtn[CAPS_LOCK],    FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (display_hbox),
		      checkbtn[SCROLL_LOCK],  FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (minmax_hbox),
		      checkbtn[MIN_CHK],      FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (minmax_hbox), min_spinbutton, FALSE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (minmax_hbox),
		      checkbtn[MAX_CHK],      FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (minmax_hbox), max_spinbutton, FALSE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (size_vbox),
		      checkbtn[OPTIMIZE_CHK], FALSE, FALSE, 0);

  gtk_widget_ref (aspect_label);
  gtk_widget_ref (min_spinbutton);
  gtk_widget_ref (max_spinbutton);
  gtk_widget_ref (aspect_spinbutton);

  return general_vbox;
}



/*
 * Function images_page ()
 *
 *    Description and creation of the images-page.
 *
 */
GtkWidget *images_page (void)
{
  GtkWidget *images_table;
  GtkWidget *pixmap_label[6];
  gint i;

  images_table = gtk_table_new (6, 2, FALSE);
  gtk_widget_ref (images_table);

  gtk_container_set_border_width (GTK_CONTAINER (images_table), 7);
  gtk_table_set_row_spacings (GTK_TABLE (images_table), 5);
  gtk_table_set_col_spacings (GTK_TABLE (images_table), 7);

  for (i = 0; i < 6; i++) {
    pixmapentry[i] = gnome_file_entry_new (NULL, _(str_label[FILE_DIALOG_TITLE]));
    gtk_widget_ref (pixmapentry[i]);
    gtk_table_attach (GTK_TABLE (images_table), pixmapentry[i], 1, 2, i, i + 1,
		      (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		      (GtkAttachOptions) (0), 0, 0);

    pixmap_label[i] = gtk_label_new (_(str_label[PIXMAP_FILE_LABEL_BASE + i]));
    gtk_widget_ref (pixmap_label[i]);
    gtk_table_attach (GTK_TABLE (images_table), pixmap_label[i],
		      0, 1, i + 0, i + 1,
		      (GtkAttachOptions) (GTK_FILL),
		      (GtkAttachOptions) (0), 0, 0);
    gtk_label_set_justify (GTK_LABEL (pixmap_label[i]), GTK_JUSTIFY_LEFT);
    gtk_misc_set_alignment (GTK_MISC (pixmap_label[i]), 0, 0.5);
  }

  return images_table;
}



/*
 * Function configure_property_window ()
 *
 *    (De-)activate widgets based on intial settings and connect signals.
 *
 */
void configure_property_window (void)
{
  gint i;

  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (checkbtn[DEFAULT_CHK]),
				settings_tmp.use_default);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (checkbtn[MIN_CHK]),
				settings_tmp.minimum_checked);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (checkbtn[MAX_CHK]),
				settings_tmp.maximum_checked);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (checkbtn[OPTIMIZE_CHK]),
				settings_tmp.optimize_checked);
  for (i = 0; i < 3; i++)
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (checkbtn[i]),
				  settings_tmp.lock_key_checked[i]);

  gtk_spin_button_set_value (GTK_SPIN_BUTTON (min_spinbutton),
			     settings_tmp.minimum);
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (max_spinbutton),
			     settings_tmp.maximum);
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (aspect_spinbutton),
			     settings_tmp.aspect_ratio);

  for (i = 0; i < 6; i++)
    gtk_entry_set_text (GTK_ENTRY (gnome_file_entry_gtk_entry
				   (GNOME_FILE_ENTRY (pixmapentry[i]))),
			settings_tmp.pixmap_file[i]);
  
  control_active_widgets();
  
  gtk_signal_connect (GTK_OBJECT (checkbtn[DEFAULT_CHK]), "toggled",
                      GTK_SIGNAL_FUNC (on_default_checkbutton_toggled),
                      NULL);
  gtk_signal_connect (GTK_OBJECT (checkbtn[MIN_CHK]), "toggled",
                      GTK_SIGNAL_FUNC (on_min_checkbutton_toggled),
                      NULL);
  gtk_signal_connect (GTK_OBJECT (checkbtn[MAX_CHK]), "toggled",
                      GTK_SIGNAL_FUNC (on_max_checkbutton_toggled),
                      NULL);
  for (i = 0; i < 3; i++)
    gtk_signal_connect (GTK_OBJECT (checkbtn[i]), "toggled",
			GTK_SIGNAL_FUNC (on_lock_key_checkbutton_toggled),
			NULL);
  gtk_signal_connect (GTK_OBJECT (min_spinbutton), "changed",
		      GTK_SIGNAL_FUNC (on_min_spinbutton_changed), NULL);
  gtk_signal_connect (GTK_OBJECT (max_spinbutton), "changed",
		      GTK_SIGNAL_FUNC (on_max_spinbutton_changed), NULL);
  gtk_signal_connect_object (GTK_OBJECT (aspect_spinbutton), "changed",
			     GTK_SIGNAL_FUNC (gnome_property_box_changed),
			     GTK_OBJECT (dialog_property));
  gtk_signal_connect_object (GTK_OBJECT (checkbtn[OPTIMIZE_CHK]), "toggled",
			     GTK_SIGNAL_FUNC (gnome_property_box_changed),
			     GTK_OBJECT (dialog_property));
  for (i = 0; i < 6; i++)
    gtk_signal_connect_object (GTK_OBJECT (gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY (pixmapentry[i]))),
			       "changed",
			       GTK_SIGNAL_FUNC (gnome_property_box_changed),
			       GTK_OBJECT (dialog_property));
}



/*
 * Function save_properties (path)
 *
 *    Saves persistent properties.
 *
 */
void save_properties (const gchar *path)
{
  gnome_config_push_prefix (path);
  gnome_config_set_bool ("keyboard_led/use_default",
			 settings.use_default);
  gnome_config_set_bool ("keyboard_led/minimum",
			 settings.minimum_checked);
  gnome_config_set_bool ("keyboard_led/maximum",
			 settings.maximum_checked);
  gnome_config_set_bool ("keyboard_led/optimize",
			 settings.optimize_checked);
  gnome_config_set_bool ("keyboard_led/numlock",
			 settings.lock_key_checked[NUM_CHK]);
  gnome_config_set_bool ("keyboard_led/capslock",
			 settings.lock_key_checked[CAPS_CHK]);
  gnome_config_set_bool ("keyboard_led/scrollock",
			 settings.lock_key_checked[SCROLL_CHK]);
  gnome_config_set_string ("keyboard_led/file0",
			   settings.pixmap_file[NUM_LOCK_OFF]);
  gnome_config_set_string ("keyboard_led/file1",
			   settings.pixmap_file[NUM_LOCK_ON]);
  gnome_config_set_string ("keyboard_led/file2",
			   settings.pixmap_file[CAPS_LOCK_OFF]);
  gnome_config_set_string ("keyboard_led/file3",
			   settings.pixmap_file[CAPS_LOCK_ON]);
  gnome_config_set_string ("keyboard_led/file4",
			   settings.pixmap_file[SCROLL_LOCK_OFF]);
  gnome_config_set_string ("keyboard_led/file5",
			   settings.pixmap_file[SCROLL_LOCK_ON]);
  /* TODO: fix to dyn. value (MIN/MAX/ASP_VAL) */
  gnome_config_set_int   ("keyboard_led/minval", settings.minimum);
  gnome_config_set_int   ("keyboard_led/maxval", settings.maximum);
  gnome_config_set_float ("keyboard_led/aspval", settings.aspect_ratio);
}



/*
 * Function load_properties (path)
 *
 *    Load persistent properties.
 *
 */
void load_properties (const gchar *path)
{
  gnome_config_push_prefix (path);

  /* note: we don't specify default settings for each key, since we
     assume that if one key isn't in the config-file, none are. and in
     addition we only wan't to maintain default settings one place; in
     the default structure we got to keep anyhow... */
  settings.use_default =
    gnome_config_get_bool ("keyboard_led/use_default=true");
  settings.minimum_checked =
    gnome_config_get_bool ("keyboard_led/minimum");
  settings.maximum_checked =
    gnome_config_get_bool ("keyboard_led/maximum");
  settings.optimize_checked =
    gnome_config_get_bool ("keyboard_led/optimize");
  settings.lock_key_checked[NUM_CHK] =
    gnome_config_get_bool ("keyboard_led/numlock");
  settings.lock_key_checked[CAPS_CHK] =
    gnome_config_get_bool ("keyboard_led/capslock");
  settings.lock_key_checked[SCROLL_CHK] =
    gnome_config_get_bool ("keyboard_led/scrollock");
  settings.pixmap_file[NUM_LOCK_OFF] =
    gnome_config_get_string ("keyboard_led/file0");
  settings.pixmap_file[NUM_LOCK_ON] =
    gnome_config_get_string ("keyboard_led/file1");
  settings.pixmap_file[CAPS_LOCK_OFF] =
    gnome_config_get_string ("keyboard_led/file2");
  settings.pixmap_file[CAPS_LOCK_ON] =
    gnome_config_get_string ("keyboard_led/file3");
  settings.pixmap_file[SCROLL_LOCK_OFF] =
    gnome_config_get_string ("keyboard_led/file4");
  settings.pixmap_file[SCROLL_LOCK_ON] =
    gnome_config_get_string ("keyboard_led/file5");
  settings.minimum      = gnome_config_get_int   ("keyboard_led/minval");
  settings.maximum      = gnome_config_get_int   ("keyboard_led/maxval");
  settings.aspect_ratio = gnome_config_get_float ("keyboard_led/aspval");
  gnome_config_pop_prefix();

  /* if no settings are in the config file; fetch the default settings */
  if (settings.use_default)
    (void)memcpy (&settings, &settings_default, sizeof (struct _settings));

  /* make tmp_structure ready. it will be used while operating the properties
     dialog */
  (void)memcpy (&settings_tmp, &settings, sizeof (struct _settings));
}



/*
 * Function control_active_widgets ()
 *
 *    (De-)activate widgets based on logical rules and widget values.
 *
 */
void control_active_widgets (void)
{
  gint i;
  
  if (settings_tmp.use_default == TRUE) {
    /* deactivate all the other controls */
    
    gtk_widget_set_sensitive (GTK_WIDGET (checkbtn[MIN_CHK]), FALSE);
    gtk_widget_set_sensitive (GTK_WIDGET (min_spinbutton), FALSE);
    gtk_widget_set_sensitive (GTK_WIDGET (checkbtn[MAX_CHK]), FALSE);
    gtk_widget_set_sensitive (GTK_WIDGET (max_spinbutton), FALSE);
    gtk_widget_set_sensitive (GTK_WIDGET (aspect_spinbutton), FALSE);
    gtk_widget_set_sensitive (GTK_WIDGET (checkbtn[OPTIMIZE_CHK]), FALSE);
    for (i = 0; i < 3; i++)
      gtk_widget_set_sensitive (GTK_WIDGET (checkbtn[i]), FALSE);
    for (i = 0; i < 6; i++)
      gtk_widget_set_sensitive (GTK_WIDGET (pixmapentry[i]), FALSE);
    
  } else {
    /* activate controls based on values on check-buttons */
    gtk_widget_set_sensitive (GTK_WIDGET (checkbtn[MIN_CHK]), TRUE);
    if (gtk_toggle_button_get_active ( GTK_TOGGLE_BUTTON (checkbtn[MIN_CHK])))
      gtk_widget_set_sensitive (GTK_WIDGET (min_spinbutton), TRUE);
    else
      gtk_widget_set_sensitive (GTK_WIDGET (min_spinbutton), FALSE);
    
    gtk_widget_set_sensitive (GTK_WIDGET (checkbtn[MAX_CHK]), TRUE);
    if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (checkbtn[MAX_CHK])))
      gtk_widget_set_sensitive (GTK_WIDGET (max_spinbutton), TRUE);
    else
      gtk_widget_set_sensitive (GTK_WIDGET (max_spinbutton), FALSE);
    
    gtk_widget_set_sensitive (GTK_WIDGET (aspect_spinbutton), TRUE);
    gtk_widget_set_sensitive (GTK_WIDGET (checkbtn[OPTIMIZE_CHK]), TRUE);
    
    check_and_set_lock_toggles_sensitivity();
    
    for (i = 0; i < 6; i++)
      gtk_widget_set_sensitive (GTK_WIDGET (pixmapentry[i]), TRUE);
  }
}



/*
 * Function check_and_set_lock_toggles_sensitivity ()
 *
 *    At a minimum at least one of the num-, caps-, or scroll-lock
 *    buttons have to be checked. If only one is checked; Make it
 *    non-sensitive to input.
 *
 */
void check_and_set_lock_toggles_sensitivity (void)
{
  gboolean checked[3];
  gboolean sensitive[3] = {TRUE, TRUE, TRUE};
  gint i;

  for (i = 0; i < 3; i++)
    checked[i] = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(checkbtn[i]));
  
  if      ( checked[NUM_CHK] && !checked[CAPS_CHK] && !checked[SCROLL_CHK])
    sensitive[NUM_CHK] = FALSE;
  else if (!checked[NUM_CHK] &&  checked[CAPS_CHK] && !checked[SCROLL_CHK])
    sensitive[CAPS_CHK] = FALSE;
  else if (!checked[NUM_CHK] && !checked[CAPS_CHK] &&  checked[SCROLL_CHK])
    sensitive[SCROLL_CHK] = FALSE;

  for (i = 0; i < 3; i++)
    gtk_widget_set_sensitive (GTK_WIDGET (checkbtn[i]), sensitive[i]);
}



void on_default_checkbutton_toggled (GtkToggleButton *togglebutton,
				     gpointer user_data)
{
  gnome_property_box_changed (GNOME_PROPERTY_BOX (dialog_property));
  settings_tmp.use_default = gtk_toggle_button_get_active (togglebutton);
  control_active_widgets();
}



void on_min_checkbutton_toggled (GtkToggleButton *togglebutton,
				 gpointer user_data)
{
  gnome_property_box_changed (GNOME_PROPERTY_BOX (dialog_property));
  settings_tmp.minimum_checked = gtk_toggle_button_get_active (togglebutton);
  gtk_widget_set_sensitive (GTK_WIDGET (min_spinbutton),
			    settings_tmp.minimum_checked);
}



void on_max_checkbutton_toggled (GtkToggleButton *togglebutton,
				 gpointer user_data)
{
  gnome_property_box_changed (GNOME_PROPERTY_BOX (dialog_property));
  settings_tmp.minimum_checked = gtk_toggle_button_get_active (togglebutton);
  gtk_widget_set_sensitive (GTK_WIDGET (max_spinbutton),
			    settings_tmp.minimum_checked);
}



void on_min_spinbutton_changed (GtkSpinButton *spinbutton, gpointer user_data)
{
  int max, min;

  gnome_property_box_changed (GNOME_PROPERTY_BOX (dialog_property));

  min = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (min_spinbutton));
  max = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (max_spinbutton));

  if (max < min)
    gtk_spin_button_set_value (GTK_SPIN_BUTTON (max_spinbutton), (gfloat)min);
}



void on_max_spinbutton_changed (GtkSpinButton *spinbutton, gpointer user_data)
{
  gint max, min;

  gnome_property_box_changed (GNOME_PROPERTY_BOX (dialog_property));

  min = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (min_spinbutton));
  max = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (max_spinbutton));

  if (min > max)
    gtk_spin_button_set_value (GTK_SPIN_BUTTON (min_spinbutton), (gfloat)max);
}



void on_lock_key_checkbutton_toggled (GtkToggleButton *togglebutton,
				      gpointer user_data)
{
  int i;
  gnome_property_box_changed (GNOME_PROPERTY_BOX (dialog_property));
  check_and_set_lock_toggles_sensitivity();
  for (i = 0; i < 3; i++)
    settings_tmp.lock_key_checked[i]
      = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (checkbtn[i]));
}



void property_apply_callback (GnomePropertyBox *property_box, gint page_num)
{
  gint i;
  
  switch (page_num) {
  case 0:
    /* fetch values */
    settings_tmp.minimum = 
      gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (min_spinbutton));
    settings_tmp.maximum =
      gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (max_spinbutton));
    settings_tmp.aspect_ratio =
      gtk_spin_button_get_value_as_float (GTK_SPIN_BUTTON (aspect_spinbutton));
    settings_tmp.optimize_checked =
      gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(checkbtn[OPTIMIZE_CHK]));
    settings_tmp.minimum_checked =
      gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(checkbtn[MIN_CHK]));
    settings_tmp.maximum_checked =
      gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(checkbtn[MAX_CHK]));
    break;
  case 1:
    /* since there are changes in one or more image-files we update them all */
    for (i = 0; i < 6; i++) {
      /* free the tmp_settings pointer to the pixmap file, but only
	 if it doesn't point to the default path */
      if (settings_tmp.pixmap_file[i] != settings_default.pixmap_file[i])
	g_free (settings_tmp.pixmap_file[i]);
      
      /* update pixmap-path */
      settings_tmp.pixmap_file[i] = gtk_editable_get_chars
	(GTK_EDITABLE (gnome_file_entry_gtk_entry
		       (GNOME_FILE_ENTRY (pixmapentry[i]))), 0, -1);
    }
    break;
  default:
    break;
  }

  if (page_num != -1) {
    if (settings_tmp.use_default == TRUE)
      (void)memcpy (&settings, &settings_default, sizeof (struct _settings));
    else
      (void)memcpy (&settings, &settings_tmp, sizeof (struct _settings));
    
    /* reshape applet and reload/reshape all images regardless of changes */
    (void) reshape_applet(TRUE);
  }
}


void property_help_callback (GnomePropertyBox *property_box, gint page_num)
{
  GnomeHelpMenuEntry help_entry = { PACKAGE, "index.html" };
  gnome_help_display(NULL, &help_entry);
}


gint property_destroy_callback (GtkWidget *widget, void *data)
{
  dialog_property = NULL;
  return FALSE;
}
