/***************************************************************************
 *   Copyright (C) 2004-2019 by Thomas Fischer <fischer@unix-ag.uni-kl.de> *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, see <https://www.gnu.org/licenses/>. *
 ***************************************************************************/

#include "encoder.h"

#ifdef HAVE_ICU
#include <unicode/translit.h>
#endif // HAVE_ICU

#include "encoderlatex.h"
#include "logging_io.h"

class Encoder::Private
{
public:
#ifdef HAVE_ICU
    icu::Transliterator *translit;
#else // HAVE_ICU
    static const char *unidecode_text;
    static const size_t unidecode_pos[];
#endif // HAVE_ICU

    Private()
#ifdef HAVE_ICU
            : translit(nullptr)
#endif // HAVE_ICU
    {
#ifdef HAVE_ICU
        /// Create an ICU Transliterator, configured to
        /// transliterate virtually anything into plain ASCII
        UErrorCode uec = U_ZERO_ERROR;
        translit = icu::Transliterator::createInstance("Any-Latin;Latin-ASCII", UTRANS_FORWARD, uec);
        if (U_FAILURE(uec)) {
            qCWarning(LOG_KBIBTEX_IO) << "Error creating an ICU Transliterator instance: " << u_errorName(uec);
            if (translit != nullptr) delete translit;
            translit = nullptr;
        }
#endif // HAVE_ICU
    }

    ~Private()
    {
#ifdef HAVE_ICU
        if (translit != nullptr)
            delete translit;
#endif // HAVE_ICU
    }
};

#ifndef HAVE_ICU
/// The following array contains only 65536-2048 entries
/// as it skips the range reserved for surrogates (U+D800 to U+DFFF).
/// To get the index for a code point c in the range 0x0 to 0xffff,
/// apply the following formula:
///   index = c < 0xd800 ? c : c - 2048;
const size_t Encoder::Private::unidecode_pos[] = {
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    97, 913569, 921249, 921281, 921313, 921345, 921377, 871489, 358817, 358945, 921409, 921441, 914081, 921473, 719073, 662913, 715521, 713569, 713633, 713761, 713889, 714017, 714145, 715361, 714273, 716673, 876289, 921505, 914145, 876353, 914209, 913665,
    921537, 876481, 914529, 718785, 914849, 877185, 876673, 876865, 876897, 718945, 915329, 915553, 877793, 877345, 916257, 877281, 876993, 921569, 872577, 916673, 877729, 914497, 718913, 916225, 876737, 915265, 914881, 921601, 921633, 921665, 921697, 921729,
    921761, 129, 705, 545, 84801, 673, 764609, 1153, 225, 81473, 289, 1377, 33, 2305, 3169, 7393, 2529, 662497, 662817, 1, 2977, 75521, 869057, 609, 712577, 65, 358625, 921793, 920865, 921825, 871361, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    97, 0, 0, 0, 0, 0, 0, 0, 0, 872451, 0, 914146, 0, 921473, 872547, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 914210, 713796, 713540, 714564, 0,
    876481, 876481, 876481, 876481, 876481, 876481, 914402, 718785, 877185, 877185, 877185, 877185, 718945, 718945, 718945, 718945, 914849, 916257, 877281, 877281, 877281, 877281, 877281, 921409, 877281, 914497, 914497, 914497, 914497, 915265, 877826, 3842,
    129, 129, 129, 129, 129, 129, 642, 545, 673, 673, 673, 673, 81473, 81473, 81473, 81473, 84801, 3169, 7393, 7393, 7393, 7393, 7393, 662913, 7393, 75521, 75521, 75521, 75521, 65, 891426, 65,
    876481, 129, 876481, 129, 876481, 129, 718785, 545, 718785, 545, 718785, 545, 718785, 545, 914849, 84801, 914849, 84801, 877185, 673, 877185, 673, 877185, 673, 877185, 673, 877185, 673, 876865, 1153, 876865, 1153,
    876865, 1153, 876865, 1153, 876897, 225, 876897, 225, 718945, 81473, 718945, 81473, 718945, 81473, 718945, 81473, 718945, 81473, 915298, 407554, 915329, 289, 915553, 1377, 662497, 877793, 33, 877793, 33, 877793, 33, 877793,
    33, 877793, 33, 916257, 3169, 916257, 3169, 916257, 3169, 913826, 916257, 3169, 877281, 7393, 877281, 7393, 877281, 7393, 916450, 78370, 872577, 662817, 872577, 662817, 872577, 662817, 916673, 1, 916673, 1, 916673, 1,
    916673, 1, 877729, 2977, 877729, 2977, 877729, 2977, 914497, 75521, 914497, 75521, 914497, 75521, 914497, 75521, 914497, 75521, 914497, 75521, 916225, 609, 915265, 65, 915265, 914881, 358625, 914881, 358625, 914881, 358625, 1,
    705, 914529, 914529, 705, 0, 0, 0, 718785, 545, 914849, 914849, 914849, 84801, 0, 0, 0, 877185, 876673, 764609, 876865, 0, 918242, 718945, 718945, 915553, 1377, 33, 0, 0, 916257, 3169, 0,
    877281, 7393, 877282, 891682, 876993, 2529, 0, 0, 0, 0, 0, 2977, 877729, 2977, 877729, 914497, 75521, 0, 718913, 915265, 65, 914881, 358625, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 914850, 914914, 886850, 915874, 916002, 895682, 916258, 916322, 3618, 876481, 129, 718945, 81473, 877281, 7393, 914497, 75521, 914497, 75521, 914497, 75521, 914497, 75521, 914497, 75521, 0, 876481, 129,
    876481, 129, 914402, 642, 876865, 1153, 876865, 1153, 915553, 1377, 877281, 7393, 877281, 7393, 0, 0, 289, 914850, 914914, 886850, 876865, 1153, 0, 0, 916257, 3169, 876481, 129, 914402, 642, 877281, 7393,
    876481, 129, 876481, 129, 877185, 673, 877185, 673, 718945, 81473, 718945, 81473, 877281, 7393, 877281, 7393, 872577, 662817, 872577, 662817, 914497, 75521, 914497, 75521, 916673, 1, 877729, 2977, 0, 0, 876897, 225,
    0, 84801, 0, 0, 914881, 358625, 876481, 129, 877185, 673, 877281, 7393, 877281, 7393, 877281, 7393, 877281, 7393, 915265, 65, 33, 3169, 2977, 289, 371426, 919970, 876481, 718785, 545, 877793, 877729, 1,
    358625, 0, 0, 914529, 914497, 0, 877185, 673, 915329, 289, 0, 0, 872577, 662817, 915265, 65, 0, 0, 0, 705, 0, 545, 84801, 84801, 0, 0, 0, 673, 0, 0, 0, 289,
    1153, 1153, 876865, 0, 0, 0, 225, 225, 81473, 0, 718945, 33, 33, 33, 0, 0, 0, 2305, 3169, 3169, 916257, 0, 916450, 0, 0, 0, 0, 0, 662817, 662817, 662817, 0,
    872577, 0, 1, 0, 0, 0, 0, 0, 2977, 75521, 0, 869057, 0, 0, 0, 915265, 358625, 358625, 0, 0, 0, 0, 0, 0, 0, 914529, 0, 876865, 876897, 289, 0, 877793,
    662497, 0, 0, 886850, 0, 886850, 43298, 0, 0, 0, 2722, 918690, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 871489, 921249, 871489, 871489, 871489, 0, 0,
    0, 0, 914145, 914209, 921697, 0, 921697, 0, 871489, 0, 0, 921761, 0, 0, 0, 0, 876289, 0, 0, 0, 0, 0, 921441, 921473, 0, 0, 0, 0, 871361, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 871489, 0, 0, 0, 0, 0, 81473, 0, 0, 0, 921505, 0,
    0, 0, 0, 0, 0, 0, 876481, 0, 877185, 877185, 718945, 0, 877281, 0, 915265, 877281, 81473, 876481, 914529, 876865, 914849, 877185, 914881, 877185, 877826, 718945, 915553, 877793, 877345, 916257, 876737, 877281,
    876993, 872577, 0, 916673, 877729, 915265, 915202, 914722, 916642, 877281, 718945, 915265, 129, 673, 673, 81473, 65, 129, 705, 1153, 84801, 673, 358625, 673, 891426, 81473, 1377, 33, 2305, 3169, 712577, 7393,
    2529, 662817, 1, 1, 2977, 65, 905954, 546, 42850, 7393, 81473, 65, 7393, 65, 7393, 0, 705, 891426, 915265, 915265, 915265, 905954, 2529, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1377, 662817, 1, 289, 877826, 673, 0, 916673, 1, 916673, 916673, 1, 0, 0, 0, 0,
    877185, 877185, 914849, 876865, 877185, 914881, 718945, 718945, 915329, 877793, 916257, 718785, 915553, 718945, 914497, 914849, 876481, 914529, 718913, 876865, 914849, 877185, 914881, 914881, 718945, 915329, 915553, 877793, 877345, 916257, 877281, 876993,
    872577, 916673, 877729, 914497, 876673, 876897, 718785, 718785, 916673, 916673, 921249, 915265, 871489, 877185, 914497, 876481, 129, 705, 869057, 1153, 84801, 673, 358625, 358625, 81473, 289, 1377, 33, 2305, 3169, 7393, 2529,
    662817, 1, 2977, 75521, 764609, 225, 545, 545, 1, 1, 921249, 65, 871489, 673, 75521, 129, 673, 673, 84801, 1153, 673, 358625, 81473, 81473, 289, 33, 3169, 545, 1377, 81473, 75521, 84801,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 876865, 1153, 876865, 1153, 876865, 1153, 0, 0, 914881, 358625, 915553, 1377, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 914881, 358625, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 876481, 129, 876481, 129, 914402, 642, 877185, 673, 0, 0, 0, 0, 914881, 358625, 914881, 358625,
    0, 0, 718945, 81473, 718945, 81473, 877281, 7393, 0, 0, 0, 0, 877185, 673, 914497, 75521, 914497, 75521, 914497, 75521, 718785, 545, 0, 0, 915265, 65, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 876481, 914529, 876865, 914849, 877185, 914881, 877185, 0, 917026, 914881, 718945, 877793, 876737, 718785, 915553,
    876897, 915329, 876865, 718785, 877345, 915265, 916257, 916673, 877281, 914594, 876993, 915329, 872577, 916673, 718913, 877729, 872577, 914594, 916225, 916514, 915554, 877281, 876673, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 129, 705, 1153, 84801, 673, 358625, 673, 0, 908546, 358625, 81473, 33, 712577, 545, 1377, 225, 289, 1153, 545, 2305, 65, 3169, 1, 7393, 917474, 2529, 289, 662817, 1, 869057, 2977,
    662817, 917474, 609, 919330, 892322, 7393, 764609, 917922, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 673, 673, 129, 7393, 81473, 673, 673, 129, 129, 7393, 0, 75521, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 871489, 705, 1153, 84801, 225, 609, 358625, 225, 2977, 65, 1377, 1377, 33, 2305, 2305, 3169,
    3169, 1, 871489, 2529, 2529, 358625, 358625, 662497, 662817, 1, 2977, 0, 0, 0, 0, 0, 920546, 920610, 911202, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 129, 129, 609, 129, 65, 129, 705, 2977, 2977, 891426, 289, 225, 663394, 84801, 884450, 662817, 358625, 1, 276642, 1, 84801, 2977, 358625, 0, 771266, 0, 0, 0, 0, 0,
    0, 764609, 662497, 1377, 33, 2305, 3169, 225, 609, 65, 65, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    715521, 713569, 713633, 713761, 713889, 714017, 714145, 715361, 714273, 716673, 921345, 914081, 719073, 0, 0, 0, 0, 0, 0, 0, 0, 129, 609, 0, 65, 0, 0, 0, 0, 0, 2529, 0,
    0, 0, 0, 0, 0, 0, 546, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 358626, 0, 1, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 869057, 0, 0, 0, 0, 1377, 0, 0, 0, 3170, 0, 1153, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 869057, 65, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 715521, 713569, 713633, 713761, 713889, 714017, 714145, 715361, 714273, 716673, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 921409, 0, 0, 0, 0, 705, 1153, 1153, 84801, 917794, 225, 609, 358625, 225, 2977, 2977, 65, 911138, 1377,
    33, 2305, 3169, 1, 1, 0, 2529, 2529, 1, 662497, 662817, 276642, 2977, 0, 0, 0, 129, 129, 129, 7393, 7393, 129, 673, 673, 673, 673, 81473, 81473, 75521, 75521, 75521, 7393,
    0, 0, 81473, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    225, 1, 3169, 662817, 705, 33, 1377, 871489, 869057, 2305, 764609, 84801, 2977, 33, 1153, 3169, 1, 84801, 358625, 2977, 65, 2529, 289, 545, 50946, 225, 663394, 884450, 0, 1, 1, 84801,
    2977, 0, 0, 1153, 662497, 0, 129, 129, 81473, 81473, 75521, 75521, 673, 673, 7393, 7393, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 2305, 2305, 225, 0, 129, 129, 81473, 81473, 75521, 75521, 662817, 33, 673, 673, 673, 406594, 7393, 7393, 7393, 917346, 168962, 892803, 451778, 889251, 603778, 739394, 69667, 128066, 891747, 603778, 317762,
    908835, 417698, 884451, 603778, 317762, 908835, 417698, 884451, 603778, 603778, 235778, 905955, 358978, 880131, 569698, 3970, 662818, 662818, 418, 418, 418, 920290, 276866, 276866, 276866, 69698, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 871491, 0, 0, 0, 0, 0, 0, 0, 919778, 892803, 451778, 869954, 662818, 907011, 764610, 3970,
    662817, 33, 0, 0, 0, 0, 715521, 713569, 713633, 713761, 713889, 714017, 714145, 715361, 714273, 716673, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 2305, 2305, 225, 0, 129, 129, 81473, 81473, 75521, 75521, 662817, 33, 0, 0, 673, 406594, 0, 0, 7393, 917346, 168962, 892803, 451778, 889251, 603778, 739394, 69667, 128066, 891747, 603778, 317762,
    908835, 417698, 884451, 603778, 317762, 908835, 417698, 884451, 603778, 0, 235778, 905955, 358978, 880131, 569698, 3970, 662818, 0, 418, 0, 0, 0, 276866, 276866, 276866, 69698, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2977, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 662818, 907011, 0, 3970,
    662817, 33, 0, 0, 0, 0, 715521, 713569, 713633, 713761, 713889, 714017, 714145, 715361, 714273, 716673, 662818, 662818, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 2305, 2305, 0, 0, 129, 129, 81473, 81473, 75521, 75521, 0, 0, 0, 0, 673, 406594, 0, 0, 7393, 917346, 168962, 892803, 451778, 889251, 603778, 739394, 69667, 128066, 891747, 603778, 317762,
    908835, 417698, 884451, 603778, 317762, 908835, 417698, 884451, 603778, 0, 235778, 905955, 358978, 880131, 569698, 3970, 662818, 0, 418, 418, 0, 920290, 276866, 0, 276866, 69698, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 892803, 451778, 869954, 662818, 0, 764610, 0,
    0, 0, 0, 0, 0, 0, 715521, 713569, 713633, 713761, 713889, 714017, 714145, 715361, 714273, 716673, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 2305, 2305, 225, 0, 129, 129, 81473, 81473, 75521, 75521, 662817, 33, 673, 0, 673, 406594, 7393, 0, 7393, 917346, 168962, 892803, 451778, 889251, 603778, 739394, 69667, 128066, 891747, 603778, 317762,
    908835, 417698, 884451, 603778, 317762, 908835, 417698, 884451, 603778, 0, 235778, 905955, 358978, 880131, 569698, 3970, 662818, 0, 418, 418, 0, 920290, 276866, 276866, 276866, 69698, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 871491, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    662817, 33, 0, 0, 0, 0, 715521, 713569, 713633, 713761, 713889, 714017, 714145, 715361, 714273, 716673, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 2305, 2305, 225, 0, 129, 129, 81473, 81473, 75521, 75521, 662817, 33, 0, 0, 673, 406594, 0, 0, 7393, 917346, 168962, 892803, 451778, 889251, 603778, 739394, 69667, 128066, 891747, 603778, 317762,
    908835, 417698, 884451, 603778, 317762, 908835, 417698, 884451, 603778, 0, 235778, 905955, 358978, 880131, 569698, 3970, 662818, 0, 418, 418, 0, 920290, 276866, 276866, 276866, 69698, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 662818, 907011, 0, 3970,
    662817, 33, 0, 0, 0, 0, 715521, 713569, 713633, 713761, 713889, 714017, 714145, 715361, 714273, 716673, 0, 610, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 2305, 225, 0, 129, 129, 81473, 81473, 75521, 75521, 0, 0, 0, 673, 673, 406594, 0, 7393, 7393, 917346, 168962, 0, 0, 0, 603778, 739394, 0, 128066, 0, 603778, 317762,
    0, 0, 0, 603778, 317762, 0, 0, 0, 603778, 603778, 235778, 0, 0, 0, 569698, 3970, 662818, 662818, 418, 418, 418, 920290, 276866, 276866, 276866, 69698, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 715521, 713569, 713633, 713761, 713889, 714017, 714145, 715361, 714273, 716673, 715490, 718659, 718660, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 2305, 2305, 225, 0, 129, 129, 81473, 81473, 75521, 75521, 662817, 33, 0, 673, 673, 406594, 0, 7393, 7393, 917346, 168962, 892803, 451778, 889251, 603778, 739394, 69667, 128066, 891747, 603778, 317762,
    908835, 417698, 884451, 603778, 317762, 908835, 417698, 884451, 603778, 0, 235778, 905955, 358978, 880131, 569698, 3970, 662818, 662818, 418, 418, 0, 920290, 276866, 276866, 276866, 69698, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    662817, 33, 0, 0, 0, 0, 715521, 713569, 713633, 713761, 713889, 714017, 714145, 715361, 714273, 716673, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 2305, 225, 0, 129, 129, 81473, 81473, 75521, 75521, 662817, 33, 0, 673, 673, 406594, 0, 7393, 7393, 917346, 168962, 892803, 451778, 889251, 603778, 739394, 69667, 128066, 891747, 603778, 317762,
    908835, 417698, 884451, 603778, 317762, 908835, 417698, 884451, 603778, 0, 235778, 905955, 358978, 880131, 569698, 3970, 662818, 662818, 418, 418, 0, 920290, 276866, 276866, 276866, 69698, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 418, 0,
    662817, 33, 0, 0, 0, 0, 715521, 713569, 713633, 713761, 713889, 714017, 714145, 715361, 714273, 716673, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 2305, 225, 0, 129, 129, 81473, 81473, 75521, 75521, 662817, 33, 0, 673, 673, 406594, 0, 7393, 7393, 917346, 168962, 892803, 451778, 889251, 603778, 739394, 69667, 128066, 891747, 603778, 317762,
    908835, 417698, 884451, 603778, 317762, 908835, 417698, 884451, 603778, 0, 235778, 905955, 358978, 880131, 569698, 3970, 662818, 662818, 418, 418, 418, 920290, 276866, 276866, 276866, 69698, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    662817, 33, 0, 0, 0, 0, 715521, 713569, 713633, 713761, 713889, 714017, 714145, 715361, 714273, 716673, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 1377, 663394, 663394, 663394, 892323, 663394, 3170, 545, 546, 546, 1, 546, 65, 84801, 2977, 891426, 891426, 908547, 3169, 84801, 2977, 891426, 891426, 891426, 3169, 705, 2529, 905954, 764609, 905954, 764609,
    905954, 2305, 65, 662817, 869057, 33, 33, 609, 1, 910114, 1, 225, 33, 712577, 225, 0, 129, 129, 129, 129, 81473, 81473, 75521, 75521, 75521, 75521, 0, 0, 0, 0, 0, 0,
    673, 642, 7393, 81473, 81473, 81473, 914146, 0, 0, 0, 0, 0, 0, 0, 871361, 0, 715521, 713569, 713633, 713761, 713889, 714017, 714145, 715361, 714273, 716673, 920866, 914210, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 129, 705, 1153, 84801, 673, 869057, 358625, 2977, 81473, 892322, 33, 2305, 3169, 7393, 919330, 358626,
    662817, 1, 908546, 75521, 2529, 1377, 771266, 919714, 276642, 546, 43298, 886850, 910083, 882531, 663394, 289, 225, 0, 0, 81442, 662497, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    1153, 1378, 3169, 84801, 50946, 33, 2305, 705, 30786, 1, 3842, 0, 289, 10626, 546, 1377, 2977, 2529, 225, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 129, 642, 3970, 3971, 7362, 673, 7331, 7330, 7393, 610, 611, 78370, 105058, 75521, 94114, 88354, 91234, 107938, 75490, 81442, 81473, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 1153, 1378, 1154, 3169, 3618, 3394, 84801, 33, 1826, 2274, 1602, 2722, 2946, 2498, 194, 2305, 705, 706, 1, 3842, 3170, 289, 546, 1377,
    2977, 2529, 225, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    876481, 914402, 0, 914529, 718785, 914849, 914849, 877185, 0, 0, 915329, 915553, 877793, 877345, 0, 877281, 0, 0, 0, 0, 0, 0, 0, 0, 876993, 0, 0, 877729, 914497, 0, 0, 0,
    718913, 916225, 914881, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 886914, 705, 84801, 764609, 2305, 3169, 2529, 662817, 662817, 1, 2977, 358625, 0, 0, 0, 891426, 718945, 0, 2529, 914497, 0,
    705, 84801, 764609, 1153, 1377, 33, 2305, 3169, 2529, 662817, 1, 0, 869057, 712577, 358625, 129, 0, 84801, 673, 673, 0, 0, 81473, 0, 0, 75521, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    876481, 129, 914529, 705, 914529, 705, 914529, 705, 718785, 545, 914849, 84801, 914849, 84801, 914849, 84801, 914849, 84801, 914849, 84801, 877185, 673, 877185, 673, 877185, 673, 877185, 673, 877185, 673, 876673, 764609,
    876865, 1153, 876897, 225, 876897, 225, 876897, 225, 876897, 225, 876897, 225, 718945, 81473, 718945, 81473, 915553, 1377, 915553, 1377, 915553, 1377, 877793, 33, 877793, 33, 877793, 33, 877793, 33, 877345, 2305,
    877345, 2305, 877345, 2305, 916257, 3169, 916257, 3169, 916257, 3169, 916257, 3169, 877281, 7393, 877281, 7393, 877281, 7393, 877281, 7393, 876993, 2529, 876993, 2529, 872577, 662817, 872577, 662817, 872577, 662817, 872577, 662817,
    916673, 1, 916673, 1, 916673, 1, 916673, 1, 916673, 1, 877729, 2977, 877729, 2977, 877729, 2977, 877729, 2977, 914497, 75521, 914497, 75521, 914497, 75521, 914497, 75521, 914497, 75521, 718913, 869057, 718913, 869057,
    916225, 609, 916225, 609, 916225, 609, 916225, 609, 916225, 609, 876737, 712577, 876737, 712577, 915265, 65, 914881, 358625, 914881, 358625, 914881, 358625, 225, 2977, 609, 65, 129, 1, 1, 1, 916898, 0,
    876481, 129, 876481, 129, 876481, 129, 876481, 129, 876481, 129, 876481, 129, 876481, 129, 876481, 129, 876481, 129, 876481, 129, 876481, 129, 876481, 129, 877185, 673, 877185, 673, 877185, 673, 877185, 673,
    877185, 673, 877185, 673, 877185, 673, 877185, 673, 718945, 81473, 718945, 81473, 877281, 7393, 877281, 7393, 877281, 7393, 877281, 7393, 877281, 7393, 877281, 7393, 877281, 7393, 877281, 7393, 877281, 7393, 877281, 7393,
    877281, 7393, 877281, 7393, 914497, 75521, 914497, 75521, 914497, 75521, 914497, 75521, 914497, 75521, 914497, 75521, 914497, 75521, 915265, 65, 915265, 65, 915265, 65, 915265, 65, 915938, 162, 718913, 869057, 915265, 65,
    129, 69698, 129, 69698, 129, 69698, 129, 69698, 876481, 877058, 876481, 877058, 876481, 877058, 876481, 877058, 673, 72578, 673, 72578, 673, 72578, 0, 0, 877185, 877154, 877185, 877154, 877185, 877154, 0, 0,
    673, 72578, 673, 72578, 673, 72578, 673, 72578, 877185, 877154, 877185, 877154, 877185, 877154, 877185, 877154, 81473, 400770, 81473, 400770, 81473, 400770, 81473, 400770, 718945, 915106, 718945, 915106, 718945, 915106, 718945, 915106,
    7393, 78338, 7393, 78338, 7393, 78338, 0, 0, 877281, 877250, 877281, 877250, 877281, 877250, 0, 0, 65, 3938, 65, 3938, 65, 3938, 65, 3938, 0, 915234, 0, 915234, 0, 915234, 0, 915234,
    7393, 78338, 7393, 78338, 7393, 78338, 7393, 78338, 877281, 877250, 877281, 877250, 877281, 877250, 877281, 877250, 129, 129, 673, 673, 673, 673, 81473, 81473, 7393, 7393, 65, 65, 7393, 7393, 0, 0,
    406594, 739939, 406594, 739939, 406594, 739939, 406594, 739939, 877090, 877059, 877090, 877059, 877090, 877059, 877090, 877059, 859010, 858979, 859010, 858979, 859010, 858979, 859010, 858979, 877186, 877155, 877186, 877155, 877186, 877155, 877186, 877155,
    891682, 891651, 891682, 891651, 891682, 891651, 891682, 891651, 877282, 877251, 877282, 877251, 877282, 877251, 877282, 877251, 129, 129, 406594, 406594, 406594, 0, 129, 406594, 876481, 876481, 876481, 876481, 877090, 0, 81473, 0,
    0, 0, 859010, 859010, 859010, 0, 673, 859010, 877185, 877185, 877185, 877185, 877186, 0, 0, 0, 81473, 81473, 81473, 81473, 0, 0, 81473, 81473, 718945, 718945, 718945, 718945, 0, 0, 0, 0,
    65, 65, 65, 65, 662817, 907010, 65, 65, 915265, 915265, 915265, 915265, 916706, 0, 0, 0, 0, 0, 891682, 891682, 891682, 0, 7393, 891682, 877281, 877281, 877281, 877281, 877282, 0, 0, 0,
    97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 97, 0, 0, 0, 0, 0, 921473, 921473, 921473, 921473, 921473, 921473, 920866, 0, 871489, 871489, 914081, 871489, 921249, 921249, 914082, 921249,
    0, 0, 0, 0, 719073, 875138, 875139, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 871489, 921249, 0, 0, 0, 0, 0, 914145, 914209, 0, 913570, 0, 0, 0,
    0, 0, 0, 0, 662913, 921601, 921665, 914338, 914274, 913634, 0, 0, 0, 0, 921409, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 97,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    914658, 0, 914786, 876771, 915810, 0, 0, 877635, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 916770, 917090, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    878115, 878211, 718785, 0, 0, 882147, 882243, 0, 0, 0, 1153, 876897, 712577, 876897, 225, 0, 718945, 718945, 877793, 33, 0, 916257, 916386, 0, 0, 876993, 921569, 872577, 872577, 872577, 916834, 0,
    0, 877731, 0, 0, 914881, 0, 877281, 0, 914881, 0, 915553, 876481, 914529, 718785, 0, 673, 877185, 876673, 0, 877345, 7393, 0, 0, 0, 0, 81473, 0, 876675, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 914849, 84801, 673, 81473, 289, 0, 0, 0, 0, 0, 0, 0, 0, 0, 713668, 714308, 713924, 714436, 714692, 714948, 714052, 715076, 714180, 714820, 715204, 715332, 713539,
    718945, 718946, 718947, 915426, 718913, 718914, 718915, 718916, 915490, 876737, 878018, 878019, 877793, 718785, 914849, 877345, 81473, 869090, 869091, 918306, 869057, 869058, 869059, 869060, 910722, 712577, 712578, 910659, 33, 545, 84801, 2305,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 921473, 0, 0, 662913, 921633, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 920865, 0, 920866, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 914146, 914210, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 914145, 914209, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 871587, 871683, 871779, 871875, 871971, 872067, 872163, 872259, 872355, 715460, 715588, 715716,
    715844, 715972, 716100, 716228, 716356, 716484, 716612, 716740, 875362, 875458, 875554, 875650, 875746, 875842, 875938, 876034, 876130, 875235, 875331, 875427, 875523, 875619, 875715, 875811, 875907, 876003, 876099, 876195, 872643, 872739, 872835, 872931,
    873027, 873123, 873219, 873315, 873411, 873507, 873603, 873699, 873795, 873891, 873987, 874083, 874179, 874275, 874371, 874467, 874563, 874659, 874755, 874851, 874947, 875043, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 913890, 914018, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 876291, 876354, 876387, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    911715, 395781, 3970, 869699, 3970, 858051, 486181, 0, 883587, 883587, 366178, 891939, 869316, 869316, 910466, 87588, 87588, 910466, 671330, 911043, 486018, 486018, 869443, 869443, 859844, 247298, 0, 907106, 907106, 911907, 883299, 579298,
    821059, 859972, 859972, 725700, 870340, 870340, 662501, 903651, 857924, 0, 0, 579298, 859331, 859331, 859331, 671330, 671330, 767300, 87588, 0, 87588, 0, 858051, 858051, 0, 893859, 107938, 107938, 907203, 139586, 882435, 882435,
    882435, 81218, 869698, 712578, 712578, 486020, 0, 0, 100259, 879651, 0, 72547, 743748, 743748, 743748, 918114, 495811, 395781, 395781, 712901, 817219, 0, 918114, 107938, 857796, 910371, 7330, 764740, 887331, 859331, 0, 859331,
    859331, 0, 569698, 461378, 461539, 107938, 839748, 545218, 897891, 81221, 819396, 662498, 662498, 400739, 400739, 803652, 803652, 461539, 461539, 461539, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    869698, 778435, 358627, 906051, 869698, 892227, 887234, 905539, 858051, 887234, 858434, 358978, 486181, 819394, 727236, 486018, 662500, 883587, 542658, 879075, 725570, 764612, 712578, 859331, 366178, 891939, 395781, 671330, 911715, 893187, 910371, 329282,
    859331, 870467, 288387, 712578, 417698, 613538, 912386, 819396, 882915, 869316, 911715, 859331, 72547, 662980, 81189, 775620, 486018, 495811, 767299, 911043, 461381, 869699, 775620, 869698, 775620, 486018, 662980, 400739, 869443, 454178, 81218, 859844,
    870467, 247298, 90467, 886083, 495811, 764612, 910466, 907106, 911907, 911907, 579298, 662500, 870467, 883299, 276643, 910466, 725570, 898179, 859331, 662498, 859972, 743779, 870340, 918114, 911043, 640771, 853572, 3970, 903651, 857924, 869572, 107938,
    461379, 610, 767299, 663141, 107172, 864452, 645730, 606178, 366178, 878787, 645730, 821059, 579298, 898179, 859331, 859331, 859331, 907203, 72578, 910851, 542658, 358627, 819394, 819394, 887715, 87588, 100260, 107938, 893859, 887234, 894339, 887234,
    107938, 907203, 398180, 912386, 870467, 892035, 663139, 81189, 870596, 769219, 279746, 882435, 81218, 405861, 910851, 869444, 869698, 3970, 486020, 785092, 100259, 461378, 886083, 859331, 870467, 879651, 400739, 912483, 871234, 663140, 72547, 869443,
    398180, 743748, 869698, 911715, 725572, 542658, 495811, 712901, 817219, 918114, 542658, 870724, 107938, 857796, 887331, 819396, 454178, 910371, 892035, 869699, 7330, 764740, 887331, 859331, 859844, 712581, 569698, 461378, 888291, 725700, 886083, 395781,
    454178, 461539, 107938, 839748, 545218, 545218, 897891, 569698, 81221, 276643, 858979, 870467, 819396, 752196, 461378, 276643, 725570, 662498, 400739, 803652, 461539, 911907, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    97, 914081, 719073, 0, 0, 0, 0, 800836, 914145, 914209, 914146, 914210, 0, 0, 0, 0, 0, 0, 0, 0, 921601, 921665, 0, 0, 921601, 921665, 921601, 921665, 0, 921249, 921249, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 859331, 610980, 276866, 0, 0, 0, 0, 0,
    0, 920930, 129, 921058, 81473, 921186, 75521, 920994, 673, 921122, 7393, 168962, 451778, 518530, 771522, 180482, 461378, 171842, 454178, 177602, 458978, 276866, 869954, 859331, 486018, 288386, 871234, 279746, 870082, 285506, 871106, 317762,
    417698, 400739, 885411, 871364, 864291, 886851, 320642, 420098, 326402, 424898, 603778, 610978, 613538, 606178, 611138, 69698, 358978, 235778, 400770, 725570, 645730, 918114, 368578, 247298, 72578, 361378, 238658, 78338, 366178, 244418, 569698, 819394,
    579298, 572098, 576898, 913155, 3970, 913347, 107938, 913251, 105058, 662818, 907106, 858434, 858050, 858178, 913059, 610, 91234, 88354, 94114, 3169, 920482, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 920930, 129, 921058, 81473, 921186, 75521, 920994, 673, 921122, 7393, 168962, 451778, 518530, 771522, 180482, 461378, 171842, 454178, 177602, 458978, 276866, 869954, 859331, 486018, 288386, 871234, 279746, 870082, 285506, 871106, 317762,
    417698, 400739, 885411, 871364, 864291, 886851, 320642, 420098, 326402, 424898, 603778, 610978, 613538, 606178, 611138, 69698, 358978, 235778, 400770, 725570, 645730, 918114, 368578, 247298, 72578, 361378, 238658, 78338, 366178, 244418, 569698, 819394,
    579298, 572098, 576898, 913155, 3970, 913347, 107938, 913251, 105058, 662818, 907106, 858434, 858050, 858178, 913059, 610, 91234, 88354, 94114, 3169, 920482, 912867, 912963, 920290, 869058, 920354, 920418, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 705, 2529, 918754, 764609, 84801, 2977, 878434, 33, 1153, 1377, 225, 289, 662497, 712577, 870468, 741444, 859332, 907107, 912387, 882627, 907875, 917218, 878530, 917858, 886947, 878307, 887043,
    878499, 905571, 878403, 887139, 719300, 764484, 887235, 911235, 910467, 911811, 0, 0, 0, 0, 0, 0, 0, 1153, 1378, 1154, 3169, 3618, 3394, 84801, 50946, 33, 1826, 2274, 1602, 2722, 2946, 2498,
    0, 2305, 705, 30786, 0, 1, 3842, 0, 289, 10626, 546, 1377, 2977, 2529, 225, 129, 642, 3970, 3971, 7362, 673, 7331, 7330, 7393, 610, 611, 78370, 105058, 75521, 94114, 88354, 91234,
    107938, 75490, 81442, 81473, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    873219, 873891, 872931, 873699, 873795, 872739, 874371, 913954, 873507, 716996, 873603, 874467, 874083, 873315, 717252, 718148, 717124, 717892, 718020, 716868, 718404, 872643, 717508, 358821, 717764, 718532, 718276, 717380, 717636, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    1153, 3169, 84801, 33, 2305, 705, 1, 0, 289, 546, 1377, 2977, 2529, 225, 451778, 603778, 417698, 418, 569698, 358978, 276866, 129, 128066, 69667, 168962, 317762, 235778, 69698, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 891459, 417698, 914466, 879267, 919266, 2530, 582146, 0, 0, 915362, 0, 0, 0, 0, 0,
    919394, 919010, 0, 918818, 918370, 915618, 916066, 914978, 789219, 789188, 919458, 919074, 0, 0, 120898, 120130, 876898, 892419, 877347, 876867, 877827, 0, 0, 0, 0, 887618, 819490, 0, 219970, 917666, 219202, 0,
    0, 0, 0, 0, 0, 0, 0, 897603, 0, 876994, 892515, 877443, 876963, 662819, 662821, 0, 42850, 860194, 0, 42626, 919586, 919138, 0, 918882, 918434, 916130, 919650, 919202, 0, 918946, 918498, 916194,
    0, 0, 719044, 914530, 917538, 917602, 718788, 876579, 917730, 915042, 69698, 915170, 83586, 915682, 915746, 51810, 2274, 918562, 802499, 918626, 62530, 820163, 823363, 919522, 853444, 877539, 916578, 920098, 916962, 917154, 877923, 876483,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    906339, 864452, 0, 0, 535299, 910466, 869699, 0, 0, 0, 0, 0, 869698, 0, 0, 0, 0, 0, 0, 0, 0, 0, 910563, 0, 0, 0, 0, 0, 743108, 0, 0, 0,
    0, 905283, 0, 0, 746563, 0, 0, 0, 869570, 869444, 0, 712741, 895587, 800835, 712581, 107172, 869443, 713060, 883299, 910466, 640771, 858434, 0, 569698, 662500, 869698, 869699, 902403, 398181, 764740, 0, 0,
    0, 870980, 764612, 725762, 910466, 912771, 0, 870596, 755012, 288386, 869698, 662661, 535301, 894339, 901923, 358627, 276643, 0, 0, 0, 869570, 0, 0, 663140, 891939, 884739, 905283, 288386, 869698, 803652, 869700, 723524,
    0, 0, 0, 795331, 819524, 869698, 542658, 486018, 107938, 897315, 739908, 0, 0, 0, 409411, 869570, 81411, 906915, 0, 870596, 0, 395811, 712578, 908643, 911043, 81411, 502531, 569698, 896931, 690852, 911043, 870340,
    870212, 107938, 870980, 887234, 719267, 662498, 400739, 910466, 395811, 690852, 279746, 0, 662661, 894339, 276866, 0, 0, 180483, 247298, 317762, 276643, 100260, 743170, 908643, 0, 819396, 869699, 750660, 107938, 899331, 502531, 839748,
    910563, 911715, 0, 0, 72547, 764740, 894339, 542658, 0, 897315, 0, 486018, 0, 0, 0, 0, 857924, 0, 882339, 542661, 461378, 898179, 0, 461379, 288387, 0, 0, 898179, 815299, 857924, 859331, 542658,
    0, 87588, 893187, 427298, 713060, 697572, 0, 0, 727236, 743779, 755012, 775620, 398181, 0, 857795, 486181, 545218, 869444, 0, 835779, 910563, 0, 725570, 891939, 288386, 0, 775620, 0, 911715, 869444, 662499, 645730,
    750532, 918114, 897315, 662499, 662499, 690852, 878787, 767299, 882626, 869572, 795332, 0, 0, 663139, 0, 542658, 81219, 905539, 853572, 750530, 858436, 673, 906243, 869698, 870980, 907299, 486020, 0, 400739, 405861, 712578, 0,
    896931, 748484, 800835, 892227, 288386, 869316, 869955, 0, 0, 358627, 712900, 486020, 912483, 81188, 910563, 542658, 0, 400739, 712578, 486020, 0, 486018, 0, 887331, 81187, 723524, 891939, 0, 358978, 542661, 793156, 0,
    712579, 880419, 892227, 894339, 869443, 878787, 100260, 545218, 879651, 673, 545218, 0, 0, 72547, 905283, 869572, 398212, 107938, 0, 461539, 869698, 869572, 775620, 896739, 695810, 537858, 859331, 0, 679811, 911043, 712580, 912483,
    0, 906723, 869699, 712578, 870467, 486019, 81218, 418, 869698, 171842, 918114, 857795, 406594, 0, 171842, 81187, 910563, 81187, 910371, 0, 0, 81220, 288386, 911715, 0, 502531, 870340, 869570, 859331, 0, 276644, 180483,
    276646, 72578, 888003, 100259, 906339, 663140, 81219, 712578, 764611, 640772, 746563, 764612, 775620, 725762, 918114, 606178, 910851, 911715, 81219, 0, 398180, 891171, 3169, 81219, 542658, 764610, 869316, 906147, 0, 671330, 0, 0,
    537858, 800835, 869698, 743139, 0, 869570, 857922, 887234, 0, 0, 910947, 0, 0, 0, 0, 903075, 910371, 910563, 695810, 405892, 704291, 903075, 903075, 869699, 713060, 0, 0, 0, 0, 0, 910275, 859844,
    905283, 7330, 662498, 905539, 395811, 502531, 755012, 409411, 545218, 139586, 743779, 800836, 0, 864452, 806467, 0, 0, 0, 0, 0, 0, 454178, 100259, 859331, 910851, 643171, 409380, 903651, 886563, 870082, 673, 910563,
    911715, 673, 663141, 90467, 180482, 81218, 454178, 712579, 815299, 896931, 486018, 743139, 870467, 0, 0, 910275, 0, 878787, 406594, 870724, 662500, 890691, 746563, 879651, 366178, 81187, 542658, 869316, 910755, 0, 0, 0,
    0, 0, 405892, 695810, 744258, 893283, 893859, 870467, 910563, 712578, 0, 906243, 712899, 712578, 0, 0, 744132, 486018, 743779, 317762, 100259, 869570, 244418, 878275, 0, 0, 0, 891171, 7330, 712581, 910851, 72578,
    912771, 869698, 882626, 0, 797252, 712580, 908643, 858180, 869698, 870467, 712578, 712580, 139586, 486018, 395811, 0, 905763, 542658, 0, 795331, 878275, 395811, 100259, 857922, 0, 100259, 395811, 513571, 400739, 903075, 743779, 0,
    725570, 712579, 90468, 869572, 87587, 911715, 857795, 869570, 903075, 725570, 740323, 495812, 725762, 725762, 0, 0, 713060, 690851, 139586, 0, 912771, 368578, 891939, 406594, 869956, 882626, 764610, 0, 0, 0, 0, 903075,
    895587, 898659, 427299, 721604, 725570, 879075, 0, 81187, 712579, 864452, 395781, 0, 0, 886563, 910371, 918114, 886563, 107938, 7330, 180483, 910371, 793156, 0, 910371, 911043, 803652, 869444, 368578, 400739, 910563, 903075, 795332,
    869698, 871108, 815299, 712901, 712579, 778435, 910563, 0, 486018, 799300, 869698, 486018, 869699, 0, 417698, 869698, 910563, 740323, 107172, 513571, 395780, 908643, 690852, 870467, 879075, 817220, 180483, 395780, 894339, 0, 712578, 0,
    712578, 857668, 835780, 110051, 0, 803652, 918114, 871108, 0, 461378, 892611, 750660, 81219, 180483, 0, 888291, 908739, 0, 662980, 893571, 903075, 918114, 888291, 906243, 721603, 486019, 523652, 712578, 107938, 870724, 663140, 743748,
    869316, 486018, 613538, 869316, 869698, 107938, 869698, 100259, 663140, 719267, 740323, 276866, 502531, 911715, 0, 869443, 905859, 906339, 395780, 0, 368578, 755012, 671330, 887234, 0, 898179, 110051, 486018, 0, 857668, 712741, 905763,
    81187, 643172, 905283, 891939, 869698, 887234, 886563, 0, 0, 0, 886563, 0, 0, 906243, 545218, 906339, 908355, 739395, 886083, 712578, 764740, 869698, 908451, 906243, 244418, 869443, 700932, 869443, 911715, 879651, 803652, 0,
    0, 0, 0, 110051, 542658, 317762, 795331, 815299, 662501, 870596, 100259, 712578, 545218, 712578, 907587, 764611, 0, 910371, 764610, 869698, 901923, 398181, 690851, 486018, 276643, 853572, 81282, 535299, 69667, 0, 712580, 870467,
    0, 0, 764740, 542660, 910947, 869570, 819394, 81411, 579298, 107172, 712900, 869698, 904707, 690852, 712578, 110051, 276643, 918114, 869698, 417698, 0, 542660, 882435, 739395, 139586, 545218, 288386, 837699, 725762, 81282, 662500, 0,
    882819, 744132, 0, 719267, 610980, 897891, 869443, 911907, 901635, 725762, 663140, 569698, 0, 0, 795331, 712578, 911907, 870467, 90468, 406532, 817220, 839748, 87587, 819394, 903075, 857922, 869955, 542660, 870467, 912386, 739939, 869570,
    740323, 869572, 870467, 819396, 409380, 890691, 0, 409380, 804164, 358627, 859844, 799300, 892035, 910563, 752196, 891939, 858180, 815300, 0, 171842, 911043, 841284, 869698, 795332, 107172, 869699, 100259, 288386, 0, 800835, 3970, 898179,
    821060, 912579, 107938, 869698, 890691, 819394, 502531, 90467, 0, 513572, 750532, 803652, 0, 869444, 882819, 857668, 819396, 817220, 857795, 0, 87587, 420098, 406594, 0, 725572, 904707, 542660, 495811, 107938, 81380, 912771, 366178,
    81411, 911043, 329283, 486018, 81282, 897315, 486018, 910371, 366178, 869954, 869570, 610980, 110051, 0, 358978, 713059, 139586, 910371, 910563, 662498, 869698, 910563, 882626, 906339, 427298, 839748, 662498, 486018, 329283, 0, 676452, 750532,
    893859, 712900, 0, 0, 869699, 739523, 486018, 81411, 912386, 795331, 901923, 139586, 857795, 883299, 0, 891939, 869570, 744132, 107172, 886083, 400739, 0, 821059, 81221, 288387, 171842, 871234, 740323, 398181, 910851, 610978, 400739,
    542660, 81282, 579298, 671330, 712581, 100260, 81219, 883011, 906339, 893859, 918114, 427299, 815300, 795332, 910179, 395811, 815300, 366178, 906819, 662498, 395811, 0, 803652, 0, 864580, 870082, 662498, 869955, 819394, 905859, 712900, 712581,
    767300, 0, 662498, 0, 545218, 0, 110051, 673, 755524, 821059, 910371, 857924, 908355, 821059, 329282, 0, 821060, 911043, 892227, 542658, 793156, 767300, 869828, 417698, 0, 893859, 896739, 662500, 725762, 725700, 107172, 815300,
    883587, 0, 725762, 0, 712578, 918114, 746563, 892035, 907395, 700932, 857922, 673, 662498, 486018, 486018, 81219, 785092, 912579, 725700, 817220, 878787, 910371, 869698, 725762, 107938, 740323, 427299, 94114, 610978, 744260, 0, 542658,
    545218, 839748, 406532, 542658, 0, 545218, 764740, 819394, 107938, 0, 139586, 0, 0, 712900, 643172, 869698, 0, 486018, 725570, 0, 858051, 81221, 764611, 454178, 180482, 891939, 662979, 0, 671330, 700932, 869828, 912386,
    725570, 535299, 542658, 81221, 910947, 905283, 0, 713059, 90467, 712580, 662499, 7330, 898179, 0, 0, 276643, 0, 857668, 870852, 527011, 910466, 869700, 81190, 695810, 542660, 725570, 890691, 815300, 910563, 764740, 896739, 912195,
    713061, 81187, 815299, 803652, 0, 869699, 647491, 713061, 486020, 804164, 903075, 869698, 0, 486018, 486018, 870212, 107938, 892035, 81220, 870467, 418, 800836, 870467, 723523, 712899, 139586, 746563, 799300, 869698, 870340, 712580, 400739,
    882626, 400739, 100259, 795332, 886083, 803652, 395780, 0, 329283, 69667, 406594, 400739, 0, 869700, 713059, 905539, 0, 329283, 69667, 911043, 871108, 0, 640771, 857668, 542660, 857795, 545218, 100259, 513572, 288386, 869698, 395780,
    486181, 486021, 0, 495812, 0, 755012, 0, 789060, 395811, 750530, 0, 0, 405892, 0, 400739, 750660, 725570, 0, 910947, 545218, 0, 910563, 725570, 0, 725570, 0, 712580, 907299, 880419, 887234, 789060, 0,
    713060, 879651, 673, 107938, 857922, 869955, 819394, 869698, 671330, 0, 0, 0, 662980, 908643, 579298, 495812, 725572, 858180, 739524, 739395, 752196, 0, 0, 0, 0, 750530, 700932, 317762, 869444, 676452, 886563, 712578,
    908739, 0, 695810, 662980, 486020, 870467, 910371, 869699, 0, 0, 81220, 713221, 662498, 871108, 0, 910563, 910563, 870082, 910371, 0, 0, 317762, 712900, 841284, 0, 0, 0, 869698, 858051, 276643, 69667, 870980,
    0, 819396, 486018, 764612, 905859, 406594, 764611, 725762, 857795, 662499, 869316, 764739, 767299, 857668, 454178, 700932, 395780, 911715, 888291, 723523, 918114, 81187, 358627, 0, 870596, 0, 395812, 841283, 892227, 405861, 69667, 523652,
    895011, 542658, 107938, 0, 107938, 739939, 542658, 743139, 775620, 171842, 869828, 420098, 81411, 0, 461381, 486181, 912771, 918114, 906243, 879651, 72547, 882626, 815300, 395811, 712578, 906339, 81221, 0, 0, 743108, 666371, 100259,
    870467, 420098, 320642, 817219, 800836, 859844, 329283, 739395, 884739, 72547, 643172, 869698, 139586, 486018, 893571, 864452, 869828, 0, 882339, 662498, 107938, 542660, 744132, 0, 0, 0, 107938, 486018, 910371, 819394, 288387, 910563,
    869570, 400739, 906339, 81411, 0, 107938, 906243, 860100, 859972, 886563, 896739, 0, 640772, 908643, 870596, 869699, 907587, 887331, 398180, 869828, 869698, 409411, 279746, 7330, 821059, 764739, 72578, 0, 869699, 739522, 610978, 725762,
    764740, 542660, 395781, 395780, 569698, 884739, 81218, 545218, 0, 869698, 81219, 712899, 81218, 673, 743779, 679811, 610978, 712580, 542658, 712580, 100259, 803652, 817219, 495811, 486018, 0, 725572, 107938, 743779, 819524, 743108, 897891,
    0, 537858, 891939, 910371, 869698, 869572, 712578, 739395, 795331, 869699, 910563, 869954, 897315, 800836, 662500, 743779, 486020, 94114, 0, 0, 454178, 358627, 884739, 107172, 486018, 100260, 858434, 712578, 276646, 107938, 869698, 662500,
    486018, 857922, 864452, 859844, 662500, 579298, 495811, 898179, 869699, 888003, 244418, 869572, 898179, 764612, 3970, 767300, 676452, 81411, 107938, 461379, 776132, 895587, 673, 912386, 912386, 725570, 610, 0, 895011, 0, 0, 793156,
    0, 739939, 869699, 358627, 405861, 712580, 869572, 0, 906339, 905859, 461539, 887234, 775620, 662661, 81218, 893859, 542658, 398180, 666371, 870980, 94114, 906147, 515972, 704291, 643172, 320642, 317762, 870980, 725700, 461378, 81218, 0,
    727236, 870467, 755012, 427299, 870596, 902403, 800835, 244418, 486018, 821059, 910371, 72547, 890691, 721604, 858434, 690851, 368578, 871108, 180483, 893859, 395811, 869700, 870467, 891939, 869444, 910563, 910947, 662980, 662500, 910563, 288386, 739939,
    819394, 409411, 645730, 0, 81411, 603778, 676452, 723523, 743108, 891939, 81221, 795331, 0, 81218, 886083, 743779, 778435, 911043, 739522, 461539, 486020, 486020, 883587, 495811, 569698, 81411, 819396, 739395, 896931, 645730, 100260, 139586,
    139586, 906723, 0, 662500, 858692, 0, 892035, 743779, 110051, 417698, 869572, 869316, 887331, 739522, 7330, 0, 748483, 0, 857795, 81411, 704291, 0, 662501, 712578, 610978, 878275, 817220, 868932, 795331, 740323, 882626, 870212,
    725762, 897315, 899331, 0, 918114, 0, 910563, 366178, 81411, 857796, 910563, 0, 0, 366178, 662500, 244418, 785092, 892227, 527011, 676452, 139586, 673, 903075, 662500, 884739, 884739, 0, 662498, 870467, 662498, 870724, 180482,
    107938, 857795, 180482, 72578, 918114, 0, 750530, 712580, 461539, 72578, 906819, 395811, 700932, 366178, 662980, 725570, 545218, 7330, 610978, 406501, 666371, 750660, 545218, 905539, 542660, 171842, 666371, 713060, 406501, 542660, 898179, 0,
    662500, 892611, 858820, 869316, 725570, 712899, 869699, 712578, 662980, 288386, 276866, 907299, 743748, 545218, 800836, 69667, 0, 81220, 0, 0, 486019, 721603, 81218, 886083, 0, 896739, 139586, 789060, 171842, 908451, 897315, 713059,
    752196, 755524, 358627, 100259, 640772, 69667, 0, 0, 0, 0, 869698, 0, 0, 911715, 81411, 911043, 911043, 870467, 775620, 662498, 769219, 0, 0, 743139, 819394, 918114, 81218, 461381, 690851, 750530, 0, 100259,
    0, 0, 857922, 0, 395781, 821060, 908739, 879075, 81282, 0, 0, 712580, 0, 0, 0, 898179, 795332, 835779, 879651, 398180, 0, 887331, 870596, 486018, 891939, 276643, 0, 527011, 884739, 545218, 0, 0,
    0, 0, 107938, 908643, 395780, 815299, 821059, 81220, 90467, 844484, 81220, 743139, 495812, 366178, 712580, 542658, 495811, 0, 815300, 853700, 740323, 100260, 0, 712580, 288386, 910371, 72547, 712578, 495811, 739524, 72578, 764739,
    858692, 800836, 0, 427299, 662498, 247298, 911907, 366178, 0, 81411, 884739, 100259, 139586, 785092, 835779, 895011, 107938, 695810, 864452, 910466, 405892, 869316, 740323, 0, 864580, 713061, 0, 81221, 918114, 0, 0, 704291,
    0, 858052, 785092, 0, 869443, 0, 0, 869828, 892227, 81219, 0, 721604, 900771, 0, 767300, 910371, 0, 898659, 671330, 725572, 545218, 857922, 0, 0, 454178, 713059, 545218, 905667, 858180, 906339, 895011, 775620,
    712580, 712578, 869443, 0, 839748, 0, 0, 0, 910563, 895011, 918114, 883011, 870980, 358978, 912771, 713059, 912579, 72578, 486018, 0, 486020, 0, 0, 0, 329282, 712580, 100259, 690852, 317762, 750530, 892227, 81283,
    395811, 869316, 139586, 910371, 721604, 870724, 903075, 864452, 901635, 0, 0, 911715, 819396, 0, 0, 901635, 663141, 69667, 100259, 769219, 405861, 858436, 486019, 857795, 898179, 673, 542658, 400739, 869956, 72578, 891939, 610980,
    0, 461380, 743139, 888003, 0, 723523, 908451, 910466, 486018, 712578, 662661, 72578, 90468, 712580, 891939, 409411, 645730, 663140, 743108, 713060, 0, 712900, 860228, 486018, 676452, 870467, 723523, 0, 0, 0, 795332, 725570,
    869572, 905859, 883299, 0, 870467, 645730, 395780, 725570, 288386, 743779, 398211, 486181, 81189, 486021, 837699, 461378, 764612, 0, 0, 317762, 882819, 712578, 420098, 712580, 535300, 713059, 317762, 81218, 882819, 545218, 789060, 545218,
    662500, 905763, 713060, 0, 542658, 882435, 662498, 0, 0, 695810, 800836, 857922, 542660, 545218, 276643, 775620, 713059, 905763, 495811, 857796, 0, 0, 871108, 247298, 495811, 725700, 486020, 778435, 0, 0, 912195, 895011,
    542658, 897315, 663140, 819396, 486020, 750530, 879651, 0, 542660, 0, 712580, 647491, 906723, 803652, 912579, 0, 892227, 662980, 910851, 0, 910563, 0, 795331, 662498, 869698, 905283, 542658, 911907, 0, 869698, 400739, 486018,
    395812, 910563, 515972, 912386, 72578, 712578, 857922, 739939, 712579, 739939, 461539, 395780, 910947, 869570, 663140, 893187, 712579, 662979, 107938, 3970, 906147, 871234, 911715, 912386, 542660, 712580, 712579, 869698, 662979, 100259, 785092, 712578,
    400739, 859331, 513572, 869699, 858979, 869698, 712578, 279746, 495811, 7330, 911715, 906723, 7330, 804164, 527011, 713061, 0, 0, 0, 0, 910563, 0, 882819, 910755, 81282, 910755, 739395, 81189, 712899, 0, 869698, 645730,
    180482, 663141, 795332, 329283, 712578, 800836, 662498, 94114, 542660, 427298, 817219, 795331, 910371, 755524, 793156, 406594, 912099, 81411, 869698, 576898, 912386, 764739, 643172, 0, 725570, 542658, 545218, 897315, 739939, 713060, 888003, 906723,
    713060, 523652, 398181, 892035, 892227, 486018, 800836, 0, 858820, 906723, 907299, 743748, 837700, 870467, 896739, 905763, 0, 0, 879075, 858180, 712580, 894339, 869316, 918114, 857922, 0, 662979, 870467, 690851, 858180, 288386, 869700,
    898179, 901635, 725572, 0, 663461, 690852, 395811, 907587, 858180, 0, 748484, 247298, 785092, 690851, 0, 719267, 841284, 895011, 884739, 884739, 713221, 0, 545218, 746563, 712578, 461539, 486018, 610978, 869698, 610980, 107938, 87588,
    891171, 870082, 100259, 882819, 712580, 785092, 905539, 918114, 905859, 0, 911715, 906339, 3970, 368578, 725572, 859331, 712899, 869698, 725572, 0, 427299, 795331, 869698, 739908, 405861, 869572, 869570, 107938, 910755, 0, 0, 0,
    317762, 891171, 0, 0, 0, 803652, 910563, 72547, 486020, 690851, 645730, 869955, 869572, 712580, 839748, 0, 0, 0, 0, 0, 819394, 486018, 904707, 81218, 81219, 87588, 911715, 870082, 725570, 819394, 662501, 910563,
    764611, 869698, 690851, 894339, 107172, 0, 495811, 663139, 869699, 486018, 0, 288386, 0, 0, 901635, 87588, 819396, 288386, 869698, 858692, 712578, 486018, 897315, 911715, 898179, 712899, 288387, 870467, 725572, 542658, 0, 0,
    0, 0, 0, 0, 0, 857668, 461380, 712578, 713060, 107172, 903075, 502531, 910563, 911043, 910563, 870467, 837700, 0, 671330, 803652, 398180, 819394, 906723, 746563, 662980, 0, 0, 0, 288386, 910563, 366178, 752196,
    871234, 0, 276643, 663139, 395811, 690851, 888291, 0, 0, 0, 603778, 819394, 910947, 817219, 486020, 882819, 892227, 72578, 887331, 859331, 72547, 663140, 613538, 647492, 815299, 0, 0, 0, 0, 869698, 743108, 0,
    180482, 879075, 894339, 171842, 662979, 725570, 288387, 454178, 645730, 869698, 712580, 610978, 869700, 358627, 409380, 764740, 869570, 853700, 910466, 799300, 739396, 912483, 912771, 725572, 911043, 81220, 905667, 910755, 0, 894339, 857796, 869316,
    785092, 891171, 0, 0, 100259, 910851, 358627, 398212, 869700, 712578, 0, 0, 542660, 712580, 81220, 869699, 0, 542660, 662980, 739396, 879651, 486020, 276643, 764611, 750532, 0, 358978, 107938, 0, 0, 835780, 894339,
    869698, 883299, 0, 395780, 740292, 767299, 495811, 837699, 0, 0, 0, 799300, 576898, 911715, 0, 895587, 395811, 0, 107172, 495811, 400739, 858051, 843972, 0, 0, 405892, 864452, 0, 406594, 461379, 725700, 366178,
    662661, 0, 276643, 81380, 81411, 740292, 918114, 81411, 673, 910371, 764739, 690851, 0, 806467, 72578, 107172, 81411, 0, 107938, 871108, 100259, 906339, 870340, 486181, 908643, 0, 0, 0, 0, 0, 0, 329283,
    800835, 486181, 712899, 869444, 81218, 0, 869570, 0, 0, 0, 882819, 857796, 576898, 0, 912195, 723524, 400739, 0, 0, 100259, 454178, 576898, 879651, 789060, 884739, 870340, 0, 910466, 100259, 0, 892227, 712580,
    908643, 395811, 0, 750532, 486018, 891939, 0, 871236, 0, 910563, 893571, 764611, 743779, 712578, 903075, 819394, 719267, 744260, 869699, 819394, 0, 892227, 857922, 700932, 87587, 713059, 542658, 858820, 523652, 712580, 713059, 870467,
    864580, 276643, 879651, 7330, 853572, 395780, 81218, 515971, 892035, 81282, 409380, 662500, 879651, 358978, 764739, 171842, 910179, 910179, 912771, 800836, 0, 461539, 100259, 859331, 743139, 895011, 662979, 671330, 0, 879651, 858051, 427298,
    366178, 542661, 662500, 887331, 486018, 871108, 81411, 72578, 542658, 869828, 911907, 910755, 395780, 750530, 894339, 495811, 405861, 671330, 247298, 911043, 486021, 81220, 81220, 908739, 858434, 90468, 869700, 906915, 869699, 859331, 869699, 892227,
    704291, 869572, 486019, 713221, 906243, 358627, 750660, 0, 911715, 0, 0, 869698, 859331, 869698, 576898, 0, 0, 906723, 869316, 910466, 769220, 869700, 697572, 859331, 610978, 769220, 317762, 94114, 139586, 395780, 853700, 870980,
    81218, 901923, 100259, 890691, 107938, 743139, 0, 671330, 400739, 81218, 100260, 90468, 712580, 647491, 858180, 896739, 893859, 662980, 869316, 870082, 739939, 764611, 395811, 395780, 712900, 0, 317762, 358627, 843972, 395811, 107938, 870980,
    911715, 542658, 743779, 712578, 712580, 395780, 542660, 0, 671330, 892035, 247298, 906339, 775620, 912386, 107938, 0, 0, 858052, 903651, 898659, 358978, 892035, 0, 869570, 647492, 725572, 898179, 0, 0, 0, 0, 869698,
    107938, 0, 647492, 857922, 879075, 81411, 0, 0, 0, 368578, 815300, 418, 329282, 910466, 542658, 800836, 0, 486018, 502531, 912483, 886563, 892227, 883299, 879651, 0, 0, 0, 0, 0, 418, 727235, 288387,
    329282, 910851, 0, 0, 0, 0, 0, 886563, 0, 0, 288387, 725570, 329282, 279746, 739395, 329282, 819396, 495811, 545218, 0, 0, 712900, 725570, 486018, 870083, 869572, 542658, 0, 0, 288387, 107172, 276643,
    0, 0, 673, 0, 0, 0, 0, 662661, 897315, 713060, 704291, 461378, 107938, 894339, 366178, 902403, 853572, 542660, 690852, 542660, 90467, 746564, 542658, 697572, 610, 870596, 767300, 869444, 81283, 764611, 643172, 366178,
    910179, 276643, 869698, 366178, 906243, 905539, 775620, 700932, 395811, 398181, 891939, 81220, 869444, 750532, 739908, 755012, 645730, 858436, 895011, 663141, 743170, 750530, 107938, 81189, 858180, 513572, 690852, 744132, 853700, 81190, 545218, 700932,
    713061, 542658, 276866, 640771, 671330, 0, 746564, 81218, 869698, 712580, 910563, 897315, 895587, 0, 690851, 767299, 0, 690851, 0, 0, 0, 911715, 835779, 0, 767300, 502531, 400739, 890691, 87587, 542658, 895587, 895011,
    712579, 879651, 81282, 107938, 139586, 907203, 910947, 912386, 883011, 739395, 870084, 107172, 918114, 858436, 0, 712578, 276643, 785092, 785092, 869570, 712901, 0, 0, 859972, 398180, 764611, 486018, 870467, 0, 461378, 910466, 0,
    906243, 276643, 739939, 910179, 427298, 912386, 719267, 579298, 918114, 800836, 486018, 910755, 869572, 901635, 3970, 891939, 542658, 417698, 858434, 869828, 545218, 663140, 542658, 542661, 769220, 869443, 910563, 857795, 906243, 72547, 911715, 368578,
    535301, 906723, 406594, 857795, 662501, 81187, 905859, 893283, 869698, 776132, 663141, 853572, 0, 870596, 81221, 81411, 81218, 879651, 0, 0, 712899, 486018, 461378, 712578, 888291, 739908, 569698, 358627, 329283, 870724, 712580, 795332,
    0, 0, 0, 870467, 406594, 712580, 891171, 712578, 0, 329283, 739395, 907587, 712580, 891939, 764739, 906819, 0, 911043, 883587, 486019, 894339, 100259, 545218, 329283, 869700, 645730, 897315, 542658, 880419, 0, 898179, 878787,
    81221, 0, 911043, 72578, 409380, 72578, 841284, 743108, 542658, 690852, 81220, 725570, 358978, 72547, 100260, 417698, 725762, 910851, 0, 912386, 417698, 719267, 721604, 883011, 87587, 317762, 879075, 767299, 100259, 712578, 358627, 3970,
    764611, 911715, 81282, 329283, 817220, 663139, 495811, 461378, 486018, 857668, 785092, 100259, 712578, 513571, 819396, 869572, 662980, 94114, 662500, 81220, 858051, 713060, 864452, 892227, 910563, 662498, 81283, 898659, 461378, 0, 908739, 764611,
    139586, 395780, 860100, 725570, 898179, 860228, 461378, 405892, 81219, 897315, 395812, 486019, 857924, 888003, 81221, 368578, 461378, 764740, 579298, 406594, 869700, 860100, 542661, 891939, 400739, 891939, 743108, 647492, 398180, 100259, 427298, 750530,
    0, 542661, 712580, 725700, 869444, 817220, 7330, 819394, 662498, 662498, 94114, 910563, 107938, 662499, 398181, 911043, 869700, 100260, 486018, 871108, 869572, 821059, 896739, 892611, 911043, 100259, 679811, 461539, 81221, 869700, 663141, 69667,
    542660, 0, 869572, 81189, 72547, 610978, 857922, 819524, 743779, 107938, 712900, 81218, 739524, 725700, 662500, 712578, 486021, 893187, 897891, 815300, 712900, 725572, 486018, 892227, 835780, 725570, 859331, 860228, 576898, 895011, 899331, 576898,
    712578, 395780, 857922, 785092, 743779, 712580, 869570, 903651, 700932, 743139, 107938, 0, 405861, 366178, 871236, 750660, 870980, 486018, 662499, 0, 869444, 81411, 859331, 180482, 0, 427299, 911043, 107938, 721604, 891939, 713059, 486019,
    859331, 750530, 755012, 882626, 918114, 821059, 713060, 713060, 0, 100259, 857668, 395812, 775620, 857668, 896931, 776132, 418, 907299, 764610, 883011, 100259, 775620, 891939, 776132, 891171, 908451, 94114, 713061, 903075, 750660, 893571, 317762,
    882819, 3970, 778435, 0, 0, 750530, 0, 819396, 891939, 821059, 139586, 107938, 713060, 870340, 712899, 869444, 0, 721603, 72578, 890691, 405892, 893859, 910466, 366178, 515972, 545218, 744258, 542660, 869698, 857668, 276643, 0,
    869572, 495811, 857795, 81411, 288386, 81190, 757827, 803652, 0, 901923, 690851, 746563, 910371, 767299, 417698, 542658, 739394, 712580, 640771, 418, 358627, 839748, 406532, 869700, 712580, 795331, 576898, 358978, 0, 461539, 725570, 918114,
    743779, 869698, 895587, 0, 869699, 789060, 743779, 398181, 886083, 673, 0, 100259, 870724, 712899, 662498, 107938, 857924, 743779, 903075, 81221, 139586, 663139, 0, 0, 643172, 821060, 882435, 896739, 542658, 81190, 0, 882819,
    662980, 746563, 662498, 0, 893571, 800836, 799300, 858052, 107938, 869698, 750660, 662498, 869698, 610980, 918114, 486020, 3970, 764612, 907299, 712580, 0, 0, 725570, 859331, 244418, 610980, 870467, 908739, 864452, 864452, 858434, 869698,
    895011, 81282, 72578, 662661, 542658, 461539, 912386, 288386, 869828, 3970, 69667, 87587, 789060, 697572, 911715, 81411, 486020, 907299, 815300, 139586, 912386, 139586, 81282, 288387, 893571, 409411, 857924, 395781, 886563, 523652, 606178, 739395,
    695810, 869570, 892035, 81221, 662498, 891939, 898179, 100259, 0, 870467, 329283, 0, 406594, 640772, 739396, 690852, 3138, 409411, 662498, 81187, 908451, 870980, 904707, 329282, 663140, 896739, 725700, 542658, 815299, 869443, 739523, 81221,
    898659, 888291, 542660, 883587, 712900, 912386, 0, 0, 870467, 358978, 882819, 906339, 0, 803652, 712580, 887331, 891171, 398181, 892035, 673, 405861, 911907, 405892, 911043, 3970, 911043, 700932, 712899, 911715, 910851, 911043, 171842,
    81220, 795332, 911907, 398180, 0, 0, 663140, 0, 841284, 821060, 405892, 81190, 110051, 869572, 495811, 870980, 107938, 690851, 513572, 662661, 0, 398181, 892035, 910851, 713061, 405861, 640771, 857668, 0, 857922, 795331, 869698,
    858180, 671330, 662500, 671330, 0, 764610, 0, 817220, 81219, 0, 0, 739939, 857668, 81187, 906723, 427299, 542658, 358978, 891939, 869570, 897315, 0, 110051, 713221, 81218, 869699, 0, 870467, 662500, 0, 767299, 486020,
    358627, 358627, 180482, 903075, 907299, 870082, 81283, 870467, 775620, 869698, 400739, 486018, 358627, 893859, 858051, 858180, 713061, 603778, 739522, 0, 0, 869698, 892227, 880419, 398181, 502531, 886083, 910371, 869698, 713059, 100259, 0,
    666371, 806467, 647492, 870340, 395811, 107938, 883299, 870340, 887331, 662979, 800836, 317762, 857922, 815300, 7330, 879075, 461539, 461379, 835779, 454178, 0, 859331, 171842, 908451, 882626, 870596, 908643, 793156, 857795, 869570, 427298, 739522,
    81220, 744132, 878275, 713061, 662500, 495811, 871108, 910371, 0, 0, 712578, 603778, 247298, 908355, 139586, 713060, 858820, 908739, 721603, 317762, 662500, 90468, 858180, 897315, 81218, 908355, 713221, 247298, 899331, 495811, 858820, 819394,
    276643, 800836, 894339, 486021, 797252, 870467, 750660, 0, 666371, 461378, 764611, 898659, 288387, 486020, 690852, 910563, 180482, 910466, 764611, 897315, 739395, 739524, 800836, 869698, 744132, 110051, 817220, 107938, 870467, 869698, 746563, 743779,
    910371, 690851, 279746, 910563, 908355, 676452, 662500, 895587, 869698, 0, 894339, 542658, 887331, 895011, 800835, 712580, 869316, 743170, 819394, 712580, 719268, 358629, 276646, 100259, 725572, 800836, 405892, 662498, 799300, 721603, 725570, 81218,
    81218, 0, 739522, 905859, 662661, 821060, 892035, 368578, 898659, 666371, 910371, 0, 0, 542658, 857924, 0, 409411, 712581, 0, 859331, 869700, 0, 835779, 81221, 892035, 100259, 0, 276866, 868932, 910563, 713059, 817219,
    712578, 815299, 0, 81221, 690851, 869316, 7330, 725570, 897315, 764611, 542658, 882819, 81188, 883587, 750530, 535301, 81187, 712580, 395780, 819394, 662500, 906339, 713060, 0, 0, 0, 81218, 767299, 400739, 776132, 579298, 366178,
    81219, 769220, 911043, 898179, 87588, 0, 0, 0, 858434, 910851, 713061, 821059, 486021, 0, 712900, 912771, 911907, 895011, 0, 870596, 725570, 858051, 107938, 0, 743748, 887234, 869698, 819394, 857796, 0, 87588, 486018,
    368578, 0, 880419, 764611, 911907, 542658, 764611, 857922, 918114, 887234, 673, 713061, 864452, 107938, 495811, 662498, 139586, 893571, 72547, 879651, 903651, 869698, 869570, 900771, 910947, 918114, 0, 841283, 697572, 723524, 712899, 910371,
    171842, 911043, 743170, 869316, 769220, 690852, 461539, 81411, 317762, 0, 911043, 417698, 662498, 495811, 896931, 819394, 819394, 486020, 545218, 764611, 743170, 819394, 891939, 918114, 880419, 81221, 288386, 911043, 903075, 495811, 542660, 366178,
    486020, 695810, 800836, 871236, 859331, 869699, 883587, 743108, 739394, 899331, 100259, 795331, 405861, 785092, 276646, 857924, 903075, 897315, 0, 859331, 897315, 358627, 0, 743108, 789060, 486181, 887234, 869698, 907299, 882339, 858051, 918114,
    795331, 288387, 107938, 911715, 750532, 800836, 358627, 317762, 647492, 870212, 785092, 81380, 368578, 893187, 882915, 0, 395811, 395811, 900771, 81218, 775620, 750530, 918114, 869572, 819394, 898659, 795332, 461378, 870340, 317762, 107938, 871108,
    542658, 545218, 910466, 894339, 486018, 542658, 542658, 0, 244418, 100260, 610, 910179, 643172, 0, 870340, 461539, 0, 869570, 901635, 906723, 910371, 713061, 755012, 910371, 366178, 0, 81220, 869572, 869955, 542658, 100259, 81221,
    910851, 81218, 879075, 719267, 869316, 244418, 799300, 870596, 869698, 869570, 897315, 892707, 81187, 0, 603778, 395811, 740292, 545218, 712900, 317762, 918114, 405892, 870084, 857668, 288386, 647491, 461380, 0, 409411, 81187, 0, 887234,
    887234, 858436, 662498, 671330, 139586, 0, 100259, 721604, 7330, 912386, 606178, 81190, 358978, 882435, 695810, 395811, 912771, 358978, 713059, 610, 769220, 725570, 887234, 358627, 910466, 90467, 870467, 870596, 545218, 90467, 778435, 906339,
    418, 912099, 908355, 819396, 750530, 662498, 882435, 853700, 542660, 859331, 803652, 288386, 662498, 869828, 764740, 869570, 892227, 750530, 853572, 461380, 903651, 858051, 713060, 888003, 645730, 690851, 740292, 409380, 72578, 358629, 576898, 880419,
    662498, 859331, 725570, 892227, 671330, 0, 461379, 603778, 81411, 712578, 887234, 910755, 900771, 0, 712578, 870467, 907395, 139586, 884739, 713059, 858820, 817220, 725570, 395811, 107938, 712580, 640772, 837700, 739395, 368578, 0, 662498,
    486018, 870980, 545218, 502531, 712580, 712578, 882339, 90467, 870467, 912386, 527011, 744132, 864452, 81187, 750530, 409380, 906339, 713059, 712899, 907203, 727235, 486018, 712578, 358627, 892227, 454178, 486018, 417698, 398180, 908451, 907491, 712581,
    81221, 662498, 358627, 679811, 739908, 90468, 171842, 892707, 461378, 888003, 764611, 744132, 882435, 870467, 395780, 894339, 910755, 870212, 713059, 107938, 461539, 775620, 869955, 746563, 743779, 908355, 690851, 461378, 712578, 815299, 886563, 725762,
    645730, 910466, 406594, 817220, 645730, 817220, 100260, 870467, 366178, 869700, 910371, 719268, 795331, 100259, 395780, 857924, 713060, 247298, 0, 908643, 887331, 276643, 0, 746564, 883011, 690851, 864452, 400739, 317762, 486019, 860100, 81221,
    799300, 0, 0, 398180, 495811, 673, 890691, 918114, 886563, 0, 673, 723524, 908739, 750530, 0, 750530, 368578, 87587, 870340, 806467, 662498, 579298, 662500, 0, 871108, 908355, 0, 911715, 870596, 317762, 0, 288386,
    368578, 712578, 486021, 882435, 918114, 693252, 72547, 918114, 887331, 910466, 712578, 100260, 821060, 640772, 815300, 668772, 817220, 882435, 864580, 892611, 878787, 869316, 869443, 662498, 0, 0, 858820, 81220, 903651, 869316, 398180, 746563,
    764740, 869699, 81283, 719267, 907106, 815299, 764611, 857922, 859331, 72578, 725572, 883299, 576898, 748484, 0, 0, 535301, 0, 69667, 886563, 911715, 740323, 0, 461379, 910851, 894339, 495811, 662498, 857922, 87588, 869698, 799300,
    0, 0, 100259, 869698, 869699, 662498, 713059, 712578, 869698, 7330, 910466, 870467, 870467, 395811, 743748, 918114, 409380, 647492, 793156, 743108, 0, 910179, 662661, 744132, 888291, 535299, 857922, 857922, 870467, 817220, 542658, 870596,
    317762, 870467, 461378, 542661, 81218, 418, 750532, 882626, 869700, 0, 0, 662498, 0, 69667, 898179, 427298, 869699, 739908, 907299, 398211, 858436, 918114, 893571, 869444, 486020, 869698, 898659, 0, 815300, 486018, 908451, 395811,
    0, 542658, 912386, 871234, 911043, 454178, 542658, 662498, 775620, 542658, 727236, 908451, 0, 0, 288386, 743108, 486020, 910563, 879651, 869570, 495812, 247298, 800836, 712581, 912771, 750660, 409380, 857796, 835779, 870212, 545218, 869698,
    858820, 107938, 81219, 542658, 235778, 0, 0, 542658, 0, 0, 276646, 0, 869698, 841284, 671330, 180482, 918114, 869698, 748484, 719267, 739522, 0, 695810, 857795, 725700, 288387, 910371, 757827, 279746, 406594, 662498, 912675,
    535300, 887331, 0, 869699, 0, 907587, 886083, 81411, 910563, 870082, 690851, 690852, 870467, 869698, 918114, 673, 0, 502531, 486019, 69667, 712580, 815299, 0, 725570, 800836, 891939, 180483, 486019, 0, 398181, 795332, 869444,
    887331, 545218, 712899, 72578, 486018, 610978, 869700, 869316, 693252, 893859, 870082, 180483, 0, 662500, 139586, 853700, 764611, 905539, 800835, 819394, 870980, 910563, 81218, 819394, 891939, 869954, 744132, 542658, 719267, 358627, 869699, 395811,
    0, 869698, 804164, 911907, 719267, 800836, 610981, 107938, 904899, 0, 869698, 904899, 869698, 662500, 712579, 81187, 869699, 819394, 712578, 603778, 513571, 871234, 712579, 100259, 329282, 695810, 910466, 908451, 869699, 405861, 870596, 815300,
    869828, 613538, 819524, 912195, 87587, 542658, 857922, 603778, 859331, 725570, 912386, 721604, 0, 789060, 712581, 180483, 905667, 535301, 910947, 712899, 911043, 527011, 81411, 712578, 673, 100260, 864580, 911715, 892227, 542658, 0, 542658,
    398181, 486018, 81218, 712900, 918114, 395781, 461380, 139586, 817220, 395781, 690851, 900771, 869444, 542658, 100259, 908355, 859331, 869698, 727236, 744132, 743139, 87587, 750530, 486018, 454178, 395811, 366178, 910755, 895587, 739395, 739395, 869698,
    869572, 100259, 912195, 395811, 107172, 871108, 0, 513572, 107938, 662498, 713059, 569698, 0, 0, 276646, 495811, 461380, 247298, 800835, 0, 697572, 486021, 418, 869698, 107172, 882626, 100259, 891939, 910947, 910371, 712580, 841284,
    918114, 454178, 0, 576898, 358627, 901635, 712580, 90467, 542658, 739395, 899331, 486020, 610978, 739908, 87587, 869570, 613538, 897891, 912579, 513571, 168962, 395812, 0, 0, 107938, 910371, 358627, 0, 0, 869698, 0, 750660,
    918114, 725570, 358627, 912386, 276643, 712579, 610978, 0, 785092, 910947, 405861, 904707, 858180, 870467, 666372, 0, 662980, 107938, 0, 495811, 0, 545218, 395811, 880419, 869698, 912579, 712900, 107938, 87587, 610978, 461380, 892227,
    723524, 739395, 0, 886563, 662498, 911043, 180483, 858436, 743139, 910947, 910563, 0, 180483, 0, 910563, 366178, 171842, 882819, 869570, 878787, 743170, 871108, 0, 695810, 81187, 400739, 839748, 461380, 764740, 910563, 748484, 910371,
    892227, 180483, 870084, 276866, 886563, 800836, 817220, 0, 891171, 817220, 803652, 0, 869700, 0, 461380, 744258, 542658, 427298, 0, 725700, 0, 712578, 0, 420098, 420098, 712580, 542660, 0, 858820, 910563, 859331, 910371,
    0, 0, 72578, 911715, 545218, 893571, 869700, 663141, 789060, 662498, 486020, 110051, 0, 662498, 0, 800835, 486018, 897891, 81190, 610980, 727235, 542658, 800836, 767300, 398181, 869572, 712580, 81218, 725570, 871234, 883299, 883299,
    409411, 878275, 72547, 695810, 0, 905283, 870467, 895587, 887331, 785092, 461380, 712578, 800835, 869572, 858052, 908739, 645730, 869443, 662980, 870467, 610, 905539, 864452, 869698, 910563, 645730, 911043, 911043, 613538, 740323, 841283, 869699,
    764611, 835779, 911043, 87587, 869828, 712579, 870596, 869828, 859331, 819396, 712578, 486018, 908739, 887331, 910851, 610978, 882626, 819394, 725572, 0, 603778, 107938, 673, 870467, 858051, 869570, 896931, 81411, 910947, 901923, 395811, 486019,
    886083, 81219, 329282, 647492, 744258, 712578, 676452, 819394, 869443, 910466, 662661, 712901, 908739, 869444, 892035, 139586, 409411, 695810, 815299, 100259, 486018, 859844, 894339, 87587, 72547, 739395, 891939, 911715, 81411, 712899, 288386, 454178,
    901923, 712578, 0, 427299, 400739, 910371, 713059, 778435, 740292, 400739, 912195, 81411, 804164, 799300, 893859, 910179, 81411, 910466, 725762, 663139, 288387, 897891, 690851, 869443, 495812, 81282, 317762, 395780, 910371, 868932, 486018, 398180,
    72547, 107938, 712580, 869443, 0, 0, 0, 901923, 0, 100259, 906339, 486021, 676452, 502531, 799300, 139586, 0, 815299, 895011, 0, 81187, 400739, 712581, 857795, 898659, 276643, 739908, 400739, 461378, 107938, 869699, 0,
    895587, 893859, 276643, 713059, 276646, 81411, 0, 0, 673, 0, 662979, 871108, 892227, 502531, 868932, 896739, 910371, 405861, 358627, 895011, 0, 713059, 870340, 0, 869698, 81187, 610978, 366178, 864324, 869698, 740323, 3970,
    81220, 815299, 815299, 857922, 893859, 740323, 713221, 821059, 712580, 713060, 276643, 912771, 358627, 890691, 869572, 869698, 870467, 910563, 495811, 739395, 0, 368578, 542661, 870467, 486018, 87587, 461380, 139586, 495812, 406594, 918114, 461539,
    743139, 100259, 858436, 870467, 725700, 869698, 908451, 884739, 461539, 663141, 910947, 398180, 663139, 857796, 0, 0, 409380, 405892, 755012, 398181, 910371, 858434, 276643, 882339, 486018, 869954, 662498, 100259, 918114, 107938, 918114, 244418,
    870467, 690851, 912771, 72547, 857922, 911715, 72578, 743139, 461539, 673, 486021, 110051, 905539, 882915, 329282, 918114, 912771, 81218, 0, 366178, 870340, 892227, 690852, 892227, 918114, 81221, 409380, 107172, 81380, 908451, 400739, 662500,
    882339, 869316, 815299, 739395, 662498, 486020, 725570, 486018, 870467, 358627, 857922, 712900, 486018, 725572, 0, 542658, 542658, 911907, 857924, 398181, 918114, 69667, 690852, 859331, 395812, 906243, 662498, 366178, 603778, 905539, 81187, 744258,
    911907, 870467, 398180, 81187, 725570, 817220, 358978, 864452, 821059, 895011, 764740, 398181, 906339, 864580, 918114, 893283, 486020, 0, 0, 0, 713060, 906339, 912771, 400739, 180483, 895011, 879651, 427298, 910466, 0, 870980, 545218,
    690852, 0, 81187, 542661, 864452, 527011, 395781, 892227, 329282, 81220, 887331, 725570, 0, 712579, 94114, 486018, 857922, 180483, 81218, 906339, 288387, 882339, 0, 906339, 645730, 640772, 610, 911043, 858180, 910947, 744258, 884739,
    400739, 883011, 817220, 869572, 886563, 880419, 713059, 81187, 395780, 461539, 755524, 912483, 748484, 893571, 693252, 911907, 857924, 358627, 800836, 398180, 713060, 918114, 663139, 864580, 535299, 406594, 0, 662661, 276643, 739939, 662980, 910275,
    712900, 803652, 892035, 542658, 0, 409380, 858180, 911907, 892227, 513572, 764611, 662498, 405892, 918114, 545218, 405892, 910179, 821059, 864452, 789060, 662498, 713061, 857796, 775620, 864452, 795332, 898179, 869699, 545218, 869828, 139586, 645730,
    0, 910563, 725572, 409411, 358627, 858180, 666372, 910466, 69667, 515972, 662980, 643172, 815299, 910755, 0, 744132, 515972, 358629, 395780, 671330, 405861, 288387, 879651, 892611, 0, 870467, 910371, 821059, 800836, 871236, 903075, 800836,
    662498, 911907, 0, 869698, 712578, 398180, 0, 858180, 398180, 843972, 911715, 486018, 366178, 764612, 0, 0, 744258, 750530, 785092, 107938, 72578, 869570, 107938, 857922, 0, 878787, 769220, 486181, 0, 3970, 276643, 911715,
    676452, 7330, 739396, 911043, 276643, 100259, 663461, 799300, 744132, 107938, 366178, 288387, 0, 100259, 894339, 800835, 695810, 427298, 911907, 486018, 0, 110051, 0, 0, 139586, 139586, 81187, 398180, 775620, 712581, 712580, 81282,
    461539, 107938, 894339, 0, 329282, 398180, 869444, 906339, 395812, 0, 746564, 882339, 750530, 100259, 912386, 0, 869700, 395780, 0, 542658, 908451, 569698, 569698, 0, 690852, 905859, 896739, 662498, 883011, 329282, 673, 739395,
    891939, 869698, 486018, 746564, 892227, 725570, 894339, 869698, 409380, 409380, 244418, 542658, 912099, 908643, 244418, 744258, 139586, 869570, 764611, 0, 869570, 887234, 743779, 358627, 719267, 764610, 789060, 395811, 542661, 870467, 819394, 107938,
    0, 739523, 898659, 869699, 819396, 329282, 180483, 0, 0, 819394, 858180, 107938, 662501, 819394, 139586, 645730, 495811, 87588, 486018, 817220, 486020, 910851, 879075, 767299, 395780, 542658, 542658, 906339, 757827, 869700, 110051, 398180,
    870467, 719267, 0, 896931, 892611, 461539, 911907, 81411, 645730, 69667, 886563, 395780, 662979, 859331, 663139, 869444, 869700, 859331, 400739, 7330, 395811, 887331, 7330, 100259, 871236, 908355, 495811, 886563, 712580, 461380, 908739, 906243,
    395780, 395811, 817220, 911907, 744258, 662500, 495811, 662980, 579298, 869828, 0, 643172, 713061, 870467, 409380, 107938, 900771, 87587, 486021, 662498, 288386, 906051, 864452, 535300, 744258, 288387, 0, 891939, 486020, 725762, 785092, 7330,
    0, 7330, 803652, 912195, 879075, 542660, 0, 81220, 545218, 910371, 712580, 909603, 366178, 713061, 358627, 879651, 817220, 910563, 743170, 911715, 0, 869316, 542658, 712899, 819394, 0, 7330, 0, 0, 244418, 910563, 0,
    0, 0, 662980, 870980, 0, 662980, 892227, 486018, 891939, 0, 839748, 725762, 81187, 910466, 461380, 910563, 697572, 910851, 746564, 712900, 690851, 643172, 910563, 869570, 712580, 671330, 535299, 713061, 910466, 743779, 907395, 90467,
    427298, 81220, 893283, 918114, 406501, 712580, 857795, 906243, 795331, 0, 3970, 869700, 906723, 395812, 409380, 870467, 0, 910371, 100259, 712581, 869698, 610978, 713061, 406501, 0, 859331, 752196, 912386, 892227, 869570, 869828, 0,
    0, 869570, 883587, 864452, 454178, 869698, 405892, 869698, 0, 542658, 180482, 712580, 288387, 712578, 869572, 0, 0, 750530, 893571, 870596, 610980, 398181, 486020, 725570, 358629, 800836, 740323, 721604, 690852, 400739, 569698, 712580,
    276645, 107172, 857922, 0, 247298, 81411, 910371, 869698, 7330, 0, 72547, 740323, 727235, 0, 712580, 395780, 409411, 0, 395811, 882626, 870467, 662498, 180483, 907203, 0, 869700, 712741, 0, 81218, 882819, 0, 906723,
    750530, 910466, 906339, 0, 100259, 799300, 725570, 0, 727235, 0, 869828, 904899, 879075, 869700, 405892, 882626, 662499, 695810, 107938, 894339, 879075, 0, 486018, 918114, 712580, 739523, 81218, 279746, 723524, 857796, 107938, 610,
    406594, 395811, 746563, 454178, 750530, 743779, 640772, 0, 870724, 800836, 897891, 897891, 542660, 869316, 910851, 713060, 244418, 918114, 904707, 712578, 427299, 746563, 110051, 712580, 869699, 276643, 427299, 723524, 81218, 887331, 887331, 869954,
    879651, 887331, 712580, 859331, 819396, 712900, 712900, 712900, 81411, 918114, 87587, 576898, 857668, 799300, 0, 899331, 81218, 405892, 107938, 662498, 886563, 81283, 0, 358978, 750530, 869572, 750530, 725570, 870596, 905763, 909603, 869698,
    0, 486019, 870467, 329282, 910563, 746563, 864580, 910563, 395781, 869828, 461380, 542661, 723524, 0, 545218, 486018, 869572, 276643, 427298, 908355, 81218, 110051, 395780, 721604, 858180, 673, 90468, 358978, 764740, 107938, 713059, 764739,
    461380, 368578, 454178, 757827, 81221, 427298, 695810, 366178, 662500, 895011, 803652, 910371, 712900, 795331, 288387, 603778, 725570, 910179, 358627, 884739, 368578, 139586, 244418, 712579, 910371, 244418, 417698, 764611, 395780, 81218, 869954, 0,
    0, 0, 0, 0, 764611, 910563, 405892, 400739, 879075, 869699, 0, 495812, 366178, 858436, 743108, 869700, 869698, 888003, 527011, 110051, 713060, 3970, 139586, 743139, 821059, 878787, 454178, 725572, 870980, 740323, 713060, 663141,
    769219, 725570, 886563, 409380, 81188, 666371, 398181, 719267, 398180, 898179, 905859, 910371, 645730, 918114, 870980, 662498, 800835, 869698, 817219, 910466, 662498, 884739, 398180, 712579, 72578, 666372, 461379, 743139, 725762, 918114, 857668, 409411,
    645730, 100259, 671330, 712578, 821060, 180483, 454178, 0, 725762, 666371, 276646, 896739, 713060, 81411, 395780, 0, 800835, 603778, 395811, 427298, 495811, 819396, 764611, 673, 740292, 405892, 405892, 107938, 910851, 905763, 725570, 740292,
    911715, 869698, 910851, 276866, 869570, 542658, 542658, 869828, 427299, 743779, 662979, 797252, 906147, 81218, 891171, 368578, 907299, 910371, 908355, 81282, 107938, 712581, 398212, 100260, 869316, 911043, 0, 725570, 0, 398212, 908739, 895587,
    0, 358627, 0, 712578, 869955, 869698, 886083, 869828, 892035, 0, 366178, 695810, 869700, 0, 869698, 610980, 858820, 723523, 890691, 721603, 576898, 888003, 3138, 663139, 0, 870467, 100260, 486020, 869828, 859972, 695810, 910371,
    910947, 870467, 869698, 858051, 859331, 81218, 606178, 7330, 486020, 288387, 869700, 879075, 81218, 81218, 7330, 0, 100260, 542660, 712578, 3138, 427299, 869955, 358627, 869700, 869700, 495811, 81190, 746563, 0, 793156, 869698, 7330,
    486020, 3138, 841284, 882626, 662500, 910851, 366178, 819394, 859972, 576898, 542661, 662498, 662498, 859844, 918114, 366178, 723524, 880419, 869698, 910371, 81220, 764611, 662498, 898179, 879075, 81283, 81283, 918114, 662498, 906819, 910179, 869698,
    366178, 853572, 358978, 0, 869572, 0, 0, 107938, 400739, 545218, 869698, 542658, 0, 839748, 712578, 910466, 0, 894339, 247298, 870980, 912579, 870980, 395781, 81282, 887234, 107938, 797252, 918114, 712899, 409411, 409380, 908355,
    725570, 725570, 712899, 0, 72578, 542658, 0, 395811, 912099, 461378, 398181, 896739, 576898, 819394, 897891, 725762, 713059, 358627, 81221, 764611, 748484, 700932, 0, 427298, 94114, 910371, 486018, 400739, 800835, 725700, 803652, 486020,
    903075, 897315, 663140, 0, 461379, 903075, 869698, 180482, 87587, 610, 662499, 366178, 892707, 800836, 767299, 461379, 739939, 535301, 398212, 180483, 870082, 697572, 795332, 725570, 81220, 244418, 911043, 87587, 695810, 288387, 535299, 427299,
    725762, 486020, 576898, 180483, 793156, 81282, 569698, 857796, 857668, 0, 892707, 740323, 886563, 712580, 901635, 908451, 891939, 645730, 235778, 676452, 395781, 903075, 815299, 676452, 882626, 712580, 893283, 0, 750530, 906147, 864580, 871234,
    94114, 887331, 882339, 643172, 878275, 0, 907203, 662498, 883011, 640771, 366178, 815299, 871108, 882626, 180483, 486018, 795331, 0, 817220, 819396, 640771, 545218, 871236, 0, 895587, 869698, 90467, 542658, 542658, 870084, 358627, 409411,
    663140, 400739, 869444, 87588, 755012, 743779, 645730, 81218, 898659, 72547, 898659, 740292, 139586, 904707, 0, 869698, 858434, 800836, 3970, 0, 662498, 912386, 0, 721604, 775620, 870082, 891939, 107938, 857795, 879651, 358978, 910179,
    100260, 857668, 911715, 870467, 891939, 576898, 663141, 662980, 662498, 662980, 819394, 775620, 869698, 769220, 769220, 905539, 918114, 910851, 7330, 697572, 864580, 900771, 895587, 739395, 542658, 276643, 545218, 743779, 883011, 905667, 895587, 139586,
    712900, 139586, 713061, 871234, 712580, 870467, 0, 0, 418, 0, 0, 418, 869570, 769220, 673, 579298, 713221, 695810, 869828, 712900, 769220, 90468, 795332, 107938, 908355, 712899, 739939, 81219, 712900, 0, 896739, 395780,
    870467, 910371, 869572, 912195, 821059, 461539, 288386, 0, 0, 671330, 886563, 739523, 535300, 693252, 902403, 893859, 545218, 869698, 910563, 100259, 857796, 247298, 743108, 712580, 461380, 891939, 893571, 817220, 7330, 0, 542658, 869699,
    409380, 906339, 693252, 107938, 0, 0, 883299, 427298, 405892, 0, 712578, 0, 662498, 0, 869828, 486018, 110051, 764612, 775620, 395812, 713060, 906723, 0, 0, 891939, 645730, 767299, 869572, 663141, 859331, 857668, 882626,
    884739, 366178, 750660, 87587, 882626, 870467, 878787, 910466, 879075, 746563, 358978, 700932, 0, 775620, 892035, 461539, 882626, 911715, 869828, 893859, 139586, 918114, 903075, 673, 673, 869444, 513571, 100259, 329282, 906147, 723524, 821060,
    859972, 100259, 662498, 869828, 880419, 0, 869572, 743139, 81221, 911043, 789060, 180483, 673, 486018, 576898, 405861, 879075, 910466, 713060, 869570, 317762, 400739, 712578, 744132, 569698, 893187, 100259, 739395, 0, 72578, 748484, 719267,
    700932, 107938, 712581, 901923, 860100, 764739, 247298, 800836, 725762, 81220, 869698, 81220, 817220, 869700, 894339, 100259, 879075, 884739, 800836, 859331, 785092, 895011, 495812, 139586, 695810, 645730, 767300, 869316, 910275, 406501, 750530, 81220,
    911043, 542658, 819394, 81218, 663141, 486019, 869699, 910371, 0, 853700, 545218, 800836, 869698, 882339, 662980, 81218, 276643, 910179, 576898, 81219, 909603, 727236, 643172, 409411, 918114, 891171, 368578, 542658, 395780, 645730, 883011, 817220,
    908451, 662501, 870467, 535301, 725570, 725762, 817220, 712580, 180482, 905539, 868932, 910371, 712580, 0, 868932, 893859, 395780, 610978, 610978, 542658, 755012, 139586, 662500, 366178, 858692, 712899, 908739, 662500, 843972, 869698, 495812, 767299,
    750530, 486020, 898659, 417698, 486020, 107938, 910563, 912099, 815300, 542658, 778435, 910947, 317762, 713059, 100260, 868932, 662981, 712578, 857668, 910371, 869700, 81188, 857922, 610, 0, 870467, 697572, 461378, 662981, 739394, 918114, 909603,
    317762, 317762, 870980, 395811, 647492, 72578, 870724, 870596, 366178, 895587, 613538, 712578, 905763, 750530, 72578, 695810, 910275, 695810, 662498, 486018, 400739, 358978, 495811, 171842, 542658, 139586, 857922, 418, 461378, 662499, 662498, 712580,
    486020, 859331, 486020, 406594, 81219, 712899, 870082, 911043, 712900, 486018, 69667, 100259, 486020, 0, 100259, 0, 785092, 700932, 835779, 911907, 0, 400739, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    869698, 752196, 892707, 662498, 662981, 712579, 395811, 87587, 712901, 666371, 662981, 712579, 486018, 368578, 107938, 819396, 888003, 743108, 743108, 358629, 906243, 645730, 859331, 859331, 906339, 727236, 7330, 744132, 755012, 671330, 398181, 885315,
    906339, 542661, 885315, 911715, 542661, 100259, 727236, 666372, 778435, 892035, 454178, 3970, 662501, 713221, 486018, 891939, 764740, 461380, 81189, 395780, 800835, 870980, 358627, 358978, 87587, 746563, 910371, 358627, 495812, 542658, 139586, 906051,
    918114, 869698, 869698, 901635, 910466, 892035, 892035, 910179, 572098, 869698, 869698, 870467, 910466, 712899, 81218, 764610, 537858, 869699, 647492, 640772, 857668, 81218, 776132, 398181, 398181, 869698, 869699, 3970, 899331, 892035, 662498, 7330,
    712578, 712581, 888003, 892035, 712579, 81218, 276643, 886083, 859331, 486018, 835780, 486019, 139586, 859331, 898179, 81218, 897891, 804164, 912386, 858434, 910851, 100259, 918114, 662979, 603778, 767299, 908451, 107938, 882819, 713059, 767299, 870467,
    461539, 767299, 804164, 800835, 869698, 892227, 537858, 569698, 107938, 713061, 859331, 859331, 887234, 81187, 107938, 180483, 107938, 110051, 81218, 662498, 910466, 495812, 671330, 288387, 769219, 769219, 3970, 910563, 3970, 662498, 3970, 486018,
    905539, 87588, 513572, 417698, 785092, 739939, 869698, 395780, 398212, 579298, 7330, 712581, 495812, 697572, 542661, 712581, 495812, 7330, 857795, 366178, 911715, 910563, 746563, 542660, 886563, 817219, 858051, 858051, 486018, 486018, 87588, 869698,
    663140, 858051, 537858, 752196, 870082, 495811, 247298, 743108, 358978, 712901, 495811, 891939, 727236, 858052, 744132, 887714, 666371, 806467, 727236, 739396, 912386, 859331, 317762, 712901, 918114, 712580, 712580, 910179, 405892, 700932, 858051, 662500,
    767299, 454178, 366178, 883299, 800836, 869698, 740292, 395781, 276866, 395781, 869698, 579298, 817219, 858051, 764611, 740292, 100260, 662500, 713221, 645730, 94114, 910466, 486020, 486019, 911043, 764740, 739396, 858051, 87588, 764739, 750530, 764612,
    713221, 662498, 905859, 107938, 750660, 757827, 910466, 869698, 869443, 513572, 869698, 486018, 406594, 910466, 486018, 918114, 764610, 910755, 495811, 645730, 746563, 918114, 690852, 713221, 911715, 743779, 81411, 107938, 882819, 110051, 666371, 910371,
    81189, 72547, 3970, 712580, 662981, 395781, 806467, 739396, 910947, 869443, 910371, 358627, 870082, 712580, 613538, 366178, 461378, 610978, 610978, 910563, 721603, 869570, 800836, 870596, 663140, 857922, 882626, 723524, 859331, 486019, 645730, 869698,
    671330, 869698, 713061, 750532, 395811, 897891, 746563, 358627, 368578, 857922, 725570, 870340, 882626, 910371, 750530, 358627, 912771, 911715, 100260, 695810, 712900, 72578, 725570, 910179, 663139, 107938, 869698, 918114, 912771, 890691, 841284, 700932,
    610978, 712580, 857922, 107172, 610, 662500, 859331, 168962, 879075, 905859, 81411, 72578, 893859, 712581, 454178, 100260, 878787, 764610, 821060, 486019, 887234, 727236, 486018, 398211, 743779, 461539, 857924, 864580, 785092, 882626, 869698, 859331,
    869444, 663140, 910179, 513571, 870467, 888003, 893571, 869698, 400739, 535299, 461381, 542658, 869699, 859331, 819394, 358627, 869570, 911715, 81282, 545218, 900771, 887234, 806467, 755012, 69667, 400739, 910947, 775620, 870596, 869698, 858434, 882915,
    712579, 671330, 883299, 545218, 317762, 785092, 713060, 739522, 857668, 793156, 739908, 841284, 843972, 495811, 910466, 743139, 486181, 398181, 713060, 912771, 743108, 857795, 545218, 139586, 276643, 697572, 663140, 329283, 366178, 835779, 869316, 725572,
    329283, 107938, 712578, 744258, 673, 906339, 869570, 461381, 180482, 910466, 502531, 869698, 918114, 542661, 871234, 857668, 542658, 107172, 409411, 495812, 662500, 666371, 905859, 288386, 918114, 712578, 542658, 918114, 647492, 879075, 107938, 662498,
    712579, 869443, 910755, 107938, 750530, 72547, 743108, 870467, 100259, 542659, 542658, 893571, 671330, 486020, 910755, 918114, 743779, 139586, 869316, 905667, 486020, 725700, 81187, 887331, 764740, 3970, 81282, 879651, 107938, 869443, 725570, 81218,
    395781, 870467, 727236, 892035, 911043, 882819, 542659, 87587, 893571, 739396, 871108, 454178, 461380, 879651, 864452, 276643, 276643, 817219, 883587, 690851, 892227, 81380, 869444, 643172, 690852, 743139, 869698, 662498, 695810, 767299, 495812, 891939,
    288387, 395781, 891939, 764612, 870467, 523652, 789060, 871108, 139586, 662500, 610978, 806467, 870980, 94114, 897315, 676452, 797252, 409411, 755012, 912386, 723523, 910466, 139586, 901635, 882339, 486020, 870212, 7330, 870467, 662979, 857796, 841284,
    869700, 398181, 662500, 100259, 858436, 713221, 409380, 486019, 486018, 910371, 107938, 727236, 907491, 695810, 910371, 853572, 100259, 764740, 690852, 94114, 673, 910563, 72547, 663141, 513571, 750530, 912771, 69667, 697572, 879651, 910563, 81221,
    911043, 712900, 743108, 100259, 911715, 486020, 869570, 712899, 882626, 918114, 725570, 870467, 871108, 819396, 486018, 869698, 910563, 910947, 882339, 755524, 739522, 713060, 743170, 905539, 905539, 879651, 869954, 896739, 891939, 910371, 764739, 395781,
    461539, 908355, 870467, 288386, 712579, 918114, 869828, 858180, 542658, 613538, 110051, 486021, 569698, 721604, 750532, 690852, 740323, 891939, 712578, 662980, 662500, 892227, 739396, 81187, 666371, 879651, 869316, 107172, 911043, 690851, 908451, 100260,
    764610, 727236, 486019, 883299, 912099, 690852, 461378, 727235, 81187, 905283, 739395, 894339, 882819, 107172, 912195, 871108, 723524, 676452, 725762, 81189, 107938, 870212, 871234, 662981, 81190, 495812, 400739, 662979, 395811, 712901, 857796, 100259,
    750530, 910563, 896739, 879651, 853700, 495811, 542660, 545218, 815299, 662500, 712580, 690851, 869700, 755012, 358629, 712581, 662980, 857668, 486181, 329283, 912675, 247298, 712578, 893859, 395781, 461381, 799300, 662498, 398181, 395780, 910371, 486018,
    366178, 81411, 81189, 909603, 746563, 785092, 892035, 668772, 764739, 712580, 139586, 673, 785092, 486020, 700932, 800835, 366178, 461378, 712580, 288386, 712580, 486021, 821059, 7330, 495811, 486019, 857668, 645730, 764740, 870596, 406594, 878275,
    869698, 502531, 843972, 395780, 869698, 746564, 495812, 869572, 793156, 486020, 81187, 746563, 785092, 662979, 912099, 739395, 727235, 81282, 858434, 908643, 743108, 739908, 795331, 610978, 495811, 662500, 817220, 910466, 841284, 662661, 610978, 395781,
    895011, 894339, 545218, 535301, 879075, 107938, 725700, 869955, 870467, 671330, 911715, 740323, 857796, 398180, 542658, 693252, 910371, 803652, 81187, 395780, 719268, 276643, 81411, 542658, 897315, 869955, 905283, 690852, 100259, 894339, 835780, 887234,
    910466, 110051, 869955, 869828, 712741, 405861, 870340, 712741, 712580, 461381, 427299, 171842, 427299, 819396, 329282, 395781, 887234, 427299, 887234, 495811, 329282, 671330, 100259, 100259, 859331, 0, 746564, 662500, 886083, 764739, 898179, 663140,
    886083, 0, 495812, 542658, 81221, 858434, 87588, 902403, 857924, 542661, 107938, 358978, 775620, 895587, 712578, 395811, 795331, 775620, 864452, 461380, 869444, 727236, 662498, 139586, 750532, 912386, 764739, 100260, 486020, 859844, 486018, 869698,
    486018, 395780, 486181, 898179, 719267, 902403, 869828, 898179, 767300, 719267, 739522, 486181, 739522, 912099, 461379, 486181, 898179, 870596, 898179, 890691, 869570, 819396, 819394, 858180, 869699, 910563, 513571, 502531, 843972, 869698, 819394, 859331,
    461380, 817220, 713221, 139586, 869828, 821060, 893187, 800835, 918114, 910563, 819394, 727236, 755012, 908643, 767300, 764740, 727236, 81218, 405861, 892227, 81218, 535301, 7330, 797252, 640771, 918114, 821059, 755012, 712580, 895011, 662499, 486020,
    495812, 908355, 898659, 329282, 662498, 461378, 870852, 676452, 495812, 542661, 857796, 750660, 800836, 755012, 767299, 486020, 869699, 882723, 406594, 542658, 81190, 821060, 870852, 882819, 671330, 886563, 495811, 800835, 800835, 841284, 712578, 427298,
    486018, 764611, 764611, 764611, 764740, 139586, 81187, 713061, 764740, 579298, 870467, 918114, 764740, 647492, 764740, 892611, 81221, 892611, 767299, 748484, 647492, 662500, 712741, 793156, 329282, 725762, 81187, 486018, 746564, 395811, 395811, 912195,
    883587, 750660, 883587, 858051, 858051, 81190, 764739, 906243, 869698, 486018, 513571, 662500, 882915, 81187, 90467, 486018, 746563, 869444, 81219, 87587, 892227, 542658, 911907, 895011, 895587, 870082, 767300, 81190, 918114, 81187, 857922, 750660,
    662980, 821059, 800836, 713221, 640771, 880419, 891939, 891939, 905763, 542658, 662980, 880419, 395780, 495812, 461379, 769220, 883587, 81190, 180483, 180482, 886563, 887234, 870467, 276644, 857924, 662979, 882626, 171842, 891939, 461539, 882626, 461539,
    892611, 886563, 486018, 695810, 495812, 896739, 897315, 870082, 869828, 883011, 910851, 171842, 418, 662500, 662979, 81190, 461379, 486020, 883011, 542658, 695810, 887331, 906147, 395780, 662498, 81190, 912386, 767300, 87587, 366178, 486018, 886563,
    857796, 662980, 427298, 486020, 486018, 366178, 100259, 139586, 743779, 663141, 486020, 886563, 755524, 910466, 461379, 918114, 663141, 486020, 454178, 417698, 892611, 81190, 81189, 395780, 868932, 545218, 542658, 643172, 662980, 853700, 893187, 785092,
    461379, 857668, 892227, 81219, 712899, 870980, 542660, 139586, 645730, 895587, 461539, 785092, 461539, 486020, 486020, 690852, 743779, 486018, 486020, 869698, 486020, 870467, 395780, 486020, 576898, 542658, 358627, 542658, 3970, 857924, 721603, 775620,
    486019, 910466, 897891, 895011, 495811, 515972, 910563, 870467, 755012, 358627, 613538, 891939, 857922, 858820, 869698, 358627, 576898, 542658, 495811, 893859, 893859, 789060, 893187, 100260, 610, 869316, 900771, 535301, 891939, 895011, 72578, 859331,
    171842, 495811, 888291, 366178, 821059, 400739, 795332, 107172, 107172, 819396, 171842, 910947, 789060, 857796, 545218, 368578, 817220, 400739, 894339, 892611, 819396, 755012, 869570, 869570, 513571, 910466, 869698, 910947, 90468, 663141, 893859, 579298,
    545218, 853700, 859331, 486018, 857795, 486021, 740292, 857924, 712581, 869698, 892227, 764611, 789060, 700932, 139586, 746563, 910563, 897891, 910947, 910947, 545218, 542658, 72547, 719268, 857924, 879075, 858820, 110051, 892035, 879075, 890691, 910466,
    110051, 90467, 712741, 888003, 888003, 879075, 744132, 869698, 712741, 643172, 139586, 908739, 454178, 247298, 673, 905763, 918114, 775620, 417698, 892035, 775620, 725570, 81219, 879651, 901923, 859331, 764612, 892035, 869698, 869954, 486021, 513572,
    486021, 535301, 81218, 712579, 857922, 764611, 461539, 906243, 869956, 535301, 887331, 81218, 107938, 461539, 180483, 81411, 746563, 461539, 542660, 542660, 864324, 427298, 892035, 892227, 712578, 645730, 857922, 869698, 171842, 100259, 725572, 610978,
    857922, 859331, 910947, 662500, 610980, 276866, 871234, 663141, 910466, 81411, 721603, 859331, 712578, 87587, 81219, 910563, 87587, 879651, 871234, 870980, 910563, 746563, 897891, 835779, 746563, 486018, 366178, 663461, 366178, 535301, 725572, 368578,
    712900, 168962, 545218, 911715, 545218, 712578, 461379, 94114, 910563, 891939, 891939, 910371, 81283, 662661, 870467, 898179, 869699, 910371, 858820, 486018, 906723, 804164, 400739, 789060, 910563, 869570, 495811, 906723, 910466, 486018, 673, 857796,
    712578, 666371, 395781, 910371, 673, 697572, 542658, 713059, 395811, 542658, 3970, 3970, 100259, 663139, 750530, 712899, 640772, 3970, 906243, 3970, 870467, 739522, 640772, 695810, 542658, 663139, 743139, 697572, 912579, 883011, 887331, 869828,
    739522, 869828, 712581, 100259, 542658, 892227, 662979, 750532, 81187, 892035, 495811, 725762, 461539, 100259, 671330, 542658, 395781, 795331, 542658, 100259, 100259, 869828, 671330, 775620, 800835, 907203, 857922, 857922, 887234, 894339, 427298, 712580,
    358629, 666371, 739395, 739395, 739395, 739395, 406594, 883299, 911715, 69667, 486018, 911715, 276646, 764611, 859844, 776132, 358978, 764610, 907491, 859331, 276643, 870980, 857922, 859844, 725572, 869570, 486019, 640771, 908355, 486018, 910371, 908355,
    884739, 907299, 744132, 893187, 461378, 139586, 800836, 461379, 883587, 893187, 870467, 785092, 870340, 358978, 752196, 171842, 908643, 400739, 859331, 911715, 906339, 244418, 7330, 740323, 671330, 690851, 400739, 537858, 750660, 486018, 799300, 405892,
    899331, 869570, 815300, 400739, 454178, 869572, 911043, 912386, 72578, 486018, 750660, 882915, 700932, 821060, 743139, 542658, 329282, 712581, 712899, 712579, 7330, 545218, 3970, 569698, 743170, 743779, 869698, 502531, 743108, 800835, 704291, 869699,
    887331, 725570, 857795, 857795, 891939, 368578, 887715, 358978, 757827, 764739, 673, 395811, 697572, 515972, 860100, 662498, 405892, 870467, 869699, 910466, 910466, 740292, 603778, 910851, 712578, 81380, 886083, 90467, 743139, 405892, 910466, 888291,
    3970, 502531, 545218, 673, 454178, 898659, 883299, 662498, 398181, 910466, 888291, 918114, 785092, 405892, 400739, 663141, 603778, 704291, 918114, 869698, 883299, 743170, 542658, 879651, 869828, 891171, 90467, 662501, 910466, 673, 859331, 789060,
    643171, 90467, 606178, 2305, 800836, 719267, 911715, 750530, 870596, 859331, 870596, 909603, 712578, 869698, 662498, 647492, 912386, 461378, 882626, 910371, 869570, 72578, 901923, 451778, 905859, 869698, 869316, 663140, 81218, 821060, 417698, 857922,
    139586, 395811, 869954, 910179, 886563, 906147, 905763, 880419, 918114, 100260, 72578, 869954, 72578, 739939, 892035, 107172, 918114, 417698, 870596, 610, 168962, 461378, 168962, 912771, 368578, 803652, 755012, 841284, 317762, 671330, 712580, 743779,
    662498, 887234, 673, 461381, 712899, 712578, 869698, 895011, 912386, 899331, 819394, 870467, 911043, 486018, 870596, 454178, 276643, 869955, 869316, 739939, 81411, 535299, 406532, 908739, 712580, 673, 869572, 910755, 891171, 100259, 893859, 869698,
    406594, 647491, 663140, 700932, 405892, 712741, 886563, 610, 69698, 912099, 911715, 884739, 905667, 712581, 406594, 769219, 535301, 3970, 417698, 869316, 725570, 81411, 610980, 81219, 869444, 793156, 886563, 764739, 486018, 843972, 900771, 105058,
    740323, 869828, 803652, 906147, 815300, 454178, 7393, 400739, 858820, 542658, 603778, 871234, 72578, 180482, 869316, 712580, 893859, 366178, 713059, 712899, 542661, 358978, 899331, 895011, 288387, 918114, 368578, 395811, 398212, 769220, 860228, 454178,
    911715, 100259, 461378, 461378, 879651, 395811, 908451, 409380, 869698, 406594, 486019, 329282, 712580, 87587, 542658, 712578, 690852, 912771, 906339, 72547, 910466, 912195, 3970, 886083, 662498, 750530, 857795, 569698, 576898, 775620, 886083, 857922,
    893859, 542661, 908451, 912195, 81220, 795332, 662979, 486018, 871234, 94114, 764740, 495811, 81218, 662498, 859844, 910371, 276644, 395781, 887234, 542658, 662501, 81282, 870082, 105058, 610980, 107938, 864452, 893571, 662979, 712578, 910179, 81218,
    406594, 870340, 904707, 515971, 870980, 870980, 662981, 750530, 398212, 800835, 129, 882339, 712581, 704291, 910466, 90467, 882819, 662979, 461378, 662498, 662498, 908739, 746563, 746563, 7330, 912386, 725570, 882819, 406501, 72578, 3970, 662498,
    713059, 887331, 542661, 712580, 645730, 662979, 418, 870082, 869700, 461379, 235778, 713059, 279746, 358629, 903075, 891171, 897315, 100259, 750530, 857924, 395780, 366178, 752196, 795332, 869316, 139586, 690852, 400739, 695810, 81282, 892035, 746563,
    168962, 107172, 910371, 835779, 662980, 107938, 713059, 418, 891939, 743139, 395811, 884739, 870596, 739908, 910275, 905283, 107938, 869699, 869954, 911043, 7393, 819396, 81218, 110051, 81189, 81411, 81220, 81220, 712578, 72578, 486018, 180483,
    713221, 910371, 662979, 869570, 81221, 886563, 903075, 869572, 542661, 107938, 666372, 400739, 857668, 100259, 746563, 643171, 739395, 542658, 105058, 712899, 910371, 819524, 869700, 643171, 368578, 180483, 712578, 107938, 891939, 896739, 180482, 912195,
    81218, 695810, 911043, 72578, 129, 910755, 662501, 279746, 107172, 288386, 405892, 910563, 406594, 908451, 569698, 69667, 739939, 171842, 417698, 666372, 398180, 858434, 908355, 610, 486018, 640772, 910466, 662500, 859331, 454178, 912386, 891939,
    893859, 90468, 610, 671330, 400739, 740323, 908451, 0, 739939, 908451, 857795, 903075, 72578, 870467, 878275, 3169, 454178, 603778, 884739, 406594, 662501, 700932, 725570, 725762, 725762, 542660, 912579, 713059, 576898, 908355, 908355, 690851,
    750530, 662498, 785092, 405861, 785092, 892611, 690851, 662980, 882435, 486019, 406594, 869316, 853700, 896739, 451778, 461378, 869316, 81218, 81411, 891171, 743170, 712580, 870082, 395781, 869570, 244418, 420098, 569698, 569698, 81218, 894339, 427298,
    451778, 690852, 7330, 723524, 869700, 878275, 785092, 819394, 869316, 81219, 897891, 719267, 406501, 643172, 893859, 869316, 486018, 358627, 740292, 180483, 912579, 869316, 671330, 740323, 918114, 799300, 857668, 712578, 81187, 395780, 746563, 858979,
    910947, 673, 912675, 764611, 400739, 81411, 869955, 81190, 744258, 746563, 107938, 704291, 739524, 785092, 7330, 712578, 662498, 740323, 542660, 869570, 748484, 81411, 869699, 247298, 892227, 857795, 910947, 903075, 545218, 671330, 100259, 869700,
    417698, 712900, 7393, 870596, 495811, 843972, 81411, 910563, 662498, 673, 912195, 869698, 859331, 785092, 869828, 406594, 107172, 892227, 793156, 107938, 643171, 883587, 451778, 218818, 757827, 746564, 869443, 878275, 645730, 645730, 869699, 912579,
    841284, 750530, 795331, 317762, 743779, 858434, 740323, 712579, 7330, 886563, 645730, 743108, 486018, 495811, 740323, 695810, 395781, 910947, 572098, 739394, 695810, 545218, 81411, 366178, 911715, 903075, 869699, 81218, 572098, 405892, 713059, 542658,
    895587, 739939, 835780, 869316, 576898, 100259, 542658, 545218, 803652, 576898, 746563, 398180, 647491, 645730, 712581, 743779, 576898, 712578, 886563, 180482, 100259, 395780, 869700, 719268, 750532, 418, 317762, 869316, 892227, 743748, 81220, 743779,
    358629, 903075, 869316, 739394, 542658, 395780, 739908, 542658, 869698, 897315, 835780, 869954, 288386, 712578, 870083, 486020, 869954, 358627, 795331, 903075, 835780, 795331, 542818, 910371, 81411, 869699, 906339, 671330, 841283, 486020, 81411, 869443,
    869699, 835779, 868932, 868932, 757827, 513572, 869828, 486181, 853572, 110051, 744132, 81218, 81411, 869828, 673, 891171, 527011, 744132, 700932, 329282, 910371, 806467, 891171, 906819, 907106, 800836, 461378, 891171, 908643, 891171, 329282, 911715,
    891171, 869699, 409411, 247298, 107938, 395811, 869828, 806467, 857924, 107938, 857796, 891171, 81189, 910371, 869828, 857924, 180482, 247298, 869828, 869828, 3970, 329282, 329282, 329282, 868932, 896931, 81411, 869698, 81220, 804164, 804164, 329282,
    3970, 329282, 697572, 663141, 247298, 545218, 793156, 3970, 912099, 910371, 454178, 107938, 910466, 461539, 645730, 869698, 420098, 662500, 662500, 713060, 870980, 746564, 662499, 712579, 662980, 535301, 395781, 662498, 903075, 576898, 486018, 486019,
    870467, 870467, 721603, 910947, 869698, 857795, 898659, 502531, 858180, 704291, 764612, 723523, 723523, 690851, 513571, 406532, 912771, 515972, 725570, 495812, 750530, 495812, 486018, 793156, 750530, 495812, 486020, 690851, 542658, 358978, 910466, 764739,
    870724, 244418, 721603, 690852, 527011, 857922, 690851, 870467, 910179, 767299, 647492, 750532, 461379, 610978, 908643, 645730, 486181, 100260, 887714, 725762, 545218, 906339, 579298, 171842, 890691, 910851, 358978, 400739, 72547, 800836, 358627, 918114,
    81218, 870467, 81380, 418, 803652, 803652, 545218, 725762, 883299, 905763, 821059, 869444, 755012, 486018, 72578, 545218, 882626, 400739, 894339, 888003, 869699, 743139, 427299, 870340, 918114, 461381, 911043, 886563, 886563, 461539, 69667, 100260,
    869699, 764610, 890691, 869828, 884739, 910563, 515971, 662981, 859844, 673, 727236, 750532, 405892, 3970, 535299, 417698, 168962, 746564, 892611, 395812, 901923, 81282, 869444, 712580, 869828, 721604, 918114, 358978, 869698, 869699, 395811, 869570,
    81380, 857795, 769220, 406594, 723524, 764612, 906723, 107172, 502531, 486019, 750530, 897891, 795332, 789060, 398181, 662980, 495811, 713059, 895011, 895011, 368578, 398181, 81219, 368578, 859331, 910947, 891171, 486181, 7330, 610980, 750530, 107938,
    368578, 3970, 857924, 288387, 645730, 857796, 87587, 139586, 806467, 713061, 523652, 405861, 755012, 883299, 690851, 81282, 882339, 81187, 723524, 513571, 870467, 886563, 869698, 870467, 869698, 905859, 486018, 870852, 662498, 907587, 139586, 610978,
    180482, 171842, 690852, 527011, 610978, 486020, 427299, 495811, 767300, 107938, 673, 643172, 461378, 329282, 797252, 764612, 3970, 662500, 527011, 81282, 663140, 886563, 901923, 329282, 398181, 869699, 409411, 725570, 542660, 891171, 884739, 358629,
    743139, 879075, 879075, 107938, 750530, 898179, 891939, 858436, 7330, 769220, 513571, 871108, 107938, 81221, 673, 911043, 100259, 879075, 882626, 898659, 395781, 427298, 910179, 869699, 764740, 713221, 891939, 495811, 398212, 767300, 409380, 486020,
    647492, 894339, 712581, 81221, 797252, 755524, 87587, 869572, 486018, 486018, 793156, 869700, 317762, 398181, 107172, 892611, 288386, 288386, 859331, 819394, 317762, 90468, 398181, 329282, 690852, 906723, 713221, 542658, 713221, 721604, 878275, 869956,
    427299, 864452, 910466, 713061, 910947, 454178, 713060, 406594, 775620, 100259, 513571, 864452, 869828, 90467, 910563, 895587, 739939, 795332, 395781, 643172, 723524, 398180, 545218, 545218, 743170, 662500, 898659, 576898, 358629, 276646, 276643, 896739,
    400739, 815299, 725700, 495812, 739522, 276643, 870467, 712901, 513571, 107172, 750532, 398180, 870467, 712578, 891171, 662501, 495811, 750530, 662981, 579298, 882819, 100259, 317762, 870084, 662500, 662501, 542661, 910371, 870724, 857668, 870084, 869570,
    662980, 662980, 358978, 247298, 793156, 755012, 764611, 906723, 576898, 757827, 757827, 912675, 750530, 663141, 886563, 886563, 690851, 748484, 579298, 764739, 81221, 690851, 417698, 7330, 358627, 486020, 725762, 662501, 486018, 857668, 515971, 869698,
    645730, 725570, 750532, 486021, 7330, 107172, 910851, 690851, 795331, 139586, 406532, 746564, 719268, 662500, 910947, 712580, 712578, 72578, 406594, 3970, 883587, 740323, 858436, 495811, 894339, 535301, 545218, 100259, 690851, 910371, 406532, 803652,
    803652, 907299, 542658, 800835, 719268, 395780, 910947, 100259, 894339, 358978, 87587, 859331, 858051, 666371, 358630, 358630, 663141, 869698, 897891, 171842, 358627, 358630, 81218, 81218, 527011, 869698, 81218, 869570, 527011, 859844, 815300, 912675,
    859844, 869698, 870467, 461378, 81187, 486021, 764740, 879651, 870212, 725572, 288387, 906819, 800836, 918114, 883011, 712579, 712741, 910563, 901923, 712579, 180483, 712578, 910275, 869828, 898179, 288386, 886563, 886563, 7330, 857796, 910275, 890691,
    890691, 662498, 817220, 817220, 869699, 743779, 398180, 417698, 870082, 864452, 908643, 918114, 776132, 911043, 100260, 395812, 888291, 859331, 908739, 908643, 905539, 100259, 725570, 869698, 535299, 486019, 886563, 81219, 535301, 110051, 486019, 358978,
    3138, 542660, 81220, 750530, 100259, 905763, 789060, 662498, 901635, 764740, 910563, 764739, 750532, 857924, 180483, 912483, 81220, 662498, 892611, 712899, 723523, 869698, 486021, 908739, 869956, 723523, 712578, 81221, 887331, 750660, 910947, 723524,
    750532, 725762, 663139, 90468, 69698, 725762, 910466, 725762, 486021, 542660, 886563, 110051, 486021, 859331, 764739, 743779, 725570, 804164, 886563, 613538, 613538, 752196, 901635, 662500, 486020, 317762, 892035, 844484, 69667, 740323, 712580, 764611,
    486018, 860228, 858434, 887331, 87588, 405892, 358630, 918114, 569698, 746563, 858051, 918114, 495812, 100259, 739939, 90467, 713221, 235778, 427298, 486018, 515972, 713221, 911043, 495811, 110051, 819524, 887715, 400739, 911907, 358630, 903651, 100259,
    603778, 869443, 764739, 725570, 107938, 910179, 764740, 87587, 764612, 910466, 107938, 461539, 427298, 358978, 610978, 870596, 870980, 870340, 417698, 901635, 869828, 905539, 712580, 870467, 673, 898659, 576898, 662498, 725570, 663140, 906243, 673,
    72578, 869570, 764610, 713061, 821059, 721603, 579298, 918114, 800836, 912386, 912386, 859331, 719267, 662980, 100260, 815299, 891939, 461378, 671330, 869444, 910371, 912386, 139586, 662980, 647491, 858051, 911043, 755012, 486021, 276643, 486018, 888003,
    712581, 81219, 789060, 785092, 890691, 893859, 486020, 486020, 869698, 610980, 870467, 486018, 486018, 712580, 398212, 461381, 502531, 535299, 100259, 821060, 895011, 905859, 673, 911715, 100259, 69667, 663140, 869699, 859331, 461539, 857924, 912386,
    676452, 910371, 405892, 610, 896739, 3970, 906915, 785092, 804164, 647492, 712580, 858820, 542658, 398181, 910563, 815300, 918114, 908451, 898659, 910371, 171842, 743748, 743748, 697572, 610981, 869444, 835779, 107938, 603778, 906147, 902403, 789060,
    663140, 870467, 395811, 750530, 358630, 673, 647491, 329283, 712580, 819396, 910466, 100259, 910466, 406594, 100259, 107938, 671330, 107938, 610, 542658, 712580, 139586, 857922, 870724, 662498, 712580, 870980, 755012, 395781, 545218, 406594, 673,
    673, 896739, 819396, 744132, 906147, 139586, 244418, 882339, 800836, 87587, 725700, 869316, 276643, 662498, 81411, 764611, 94114, 907299, 690851, 887331, 887331, 891939, 864452, 610978, 857924, 495812, 409411, 495812, 662500, 750532, 869444, 81218,
    87587, 893571, 725570, 869699, 743108, 901923, 918114, 495812, 806467, 81282, 795331, 527011, 869699, 3970, 139586, 542658, 750532, 712580, 81219, 81219, 869700, 395780, 663140, 697572, 746564, 911043, 910466, 835779, 743748, 486019, 905539, 869570,
    107938, 910371, 750530, 907203, 898659, 746563, 858436, 857795, 81411, 94114, 662500, 409380, 819524, 918114, 891939, 755524, 869698, 713221, 898659, 81221, 819396, 81282, 869700, 869572, 891939, 910371, 898659, 869828, 713061, 906339, 859331, 910563,
    910179, 542660, 898179, 719267, 671330, 853572, 910371, 610, 744258, 81218, 725762, 891939, 879075, 869570, 905539, 461539, 81187, 911043, 645730, 712578, 869828, 869700, 858180, 858434, 400739, 895587, 898659, 640771, 725762, 569698, 890691, 180483,
    857795, 486019, 907587, 713060, 869828, 891939, 858180, 821060, 869700, 486018, 288386, 839748, 712580, 908739, 640772, 795332, 901923, 879075, 406594, 645730, 647491, 869698, 853700, 107938, 894339, 869572, 815299, 869698, 712901, 513572, 107172, 610978,
    542658, 750530, 461539, 100259, 495811, 358629, 395781, 870082, 395811, 837699, 893859, 576898, 713059, 81218, 81218, 725762, 837699, 662501, 569698, 906051, 461378, 910466, 857668, 910179, 712900, 819524, 712580, 712580, 576898, 799300, 542660, 81219,
    461539, 748484, 870467, 869570, 869698, 81219, 712578, 180483, 906915, 712578, 100259, 395780, 785092, 898659, 764611, 764611, 712580, 869698, 81411, 785092, 918114, 859331, 725570, 662980, 288387, 662501, 542660, 81220, 869443, 839748, 755012, 869698,
    739395, 406594, 610981, 841284, 569698, 864580, 743108, 495811, 882626, 107938, 647491, 858180, 858434, 901635, 100259, 908643, 869700, 662500, 839748, 911907, 869700, 819396, 725570, 569698, 663140, 869444, 610978, 427298, 895587, 869828, 795331, 100259,
    276646, 800836, 785092, 610981, 795331, 662500, 869700, 276646, 81411, 857924, 819394, 542658, 804164, 100259, 358627, 795331, 912386, 891939, 892227, 892227, 523652, 110051, 569698, 912386, 882915, 679811, 918114, 879651, 912386, 869316, 869443, 817220,
    671330, 908643, 879075, 486018, 461378, 613538, 910851, 911715, 358629, 739939, 804164, 679811, 901923, 899331, 744132, 662500, 276643, 739395, 3970, 912386, 610978, 918114, 912386, 542658, 910851, 366178, 858434, 901635, 903075, 903075, 869700, 804164,
    819396, 841284, 858180, 317762, 461539, 870212, 662661, 107938, 859844, 81282, 329282, 676452, 87587, 907203, 911043, 405892, 869698, 495812, 870852, 819394, 358627, 746564, 405892, 871108, 461380, 870596, 752196, 87587, 869698, 879075, 859331, 859331,
    405861, 663140, 171842, 869572, 859331, 911715, 81220, 869698, 864580, 859331, 712580, 775620, 398181, 906819, 775620, 869316, 912099, 712899, 879075, 739939, 100259, 869316, 486019, 663140, 398180, 858180, 81221, 819394, 893187, 535300, 727235, 288386,
    882339, 869955, 486018, 869828, 486018, 869699, 819394, 893187, 857796, 72578, 713060, 486020, 918114, 841284, 727236, 81220, 898659, 857795, 395811, 107938, 859331, 841284, 495811, 841284, 870467, 107938, 879075, 535300, 841284, 857795, 576898, 69667,
    139586, 461379, 857795, 81218, 910466, 799300, 859331, 841284, 870212, 663140, 910371, 910563, 535300, 81411, 799300, 502531, 81220, 869698, 869698, 879075, 857795, 405861, 879075, 764740, 882915, 427299, 671330, 910947, 883587, 545218, 427299, 859844,
    244418, 764740, 358629, 918114, 663139, 171842, 486021, 486021, 358629, 910371, 912675, 910947, 276643, 427299, 883587, 869316, 891939, 858820, 887234, 887234, 887234, 451778, 486020, 276643, 398180, 662981, 662981, 576898, 451778, 395781, 799300, 712580,
    712580, 527011, 911715, 87588, 911715, 799300, 799300, 911043, 815300, 87588, 87588, 87588, 451778, 911043, 886563, 180483, 713221, 892035, 767299, 461378, 767299, 329283, 767299, 767299, 859331, 869699, 400739, 892707, 610978, 495811, 910371, 839748,
    139586, 645730, 739524, 712578, 725570, 139586, 891939, 864452, 857922, 695810, 891939, 910466, 750660, 859331, 859331, 647492, 486018, 910563, 713060, 910563, 610978, 712900, 712578, 910371, 815299, 673, 896739, 647492, 695810, 887331, 276643, 910563,
    329282, 545218, 545218, 712578, 739524, 545218, 139586, 910563, 139586, 892227, 799300, 892227, 276643, 712578, 72547, 704291, 610978, 662980, 610, 712580, 542658, 673, 81411, 81411, 803652, 869698, 662498, 858051, 910466, 395811, 663140, 107938,
    81187, 288387, 662498, 858051, 911907, 721603, 911043, 81283, 3970, 910466, 891939, 673, 486018, 662500, 764739, 87587, 662498, 739523, 662500, 662498, 69667, 891939, 857922, 767300, 712580, 725762, 795331, 883587, 358978, 912771, 912771, 100260,
    139586, 767300, 171842, 890691, 910851, 244418, 542658, 864580, 857922, 100259, 918114, 910755, 486019, 800836, 910179, 645730, 725762, 883299, 535301, 911907, 857922, 81218, 244418, 821059, 81282, 864580, 800836, 400739, 647492, 755012, 395811, 180483,
    910755, 898179, 700932, 910851, 869698, 725572, 72578, 358978, 897315, 673, 918114, 910947, 884739, 545218, 3138, 887234, 888003, 857924, 755012, 869698, 579298, 859331, 81282, 910371, 81220, 870467, 819394, 542658, 486018, 700932, 910371, 911715,
    662499, 712579, 542658, 911043, 785092, 713061, 804164, 785092, 673, 673, 107938, 910563, 368578, 857668, 906819, 764740, 764740, 901923, 542658, 911715, 712580, 858180, 883587, 663140, 398181, 329282, 769220, 502531, 888291, 712579, 869699, 107938,
    795332, 513571, 893859, 893571, 712580, 906723, 523652, 405861, 405861, 317762, 800835, 81219, 139586, 893571, 662498, 821059, 527011, 527011, 871234, 461378, 882819, 3970, 3970, 767300, 806467, 806467, 797252, 892227, 886563, 713061, 891171, 869699,
    755012, 395811, 713061, 910371, 869316, 645730, 100259, 676452, 891939, 723524, 871234, 180482, 755012, 712900, 461378, 869699, 912386, 870082, 81221, 107938, 910275, 100260, 764740, 906339, 100260, 695810, 869698, 870467, 859331, 912099, 911043, 673,
    358627, 513571, 545218, 100259, 898659, 395811, 486018, 486018, 81220, 697572, 663141, 898659, 662500, 910466, 107938, 871108, 795331, 171842, 100259, 100259, 910371, 871108, 69667, 288387, 858180, 171842, 857795, 107938, 662498, 896739, 329282, 427299,
    712578, 90468, 739396, 746564, 858180, 891939, 892611, 895587, 910466, 676452, 857668, 912386, 910371, 723524, 750532, 883011, 662500, 107172, 903075, 883011, 486018, 859331, 907491, 676452, 871108, 486021, 799300, 513572, 395780, 884739, 739523, 752196,
    329282, 896739, 712901, 712900, 712900, 725762, 882435, 857922, 662501, 882819, 912579, 883587, 883587, 712578, 107938, 905859, 803652, 712581, 739524, 366178, 857795, 785092, 100259, 893859, 712900, 800835, 799300, 799300, 495811, 748484, 886563, 912675,
    785092, 461539, 911043, 785092, 911043, 892227, 712900, 869698, 910851, 901923, 7330, 7330, 869698, 903075, 712580, 486018, 910563, 171842, 712578, 750530, 725762, 912579, 910371, 869698, 858180, 883587, 800836, 891939, 107938, 911907, 869699, 858434,
    891939, 542658, 461539, 803652, 803652, 750532, 858180, 712578, 139586, 395780, 869700, 180483, 100259, 910371, 901923, 857924, 740292, 744260, 804164, 750532, 750532, 903075, 100259, 100259, 100259, 180483, 100259, 81189, 793156, 81189, 870596, 81221,
    495812, 910947, 740292, 740292, 895011, 775620, 912771, 857668, 139586, 775620, 139586, 910466, 247298, 247298, 69667, 906339, 906339, 486018, 869698, 671330, 358978, 870467, 870340, 712581, 869698, 495811, 910947, 789060, 358978, 910947, 495811, 918114,
    869954, 725570, 859331, 368578, 752196, 663461, 764611, 903075, 859331, 764739, 235778, 870467, 712578, 81218, 746563, 910371, 712901, 690852, 883299, 576898, 905859, 235778, 909603, 366178, 542660, 870467, 870596, 366178, 870467, 750530, 576898, 869698,
    869698, 647492, 662499, 789060, 858434, 663461, 883299, 713061, 859972, 857668, 713060, 859331, 906819, 712578, 721604, 883299, 461539, 743108, 647492, 712901, 666371, 87587, 883299, 910371, 395781, 662979, 662498, 870082, 891171, 898179, 427298, 743139,
    713061, 869570, 819394, 910371, 94114, 918114, 869698, 721604, 647492, 884739, 775620, 640771, 81221, 908739, 819394, 486019, 693252, 81411, 713221, 662980, 815299, 579298, 725700, 891171, 870082, 579298, 721604, 712901, 495812, 395780, 918114, 870467,
    81218, 764611, 81190, 725570, 725570, 712901, 819394, 857668, 395780, 764739, 817220, 721604, 743108, 899331, 81187, 891939, 712580, 795331, 767299, 647492, 610980, 486020, 727236, 727236, 869444, 767299, 911043, 81220, 911715, 911715, 486018, 461381,
    645730, 697572, 870082, 461381, 358630, 576898, 857796, 725570, 857795, 757827, 81190, 461539, 3970, 878787, 891939, 869570, 545218, 910466, 358630, 180482, 869700, 750530, 905763, 750532, 3970, 819524, 769220, 882626, 918114, 700932, 640772, 887331,
    712581, 869698, 870467, 864580, 870467, 910755, 427298, 912771, 869316, 329282, 461539, 180482, 815300, 697572, 911715, 368578, 727236, 398181, 893571, 725570, 486018, 81282, 276643, 513572, 107172, 910179, 676452, 276643, 857796, 107938, 107938, 819524,
    908355, 739522, 712581, 887331, 892035, 673, 461539, 895587, 662979, 542660, 795332, 908355, 870467, 368578, 857796, 892035, 892035, 495811, 725762, 893283, 896739, 869699, 799300, 883299, 545218, 869698, 81187, 395780, 329282, 671330, 869443, 819524,
    395781, 910466, 887331, 461381, 180482, 793156, 725570, 662501, 910563, 800835, 800835, 799300, 545218, 486018, 869700, 712580, 697572, 107172, 542658, 697572, 869699, 910947, 100259, 697572, 750530, 905667, 486020, 81411, 901635, 81411, 775620, 610980,
    892611, 725572, 869698, 662498, 843972, 764739, 139586, 100259, 869698, 869956, 725570, 869698, 869698, 887234, 666371, 859331, 887234, 859331, 859331, 775620, 750660, 869699, 81218, 918114, 405892, 910466, 329283, 400739, 486021, 358978, 663140, 750530,
    712901, 892227, 908739, 918114, 750530, 819394, 712580, 81218, 740292, 613538, 495812, 713060, 869698, 819394, 857924, 87587, 858820, 907491, 869572, 495812, 750660, 712901, 486021, 662501, 643172, 746563, 662501, 725570, 725570, 663139, 746563, 486020,
    890691, 454178, 764610, 725570, 893187, 486020, 880419, 869316, 746563, 891171, 486021, 405892, 819394, 891171, 87587, 892227, 486018, 486018, 461539, 746564, 545218, 545218, 868932, 81411, 870467, 81411, 81411, 869698, 869698, 869698, 869698, 911907,
    911907, 662980, 869444, 90467, 700932, 100259, 100259, 107938, 400739, 882339, 725700, 750660, 727235, 643172, 107172, 853700, 712901, 869700, 400739, 400739, 870980, 910179, 486018, 764612, 713221, 869698, 87588, 72547, 725570, 750530, 800836, 918114,
    87588, 713061, 744258, 87588, 495812, 883299, 712578, 910947, 398211, 100260, 406532, 545218, 743139, 87588, 398181, 870467, 869570, 495812, 329282, 744132, 870467, 893571, 744132, 420098, 905667, 712578, 755012, 486018, 395781, 870467, 744132, 870596,
    893571, 107938, 910563, 891939, 486020, 859331, 486019, 725572, 81221, 918114, 910947, 910371, 640772, 911043, 910371, 712578, 713061, 853700, 695810, 420098, 713061, 713061, 880419, 420098, 405861, 72547, 785092, 81411, 785092, 81411, 898659, 803652,
    712581, 879075, 857922, 869443, 869443, 725570, 869698, 537858, 858051, 883587, 752196, 888003, 486018, 858051, 858051, 395780, 690851, 320642, 320642, 767299, 662498, 859331, 882915, 870467, 87588, 815300, 712578, 764611, 869700, 864452, 821059, 90467,
    713221, 405861, 910466, 486018, 910466, 712578, 486019, 911715, 87587, 744132, 676452, 793156, 107938, 725572, 870467, 662498, 882819, 398180, 908643, 704291, 662500, 610980, 409411, 712741, 903651, 535301, 712580, 869443, 513572, 81218, 892611, 764739,
    406532, 908643, 676452, 910466, 743170, 395781, 81190, 139586, 869698, 879075, 740292, 821059, 905859, 912771, 870083, 100260, 139586, 721603, 613538, 901923, 713061, 235778, 368578, 909603, 81218, 81218, 139586, 417698, 542660, 671330, 743108, 750530,
    883299, 869698, 329282, 911715, 918114, 486018, 643172, 869444, 869828, 610978, 776132, 918114, 712578, 725570, 911715, 906243, 869572, 744132, 727236, 81221, 869570, 81187, 725570, 276643, 712578, 690851, 107172, 871108, 427299, 576898, 870467, 869698,
    859331, 837699, 910947, 859331, 712578, 893859, 398212, 535301, 900771, 870467, 910563, 542660, 864580, 81221, 884739, 740323, 523652, 461539, 398212, 712578, 785092, 276643, 671330, 81218, 906339, 100260, 81411, 81411, 400739, 486019, 869698, 712741,
    776132, 800835, 81411, 912386, 869570, 400739, 662981, 613538, 398211, 3138, 171842, 755012, 864452, 775620, 857924, 712578, 662499, 911907, 643172, 515971, 420098, 81411, 673, 869316, 700932, 100259, 892611, 739522, 901923, 110051, 815300, 107172,
    107172, 869828, 645730, 527011, 857668, 911907, 107938, 329282, 891939, 712578, 713059, 800835, 695810, 395811, 740323, 906243, 695810, 368578, 869698, 662500, 81411, 712578, 879651, 815299, 869698, 398212, 676452, 857924, 398181, 180483, 910466, 910466,
    911715, 542658, 542661, 81220, 744132, 869698, 911907, 542658, 841283, 901923, 673, 906723, 869572, 662500, 910466, 821059, 744132, 887331, 879651, 420098, 882819, 395781, 817219, 666371, 486018, 461380, 461380, 869444, 883587, 662498, 523652, 864452,
    806467, 712578, 513571, 778435, 610978, 857796, 743108, 757827, 891171, 712900, 495812, 87587, 869828, 495811, 486018, 795331, 107938, 743779, 72578, 857924, 690851, 695810, 695810, 903075, 87588, 743748, 81218, 409411, 712578, 395781, 869443, 910371,
    81411, 673, 908451, 871108, 486020, 107172, 750532, 139586, 739395, 398181, 420098, 879651, 906243, 739395, 746563, 461380, 886563, 901923, 110051, 712581, 870724, 884739, 81221, 409380, 662661, 858050, 869444, 739522, 725572, 821059, 871108, 695810,
    857668, 743108, 879651, 869572, 910371, 454178, 662500, 910371, 107938, 107938, 725570, 869572, 81220, 821059, 725570, 869698, 819396, 107172, 892611, 746564, 869699, 673, 398180, 898179, 662499, 171842, 107938, 406594, 906243, 100259, 905283, 767299,
    110051, 871108, 878275, 797252, 764739, 869700, 180483, 180483, 906723, 775620, 110051, 288386, 288386, 662498, 911043, 676452, 81221, 486018, 461378, 139586, 81190, 610978, 910563, 892611, 713061, 107172, 882435, 910947, 663140, 366178, 892611, 869828,
    712578, 409411, 107172, 100260, 542658, 907587, 908739, 869699, 882626, 869570, 662500, 908643, 81221, 110051, 663140, 821060, 775620, 663139, 744132, 853700, 579298, 579298, 891171, 400739, 739395, 739395, 739395, 882819, 821059, 320642, 712901, 700932,
    725762, 276646, 815299, 461380, 906723, 912195, 892035, 81411, 892611, 542660, 743170, 676452, 857795, 869699, 545218, 662981, 910371, 868932, 815299, 662500, 663139, 107172, 857796, 513572, 750530, 870467, 896739, 789060, 662498, 662498, 107938, 647492,
    799300, 744132, 911715, 405861, 870467, 700932, 398181, 662498, 857922, 643172, 879651, 880419, 662661, 785092, 870084, 400739, 542660, 647492, 180483, 81411, 857668, 398181, 869699, 869699, 712578, 712578, 746563, 690851, 886563, 427299, 427299, 288386,
    892227, 739522, 869316, 764611, 764739, 893859, 893859, 405861, 395811, 662498, 712580, 821059, 495812, 799300, 910466, 739395, 892227, 744258, 712580, 690851, 663141, 645730, 869698, 81187, 712580, 901923, 746563, 690851, 495812, 676452, 395811, 785092,
    910371, 869572, 755012, 857795, 857795, 139586, 882435, 515971, 910563, 869700, 725762, 898179, 869698, 800835, 279746, 502531, 406532, 817219, 795331, 406594, 800835, 100259, 893283, 712579, 400739, 107938, 869699, 883299, 817220, 406594, 817220, 427299,
    662498, 576898, 795331, 817219, 743108, 870467, 905283, 905283, 100259, 100260, 366178, 870467, 535301, 535301, 911715, 918114, 895587, 899331, 398181, 81411, 395780, 817220, 795331, 406532, 869572, 719268, 395780, 486018, 139586, 81220, 663139, 869698,
    542660, 835779, 819394, 690852, 892227, 767300, 767300, 358630, 454178, 911907, 910466, 486020, 869570, 276643, 858180, 712578, 398181, 94114, 891939, 454178, 486020, 662501, 743779, 662501, 712900, 755012, 662498, 486019, 884739, 912291, 486019, 486018,
    870467, 513571, 486018, 180483, 888003, 748484, 712900, 662501, 454178, 486020, 891939, 107938, 486020, 100259, 545218, 81218, 712900, 712578, 712578, 743748, 883299, 857922, 81218, 81218, 81218, 673, 859331, 695810, 898179, 81218, 542658, 764612,
    908451, 725572, 750532, 486181, 662981, 869698, 869698, 662980, 81218, 887331, 100259, 859844, 859844, 882339, 712899, 906339, 537858, 247298, 358978, 417698, 858052, 764611, 858434, 912099, 910179, 712901, 750660, 513572, 107938, 180482, 767299, 663140,
    69667, 910179, 461378, 893187, 910466, 748483, 662500, 870467, 858051, 893283, 817219, 907587, 100260, 903651, 721603, 72547, 906915, 712578, 662500, 721603, 486019, 107938, 918114, 725762, 712578, 645730, 870467, 870467, 673, 748483, 870340, 398181,
    486018, 100259, 535301, 725572, 740292, 139586, 90467, 81218, 911907, 892227, 358978, 857795, 746563, 713061, 110051, 87587, 606178, 869698, 276643, 358628, 906147, 905539, 886083, 513572, 713059, 906147, 918114, 905763, 358978, 725762, 870082, 868932,
    893187, 806467, 662501, 110051, 81218, 879075, 727236, 870467, 643172, 835779, 368578, 645730, 908643, 911043, 713060, 712899, 100260, 879075, 72578, 610978, 7330, 750530, 400739, 645730, 486019, 576898, 898659, 398180, 3970, 743108, 857922, 821059,
    81187, 486019, 918114, 712899, 358627, 746563, 739908, 579298, 610980, 418, 918114, 905763, 721603, 905667, 800835, 603778, 776132, 662500, 139586, 317762, 358978, 910179, 910179, 725762, 139586, 870980, 640771, 870340, 878787, 878787, 750530, 610978,
    139586, 893283, 803652, 486020, 662499, 107172, 795331, 841284, 366178, 870082, 662500, 398211, 893283, 859331, 891939, 713061, 841283, 775620, 775620, 857924, 276645, 882915, 869954, 892707, 869698, 910563, 739522, 81411, 647491, 713381, 859331, 603778,
    878787, 400739, 461379, 870467, 893283, 886563, 886563, 870467, 906243, 81282, 843972, 713060, 454178, 785092, 535299, 755012, 603778, 864580, 895011, 712899, 545218, 884739, 610, 892227, 895011, 139586, 870467, 804164, 3970, 94114, 317762, 910563,
    901923, 746564, 785092, 713061, 486018, 81411, 712580, 107938, 406594, 910179, 905283, 883011, 366178, 769220, 695810, 713060, 398181, 276866, 276866, 515972, 898659, 843972, 139586, 643172, 486020, 869698, 697572, 662980, 858435, 87587, 910563, 69667,
    764740, 785092, 910466, 502531, 892035, 700932, 527011, 743779, 329282, 870980, 906147, 545218, 358978, 395811, 858820, 903075, 789060, 870082, 276643, 7330, 892227, 368578, 87587, 368578, 912675, 7330, 870212, 545218, 908355, 910179, 893859, 679811,
    721604, 486020, 81220, 883587, 910371, 87587, 857795, 643172, 663139, 895011, 821059, 817219, 918114, 878787, 139586, 883587, 94114, 406594, 789060, 911907, 871108, 398180, 81380, 891939, 329282, 723523, 603778, 610980, 907491, 912771, 94114, 662498,
    712580, 398181, 750532, 907587, 806467, 857796, 767300, 886563, 859844, 750660, 906147, 750530, 712901, 409411, 486018, 908739, 662499, 662498, 905667, 276643, 662500, 800836, 7330, 3970, 892227, 713061, 542661, 461379, 869698, 743779, 662980, 713061,
    896931, 882339, 690851, 72547, 727236, 891939, 695810, 523652, 329283, 100259, 883011, 870596, 139586, 864452, 662500, 515971, 878787, 235778, 891939, 545218, 776132, 821060, 891939, 870467, 746563, 817220, 739395, 907587, 461380, 643172, 869828, 905283,
    486020, 713061, 892035, 486020, 107938, 100259, 180483, 835779, 405892, 907203, 645730, 910371, 878275, 912483, 869572, 819524, 695810, 903075, 69667, 859331, 871108, 713060, 869698, 910947, 107172, 725572, 100260, 81220, 100259, 869955, 81282, 869570,
    3970, 94114, 171842, 406501, 486018, 695810, 418, 418, 398180, 892611, 892035, 892035, 329282, 891939, 81411, 769219, 405861, 869316, 884739, 910563, 869828, 662500, 7330, 69667, 712899, 879651, 911043, 910371, 723524, 795331, 90467, 857795,
    395780, 454178, 896739, 871108, 769219, 785092, 890691, 857795, 858180, 906723, 743108, 406501, 712900, 679811, 679811, 366178, 81187, 858180, 721604, 883011, 907587, 171842, 911043, 883587, 870467, 613538, 418, 486020, 908355, 906339, 888291, 712580,
    860228, 666372, 495811, 899331, 673, 81380, 905283, 662980, 317762, 712899, 690852, 640771, 721603, 417698, 542658, 908739, 81218, 870467, 610, 81219, 662500, 90467, 662501, 864452, 713060, 673, 910563, 905283, 857924, 69667, 712899, 454178,
    910466, 3138, 663139, 513572, 663139, 276643, 878787, 911043, 727235, 908355, 690851, 276866, 395780, 908451, 892035, 405861, 81190, 776132, 727236, 764740, 663461, 750530, 662498, 908355, 870212, 542660, 398181, 400739, 461380, 545218, 897315, 896739,
    871108, 888003, 81218, 712899, 81190, 690852, 81219, 882819, 901635, 576898, 486021, 461539, 869700, 870467, 725762, 870467, 903075, 815299, 395780, 893187, 81187, 663139, 868932, 785092, 576898, 576898, 713059, 739395, 515972, 725700, 486021, 911043,
    890691, 662500, 799300, 486018, 869700, 892227, 906051, 906051, 893859, 757827, 712580, 858436, 461539, 869955, 869698, 712580, 398181, 398181, 276866, 901923, 405892, 671330, 395811, 461381, 417698, 912675, 610980, 800835, 713061, 81411, 358630, 785092,
    486018, 882435, 746563, 746563, 72547, 366178, 72547, 892227, 918114, 799300, 723523, 918114, 857668, 366178, 883011, 870980, 358629, 910371, 247298, 857795, 757827, 610980, 81219, 910563, 545218, 785092, 744260, 317762, 395811, 857668, 94114, 486020,
    767299, 107172, 894339, 835780, 545218, 662980, 870980, 870082, 247298, 743748, 486018, 746564, 279746, 882435, 857796, 857796, 81220, 891939, 857795, 793156, 746563, 910563, 168962, 645730, 878787, 725762, 139586, 7330, 673, 817220, 908355, 819394,
    486018, 908643, 870980, 883587, 869444, 795331, 739394, 139586, 7330, 858434, 7330, 7330, 610978, 94114, 891939, 727235, 841284, 454178, 870467, 870467, 893283, 576898, 486020, 910563, 895011, 690851, 878787, 908355, 545218, 896931, 906915, 695810,
    640771, 100260, 894339, 739394, 276643, 869955, 610980, 712580, 502531, 743779, 542658, 418, 81220, 869700, 545218, 803652, 662500, 662500, 869955, 662500, 795331, 712580, 869700, 898659, 719268, 395780, 90468, 744260, 910563, 663139, 897315, 502531,
    819394, 400739, 869955, 804164, 690851, 871236, 542658, 750532, 610, 746564, 785092, 892227, 795331, 542658, 835780, 870467, 461539, 461539, 662498, 910947, 247298, 247298, 859844, 892707, 911715, 888003, 869698, 775620, 767299, 721603, 764612, 713061,
    244418, 750532, 893187, 821059, 910466, 461378, 72578, 739522, 869316, 819394, 81187, 454178, 750530, 869570, 785092, 821059, 398180, 892035, 663140, 886563, 107938, 400739, 725762, 878787, 869570, 785092, 886563, 542660, 903075, 725570, 395781, 750532,
    886563, 869698, 767299, 666371, 171842, 100259, 757827, 486018, 905539, 869316, 886563, 785092, 495812, 100260, 712579, 821059, 276643, 406594, 857668, 406594, 713061, 750530, 713060, 918114, 276643, 799300, 857922, 712741, 869698, 785092, 662980, 785092,
    870980, 869698, 542660, 725570, 542658, 869316, 869316, 90467, 910851, 662498, 662498, 870212, 727235, 892227, 870212, 795332, 887331, 721603, 721603, 795331, 107938, 795331, 910371, 886083, 663141, 799300, 486019, 81218, 910563, 486019, 107938, 713060,
    785092, 94114, 864580, 886083, 495811, 400739, 869699, 918114, 662501, 712900, 857922, 870980, 712900, 755524, 883011, 671330, 869443, 870980, 870980, 857795, 800835, 870980, 81187, 755524, 358627, 764612, 395780, 395812, 107938, 859331, 905859, 911715,
    898659, 640772, 662498, 712900, 898179, 545218, 905859, 645730, 895587, 918114, 764612, 869572, 495812, 495812, 610978, 871234, 870340, 869698, 895587, 858820, 486020, 107938, 869698, 662498, 870467, 764611, 853700, 764611, 712900, 793156, 288387, 107938,
    910466, 486018, 486018, 486018, 743779, 907106, 746563, 892035, 870467, 912195, 910563, 864580, 910947, 869570, 451778, 418, 767299, 395811, 908643, 750530, 869570, 395780, 859331, 535301, 100260, 859331, 87588, 821059, 821059, 704291, 409380, 910466,
    110051, 879651, 81283, 870082, 721603, 891939, 527011, 663141, 81218, 764612, 740323, 461539, 395781, 869572, 821060, 409411, 764739, 857795, 81218, 869698, 712578, 869443, 100259, 870082, 764612, 690851, 663140, 139586, 100260, 869955, 727236, 869444,
    869700, 869572, 244418, 713060, 800836, 409380, 740323, 898659, 912771, 576898, 725572, 869570, 409411, 870340, 871108, 859331, 859331, 107938, 887331, 884739, 898179, 610978, 395781, 90467, 755012, 406594, 727236, 81283, 870596, 803652, 712580, 535301,
    864580, 740292, 859331, 81221, 81221, 869572, 180483, 869570, 785092, 495811, 870467, 495811, 662981, 700932, 405892, 100259, 888003, 712581, 858692, 869316, 7330, 110051, 81411, 395811, 395811, 502531, 87587, 712580, 527011, 870596, 712578, 398181,
    663141, 368578, 713059, 713059, 910466, 87587, 81411, 740323, 398180, 87587, 864452, 870980, 912579, 870596, 247298, 495812, 712578, 662980, 610978, 712578, 857796, 662498, 495812, 461539, 713061, 869698, 870467, 81282, 87587, 800835, 542661, 395781,
    87588, 869316, 869955, 887331, 869572, 769220, 869698, 712579, 110051, 81411, 869570, 821059, 180483, 7330, 869700, 276643, 910371, 276643, 857796, 898179, 835779, 486020, 844484, 81282, 100260, 409380, 911043, 908451, 247298, 821060, 785092, 892611,
    888291, 90468, 395781, 662498, 740323, 100259, 542658, 406594, 486018, 486018, 817219, 869955, 910563, 740323, 579298, 576898, 744132, 610978, 712901, 81411, 879075, 395811, 869572, 81189, 799300, 712580, 690851, 495812, 906051, 800835, 704291, 712578,
    869698, 486018, 81221, 883299, 7330, 7330, 542658, 690851, 700932, 869316, 887331, 663140, 870340, 740323, 869698, 712581, 869444, 663140, 785092, 879075, 495812, 100259, 406594, 7330, 858434, 276643, 817220, 910947, 911043, 247298, 542658, 398180,
    535301, 884739, 799300, 100259, 743779, 545218, 712578, 858180, 803652, 835780, 897315, 804164, 858692, 690852, 100259, 358627, 911907, 911907, 857922, 7330, 769220, 7330, 81218, 72578, 276643, 882435, 882435, 663141, 815299, 739524, 739524, 695810,
    912579, 739395, 869570, 81411, 869699, 906243, 764739, 645730, 911907, 911715, 858436, 643172, 764739, 918114, 800836, 887331, 857922, 695810, 613538, 864580, 860228, 713060, 795332, 795332, 912579, 821060, 81221, 87588, 704291, 740292, 486018, 662498,
    869700, 871108, 87588, 700932, 795332, 893859, 817220, 803652, 579298, 748484, 910371, 576898, 723523, 712899, 276643, 276643, 579298, 358627, 858051, 358978, 247298, 886563, 886563, 883587, 542658, 461539, 486018, 892035, 725570, 910755, 398181, 882626,
    662979, 858434, 869954, 857924, 662500, 107938, 767299, 910466, 69667, 662980, 910947, 764611, 910466, 912386, 542658, 869444, 882339, 882915, 858051, 725700, 910179, 750530, 712901, 815300, 400739, 869698, 888003, 775620, 427298, 542658, 662498, 276643,
    767300, 864580, 486021, 819396, 87587, 893571, 892035, 815300, 100260, 569698, 819524, 671330, 869828, 395812, 887331, 879651, 891939, 755012, 888291, 911043, 712580, 81187, 409380, 235778, 276643, 81219, 869443, 743108, 358627, 743108, 676452, 721603,
    676452, 486018, 94114, 495811, 890691, 486018, 898179, 645730, 725570, 87588, 81283, 764612, 764739, 869698, 918114, 835779, 712578, 81218, 3970, 886083, 869443, 713060, 911043, 800835, 907299, 673, 898659, 870340, 891171, 870467, 744132, 110051,
    912579, 663141, 276643, 912195, 750530, 542658, 545218, 486020, 398181, 676452, 662501, 764740, 712900, 869316, 712580, 180482, 647492, 908643, 712578, 870467, 776132, 869316, 486019, 486019, 890691, 879075, 576898, 869698, 7330, 7330, 859331, 903075,
    725570, 886563, 869698, 800836, 727236, 610978, 418, 72578, 721603, 764611, 713221, 883299, 882626, 100260, 918114, 878787, 900771, 767299, 662498, 719267, 907203, 898179, 858820, 676452, 713059, 712579, 911715, 663140, 461539, 910179, 712899, 835779,
    841284, 107172, 750530, 870467, 712899, 69667, 746563, 461378, 368578, 892035, 725762, 918114, 486020, 358978, 886563, 171842, 901635, 358627, 725570, 895587, 739908, 662980, 671330, 81187, 905859, 859331, 776132, 712899, 911043, 398181, 892035, 859331,
    870467, 895587, 898659, 542658, 858180, 712899, 912195, 725700, 712900, 870467, 803652, 755012, 545218, 663141, 542658, 795331, 107172, 276643, 910947, 276645, 662498, 713060, 662498, 542658, 869698, 712581, 713060, 542658, 279746, 461379, 513571, 723523,
    858051, 869316, 878787, 858051, 727236, 912386, 743108, 869698, 882626, 869570, 358627, 486020, 912579, 887234, 887234, 911715, 764610, 775620, 892707, 893859, 712900, 895011, 869699, 100260, 72578, 769219, 869698, 859331, 454178, 912099, 804164, 918114,
    891939, 398212, 461539, 908739, 461381, 910371, 535301, 858434, 81282, 81282, 789060, 869698, 870980, 180482, 870467, 662661, 700932, 666372, 666372, 81220, 139586, 892035, 910851, 886563, 870724, 107938, 869955, 0, 869700, 891939, 895587, 712900,
    3970, 906915, 713060, 746564, 662498, 857668, 81219, 461539, 486021, 358630, 910947, 908451, 662979, 713060, 879651, 697572, 893283, 495812, 244418, 723523, 918114, 907299, 700932, 892227, 712578, 795332, 895587, 764740, 662498, 90467, 502531, 767299,
    288386, 542661, 906339, 697572, 911715, 898659, 721604, 803652, 643172, 358630, 750530, 869572, 329282, 912195, 725762, 461378, 725570, 750530, 395811, 912386, 870467, 858051, 879651, 769220, 486020, 81220, 87587, 905283, 486019, 864580, 486018, 869316,
    545218, 409411, 858820, 739523, 764739, 676452, 817220, 910466, 542658, 542658, 886083, 857795, 869700, 908451, 139586, 695810, 910563, 527011, 870980, 276643, 395780, 764611, 910371, 495812, 542658, 727235, 712579, 887714, 908739, 870467, 893571, 542660,
    486020, 870980, 800836, 542658, 662498, 727236, 806467, 744132, 662500, 819396, 662498, 662498, 882339, 778435, 395780, 420098, 887331, 905667, 721604, 721604, 409411, 871108, 398181, 912195, 486018, 542658, 643172, 107938, 107938, 461378, 502531, 755012,
    690852, 767300, 87588, 750530, 883011, 764611, 398181, 712900, 662498, 869828, 100259, 107938, 857924, 869698, 668771, 858051, 81380, 797252, 662498, 870980, 918114, 171842, 893571, 912483, 912483, 870340, 461380, 764739, 764739, 663140, 857796, 610978,
    87587, 891171, 545218, 740323, 891939, 869698, 743108, 139586, 139586, 398181, 912771, 542661, 662501, 870467, 81380, 3970, 139586, 879651, 785092, 870980, 912386, 727235, 643172, 752196, 81187, 395781, 817219, 81219, 486020, 461539, 712578, 427298,
    662500, 883587, 461539, 750532, 897315, 870467, 857924, 821060, 918114, 769220, 643172, 662980, 869698, 910179, 668771, 886563, 7330, 918114, 910371, 910371, 755524, 486019, 871108, 486020, 869698, 663140, 712578, 100259, 100259, 81189, 486020, 409380,
    107938, 72578, 712899, 94114, 853572, 725570, 911043, 743779, 869570, 907491, 100260, 418, 100259, 723523, 81411, 180483, 891939, 180483, 671330, 764740, 910563, 910179, 870467, 486020, 579298, 898179, 81187, 81218, 81218, 542660, 797252, 697572,
    835779, 107938, 911715, 898659, 676452, 869572, 869572, 100260, 713060, 853572, 7330, 486018, 891939, 7330, 81187, 757827, 107938, 912483, 910371, 898659, 695810, 486018, 891939, 892611, 906339, 869700, 907203, 81221, 896739, 537858, 857924, 712581,
    647491, 859331, 888003, 690851, 795331, 90467, 107938, 398180, 545218, 139586, 663140, 81187, 725570, 910563, 486019, 869698, 712900, 918114, 905283, 819394, 795332, 858180, 461378, 486020, 139586, 317762, 911043, 713060, 721604, 662979, 869828, 912386,
    821060, 288386, 486019, 911043, 891939, 81221, 767299, 887331, 712899, 662500, 569698, 679811, 869828, 910563, 858180, 859331, 870467, 882819, 90467, 697572, 895587, 858180, 690852, 906723, 870212, 671330, 663141, 317762, 171842, 712578, 461378, 662498,
    888291, 888291, 679811, 640771, 908739, 454178, 409380, 750532, 904707, 486018, 860228, 890691, 81380, 662501, 69667, 662500, 406532, 898659, 869570, 767300, 888291, 870980, 910179, 857668, 100260, 750532, 486019, 513571, 912579, 883587, 803652, 727235,
    358627, 666372, 712578, 486018, 542660, 81411, 107172, 662500, 891171, 888003, 888003, 868932, 81219, 662498, 668771, 882819, 643172, 911715, 81218, 486021, 81218, 81220, 461539, 903075, 869698, 888291, 513572, 461539, 461539, 882435, 815299, 495811,
    750530, 358630, 537858, 795332, 398180, 744132, 542658, 910755, 857796, 276646, 764611, 700932, 461380, 870082, 288386, 894339, 545218, 542661, 819394, 896739, 740292, 288386, 171842, 81187, 690852, 725700, 545218, 892035, 713059, 712899, 276643, 712901,
    815299, 576898, 839748, 100260, 864580, 643172, 358627, 662979, 712578, 857924, 398212, 486020, 744132, 486018, 100259, 662501, 910851, 869700, 887234, 910947, 870467, 857668, 912579, 744132, 247298, 276643, 81219, 180483, 713060, 912675, 911907, 662980,
    712578, 409380, 750532, 764610, 767299, 576898, 910466, 857668, 906915, 800835, 895587, 857668, 712580, 907395, 764611, 712900, 910179, 893859, 110051, 860100, 757827, 398181, 690852, 817220, 139586, 398181, 288386, 892227, 892227, 750532, 81411, 486018,
    905283, 712581, 910179, 841284, 907299, 358627, 700932, 870084, 764739, 662661, 719267, 398212, 662500, 461378, 895587, 893859, 888291, 81187, 712578, 663141, 912386, 666371, 486018, 886083, 495812, 545218, 486020, 81187, 869828, 317762, 276643, 486021,
    690851, 800835, 843972, 869699, 712578, 81411, 662980, 912579, 869572, 398181, 767299, 139586, 912579, 869698, 857795, 247298, 100259, 894339, 764740, 81411, 746564, 486018, 288387, 366178, 647492, 398181, 81187, 358628, 461539, 486018, 891939, 486019,
    857796, 870212, 486020, 662501, 883587, 869698, 725700, 676452, 663139, 800835, 542658, 69667, 817220, 869699, 908739, 908643, 819396, 662498, 868932, 727235, 743779, 486018, 662500, 610978, 841284, 869698, 888291, 513571, 869699, 904707, 857796, 100259,
    662498, 819394, 870340, 461539, 409380, 486018, 180483, 244418, 748484, 81187, 454178, 819396, 911715, 870467, 81221, 662500, 894339, 894339, 276866, 545218, 542658, 744260, 545218, 899331, 81411, 743170, 545218, 870467, 888291, 427298, 869828, 542658,
    887331, 870980, 908355, 542660, 486021, 81187, 857796, 358627, 545218, 100259, 542658, 358627, 398180, 891939, 673, 288386, 406532, 903075, 107938, 803652, 893571, 785092, 712580, 461539, 139586, 869316, 800836, 869700, 486020, 869699, 911715, 869700,
    712581, 843972, 366178, 395780, 795331, 139586, 276646, 663139, 910371, 744132, 857924, 857922, 739396, 892035, 107938, 897315, 542658, 744260, 804164, 746564, 892227, 100259, 795331, 795331, 358627, 894339, 542658, 358978, 835780, 107938, 800836, 461381,
    662500, 882626, 81220, 869443, 107938, 869698, 662500, 743170, 869570, 740292, 81187, 662498, 892611, 869698, 892227, 712578, 869570, 72578, 107938, 180483, 795332, 535300, 860228, 712578, 406594, 869698, 662498, 81188, 400739, 857795, 535300, 513571,
    535300, 513571, 81189, 662979, 461379, 869699, 869443, 910563, 107938, 662500, 869316, 7330, 454178, 910466, 690851, 495811, 743170, 81218, 695810, 81220, 869570, 643171, 712578, 869316, 81188, 663139, 662980, 395811, 81187, 869698, 673, 107938,
    743748, 81220, 870467, 713061, 882626, 368578, 910466, 662498, 368578, 368578, 910275, 139586, 662500, 400739, 279746, 400739, 279746, 713221, 288387, 288387, 542658, 870082, 107938, 542658, 461539, 883299, 673, 671330, 486020, 713059, 576898, 576898,
    911043, 576898, 744258, 100260, 864452, 663141, 883299, 662981, 869570, 910947, 276643, 739395, 892227, 853700, 662499, 906339, 288386, 857796, 110051, 542660, 869698, 887715, 870467, 7330, 739395, 409411, 746563, 486018, 884739, 713060, 110051, 90467,
    743108, 727235, 695810, 495811, 662981, 869699, 750660, 892035, 81411, 744260, 869698, 746563, 427298, 486021, 542660, 727235, 427298, 486020, 486020, 276643, 743170, 755524, 358627, 869699, 857796, 869698, 662979, 857668, 171842, 869316, 910947, 750532,
    81411, 81411, 461378, 857668, 486018, 869698, 743170, 81411, 755524, 869698, 869316, 910466, 461380, 579298, 898659, 898659, 406594, 891939, 427298, 107938, 725570, 725570, 725570, 645730, 645730, 725570, 395780, 898179, 740323, 882339, 645730, 895011,
    486019, 712900, 878275, 579298, 910179, 910947, 887234, 858180, 712580, 139586, 579298, 740323, 906339, 886083, 662979, 690851, 905859, 139586, 886563, 882819, 725570, 666371, 666371, 898179, 878275, 276643, 276643, 910179, 72578, 486020, 317762, 666371,
    545218, 579298, 898179, 700932, 858180, 395781, 247298, 545218, 712900, 907587, 712900, 817220, 545218, 857922, 884739, 859331, 750530, 821059, 892227, 815300, 662498, 906051, 901635, 662498, 883587, 712580, 81189, 764739, 100260, 902403, 727235, 918114,
    663140, 755012, 857796, 662498, 869699, 712578, 739939, 100260, 81282, 3970, 171842, 857796, 3970, 755012, 746563, 545218, 857796, 100260, 110051, 110051, 859972, 859972, 713061, 727236, 107172, 746564, 859972, 537858, 610978, 704291, 764611, 461539,
    697572, 870467, 906339, 727235, 870082, 819396, 744260, 81411, 750660, 395811, 69667, 870980, 81189, 87587, 764611, 417698, 712578, 910179, 815300, 906339, 662498, 662980, 647491, 395811, 662500, 910466, 910466, 910947, 671330, 858434, 775620, 486021,
    400739, 910466, 329282, 892035, 690852, 870467, 870467, 662500, 819394, 461378, 87588, 495812, 495812, 907299, 502531, 405892, 908643, 857924, 486018, 725572, 725572, 767299, 90467, 713221, 764612, 712741, 892227, 81218, 903651, 662498, 764739, 869570,
    869570, 857795, 869698, 94114, 110051, 869828, 395812, 100259, 398180, 398180, 746563, 911715, 757827, 81218, 743779, 662498, 579298, 613538, 898659, 417698, 819396, 819394, 405861, 640772, 725570, 662979, 870467, 905859, 640771, 870724, 869954, 890691,
    895587, 898659, 870082, 764740, 743170, 542658, 806467, 739396, 764740, 910371, 81218, 576898, 898659, 276643, 139586, 869954, 910179, 910179, 910179, 72578, 542658, 819394, 869698, 764610, 887331, 911715, 864452, 870467, 870340, 461378, 712900, 100259,
    671330, 535301, 486181, 139586, 910563, 906339, 869698, 486019, 713221, 857924, 244418, 81411, 819394, 723523, 870082, 358627, 537858, 911715, 461378, 405892, 767299, 764610, 898179, 671330, 81218, 647492, 882626, 764611, 870467, 288386, 841284, 398181,
    800836, 905763, 366178, 662498, 671330, 610978, 139586, 276866, 358627, 663141, 894339, 869572, 892227, 918114, 640771, 821059, 908643, 100260, 486018, 107172, 461380, 723524, 910851, 803652, 545218, 746563, 897315, 910563, 244418, 870082, 495812, 869699,
    640771, 891939, 7330, 81411, 81411, 912099, 398181, 869699, 910371, 743139, 486020, 100260, 895011, 671330, 486018, 887234, 869444, 918114, 276866, 279746, 870467, 869699, 910466, 712578, 892707, 358627, 486021, 897315, 897315, 81282, 755012, 695810,
    900771, 894339, 869698, 819394, 857924, 495811, 244418, 910371, 869316, 910563, 405892, 869570, 288386, 535301, 908739, 906243, 139586, 887234, 870596, 858434, 647492, 910947, 712741, 870467, 461381, 81220, 821060, 743779, 610, 662499, 905667, 910466,
    857922, 895587, 869698, 486019, 495812, 662500, 486021, 785092, 713060, 859331, 870980, 739522, 764610, 81411, 486018, 895587, 395780, 409411, 81218, 843972, 910947, 495811, 895011, 906339, 910371, 713059, 502531, 395811, 721604, 815300, 870980, 911715,
    712578, 366178, 886083, 81220, 405892, 869698, 247298, 869700, 795331, 740323, 795332, 395811, 542658, 769220, 918114, 910466, 542660, 409380, 764740, 869698, 107938, 700932, 893859, 739939, 495811, 486019, 405861, 486181, 898659, 288387, 398181, 905859,
    712580, 663140, 329282, 527011, 647492, 903075, 395811, 495812, 869316, 663139, 610980, 329282, 107172, 869316, 712580, 697572, 673, 288386, 704291, 789060, 739523, 695810, 542658, 859972, 671330, 894339, 859972, 908739, 427298, 893859, 893571, 542660,
    910371, 94114, 110051, 81220, 750530, 398212, 907395, 486020, 712901, 279746, 918114, 461380, 869444, 859844, 276645, 3970, 743748, 712901, 7330, 523652, 94114, 395811, 910179, 755012, 72578, 94114, 139586, 663139, 542661, 409411, 317762, 870980,
    750532, 906243, 420098, 789060, 912386, 712578, 869316, 662498, 461378, 891171, 100259, 800835, 690852, 870596, 643172, 740323, 395781, 276643, 662498, 764612, 870467, 545218, 901923, 139586, 908739, 744132, 894339, 713059, 647492, 887331, 676452, 864452,
    645730, 746563, 107938, 610978, 107938, 545218, 767299, 819394, 495812, 800836, 806467, 869699, 882819, 857922, 406532, 107938, 610980, 663140, 725700, 409380, 81218, 869828, 893571, 409411, 857796, 100259, 662500, 864452, 819524, 870467, 869699, 366178,
    723523, 869828, 90467, 907491, 887331, 857796, 869828, 171842, 486018, 663139, 869828, 279746, 545218, 912386, 427298, 869698, 486020, 819396, 905667, 712578, 107938, 869828, 663140, 663140, 907203, 81220, 358627, 486020, 844484, 107938, 906339, 697572,
    857922, 427298, 764611, 712899, 366178, 94114, 94114, 750530, 910371, 90467, 858434, 910563, 739522, 910371, 72578, 767300, 100259, 405892, 869572, 819394, 171842, 898179, 869700, 100259, 911715, 405892, 819524, 663141, 898659, 912099, 409411, 901635,
    461539, 400739, 673, 905667, 898659, 542660, 662498, 662498, 898659, 864452, 882723, 910371, 739395, 868932, 819396, 81411, 576898, 869570, 486018, 643171, 486020, 486020, 81218, 764740, 712581, 869698, 869699, 712900, 859331, 891939, 398181, 81221,
    690851, 107938, 725570, 821059, 859331, 329282, 663141, 107172, 139586, 755012, 868932, 785092, 785092, 906339, 100259, 690852, 803652, 743779, 869828, 835779, 721603, 911715, 857924, 358630, 542661, 395780, 712580, 409380, 903075, 912386, 87587, 859331,
    815299, 869700, 418, 180483, 764740, 486020, 869570, 896739, 910371, 888003, 366178, 869700, 244418, 495811, 100259, 690852, 869828, 908451, 869828, 542660, 911043, 817220, 870852, 398181, 171842, 908643, 317762, 610, 895587, 890691, 907587, 821060,
    712899, 859331, 869698, 806467, 569698, 247298, 910371, 542658, 912099, 910466, 712578, 90467, 662501, 870082, 859331, 288386, 406594, 857795, 908355, 110051, 910755, 869699, 858180, 409411, 288386, 908451, 610978, 317762, 859331, 858434, 406594, 640771,
    81187, 81187, 640772, 90468, 739396, 899331, 454178, 750532, 740323, 81221, 712578, 912386, 750530, 870467, 869444, 918114, 891939, 81219, 454178, 912386, 908739, 693252, 288387, 725570, 785092, 81411, 778435, 869699, 888291, 803652, 870467, 100259,
    663139, 815299, 869700, 409380, 545218, 795331, 804164, 911043, 727235, 690851, 107938, 910755, 81218, 725570, 725700, 870467, 486021, 893187, 663140, 662981, 750530, 819394, 725762, 545218, 81218, 81218, 911715, 395780, 764611, 107172, 778435, 815299,
    857796, 107938, 853700, 486018, 3970, 740292, 662498, 712578, 486018, 545218, 896739, 803652, 495811, 891171, 744132, 896739, 870467, 888003, 662501, 542658, 100259, 882435, 785092, 744132, 409380, 868932, 743170, 693252, 7330, 712578, 819394, 690852,
    576898, 662981, 395811, 542660, 795331, 610, 400739, 767299, 764740, 869572, 869698, 815299, 912386, 815300, 513572, 897315, 643172, 276643, 712901, 712901, 358630, 869570, 81220, 743779, 486020, 100259, 276646, 799300, 882819, 695810, 100260, 486021,
    744132, 869700, 405892, 910755, 276643, 461380, 869700, 869316, 871108, 527011, 869570, 542660, 870467, 910371, 645730, 107938, 785092, 244418, 746564, 81411, 891939, 910466, 235778, 486018, 640771, 910371, 288386, 662500, 662500, 712578, 545218, 712578,
    910947, 757827, 81221, 821059, 907395, 288386, 893859, 713060, 744132, 869698, 713059, 87587, 662980, 690851, 740292, 910947, 180483, 7330, 858820, 329282, 358627, 276866, 858979, 725570, 662980, 395780, 395780, 276643, 700932, 247298, 800835, 910371,
    279746, 279746, 398181, 486181, 398181, 81219, 785092, 893859, 72547, 767299, 882915, 405892, 671330, 276643, 643172, 395811, 110051, 895587, 405892, 918114, 740323, 72578, 712580, 486020, 662980, 712578, 107938, 545218, 795331, 841284, 107938, 800835,
    819396, 912195, 746564, 81220, 870082, 910563, 107938, 542658, 859331, 910851, 800836, 87587, 912386, 107172, 81411, 739395, 542660, 750532, 7330, 725762, 81220, 713060, 395780, 815299, 746563, 746563, 869698, 288387, 645730, 139586, 317762, 857795,
    486018, 870980, 542660, 843972, 891171, 495811, 764739, 279746, 486018, 288387, 81411, 81187, 317762, 676452, 752196, 279746, 358627, 893571, 727235, 542660, 819394, 859331, 276643, 819394, 841284, 869700, 869700, 817220, 495811, 662498, 725570, 486018,
    740323, 858434, 882819, 94114, 908739, 869699, 869699, 427299, 882626, 743779, 857796, 795331, 502531, 406594, 247298, 870980, 910371, 727235, 461378, 662500, 869700, 727235, 893283, 887331, 739396, 572098, 486020, 910371, 897315, 869955, 545218, 542658,
    911715, 100260, 545218, 671330, 870467, 869700, 427298, 87588, 81411, 910563, 640771, 663140, 725700, 395780, 576898, 895587, 486020, 247298, 279746, 398181, 461378, 727235, 743779, 712580, 545218, 857795, 395811, 869700, 858180, 542658, 495812, 869316,
    869700, 288387, 910371, 910563, 406532, 910851, 358627, 803652, 893571, 427299, 764611, 81218, 893571, 276643, 800836, 869700, 819394, 486018, 542660, 486020, 869700, 764739, 800835, 869698, 486020, 911907, 395780, 883299, 719268, 486020, 795331, 764611,
    276646, 869828, 870980, 764740, 663139, 894339, 795331, 744132, 857922, 107172, 662500, 764610, 461380, 892227, 100259, 740323, 869700, 276866, 869955, 804164, 100259, 542658, 819394, 662980, 690851, 746564, 785092, 395780, 869700, 740323, 358978, 358627,
    795331, 795331, 835780, 87587, 804164, 910947, 712580, 100259, 767299, 100259, 107938, 743779, 725700, 899331, 461381, 748484, 81411, 869316, 869316, 81411, 405892, 800836, 912195, 358629, 892035, 712899, 910563, 400739, 870980, 912099, 912099, 739395,
    100260, 662498, 713221, 764739, 903651, 486181, 90467, 247298, 869698, 545218, 81380, 645730, 892611, 640771, 100259, 892611, 640772, 579298, 740292, 799300, 461539, 513572, 757827, 461381, 869443, 870467, 461381, 461381, 910371, 662501, 725572, 417698,
    712579, 713061, 358627, 171842, 870340, 918114, 358978, 910563, 910563, 800836, 870980, 869572, 139586, 690851, 905763, 486181, 905763, 908643, 908643, 727236, 100260, 700932, 662980, 358627, 712899, 750532, 910371, 859331, 542660, 400739, 81221, 870596,
    81218, 860228, 795331, 697572, 785092, 869570, 398212, 857924, 895011, 81220, 100260, 910755, 910755, 712580, 869699, 910466, 870596, 911043, 859331, 910371, 700932, 899331, 912099, 892611, 405892, 893859, 712579, 358627, 869572, 713061, 244418, 100259,
    81411, 461381, 72547, 81411, 892707, 139586, 764611, 858820, 7330, 81411, 0, 690852, 495811, 858050, 895011, 712578, 918114, 486181, 910563, 247298, 697572, 870980, 697572, 87587, 739939, 643172, 795332, 100259, 869570, 764740, 400739, 858180,
    81218, 712578, 276643, 72578, 910947, 180482, 789060, 869316, 712578, 100259, 395811, 358630, 502531, 750530, 910563, 486018, 910466, 100259, 545218, 395811, 100259, 81220, 817219, 139586, 883587, 879651, 764739, 800835, 527011, 409411, 704291, 712578,
    882819, 910466, 405892, 740292, 918114, 94114, 785092, 744132, 764740, 647492, 662661, 907491, 712578, 662661, 869443, 740292, 100259, 100259, 869698, 892227, 107938, 767300, 719267, 645730, 712741, 767300, 663141, 395781, 858820, 712741, 610980, 769220,
    910371, 398180, 72578, 180483, 713221, 755524, 712579, 81411, 764740, 542660, 869572, 869444, 81221, 785092, 486020, 725570, 869700, 358627, 910371, 868932, 662980, 712578, 844484, 844484, 395780, 100259, 486181, 486181, 107938, 898659, 662979, 910371,
    712899, 495811, 662661, 907203, 898659, 81220, 869570, 870340, 910371, 764611, 906339, 288387, 100260, 895011, 358627, 891939, 912195, 461379, 879075, 81218, 110051, 835779, 859331, 542661, 725572, 890691, 329283, 690852, 740292, 662980, 3138, 366178,
    81221, 910563, 712578, 910466, 712578, 110051, 72578, 72578, 712578, 110051, 712741, 901635, 662980, 662661, 911043, 910947, 819394, 542660, 869700, 910466, 858180, 775620, 100259, 662501, 895587, 712578, 725570, 725700, 744132, 545218, 486020, 276643,
    869698, 896739, 643172, 288387, 869698, 693252, 892227, 871108, 110051, 81218, 869698, 870467, 725762, 910371, 895587, 395811, 743170, 858050, 486181, 815299, 527011, 662981, 744260, 870084, 486020, 712578, 712578, 712578, 869698, 869316, 400739, 81221,
    395780, 7330, 690851, 719267, 100259, 910947, 857668, 502531, 748484, 757827, 663140, 785092, 764739, 671330, 799300, 107938, 800835, 700932, 858820, 764739, 764611, 100259, 910947, 795331, 898659, 690852, 869698, 486181, 817219, 495812, 785092, 869700,
    107938, 869698, 910851, 795331, 908643, 912195, 739395, 288387, 712578, 906723, 871108, 542660, 81411, 358627, 910563, 800836, 910371, 869698, 910563, 870340, 81411, 417698, 843972, 795331, 858434, 712580, 72578, 910947, 495811, 743108, 883587, 911043,
    72578, 795331, 725700, 858180, 542658, 576898, 879075, 907491, 545218, 418, 725762, 910947, 535301, 860228, 799300, 542658, 545218, 892227, 799300, 100259, 712578, 910563, 803652, 7330, 739395, 719268, 911907, 795331, 744132, 892227, 405861, 461380,
    139586, 72547, 819394, 690852, 795331, 358627, 795331, 800836, 744260, 107938, 870340, 870340, 235778, 713061, 905763, 398181, 869828, 406594, 910371, 395811, 892227, 892227, 918114, 7330, 358978, 884739, 7330, 911043, 871234, 276646, 887234, 640771,
    81190, 171842, 869956, 884739, 662501, 107172, 662501, 853572, 721603, 640771, 740292, 486020, 905667, 427298, 81190, 107938, 712899, 725572, 884739, 721604, 366178, 81190, 911715, 911715, 427298, 3970, 398181, 903651, 903651, 647491, 892035, 900771,
    317762, 579298, 893859, 858051, 815300, 764612, 898179, 579298, 767300, 910466, 100259, 454178, 879651, 671330, 486020, 461378, 911715, 454178, 663141, 579298, 750530, 662500, 857924, 857924, 912386, 320642, 712578, 815300, 515972, 662500, 910466, 461378,
    712578, 542658, 542658, 906147, 486018, 767300, 870467, 723523, 857924, 409380, 427298, 139586, 486019, 486020, 764740, 853572, 171842, 139586, 892707, 81187, 712578, 879651, 897315, 891939, 569698, 666371, 910371, 898179, 757827, 700932, 857668, 486021,
    712578, 542658, 427298, 895011, 905667, 853700, 366178, 712578, 743108, 910371, 180483, 743108, 857924, 857924, 358978, 764611, 906339, 486018, 739908, 870980, 81282, 454178, 358630, 461381, 569698, 911715, 513572, 366178, 743139, 3970, 869699, 81220,
    358630, 110051, 535301, 903651, 750530, 535301, 713221, 579298, 879651, 645730, 139586, 869698, 663141, 905763, 712579, 910179, 81218, 800836, 887331, 645730, 610978, 911043, 911715, 890691, 910851, 139586, 746563, 366178, 180482, 712580, 841284, 81220,
    398211, 785092, 72578, 870340, 486018, 910947, 662980, 317762, 858180, 859844, 700932, 893859, 427298, 712579, 859331, 793156, 713061, 107938, 679811, 107938, 725570, 815300, 712578, 789060, 542658, 712579, 869699, 864324, 795332, 879651, 870467, 100259,
    662979, 542658, 395811, 712580, 495812, 905667, 887331, 869316, 878787, 662498, 610978, 725700, 869699, 893571, 895011, 486020, 662501, 527011, 100259, 891171, 871108, 819394, 395781, 869698, 870467, 713061, 3970, 817220, 882339, 744258, 663139, 895011,
    750532, 897315, 81218, 871108, 461539, 910371, 764740, 94114, 869828, 869444, 358627, 898179, 910371, 81189, 712580, 868932, 3970, 901923, 910563, 486019, 743139, 725572, 911715, 911715, 898659, 69667, 911043, 679811, 366178, 821060, 81219, 869828,
    908355, 569698, 869828, 883299, 107938, 859331, 740323, 662501, 869698, 713060, 739396, 740323, 815299, 495812, 486021, 576898, 712901, 395780, 725762, 725762, 740323, 882819, 723523, 892227, 725570, 725570, 81221, 247298, 800835, 869570, 700932, 911043,
    799300, 860228, 869316, 859844, 757827, 785092, 454178, 789060, 427298, 81411, 793156, 712580, 910563, 317762, 712580, 910947, 841284, 725572, 743779, 904707, 817220, 895011, 901923, 461381, 859844, 545218, 317762, 712580, 819394, 719268, 81220, 901923,
    897315, 712580, 662498, 892227, 869572, 819524, 912386, 545218, 545218, 107938, 288386, 87588, 906339, 451778, 752196, 537858, 358978, 486018, 405892, 750530, 81189, 767299, 892035, 107938, 662498, 107938, 395781, 569698, 405892, 910466, 918114, 90467,
    891939, 3970, 727235, 725572, 721604, 911907, 892227, 817219, 892227, 87587, 486020, 898659, 746563, 647491, 910371, 81220, 712580, 662501, 800836, 883299, 869698, 81282, 647492, 750532, 918114, 869572, 712578, 366178, 882626, 890691, 486019, 858820,
    244418, 882626, 171842, 719267, 663141, 663140, 869698, 871234, 486019, 821059, 662980, 895587, 725570, 713060, 713060, 892227, 764610, 803652, 495811, 785092, 486020, 542658, 461381, 712580, 870596, 775620, 100259, 910755, 100260, 869570, 897315, 288386,
    358627, 857795, 869699, 910947, 879075, 887234, 712581, 911043, 712579, 395812, 461539, 405861, 869570, 721603, 905859, 893859, 746564, 869700, 81411, 90467, 673, 398181, 750530, 910466, 910466, 398181, 502531, 898659, 879651, 697572, 712580, 81187,
    395811, 869572, 100259, 906339, 869572, 795332, 542658, 910755, 918114, 895587, 3970, 712578, 800836, 542658, 495811, 542660, 908451, 908451, 764740, 87587, 750532, 647491, 712900, 279746, 821059, 107938, 139586, 398180, 893571, 821059, 663141, 910371,
    864452, 81187, 912771, 723524, 398181, 81218, 662498, 673, 527011, 395781, 662498, 723524, 87587, 545218, 744132, 461380, 100259, 750660, 879651, 800835, 857795, 645730, 235778, 906723, 870980, 857795, 764610, 495811, 662661, 427298, 891939, 409411,
    107938, 898179, 898659, 409380, 869572, 695810, 869444, 883299, 907203, 821059, 486020, 910371, 858436, 81220, 910563, 81189, 486020, 358629, 395781, 542660, 857924, 712579, 755524, 869828, 3970, 901923, 81218, 869700, 107938, 81221, 907299, 279746,
    895587, 859331, 858180, 908451, 911043, 90467, 910466, 713060, 495811, 869700, 569698, 908739, 895587, 690852, 542658, 795332, 461539, 713060, 662501, 883011, 892227, 870340, 911043, 406594, 727235, 276643, 395781, 527011, 358629, 744132, 495811, 869698,
    882819, 744132, 662498, 542658, 495812, 908451, 906339, 869572, 725762, 542660, 817219, 712901, 869699, 7330, 869700, 910371, 545218, 910466, 748484, 910755, 870084, 910947, 857922, 746564, 800835, 799300, 662661, 288386, 81221, 461539, 247298, 495812,
    764611, 495811, 895587, 486018, 81411, 495812, 406594, 725570, 739395, 857922, 912195, 746564, 785092, 778435, 690851, 81411, 81220, 279746, 288387, 864452, 81187, 107938, 495811, 545218, 727235, 276643, 90467, 912579, 795331, 712578, 912386, 869572,
    858436, 94114, 888003, 894339, 427298, 542658, 870467, 907203, 542658, 869955, 662661, 695810, 461539, 288387, 418, 803652, 545218, 542658, 869955, 795331, 869700, 819394, 712581, 662661, 461380, 883587, 869955, 81220, 461379, 366178, 884739, 366178,
    81218, 870467, 853700, 721603, 719268, 542658, 610, 0, 712581, 662500, 721603, 643171, 764612, 746563, 90468, 743170, 0, 0, 610, 81218, 800836, 869698, 647492, 882626, 878787, 789060, 395781, 400739, 0, 746564, 817220, 368578,
    870724, 647492, 725572, 870596, 713060, 0, 882626, 869700, 662498, 712580, 896739, 750530, 743170, 817220, 358629, 723524, 800835, 870084, 910466, 645730, 746563, 90468, 869700, 100259, 767299, 883299, 663140, 864452, 864452, 395811, 395781, 663141,
    857796, 663140, 395780, 395780, 907299, 663141, 288386, 663140, 107172, 663461, 545218, 918114, 107172, 723524, 764740, 841284, 864452, 911715, 486019, 663140, 712899, 750532, 918114, 835779, 750532, 647492, 697572, 81219, 697572, 713060, 912099, 817220,
    725570, 725570, 895587, 910947, 895587, 395781, 579298, 110051, 764611, 918114, 769220, 864452, 891939, 891939, 857924, 910371, 918114, 864452, 579298, 886563, 640771, 486021, 610, 417698, 835779, 895587, 723523, 713060, 81187, 579298, 579298, 739522,
    864452, 888003, 725570, 417698, 870467, 896931, 662498, 896931, 640771, 869698, 764611, 81219, 663139, 107938, 579298, 502531, 869698, 895587, 663139, 884739, 743108, 81219, 746564, 870724, 486018, 87587, 486021, 398181, 395781, 704291, 894339, 486018,
    69667, 895587, 884739, 868932, 800835, 486021, 486021, 743108, 645730, 884739, 884739, 645730, 891939, 746563, 276643, 276643, 870467, 869698, 606178, 901635, 752196, 725570, 891939, 799300, 767300, 454178, 892035, 870596, 712579, 662980, 869570, 904899,
    542658, 100260, 398180, 911715, 358978, 891939, 892227, 662498, 712579, 882819, 725570, 869698, 542658, 871108, 81190, 764740, 358627, 905763, 645730, 767299, 171842, 882626, 910851, 870467, 746563, 713060, 764610, 870467, 693252, 139586, 486018, 887331,
    139586, 662980, 486019, 869572, 712899, 727236, 903075, 713061, 107172, 495812, 857924, 693252, 700932, 869698, 891939, 910371, 81411, 690851, 100260, 400739, 870467, 398211, 3970, 898659, 886083, 495812, 869316, 700932, 329282, 815300, 645730, 869316,
    864324, 918114, 542658, 870467, 883011, 886563, 910466, 662979, 893859, 859844, 81220, 712580, 869698, 723524, 712901, 461380, 690851, 887331, 569698, 800835, 400739, 486018, 864452, 81282, 400739, 725570, 725570, 821059, 461378, 427299, 673, 910371,
    107938, 882819, 3970, 358627, 744258, 746563, 663140, 713221, 400739, 107938, 743139, 764740, 418, 100260, 398180, 329282, 107938, 891171, 90467, 81220, 180482, 486019, 869699, 869698, 896739, 907587, 892227, 400739, 712578, 461380, 869698, 90467,
    486018, 81190, 721603, 81411, 895587, 739908, 859844, 904899, 750532, 417698, 880419, 690851, 712901, 725700, 663140, 744258, 897315, 869698, 871108, 743108, 712901, 870212, 908355, 279746, 906723, 750660, 896739, 896739, 576898, 857795, 869699, 869700,
    81221, 918114, 799300, 803652, 857668, 895587, 893859, 712580, 887331, 746563, 869699, 72578, 406594, 721603, 712580, 461380, 461539, 843972, 107938, 910371, 869698, 107172, 645730, 894339, 542658, 276643, 746563, 800835, 750532, 800835, 893571, 880419,
    486018, 400739, 100260, 869572, 891939, 713061, 572098, 542658, 743779, 893571, 486018, 750532, 869572, 869700, 869699, 857922, 107172, 690851, 750532, 897315, 804164, 804164, 366178, 366178, 461539, 358978, 764610, 748484, 764610, 878787, 878787, 906243,
    486018, 912195, 912195, 898179, 420098, 235778, 891939, 81221, 461539, 882626, 800836, 888291, 576898, 486018, 785092, 643172, 888291, 406594, 673, 740323, 395811, 725570, 87587, 743108, 662500, 712578, 406594, 869316, 740323, 81221, 740323, 870082,
    882819, 740323, 869316, 7330, 244418, 740323, 785092, 406594, 869444, 81221, 542658, 853700, 72578, 785092, 645730, 767299, 905763, 870596, 502531, 906339, 882915, 906723, 712899, 461378, 502531, 502531, 870596, 712899, 461378, 870340, 427298, 821059,
    662498, 869700, 107938, 879651, 870340, 713221, 643171, 72578, 869700, 72578, 869698, 366178, 87587, 72578, 81283, 712900, 100259, 486020, 72578, 107938, 180483, 764611, 888003, 883587, 640771, 918114, 906339, 663141, 883587, 545218, 712900, 817220,
    542658, 495811, 869570, 486020, 640771, 461380, 81282, 545218, 869570, 870596, 746564, 81282, 461378, 542658, 579298, 752196, 767299, 869570, 815300, 87588, 870467, 662498, 869828, 864452, 712581, 757827, 869443, 712578, 640771, 764740, 757827, 821059,
    821060, 663141, 859331, 110051, 819396, 640771, 764612, 819524, 746563, 898659, 898179, 513571, 712580, 893187, 859331, 100260, 713061, 911043, 663140, 743779, 417698, 713060, 535301, 139586, 663140, 869698, 663141, 898659, 576898, 358627, 713060, 713060,
    819396, 859331, 869828, 884739, 610978, 912386, 912386, 740292, 712899, 869572, 727236, 819394, 803652, 288387, 700932, 819394, 884739, 750530, 606178, 821060, 869572, 400739, 535301, 789060, 900771, 713060, 864580, 100260, 100259, 576898, 713221, 576898,
    713059, 713061, 898659, 908451, 858820, 395811, 81220, 750530, 398181, 883011, 789060, 673, 815299, 712580, 712578, 527011, 893571, 486020, 662980, 864452, 778435, 87587, 797252, 859331, 662661, 895011, 3970, 495812, 713061, 542658, 893571, 288387,
    789060, 859972, 288387, 427298, 725570, 645730, 579298, 409411, 610978, 545218, 869698, 891939, 882339, 870596, 107938, 409411, 569698, 712579, 869444, 81411, 778435, 912099, 409380, 486020, 898659, 427298, 743139, 869572, 864452, 180483, 888291, 907299,
    898179, 869570, 764610, 94114, 819524, 743108, 180483, 819394, 90468, 893187, 746564, 398180, 171842, 908355, 712579, 662661, 576898, 821060, 815299, 859972, 870082, 712901, 869698, 750660, 893187, 576898, 860100, 744132, 896739, 400739, 815299, 853700,
    398181, 461539, 817220, 87587, 907395, 906051, 712578, 857668, 247298, 358627, 748484, 663140, 860100, 799300, 72547, 712580, 513571, 7330, 869570, 700932, 900771, 800835, 461539, 486020, 7330, 406594, 81411, 712900, 486020, 461378, 870340, 857922,
    898659, 743108, 907587, 841284, 910947, 911043, 743779, 817220, 819396, 647491, 819396, 894339, 535301, 892227, 869572, 819396, 743779, 545218, 817220, 803652, 461380, 815299, 712578, 81187, 690852, 513571, 358627, 898179, 495811, 495811, 107938, 860228,
    870082, 892227, 859331, 869698, 663140, 870467, 743139, 663140, 869700, 139586, 870596, 785092, 883011, 755524, 406594, 785092, 870084, 911907, 358978, 859331, 752196, 662498, 486018, 912386, 767299, 910466, 713059, 180482, 767300, 712578, 764611, 535301,
    746564, 569698, 662979, 746563, 892227, 542658, 918114, 821059, 673, 743779, 513572, 870467, 662498, 513571, 891939, 727235, 673, 3970, 645730, 713059, 100259, 288387, 358629, 72547, 757827, 610, 100259, 495811, 764740, 764610, 576898, 712899,
    139586, 107938, 171842, 910179, 910179, 750530, 870212, 713060, 673, 918114, 579298, 358627, 418, 725572, 613538, 647492, 643172, 800836, 905763, 537858, 244418, 366178, 244418, 663140, 869954, 406594, 542658, 803652, 700932, 107172, 542658, 535301,
    81187, 515972, 857924, 358627, 535301, 461539, 673, 901923, 662499, 545218, 910371, 406594, 454178, 712580, 869444, 100259, 755012, 643172, 712578, 893859, 405892, 860228, 712579, 857668, 857796, 910371, 857668, 869698, 515972, 869316, 906723, 395780,
    795332, 405892, 107938, 869316, 712579, 815300, 897315, 107172, 72547, 72547, 94114, 895587, 869700, 815300, 906723, 100259, 662979, 527011, 107938, 400739, 81219, 545218, 398180, 486020, 904899, 676452, 870980, 515972, 643172, 100259, 870724, 523652,
    398181, 662498, 871108, 857796, 800835, 502531, 366178, 752196, 821059, 750660, 486020, 72578, 545218, 406594, 288387, 906723, 797252, 879651, 869699, 427299, 910466, 662498, 806467, 87587, 750532, 901923, 879651, 662498, 398180, 858436, 100259, 884739,
    752196, 427298, 910179, 891939, 869700, 725572, 171842, 725570, 910371, 860228, 713060, 755524, 712579, 746564, 695810, 901923, 643172, 486020, 750530, 690851, 69667, 864452, 662498, 757827, 764740, 869572, 906723, 906723, 569698, 775620, 610980, 288386,
    673, 882626, 895587, 671330, 690852, 721604, 81219, 645730, 910371, 666372, 894339, 883011, 864452, 712579, 712578, 542660, 640771, 910371, 110051, 427299, 713059, 171842, 418, 358629, 911043, 778435, 358629, 395780, 662498, 725762, 643172, 895587,
    545218, 513571, 81190, 398180, 869699, 894339, 725700, 662498, 576898, 662498, 882819, 871108, 857796, 743748, 806467, 486018, 662980, 893859, 857922, 870084, 748484, 486020, 712578, 800835, 752196, 690851, 81221, 640771, 869954, 857668, 750530, 542658,
    486020, 785092, 712578, 712901, 857668, 757827, 486020, 107938, 870724, 72578, 171842, 870082, 894339, 891939, 81187, 7330, 906723, 746564, 869698, 486021, 645730, 645730, 107938, 647491, 673, 406594, 171842, 486020, 107938, 858436, 817220, 905763,
    882626, 366178, 100260, 569698, 739394, 712580, 535301, 894339, 894339, 870467, 542658, 542658, 764611, 906723, 905763, 869700, 542658, 803652, 803652, 576898, 366178, 276646, 461380, 795331, 739394, 100259, 859331, 859331, 542658, 858052, 663139, 911907,
    671330, 662498, 317762, 569698, 910563, 911043, 712580, 662498, 662498, 870467, 723524, 427299, 713221, 858051, 869698, 859331, 911715, 870467, 864580, 918114, 918114, 819394, 871234, 870467, 864324, 898659, 912771, 857922, 81218, 358627, 663140, 288387,
    882626, 739908, 819394, 545218, 107938, 712581, 910466, 864580, 853700, 358627, 461539, 712579, 870467, 486018, 888291, 713060, 888291, 859972, 495811, 663140, 888003, 527011, 750530, 883587, 743779, 908739, 662498, 461378, 461380, 912579, 800836, 545218,
    727236, 495811, 883587, 870467, 545218, 395780, 725570, 713059, 81411, 911715, 712578, 869699, 912386, 743779, 713060, 918114, 869828, 910466, 712580, 100260, 870467, 869698, 898659, 671330, 750530, 879651, 870980, 713060, 107172, 486018, 888291, 690852,
    671330, 569698, 317762, 918114, 869572, 662498, 107938, 712578, 486018, 671330, 395780, 746563, 461539, 288387, 542658, 843972, 819394, 883587, 542658, 719268, 911907, 695810, 869955, 894339, 907203, 107938, 107938, 542658, 910563, 857795, 72578, 329282,
    910755, 671330, 858051, 329282, 912386, 69667, 767299, 869698, 712580, 727236, 610980, 906339, 906339, 713221, 764739, 740323, 110051, 171842, 819524, 870467, 495812, 725570, 870467, 107938, 819394, 180482, 721603, 645730, 610978, 542658, 911715, 871234,
    645730, 366178, 800836, 576898, 398181, 610980, 857795, 100260, 912771, 870467, 870467, 276643, 139586, 912386, 743779, 486018, 398181, 700932, 870467, 743779, 72578, 869699, 912386, 870467, 891939, 858051, 427298, 869698, 358627, 81411, 843972, 918114,
    712578, 888291, 795332, 918114, 910947, 859972, 545218, 527011, 767299, 495812, 695810, 398181, 329282, 858820, 859972, 3970, 806467, 545218, 461378, 912771, 858051, 870852, 721604, 878787, 486018, 870467, 870467, 527011, 797252, 643172, 171842, 727236,
    743108, 912579, 107938, 288386, 896931, 712581, 869698, 712578, 725572, 486018, 918114, 645730, 905283, 891939, 713221, 871108, 869570, 398181, 883587, 90467, 712580, 912386, 107938, 486018, 869570, 713060, 870467, 883587, 486019, 486018, 888291, 888291,
    461378, 858180, 288387, 858180, 486018, 513572, 579298, 739395, 898659, 870467, 486018, 545218, 288386, 486018, 869700, 90467, 906339, 279746, 72578, 869698, 81221, 906243, 486018, 288387, 869316, 247298, 785092, 870980, 713221, 912579, 545218, 288387,
    843972, 279746, 81411, 719268, 905283, 107938, 647491, 486018, 329283, 90467, 398181, 743779, 535301, 545218, 725700, 279746, 719268, 870980, 542658, 744260, 910851, 610, 892035, 662661, 712578, 662661, 523652, 107938, 663140, 495812, 911043, 81189,
    870852, 329282, 893859, 906243, 870212, 911043, 725572, 879075, 911043, 727236, 610, 358627, 785092, 857668, 750660, 910466, 461539, 911043, 870467, 81190, 911043, 864580, 785092, 81190, 486181, 869316, 398181, 893187, 744260, 94114, 746563, 180482,
    171842, 870980, 869570, 288386, 461380, 180483, 886083, 870980, 910947, 94114, 610, 3970, 107938, 139586, 662661, 911043, 911043, 864580, 740292, 107938, 864452, 750660, 139586, 799300, 712578, 910466, 180483, 81190, 870340, 535300, 535300, 803652,
    398181, 882819, 799300, 912195, 744260, 857668, 662661, 886083, 912195, 803652, 906243, 542658, 81187, 859331, 918114, 662500, 81187, 405892, 662498, 740323, 663141, 764739, 276643, 819524, 857922, 712900, 358627, 800836, 803652, 727236, 495812, 495812,
    712901, 878787, 671330, 502531, 405892, 700932, 676452, 495812, 750660, 869698, 276643, 495812, 857922, 891939, 647492, 755524, 542658, 358629, 739524, 748484, 882915, 910275, 495812, 513571, 495812, 358627, 358627, 537858, 643172, 107938, 400739, 767299,
    815300, 358627, 87587, 427298, 486018, 785092, 358978, 864324, 486018, 857795, 870340, 679811, 3970, 870724, 869828, 81218, 395812, 869316, 739523, 725570, 725570, 486020, 869698, 755012, 662980, 663141, 417698, 750530, 358627, 603778, 400739, 461378,
    542658, 906243, 821059, 879075, 864580, 671330, 918114, 739522, 723523, 764610, 417698, 912386, 750530, 800836, 870082, 613538, 918114, 890691, 764611, 486019, 767299, 764611, 859331, 898179, 244418, 695810, 486020, 662661, 803652, 821059, 725572, 897315,
    461539, 857922, 400739, 869699, 911043, 712580, 725570, 662661, 893283, 748484, 869316, 495811, 857924, 679811, 858434, 764610, 535301, 358627, 700932, 486018, 417698, 395812, 739522, 713221, 893187, 893571, 725570, 858692, 746564, 713061, 739522, 918114,
    110051, 329282, 235778, 542658, 795332, 139586, 461380, 486020, 395811, 700932, 712579, 870467, 398181, 864324, 859331, 358627, 912771, 869316, 858820, 697572, 739522, 100259, 888291, 793156, 767299, 743108, 535301, 767300, 110051, 7393, 662500, 869316,
    486020, 906147, 893571, 912483, 725570, 725570, 725570, 454178, 908643, 776132, 107938, 486020, 883587, 461378, 400739, 713061, 857796, 662979, 870596, 545218, 366178, 486018, 800835, 864324, 502531, 918114, 712899, 461378, 523652, 662500, 662500, 502531,
    81380, 461380, 869828, 739522, 871234, 366178, 870082, 906243, 910179, 897315, 746563, 869316, 907491, 486020, 869572, 725572, 679811, 712581, 712580, 647492, 713060, 869444, 81218, 869698, 358627, 911907, 409380, 545218, 910466, 755012, 860228, 486018,
    891939, 81221, 869444, 898659, 764611, 81189, 358629, 853572, 764740, 358627, 81221, 906243, 743139, 906339, 819524, 662500, 461378, 180483, 859331, 896739, 110051, 72578, 690852, 911907, 743108, 888291, 887331, 907491, 713061, 890691, 903075, 662500,
    869316, 744260, 803652, 643172, 427298, 542658, 725570, 870980, 81187, 858692, 400739, 358627, 662501, 803652, 795331, 486020, 368578, 542658, 81411, 725570, 750530, 744132, 100259, 643172, 739395, 358629, 645730, 853700, 886083, 107938, 899331, 868932,
    870082, 858692, 461539, 869698, 81218, 395780, 893187, 744258, 647492, 912195, 486018, 461539, 288386, 896739, 739522, 545218, 610980, 908451, 744260, 750660, 908451, 537858, 755524, 542661, 869316, 366178, 819394, 858692, 746564, 799300, 746563, 750532,
    918114, 486020, 821059, 180483, 883299, 785092, 748484, 81221, 679811, 893859, 869955, 869316, 545218, 859331, 869955, 662498, 905667, 662498, 905667, 767299, 139586, 545218, 545218, 100259, 366178, 746564, 878275, 358628, 890691, 662500, 542660, 368578,
    870596, 893571, 859331, 795331, 180483, 107938, 911907, 740323, 713060, 908643, 695810, 903075, 743108, 486018, 869698, 662498, 693252, 358629, 870596, 764611, 908355, 870596, 662500, 870980, 693252, 545218, 545218, 486020, 910179, 869700, 107938, 893571,
    803652, 906243, 542660, 795331, 662500, 911907, 713221, 857922, 542660, 725572, 755524, 871236, 542658, 671330, 897315, 869700, 911907, 870980, 107938, 819394, 750530, 764611, 663140, 713059, 663140, 613538, 72578, 894339, 712580, 912386, 610978, 882915,
    712901, 662500, 870212, 725570, 721603, 910466, 662979, 513572, 907203, 764739, 725570, 882819, 869699, 713059, 819394, 908643, 81218, 358978, 542658, 767299, 139586, 244418, 576898, 744258, 712900, 870596, 400739, 288386, 864580, 542658, 712578, 288386,
    405892, 700932, 912386, 739522, 911907, 870596, 800835, 358630, 878787, 893859, 764739, 887234, 857922, 72578, 542661, 712580, 918114, 542661, 739395, 495812, 542658, 911907, 545218, 139586, 662498, 882819, 878787, 712901, 800835, 871108, 495812, 891171,
    81219, 666371, 666371, 690852, 725572, 907203, 819396, 743139, 869570, 871108, 81218, 486020, 869955, 882626, 542658, 910563, 918114, 905283, 879651, 461378, 910755, 888291, 690852, 906339, 486019, 882435, 358630, 690852, 819394, 666371, 764739, 912195,
    513572, 486021, 576898, 666371, 666371, 905283, 712578, 542661, 486021, 793156, 366178, 81220, 276643, 871108, 712580, 905283, 868932, 903075, 542658, 912771, 750530, 903075, 864580, 795331, 819394, 671330, 892035, 712578, 775620, 713061, 892035, 911715,
    486018, 69667, 870596, 910947, 911907, 405892, 107938, 72578, 87587, 858051, 90467, 90467, 906339, 603778, 912386, 905539, 903651, 887715, 486018, 276643, 409380, 645730, 713060, 662979, 405892, 870467, 486018, 764739, 110051, 858051, 746563, 495811,
    288386, 764612, 908451, 882819, 892035, 869954, 358978, 495811, 918114, 870467, 662498, 912386, 743108, 405892, 869954, 894339, 712578, 918114, 910563, 663140, 366178, 358627, 857922, 800836, 358627, 858820, 767299, 100260, 918114, 910179, 713060, 883299,
    81187, 859331, 713221, 712580, 871234, 486181, 721603, 857922, 576898, 276643, 912579, 535301, 495812, 858051, 395812, 910563, 891939, 358627, 743108, 461379, 878787, 892227, 535301, 81218, 882626, 81220, 769220, 908739, 891939, 180482, 785092, 857924,
    888003, 897315, 869572, 723524, 712580, 918114, 888771, 755012, 858180, 864580, 869699, 894339, 910563, 789060, 869570, 888003, 884739, 700932, 671330, 486021, 712581, 81411, 892227, 870467, 486020, 789060, 400739, 819396, 713060, 545218, 398181, 906339,
    276643, 721604, 700932, 869316, 81220, 857795, 769220, 910755, 695810, 905539, 910563, 405892, 712578, 918114, 697572, 288387, 427299, 527011, 918114, 495812, 81218, 870467, 100259, 486181, 764740, 486018, 869570, 858051, 871108, 398180, 886563, 542658,
    545218, 542661, 743108, 857924, 858820, 662498, 662498, 870852, 662498, 87587, 662500, 712580, 859844, 910371, 662498, 908739, 87587, 767300, 87588, 723524, 870724, 882339, 891171, 882819, 806467, 895587, 662498, 712900, 725570, 743748, 800836, 819396,
    662498, 906243, 864452, 871108, 778435, 912483, 712578, 912386, 869444, 542661, 495811, 887331, 907299, 821059, 107938, 871108, 764611, 545218, 869570, 869700, 662981, 662498, 869570, 712581, 486020, 171842, 712580, 858436, 819396, 486018, 755524, 405861,
    750530, 821059, 819524, 869828, 910563, 879075, 671330, 906339, 725572, 81220, 769220, 744132, 819396, 910371, 918114, 910371, 905539, 890691, 819524, 910563, 542660, 871108, 725572, 110051, 869699, 695810, 461379, 870467, 110051, 398181, 395780, 883299,
    712579, 869828, 871108, 869570, 663141, 910371, 769220, 869572, 869700, 495811, 869698, 870724, 610978, 721604, 461378, 640771, 870596, 486020, 882626, 857924, 276646, 110051, 712579, 882819, 712578, 858180, 908739, 918114, 110051, 398180, 888291, 858434,
    81218, 912099, 693252, 712580, 288386, 713060, 871108, 908739, 81221, 882339, 725570, 764740, 744258, 542658, 908451, 100259, 712578, 871108, 894339, 789060, 662500, 815299, 870467, 545218, 579298, 853700, 542660, 819394, 869572, 871108, 486018, 662980,
    288387, 764611, 545218, 723524, 869698, 907587, 900771, 911043, 662501, 409411, 712580, 486018, 662979, 910755, 719267, 869572, 288387, 857668, 870084, 912771, 870467, 662980, 666371, 800835, 107938, 764611, 799300, 743748, 912675, 486020, 906915, 395780,
    907299, 910755, 81411, 81219, 871236, 712578, 662501, 110051, 417698, 663141, 81411, 712578, 279746, 486020, 486021, 81220, 912195, 744132, 910563, 785092, 725570, 746563, 869698, 843972, 288387, 869698, 858692, 869570, 486018, 727235, 662500, 795331,
    247298, 910947, 871236, 662498, 643172, 911043, 576898, 894339, 910563, 871236, 535301, 911715, 869570, 894339, 712580, 395780, 785092, 545218, 395780, 869700, 882339, 719268, 712580, 912579, 871236, 897315, 542658, 883587, 795331, 894339, 542660, 671330,
    892035, 107938, 405892, 870596, 712580, 454178, 911907, 486018, 87587, 535301, 486018, 858051, 910371, 110051, 405892, 409380, 645730, 662979, 767300, 603778, 858051, 871108, 806467, 764739, 870467, 90467, 764612, 358627, 713060, 903651, 276643, 712580,
    767299, 910563, 918114, 542660, 871234, 663140, 712578, 870467, 713221, 870596, 721603, 918114, 81187, 858820, 869698, 495812, 883299, 721604, 858180, 891939, 180482, 906915, 884739, 395812, 81411, 888771, 869572, 486021, 897315, 892227, 785092, 700932,
    769220, 869316, 789060, 910755, 712578, 288387, 908739, 486018, 695810, 486018, 869570, 800836, 869700, 869570, 662498, 887331, 743748, 662981, 778435, 663141, 910371, 819396, 859844, 723524, 743108, 908739, 895587, 857924, 871108, 712900, 87587, 545218,
    870724, 912386, 171842, 712581, 486020, 819396, 795331, 695810, 819524, 486018, 110051, 81411, 671330, 886563, 755524, 725572, 712580, 890691, 870724, 81220, 750530, 545218, 725572, 821059, 869828, 495811, 918114, 858434, 713060, 764740, 882819, 888291,
    395780, 542658, 869698, 486020, 727235, 853700, 815299, 894339, 869700, 908451, 900771, 907587, 910563, 799300, 662980, 870084, 486021, 662500, 857668, 81220, 785092, 871236, 887715, 910563, 767300, 887715, 906723, 887715, 662498, 366178, 647492, 712581,
    870340, 767300, 869700, 869700, 857796, 712579, 461380, 912675, 690851, 398181, 662498, 90468, 869700, 894339, 690851, 545218, 461380, 87588, 87588, 767300, 87588, 395811, 897315, 897315, 918114, 663140, 764610, 461378, 358627, 139586, 898179, 461378,
    821059, 767300, 358978, 461379, 695810, 789060, 918114, 663140, 100259, 870340, 912579, 461379, 870980, 107938, 870467, 81282, 764610, 795331, 276643, 671330, 645730, 569698, 895587, 358978, 764610, 542658, 740292, 910371, 725570, 486018, 870084, 405861,
    895587, 486018, 789060, 819394, 870340, 897315, 645730, 486018, 486018, 804164, 100260, 819394, 662501, 417698, 898659, 100260, 911715, 911715, 764739, 358978, 888291, 100260, 461378, 662501, 869956, 888291, 800836, 869698, 358627, 750530, 910755, 662501,
    869698, 712580, 858180, 906819, 906819, 662501, 81220, 908451, 712580, 869698, 100260, 662501, 662500, 107938, 769220, 891939, 690852, 869828, 712578, 764611, 662980, 764739, 662980, 542660, 894339, 769220, 904707, 662501, 395780, 107938, 775620, 869698,
    405861, 90468, 764739, 405892, 400739, 400739, 882819, 918114, 712579, 723523, 869698, 418, 869698, 645730, 800836, 895587, 870467, 857922, 712578, 910563, 712581, 712578, 712578, 171842, 857668, 81411, 81411, 869316, 662979, 405892, 486021, 750530,
    882819, 887331, 883587, 662979, 400739, 358627, 486020, 869572, 400739, 853572, 871108, 87587, 81411, 743139, 72578, 72578, 395811, 725762, 853700, 869698, 542660, 743139, 725762, 800835, 643171, 857668, 725762, 764611, 869698, 81411, 869572, 883587,
    911043, 893859, 893859, 892707, 898179, 713059, 662498, 890691, 890691, 890691, 884739, 884739, 887234, 276644, 858436, 901635, 901635, 755524, 894339, 697572, 912386, 769220, 740292, 740323, 110051, 358978, 645730, 869698, 671330, 857922, 486019, 139586,
    743779, 81187, 893859, 806467, 486018, 690852, 743170, 896739, 904707, 486021, 640772, 712899, 896739, 486018, 893859, 743779, 911715, 576898, 406532, 887234, 869698, 752196, 7330, 417698, 676452, 857795, 110051, 400739, 746563, 746563, 405892, 769220,
    870467, 640771, 903075, 746563, 713060, 72547, 800836, 713061, 911715, 610, 799300, 803652, 870467, 841284, 864580, 887234, 3970, 909603, 461379, 869570, 542660, 740323, 663141, 895011, 647491, 495812, 139586, 725570, 750530, 891171, 90467, 869570,
    647492, 744132, 752196, 610978, 697572, 139586, 744132, 180483, 542660, 180483, 744132, 542660, 90468, 180483, 542660, 542660, 744132, 725762, 663141, 676452, 697572, 180483, 903075, 870467, 746563, 841284, 906243, 610978, 697572, 697572, 803652, 107938,
    107938, 870340, 671330, 288386, 869698, 288386, 671330, 870340, 870340, 907203, 869698, 537858, 486018, 906339, 515971, 882435, 454178, 366178, 81220, 81221, 400739, 858051, 869316, 858434, 870596, 869828, 427298, 767300, 858180, 767299, 69667, 94114,
    395781, 461378, 870467, 395811, 918114, 887331, 764739, 905859, 640772, 486020, 764612, 870852, 911715, 603778, 81283, 515971, 719267, 775620, 107938, 90467, 911043, 662498, 645730, 662500, 712578, 712578, 887331, 515971, 495812, 908643, 663140, 713221,
    712901, 910563, 663140, 910371, 870596, 884739, 746563, 887331, 358978, 366178, 857922, 864452, 879651, 461379, 908643, 912386, 887331, 870467, 610978, 647492, 912386, 918114, 640772, 713060, 712580, 912771, 905859, 486019, 663141, 870467, 879075, 579298,
    857922, 81218, 171842, 400739, 869699, 869570, 100260, 803652, 755012, 168962, 545218, 495812, 613538, 100259, 640772, 535299, 869698, 461381, 739939, 454178, 755012, 400739, 785092, 712741, 712741, 887234, 81282, 398212, 853572, 837700, 912386, 461539,
    398181, 864580, 870467, 882819, 898659, 910563, 882819, 910563, 897891, 897891, 486018, 910563, 841283, 793156, 276866, 869956, 662498, 901923, 819394, 843972, 804164, 87587, 366178, 90467, 87587, 910755, 785092, 495812, 911715, 398212, 883011, 895011,
    662980, 697572, 898659, 409380, 663140, 662500, 420098, 789060, 744258, 910755, 869443, 910179, 905763, 398181, 902403, 247298, 886083, 910179, 839748, 901923, 645730, 461378, 897315, 542658, 542660, 712901, 882819, 891939, 542661, 859972, 645730, 725700,
    806467, 853572, 894339, 180483, 81380, 746563, 864452, 902403, 495812, 901635, 418, 7330, 100259, 858051, 663140, 743748, 918114, 918114, 139586, 887331, 662501, 87587, 755012, 645730, 891171, 871108, 752196, 94114, 898659, 610978, 358629, 400739,
    882723, 897315, 743170, 750530, 81282, 869444, 901923, 276643, 276645, 835779, 110051, 713221, 907203, 673, 878275, 329282, 911043, 486020, 910371, 785092, 107938, 486019, 755524, 725570, 395781, 918114, 712580, 610978, 819396, 610, 693252, 329283,
    721604, 662500, 545218, 610, 859844, 690852, 288386, 870724, 454178, 869698, 366178, 799300, 486018, 645730, 910563, 888291, 545218, 727235, 743170, 395781, 545218, 891171, 640772, 406501, 725700, 486021, 918114, 690852, 576898, 712578, 358629, 545218,
    785092, 869700, 545218, 870467, 910851, 882915, 800835, 700932, 643172, 610978, 406501, 799300, 882819, 461539, 869316, 693252, 764611, 870467, 785092, 662980, 81218, 882819, 907395, 712581, 288387, 764739, 869700, 662980, 358628, 746563, 793156, 843972,
    704291, 542660, 725570, 107172, 892227, 81187, 869698, 789060, 418, 542660, 907587, 704291, 461378, 662498, 882819, 727235, 910947, 901923, 94114, 869956, 712580, 725700, 869444, 535300, 418, 100259, 545218, 743779, 869954, 897315, 857922, 869956,
    804164, 610978, 869954, 398180, 662500, 94114, 461381, 869956, 800835, 461381, 912386, 785092, 903075, 743108, 486018, 888291, 743108, 819396, 903075, 870467, 870467, 454178, 486020, 884739, 870467, 910755, 908643, 713060, 892035, 712580, 107938, 69667,
    911043, 107938, 405861, 712578, 712578, 892035, 107938, 107938, 869444, 139586, 892035, 869443, 663139, 663139, 663139, 892035, 859331, 690851, 276643, 859331, 864452, 690851, 247298, 247298, 461380, 81219, 864452, 81189, 860100, 712579, 910466, 870596,
    883587, 81189, 662980, 869698, 764611, 235778, 908643, 764611, 721603, 81189, 395812, 764612, 721603, 725570, 545218, 713221, 486020, 739396, 800836, 358627, 870082, 886563, 366178, 712580, 454178, 81189, 712579, 545218, 662661, 640772, 712578, 535299,
    918114, 912195, 764740, 542658, 858820, 107938, 795332, 697572, 107938, 910371, 366178, 817220, 610980, 139586, 81221, 859844, 171842, 725572, 579298, 884739, 883587, 721604, 69667, 869698, 908355, 739396, 882435, 896739, 883299, 910851, 911043, 405861,
    748484, 746564, 662501, 545218, 869698, 486018, 486020, 743779, 817220, 662498, 545218, 545218, 395780, 276646, 769219, 542661, 486020, 486020, 279746, 100259, 918114, 647492, 100259, 100259, 882435, 882435, 869698, 537858, 697572, 785092, 406594, 901635,
    864580, 785092, 891939, 643172, 87587, 869698, 739908, 819396, 819394, 767299, 662500, 107938, 107938, 858820, 662661, 427298, 81218, 662498, 815300, 912386, 81411, 288387, 870467, 712581, 645730, 918114, 704291, 910371, 910466, 870467, 662498, 662980,
    90467, 662500, 858051, 918114, 893187, 891939, 545218, 869570, 486018, 857795, 662498, 100259, 764739, 358978, 907299, 869443, 486018, 81219, 81219, 764612, 910466, 892227, 890691, 870467, 110051, 857795, 725762, 81187, 898179, 3970, 887331, 858052,
    395812, 744132, 869699, 911715, 725572, 869698, 906243, 910371, 542658, 645730, 673, 712580, 395781, 739396, 358627, 288386, 695810, 869828, 719267, 800836, 908643, 858820, 750530, 819524, 857796, 542658, 107172, 171842, 579298, 879651, 879075, 890691,
    821059, 869698, 869698, 139586, 906051, 907491, 180482, 841284, 610978, 366178, 727236, 662980, 910755, 911043, 712580, 723523, 405892, 869700, 712899, 755012, 139586, 884739, 903075, 767299, 81218, 647492, 898659, 918114, 663141, 461378, 725570, 910371,
    918114, 870980, 898179, 764611, 486019, 898179, 898179, 358978, 882626, 576898, 912386, 870467, 400739, 486018, 495812, 803652, 744132, 839748, 869828, 910851, 869700, 662661, 454178, 821060, 542658, 858180, 869699, 769219, 662500, 739908, 398180, 107938,
    740323, 912386, 895011, 910466, 486018, 461539, 882626, 486020, 882626, 890691, 461381, 815300, 69667, 785092, 785092, 918114, 107938, 358627, 912386, 486021, 81411, 869699, 69667, 764610, 858180, 858434, 405861, 815300, 700932, 713221, 662500, 358627,
    910947, 81220, 918114, 857924, 888003, 417698, 495812, 869444, 81189, 882435, 495812, 887234, 81282, 857668, 400739, 858051, 486020, 695810, 81221, 647492, 542658, 495811, 893859, 276643, 358630, 417698, 486019, 906915, 725570, 739522, 857668, 81411,
    486018, 746564, 912386, 858180, 409411, 869444, 897315, 869700, 910947, 495811, 679811, 869699, 897891, 405892, 870596, 911043, 427298, 910371, 542658, 886083, 918114, 858051, 869699, 72578, 725570, 368578, 110051, 750530, 329282, 288387, 288387, 398181,
    398180, 910466, 880419, 712578, 769220, 542658, 247298, 358627, 576898, 542658, 358630, 912771, 910179, 906339, 662979, 908451, 398180, 643172, 139586, 898659, 817220, 869444, 495812, 72547, 663140, 502531, 100259, 697572, 911715, 883011, 461380, 395811,
    911715, 883011, 486019, 87588, 288386, 903651, 858820, 712580, 795332, 918114, 673, 576898, 90467, 891939, 835779, 579298, 513571, 893571, 542660, 859331, 94114, 329282, 712580, 743779, 911715, 869700, 869700, 775620, 409380, 815300, 815300, 882626,
    87587, 495812, 750530, 857922, 755012, 486020, 912483, 461378, 418, 545218, 139586, 910371, 502531, 903075, 527011, 72578, 247298, 912099, 888291, 891171, 918114, 806467, 395781, 743108, 676452, 81380, 712900, 817219, 882339, 358978, 542658, 329282,
    366178, 395811, 879075, 857795, 789060, 712578, 857795, 750530, 891939, 247298, 746564, 495811, 857668, 908643, 769220, 81219, 461378, 800836, 887331, 857795, 81282, 87588, 723524, 870596, 100259, 139586, 486020, 800835, 690851, 276643, 864452, 883587,
    81218, 662498, 72578, 882819, 908739, 409380, 725570, 395781, 81220, 887331, 893571, 662498, 817220, 647492, 910371, 746563, 662979, 81220, 100259, 869698, 864580, 662498, 87587, 739522, 901635, 713060, 910179, 892035, 909603, 897315, 725570, 869698,
    640771, 366178, 905763, 752196, 869700, 869700, 869700, 869316, 276866, 906339, 171842, 712581, 87587, 107938, 107938, 918114, 542660, 869572, 869572, 835779, 739522, 94114, 409380, 869316, 107938, 725572, 898179, 81282, 673, 897315, 869700, 893283,
    893283, 486021, 819396, 912771, 912771, 871234, 879075, 907203, 712578, 7330, 81282, 857922, 486020, 918114, 545218, 495812, 643171, 764740, 405892, 405892, 743139, 100259, 329282, 713059, 912386, 712581, 858051, 454178, 662499, 857796, 819394, 81221,
    663140, 247298, 888003, 755012, 870596, 486020, 910371, 366178, 910371, 235778, 486018, 81218, 869956, 486019, 755524, 911043, 288387, 744132, 857924, 910371, 713060, 180483, 697572, 409411, 712578, 859331, 662498, 795331, 871108, 911043, 869828, 898659,
    110051, 276643, 750530, 358629, 461380, 719267, 910851, 395780, 892611, 180483, 81219, 486021, 896739, 910371, 905667, 911715, 908355, 869699, 859331, 409380, 859331, 110051, 713060, 795332, 858434, 817220, 542658, 906723, 864324, 869828, 542658, 139586,
    712578, 721604, 81187, 869570, 329282, 895587, 743779, 750532, 662500, 871234, 244418, 883011, 869828, 81187, 107938, 793156, 640771, 247298, 247298, 603778, 860228, 712578, 764739, 110051, 713061, 486020, 486018, 907491, 739396, 3138, 819394, 740323,
    679811, 713060, 821060, 908355, 869570, 895587, 712578, 461378, 795332, 858180, 90468, 888003, 883011, 859331, 690852, 897315, 858434, 908451, 869572, 879651, 911043, 461539, 725570, 871108, 778435, 912771, 864580, 739522, 905859, 795331, 746563, 486018,
    542658, 663140, 795332, 107938, 800836, 869700, 576898, 750660, 864580, 898179, 700932, 81187, 643172, 81282, 542660, 744132, 712578, 647492, 906339, 495811, 409380, 891939, 910371, 329283, 882435, 107938, 869698, 912386, 799300, 725570, 545218, 869570,
    368578, 712901, 894339, 662501, 815299, 100259, 800836, 486018, 725700, 778435, 395811, 750530, 288386, 545218, 663139, 662981, 750530, 899331, 910947, 815299, 366178, 750530, 883011, 713059, 663140, 869572, 910371, 81218, 725762, 819394, 896739, 744258,
    713221, 882339, 244418, 486021, 819394, 744132, 839748, 81411, 789060, 869699, 486020, 610980, 276643, 869699, 891171, 398180, 81218, 662979, 893187, 662500, 569698, 869956, 870082, 662501, 886083, 542660, 800835, 893187, 406594, 725570, 542658, 910371,
    486018, 662500, 663141, 764611, 817220, 743170, 395780, 750532, 910947, 785092, 907299, 907299, 894339, 107938, 857668, 81187, 81219, 486020, 897891, 110051, 879075, 911715, 857922, 545218, 906915, 81411, 673, 695810, 887331, 892227, 912579, 764610,
    858434, 764739, 180483, 860100, 907299, 3970, 869570, 918114, 892227, 746564, 910466, 755012, 671330, 869316, 712578, 803652, 90467, 858820, 662498, 486020, 110051, 679811, 800836, 107938, 712579, 90468, 486018, 405892, 671330, 843972, 894339, 869572,
    110051, 107938, 712578, 740323, 879075, 740323, 406594, 910371, 81411, 81411, 486018, 882626, 712581, 87587, 899331, 869698, 797252, 486021, 739395, 663140, 662501, 542660, 171842, 869828, 417698, 695810, 690852, 910851, 725570, 712900, 679811, 712580,
    764611, 752196, 910563, 461378, 910563, 276643, 486020, 740323, 405892, 276866, 869443, 910947, 911043, 878787, 908355, 276643, 910947, 427299, 647491, 910371, 841284, 743108, 897891, 858434, 853700, 908643, 486018, 912195, 398180, 713060, 887234, 610978,
    869700, 888291, 744132, 869316, 662498, 764610, 486020, 869570, 180483, 486018, 725572, 750660, 819394, 795331, 495811, 739396, 819524, 662661, 906243, 712580, 799300, 743170, 712580, 288386, 545218, 869698, 869570, 910563, 542658, 869698, 418, 894339,
    785092, 750530, 870467, 879651, 693252, 911043, 576898, 81220, 725700, 764611, 908355, 690851, 329283, 662661, 857668, 910371, 895587, 81411, 743170, 888291, 110051, 879075, 542658, 276643, 81187, 406594, 800835, 912195, 869572, 857795, 893571, 743779,
    910179, 910466, 907299, 907299, 662498, 398212, 545218, 288386, 329283, 817220, 110051, 647492, 107938, 910947, 486018, 486181, 869572, 576898, 906339, 288386, 486181, 643172, 903075, 366178, 719268, 869698, 712580, 107938, 139586, 542660, 542660, 869699,
    662501, 869700, 803652, 905539, 81219, 911907, 800836, 857922, 911043, 764611, 898659, 395811, 180483, 795331, 486018, 746564, 815299, 894339, 894339, 81411, 764740, 870467, 910371, 180483, 712900, 406532, 542658, 486018, 819394, 894339, 406532, 897315,
    486018, 180483, 545218, 486020, 276866, 693252, 894339, 857924, 869316, 869698, 804164, 817219, 880419, 81218, 81218, 545218, 904899, 545218, 671330, 869316, 662500, 81187, 81218, 869570, 883011, 918114, 869570, 869570, 545218, 81218, 107938, 740323,
    785092, 139586, 891171, 879075, 100259, 712900, 712900, 180483, 727235, 712578, 276643, 405861, 906339, 750660, 486018, 906339, 752196, 859331, 712579, 892227, 713059, 663139, 107938, 395811, 912386, 405892, 81411, 817220, 454178, 288387, 712579, 739908,
    859331, 869698, 569698, 712581, 764612, 673, 358978, 400739, 662500, 90467, 90467, 907299, 721604, 645730, 911907, 911907, 502531, 662498, 700932, 869699, 662498, 739395, 869828, 892227, 81411, 857795, 662498, 713221, 3970, 740323, 579298, 87588,
    764739, 764739, 395812, 775620, 912195, 918114, 719267, 891939, 918114, 400739, 886083, 879075, 712580, 610978, 883299, 906339, 911715, 712899, 647492, 400739, 911715, 72578, 395811, 139586, 542658, 918114, 719267, 712899, 890691, 645730, 645730, 712580,
    358627, 750660, 880419, 727236, 461378, 712900, 857922, 663139, 909603, 800836, 461378, 746563, 461378, 869700, 542658, 398181, 857922, 900771, 454178, 882626, 81411, 81411, 815300, 918114, 100260, 610, 895011, 358627, 869698, 712580, 893283, 785092,
    542658, 869698, 647492, 662498, 69698, 663139, 869698, 87588, 576898, 662661, 906243, 461539, 662661, 870467, 815299, 893859, 713059, 486019, 901923, 671330, 662498, 869444, 891939, 906339, 858820, 107172, 486019, 329283, 72547, 879651, 673, 395811,
    276643, 869572, 764740, 663140, 663140, 918114, 712580, 713059, 910466, 918114, 542658, 795332, 725570, 81187, 869828, 911715, 891939, 746563, 100259, 697572, 750532, 329283, 81411, 94114, 870467, 676452, 887331, 139586, 819394, 662498, 662498, 107938,
    502531, 418, 817220, 662501, 671330, 712578, 806467, 542658, 884739, 864580, 908739, 527011, 395811, 395811, 107938, 721604, 887331, 645730, 910371, 757827, 869698, 869828, 908451, 857924, 662500, 907299, 610978, 857796, 910371, 542661, 891171, 87587,
    755012, 673, 721603, 750530, 87588, 739395, 100260, 869700, 891171, 395780, 752196, 418, 171842, 891939, 910563, 697572, 898179, 869570, 819396, 107938, 891939, 859331, 869572, 81221, 100259, 725572, 907203, 910371, 918114, 869828, 898659, 910371,
    918114, 858434, 910563, 911715, 906339, 898179, 712579, 869700, 859331, 405861, 690852, 358627, 871108, 695810, 918114, 869828, 180483, 817220, 418, 427298, 81218, 906339, 884739, 542658, 94114, 110051, 857922, 835779, 896739, 409380, 858180, 869700,
    486021, 721603, 795332, 640772, 671330, 712578, 882626, 712578, 869828, 90468, 542660, 908355, 721603, 858180, 858180, 486018, 910466, 910755, 395811, 857795, 869698, 725570, 81219, 690852, 869698, 427298, 901635, 72578, 81218, 461539, 569698, 821060,
    869698, 90467, 869700, 320642, 713221, 739396, 907587, 662498, 815299, 864580, 662981, 859331, 882435, 400739, 750530, 725762, 545218, 910371, 870467, 690852, 398180, 853700, 857922, 645730, 107938, 486020, 897315, 896739, 857795, 713221, 869699, 486021,
    663461, 90467, 869316, 87587, 713059, 713059, 569698, 569698, 891171, 895587, 898179, 712578, 744132, 542658, 815299, 869316, 395781, 712901, 815300, 712581, 576898, 912579, 671330, 906339, 320642, 870467, 643172, 643172, 785092, 857922, 880419, 799300,
    640771, 461539, 712578, 486018, 358629, 81221, 887331, 893859, 892227, 892227, 81411, 869699, 395780, 785092, 662980, 901923, 869316, 910466, 405861, 910947, 671330, 81187, 398181, 746564, 542658, 910563, 662980, 869698, 495812, 417698, 395780, 662498,
    882626, 712581, 663139, 897315, 857795, 869700, 739908, 542658, 912291, 869572, 542660, 358627, 870082, 910563, 815300, 910563, 662498, 858180, 486020, 817220, 740323, 858434, 743779, 870980, 891939, 647491, 72578, 899331, 764611, 894339, 891939, 418,
    821059, 542658, 409380, 542658, 906339, 903075, 545218, 427298, 869316, 358627, 803652, 542658, 803652, 764740, 7330, 645730, 835780, 461378, 789060, 869700, 276643, 712578, 739395, 857922, 857924, 427298, 739395, 815299, 857922, 891939, 358627, 870980,
    910851, 81221, 613538, 905859, 613538, 869443, 713221, 897891, 887234, 168962, 899331, 712578, 869444, 100259, 513571, 869828, 857922, 800836, 869572, 276643, 712580, 700932, 712581, 891939, 712580, 3970, 81218, 910371, 883587, 405861, 910371, 883587,
    870852, 398212, 857922, 869698, 869698, 368578, 767299, 107938, 725700, 69667, 869698, 662980, 398180, 918114, 778435, 764739, 663461, 891939, 603778, 713221, 746563, 869698, 713221, 713221, 891939, 870467, 910563, 719267, 870467, 858051, 857795, 495811,
    502531, 869828, 898659, 739908, 725762, 839748, 81411, 719267, 486019, 910179, 800836, 883299, 879075, 905763, 911043, 912771, 725570, 858820, 690851, 139586, 72578, 910851, 910755, 713060, 869698, 235778, 366178, 750530, 610, 918114, 778435, 870467,
    870467, 719267, 640771, 869698, 898179, 910179, 603778, 890691, 869572, 713059, 857922, 879651, 107938, 712578, 819394, 366178, 366178, 918114, 400739, 400739, 180482, 858051, 486021, 486019, 486020, 366178, 891939, 887234, 454178, 858434, 358627, 461539,
    869699, 882339, 895011, 168962, 869444, 358630, 746564, 869570, 527011, 515971, 839748, 276643, 486019, 527011, 398181, 542658, 789060, 663140, 906147, 454178, 869698, 107938, 713060, 895587, 906339, 906819, 486018, 869698, 368578, 358630, 859972, 662979,
    906819, 542658, 542660, 542660, 180482, 486020, 887715, 395780, 725570, 527011, 908739, 869828, 800836, 400739, 395781, 743108, 886563, 725700, 542661, 662981, 905859, 905859, 887331, 869828, 897315, 891171, 100259, 427298, 695810, 870467, 139586, 869698,
    662498, 891171, 461379, 515971, 662498, 695810, 695810, 918114, 405861, 910563, 725572, 884739, 527011, 755524, 910755, 910755, 72578, 869828, 879075, 879075, 918114, 107938, 868932, 100259, 81411, 879651, 81187, 545218, 905763, 746563, 110051, 317762,
    890691, 417698, 406532, 858180, 869828, 858434, 901635, 486181, 908451, 721603, 329283, 400739, 666372, 839748, 869700, 891939, 662500, 406532, 180482, 542660, 795331, 542658, 713059, 859331, 545218, 869698, 884739, 910563, 712580, 910371, 725700, 882435,
    486018, 662501, 668771, 879075, 712581, 725570, 918114, 486020, 358629, 486020, 882819, 486018, 746563, 869954, 764611, 366178, 712581, 869443, 880419, 906915, 815299, 795331, 725762, 870082, 461539, 882435, 288387, 843972, 395780, 542660, 725570, 495811,
    746564, 276643, 690851, 725570, 795331, 918114, 858434, 870467, 427299, 276643, 610, 859331, 878787, 910563, 366178, 398180, 893571, 803652, 712578, 712580, 795331, 713059, 883299, 139586, 869955, 859331, 486020, 640771, 869698, 795331, 3970, 712578,
    712578, 911043, 764740, 690851, 918114, 764868, 918114, 358978, 72578, 486018, 486018, 486020, 461380, 725572, 100259, 461539, 892227, 853572, 898179, 819394, 819394, 899331, 859331, 671330, 395780, 897315, 892227, 819394, 864580, 542660, 911043, 870467,
    502531, 712578, 662980, 910371, 712578, 864452, 107938, 795331, 673, 427298, 857795, 640772, 486018, 821060, 869700, 890691, 857922, 712900, 495811, 461380, 748484, 486020, 897315, 857922, 486020, 910371, 892227, 857922, 897315, 795331, 663140, 750530,
    461380, 486020, 461380, 100259, 461539, 819394, 859331, 395780, 795331, 892227, 486018, 712578, 750530, 864452, 107938, 890691, 495811, 857922, 785092, 906339, 495811, 744258, 892227, 870467, 740292, 486018, 461378, 746563, 912386, 750530, 662981, 81219,
    857924, 454178, 859331, 891939, 461539, 775620, 81187, 891939, 409411, 906339, 869444, 288386, 610978, 486018, 545218, 870467, 712899, 725570, 869444, 81218, 662981, 775620, 870467, 910851, 81187, 712578, 869698, 542658, 892227, 712578, 100259, 712578,
    100259, 100259, 752196, 918114, 906339, 906339, 785092, 405892, 486018, 764611, 910947, 750660, 405892, 739908, 908739, 869570, 891939, 869698, 858051, 910947, 869699, 662980, 662498, 910179, 486018, 910947, 869699, 673, 764739, 3970, 911043, 676452,
    663140, 869699, 869443, 892227, 869316, 606178, 398180, 911715, 870467, 712741, 764612, 869443, 740292, 663139, 100259, 276866, 870852, 869570, 869698, 869698, 288386, 400739, 72578, 663140, 72578, 869570, 713060, 358627, 713061, 890691, 912386, 912386,
    712900, 461378, 918114, 486020, 884739, 800836, 750530, 100260, 542658, 901923, 640771, 870596, 767299, 869698, 139586, 911043, 712899, 869698, 869698, 857922, 870340, 647492, 725570, 712741, 857922, 358978, 417698, 871234, 908739, 358627, 882626, 713059,
    107172, 869570, 910947, 869698, 81221, 72578, 859331, 69667, 869316, 859331, 398211, 69667, 890691, 461539, 857924, 81411, 891939, 81219, 888003, 712581, 910371, 663140, 870596, 700932, 819394, 712900, 821060, 673, 81411, 100259, 712741, 461379,
    887234, 727236, 864580, 869698, 894339, 358627, 535301, 535299, 910466, 107938, 693252, 486018, 870467, 858051, 744258, 795332, 673, 535301, 859010, 859331, 697572, 746563, 879651, 395780, 911715, 515972, 857668, 857795, 276644, 81282, 107938, 869316,
    398181, 891939, 712580, 910466, 910466, 888291, 676452, 368578, 81411, 495812, 860228, 713060, 860228, 427298, 81219, 395781, 859972, 891939, 171842, 857922, 744132, 869316, 288387, 87588, 712580, 887331, 400739, 317762, 869698, 610978, 869699, 750660,
    645730, 870980, 395780, 398180, 870852, 486018, 662498, 690851, 870724, 910371, 139586, 857796, 755012, 713061, 870082, 912483, 662500, 870980, 542661, 486020, 81187, 740323, 806467, 663140, 725700, 81219, 853572, 107938, 884739, 869570, 853572, 859331,
    869572, 859331, 409411, 81219, 673, 713221, 750530, 910563, 918114, 247298, 697572, 486020, 662498, 107938, 912386, 358629, 712578, 81411, 869699, 81282, 712580, 835779, 398180, 764740, 358627, 100260, 100259, 81221, 869572, 454178, 905283, 662498,
    900771, 7330, 910371, 869444, 693252, 870596, 662980, 486020, 244418, 180483, 81221, 743779, 454178, 869700, 819394, 869316, 819394, 712578, 662501, 398180, 910851, 695810, 288386, 721604, 400739, 662500, 859331, 486021, 869828, 910563, 72578, 908739,
    911043, 911043, 545218, 107938, 725700, 744132, 857796, 542658, 576898, 576898, 662981, 713059, 899907, 486020, 870082, 891939, 542660, 896739, 739395, 743170, 778435, 712578, 870980, 725762, 725762, 495811, 713059, 869698, 81218, 486021, 815299, 740292,
    395811, 81219, 395780, 869570, 870084, 279746, 712578, 712899, 427299, 713061, 901923, 795331, 673, 869700, 892227, 486018, 912675, 785092, 366178, 81411, 358629, 910466, 870083, 712899, 859331, 857668, 690851, 870083, 247298, 663141, 869572, 912195,
    690851, 746564, 288387, 712580, 486018, 785092, 495812, 712900, 835780, 869698, 406594, 712900, 645730, 81411, 81219, 869698, 869698, 662980, 719268, 904707, 662500, 427299, 317762, 81218, 870596, 740323, 406594, 869700, 486020, 107938, 486020, 81411,
    427298, 713059, 869572, 869955, 894339, 663140, 910371, 395780, 542658, 869698, 725572, 713059, 100259, 673, 743108, 910371, 743108, 911043, 395780, 719268, 869699, 795331, 398180, 910563, 903075, 81220, 869955, 869698, 746564, 712900, 100259, 427298,
    100259, 486018, 752196, 918114, 858051, 486018, 891939, 405892, 908739, 719268, 662980, 662498, 910179, 910947, 869698, 910947, 486018, 858051, 486021, 81411, 743170, 139586, 3970, 606178, 869570, 673, 806467, 712741, 676452, 764740, 663139, 764612,
    892227, 713061, 461378, 72578, 647492, 871234, 859331, 712741, 712899, 288386, 713060, 750530, 870596, 882626, 857922, 870340, 725570, 869698, 869698, 535301, 894339, 859331, 461379, 859331, 486018, 81411, 398181, 358627, 663140, 81219, 746563, 890691,
    857924, 461539, 910947, 869698, 713061, 888003, 712581, 69667, 409411, 869570, 870596, 891939, 910466, 107938, 857668, 910466, 888291, 911715, 81411, 535301, 860228, 676452, 859010, 857796, 358627, 912483, 905283, 427298, 870980, 887331, 171842, 910371,
    107938, 858948, 663140, 750660, 395780, 542661, 870852, 288387, 690851, 663140, 869698, 900771, 398180, 884739, 81221, 486020, 910563, 910851, 7330, 910371, 673, 107938, 869572, 395780, 912386, 81282, 100259, 750530, 819394, 853572, 869570, 576898,
    746564, 288386, 910563, 911043, 721604, 859331, 662500, 819394, 495811, 815299, 713059, 486020, 899907, 690851, 870083, 857668, 795331, 247298, 892227, 100259, 662500, 712900, 398180, 461378, 662500, 405892, 712579, 486018, 405892, 395811, 405892, 712578,
    712578, 743779, 799300, 395811, 427298, 803652, 886083, 486021, 662498, 859331, 542658, 748484, 87587, 725570, 276643, 712580, 764740, 870467, 870467, 100259, 100259, 859331, 81187, 81411, 704291, 869698, 704291, 869698, 486020, 358978, 743139, 673,
    81187, 712581, 81220, 486020, 515971, 888003, 139586, 918114, 712578, 727235, 740323, 107938, 358627, 486019, 764739, 712578, 366178, 90467, 81220, 727235, 750530, 871108, 764739, 869698, 870467, 879075, 739908, 81282, 645730, 603778, 645730, 890691,
    603778, 911715, 750660, 576898, 671330, 910755, 81220, 527011, 72578, 740323, 576898, 81282, 898179, 542658, 610978, 725570, 107938, 486019, 868932, 898179, 645730, 712578, 869698, 139586, 576898, 81187, 690851, 81220, 892227, 879651, 713060, 869828,
    918114, 882339, 775620, 320642, 869698, 395812, 87587, 647491, 743779, 764611, 690851, 461380, 870082, 870467, 887234, 358627, 859331, 725570, 912386, 887234, 461539, 853572, 725572, 897891, 883299, 663141, 535301, 887331, 909603, 869698, 400739, 898179,
    72578, 725570, 545218, 800835, 81411, 888003, 853572, 912386, 486019, 869570, 912291, 785092, 888003, 869956, 486020, 869700, 910947, 713060, 663139, 727235, 727235, 906339, 663139, 81189, 869956, 870596, 893571, 869955, 882626, 398180, 662981, 864452,
    905859, 769220, 712580, 897891, 486020, 288387, 918114, 690851, 744132, 744132, 870467, 486018, 712901, 427298, 495811, 712741, 409380, 110051, 879075, 912099, 893571, 764740, 739396, 486018, 663141, 869698, 358629, 918114, 890691, 878275, 870082, 799300,
    869698, 878787, 398180, 87587, 870467, 870724, 725700, 110051, 870084, 746563, 869955, 100259, 247298, 662980, 87587, 869700, 495811, 767299, 712580, 869956, 725570, 427298, 276643, 100259, 662981, 869572, 803652, 767299, 869956, 879651, 713060, 918114,
    869828, 775620, 882339, 870082, 712580, 878787, 712901, 743779, 870467, 764611, 690851, 647491, 725572, 890691, 358627, 461380, 887234, 486020, 723523, 859331, 909603, 461539, 535301, 883299, 898179, 887331, 72578, 869698, 912291, 870467, 486019, 81411,
    912386, 800835, 545218, 869956, 912386, 888003, 495811, 906339, 713060, 893571, 663139, 918114, 427298, 486018, 276643, 662981, 882626, 725570, 870596, 769220, 905859, 746563, 893571, 764740, 870724, 918114, 358629, 878275, 870082, 100259, 869955, 110051,
    870084, 662980, 869700, 767299, 400739, 712578, 663139, 835779, 700932, 712578, 398181, 72578, 398181, 713059, 712579, 690852, 912483, 912483, 542658, 892035, 918114, 870340, 767299, 662498, 662980, 662661, 869699, 712580, 912386, 892227, 857795, 400739,
    882626, 398180, 398180, 884739, 139586, 740292, 750530, 712578, 712900, 892227, 911907, 857922, 486018, 400739, 81187, 461379, 910851, 912386, 864580, 886563, 895011, 767299, 908451, 744258, 712578, 870340, 288386, 869699, 139586, 486020, 906723, 690852,
    743748, 882819, 545218, 857922, 746564, 906339, 912386, 695810, 857922, 400739, 81221, 857668, 857668, 785092, 912195, 695810, 887234, 869955, 869955, 871234, 235778, 879075, 180482, 171842, 757827, 892227, 918114, 398180, 486020, 764612, 870467, 317762,
    911907, 358978, 662498, 911907, 662501, 910179, 908643, 869698, 610980, 800836, 898659, 358978, 884739, 180482, 910179, 486019, 882626, 905763, 662499, 358627, 139586, 750532, 870467, 918114, 640771, 139586, 662980, 366178, 610978, 139586, 542658, 769219,
    869698, 486018, 886563, 712580, 785092, 886563, 358627, 857924, 535299, 713381, 461539, 662661, 180483, 712581, 400739, 545218, 853572, 870467, 486019, 864580, 882339, 486020, 417698, 857668, 725570, 712580, 886563, 486018, 139586, 486018, 276643, 329282,
    81187, 495812, 903075, 869316, 368578, 910851, 882915, 579298, 276643, 542661, 107172, 785092, 743108, 857668, 900771, 317762, 486020, 662498, 94114, 910371, 743748, 891939, 486018, 903075, 139586, 903075, 806467, 545218, 797252, 406532, 139586, 400739,
    87587, 857924, 695810, 366178, 871234, 906243, 869698, 744258, 871108, 882339, 871108, 643172, 870467, 713061, 750532, 870467, 107938, 886563, 757827, 81189, 107172, 713221, 750530, 712899, 398180, 406501, 486020, 461379, 690852, 139586, 918114, 871234,
    884739, 853572, 907203, 905283, 695810, 69667, 329283, 486020, 883587, 883011, 662498, 317762, 662501, 610980, 750532, 695810, 486018, 903075, 815299, 895587, 869955, 725570, 405861, 545218, 799300, 744258, 690852, 883299, 288386, 712578, 180483, 486018,
    870467, 662501, 750530, 640771, 871108, 542660, 723524, 912195, 610980, 880419, 329283, 139586, 748484, 739524, 712580, 764611, 81187, 713221, 757827, 366178, 744258, 744258, 892227, 892227, 800835, 317762, 857668, 892227, 247298, 799300, 757827, 744260,
    461380, 912195, 417698, 725570, 725570, 358627, 139586, 81187, 857668, 757827, 743108, 486018, 910466, 911907, 610980, 800835, 895011, 870467, 542658, 870467, 395780, 81187, 755524, 910371, 803652, 800835, 712580, 910371, 871236, 795331, 910563, 719268,
    276866, 903075, 317762, 857922, 486018, 744260, 883011, 712578, 180483, 892227, 800835, 663140, 775620, 746563, 764739, 857922, 695810, 886563, 886563, 775620, 795332, 858051, 897315, 406594, 486018, 139586, 690852, 523652, 893859, 100259, 898659, 513572,
    857922, 896739, 893859, 886563, 870467, 100259, 695810, 883587, 869700, 107938, 72547, 3970, 461539, 502531, 910371, 911907, 869443, 883299, 869572, 764611, 858051, 662980, 535301, 276643, 704291, 398180, 883299, 673, 603778, 662498, 898179, 858436,
    535301, 662500, 358629, 405892, 81218, 857922, 535301, 750530, 800836, 883299, 725762, 713060, 764611, 535301, 100260, 643172, 879651, 461378, 461378, 905763, 358627, 858180, 673, 358978, 870596, 870467, 911043, 171842, 869698, 870467, 859331, 647492,
    887234, 775620, 139586, 785092, 461381, 72578, 892611, 857924, 870596, 912099, 870467, 663139, 542661, 107938, 858820, 911715, 87587, 869699, 713059, 87587, 918114, 857796, 870596, 610978, 797252, 713059, 712900, 542661, 912386, 81411, 87588, 743748,
    891171, 513571, 869698, 643172, 662500, 778435, 610980, 647492, 461380, 879651, 806467, 905667, 542661, 858436, 907203, 486018, 100260, 712580, 81189, 882723, 409380, 454178, 911715, 405892, 276643, 918114, 912386, 918114, 90467, 723523, 712900, 107938,
    90467, 908739, 461378, 713060, 712579, 869828, 545218, 785092, 740292, 358629, 910371, 409411, 910851, 713059, 785092, 712900, 368578, 893859, 764739, 764611, 800835, 454178, 279746, 513571, 81220, 869698, 486018, 870724, 887234, 107938, 486020, 405892,
    894339, 905859, 542658, 542658, 545218, 800835, 72547, 3970, 461539, 869572, 883299, 858051, 358629, 673, 806467, 858436, 405892, 461378, 171842, 545218, 870596, 870467, 869698, 81218, 713060, 542658, 911043, 857796, 859331, 912099, 870467, 785092,
    870596, 857924, 545218, 785092, 713059, 918114, 542661, 610980, 879651, 81411, 778435, 87588, 542661, 743748, 912386, 882723, 918114, 486018, 90467, 276643, 905859, 869828, 712579, 610980, 545218, 713059, 800835, 869443, 461378, 882626, 882626, 645730,
    912579, 725572, 418, 418, 882626, 910851, 721603, 725572, 725572, 725572, 910851, 725572, 721603, 882626, 725572, 725572, 398180, 858434, 843972, 843972, 395780, 743748, 743748, 869698, 858052, 725572, 725572, 859331, 107938, 799300, 417698, 395780,
    767299, 662500, 107938, 107938, 662498, 910947, 869698, 891171, 897891, 662498, 869954, 87588, 329282, 870852, 869700, 417698, 110051, 495811, 395812, 3970, 764611, 910466, 417698, 673, 739939, 713059, 417698, 495811, 869828, 910371, 542660, 400739,
    72547, 610978, 864580, 870467, 869698, 486181, 486019, 398180, 883299, 887234, 750530, 244418, 358627, 884739, 870082, 908739, 276643, 910179, 857922, 495812, 81411, 755012, 911715, 819394, 723524, 486018, 901635, 869698, 891939, 870724, 895011, 910947,
    329283, 676452, 859331, 908739, 640772, 743139, 610978, 757827, 486181, 869572, 910947, 368578, 911715, 869316, 906339, 905539, 358627, 906339, 750530, 750530, 329282, 495812, 695810, 886083, 869698, 713059, 700932, 461381, 910466, 859331, 398181, 288386,
    912195, 906819, 764740, 542660, 908451, 81411, 542658, 461378, 893571, 723523, 883011, 892227, 723524, 81220, 883299, 545218, 911715, 870596, 495811, 107938, 743748, 180483, 910371, 695810, 869698, 417698, 869828, 897315, 725570, 905283, 107938, 746564,
    288387, 757827, 288387, 100259, 81189, 400739, 695810, 107938, 859331, 713060, 911715, 110051, 673, 725572, 891171, 673, 712579, 81221, 906339, 883587, 417698, 910371, 835779, 869698, 890691, 911043, 743108, 895587, 910947, 317762, 750530, 400739,
    869828, 288386, 317762, 662500, 569698, 911043, 461380, 712901, 725762, 859331, 739394, 400739, 288386, 912195, 713059, 757827, 750530, 896739, 400739, 883011, 800835, 912675, 906915, 662500, 869572, 107938, 869698, 673, 799300, 139586, 859331, 725570,
    911043, 897891, 910563, 288387, 739939, 712900, 693252, 887234, 819524, 725572, 725572, 418, 542658, 869828, 911043, 897315, 542658, 869698, 697572, 748484, 662498, 107172, 662980, 395811, 107938, 815300, 858434, 662661, 712578, 535301, 918114, 513572,
    727235, 764612, 869444, 603778, 869443, 663140, 721604, 869828, 882915, 743779, 910563, 721604, 910466, 139586, 911715, 395811, 908643, 906339, 725570, 645730, 727236, 858820, 879651, 610, 750530, 912483, 7330, 800835, 535301, 461539, 358627, 859331,
    180482, 107938, 888003, 72578, 906243, 870467, 486018, 81220, 743139, 869444, 785092, 712578, 461539, 905283, 795332, 486019, 793156, 713061, 795332, 110051, 100259, 398181, 886083, 712578, 545218, 918114, 910466, 918114, 888291, 740323, 795332, 486019,
    769220, 502531, 869700, 366178, 712578, 879651, 542658, 110051, 368578, 869316, 662498, 645730, 857796, 891171, 870596, 690851, 912483, 647492, 893571, 610978, 398180, 911715, 368578, 712581, 746563, 139586, 107172, 857668, 869698, 886083, 100259, 898659,
    907491, 879651, 673, 276643, 789060, 107938, 110051, 743139, 180483, 712581, 712581, 908355, 690852, 821060, 712578, 858434, 81187, 912386, 912483, 7330, 910466, 712581, 110051, 740323, 107172, 725570, 898179, 740292, 918114, 799300, 869699, 358629,
    81218, 857668, 100259, 712901, 815299, 857668, 869570, 748484, 725570, 910947, 725570, 870084, 910371, 713061, 898179, 662980, 800835, 244418, 746563, 817220, 7330, 882435, 793156, 764740, 817220, 912483, 535301, 542660, 869955, 395780, 911715, 486018,
    100259, 395780, 883011, 800836, 81220, 712578, 764740, 869955, 542658, 911715, 752196, 906339, 870980, 905859, 870596, 869698, 767299, 107938, 892035, 100259, 912579, 898179, 713060, 869570, 886083, 713060, 764739, 869828, 918114, 110051, 908643, 864452,
    662499, 910179, 744258, 395811, 461378, 288386, 244418, 743108, 912099, 821060, 893859, 743748, 743108, 911715, 700932, 870467, 712580, 486021, 398181, 869699, 329282, 785092, 898659, 180482, 864324, 894339, 247298, 912579, 739939, 100259, 858692, 610981,
    910371, 545218, 795331, 100259, 908739, 905859, 712900, 409380, 690851, 912579, 870724, 744258, 527011, 695810, 712580, 427298, 81218, 869570, 869444, 690851, 906339, 409380, 110051, 244418, 171842, 908355, 819394, 857924, 743108, 883011, 110051, 107172,
    81283, 712899, 739939, 690852, 486021, 853700, 398180, 107938, 542658, 912195, 893859, 869698, 486021, 368578, 785092, 712578, 690851, 764610, 843972, 869698, 542658, 139586, 100259, 869698, 610981, 858434, 910947, 743108, 100259, 800836, 819394, 819394,
    610981, 869443, 785092, 858692, 819394, 100259, 725572, 882339, 859331, 911715, 859331, 859331, 542658, 713221, 7330, 542661, 542658, 495811, 495811, 906339, 869698, 799300, 883587, 870340, 752196, 244418, 906339, 358978, 918114, 713060, 870467, 358978,
    804164, 918114, 901635, 750660, 662980, 857668, 893187, 81189, 912386, 764611, 81219, 81219, 395811, 767300, 662498, 815300, 907106, 750530, 671330, 712578, 869698, 739908, 859331, 329282, 712578, 613538, 662500, 906339, 486020, 645730, 7330, 495811,
    358978, 764612, 398180, 869444, 886083, 911907, 662500, 918114, 645730, 603778, 869443, 673, 892227, 757827, 890691, 869699, 662500, 721603, 276866, 858051, 740292, 903651, 764739, 110051, 869698, 857795, 645730, 891171, 405892, 869699, 502531, 750660,
    869698, 713221, 712578, 888003, 907106, 743779, 908643, 513572, 869828, 545218, 673, 857795, 886563, 912386, 610978, 329282, 859331, 821059, 461378, 171842, 800836, 727236, 671330, 461378, 366178, 645730, 107938, 671330, 912771, 368578, 911715, 864452,
    486019, 713060, 859331, 859331, 870467, 139586, 395780, 859331, 859331, 869572, 870340, 879075, 72578, 725570, 663141, 81187, 859331, 366178, 358627, 400739, 869954, 244418, 700932, 662500, 918114, 870212, 895587, 662500, 918114, 542658, 911907, 645730,
    100260, 721603, 366178, 891939, 890691, 276643, 713061, 579298, 712578, 712578, 750530, 486019, 579298, 690851, 81220, 869698, 671330, 535301, 168962, 879651, 486020, 700932, 869444, 405892, 785092, 400739, 887234, 897315, 727236, 859331, 900771, 486019,
    869699, 502531, 870596, 405861, 712581, 700932, 576898, 894339, 486018, 107938, 869570, 858051, 912675, 870467, 662661, 662980, 400739, 712580, 869444, 857924, 645730, 909603, 358627, 712581, 821060, 535299, 911043, 712580, 712580, 910755, 502531, 69667,
    893859, 486018, 645730, 858434, 819394, 869698, 869699, 461381, 81282, 885315, 911715, 279746, 892707, 662500, 804164, 671330, 406594, 750660, 395811, 907299, 859331, 515972, 906339, 869316, 713059, 910755, 869956, 695810, 883011, 461379, 405892, 713221,
    905539, 545218, 898659, 795332, 87587, 869443, 110051, 879651, 910466, 288386, 107938, 395780, 752196, 366178, 395811, 486019, 405892, 744260, 764740, 395780, 87587, 870467, 671330, 869572, 81219, 107938, 864580, 535301, 870980, 896931, 869444, 857795,
    663140, 395811, 896931, 7330, 81187, 870084, 139586, 712580, 909603, 815300, 247298, 542658, 640771, 907299, 398181, 888291, 542658, 320642, 727236, 358627, 713060, 329282, 895587, 912579, 139586, 395781, 869828, 486020, 767300, 750660, 908739, 395781,
    806467, 891171, 800836, 645730, 545218, 542658, 662501, 906147, 789060, 821059, 912579, 643172, 81282, 645730, 712580, 3970, 870724, 894339, 171842, 523652, 317762, 527011, 427298, 902403, 81380, 912386, 713061, 723523, 903075, 871108, 409380, 690851,
    752196, 662498, 662500, 870724, 486018, 107938, 495811, 461380, 898179, 395781, 864452, 712578, 542660, 908739, 461378, 883011, 276643, 713060, 545218, 817220, 545218, 81219, 725700, 451778, 893571, 515971, 764612, 910466, 901635, 87587, 869955, 81218,
    420098, 712580, 853572, 743779, 542661, 764610, 817219, 892611, 869700, 750530, 542660, 891171, 712580, 427298, 329282, 910371, 871108, 918114, 907203, 486018, 673, 502531, 398180, 695810, 712899, 81218, 100260, 755524, 712579, 107938, 515972, 663141,
    81221, 910371, 918114, 870340, 69667, 906243, 859331, 405892, 180483, 864452, 900771, 857668, 857668, 743139, 905539, 744132, 81220, 7330, 821059, 486020, 755524, 486020, 676452, 180483, 81218, 869572, 886563, 891939, 713060, 725572, 713221, 912386,
    910755, 7330, 898659, 905667, 406594, 891939, 662500, 898659, 908451, 417698, 721604, 712579, 542660, 908451, 892611, 895587, 911043, 7330, 904707, 90468, 858180, 690852, 908451, 662501, 542658, 860228, 81380, 366178, 640771, 417698, 725570, 666372,
    767300, 912386, 910466, 869700, 81221, 864580, 895587, 892611, 679811, 662979, 908355, 87587, 740323, 713060, 713060, 795332, 869698, 869828, 690852, 903075, 712578, 486019, 454178, 569698, 789060, 676452, 871234, 908451, 712579, 764740, 90467, 603778,
    545218, 908451, 743170, 871234, 868932, 910755, 461380, 869572, 542660, 859844, 725762, 815299, 576898, 897315, 725570, 910371, 895587, 750530, 666371, 871108, 869698, 545218, 725762, 515972, 662501, 882819, 662498, 395781, 690852, 815299, 107172, 395780,
    764740, 495812, 725700, 276643, 896739, 910755, 744132, 803652, 869955, 486020, 882435, 542658, 712579, 712578, 513572, 276646, 723524, 712901, 662500, 398181, 545218, 81219, 486018, 247298, 81411, 662501, 244418, 800835, 279746, 910755, 666371, 398181,
    180483, 671330, 895587, 901923, 81221, 906051, 288387, 764611, 857668, 857924, 100260, 690852, 712581, 892227, 785092, 912675, 799300, 906243, 893859, 427299, 869443, 869955, 486018, 486020, 713221, 748484, 3970, 869700, 427299, 892227, 904707, 869955,
    247298, 909603, 764611, 398181, 752196, 662980, 892611, 486020, 887331, 288387, 545218, 789060, 81411, 107938, 542660, 870980, 857668, 486020, 870980, 894339, 725570, 909603, 81220, 7330, 886563, 891171, 746564, 139586, 764739, 417698, 879651, 869698,
    406594, 871108, 910947, 750660, 358627, 398212, 870724, 486018, 903075, 72578, 743779, 857796, 727235, 869700, 180483, 841284, 869570, 486020, 486020, 662500, 69667, 870467, 899331, 542658, 894339, 486018, 871236, 535301, 662981, 643172, 418, 427298,
    860228, 743748, 545218, 725700, 879075, 545218, 712580, 535300, 803652, 673, 545218, 869443, 486020, 795331, 366178, 486020, 911043, 395780, 712581, 486020, 712578, 461380, 739396, 903075, 894339, 744260, 857922, 640771, 897315, 871236, 804164, 912195,
    903075, 892227, 690852, 358627, 795331, 495811, 451778, 869698, 713060, 752196, 870340, 244418, 799300, 329282, 662500, 81189, 662980, 276866, 764611, 750660, 817219, 613538, 100260, 739908, 869444, 888003, 368578, 908643, 139586, 757827, 740292, 713221,
    603778, 879651, 767300, 721603, 662500, 911043, 857795, 502531, 910466, 890691, 513572, 764612, 743779, 905539, 903651, 358978, 107938, 662500, 713061, 662500, 461378, 366178, 171842, 244418, 368578, 366178, 911907, 871236, 579298, 690851, 486019, 750532,
    911715, 909603, 366178, 800836, 860228, 662500, 898179, 879075, 859331, 869572, 317762, 725570, 610978, 645730, 886563, 869444, 892707, 893859, 887234, 815300, 3970, 911715, 398181, 486019, 7330, 901923, 870467, 746564, 700932, 545218, 750660, 869699,
    892611, 712899, 358627, 712578, 752196, 885315, 712580, 81219, 857924, 662979, 69698, 750660, 454178, 821060, 713061, 279746, 785092, 869698, 395780, 405861, 690852, 81282, 869699, 858434, 358627, 893859, 247298, 910466, 893571, 320642, 542660, 515972,
    869316, 908451, 542658, 870084, 81187, 891171, 888291, 673, 910755, 883011, 896931, 764740, 869443, 895587, 892611, 486020, 907299, 695810, 795332, 857795, 139586, 129, 662501, 713059, 905283, 883011, 898179, 723523, 662498, 420098, 171842, 527011,
    395781, 712578, 461378, 897315, 81380, 870724, 495811, 870467, 712580, 789060, 743779, 905859, 690851, 752196, 486020, 139586, 817220, 912386, 906243, 869700, 892611, 662501, 671330, 673, 69667, 857668, 713221, 755524, 908355, 81221, 81220, 406594,
    427298, 898659, 896739, 912386, 887331, 898659, 576898, 713060, 366178, 454178, 903075, 690852, 789060, 903075, 603778, 895587, 888291, 721604, 869698, 486019, 727235, 858180, 725700, 690852, 815299, 897315, 723524, 107172, 495812, 750530, 871234, 869572,
    895587, 395780, 892227, 799300, 247298, 545218, 427299, 795331, 247298, 744260, 662501, 748484, 743779, 894339, 81220, 870980, 542660, 869698, 69667, 725700, 418, 395780, 712581, 712901, 395781, 892035, 725762, 884739, 857922, 799300, 819394, 712901,
    817219, 569698, 276645, 662980, 743779, 817219, 100259, 725570, 395811, 725570, 662980, 892611, 513572, 723524, 405892, 907395, 666371, 712580, 712580, 486020, 821059, 712579, 859972, 886083, 712899, 901923, 712900, 643172, 712579, 800836, 725572, 725570,
    907395, 406594, 461380, 454178, 454178, 764610, 81187, 405892, 461539, 821059, 279746, 527011, 795332, 545218, 697572, 662979, 139586, 911907, 911907, 395780, 857922, 800835, 395781, 858692, 527011, 100259, 90467, 100259, 673, 409411, 107938, 90467,
    405892, 879075, 405892, 857922, 911043, 90467, 721603, 81282, 910371, 869699, 893283, 906723, 795331, 427298, 857924, 764740, 864452, 903075, 317762, 892611, 72578, 906723, 81190, 461380, 886083, 662498, 180483, 690852, 461380, 853700, 513571, 712578,
    81411, 395780, 645730, 746564, 81220, 317762, 90467, 317762, 817219, 276645, 662980, 100259, 395811, 725570, 90467, 81190, 907395, 910371, 712580, 405892, 486020, 821059, 513572, 817219, 712899, 901923, 461539, 90467, 317762, 821059, 545218, 892611,
    764610, 454178, 72578, 527011, 892035, 911907, 795332, 427298, 107938, 100259, 395781, 712578, 90467, 409411, 100259, 673, 395780, 795331, 857922, 81411, 893283, 906723, 72578, 864452, 417698, 906723, 395811, 81220, 918114, 918114, 537858, 427299,
    869443, 662500, 910466, 888003, 870467, 869699, 100260, 886083, 673, 663141, 721603, 905859, 515972, 110051, 858436, 870467, 645730, 495812, 764612, 100260, 869699, 713060, 891939, 398181, 673, 857922, 750530, 871234, 912771, 750532, 800836, 129,
    910179, 910179, 879651, 727236, 918114, 486018, 545218, 803652, 398180, 869444, 886563, 896739, 576898, 486021, 276643, 886563, 712580, 887234, 461539, 107938, 888003, 662980, 502531, 857668, 869444, 409380, 918114, 725570, 712579, 662980, 663141, 870467,
    247298, 886083, 869828, 713060, 81187, 712580, 883587, 903075, 110051, 712580, 905859, 887331, 912483, 869698, 427299, 806467, 869699, 139586, 81380, 398180, 645730, 800836, 908739, 712580, 545218, 663141, 712580, 869699, 358627, 100260, 858052, 712579,
    405861, 100259, 869699, 276643, 750530, 107938, 803652, 910371, 910371, 903075, 427299, 288387, 81282, 81221, 891939, 288387, 869699, 888003, 100259, 81411, 454178, 110051, 910466, 180483, 406594, 712578, 690852, 486018, 712901, 883587, 725762, 712578,
    869699, 276866, 906915, 800835, 329283, 748484, 785092, 288387, 288387, 725762, 712580, 764739, 610978, 887234, 486018, 883587, 712578, 869699, 870467, 81411, 803652, 712578, 542658, 542658, 542658, 870724, 81218, 870467, 679811, 789060, 835779, 869698,
    906723, 100259, 857795, 662500, 712741, 3970, 486018, 461378, 81220, 870467, 890691, 789060, 882626, 107172, 139586, 81187, 81218, 869954, 897315, 107938, 743108, 750660, 288387, 395811, 94114, 276646, 461380, 81187, 869954, 107172, 486018, 712578,
    743108, 895587, 542658, 835779, 910851, 869954, 486018, 486018, 107938, 107938, 910851, 603778, 887715, 279746, 579298, 90467, 764739, 640772, 110051, 542658, 400739, 100260, 800836, 894339, 81282, 879075, 910466, 750532, 746564, 81218, 910466, 750660,
    869570, 486018, 579298, 398180, 869316, 712899, 697572, 713060, 905859, 898659, 800836, 662498, 870596, 743779, 662979, 887331, 405892, 712900, 869699, 610978, 358627, 704291, 800835, 800836, 755012, 869700, 910466, 800836, 276646, 800836, 712579, 405892,
    869699, 897891, 897891, 110051, 895587, 817220, 727235, 910466, 910371, 893283, 869699, 712578, 869698, 406594, 746563, 693252, 666371, 107938, 545218, 803652, 883299, 486018, 640772, 100260, 358978, 645730, 910371, 764740, 712578, 486018, 897891, 817220,
    817220, 894339, 542658, 743779, 406594, 887331, 883299, 803652, 800836, 406594, 764740, 542658, 879075, 72578, 72578, 72578, 727236, 857796, 857796, 495812, 864452, 713060, 495812, 398181, 857796, 495812, 495812, 750532, 495812, 864452, 887331, 887331,
    892707, 819394, 819396, 819396, 879075, 7330, 864452, 81411, 7330, 454178, 752196, 69667, 662500, 858051, 750530, 427298, 910466, 858051, 857795, 495811, 910851, 903651, 358978, 869699, 276866, 603778, 576898, 871234, 417698, 721603, 869698, 911043,
    908739, 879651, 891939, 405892, 905763, 100260, 727236, 869699, 454178, 908739, 891939, 910563, 81282, 81282, 398211, 775620, 662499, 417698, 857668, 697572, 815299, 869700, 288387, 864580, 857668, 869572, 523652, 723524, 317762, 662981, 727236, 893283,
    139586, 418, 910563, 907203, 721604, 3139, 906339, 906339, 72578, 857668, 579298, 139586, 486020, 725572, 750530, 486020, 90467, 908739, 890691, 317762, 879651, 910563, 640771, 454178, 725570, 893283, 690852, 896739, 461539, 857668, 910851, 486018,
    486020, 486021, 395780, 417698, 81218, 712580, 662500, 427298, 610, 486020, 795331, 910371, 858051, 918114, 898659, 857924, 454178, 910371, 857668, 395811, 395781, 893283, 907203, 110051, 663139, 910371, 454178, 878787, 908739, 890691, 110051, 888291,
    725570, 910371, 288387, 427298, 610, 427298, 910371, 858051, 918114, 395811, 910371, 110051, 908739, 892035, 892035, 712580, 910563, 712580, 486018, 869699, 869954, 110051, 858820, 537858, 643172, 81221, 869700, 110051, 643172, 81282, 869699, 712581,
    81218, 7330, 752196, 857796, 180483, 712581, 860100, 395811, 869570, 869698, 869570, 673, 676452, 180483, 662498, 395812, 107938, 87587, 721603, 757827, 750530, 746563, 640771, 244418, 800836, 72547, 495812, 894339, 72578, 857668, 673, 673,
    910371, 910563, 893283, 663140, 869698, 869698, 739939, 427299, 107938, 647492, 894339, 918114, 486019, 905539, 81411, 180483, 486019, 897315, 697572, 398181, 869700, 110051, 81218, 395811, 495812, 329283, 329283, 647491, 893571, 329283, 912386, 912386,
    81380, 752196, 893571, 690851, 395811, 662500, 171842, 882819, 869572, 857795, 869698, 878275, 695810, 673, 673, 100259, 90467, 513571, 107172, 358629, 100259, 712580, 869443, 869698, 869828, 666372, 750532, 750532, 486021, 180483, 894339, 893859,
    853700, 910275, 815299, 744258, 911043, 740323, 857668, 461378, 910947, 100259, 81411, 395780, 858434, 817220, 727235, 712580, 647491, 545218, 795331, 903075, 857924, 7330, 752196, 857796, 395811, 712581, 860100, 869570, 869570, 87587, 461378, 757827,
    662498, 721603, 676452, 395812, 107938, 545218, 800836, 244418, 495812, 891939, 486019, 697572, 72578, 869700, 486181, 171842, 869698, 647491, 81411, 329283, 395811, 869700, 869700, 171842, 695810, 107172, 673, 358629, 100259, 673, 903075, 815299,
    750532, 666372, 740323, 894339, 395780, 858434, 647491, 857924, 764740, 725700, 461379, 918114, 712579, 712900, 725700, 276866, 358978, 908643, 895011, 461379, 869572, 858820, 139586, 725700, 671330, 910371, 100260, 911043, 908355, 892611, 908355, 764611,
    895587, 712578, 895587, 853700, 853700, 895587, 725700, 725700, 725700, 799300, 725700, 279746, 764740, 910755, 764740, 100260, 712900, 725700, 276866, 139586, 671330, 908355, 911043, 895587, 853700, 725700, 725700, 887331, 764611, 887331, 887331, 859331,
    859331, 739395, 486018, 752196, 671330, 910179, 712900, 679811, 712581, 704291, 858051, 107938, 789060, 400739, 869699, 764611, 764611, 679811, 869699, 905539, 869698, 912771, 725570, 891939, 908739, 879075, 882626, 909603, 671330, 879075, 859331, 886563,
    739939, 858051, 864452, 785092, 486019, 727236, 911043, 700932, 882626, 712581, 100260, 789060, 887234, 100259, 537858, 712578, 739395, 366178, 902403, 673, 368578, 502531, 886083, 288386, 107938, 859331, 911043, 409411, 891171, 859331, 486020, 870724,
    727236, 712580, 368578, 7330, 690851, 887331, 712901, 910371, 461380, 673, 844484, 110051, 81218, 81221, 909603, 81411, 486020, 743139, 406594, 690852, 764739, 910371, 461378, 69667, 676452, 690852, 366178, 888291, 712578, 180483, 895587, 908355,
    908739, 7330, 90467, 576898, 690852, 815299, 725570, 107938, 910755, 495811, 666371, 180483, 358629, 662980, 400739, 746563, 869698, 486018, 906915, 398181, 107172, 908739, 910371, 712581, 712900, 764739, 739939, 817220, 100259, 576898, 395780, 712581,
    897315, 869955, 835780, 859331, 752196, 486018, 910179, 690852, 704291, 712578, 858051, 107938, 400739, 764611, 869699, 486020, 859331, 879075, 671330, 886563, 869698, 887234, 906915, 712581, 72578, 537858, 785092, 712578, 727236, 366178, 886083, 673,
    107938, 902403, 502531, 891171, 409411, 712580, 461380, 69667, 180483, 461378, 908355, 395780, 7330, 576898, 366178, 895587, 910755, 495811, 815299, 666371, 358629, 835780, 859844, 180483, 891171, 712581, 764739, 366178, 610978, 725570, 366178, 329282,
    395811, 887331, 486020, 81282, 406594, 918114, 712580, 110051, 869443, 764739, 647491, 869443, 569698, 107938, 764740, 395811, 750530, 910179, 713059, 400739, 910947, 358627, 870467, 905859, 869443, 907106, 276866, 110051, 90467, 870467, 746563, 545218,
    911715, 366178, 879075, 892227, 910179, 869698, 857922, 90467, 857922, 486181, 244418, 870340, 869828, 905859, 870596, 139586, 358627, 613538, 139586, 645730, 869956, 486019, 800836, 713060, 908643, 918114, 100260, 859331, 725570, 910179, 910179, 671330,
    895587, 569698, 853572, 908739, 870467, 858180, 693252, 755012, 910947, 857924, 663140, 486181, 887234, 739939, 366178, 358627, 869699, 897315, 870596, 746563, 739939, 895587, 139586, 676452, 857795, 815300, 795332, 395811, 329282, 869572, 329283, 502531,
    673, 398181, 869444, 406594, 545218, 870724, 870596, 663139, 853572, 527011, 908739, 893571, 871108, 171842, 662498, 662498, 100259, 887331, 907587, 100259, 454178, 911043, 910466, 853572, 744132, 853572, 662500, 887331, 81221, 662500, 743779, 107938,
    695810, 857924, 712579, 871108, 180483, 907203, 671330, 461379, 910179, 461539, 908355, 662500, 398181, 870467, 895587, 643172, 693252, 712578, 882435, 427298, 100259, 869828, 912483, 907587, 662980, 662498, 870467, 276646, 545218, 712578, 897315, 712901,
    576898, 725762, 739395, 725700, 744132, 857922, 725570, 870467, 107938, 869570, 81219, 366178, 288386, 869316, 800835, 712900, 757827, 895587, 910179, 739524, 750532, 785092, 909603, 100259, 897315, 712900, 495812, 869698, 7330, 910179, 647491, 870596,
    100259, 803652, 545218, 693252, 712581, 486018, 276646, 139586, 712578, 81220, 542658, 725700, 569698, 107938, 910179, 910947, 400739, 857922, 907106, 366178, 545218, 869956, 859331, 671330, 918114, 139586, 912483, 358627, 910179, 613538, 486019, 869698,
    883299, 869316, 569698, 869699, 785092, 81219, 897315, 739939, 853572, 725700, 542658, 398181, 100259, 869444, 857795, 502531, 662498, 662498, 171842, 870724, 871108, 288386, 739395, 853572, 870467, 180483, 907587, 910466, 725762, 895587, 662500, 662980,
    725700, 897315, 744132, 395780, 870596, 486018, 276646, 712581, 461378, 910371, 910371, 910371, 107938, 767299, 869698, 81283, 905539, 891939, 879075, 879651, 882626, 695810, 750530, 180482, 739939, 857668, 743139, 535299, 454178, 329283, 769220, 853572,
    725570, 171842, 662499, 107938, 288387, 896739, 366178, 869316, 721604, 366178, 882626, 535300, 727235, 576898, 799300, 896739, 869316, 427298, 869956, 288387, 695810, 727235, 535300, 545218, 888291, 888291, 857668, 892707, 857668, 893859, 907587, 725700,
    527011, 527011, 750530, 764612, 910755, 719267, 898179, 746563, 527011, 727235, 764610, 864580, 645730, 912386, 764610, 719267, 695810, 879075, 725570, 898179, 918114, 887234, 858180, 857922, 775620, 910755, 893283, 486018, 643172, 358628, 858820, 908451,
    695810, 542658, 727235, 871108, 750530, 643172, 676452, 713061, 857924, 871108, 860100, 486020, 910179, 81218, 418, 892035, 662498, 542660, 713060, 727235, 643172, 569698, 666371, 815299, 815299, 668772, 869570, 895011, 662500, 662500, 835780, 81220,
    893283, 841284, 727235, 895011, 719268, 886083, 886083, 901923, 405892, 712578, 886083, 395811, 886083, 886083, 892035, 395781, 107938, 107938, 454178, 100259, 918114, 857795, 461539, 871108, 895587, 461539, 662981, 107938, 461539, 898659, 486018, 662498,
    451778, 180483, 409411, 358978, 244418, 898659, 869570, 100259, 869316, 542661, 107938, 329283, 662498, 87588, 542661, 910371, 767299, 400739, 853700, 725570, 576898, 486018, 869570, 743108, 100259, 712900, 107938, 883587, 858051, 891939, 358978, 405892,
    910179, 750660, 486018, 869570, 673, 673, 662979, 395812, 704291, 576898, 891939, 663140, 721603, 869828, 645730, 545218, 90467, 81218, 545218, 869954, 764612, 764739, 603778, 911715, 853572, 576898, 72578, 712579, 857922, 395811, 645730, 800836,
    910179, 366178, 906339, 647492, 918114, 725570, 882626, 910371, 139586, 750660, 358978, 911715, 778435, 645730, 610980, 869444, 908643, 879075, 918114, 712899, 139586, 461378, 859331, 755012, 883299, 317762, 891939, 276643, 743139, 712581, 887234, 81282,
    910371, 870340, 358627, 869699, 895011, 897315, 700932, 695810, 869698, 727236, 910371, 785092, 180482, 461539, 712580, 454178, 81411, 893859, 918114, 892707, 910755, 886563, 502531, 695810, 819396, 858820, 712899, 908451, 857795, 107938, 902403, 713059,
    778435, 769220, 288386, 910466, 906339, 662980, 247298, 81220, 864580, 542658, 662979, 662979, 892707, 817220, 398181, 542658, 912483, 712578, 107172, 663140, 912386, 662498, 713061, 712581, 902403, 409380, 486018, 750660, 906243, 461378, 870596, 755012,
    893571, 887331, 610978, 869698, 527011, 545218, 892035, 395781, 495812, 806467, 800836, 912483, 542658, 817220, 871108, 870467, 610980, 81218, 107938, 750530, 859331, 663140, 81220, 695810, 743139, 869444, 358627, 418, 871108, 912291, 725572, 725572,
    81220, 857924, 912291, 910371, 910371, 107938, 409380, 907203, 884739, 81221, 542660, 100259, 906339, 906339, 486020, 725570, 673, 100260, 918114, 878275, 767299, 712579, 910179, 81218, 859331, 907491, 869572, 90467, 662500, 740323, 910466, 764612,
    907587, 895587, 569698, 859331, 859331, 461380, 912386, 693252, 317762, 911043, 673, 107172, 662500, 662498, 90467, 907491, 663140, 542660, 725762, 537858, 81411, 821059, 486018, 864580, 857922, 486020, 663140, 815299, 712578, 906339, 725700, 486018,
    486018, 358627, 486021, 910755, 358629, 107172, 712901, 513572, 910851, 880419, 107938, 857922, 712581, 366178, 785092, 910947, 288386, 81221, 912675, 662980, 662980, 764611, 461539, 800835, 910947, 819524, 712578, 870084, 712581, 764739, 461380, 743139,
    793156, 912291, 907587, 712900, 767299, 461539, 869700, 542658, 395781, 894339, 276643, 406594, 858434, 486018, 869570, 81218, 276643, 542658, 895011, 542658, 899331, 713060, 712581, 673, 545218, 461380, 542658, 712580, 107938, 883587, 486018, 911715,
    704291, 545218, 764612, 358978, 72578, 358978, 647492, 610980, 545218, 911715, 712899, 918114, 358978, 879075, 743139, 645730, 908643, 461539, 891939, 892707, 910371, 887234, 700932, 912291, 743139, 793156, 486018, 785092, 712580, 712899, 712581, 910947,
    769220, 542658, 542660, 486020, 542658, 859331, 864580, 778435, 662979, 81220, 502531, 486018, 107172, 857796, 800836, 662498, 912483, 887331, 527011, 395781, 461378, 610978, 610980, 750660, 495812, 663140, 859331, 912386, 764739, 884739, 725570, 395781,
    695810, 90467, 910371, 878275, 673, 906339, 918114, 81221, 857924, 486021, 725572, 907587, 725762, 662498, 317762, 461380, 911043, 640772, 486020, 537858, 725700, 910851, 880419, 815299, 821059, 107172, 910371, 712578, 461539, 662980, 800835, 912675,
    81218, 767299, 542658, 712900, 461380, 839748, 869698, 918114, 542658, 892035, 368578, 100259, 918114, 750660, 486018, 764740, 858434, 767299, 859331, 764740, 821060, 879075, 869828, 870467, 81218, 857795, 918114, 721603, 90467, 486020, 859331, 107938,
    887715, 911043, 892227, 892227, 645730, 81220, 713060, 879075, 100259, 3970, 713061, 764612, 764740, 90467, 743170, 883299, 454178, 858434, 800836, 899331, 918114, 910179, 821059, 542658, 725572, 870467, 454178, 869828, 882626, 857922, 869316, 400739,
    746563, 139586, 911043, 461378, 755012, 107938, 100260, 107938, 3970, 909603, 107938, 864452, 869700, 427299, 910466, 887234, 461379, 406594, 870467, 100259, 398212, 869316, 486019, 895011, 358627, 100260, 695810, 405892, 897315, 858434, 900771, 454178,
    858051, 785092, 910755, 870596, 400739, 897315, 398212, 610980, 673, 804164, 486019, 486018, 329282, 81220, 910179, 368578, 910466, 789060, 107938, 366178, 502531, 502531, 725570, 712578, 502531, 139586, 329282, 495812, 695810, 673, 673, 535301,
    81218, 910466, 663140, 893571, 785092, 640771, 545218, 645730, 276643, 918114, 81282, 870980, 643172, 857795, 662500, 879651, 750660, 545218, 906723, 486020, 139586, 329282, 3970, 869828, 662498, 542658, 7330, 870724, 523652, 886563, 527011, 663141,
    662498, 495812, 869698, 869698, 495812, 912386, 893571, 755012, 662498, 409380, 769220, 139586, 892227, 869698, 912675, 486018, 276643, 869700, 400739, 819524, 907203, 81282, 906339, 695810, 81218, 695810, 673, 891939, 898179, 918114, 409380, 329282,
    100259, 72578, 869828, 853572, 527011, 898659, 81218, 869700, 81189, 910466, 139586, 755012, 739396, 764612, 72578, 869700, 869828, 712580, 90468, 859331, 72578, 81187, 690852, 712579, 907491, 895587, 486018, 461378, 486020, 679811, 395811, 882626,
    882626, 869698, 911043, 100259, 486018, 542658, 864452, 893187, 695810, 695810, 869698, 329282, 569698, 869316, 888291, 864452, 398180, 486018, 868932, 713059, 725762, 911043, 869698, 743170, 400739, 870467, 895587, 107172, 545218, 725570, 276646, 870980,
    107938, 910466, 892227, 869699, 695810, 671330, 785092, 869698, 81219, 725570, 869700, 288386, 81221, 764611, 785092, 799300, 100259, 888291, 892035, 712580, 712580, 329282, 897891, 912675, 107938, 869700, 545218, 868932, 712580, 910851, 869698, 645730,
    81187, 897315, 712578, 869698, 486018, 870082, 107938, 712900, 7330, 100260, 645730, 841284, 81218, 819394, 869700, 817220, 750530, 911907, 107938, 894339, 368578, 545218, 72578, 803652, 276646, 911907, 869700, 461380, 857922, 542658, 804164, 839748,
    892035, 486018, 869828, 821060, 859331, 743170, 3970, 739396, 879075, 713060, 461378, 755012, 545218, 3970, 869316, 100260, 800836, 400739, 857922, 869828, 910851, 910179, 671330, 870467, 887234, 461379, 910755, 398212, 870596, 454178, 804164, 405892,
    910466, 366178, 542658, 789060, 461378, 673, 107938, 712580, 695810, 910466, 906723, 819524, 81282, 527011, 879651, 643172, 662500, 409380, 769220, 869828, 288386, 81218, 72578, 673, 461378, 906339, 882626, 898659, 910466, 869698, 911043, 90468,
    895587, 486018, 869698, 486020, 72578, 869698, 869700, 713059, 895587, 799300, 785092, 892035, 107938, 545218, 81220, 712900, 869700, 81218, 817220, 461380, 276646, 545218, 495811, 800836, 486020, 712580, 883011, 486020, 486020, 100259, 883011, 545218,
    911715, 744258, 486018, 905763, 744258, 905763, 358627, 502531, 358627, 486020, 819394, 819394, 107938, 895587, 398180, 502531, 800835, 610978, 662498, 545218, 892035, 502531, 495812, 542658, 712581, 712580, 486019, 819394, 542658, 663139, 712901, 800835,
    495812, 662498, 800836, 100259, 744258, 897891, 897891, 72578, 740292, 918114, 819396, 819396, 918114, 905763, 857922, 857922, 900771, 918114, 712580, 893571, 857922, 819396, 400739, 764740, 918114, 857922, 819396, 569698, 572098, 576898, 81411, 576898,
    912483, 846787, 764739, 81221, 81221, 495811, 461381, 864452, 905539, 405892, 81219, 535301, 405892, 276643, 542658, 610980, 400739, 858979, 858979, 869698, 662500, 746563, 712578, 704291, 576898, 576898, 662500, 883299, 81187, 911715, 750532, 869698,
    712579, 100259, 857922, 898659, 100259, 857796, 911907, 542658, 746564, 427298, 739395, 100259, 100259, 100259, 746563, 81282, 713060, 883299, 739395, 869698, 898659, 712900, 100259, 427298, 545218, 870467, 764739, 918114, 918114, 819396, 819396, 869828,
    744258, 857922, 740292, 610, 358627, 870467, 817220, 725762, 880419, 910179, 725570, 869828, 740292, 910179, 752196, 819394, 901635, 752196, 912386, 461378, 461378, 755012, 764739, 908739, 869828, 645730, 395781, 888291, 662498, 869828, 690852, 693252,
    276643, 276643, 764739, 887331, 90467, 358978, 750660, 910179, 713221, 857922, 663141, 859331, 911715, 859331, 697572, 910466, 139586, 495812, 409411, 139586, 100259, 329282, 671330, 712578, 712580, 100259, 894339, 725570, 911043, 906339, 395811, 910466,
    910466, 743139, 910563, 673, 712899, 910755, 90468, 712899, 843972, 835780, 662498, 870212, 486018, 912386, 486018, 486018, 662498, 486018, 400739, 398180, 398180, 72578, 3970, 869699, 910563, 879075, 870082, 910563, 739908, 400739, 100259, 139586,
    864580, 800836, 800836, 81187, 857924, 910563, 515971, 903075, 892035, 911043, 743748, 110051, 107938, 81187, 869698, 610978, 870082, 912483, 857922, 110051, 100259, 743170, 673, 94114, 869698, 882626, 912483, 750532, 81187, 495811, 3970, 400739,
    398180, 72578, 869699, 139586, 800836, 879075, 864580, 912386, 515971, 107938, 743748, 857922, 94114, 803652, 640772, 775620, 640772, 100259, 803652, 803652, 775620, 513571, 417698, 800836, 417698, 803652, 775620, 513571, 461539, 906339, 880419, 461539,
    911907, 81380, 72578, 892227, 910563, 107938, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 662980, 0, 0, 0, 0, 0, 0, 0, 0, 0, 767300, 317762, 897891, 0, 0, 0, 0, 454178, 746563, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 725762, 864452, 610978, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    451778, 766147, 766276, 766148, 767299, 767556, 767428, 887811, 766403, 766532, 766788, 766404, 767044, 767172, 766916, 766660, 888195, 765123, 765124, 767683, 767684, 767300, 888099, 765252, 766275, 888483, 888387, 887907, 451779, 452100, 452261, 452101,
    453540, 453861, 453701, 765380, 452420, 452581, 452901, 452421, 453221, 453381, 453061, 452741, 765764, 451780, 451781, 454020, 454021, 453541, 765636, 451941, 452260, 766020, 765892, 765508, 122275, 475780, 475941, 475781, 477220, 477541, 477381, 782020,
    476100, 476261, 476581, 476101, 476901, 477061, 476741, 476421, 782404, 474500, 474501, 477700, 477701, 477221, 782276, 474661, 475940, 782660, 782532, 782148, 122276, 122661, 122854, 122662, 124389, 124774, 124582, 474821, 123045, 123238, 123622, 123046,
    124006, 124198, 123814, 123430, 475301, 122277, 122278, 124965, 124966, 124390, 475141, 122470, 122853, 475621, 475461, 474981, 454179, 454500, 454661, 454501, 455940, 456261, 456101, 769604, 454820, 454981, 455301, 454821, 455621, 455781, 455461, 455141,
    769988, 454180, 454181, 456420, 456421, 455941, 769860, 454341, 454660, 770244, 770116, 769732, 454178, 768067, 768196, 768068, 769219, 769476, 769348, 888579, 768323, 768452, 768708, 768324, 768964, 769092, 768836, 768580, 888963, 767811, 767812, 770371,
    770372, 769220, 888867, 767940, 768195, 889155, 889059, 888675, 125156, 125541, 125734, 125542, 127269, 127654, 127462, 480101, 125925, 126118, 126502, 125926, 126886, 127078, 126694, 126310, 480581, 125157, 125158, 127845, 127846, 127270, 480421, 125350,
    125733, 480901, 480741, 480261, 125155, 478180, 478341, 478181, 479620, 479941, 479781, 782788, 478500, 478661, 478981, 478501, 479301, 479461, 479141, 478821, 783172, 477860, 477861, 481060, 481061, 479621, 783044, 478021, 478340, 783428, 783300, 782916,
    458978, 774467, 774596, 774468, 775619, 775876, 775748, 890115, 774723, 774852, 775108, 774724, 775364, 775492, 775236, 774980, 890403, 773443, 773444, 776003, 776004, 775620, 890307, 773572, 774595, 890595, 890499, 890211, 119395, 465220, 465381, 465221,
    466660, 466981, 466821, 778948, 465540, 465701, 466021, 465541, 466341, 466501, 466181, 465861, 779332, 463940, 463941, 467140, 467141, 466661, 779204, 464101, 465380, 779588, 779460, 779076, 119396, 119781, 119974, 119782, 121509, 121894, 121702, 464261,
    120165, 120358, 120742, 120166, 121126, 121318, 120934, 120550, 464741, 119397, 119398, 122085, 122086, 121510, 464581, 119590, 119973, 465061, 464901, 464421, 458979, 459300, 459461, 459301, 460740, 461061, 460901, 773700, 459620, 459781, 460101, 459621,
    460421, 460581, 460261, 459941, 774084, 458980, 458981, 461220, 461221, 460741, 773956, 459141, 459460, 774340, 774212, 773828, 481219, 481540, 481701, 481541, 482980, 483301, 483141, 783556, 481860, 482021, 482341, 481861, 482661, 482821, 482501, 482181,
    783940, 481220, 481221, 483460, 483461, 482981, 783812, 481381, 481700, 784196, 784068, 783684, 461378, 776515, 777412, 776516, 778435, 778692, 778564, 890787, 777539, 777668, 777924, 777540, 778180, 778308, 778052, 777796, 891075, 776259, 776260, 778819,
    778820, 778436, 890979, 776388, 777411, 891363, 891267, 890883, 472099, 472420, 472581, 472421, 473860, 474181, 474021, 781252, 472740, 472901, 473221, 472741, 473541, 473701, 473381, 473061, 781636, 472100, 472101, 474340, 474341, 473861, 781508, 472261,
    472580, 781892, 781764, 781380, 467299, 467620, 467781, 467621, 469060, 469381, 469221, 779716, 467940, 468101, 468421, 467941, 468741, 468901, 468581, 468261, 780100, 467300, 467301, 469540, 469541, 469061, 779972, 467461, 467780, 780356, 780228, 779844,
    469699, 470020, 470181, 470021, 471460, 471781, 471621, 780484, 470340, 470501, 470821, 470341, 471141, 471301, 470981, 470661, 780868, 469700, 469701, 471940, 471941, 471461, 780740, 469861, 470180, 781124, 780996, 780612, 483619, 483940, 484101, 483941,
    485380, 485701, 485541, 784324, 484260, 484421, 484741, 484261, 485061, 485221, 484901, 484581, 784708, 483620, 483621, 485860, 485861, 485381, 784580, 483781, 484100, 784964, 784836, 784452, 456579, 456900, 457061, 456901, 458340, 458661, 458501, 770500,
    457220, 457381, 457701, 457221, 458021, 458181, 457861, 457541, 770884, 456580, 456581, 458820, 458821, 458341, 770756, 456741, 457060, 771140, 771012, 770628, 461539, 461860, 462021, 461861, 463300, 463621, 463461, 776644, 462180, 462341, 462661, 462181,
    462981, 463141, 462821, 462501, 777028, 461540, 461541, 463780, 463781, 463301, 776900, 461701, 462020, 777284, 777156, 776772, 771522, 771779, 771908, 771780, 772931, 773188, 773060, 889539, 772035, 772164, 772420, 772036, 772676, 772804, 772548, 772292,
    889827, 771523, 771524, 773315, 773316, 772932, 889731, 771652, 771907, 890019, 889923, 889635, 168931, 512100, 512261, 512101, 513540, 513861, 513701, 789316, 512420, 512581, 512901, 512421, 513221, 513381, 513061, 512741, 789700, 510820, 510821, 514020,
    514021, 513541, 789572, 510981, 512260, 789956, 789828, 789444, 168932, 169317, 169510, 169318, 171045, 171430, 171238, 511141, 169701, 169894, 170278, 169702, 170662, 170854, 170470, 170086, 511621, 168933, 168934, 171621, 171622, 171046, 511461, 169126,
    169509, 511941, 511781, 511301, 24068, 197541, 197734, 197542, 199269, 199654, 199462, 531461, 197925, 198118, 198502, 197926, 198886, 199078, 198694, 198310, 531941, 196005, 196006, 199845, 199846, 199270, 531781, 196198, 197733, 532261, 532101, 531621,
    24069, 24518, 24743, 24519, 26534, 26983, 26759, 196390, 24966, 25191, 25639, 24967, 26087, 26311, 25863, 25415, 196966, 24070, 24071, 27206, 27207, 26535, 196774, 24295, 24742, 197350, 197158, 196582, 171812, 172197, 172390, 172198,
    173925, 174310, 174118, 516421, 172581, 172774, 173158, 172582, 173542, 173734, 173350, 172966, 516901, 171813, 171814, 174501, 174502, 173926, 516741, 172006, 172389, 517221, 517061, 516581, 171811, 514500, 514661, 514501, 515940, 516261, 516101, 790084,
    514820, 514981, 515301, 514821, 515621, 515781, 515461, 515141, 790468, 514180, 514181, 517380, 517381, 515941, 790340, 514341, 514660, 790724, 790596, 790212, 27429, 27878, 28103, 27879, 29894, 30343, 30119, 202726, 28326, 28551, 28999, 28327,
    29447, 29671, 29223, 28775, 203302, 27430, 27431, 30566, 30567, 29895, 203110, 27655, 28102, 203686, 203494, 202918, 27428, 200421, 200614, 200422, 202149, 202534, 202342, 532421, 200805, 200998, 201382, 200806, 201766, 201958, 201574, 201190,
    532901, 200037, 200038, 203877, 203878, 202150, 532741, 200230, 200613, 533221, 533061, 532581, 177571, 522180, 522341, 522181, 523620, 523941, 523781, 791620, 522500, 522661, 522981, 522501, 523301, 523461, 523141, 522821, 792004, 520900, 520901, 524100,
    524101, 523621, 791876, 521061, 522340, 792260, 792132, 791748, 20708, 184869, 185062, 184870, 186597, 186982, 186790, 527621, 185253, 185446, 185830, 185254, 186214, 186406, 186022, 185638, 528101, 183333, 183334, 187173, 187174, 186598, 527941, 183526,
    185061, 528421, 528261, 527781, 20709, 21158, 21383, 21159, 23174, 23623, 23399, 183718, 21606, 21831, 22279, 21607, 22727, 22951, 22503, 22055, 184294, 20710, 20711, 23846, 23847, 23175, 184102, 20935, 21382, 184678, 184486, 183910,
    177572, 177957, 178150, 177958, 179685, 180070, 179878, 521221, 178341, 178534, 178918, 178342, 179302, 179494, 179110, 178726, 521701, 177573, 177574, 180261, 180262, 179686, 521541, 177766, 178149, 522021, 521861, 521381, 204068, 204453, 204646, 204454,
    206181, 206566, 206374, 533381, 204837, 205030, 205414, 204838, 205798, 205990, 205606, 205222, 533861, 204069, 204070, 206757, 206758, 206182, 533701, 204262, 204645, 534181, 534021, 533541, 180451, 524580, 525701, 524581, 526980, 527301, 527141, 792388,
    525860, 526021, 526341, 525861, 526661, 526821, 526501, 526181, 792772, 524260, 524261, 527460, 527461, 526981, 792644, 524421, 525700, 793028, 792900, 792516, 193124, 193509, 193702, 193510, 195237, 195622, 195430, 530501, 193893, 194086, 194470, 193894,
    194854, 195046, 194662, 194278, 530981, 193125, 193126, 195813, 195814, 195238, 530821, 193318, 193701, 531301, 531141, 530661, 187364, 187749, 187942, 187750, 189477, 189862, 189670, 528581, 188133, 188326, 188710, 188134, 189094, 189286, 188902, 188518,
    529061, 187365, 187366, 190053, 190054, 189478, 528901, 187558, 187941, 529381, 529221, 528741, 190244, 190629, 190822, 190630, 192357, 192742, 192550, 529541, 191013, 191206, 191590, 191014, 191974, 192166, 191782, 191398, 530021, 190245, 190246, 192933,
    192934, 192358, 529861, 190438, 190821, 530341, 530181, 529701, 206948, 207333, 207526, 207334, 209061, 209446, 209254, 534341, 207717, 207910, 208294, 207718, 208678, 208870, 208486, 208102, 534821, 206949, 206950, 209637, 209638, 209062, 534661, 207142,
    207525, 535141, 534981, 534501, 174692, 175077, 175270, 175078, 176805, 177190, 176998, 517541, 175461, 175654, 176038, 175462, 176422, 176614, 176230, 175846, 518021, 174693, 174694, 177381, 177382, 176806, 517861, 174886, 175269, 518341, 518181, 517701,
    180452, 180837, 181030, 180838, 182565, 182950, 182758, 524741, 181221, 181414, 181798, 181222, 182182, 182374, 181990, 181606, 525221, 180453, 180454, 183141, 183142, 182566, 525061, 180646, 181029, 525541, 525381, 524901, 518499, 518820, 518981, 518821,
    520260, 520581, 520421, 790852, 519140, 519301, 519621, 519141, 519941, 520101, 519781, 519461, 791236, 518500, 518501, 520740, 520741, 520261, 791108, 518661, 518980, 791492, 791364, 790980, 603778, 834627, 834756, 834628, 835779, 836036, 835908, 901443,
    834883, 835012, 835268, 834884, 835524, 835652, 835396, 835140, 901827, 833603, 833604, 836163, 836164, 835780, 901731, 833732, 834755, 902115, 902019, 901539, 603779, 604100, 604261, 604101, 605540, 605861, 605701, 833860, 604420, 604581, 604901, 604421,
    605221, 605381, 605061, 604741, 834244, 603780, 603781, 606020, 606021, 605541, 834116, 603941, 604260, 834500, 834372, 833988, 229987, 627780, 627941, 627781, 629220, 629541, 629381, 850372, 628100, 628261, 628581, 628101, 628901, 629061, 628741, 628421,
    850756, 626500, 626501, 629700, 629701, 629221, 850628, 626661, 627940, 851012, 850884, 850500, 229988, 230373, 230566, 230374, 232101, 232486, 232294, 626821, 230757, 230950, 231334, 230758, 231718, 231910, 231526, 231142, 627301, 229989, 229990, 232677,
    232678, 232102, 627141, 230182, 230565, 627621, 627461, 626981, 606179, 606500, 606661, 606501, 607940, 608261, 608101, 838084, 606820, 606981, 607301, 606821, 607621, 607781, 607461, 607141, 838468, 606180, 606181, 608420, 608421, 607941, 838340, 606341,
    606660, 838724, 838596, 838212, 606178, 836547, 836676, 836548, 837699, 837956, 837828, 902211, 836803, 836932, 837188, 836804, 837444, 837572, 837316, 837060, 902595, 836291, 836292, 838851, 838852, 837700, 902499, 836420, 836675, 902787, 902691, 902307,
    232868, 233253, 233446, 233254, 234981, 235366, 235174, 632101, 233637, 233830, 234214, 233638, 234598, 234790, 234406, 234022, 632581, 232869, 232870, 235557, 235558, 234982, 632421, 233062, 233445, 632901, 632741, 632261, 232867, 630180, 630341, 630181,
    631620, 631941, 631781, 851140, 630500, 630661, 630981, 630501, 631301, 631461, 631141, 630821, 851524, 629860, 629861, 633060, 633061, 631621, 851396, 630021, 630340, 851780, 851652, 851268, 611138, 842819, 842948, 842820, 843971, 844228, 844100, 904131,
    843075, 843204, 843460, 843076, 843716, 843844, 843588, 843332, 904419, 841795, 841796, 844355, 844356, 843972, 904323, 841924, 842947, 904611, 904515, 904227, 227107, 617220, 617381, 617221, 618660, 618981, 618821, 847300, 617540, 617701, 618021, 617541,
    618341, 618501, 618181, 617861, 847684, 615940, 615941, 619140, 619141, 618661, 847556, 616101, 617380, 847940, 847812, 847428, 227108, 227493, 227686, 227494, 229221, 229606, 229414, 616261, 227877, 228070, 228454, 227878, 228838, 229030, 228646, 228262,
    616741, 227109, 227110, 229797, 229798, 229222, 616581, 227302, 227685, 617061, 616901, 616421, 611139, 611460, 611621, 611461, 612900, 613221, 613061, 842052, 611780, 611941, 612261, 611781, 612581, 612741, 612421, 612101, 842436, 611140, 611141, 613380,
    613381, 612901, 842308, 611301, 611620, 842692, 842564, 842180, 633219, 633540, 633701, 633541, 634980, 635301, 635141, 851908, 633860, 634021, 634341, 633861, 634661, 634821, 634501, 634181, 852292, 633220, 633221, 635460, 635461, 634981, 852164, 633381,
    633700, 852548, 852420, 852036, 613538, 844867, 845764, 844868, 846787, 847044, 846916, 904803, 845891, 846020, 846276, 845892, 846532, 846660, 846404, 846148, 905187, 844611, 844612, 847171, 847172, 846788, 905091, 844740, 845763, 905475, 905379, 904995,
    624099, 624420, 624581, 624421, 625860, 626181, 626021, 849604, 624740, 624901, 625221, 624741, 625541, 625701, 625381, 625061, 849988, 624100, 624101, 626340, 626341, 625861, 849860, 624261, 624580, 850244, 850116, 849732, 619299, 619620, 619781, 619621,
    621060, 621381, 621221, 848068, 619940, 620101, 620421, 619941, 620741, 620901, 620581, 620261, 848452, 619300, 619301, 621540, 621541, 621061, 848324, 619461, 619780, 848708, 848580, 848196, 621699, 622020, 622181, 622021, 623460, 623781, 623621, 848836,
    622340, 622501, 622821, 622341, 623141, 623301, 622981, 622661, 849220, 621700, 621701, 623940, 623941, 623461, 849092, 621861, 622180, 849476, 849348, 848964, 635619, 635940, 636101, 635941, 637380, 637701, 637541, 852676, 636260, 636421, 636741, 636261,
    637061, 637221, 636901, 636581, 853060, 635620, 635621, 637860, 637861, 637381, 852932, 635781, 636100, 853316, 853188, 852804, 608579, 608900, 609061, 608901, 610340, 610661, 610501, 838980, 609220, 609381, 609701, 609221, 610021, 610181, 609861, 609541,
    839364, 608580, 608581, 610820, 610821, 610341, 839236, 608741, 609060, 839620, 839492, 839108, 613539, 613860, 614021, 613861, 615300, 615621, 615461, 844996, 614180, 614341, 614661, 614181, 614981, 615141, 614821, 614501, 845380, 613540, 613541, 615780,
    615781, 615301, 845252, 613701, 614020, 845636, 845508, 845124, 610978, 840131, 840260, 840132, 841283, 841540, 841412, 902979, 840387, 840516, 840772, 840388, 841028, 841156, 840900, 840644, 903363, 839875, 839876, 841667, 841668, 841284, 903267, 840004,
    840259, 903555, 903459, 903171, 417698, 745411, 745540, 745412, 746563, 746820, 746692, 883107, 745667, 745796, 746052, 745668, 746308, 746436, 746180, 745924, 883491, 744387, 744388, 746947, 746948, 746564, 883395, 744516, 745539, 883779, 883683, 883203,
    417699, 418020, 418181, 418021, 419460, 419781, 419621, 744644, 418340, 418501, 418821, 418341, 419141, 419301, 418981, 418661, 745028, 417700, 417701, 419940, 419941, 419461, 744900, 417861, 418180, 745284, 745156, 744772, 113635, 441540, 441701, 441541,
    442980, 443301, 443141, 761412, 441860, 442021, 442341, 441861, 442661, 442821, 442501, 442181, 761796, 440260, 440261, 443460, 443461, 442981, 761668, 440421, 441700, 762052, 761924, 761540, 113636, 114021, 114214, 114022, 115749, 116134, 115942, 440581,
    114405, 114598, 114982, 114406, 115366, 115558, 115174, 114790, 441061, 113637, 113638, 116325, 116326, 115750, 440901, 113830, 114213, 441381, 441221, 440741, 420099, 420420, 420581, 420421, 421860, 422181, 422021, 748868, 420740, 420901, 421221, 420741,
    421541, 421701, 421381, 421061, 749252, 420100, 420101, 422340, 422341, 421861, 749124, 420261, 420580, 749508, 749380, 748996, 420098, 747331, 747460, 747332, 748483, 748740, 748612, 883875, 747587, 747716, 747972, 747588, 748228, 748356, 748100, 747844,
    884163, 747075, 747076, 749635, 749636, 748484, 884067, 747204, 747459, 884355, 884259, 883971, 116516, 116901, 117094, 116902, 118629, 119014, 118822, 445861, 117285, 117478, 117862, 117286, 118246, 118438, 118054, 117670, 446341, 116517, 116518, 119205,
    119206, 118630, 446181, 116710, 117093, 446661, 446501, 446021, 116515, 443940, 444101, 443941, 445380, 445701, 445541, 762180, 444260, 444421, 444741, 444261, 445061, 445221, 444901, 444581, 762564, 443620, 443621, 446820, 446821, 445381, 762436, 443781,
    444100, 762820, 762692, 762308, 424898, 753859, 753988, 753860, 755011, 755268, 755140, 885507, 754115, 754244, 754500, 754116, 754756, 754884, 754628, 754372, 885795, 752835, 752836, 755395, 755396, 755012, 885699, 752964, 753987, 885987, 885891, 885603,
    110755, 430980, 431141, 430981, 432420, 432741, 432581, 758340, 431300, 431461, 431781, 431301, 432101, 432261, 431941, 431621, 758724, 429700, 429701, 432900, 432901, 432421, 758596, 429861, 431140, 758980, 758852, 758468, 110756, 111141, 111334, 111142,
    112869, 113254, 113062, 430021, 111525, 111718, 112102, 111526, 112486, 112678, 112294, 111910, 430501, 110757, 110758, 113445, 113446, 112870, 430341, 110950, 111333, 430821, 430661, 430181, 424899, 425220, 425381, 425221, 426660, 426981, 426821, 753092,
    425540, 425701, 426021, 425541, 426341, 426501, 426181, 425861, 753476, 424900, 424901, 427140, 427141, 426661, 753348, 425061, 425380, 753732, 753604, 753220, 446979, 447300, 447461, 447301, 448740, 449061, 448901, 762948, 447620, 447781, 448101, 447621,
    448421, 448581, 448261, 447941, 763332, 446980, 446981, 449220, 449221, 448741, 763204, 447141, 447460, 763588, 763460, 763076, 427298, 755907, 756804, 755908, 757827, 758084, 757956, 886179, 756931, 757060, 757316, 756932, 757572, 757700, 757444, 757188,
    886467, 755651, 755652, 758211, 758212, 757828, 886371, 755780, 756803, 886755, 886659, 886275, 437859, 438180, 438341, 438181, 439620, 439941, 439781, 760644, 438500, 438661, 438981, 438501, 439301, 439461, 439141, 438821, 761028, 437860, 437861, 440100,
    440101, 439621, 760900, 438021, 438340, 761284, 761156, 760772, 433059, 433380, 433541, 433381, 434820, 435141, 434981, 759108, 433700, 433861, 434181, 433701, 434501, 434661, 434341, 434021, 759492, 433060, 433061, 435300, 435301, 434821, 759364, 433221,
    433540, 759748, 759620, 759236, 435459, 435780, 435941, 435781, 437220, 437541, 437381, 759876, 436100, 436261, 436581, 436101, 436901, 437061, 436741, 436421, 760260, 435460, 435461, 437700, 437701, 437221, 760132, 435621, 435940, 760516, 760388, 760004,
    449379, 449700, 449861, 449701, 451140, 451461, 451301, 763716, 450020, 450181, 450501, 450021, 450821, 450981, 450661, 450341, 764100, 449380, 449381, 451620, 451621, 451141, 763972, 449541, 449860, 764356, 764228, 763844, 422499, 422820, 422981, 422821,
    424260, 424581, 424421, 749764, 423140, 423301, 423621, 423141, 423941, 424101, 423781, 423461, 750148, 422500, 422501, 424740, 424741, 424261, 750020, 422661, 422980, 750404, 750276, 749892, 427299, 427620, 427781, 427621, 429060, 429381, 429221, 756036,
    427940, 428101, 428421, 427941, 428741, 428901, 428581, 428261, 756420, 427300, 427301, 429540, 429541, 429061, 756292, 427461, 427780, 756676, 756548, 756164, 750530, 751043, 751172, 751044, 752195, 752452, 752324, 884643, 751299, 751428, 751684, 751300,
    751940, 752068, 751812, 751556, 885027, 750787, 750788, 752579, 752580, 752196, 884931, 750916, 751171, 885219, 885123, 884835, 317731, 689380, 689541, 689381, 690820, 691141, 690981, 865092, 689700, 689861, 690181, 689701, 690501, 690661, 690341, 690021,
    865476, 688100, 688101, 691300, 691301, 690821, 865348, 688261, 689540, 865732, 865604, 865220, 317732, 318117, 318310, 318118, 319845, 320230, 320038, 688421, 318501, 318694, 319078, 318502, 319462, 319654, 319270, 318886, 688901, 317733, 317734, 320421,
    320422, 319846, 688741, 317926, 318309, 689221, 689061, 688581, 54308, 346341, 346534, 346342, 348069, 348454, 348262, 708741, 346725, 346918, 347302, 346726, 347686, 347878, 347494, 347110, 709221, 344805, 344806, 348645, 348646, 348070, 709061, 344998,
    346533, 709541, 709381, 708901, 54309, 54758, 54983, 54759, 56774, 57223, 56999, 345190, 55206, 55431, 55879, 55207, 56327, 56551, 56103, 55655, 345766, 54310, 54311, 57446, 57447, 56775, 345574, 54535, 54982, 346150, 345958, 345382,
    320612, 320997, 321190, 320998, 322725, 323110, 322918, 693701, 321381, 321574, 321958, 321382, 322342, 322534, 322150, 321766, 694181, 320613, 320614, 323301, 323302, 322726, 694021, 320806, 321189, 694501, 694341, 693861, 320611, 691780, 691941, 691781,
    693220, 693541, 693381, 865860, 692100, 692261, 692581, 692101, 692901, 693061, 692741, 692421, 866244, 691460, 691461, 694660, 694661, 693221, 866116, 691621, 691940, 866500, 866372, 865988, 57669, 58118, 58343, 58119, 60134, 60583, 60359, 351526,
    58566, 58791, 59239, 58567, 59687, 59911, 59463, 59015, 352102, 57670, 57671, 60806, 60807, 60135, 351910, 57895, 58342, 352486, 352294, 351718, 57668, 349221, 349414, 349222, 350949, 351334, 351142, 709701, 349605, 349798, 350182, 349606,
    350566, 350758, 350374, 349990, 710181, 348837, 348838, 352677, 352678, 350950, 710021, 349030, 349413, 710501, 710341, 709861, 326371, 699460, 699621, 699461, 700900, 701221, 701061, 867396, 699780, 699941, 700261, 699781, 700581, 700741, 700421, 700101,
    867780, 698180, 698181, 701380, 701381, 700901, 867652, 698341, 699620, 868036, 867908, 867524, 50948, 333669, 333862, 333670, 335397, 335782, 335590, 704901, 334053, 334246, 334630, 334054, 335014, 335206, 334822, 334438, 705381, 332133, 332134, 335973,
    335974, 335398, 705221, 332326, 333861, 705701, 705541, 705061, 50949, 51398, 51623, 51399, 53414, 53863, 53639, 332518, 51846, 52071, 52519, 51847, 52967, 53191, 52743, 52295, 333094, 50950, 50951, 54086, 54087, 53415, 332902, 51175,
    51622, 333478, 333286, 332710, 326372, 326757, 326950, 326758, 328485, 328870, 328678, 698501, 327141, 327334, 327718, 327142, 328102, 328294, 327910, 327526, 698981, 326373, 326374, 329061, 329062, 328486, 698821, 326566, 326949, 699301, 699141, 698661,
    352868, 353253, 353446, 353254, 354981, 355366, 355174, 710661, 353637, 353830, 354214, 353638, 354598, 354790, 354406, 354022, 711141, 352869, 352870, 355557, 355558, 354982, 710981, 353062, 353445, 711461, 711301, 710821, 329251, 701860, 702981, 701861,
    704260, 704581, 704421, 868164, 703140, 703301, 703621, 703141, 703941, 704101, 703781, 703461, 868548, 701540, 701541, 704740, 704741, 704261, 868420, 701701, 702980, 868804, 868676, 868292, 341924, 342309, 342502, 342310, 344037, 344422, 344230, 707781,
    342693, 342886, 343270, 342694, 343654, 343846, 343462, 343078, 708261, 341925, 341926, 344613, 344614, 344038, 708101, 342118, 342501, 708581, 708421, 707941, 336164, 336549, 336742, 336550, 338277, 338662, 338470, 705861, 336933, 337126, 337510, 336934,
    337894, 338086, 337702, 337318, 706341, 336165, 336166, 338853, 338854, 338278, 706181, 336358, 336741, 706661, 706501, 706021, 339044, 339429, 339622, 339430, 341157, 341542, 341350, 706821, 339813, 340006, 340390, 339814, 340774, 340966, 340582, 340198,
    707301, 339045, 339046, 341733, 341734, 341158, 707141, 339238, 339621, 707621, 707461, 706981, 355748, 356133, 356326, 356134, 357861, 358246, 358054, 711621, 356517, 356710, 357094, 356518, 357478, 357670, 357286, 356902, 712101, 355749, 355750, 358437,
    358438, 357862, 711941, 355942, 356325, 712421, 712261, 711781, 323492, 323877, 324070, 323878, 325605, 325990, 325798, 694821, 324261, 324454, 324838, 324262, 325222, 325414, 325030, 324646, 695301, 323493, 323494, 326181, 326182, 325606, 695141, 323686,
    324069, 695621, 695461, 694981, 329252, 329637, 329830, 329638, 331365, 331750, 331558, 702021, 330021, 330214, 330598, 330022, 330982, 331174, 330790, 330406, 702501, 329253, 329254, 331941, 331942, 331366, 702341, 329446, 329829, 702821, 702661, 702181,
    695779, 696100, 696261, 696101, 697540, 697861, 697701, 866628, 696420, 696581, 696901, 696421, 697221, 697381, 697061, 696741, 867012, 695780, 695781, 698020, 698021, 697541, 866884, 695941, 696260, 867268, 867140, 866756, 418, 794307, 794436, 794308,
    795331, 795588, 795460, 893379, 419, 794692, 794820, 794564, 795076, 795204, 794948, 420, 893763, 793283, 793284, 795715, 795716, 795332, 893667, 793412, 794435, 894051, 893955, 893475, 535459, 535780, 535941, 535781, 537220, 537541, 537381, 793540,
    536100, 536261, 536581, 536101, 536901, 537061, 536741, 536421, 793924, 535460, 535461, 537700, 537701, 537221, 793796, 535621, 535940, 794180, 794052, 793668, 212707, 559460, 559621, 559461, 560900, 561221, 561061, 810052, 559780, 559941, 560261, 559781,
    560581, 560741, 560421, 560101, 810436, 558180, 558181, 561380, 561381, 560901, 810308, 558341, 559620, 810692, 810564, 810180, 212708, 213093, 213286, 213094, 214821, 215206, 215014, 558501, 213477, 213670, 214054, 213478, 214438, 214630, 214246, 213862,
    558981, 212709, 212710, 215397, 215398, 214822, 558821, 212902, 213285, 559301, 559141, 558661, 537859, 538180, 538341, 538181, 539620, 539941, 539781, 797636, 538500, 538661, 538981, 538501, 539301, 539461, 539141, 538821, 798020, 537860, 537861, 540100,
    540101, 539621, 797892, 538021, 538340, 798276, 798148, 797764, 537858, 796099, 796228, 796100, 797251, 797508, 797380, 894147, 796355, 796484, 796740, 796356, 796996, 797124, 796868, 796612, 894531, 795843, 795844, 798403, 798404, 797252, 894435, 795972,
    796227, 894723, 894627, 894243, 215588, 215973, 216166, 215974, 217701, 218086, 217894, 563781, 216357, 216550, 216934, 216358, 217318, 217510, 217126, 216742, 564261, 215589, 215590, 218277, 218278, 217702, 564101, 215782, 216165, 564581, 564421, 563941,
    215587, 561860, 562021, 561861, 563300, 563621, 563461, 810820, 562180, 562341, 562661, 562181, 562981, 563141, 562821, 562501, 811204, 561540, 561541, 564740, 564741, 563301, 811076, 561701, 562020, 811460, 811332, 810948, 542818, 802499, 802628, 802500,
    803651, 803908, 803780, 896163, 802755, 802884, 803140, 802756, 803396, 803524, 803268, 803012, 896451, 801475, 801476, 804035, 804036, 803652, 896355, 801604, 802627, 896643, 896547, 896259, 209827, 548900, 549061, 548901, 550340, 550661, 550501, 806980,
    549220, 549381, 549701, 549221, 550021, 550181, 549861, 549541, 807364, 547620, 547621, 550820, 550821, 550341, 807236, 547781, 549060, 807620, 807492, 807108, 209828, 210213, 210406, 210214, 211941, 212326, 212134, 547941, 210597, 210790, 211174, 210598,
    211558, 211750, 211366, 210982, 548421, 209829, 209830, 212517, 212518, 211942, 548261, 210022, 210405, 548741, 548581, 548101, 542819, 543140, 543301, 543141, 544580, 544901, 544741, 801732, 543460, 543621, 543941, 543461, 544261, 544421, 544101, 543781,
    802116, 542820, 542821, 545060, 545061, 544581, 801988, 542981, 543300, 802372, 802244, 801860, 564899, 565220, 565381, 565221, 566660, 566981, 566821, 811588, 565540, 565701, 566021, 565541, 566341, 566501, 566181, 565861, 811972, 564900, 564901, 567140,
    567141, 566661, 811844, 565061, 565380, 812228, 812100, 811716, 545218, 804547, 805444, 804548, 806467, 806724, 806596, 896835, 805571, 805700, 805956, 805572, 806212, 806340, 806084, 805828, 897219, 804291, 804292, 806851, 806852, 806468, 897123, 804420,
    805443, 897507, 897411, 897027, 555779, 556100, 556261, 556101, 557540, 557861, 557701, 809284, 556420, 556581, 556901, 556421, 557221, 557381, 557061, 556741, 809668, 555780, 555781, 558020, 558021, 557541, 809540, 555941, 556260, 809924, 809796, 809412,
    550979, 551300, 551461, 551301, 552740, 553061, 552901, 807748, 551620, 551781, 552101, 551621, 552421, 552581, 552261, 551941, 808132, 550980, 550981, 553220, 553221, 552741, 808004, 551141, 551460, 808388, 808260, 807876, 553379, 553700, 553861, 553701,
    555140, 555461, 555301, 808516, 554020, 554181, 554501, 554021, 554821, 554981, 554661, 554341, 808900, 553380, 553381, 555620, 555621, 555141, 808772, 553541, 553860, 809156, 809028, 808644, 567299, 567620, 567781, 567621, 569060, 569381, 569221, 812356,
    567940, 568101, 568421, 567941, 568741, 568901, 568581, 568261, 812740, 567300, 567301, 569540, 569541, 569061, 812612, 567461, 567780, 812996, 812868, 812484, 540259, 540580, 540741, 540581, 542020, 542341, 542181, 798532, 540900, 541061, 541381, 540901,
    541701, 541861, 541541, 541221, 798916, 540260, 540261, 542500, 542501, 542021, 798788, 540421, 540740, 799172, 799044, 798660, 545219, 545540, 545701, 545541, 546980, 547301, 547141, 804676, 545860, 546021, 546341, 545861, 546661, 546821, 546501, 546181,
    805060, 545220, 545221, 547460, 547461, 546981, 804932, 545381, 545700, 805316, 805188, 804804, 542658, 799683, 799812, 799684, 800835, 801092, 800964, 894915, 799939, 800068, 800324, 799940, 800580, 800708, 800452, 800196, 895299, 799427, 799428, 801219,
    801220, 800836, 895203, 799556, 799811, 895491, 895395, 895107, 569698, 814147, 814276, 814148, 815299, 815556, 815428, 897699, 814403, 814532, 814788, 814404, 815044, 815172, 814916, 814660, 898083, 813123, 813124, 815683, 815684, 815300, 897987, 813252,
    814275, 898371, 898275, 897795, 569699, 570020, 570181, 570021, 571460, 571781, 571621, 813380, 570340, 570501, 570821, 570341, 571141, 571301, 570981, 570661, 813764, 569700, 569701, 571940, 571941, 571461, 813636, 569861, 570180, 814020, 813892, 813508,
    221347, 593540, 593701, 593541, 594980, 595301, 595141, 830532, 593860, 594021, 594341, 593861, 594661, 594821, 594501, 594181, 830916, 592260, 592261, 595460, 595461, 594981, 830788, 592421, 593700, 831172, 831044, 830660, 221348, 221733, 221926, 221734,
    223461, 223846, 223654, 592581, 222117, 222310, 222694, 222118, 223078, 223270, 222886, 222502, 593061, 221349, 221350, 224037, 224038, 223462, 592901, 221542, 221925, 593381, 593221, 592741, 572099, 572420, 572581, 572421, 573860, 574181, 574021, 817604,
    572740, 572901, 573221, 572741, 573541, 573701, 573381, 573061, 817988, 572100, 572101, 574340, 574341, 573861, 817860, 572261, 572580, 818244, 818116, 817732, 572098, 816067, 816196, 816068, 817219, 817476, 817348, 898467, 816323, 816452, 816708, 816324,
    816964, 817092, 816836, 816580, 898851, 815811, 815812, 818371, 818372, 817220, 898755, 815940, 816195, 899043, 898947, 898563, 224228, 224613, 224806, 224614, 226341, 226726, 226534, 597861, 224997, 225190, 225574, 224998, 225958, 226150, 225766, 225382,
    598341, 224229, 224230, 226917, 226918, 226342, 598181, 224422, 224805, 598661, 598501, 598021, 224227, 595940, 596101, 595941, 597380, 597701, 597541, 831300, 596260, 596421, 596741, 596261, 597061, 597221, 596901, 596581, 831684, 595620, 595621, 598820,
    598821, 597381, 831556, 595781, 596100, 831940, 831812, 831428, 576898, 823107, 823236, 823108, 824259, 824516, 824388, 900195, 823363, 823492, 823748, 823364, 824004, 824132, 823876, 823620, 900483, 822083, 822084, 824643, 824644, 824260, 900387, 822212,
    823235, 900675, 900579, 900291, 218467, 582980, 583141, 582981, 584420, 584741, 584581, 827460, 583300, 583461, 583781, 583301, 584101, 584261, 583941, 583621, 827844, 581700, 581701, 584900, 584901, 584421, 827716, 581861, 583140, 828100, 827972, 827588,
    218468, 218853, 219046, 218854, 220581, 220966, 220774, 582021, 219237, 219430, 219814, 219238, 220198, 220390, 220006, 219622, 582501, 218469, 218470, 221157, 221158, 220582, 582341, 218662, 219045, 582821, 582661, 582181, 576899, 577220, 577381, 577221,
    578660, 578981, 578821, 822340, 577540, 577701, 578021, 577541, 578341, 578501, 578181, 577861, 822724, 576900, 576901, 579140, 579141, 578661, 822596, 577061, 577380, 822980, 822852, 822468, 598979, 599300, 599461, 599301, 600740, 601061, 600901, 832068,
    599620, 599781, 600101, 599621, 600421, 600581, 600261, 599941, 832452, 598980, 598981, 601220, 601221, 600741, 832324, 599141, 599460, 832708, 832580, 832196, 579298, 825027, 825924, 825028, 826947, 827204, 827076, 900867, 826051, 826180, 826436, 826052,
    826692, 826820, 826564, 826308, 901155, 824771, 824772, 827331, 827332, 826948, 901059, 824900, 825923, 901347, 901251, 900963, 589859, 590180, 590341, 590181, 591620, 591941, 591781, 829764, 590500, 590661, 590981, 590501, 591301, 591461, 591141, 590821,
    830148, 589860, 589861, 592100, 592101, 591621, 830020, 590021, 590340, 830404, 830276, 829892, 585059, 585380, 585541, 585381, 586820, 587141, 586981, 828228, 585700, 585861, 586181, 585701, 586501, 586661, 586341, 586021, 828612, 585060, 585061, 587300,
    587301, 586821, 828484, 585221, 585540, 828868, 828740, 828356, 587459, 587780, 587941, 587781, 589220, 589541, 589381, 828996, 588100, 588261, 588581, 588101, 588901, 589061, 588741, 588421, 829380, 587460, 587461, 589700, 589701, 589221, 829252, 587621,
    587940, 829636, 829508, 829124, 601379, 601700, 601861, 601701, 603140, 603461, 603301, 832836, 602020, 602181, 602501, 602021, 602821, 602981, 602661, 602341, 833220, 601380, 601381, 603620, 603621, 603141, 833092, 601541, 601860, 833476, 833348, 832964,
    574499, 574820, 574981, 574821, 576260, 576581, 576421, 818500, 575140, 575301, 575621, 575141, 575941, 576101, 575781, 575461, 818884, 574500, 574501, 576740, 576741, 576261, 818756, 574661, 574980, 819140, 819012, 818628, 579299, 579620, 579781, 579621,
    581060, 581381, 581221, 825156, 579940, 580101, 580421, 579941, 580741, 580901, 580581, 580261, 825540, 579300, 579301, 581540, 581541, 581061, 825412, 579461, 579780, 825796, 825668, 825284, 819394, 819907, 820036, 819908, 821059, 821316, 821188, 899235,
    820163, 820292, 820548, 820164, 820804, 820932, 820676, 820420, 899619, 819651, 819652, 821443, 821444, 821060, 899523, 819780, 820035, 899811, 899715, 899427, 358978, 720451, 720580, 720452, 721603, 721860, 721732, 878595, 720707, 720836, 721092, 720708,
    721348, 721476, 721220, 720964, 878979, 719427, 719428, 721987, 721988, 721604, 878883, 719556, 720579, 879363, 879171, 878691, 358979, 359300, 359461, 359301, 360740, 361061, 360901, 719684, 359620, 359781, 360101, 359621, 360421, 360581, 360261, 359941,
    720068, 358980, 358981, 361220, 361221, 360741, 719940, 359141, 359460, 720324, 720196, 719812, 63907, 382820, 382981, 382821, 384260, 384581, 384421, 736324, 383140, 383301, 383621, 383141, 383941, 384101, 383781, 383461, 736708, 381540, 381541, 384740,
    384741, 384261, 736580, 381701, 382980, 736964, 736836, 736452, 63908, 64293, 64486, 64294, 66021, 66406, 66214, 381861, 64677, 64870, 65254, 64678, 65638, 65830, 65446, 65062, 382341, 63909, 63910, 66597, 66598, 66022, 382181, 64102,
    64485, 382661, 382501, 382021, 361379, 361700, 361861, 361701, 363140, 363461, 363301, 723908, 362020, 362181, 362501, 362021, 362821, 362981, 362661, 362341, 724292, 361380, 361381, 363620, 363621, 363141, 724164, 361541, 361860, 724548, 724420, 724036,
    361378, 722371, 722500, 722372, 723523, 723780, 723652, 879459, 722627, 722756, 723012, 722628, 723268, 723396, 723140, 722884, 879843, 722115, 722116, 724675, 724676, 723524, 879747, 722244, 722499, 880035, 879939, 879555, 66788, 67173, 67366, 67174,
    68901, 69286, 69094, 387141, 67557, 67750, 68134, 67558, 68518, 68710, 68326, 67942, 387621, 66789, 66790, 69477, 69478, 68902, 387461, 66982, 67365, 387941, 387781, 387301, 66787, 385220, 385381, 385221, 386660, 386981, 386821, 737092,
    385540, 385701, 386021, 385541, 386341, 386501, 386181, 385861, 737476, 384900, 384901, 388100, 388101, 386661, 737348, 385061, 385380, 737732, 737604, 737220, 366178, 728899, 729028, 728900, 730051, 730308, 730180, 880995, 729155, 729284, 729540, 729156,
    729796, 729924, 729668, 729412, 881283, 727875, 727876, 730435, 730436, 730052, 881187, 728004, 729027, 881475, 881379, 881091, 61027, 372260, 372421, 372261, 373700, 374021, 373861, 733252, 372580, 372741, 373061, 372581, 373381, 373541, 373221, 372901,
    733636, 370980, 370981, 374180, 374181, 373701, 733508, 371141, 372420, 733892, 733764, 733380, 61028, 61413, 61606, 61414, 63141, 63526, 63334, 371301, 61797, 61990, 62374, 61798, 62758, 62950, 62566, 62182, 371781, 61029, 61030, 63717,
    63718, 63142, 371621, 61222, 61605, 372101, 371941, 371461, 366179, 366500, 366661, 366501, 367940, 368261, 368101, 728132, 366820, 366981, 367301, 366821, 367621, 367781, 367461, 367141, 728516, 366180, 366181, 368420, 368421, 367941, 728388, 366341,
    366660, 728772, 728644, 728260, 388259, 388580, 388741, 388581, 390020, 390341, 390181, 737860, 388900, 389061, 389381, 388901, 389701, 389861, 389541, 389221, 738244, 388260, 388261, 390500, 390501, 390021, 738116, 388421, 388740, 738500, 738372, 737988,
    368578, 730819, 731716, 730820, 732739, 732996, 732868, 881571, 731843, 731972, 732228, 731844, 732484, 732612, 732356, 732100, 881859, 730563, 730564, 733123, 733124, 732740, 881763, 730692, 731715, 882051, 881955, 881667, 379139, 379460, 379621, 379461,
    380900, 381221, 381061, 735556, 379780, 379941, 380261, 379781, 380581, 380741, 380421, 380101, 735940, 379140, 379141, 381380, 381381, 380901, 735812, 379301, 379620, 736196, 736068, 735684, 374339, 374660, 374821, 374661, 376100, 376421, 376261, 734020,
    374980, 375141, 375461, 374981, 375781, 375941, 375621, 375301, 734404, 374340, 374341, 376580, 376581, 376101, 734276, 374501, 374820, 734660, 734532, 734148, 376739, 377060, 377221, 377061, 378500, 378821, 378661, 734788, 377380, 377541, 377861, 377381,
    378181, 378341, 378021, 377701, 735172, 376740, 376741, 378980, 378981, 378501, 735044, 376901, 377220, 735428, 735300, 734916, 390659, 390980, 391141, 390981, 392420, 392741, 392581, 738628, 391300, 391461, 391781, 391301, 392101, 392261, 391941, 391621,
    739012, 390660, 390661, 392900, 392901, 392421, 738884, 390821, 391140, 739268, 739140, 738756, 363779, 364100, 364261, 364101, 365540, 365861, 365701, 724804, 364420, 364581, 364901, 364421, 365221, 365381, 365061, 364741, 725188, 363780, 363781, 366020,
    366021, 365541, 725060, 363941, 364260, 725444, 725316, 724932, 368579, 368900, 369061, 368901, 370340, 370661, 370501, 730948, 369220, 369381, 369701, 369221, 370021, 370181, 369861, 369541, 731332, 368580, 368581, 370820, 370821, 370341, 731204, 368741,
    369060, 731588, 731460, 731076, 725570, 726083, 726212, 726084, 727235, 727492, 727364, 880323, 726339, 726468, 726724, 726340, 726980, 727108, 726852, 726596, 880707, 725827, 725828, 727619, 727620, 727236, 880611, 725956, 726211, 880899, 880803, 880515,
    235747, 639300, 639461, 639301, 640740, 641061, 640901, 853828, 639620, 639781, 640101, 639621, 640421, 640581, 640261, 639941, 854212, 638020, 638021, 641220, 641221, 640741, 854084, 638181, 639460, 854468, 854340, 853956, 235748, 236133, 236326, 236134,
    237861, 238246, 238054, 638341, 236517, 236710, 237094, 236518, 237478, 237670, 237286, 236902, 638821, 235749, 235750, 238437, 238438, 237862, 638661, 235942, 236325, 639141, 638981, 638501, 34148, 264357, 264550, 264358, 266085, 266470, 266278, 658661,
    264741, 264934, 265318, 264742, 265702, 265894, 265510, 265126, 659141, 262821, 262822, 266661, 266662, 266086, 658981, 263014, 264549, 659461, 659301, 658821, 34149, 34598, 34823, 34599, 36614, 37063, 36839, 263206, 35046, 35271, 35719, 35047,
    36167, 36391, 35943, 35495, 263782, 34150, 34151, 37286, 37287, 36615, 263590, 34375, 34822, 264166, 263974, 263398, 238628, 239013, 239206, 239014, 240741, 241126, 240934, 643621, 239397, 239590, 239974, 239398, 240358, 240550, 240166, 239782,
    644101, 238629, 238630, 241317, 241318, 240742, 643941, 238822, 239205, 644421, 644261, 643781, 238627, 641700, 641861, 641701, 643140, 643461, 643301, 854596, 642020, 642181, 642501, 642021, 642821, 642981, 642661, 642341, 854980, 641380, 641381, 644580,
    644581, 643141, 854852, 641541, 641860, 855236, 855108, 854724, 37509, 37958, 38183, 37959, 39974, 40423, 40199, 269542, 38406, 38631, 39079, 38407, 39527, 39751, 39303, 38855, 270118, 37510, 37511, 40646, 40647, 39975, 269926, 37735,
    38182, 270502, 270310, 269734, 37508, 267237, 267430, 267238, 268965, 269350, 269158, 659621, 267621, 267814, 268198, 267622, 268582, 268774, 268390, 268006, 660101, 266853, 266854, 270693, 270694, 268966, 659941, 267046, 267429, 660421, 660261, 659781,
    244387, 649380, 649541, 649381, 650820, 651141, 650981, 856132, 649700, 649861, 650181, 649701, 650501, 650661, 650341, 650021, 856516, 648100, 648101, 651300, 651301, 650821, 856388, 648261, 649540, 856772, 856644, 856260, 30788, 251685, 251878, 251686,
    253413, 253798, 253606, 654821, 252069, 252262, 252646, 252070, 253030, 253222, 252838, 252454, 655301, 250149, 250150, 253989, 253990, 253414, 655141, 250342, 251877, 655621, 655461, 654981, 30789, 31238, 31463, 31239, 33254, 33703, 33479, 250534,
    31686, 31911, 32359, 31687, 32807, 33031, 32583, 32135, 251110, 30790, 30791, 33926, 33927, 33255, 250918, 31015, 31462, 251494, 251302, 250726, 244388, 244773, 244966, 244774, 246501, 246886, 246694, 648421, 245157, 245350, 245734, 245158,
    246118, 246310, 245926, 245542, 648901, 244389, 244390, 247077, 247078, 246502, 648741, 244582, 244965, 649221, 649061, 648581, 270884, 271269, 271462, 271270, 272997, 273382, 273190, 660581, 271653, 271846, 272230, 271654, 272614, 272806, 272422, 272038,
    661061, 270885, 270886, 273573, 273574, 272998, 660901, 271078, 271461, 661381, 661221, 660741, 247267, 651780, 652901, 651781, 654180, 654501, 654341, 856900, 653060, 653221, 653541, 653061, 653861, 654021, 653701, 653381, 857284, 651460, 651461, 654660,
    654661, 654181, 857156, 651621, 652900, 857540, 857412, 857028, 259940, 260325, 260518, 260326, 262053, 262438, 262246, 657701, 260709, 260902, 261286, 260710, 261670, 261862, 261478, 261094, 658181, 259941, 259942, 262629, 262630, 262054, 658021, 260134,
    260517, 658501, 658341, 657861, 254180, 254565, 254758, 254566, 256293, 256678, 256486, 655781, 254949, 255142, 255526, 254950, 255910, 256102, 255718, 255334, 656261, 254181, 254182, 256869, 256870, 256294, 656101, 254374, 254757, 656581, 656421, 655941,
    257060, 257445, 257638, 257446, 259173, 259558, 259366, 656741, 257829, 258022, 258406, 257830, 258790, 258982, 258598, 258214, 657221, 257061, 257062, 259749, 259750, 259174, 657061, 257254, 257637, 657541, 657381, 656901, 273764, 274149, 274342, 274150,
    275877, 276262, 276070, 661541, 274533, 274726, 275110, 274534, 275494, 275686, 275302, 274918, 662021, 273765, 273766, 276453, 276454, 275878, 661861, 273958, 274341, 662341, 662181, 661701, 241508, 241893, 242086, 241894, 243621, 244006, 243814, 644741,
    242277, 242470, 242854, 242278, 243238, 243430, 243046, 242662, 645221, 241509, 241510, 244197, 244198, 243622, 645061, 241702, 242085, 645541, 645381, 644901, 247268, 247653, 247846, 247654, 249381, 249766, 249574, 651941, 248037, 248230, 248614, 248038,
    248998, 249190, 248806, 248422, 652421, 247269, 247270, 249957, 249958, 249382, 652261, 247462, 247845, 652741, 652581, 652101, 645699, 646020, 646181, 646021, 647460, 647781, 647621, 855364, 646340, 646501, 646821, 646341, 647141, 647301, 646981, 646661,
    855748, 645700, 645701, 647940, 647941, 647461, 855620, 645861, 646180, 856004, 855876, 855492, 276866, 664931, 665092, 664932, 666371, 666692, 666532, 860515, 665251, 665412, 665732, 665252, 666052, 666212, 665892, 665572, 860899, 663651, 663652, 666851,
    666852, 666372, 860771, 663812, 665091, 861155, 861027, 860643, 276867, 277252, 277445, 277253, 278980, 279365, 279173, 663972, 277636, 277829, 278213, 277637, 278597, 278789, 278405, 278021, 664452, 276868, 276869, 279556, 279557, 278981, 664292, 277061,
    277444, 664772, 664612, 664132, 44259, 305476, 305669, 305477, 307204, 307589, 307397, 684292, 305860, 306053, 306437, 305861, 306821, 307013, 306629, 306245, 684772, 303940, 303941, 307780, 307781, 307205, 684612, 304133, 305668, 685092, 684932, 684452,
    44260, 44709, 44934, 44710, 46725, 47174, 46950, 304325, 45157, 45382, 45830, 45158, 46278, 46502, 46054, 45606, 304901, 44261, 44262, 47397, 47398, 46726, 304709, 44486, 44933, 305285, 305093, 304517, 279747, 280132, 280325, 280133,
    281860, 282245, 282053, 669252, 280516, 280709, 281093, 280517, 281477, 281669, 281285, 280901, 669732, 279748, 279749, 282436, 282437, 281861, 669572, 279941, 280324, 670052, 669892, 669412, 279746, 667331, 667492, 667332, 668771, 669092, 668932, 861283,
    667651, 667812, 668132, 667652, 668452, 668612, 668292, 667972, 861667, 667011, 667012, 670211, 670212, 668772, 861539, 667172, 667491, 861923, 861795, 861411, 47620, 48069, 48294, 48070, 50085, 50534, 50310, 310661, 48517, 48742, 49190, 48518,
    49638, 49862, 49414, 48966, 311237, 47621, 47622, 50757, 50758, 50086, 311045, 47846, 48293, 311621, 311429, 310853, 47619, 308356, 308549, 308357, 310084, 310469, 310277, 685252, 308740, 308933, 309317, 308741, 309701, 309893, 309509, 309125,
    685732, 307972, 307973, 311812, 311813, 310085, 685572, 308165, 308548, 686052, 685892, 685412, 285506, 675011, 675172, 675012, 676451, 676772, 676612, 862819, 675331, 675492, 675812, 675332, 676132, 676292, 675972, 675652, 863203, 673731, 673732, 676931,
    676932, 676452, 863075, 673892, 675171, 863459, 863331, 862947, 40899, 292804, 292997, 292805, 294532, 294917, 294725, 680452, 293188, 293381, 293765, 293189, 294149, 294341, 293957, 293573, 680932, 291268, 291269, 295108, 295109, 294533, 680772, 291461,
    292996, 681252, 681092, 680612, 40900, 41349, 41574, 41350, 43365, 43814, 43590, 291653, 41797, 42022, 42470, 41798, 42918, 43142, 42694, 42246, 292229, 40901, 40902, 44037, 44038, 43366, 292037, 41126, 41573, 292613, 292421, 291845,
    285507, 285892, 286085, 285893, 287620, 288005, 287813, 674052, 286276, 286469, 286853, 286277, 287237, 287429, 287045, 286661, 674532, 285508, 285509, 288196, 288197, 287621, 674372, 285701, 286084, 674852, 674692, 674212, 312003, 312388, 312581, 312389,
    314116, 314501, 314309, 686212, 312772, 312965, 313349, 312773, 313733, 313925, 313541, 313157, 686692, 312004, 312005, 314692, 314693, 314117, 686532, 312197, 312580, 687012, 686852, 686372, 288386, 677411, 678532, 677412, 679811, 680132, 679972, 863587,
    678691, 678852, 679172, 678692, 679492, 679652, 679332, 679012, 863971, 677091, 677092, 680291, 680292, 679812, 863843, 677252, 678531, 864227, 864099, 863715, 301059, 301444, 301637, 301445, 303172, 303557, 303365, 683332, 301828, 302021, 302405, 301829,
    302789, 302981, 302597, 302213, 683812, 301060, 301061, 303748, 303749, 303173, 683652, 301253, 301636, 684132, 683972, 683492, 295299, 295684, 295877, 295685, 297412, 297797, 297605, 681412, 296068, 296261, 296645, 296069, 297029, 297221, 296837, 296453,
    681892, 295300, 295301, 297988, 297989, 297413, 681732, 295493, 295876, 682212, 682052, 681572, 298179, 298564, 298757, 298565, 300292, 300677, 300485, 682372, 298948, 299141, 299525, 298949, 299909, 300101, 299717, 299333, 682852, 298180, 298181, 300868,
    300869, 300293, 682692, 298373, 298756, 683172, 683012, 682532, 314883, 315268, 315461, 315269, 316996, 317381, 317189, 687172, 315652, 315845, 316229, 315653, 316613, 316805, 316421, 316037, 687652, 314884, 314885, 317572, 317573, 316997, 687492, 315077,
    315460, 687972, 687812, 687332, 282627, 283012, 283205, 283013, 284740, 285125, 284933, 670372, 283396, 283589, 283973, 283397, 284357, 284549, 284165, 283781, 670852, 282628, 282629, 285316, 285317, 284741, 670692, 282821, 283204, 671172, 671012, 670532,
    288387, 288772, 288965, 288773, 290500, 290885, 290693, 677572, 289156, 289349, 289733, 289157, 290117, 290309, 289925, 289541, 678052, 288388, 288389, 291076, 291077, 290501, 677892, 288581, 288964, 678372, 678212, 677732, 671330, 671651, 671812, 671652,
    673091, 673412, 673252, 862051, 671971, 672132, 672452, 671972, 672772, 672932, 672612, 672292, 862435, 671331, 671332, 673571, 673572, 673092, 862307, 671492, 671811, 862691, 862563, 862179, 276835, 664900, 665061, 664901, 666340, 666661, 666501, 860484,
    665220, 665381, 665701, 665221, 666021, 666181, 665861, 665541, 860868, 663620, 663621, 666820, 666821, 666341, 860740, 663781, 665060, 861124, 860996, 860612, 276836, 277221, 277414, 277222, 278949, 279334, 279142, 663941, 277605, 277798, 278182, 277606,
    278566, 278758, 278374, 277990, 664421, 276837, 276838, 279525, 279526, 278950, 664261, 277030, 277413, 664741, 664581, 664101, 44228, 305445, 305638, 305446, 307173, 307558, 307366, 684261, 305829, 306022, 306406, 305830, 306790, 306982, 306598, 306214,
    684741, 303909, 303910, 307749, 307750, 307174, 684581, 304102, 305637, 685061, 684901, 684421, 44229, 44678, 44903, 44679, 46694, 47143, 46919, 304294, 45126, 45351, 45799, 45127, 46247, 46471, 46023, 45575, 304870, 44230, 44231, 47366,
    47367, 46695, 304678, 44455, 44902, 305254, 305062, 304486, 279716, 280101, 280294, 280102, 281829, 282214, 282022, 669221, 280485, 280678, 281062, 280486, 281446, 281638, 281254, 280870, 669701, 279717, 279718, 282405, 282406, 281830, 669541, 279910,
    280293, 670021, 669861, 669381, 279715, 667300, 667461, 667301, 668740, 669061, 668901, 861252, 667620, 667781, 668101, 667621, 668421, 668581, 668261, 667941, 861636, 666980, 666981, 670180, 670181, 668741, 861508, 667141, 667460, 861892, 861764, 861380,
    47589, 48038, 48263, 48039, 50054, 50503, 50279, 310630, 48486, 48711, 49159, 48487, 49607, 49831, 49383, 48935, 311206, 47590, 47591, 50726, 50727, 50055, 311014, 47815, 48262, 311590, 311398, 310822, 47588, 308325, 308518, 308326,
    310053, 310438, 310246, 685221, 308709, 308902, 309286, 308710, 309670, 309862, 309478, 309094, 685701, 307941, 307942, 311781, 311782, 310054, 685541, 308134, 308517, 686021, 685861, 685381, 285475, 674980, 675141, 674981, 676420, 676741, 676581, 862788,
    675300, 675461, 675781, 675301, 676101, 676261, 675941, 675621, 863172, 673700, 673701, 676900, 676901, 676421, 863044, 673861, 675140, 863428, 863300, 862916, 40868, 292773, 292966, 292774, 294501, 294886, 294694, 680421, 293157, 293350, 293734, 293158,
    294118, 294310, 293926, 293542, 680901, 291237, 291238, 295077, 295078, 294502, 680741, 291430, 292965, 681221, 681061, 680581, 40869, 41318, 41543, 41319, 43334, 43783, 43559, 291622, 41766, 41991, 42439, 41767, 42887, 43111, 42663, 42215,
    292198, 40870, 40871, 44006, 44007, 43335, 292006, 41095, 41542, 292582, 292390, 291814, 285476, 285861, 286054, 285862, 287589, 287974, 287782, 674021, 286245, 286438, 286822, 286246, 287206, 287398, 287014, 286630, 674501, 285477, 285478, 288165,
    288166, 287590, 674341, 285670, 286053, 674821, 674661, 674181, 311972, 312357, 312550, 312358, 314085, 314470, 314278, 686181, 312741, 312934, 313318, 312742, 313702, 313894, 313510, 313126, 686661, 311973, 311974, 314661, 314662, 314086, 686501, 312166,
    312549, 686981, 686821, 686341, 288355, 677380, 678501, 677381, 679780, 680101, 679941, 863556, 678660, 678821, 679141, 678661, 679461, 679621, 679301, 678981, 863940, 677060, 677061, 680260, 680261, 679781, 863812, 677221, 678500, 864196, 864068, 863684,
    301028, 301413, 301606, 301414, 303141, 303526, 303334, 683301, 301797, 301990, 302374, 301798, 302758, 302950, 302566, 302182, 683781, 301029, 301030, 303717, 303718, 303142, 683621, 301222, 301605, 684101, 683941, 683461, 295268, 295653, 295846, 295654,
    297381, 297766, 297574, 681381, 296037, 296230, 296614, 296038, 296998, 297190, 296806, 296422, 681861, 295269, 295270, 297957, 297958, 297382, 681701, 295462, 295845, 682181, 682021, 681541, 298148, 298533, 298726, 298534, 300261, 300646, 300454, 682341,
    298917, 299110, 299494, 298918, 299878, 300070, 299686, 299302, 682821, 298149, 298150, 300837, 300838, 300262, 682661, 298342, 298725, 683141, 682981, 682501, 314852, 315237, 315430, 315238, 316965, 317350, 317158, 687141, 315621, 315814, 316198, 315622,
    316582, 316774, 316390, 316006, 687621, 314853, 314854, 317541, 317542, 316966, 687461, 315046, 315429, 687941, 687781, 687301, 282596, 282981, 283174, 282982, 284709, 285094, 284902, 670341, 283365, 283558, 283942, 283366, 284326, 284518, 284134, 283750,
    670821, 282597, 282598, 285285, 285286, 284710, 670661, 282790, 283173, 671141, 670981, 670501, 288356, 288741, 288934, 288742, 290469, 290854, 290662, 677541, 289125, 289318, 289702, 289126, 290086, 290278, 289894, 289510, 678021, 288357, 288358, 291045,
    291046, 290470, 677861, 288550, 288933, 678341, 678181, 677701, 671299, 671620, 671781, 671621, 673060, 673381, 673221, 862020, 671940, 672101, 672421, 671941, 672741, 672901, 672581, 672261, 862404, 671300, 671301, 673540, 673541, 673061, 862276, 671461,
    671780, 862660, 862532, 862148, 129, 85890, 86083, 85891, 81282, 88003, 87811, 410114, 130, 86467, 86851, 86275, 87235, 87427, 87043, 451, 410594, 84354, 84355, 88194, 88195, 81283, 410434, 84547, 86082, 410914, 410754, 410274,
    642, 1091, 1316, 1092, 3107, 3556, 3332, 84739, 1539, 1764, 2212, 1540, 2660, 2884, 2436, 1988, 85315, 643, 644, 3779, 3780, 3108, 85123, 868, 1315, 85699, 85507, 84931, 3970, 98531, 98724, 98532,
    100259, 100644, 100452, 413923, 98915, 99108, 99492, 98916, 99876, 100068, 99684, 99300, 414403, 96995, 96996, 100835, 100836, 100260, 414243, 97188, 98723, 414723, 414563, 414083, 3971, 4420, 4645, 4421, 6436, 6885, 6661, 97380,
    4868, 5093, 5541, 4869, 5989, 6213, 5765, 5317, 97956, 3972, 3973, 7108, 7109, 6437, 97764, 4197, 4644, 98340, 98148, 97572, 7362, 7811, 8036, 7812, 9827, 10276, 10052, 103747, 8259, 8484, 8932, 8260,
    9380, 9604, 9156, 8708, 104323, 7363, 7364, 10499, 10500, 9828, 104131, 7588, 8035, 104707, 104515, 103939, 673, 1122, 1347, 1123, 3138, 3587, 3363, 84770, 1570, 1795, 2243, 1571, 2691, 2915, 2467, 2019,
    85346, 674, 675, 3810, 3811, 3139, 85154, 899, 1346, 85730, 85538, 84962, 7331, 7780, 8005, 7781, 9796, 10245, 10021, 103716, 8228, 8453, 8901, 8229, 9349, 9573, 9125, 8677, 104292, 7332, 7333, 10468,
    10469, 9797, 104100, 7557, 8004, 104676, 104484, 103908, 7330, 101411, 101604, 101412, 103139, 103524, 103332, 414883, 101795, 101988, 102372, 101796, 102756, 102948, 102564, 102180, 415363, 101027, 101028, 104867, 104868, 103140, 415203, 101220,
    101603, 415683, 415523, 415043, 7393, 7842, 8067, 7843, 9858, 10307, 10083, 103778, 8290, 8515, 8963, 8291, 9411, 9635, 9187, 8739, 104354, 7394, 7395, 10530, 10531, 9859, 104162, 7619, 8066, 104738, 104546, 103970,
    610, 85859, 86052, 85860, 87587, 87972, 87780, 410083, 86243, 86436, 86820, 86244, 87204, 87396, 87012, 86628, 410563, 84323, 84324, 88163, 88164, 87588, 410403, 84516, 86051, 410883, 410723, 410243, 611, 1060, 1285, 1061,
    3076, 3525, 3301, 84708, 1508, 1733, 2181, 1509, 2629, 2853, 2405, 1957, 85284, 612, 613, 3748, 3749, 3077, 85092, 837, 1284, 85668, 85476, 84900, 78370, 78755, 78948, 78756, 80483, 80868, 80676, 403523,
    79139, 79332, 79716, 79140, 80100, 80292, 79908, 79524, 404003, 78371, 78372, 81059, 81060, 80484, 403843, 78564, 78947, 404323, 404163, 403683, 105058, 105443, 105636, 105444, 107171, 107556, 107364, 415843, 105827, 106020, 106404, 105828,
    106788, 106980, 106596, 106212, 416323, 105059, 105060, 107747, 107748, 107172, 416163, 105252, 105635, 416643, 416483, 416003, 75521, 75906, 76099, 75907, 77634, 78019, 77827, 399874, 76290, 76483, 76867, 76291, 77251, 77443, 77059, 76675,
    400354, 75522, 75523, 78210, 78211, 77635, 400194, 75715, 76098, 400674, 400514, 400034, 94114, 94499, 94692, 94500, 96227, 96612, 96420, 412963, 94883, 95076, 95460, 94884, 95844, 96036, 95652, 95268, 413443, 94115, 94116, 96803,
    96804, 96228, 413283, 94308, 94691, 413763, 413603, 413123, 88354, 88739, 88932, 88740, 90467, 90852, 90660, 411043, 89123, 89316, 89700, 89124, 90084, 90276, 89892, 89508, 411523, 88355, 88356, 91043, 91044, 90468, 411363, 88548,
    88931, 411843, 411683, 411203, 91234, 91619, 91812, 91620, 93347, 93732, 93540, 412003, 92003, 92196, 92580, 92004, 92964, 93156, 92772, 92388, 412483, 91235, 91236, 93923, 93924, 93348, 412323, 91428, 91811, 412803, 412643, 412163,
    107938, 108323, 108516, 108324, 110051, 110436, 110244, 416803, 108707, 108900, 109284, 108708, 109668, 109860, 109476, 109092, 417283, 107939, 107940, 110627, 110628, 110052, 417123, 108132, 108515, 417603, 417443, 416963, 75490, 75875, 76068, 75876,
    77603, 77988, 77796, 399843, 76259, 76452, 76836, 76260, 77220, 77412, 77028, 76644, 400323, 75491, 75492, 78179, 78180, 77604, 400163, 75684, 76067, 400643, 400483, 400003, 81442, 81827, 82020, 81828, 83555, 83940, 83748, 407203,
    82211, 82404, 82788, 82212, 83172, 83364, 82980, 82596, 407683, 81443, 81444, 84131, 84132, 83556, 407523, 81636, 82019, 408003, 407843, 407363, 81473, 81858, 82051, 81859, 83586, 83971, 83779, 407234, 82242, 82435, 82819, 82243,
    83203, 83395, 83011, 82627, 407714, 81474, 81475, 84162, 84163, 83587, 407554, 81667, 82050, 408034, 407874, 407394, 128066, 487651, 487812, 487652, 489091, 489412, 489252, 785251, 487971, 488132, 488452, 487972, 488772, 488932, 488612, 488292,
    785635, 486371, 486372, 489571, 489572, 489092, 785507, 486532, 487811, 785891, 785763, 785379, 128067, 128452, 128645, 128453, 130180, 130565, 130373, 486692, 128836, 129029, 129413, 128837, 129797, 129989, 129605, 129221, 487172, 128068, 128069, 130756,
    130757, 130181, 487012, 128261, 128644, 487492, 487332, 486852, 14019, 156676, 156869, 156677, 158404, 158789, 158597, 507012, 157060, 157253, 157637, 157061, 158021, 158213, 157829, 157445, 507492, 155140, 155141, 158980, 158981, 158405, 507332, 155333,
    156868, 507812, 507652, 507172, 14020, 14469, 14694, 14470, 16485, 16934, 16710, 155525, 14917, 15142, 15590, 14918, 16038, 16262, 15814, 15366, 156101, 14021, 14022, 17157, 17158, 16486, 155909, 14246, 14693, 156485, 156293, 155717,
    130947, 131332, 131525, 131333, 133060, 133445, 133253, 491972, 131716, 131909, 132293, 131717, 132677, 132869, 132485, 132101, 492452, 130948, 130949, 133636, 133637, 133061, 492292, 131141, 131524, 492772, 492612, 492132, 130946, 490051, 490212, 490052,
    491491, 491812, 491652, 786019, 490371, 490532, 490852, 490372, 491172, 491332, 491012, 490692, 786403, 489731, 489732, 492931, 492932, 491492, 786275, 489892, 490211, 786659, 786531, 786147, 17380, 17829, 18054, 17830, 19845, 20294, 20070, 161861,
    18277, 18502, 18950, 18278, 19398, 19622, 19174, 18726, 162437, 17381, 17382, 20517, 20518, 19846, 162245, 17606, 18053, 162821, 162629, 162053, 17379, 159556, 159749, 159557, 161284, 161669, 161477, 507972, 159940, 160133, 160517, 159941,
    160901, 161093, 160709, 160325, 508452, 159172, 159173, 163012, 163013, 161285, 508292, 159365, 159748, 508772, 508612, 508132, 136706, 497731, 497892, 497732, 499171, 499492, 499332, 787555, 498051, 498212, 498532, 498052, 498852, 499012, 498692, 498372,
    787939, 496451, 496452, 499651, 499652, 499172, 787811, 496612, 497891, 788195, 788067, 787683, 10659, 144004, 144197, 144005, 145732, 146117, 145925, 503172, 144388, 144581, 144965, 144389, 145349, 145541, 145157, 144773, 503652, 142468, 142469, 146308,
    146309, 145733, 503492, 142661, 144196, 503972, 503812, 503332, 10660, 11109, 11334, 11110, 13125, 13574, 13350, 142853, 11557, 11782, 12230, 11558, 12678, 12902, 12454, 12006, 143429, 10661, 10662, 13797, 13798, 13126, 143237, 10886,
    11333, 143813, 143621, 143045, 136707, 137092, 137285, 137093, 138820, 139205, 139013, 496772, 137476, 137669, 138053, 137477, 138437, 138629, 138245, 137861, 497252, 136708, 136709, 139396, 139397, 138821, 497092, 136901, 137284, 497572, 497412, 496932,
    163203, 163588, 163781, 163589, 165316, 165701, 165509, 508932, 163972, 164165, 164549, 163973, 164933, 165125, 164741, 164357, 509412, 163204, 163205, 165892, 165893, 165317, 509252, 163397, 163780, 509732, 509572, 509092, 139586, 500131, 501252, 500132,
    502531, 502852, 502692, 788323, 501411, 501572, 501892, 501412, 502212, 502372, 502052, 501732, 788707, 499811, 499812, 503011, 503012, 502532, 788579, 499972, 501251, 788963, 788835, 788451, 152259, 152644, 152837, 152645, 154372, 154757, 154565, 506052,
    153028, 153221, 153605, 153029, 153989, 154181, 153797, 153413, 506532, 152260, 152261, 154948, 154949, 154373, 506372, 152453, 152836, 506852, 506692, 506212, 146499, 146884, 147077, 146885, 148612, 148997, 148805, 504132, 147268, 147461, 147845, 147269,
    148229, 148421, 148037, 147653, 504612, 146500, 146501, 149188, 149189, 148613, 504452, 146693, 147076, 504932, 504772, 504292, 149379, 149764, 149957, 149765, 151492, 151877, 151685, 505092, 150148, 150341, 150725, 150149, 151109, 151301, 150917, 150533,
    505572, 149380, 149381, 152068, 152069, 151493, 505412, 149573, 149956, 505892, 505732, 505252, 166083, 166468, 166661, 166469, 168196, 168581, 168389, 509892, 166852, 167045, 167429, 166853, 167813, 168005, 167621, 167237, 510372, 166084, 166085, 168772,
    168773, 168197, 510212, 166277, 166660, 510692, 510532, 510052, 133827, 134212, 134405, 134213, 135940, 136325, 136133, 493092, 134596, 134789, 135173, 134597, 135557, 135749, 135365, 134981, 493572, 133828, 133829, 136516, 136517, 135941, 493412, 134021,
    134404, 493892, 493732, 493252, 139587, 139972, 140165, 139973, 141700, 142085, 141893, 500292, 140356, 140549, 140933, 140357, 141317, 141509, 141125, 140741, 500772, 139588, 139589, 142276, 142277, 141701, 500612, 139781, 140164, 501092, 500932, 500452,
    486018, 494371, 494532, 494372, 495811, 496132, 495972, 786787, 494691, 494852, 495172, 494692, 495492, 495652, 495332, 495012, 787171, 494051, 494052, 496291, 496292, 495812, 787043, 494212, 494531, 787427, 787299, 786915, 128035, 487620, 487781, 487621,
    489060, 489381, 489221, 785220, 487940, 488101, 488421, 487941, 488741, 488901, 488581, 488261, 785604, 486340, 486341, 489540, 489541, 489061, 785476, 486501, 487780, 785860, 785732, 785348, 128036, 128421, 128614, 128422, 130149, 130534, 130342, 486661,
    128805, 128998, 129382, 128806, 129766, 129958, 129574, 129190, 487141, 128037, 128038, 130725, 130726, 130150, 486981, 128230, 128613, 487461, 487301, 486821, 13988, 156645, 156838, 156646, 158373, 158758, 158566, 506981, 157029, 157222, 157606, 157030,
    157990, 158182, 157798, 157414, 507461, 155109, 155110, 158949, 158950, 158374, 507301, 155302, 156837, 507781, 507621, 507141, 13989, 14438, 14663, 14439, 16454, 16903, 16679, 155494, 14886, 15111, 15559, 14887, 16007, 16231, 15783, 15335,
    156070, 13990, 13991, 17126, 17127, 16455, 155878, 14215, 14662, 156454, 156262, 155686, 130916, 131301, 131494, 131302, 133029, 133414, 133222, 491941, 131685, 131878, 132262, 131686, 132646, 132838, 132454, 132070, 492421, 130917, 130918, 133605,
    133606, 133030, 492261, 131110, 131493, 492741, 492581, 492101, 130915, 490020, 490181, 490021, 491460, 491781, 491621, 785988, 490340, 490501, 490821, 490341, 491141, 491301, 490981, 490661, 786372, 489700, 489701, 492900, 492901, 491461, 786244, 489861,
    490180, 786628, 786500, 786116, 17349, 17798, 18023, 17799, 19814, 20263, 20039, 161830, 18246, 18471, 18919, 18247, 19367, 19591, 19143, 18695, 162406, 17350, 17351, 20486, 20487, 19815, 162214, 17575, 18022, 162790, 162598, 162022,
    17348, 159525, 159718, 159526, 161253, 161638, 161446, 507941, 159909, 160102, 160486, 159910, 160870, 161062, 160678, 160294, 508421, 159141, 159142, 162981, 162982, 161254, 508261, 159334, 159717, 508741, 508581, 508101, 136675, 497700, 497861, 497701,
    499140, 499461, 499301, 787524, 498020, 498181, 498501, 498021, 498821, 498981, 498661, 498341, 787908, 496420, 496421, 499620, 499621, 499141, 787780, 496581, 497860, 788164, 788036, 787652, 10628, 143973, 144166, 143974, 145701, 146086, 145894, 503141,
    144357, 144550, 144934, 144358, 145318, 145510, 145126, 144742, 503621, 142437, 142438, 146277, 146278, 145702, 503461, 142630, 144165, 503941, 503781, 503301, 10629, 11078, 11303, 11079, 13094, 13543, 13319, 142822, 11526, 11751, 12199, 11527,
    12647, 12871, 12423, 11975, 143398, 10630, 10631, 13766, 13767, 13095, 143206, 10855, 11302, 143782, 143590, 143014, 136676, 137061, 137254, 137062, 138789, 139174, 138982, 496741, 137445, 137638, 138022, 137446, 138406, 138598, 138214, 137830,
    497221, 136677, 136678, 139365, 139366, 138790, 497061, 136870, 137253, 497541, 497381, 496901, 163172, 163557, 163750, 163558, 165285, 165670, 165478, 508901, 163941, 164134, 164518, 163942, 164902, 165094, 164710, 164326, 509381, 163173, 163174, 165861,
    165862, 165286, 509221, 163366, 163749, 509701, 509541, 509061, 139555, 500100, 501221, 500101, 502500, 502821, 502661, 788292, 501380, 501541, 501861, 501381, 502181, 502341, 502021, 501701, 788676, 499780, 499781, 502980, 502981, 502501, 788548, 499941,
    501220, 788932, 788804, 788420, 152228, 152613, 152806, 152614, 154341, 154726, 154534, 506021, 152997, 153190, 153574, 152998, 153958, 154150, 153766, 153382, 506501, 152229, 152230, 154917, 154918, 154342, 506341, 152422, 152805, 506821, 506661, 506181,
    146468, 146853, 147046, 146854, 148581, 148966, 148774, 504101, 147237, 147430, 147814, 147238, 148198, 148390, 148006, 147622, 504581, 146469, 146470, 149157, 149158, 148582, 504421, 146662, 147045, 504901, 504741, 504261, 149348, 149733, 149926, 149734,
    151461, 151846, 151654, 505061, 150117, 150310, 150694, 150118, 151078, 151270, 150886, 150502, 505541, 149349, 149350, 152037, 152038, 151462, 505381, 149542, 149925, 505861, 505701, 505221, 166052, 166437, 166630, 166438, 168165, 168550, 168358, 509861,
    166821, 167014, 167398, 166822, 167782, 167974, 167590, 167206, 510341, 166053, 166054, 168741, 168742, 168166, 510181, 166246, 166629, 510661, 510501, 510021, 133796, 134181, 134374, 134182, 135909, 136294, 136102, 493061, 134565, 134758, 135142, 134566,
    135526, 135718, 135334, 134950, 493541, 133797, 133798, 136485, 136486, 135910, 493381, 133990, 134373, 493861, 493701, 493221, 139556, 139941, 140134, 139942, 141669, 142054, 141862, 500261, 140325, 140518, 140902, 140326, 141286, 141478, 141094, 140710,
    500741, 139557, 139558, 142245, 142246, 141670, 500581, 139750, 140133, 501061, 500901, 500421, 494019, 494340, 494501, 494341, 495780, 496101, 495941, 786756, 494660, 494821, 495141, 494661, 495461, 495621, 495301, 494981, 787140, 494020, 494021, 496260,
    496261, 495781, 787012, 494181, 494500, 787396, 787268, 786884, 69667, 394340, 394501, 394341, 395780, 396101, 395941, 739652, 394660, 394821, 395141, 394661, 395461, 395621, 395301, 394981, 740164, 393060, 393061, 396260, 396261, 395781, 740036, 393221,
    394500, 740548, 740420, 739780, 69668, 70053, 70246, 70054, 71781, 72166, 71974, 393381, 70437, 70630, 71014, 70438, 71398, 71590, 71206, 70822, 393861, 69669, 69670, 72357, 72358, 71782, 393701, 69862, 70245, 394181, 394021, 393541,
    3908, 98469, 98662, 98470, 100197, 100582, 100390, 413861, 98853, 99046, 99430, 98854, 99814, 100006, 99622, 99238, 414341, 96933, 96934, 100773, 100774, 100198, 414181, 97126, 98661, 414661, 414501, 414021, 3909, 4358, 4583, 4359,
    6374, 6823, 6599, 97318, 4806, 5031, 5479, 4807, 5927, 6151, 5703, 5255, 97894, 3910, 3911, 7046, 7047, 6375, 97702, 4135, 4582, 98278, 98086, 97510, 72548, 72933, 73126, 72934, 74661, 75046, 74854, 398661,
    73317, 73510, 73894, 73318, 74278, 74470, 74086, 73702, 399141, 72549, 72550, 75237, 75238, 74662, 398981, 72742, 73125, 399461, 399301, 398821, 72547, 396740, 396901, 396741, 398180, 398501, 398341, 740676, 397060, 397221, 397541, 397061,
    397861, 398021, 397701, 397381, 741060, 396420, 396421, 399620, 399621, 398181, 740932, 396581, 396900, 741316, 741188, 740804, 7269, 7718, 7943, 7719, 9734, 10183, 9959, 103654, 8166, 8391, 8839, 8167, 9287, 9511, 9063, 8615,
    104230, 7270, 7271, 10406, 10407, 9735, 104038, 7495, 7942, 104614, 104422, 103846, 7268, 101349, 101542, 101350, 103077, 103462, 103270, 414821, 101733, 101926, 102310, 101734, 102694, 102886, 102502, 102118, 415301, 100965, 100966, 104805,
    104806, 103078, 415141, 101158, 101541, 415621, 415461, 414981, 78307, 404420, 404581, 404421, 405860, 406181, 406021, 742340, 404740, 404901, 405221, 404741, 405541, 405701, 405381, 405061, 742724, 403140, 403141, 406340, 406341, 405861, 742596, 403301,
    404580, 742980, 742852, 742468, 548, 85797, 85990, 85798, 87525, 87910, 87718, 410021, 86181, 86374, 86758, 86182, 87142, 87334, 86950, 86566, 410501, 84261, 84262, 88101, 88102, 87526, 410341, 84454, 85989, 410821, 410661, 410181,
    549, 998, 1223, 999, 3014, 3463, 3239, 84646, 1446, 1671, 2119, 1447, 2567, 2791, 2343, 1895, 85222, 550, 551, 3686, 3687, 3015, 85030, 775, 1222, 85606, 85414, 84838, 78308, 78693, 78886, 78694,
    80421, 80806, 80614, 403461, 79077, 79270, 79654, 79078, 80038, 80230, 79846, 79462, 403941, 78309, 78310, 80997, 80998, 80422, 403781, 78502, 78885, 404261, 404101, 403621, 104996, 105381, 105574, 105382, 107109, 107494, 107302, 415781,
    105765, 105958, 106342, 105766, 106726, 106918, 106534, 106150, 416261, 104997, 104998, 107685, 107686, 107110, 416101, 105190, 105573, 416581, 416421, 415941, 81187, 406980, 408101, 406981, 409380, 409701, 409541, 743236, 408260, 408421, 408741, 408261,
    409061, 409221, 408901, 408581, 743620, 406660, 406661, 409860, 409861, 409381, 743492, 406821, 408100, 744004, 743876, 743364, 94052, 94437, 94630, 94438, 96165, 96550, 96358, 412901, 94821, 95014, 95398, 94822, 95782, 95974, 95590, 95206,
    413381, 94053, 94054, 96741, 96742, 96166, 413221, 94246, 94629, 413701, 413541, 413061, 88292, 88677, 88870, 88678, 90405, 90790, 90598, 410981, 89061, 89254, 89638, 89062, 90022, 90214, 89830, 89446, 411461, 88293, 88294, 90981,
    90982, 90406, 411301, 88486, 88869, 411781, 411621, 411141, 91172, 91557, 91750, 91558, 93285, 93670, 93478, 411941, 91941, 92134, 92518, 91942, 92902, 93094, 92710, 92326, 412421, 91173, 91174, 93861, 93862, 93286, 412261, 91366,
    91749, 412741, 412581, 412101, 107876, 108261, 108454, 108262, 109989, 110374, 110182, 416741, 108645, 108838, 109222, 108646, 109606, 109798, 109414, 109030, 417221, 107877, 107878, 110565, 110566, 109990, 417061, 108070, 108453, 417541, 417381, 416901,
    75428, 75813, 76006, 75814, 77541, 77926, 77734, 399781, 76197, 76390, 76774, 76198, 77158, 77350, 76966, 76582, 400261, 75429, 75430, 78117, 78118, 77542, 400101, 75622, 76005, 400581, 400421, 399941, 81380, 81765, 81958, 81766,
    83493, 83878, 83686, 407141, 82149, 82342, 82726, 82150, 83110, 83302, 82918, 82534, 407621, 81381, 81382, 84069, 84070, 83494, 407461, 81574, 81957, 407941, 407781, 407301, 400739, 401060, 401221, 401061, 402500, 402821, 402661, 741572,
    401380, 401541, 401861, 401381, 402181, 402341, 402021, 401701, 741956, 400740, 400741, 402980, 402981, 402501, 741828, 400901, 401220, 742212, 742084, 741700, 168962, 512131, 512292, 512132, 513571, 513892, 513732, 789347, 512451, 512612, 512932, 512452,
    513252, 513412, 513092, 512772, 789731, 510851, 510852, 514051, 514052, 513572, 789603, 511012, 512291, 789987, 789859, 789475, 168963, 169348, 169541, 169349, 171076, 171461, 171269, 511172, 169732, 169925, 170309, 169733, 170693, 170885, 170501, 170117,
    511652, 168964, 168965, 171652, 171653, 171077, 511492, 169157, 169540, 511972, 511812, 511332, 24099, 197572, 197765, 197573, 199300, 199685, 199493, 531492, 197956, 198149, 198533, 197957, 198917, 199109, 198725, 198341, 531972, 196036, 196037, 199876,
    199877, 199301, 531812, 196229, 197764, 532292, 532132, 531652, 24100, 24549, 24774, 24550, 26565, 27014, 26790, 196421, 24997, 25222, 25670, 24998, 26118, 26342, 25894, 25446, 196997, 24101, 24102, 27237, 27238, 26566, 196805, 24326,
    24773, 197381, 197189, 196613, 171843, 172228, 172421, 172229, 173956, 174341, 174149, 516452, 172612, 172805, 173189, 172613, 173573, 173765, 173381, 172997, 516932, 171844, 171845, 174532, 174533, 173957, 516772, 172037, 172420, 517252, 517092, 516612,
    171842, 514531, 514692, 514532, 515971, 516292, 516132, 790115, 514851, 515012, 515332, 514852, 515652, 515812, 515492, 515172, 790499, 514211, 514212, 517411, 517412, 515972, 790371, 514372, 514691, 790755, 790627, 790243, 27460, 27909, 28134, 27910,
    29925, 30374, 30150, 202757, 28357, 28582, 29030, 28358, 29478, 29702, 29254, 28806, 203333, 27461, 27462, 30597, 30598, 29926, 203141, 27686, 28133, 203717, 203525, 202949, 27459, 200452, 200645, 200453, 202180, 202565, 202373, 532452,
    200836, 201029, 201413, 200837, 201797, 201989, 201605, 201221, 532932, 200068, 200069, 203908, 203909, 202181, 532772, 200261, 200644, 533252, 533092, 532612, 177602, 522211, 522372, 522212, 523651, 523972, 523812, 791651, 522531, 522692, 523012, 522532,
    523332, 523492, 523172, 522852, 792035, 520931, 520932, 524131, 524132, 523652, 791907, 521092, 522371, 792291, 792163, 791779, 20739, 184900, 185093, 184901, 186628, 187013, 186821, 527652, 185284, 185477, 185861, 185285, 186245, 186437, 186053, 185669,
    528132, 183364, 183365, 187204, 187205, 186629, 527972, 183557, 185092, 528452, 528292, 527812, 20740, 21189, 21414, 21190, 23205, 23654, 23430, 183749, 21637, 21862, 22310, 21638, 22758, 22982, 22534, 22086, 184325, 20741, 20742, 23877,
    23878, 23206, 184133, 20966, 21413, 184709, 184517, 183941, 177603, 177988, 178181, 177989, 179716, 180101, 179909, 521252, 178372, 178565, 178949, 178373, 179333, 179525, 179141, 178757, 521732, 177604, 177605, 180292, 180293, 179717, 521572, 177797,
    178180, 522052, 521892, 521412, 204099, 204484, 204677, 204485, 206212, 206597, 206405, 533412, 204868, 205061, 205445, 204869, 205829, 206021, 205637, 205253, 533892, 204100, 204101, 206788, 206789, 206213, 533732, 204293, 204676, 534212, 534052, 533572,
    180482, 524611, 525732, 524612, 527011, 527332, 527172, 792419, 525891, 526052, 526372, 525892, 526692, 526852, 526532, 526212, 792803, 524291, 524292, 527491, 527492, 527012, 792675, 524452, 525731, 793059, 792931, 792547, 193155, 193540, 193733, 193541,
    195268, 195653, 195461, 530532, 193924, 194117, 194501, 193925, 194885, 195077, 194693, 194309, 531012, 193156, 193157, 195844, 195845, 195269, 530852, 193349, 193732, 531332, 531172, 530692, 187395, 187780, 187973, 187781, 189508, 189893, 189701, 528612,
    188164, 188357, 188741, 188165, 189125, 189317, 188933, 188549, 529092, 187396, 187397, 190084, 190085, 189509, 528932, 187589, 187972, 529412, 529252, 528772, 190275, 190660, 190853, 190661, 192388, 192773, 192581, 529572, 191044, 191237, 191621, 191045,
    192005, 192197, 191813, 191429, 530052, 190276, 190277, 192964, 192965, 192389, 529892, 190469, 190852, 530372, 530212, 529732, 206979, 207364, 207557, 207365, 209092, 209477, 209285, 534372, 207748, 207941, 208325, 207749, 208709, 208901, 208517, 208133,
    534852, 206980, 206981, 209668, 209669, 209093, 534692, 207173, 207556, 535172, 535012, 534532, 174723, 175108, 175301, 175109, 176836, 177221, 177029, 517572, 175492, 175685, 176069, 175493, 176453, 176645, 176261, 175877, 518052, 174724, 174725, 177412,
    177413, 176837, 517892, 174917, 175300, 518372, 518212, 517732, 180483, 180868, 181061, 180869, 182596, 182981, 182789, 524772, 181252, 181445, 181829, 181253, 182213, 182405, 182021, 181637, 525252, 180484, 180485, 183172, 183173, 182597, 525092, 180677,
    181060, 525572, 525412, 524932, 518530, 518851, 519012, 518852, 520291, 520612, 520452, 790883, 519171, 519332, 519652, 519172, 519972, 520132, 519812, 519492, 791267, 518531, 518532, 520771, 520772, 520292, 791139, 518692, 519011, 791523, 791395, 791011,
    317762, 689411, 689572, 689412, 690851, 691172, 691012, 865123, 689731, 689892, 690212, 689732, 690532, 690692, 690372, 690052, 865507, 688131, 688132, 691331, 691332, 690852, 865379, 688292, 689571, 865763, 865635, 865251, 317763, 318148, 318341, 318149,
    319876, 320261, 320069, 688452, 318532, 318725, 319109, 318533, 319493, 319685, 319301, 318917, 688932, 317764, 317765, 320452, 320453, 319877, 688772, 317957, 318340, 689252, 689092, 688612, 54339, 346372, 346565, 346373, 348100, 348485, 348293, 708772,
    346756, 346949, 347333, 346757, 347717, 347909, 347525, 347141, 709252, 344836, 344837, 348676, 348677, 348101, 709092, 345029, 346564, 709572, 709412, 708932, 54340, 54789, 55014, 54790, 56805, 57254, 57030, 345221, 55237, 55462, 55910, 55238,
    56358, 56582, 56134, 55686, 345797, 54341, 54342, 57477, 57478, 56806, 345605, 54566, 55013, 346181, 345989, 345413, 320643, 321028, 321221, 321029, 322756, 323141, 322949, 693732, 321412, 321605, 321989, 321413, 322373, 322565, 322181, 321797,
    694212, 320644, 320645, 323332, 323333, 322757, 694052, 320837, 321220, 694532, 694372, 693892, 320642, 691811, 691972, 691812, 693251, 693572, 693412, 865891, 692131, 692292, 692612, 692132, 692932, 693092, 692772, 692452, 866275, 691491, 691492, 694691,
    694692, 693252, 866147, 691652, 691971, 866531, 866403, 866019, 57700, 58149, 58374, 58150, 60165, 60614, 60390, 351557, 58597, 58822, 59270, 58598, 59718, 59942, 59494, 59046, 352133, 57701, 57702, 60837, 60838, 60166, 351941, 57926,
    58373, 352517, 352325, 351749, 57699, 349252, 349445, 349253, 350980, 351365, 351173, 709732, 349636, 349829, 350213, 349637, 350597, 350789, 350405, 350021, 710212, 348868, 348869, 352708, 352709, 350981, 710052, 349061, 349444, 710532, 710372, 709892,
    326402, 699491, 699652, 699492, 700931, 701252, 701092, 867427, 699811, 699972, 700292, 699812, 700612, 700772, 700452, 700132, 867811, 698211, 698212, 701411, 701412, 700932, 867683, 698372, 699651, 868067, 867939, 867555, 50979, 333700, 333893, 333701,
    335428, 335813, 335621, 704932, 334084, 334277, 334661, 334085, 335045, 335237, 334853, 334469, 705412, 332164, 332165, 336004, 336005, 335429, 705252, 332357, 333892, 705732, 705572, 705092, 50980, 51429, 51654, 51430, 53445, 53894, 53670, 332549,
    51877, 52102, 52550, 51878, 52998, 53222, 52774, 52326, 333125, 50981, 50982, 54117, 54118, 53446, 332933, 51206, 51653, 333509, 333317, 332741, 326403, 326788, 326981, 326789, 328516, 328901, 328709, 698532, 327172, 327365, 327749, 327173,
    328133, 328325, 327941, 327557, 699012, 326404, 326405, 329092, 329093, 328517, 698852, 326597, 326980, 699332, 699172, 698692, 352899, 353284, 353477, 353285, 355012, 355397, 355205, 710692, 353668, 353861, 354245, 353669, 354629, 354821, 354437, 354053,
    711172, 352900, 352901, 355588, 355589, 355013, 711012, 353093, 353476, 711492, 711332, 710852, 329282, 701891, 703012, 701892, 704291, 704612, 704452, 868195, 703171, 703332, 703652, 703172, 703972, 704132, 703812, 703492, 868579, 701571, 701572, 704771,
    704772, 704292, 868451, 701732, 703011, 868835, 868707, 868323, 341955, 342340, 342533, 342341, 344068, 344453, 344261, 707812, 342724, 342917, 343301, 342725, 343685, 343877, 343493, 343109, 708292, 341956, 341957, 344644, 344645, 344069, 708132, 342149,
    342532, 708612, 708452, 707972, 336195, 336580, 336773, 336581, 338308, 338693, 338501, 705892, 336964, 337157, 337541, 336965, 337925, 338117, 337733, 337349, 706372, 336196, 336197, 338884, 338885, 338309, 706212, 336389, 336772, 706692, 706532, 706052,
    339075, 339460, 339653, 339461, 341188, 341573, 341381, 706852, 339844, 340037, 340421, 339845, 340805, 340997, 340613, 340229, 707332, 339076, 339077, 341764, 341765, 341189, 707172, 339269, 339652, 707652, 707492, 707012, 355779, 356164, 356357, 356165,
    357892, 358277, 358085, 711652, 356548, 356741, 357125, 356549, 357509, 357701, 357317, 356933, 712132, 355780, 355781, 358468, 358469, 357893, 711972, 355973, 356356, 712452, 712292, 711812, 323523, 323908, 324101, 323909, 325636, 326021, 325829, 694852,
    324292, 324485, 324869, 324293, 325253, 325445, 325061, 324677, 695332, 323524, 323525, 326212, 326213, 325637, 695172, 323717, 324100, 695652, 695492, 695012, 329283, 329668, 329861, 329669, 331396, 331781, 331589, 702052, 330052, 330245, 330629, 330053,
    331013, 331205, 330821, 330437, 702532, 329284, 329285, 331972, 331973, 331397, 702372, 329477, 329860, 702852, 702692, 702212, 695810, 696131, 696292, 696132, 697571, 697892, 697732, 866659, 696451, 696612, 696932, 696452, 697252, 697412, 697092, 696772,
    867043, 695811, 695812, 698051, 698052, 697572, 866915, 695972, 696291, 867299, 867171, 866787, 235778, 639331, 639492, 639332, 640771, 641092, 640932, 853859, 639651, 639812, 640132, 639652, 640452, 640612, 640292, 639972, 854243, 638051, 638052, 641251,
    641252, 640772, 854115, 638212, 639491, 854499, 854371, 853987, 235779, 236164, 236357, 236165, 237892, 238277, 238085, 638372, 236548, 236741, 237125, 236549, 237509, 237701, 237317, 236933, 638852, 235780, 235781, 238468, 238469, 237893, 638692, 235973,
    236356, 639172, 639012, 638532, 34179, 264388, 264581, 264389, 266116, 266501, 266309, 658692, 264772, 264965, 265349, 264773, 265733, 265925, 265541, 265157, 659172, 262852, 262853, 266692, 266693, 266117, 659012, 263045, 264580, 659492, 659332, 658852,
    34180, 34629, 34854, 34630, 36645, 37094, 36870, 263237, 35077, 35302, 35750, 35078, 36198, 36422, 35974, 35526, 263813, 34181, 34182, 37317, 37318, 36646, 263621, 34406, 34853, 264197, 264005, 263429, 238659, 239044, 239237, 239045,
    240772, 241157, 240965, 643652, 239428, 239621, 240005, 239429, 240389, 240581, 240197, 239813, 644132, 238660, 238661, 241348, 241349, 240773, 643972, 238853, 239236, 644452, 644292, 643812, 238658, 641731, 641892, 641732, 643171, 643492, 643332, 854627,
    642051, 642212, 642532, 642052, 642852, 643012, 642692, 642372, 855011, 641411, 641412, 644611, 644612, 643172, 854883, 641572, 641891, 855267, 855139, 854755, 37540, 37989, 38214, 37990, 40005, 40454, 40230, 269573, 38437, 38662, 39110, 38438,
    39558, 39782, 39334, 38886, 270149, 37541, 37542, 40677, 40678, 40006, 269957, 37766, 38213, 270533, 270341, 269765, 37539, 267268, 267461, 267269, 268996, 269381, 269189, 659652, 267652, 267845, 268229, 267653, 268613, 268805, 268421, 268037,
    660132, 266884, 266885, 270724, 270725, 268997, 659972, 267077, 267460, 660452, 660292, 659812, 244418, 649411, 649572, 649412, 650851, 651172, 651012, 856163, 649731, 649892, 650212, 649732, 650532, 650692, 650372, 650052, 856547, 648131, 648132, 651331,
    651332, 650852, 856419, 648292, 649571, 856803, 856675, 856291, 30819, 251716, 251909, 251717, 253444, 253829, 253637, 654852, 252100, 252293, 252677, 252101, 253061, 253253, 252869, 252485, 655332, 250180, 250181, 254020, 254021, 253445, 655172, 250373,
    251908, 655652, 655492, 655012, 30820, 31269, 31494, 31270, 33285, 33734, 33510, 250565, 31717, 31942, 32390, 31718, 32838, 33062, 32614, 32166, 251141, 30821, 30822, 33957, 33958, 33286, 250949, 31046, 31493, 251525, 251333, 250757,
    244419, 244804, 244997, 244805, 246532, 246917, 246725, 648452, 245188, 245381, 245765, 245189, 246149, 246341, 245957, 245573, 648932, 244420, 244421, 247108, 247109, 246533, 648772, 244613, 244996, 649252, 649092, 648612, 270915, 271300, 271493, 271301,
    273028, 273413, 273221, 660612, 271684, 271877, 272261, 271685, 272645, 272837, 272453, 272069, 661092, 270916, 270917, 273604, 273605, 273029, 660932, 271109, 271492, 661412, 661252, 660772, 247298, 651811, 652932, 651812, 654211, 654532, 654372, 856931,
    653091, 653252, 653572, 653092, 653892, 654052, 653732, 653412, 857315, 651491, 651492, 654691, 654692, 654212, 857187, 651652, 652931, 857571, 857443, 857059, 259971, 260356, 260549, 260357, 262084, 262469, 262277, 657732, 260740, 260933, 261317, 260741,
    261701, 261893, 261509, 261125, 658212, 259972, 259973, 262660, 262661, 262085, 658052, 260165, 260548, 658532, 658372, 657892, 254211, 254596, 254789, 254597, 256324, 256709, 256517, 655812, 254980, 255173, 255557, 254981, 255941, 256133, 255749, 255365,
    656292, 254212, 254213, 256900, 256901, 256325, 656132, 254405, 254788, 656612, 656452, 655972, 257091, 257476, 257669, 257477, 259204, 259589, 259397, 656772, 257860, 258053, 258437, 257861, 258821, 259013, 258629, 258245, 657252, 257092, 257093, 259780,
    259781, 259205, 657092, 257285, 257668, 657572, 657412, 656932, 273795, 274180, 274373, 274181, 275908, 276293, 276101, 661572, 274564, 274757, 275141, 274565, 275525, 275717, 275333, 274949, 662052, 273796, 273797, 276484, 276485, 275909, 661892, 273989,
    274372, 662372, 662212, 661732, 241539, 241924, 242117, 241925, 243652, 244037, 243845, 644772, 242308, 242501, 242885, 242309, 243269, 243461, 243077, 242693, 645252, 241540, 241541, 244228, 244229, 243653, 645092, 241733, 242116, 645572, 645412, 644932,
    247299, 247684, 247877, 247685, 249412, 249797, 249605, 651972, 248068, 248261, 248645, 248069, 249029, 249221, 248837, 248453, 652452, 247300, 247301, 249988, 249989, 249413, 652292, 247493, 247876, 652772, 652612, 652132, 645730, 646051, 646212, 646052,
    647491, 647812, 647652, 855395, 646371, 646532, 646852, 646372, 647172, 647332, 647012, 646692, 855779, 645731, 645732, 647971, 647972, 647492, 855651, 645892, 646211, 856035, 855907, 855523, 69698, 394371, 394532, 394372, 395811, 396132, 395972, 739683,
    394691, 394852, 395172, 394692, 395492, 395652, 395332, 395012, 740195, 393091, 393092, 396291, 396292, 395812, 740067, 393252, 394531, 740579, 740451, 739811, 69699, 70084, 70277, 70085, 71812, 72197, 72005, 393412, 70468, 70661, 71045, 70469,
    71429, 71621, 71237, 70853, 393892, 69700, 69701, 72388, 72389, 71813, 393732, 69893, 70276, 394212, 394052, 393572, 3939, 98500, 98693, 98501, 100228, 100613, 100421, 413892, 98884, 99077, 99461, 98885, 99845, 100037, 99653, 99269,
    414372, 96964, 96965, 100804, 100805, 100229, 414212, 97157, 98692, 414692, 414532, 414052, 3940, 4389, 4614, 4390, 6405, 6854, 6630, 97349, 4837, 5062, 5510, 4838, 5958, 6182, 5734, 5286, 97925, 3941, 3942, 7077,
    7078, 6406, 97733, 4166, 4613, 98309, 98117, 97541, 72579, 72964, 73157, 72965, 74692, 75077, 74885, 398692, 73348, 73541, 73925, 73349, 74309, 74501, 74117, 73733, 399172, 72580, 72581, 75268, 75269, 74693, 399012, 72773,
    73156, 399492, 399332, 398852, 72578, 396771, 396932, 396772, 398211, 398532, 398372, 740707, 397091, 397252, 397572, 397092, 397892, 398052, 397732, 397412, 741091, 396451, 396452, 399651, 399652, 398212, 740963, 396612, 396931, 741347, 741219, 740835,
    7300, 7749, 7974, 7750, 9765, 10214, 9990, 103685, 8197, 8422, 8870, 8198, 9318, 9542, 9094, 8646, 104261, 7301, 7302, 10437, 10438, 9766, 104069, 7526, 7973, 104645, 104453, 103877, 7299, 101380, 101573, 101381,
    103108, 103493, 103301, 414852, 101764, 101957, 102341, 101765, 102725, 102917, 102533, 102149, 415332, 100996, 100997, 104836, 104837, 103109, 415172, 101189, 101572, 415652, 415492, 415012, 78338, 404451, 404612, 404452, 405891, 406212, 406052, 742371,
    404771, 404932, 405252, 404772, 405572, 405732, 405412, 405092, 742755, 403171, 403172, 406371, 406372, 405892, 742627, 403332, 404611, 743011, 742883, 742499, 579, 85828, 86021, 85829, 87556, 87941, 87749, 410052, 86212, 86405, 86789, 86213,
    87173, 87365, 86981, 86597, 410532, 84292, 84293, 88132, 88133, 87557, 410372, 84485, 86020, 410852, 410692, 410212, 580, 1029, 1254, 1030, 3045, 3494, 3270, 84677, 1477, 1702, 2150, 1478, 2598, 2822, 2374, 1926,
    85253, 581, 582, 3717, 3718, 3046, 85061, 806, 1253, 85637, 85445, 84869, 78339, 78724, 78917, 78725, 80452, 80837, 80645, 403492, 79108, 79301, 79685, 79109, 80069, 80261, 79877, 79493, 403972, 78340, 78341, 81028,
    81029, 80453, 403812, 78533, 78916, 404292, 404132, 403652, 105027, 105412, 105605, 105413, 107140, 107525, 107333, 415812, 105796, 105989, 106373, 105797, 106757, 106949, 106565, 106181, 416292, 105028, 105029, 107716, 107717, 107141, 416132, 105221,
    105604, 416612, 416452, 415972, 81218, 407011, 408132, 407012, 409411, 409732, 409572, 743267, 408291, 408452, 408772, 408292, 409092, 409252, 408932, 408612, 743651, 406691, 406692, 409891, 409892, 409412, 743523, 406852, 408131, 744035, 743907, 743395,
    94083, 94468, 94661, 94469, 96196, 96581, 96389, 412932, 94852, 95045, 95429, 94853, 95813, 96005, 95621, 95237, 413412, 94084, 94085, 96772, 96773, 96197, 413252, 94277, 94660, 413732, 413572, 413092, 88323, 88708, 88901, 88709,
    90436, 90821, 90629, 411012, 89092, 89285, 89669, 89093, 90053, 90245, 89861, 89477, 411492, 88324, 88325, 91012, 91013, 90437, 411332, 88517, 88900, 411812, 411652, 411172, 91203, 91588, 91781, 91589, 93316, 93701, 93509, 411972,
    91972, 92165, 92549, 91973, 92933, 93125, 92741, 92357, 412452, 91204, 91205, 93892, 93893, 93317, 412292, 91397, 91780, 412772, 412612, 412132, 107907, 108292, 108485, 108293, 110020, 110405, 110213, 416772, 108676, 108869, 109253, 108677,
    109637, 109829, 109445, 109061, 417252, 107908, 107909, 110596, 110597, 110021, 417092, 108101, 108484, 417572, 417412, 416932, 75459, 75844, 76037, 75845, 77572, 77957, 77765, 399812, 76228, 76421, 76805, 76229, 77189, 77381, 76997, 76613,
    400292, 75460, 75461, 78148, 78149, 77573, 400132, 75653, 76036, 400612, 400452, 399972, 81411, 81796, 81989, 81797, 83524, 83909, 83717, 407172, 82180, 82373, 82757, 82181, 83141, 83333, 82949, 82565, 407652, 81412, 81413, 84100,
    84101, 83525, 407492, 81605, 81988, 407972, 407812, 407332, 400770, 401091, 401252, 401092, 402531, 402852, 402692, 741603, 401411, 401572, 401892, 401412, 402212, 402372, 402052, 401732, 741987, 400771, 400772, 403011, 403012, 402532, 741859, 400932,
    401251, 742243, 742115, 741731, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    662498, 769220, 72547, 486019, 81219, 81189, 139586, 461539, 461539, 662498, 495811, 418, 901635, 795331, 893571, 897315, 897315, 897315, 897315, 897315, 537858, 897315, 893859, 897315, 897315, 893859, 897315, 804164, 804164, 795331, 795331, 795331,
    804164, 795331, 795331, 795331, 795331, 418, 418, 418, 795332, 795332, 795332, 795332, 795332, 893571, 797252, 893859, 545218, 545218, 545218, 545218, 893859, 545218, 545218, 545218, 545218, 545218, 545218, 545218, 545218, 545218, 545218, 545218,
    545218, 806467, 803652, 843972, 803652, 803652, 893859, 894339, 545218, 894339, 894339, 545218, 896739, 894339, 896739, 894339, 545218, 896739, 894339, 537858, 800835, 800836, 797252, 800836, 800836, 800836, 427298, 603778, 537858, 905283, 746563, 841284,
    613538, 545218, 869698, 879651, 640771, 725572, 918114, 368578, 819394, 276643, 908451, 739395, 878275, 663141, 7330, 860228, 662979, 398180, 398180, 859331, 907491, 896931, 896931, 542661, 542661, 542661, 542661, 542661, 542661, 542661, 542661, 542658,
    545218, 613538, 545218, 545218, 545218, 542658, 545218, 542658, 542658, 542658, 542658, 542658, 542658, 542658, 610980, 542660, 542660, 610980, 542660, 542660, 542660, 610980, 542660, 542660, 610980, 542660, 542660, 542660, 895011, 895011, 100259, 895011,
    895011, 860228, 542660, 610980, 610980, 542660, 542660, 895011, 800836, 800836, 841284, 800836, 542660, 800836, 869700, 800836, 800836, 800836, 800836, 800836, 800836, 542658, 542658, 542658, 542658, 673, 537858, 799300, 799300, 839748, 799300, 537858,
    799300, 799300, 799300, 799300, 803652, 110051, 858436, 895587, 743108, 895587, 895587, 895587, 895587, 895587, 895587, 903651, 894339, 895587, 545218, 545218, 806467, 806467, 806467, 806467, 545218, 542658, 542658, 545218, 803652, 542658, 542658, 545218,
    869698, 542658, 542658, 610978, 542658, 542658, 542658, 542658, 542658, 542658, 542658, 610978, 610978, 800835, 800835, 800835, 800835, 800835, 800835, 800835, 800835, 800835, 800835, 542658, 542658, 542658, 358630, 870467, 859331, 663140, 69667, 882626,
    906243, 427298, 317762, 690852, 870212, 755012, 879075, 918114, 869444, 486021, 486020, 893283, 910466, 81218, 0, 0, 713221, 0, 857796, 0, 0, 712578, 358627, 869698, 542658, 663140, 712581, 918114, 495812, 495812, 107938, 0,
    906339, 0, 358627, 0, 0, 869698, 886083, 0, 0, 0, 764611, 671330, 461380, 72578, 795332, 542658, 910466, 668772, 819396, 819396, 857795, 879651, 72578, 690851, 662498, 647492, 576898, 739524, 72547, 81411, 892611, 870084,
    398181, 821059, 486018, 276643, 898659, 739939, 358627, 395811, 358627, 870340, 912771, 879651, 663139, 870467, 662498, 911715, 871234, 358627, 743779, 713060, 461378, 329282, 891939, 542660, 495811, 764611, 276643, 713059, 743108, 882435, 882435, 713059,
    72578, 859331, 7330, 495811, 727235, 870084, 743748, 869698, 835779, 712581, 647491, 81411, 0, 461380, 0, 0, 727236, 535301, 857924, 869444, 405861, 486018, 107172, 858820, 72578, 908739, 81411, 610, 713221, 764739, 100259, 723523,
    725570, 882626, 725762, 869698, 882339, 911043, 87588, 663140, 107938, 870084, 725762, 398181, 883299, 107938, 908355, 727236, 725762, 857796, 795332, 87588, 712901, 883299, 662979, 895587, 869699, 912386, 395811, 495812, 358627, 857668, 892227, 764611,
    358627, 713060, 882626, 81219, 461380, 90467, 869698, 663141, 870467, 789060, 713059, 864452, 864580, 891939, 894339, 908739, 542660, 647492, 713059, 81221, 81219, 110051, 662501, 918114, 859331, 750660, 358627, 857796, 7330, 905283, 107938, 495811,
    725572, 870084, 276643, 400739, 908355, 869444, 358627, 835779, 495812, 878787, 712581, 673, 647491, 713060, 461539, 0, 0, 0, 72578, 910563, 891939, 0, 662500, 723524, 673, 640772, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    887426, 764866, 887554, 887427, 887523, 50914, 50914, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 869698, 0, 912003,
    871489, 871489, 84801, 225, 1377, 33, 2305, 662817, 2977, 921441, 1, 1, 1, 1, 913698, 913698, 871489, 705, 1153, 84801, 225, 609, 358625, 0, 2977, 65, 1377, 1377, 33, 0, 2305, 0,
    3169, 1, 0, 2529, 2529, 0, 358625, 662497, 662817, 1, 2977, 94114, 705, 1377, 2529, 913762, 0, 0, 0, 0, 0, 0, 2529, 2529, 2529, 2529, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 869057, 869057, 869057, 869057, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 546, 546, 546, 546, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 358626, 358626, 0, 0, 1377, 1377, 1377, 1377, 1153, 1153, 1153, 1153, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 3170, 3170, 3170, 3170, 0, 0, 0, 0, 0, 0, 0, 869057, 869057,
    0, 0, 0, 0, 0, 0, 0, 0, 65, 65, 3970, 3970, 65, 65, 920738, 920738, 65, 65, 65, 65, 65, 65, 65, 65, 65, 911202, 911202, 911202, 65, 65, 65, 65,
    891906, 911138, 822050, 911202, 911202, 11714, 880130, 727747, 219394, 63906, 63906, 13058, 891426, 864707, 220546, 54338, 54338, 909027, 909123, 909411, 909411, 891618, 221122, 11042, 218818, 821731, 892899, 752739, 10594, 276642, 860355, 218434,
    276642, 218434, 142978, 884450, 752707, 582146, 891426, 220546, 918722, 0, 0, 889347, 771267, 918050, 917986, 764995, 887618, 918178, 918178, 919842, 906434, 920034, 920034, 168962, 11490, 663394, 892995, 210562, 219202, 24098, 24098, 895682,
    194, 801347, 2274, 34, 34, 12386, 819266, 663363, 219970, 221346, 221346, 3618, 3394, 903939, 819490, 229986, 229986, 11042, 218818, 3938, 3938, 891906, 911138, 911427, 822050, 911202, 911202, 884450, 662817, 65, 913506, 913442,
    98, 913506, 913442, 97, 920674, 920802, 822050, 902946, 911202, 911202, 719234, 917410, 219394, 228034, 63906, 63906, 920162, 920226, 220546, 229186, 54338, 54338, 909315, 909507, 909123, 909219, 909411, 909411, 918178, 918178, 920034, 920034,
    168962, 210562, 219202, 24098, 24098, 2274, 34, 34, 569698, 219970, 858018, 869922, 819490, 844578, 229986, 229986, 65, 920674, 920802, 822050, 902946, 911202, 911202, 891906, 911138, 911427, 822050, 911138, 11714, 880130, 727747, 219394,
    880130, 13058, 891426, 864707, 220546, 891426, 909123, 891618, 221122, 11042, 218818, 821731, 752739, 10594, 276642, 860355, 218434, 276642, 860355, 218434, 142978, 884450, 752707, 582146, 891426, 918722, 0, 0, 889347, 771267, 918050, 917986,
    764995, 887618, 919842, 906434, 11490, 663394, 892995, 210562, 219202, 895682, 194, 801347, 2274, 194, 12386, 819266, 663363, 219970, 3618, 3394, 903939, 819490, 3394, 11042, 218818, 225, 891906, 911138, 911427, 822050, 911138, 822050,
    911138, 219394, 880130, 220546, 891426, 909123, 908931, 218434, 276642, 663299, 859075, 210562, 219202, 2274, 819490, 3394, 822050, 911138, 129, 75521, 81473, 54338, 54338, 0, 0, 889443, 889443, 44258, 44258, 907779, 907779, 3938,
    3938, 14018, 14018, 727779, 727779, 44258, 44258, 113634, 113634, 859459, 859075, 859588, 663299, 907683, 920098, 920098, 917794, 54338, 54338, 0, 0, 889443, 889443, 44258, 44258, 907779, 907779, 3938, 3938, 14018, 14018, 727779,
    727779, 44258, 44258, 113634, 113634, 859459, 859075, 859588, 663299, 907683, 920098, 920098, 917794, 859459, 859075, 859588, 663299, 276642, 859075, 220546, 10594, 276642, 860355, 859459, 859075, 859588, 220546, 918722, 917282, 917282, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 909699, 909027, 909027, 909123, 864708, 909891, 819235, 864964, 892131, 892131, 221699, 221699, 859459, 907971, 908067, 908163,
    908163, 821539, 908259, 908259, 859075, 859075, 908259, 859076, 859076, 859460, 663301, 663301, 859716, 859716, 884547, 752708, 752708, 819235, 819235, 909987, 223427, 0, 0, 0, 0, 771268, 771396, 771396, 764996, 764996, 906435, 906531,
    219747, 894819, 894819, 895683, 895683, 801348, 801348, 895971, 895971, 892163, 819267, 899139, 891587, 900003, 821700, 821828, 0, 0, 821572, 891555, 771299, 220899, 902883, 221091, 221091, 903843, 904035, 904035, 911523, 911523, 727748, 909795,
    909795, 864836, 864836, 223427, 223427, 224003, 891843, 224003, 860356, 907779, 859204, 884547, 895875, 896067, 911139, 911331, 911619, 222851, 906627, 902883, 906435, 219747, 0, 222083, 903747, 821956, 895779, 893091, 895779, 903747, 891843, 859491,
    900099, 887619, 880227, 893091, 0, 908259, 860356, 903843, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 919906, 132, 719172, 819268, 2, 858308, 0, 869188, 3, 9, 296, 858564, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 914081, 914081, 719073, 876289, 921505, 913569, 913665, 0, 0, 875139, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 875138, 921473, 921473, 0, 0, 358817, 358945, 921793, 921825, 921601, 921665, 0, 0, 914146, 914210, 914145,
    914209, 0, 0, 0, 0, 0, 0, 921601, 921665, 0, 0, 0, 0, 0, 0, 0, 914081, 914081, 719073, 0, 921505, 876289, 913665, 913569, 921473, 358817, 358945, 921793, 921825, 921601, 921665, 921281,
    921377, 921409, 921441, 921473, 914145, 914209, 876353, 0, 921633, 921313, 921345, 921537, 0, 0, 0, 0, 98, 129, 913506, 0, 913442, 0, 98, 129, 913506, 75521, 913442, 81473, 97, 0, 97, 0,
    0, 129, 129, 129, 129, 609, 609, 129, 129, 65, 65, 65, 65, 129, 129, 705, 705, 705, 705, 2977, 2977, 2977, 2977, 2977, 2977, 891426, 891426, 891426, 891426, 289, 289, 289,
    289, 225, 225, 225, 225, 663394, 663394, 663394, 663394, 84801, 84801, 884450, 884450, 662817, 662817, 358625, 358625, 1, 1, 1, 1, 276642, 276642, 276642, 276642, 1, 1, 1, 1, 84801, 84801, 84801,
    84801, 2977, 2977, 2977, 2977, 358625, 358625, 358625, 358625, 0, 0, 0, 0, 771266, 771266, 771266, 771266, 764609, 764609, 764609, 764609, 662497, 662497, 662497, 662497, 1377, 1377, 1377, 1377, 33, 33, 33,
    33, 2305, 2305, 2305, 2305, 3169, 3169, 3169, 3169, 225, 225, 225, 225, 609, 609, 65, 65, 65, 65, 65, 65, 418, 418, 418, 418, 418, 418, 418, 418, 0, 0, 0,
    0, 913569, 921249, 921281, 921313, 921345, 921377, 871489, 358817, 358945, 921409, 921441, 914081, 921473, 719073, 662913, 715521, 713569, 713633, 713761, 713889, 714017, 714145, 715361, 714273, 716673, 876289, 921505, 914145, 876353, 914209, 913665,
    921537, 876481, 914529, 718785, 914849, 877185, 876673, 876865, 876897, 718945, 915329, 915553, 877793, 877345, 916257, 877281, 876993, 921569, 872577, 916673, 877729, 914497, 718913, 916225, 876737, 915265, 914881, 921601, 921633, 921665, 921697, 921729,
    921761, 129, 705, 545, 84801, 673, 764609, 1153, 225, 81473, 289, 1377, 33, 2305, 3169, 7393, 2529, 662497, 662817, 1, 2977, 75521, 869057, 609, 712577, 65, 358625, 921793, 920865, 921825, 871361, 913890,
    914018, 719073, 0, 0, 914081, 0, 94114, 920930, 921058, 921186, 920994, 921122, 913155, 913347, 913251, 871364, 0, 129, 81473, 75521, 673, 7393, 168962, 518530, 180482, 171842, 177602, 276866, 859331, 288386, 279746, 285506,
    317762, 400739, 864291, 320642, 326402, 603778, 610978, 613538, 606178, 611138, 69698, 400770, 918114, 72578, 78338, 569698, 819394, 579298, 572098, 576898, 3970, 107938, 105058, 662818, 907106, 858434, 858050, 858178, 610, 3169, 0, 0,
    0, 1153, 1378, 1154, 3169, 3618, 3394, 84801, 50946, 33, 1826, 2274, 1602, 2722, 2946, 2498, 0, 2305, 705, 30786, 0, 1, 3842, 0, 289, 10626, 546, 1377, 2977, 2529, 225, 0,
    0, 0, 129, 642, 3970, 3971, 7362, 673, 0, 0, 7331, 7330, 7393, 610, 611, 78370, 0, 0, 105058, 75521, 94114, 88354, 91234, 107938, 0, 0, 75490, 81442, 81473, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
};
const char *Encoder::Private::unidecode_text =
    "sly allh jl jlalhchwaebschwaechchwaegschwaekkchwaelbchwaelgchwaelhchwaelmchwaelp"
    "chwaelschwaeltchwaengchwaenhchwaenjchwaesschyaebschyaechchyaegschyaekkchyaelbchy"
    "aelgchyaelhchyaelmchyaelpchyaelschyaeltchyaengchyaenhchyaenjchyaesschyeobschyeoc"
    "hchyeogschyeokkchyeolbchyeolgchyeolhchyeolmchyeolpchyeolschyeoltchyeongchyeonhch"
    "yeonjchyeossjjwaebsjjwaechjjwaegsjjwaekkjjwaelbjjwaelgjjwaelhjjwaelmjjwaelpjjwae"
    "lsjjwaeltjjwaengjjwaenhjjwaenjjjwaessjjyaebsjjyaechjjyaegsjjyaekkjjyaelbjjyaelgj"
    "jyaelhjjyaelmjjyaelpjjyaelsjjyaeltjjyaengjjyaenhjjyaenjjjyaessjjyeobsjjyeochjjye"
    "ogsjjyeokkjjyeolbjjyeolgjjyeolhjjyeolmjjyeolpjjyeolsjjyeoltjjyeongjjyeonhjjyeonj"
    "jjyeosskkwaebskkwaechkkwaegskkwaekkkkwaelbkkwaelgkkwaelhkkwaelmkkwaelpkkwaelskkw"
    "aeltkkwaengkkwaenhkkwaenjkkwaesskkyaebskkyaechkkyaegskkyaekkkkyaelbkkyaelgkkyael"
    "hkkyaelmkkyaelpkkyaelskkyaeltkkyaengkkyaenhkkyaenjkkyaesskkyeobskkyeochkkyeogskk"
    "yeokkkkyeolbkkyeolgkkyeolhkkyeolmkkyeolpkkyeolskkyeoltkkyeongkkyeonhkkyeonjkkyeo"
    "ssppwaebsppwaechppwaegsppwaekkppwaelbppwaelgppwaelhppwaelmppwaelpppwaelsppwaeltp"
    "pwaengppwaenhppwaenjppwaessppyaebsppyaechppyaegsppyaekkppyaelbppyaelgppyaelhppya"
    "elmppyaelpppyaelsppyaeltppyaengppyaenhppyaenjppyaessppyeobsppyeochppyeogsppyeokk"
    "ppyeolbppyeolgppyeolhppyeolmppyeolpppyeolsppyeoltppyeongppyeonhppyeonjppyeossssw"
    "aebssswaechsswaegssswaekksswaelbsswaelgsswaelhsswaelmsswaelpsswaelssswaeltsswaen"
    "gsswaenhsswaenjsswaessssyaebsssyaechssyaegsssyaekkssyaelbssyaelgssyaelhssyaelmss"
    "yaelpssyaelsssyaeltssyaengssyaenhssyaenjssyaessssyeobsssyeochssyeogsssyeokkssyeo"
    "lbssyeolgssyeolhssyeolmssyeolpssyeolsssyeoltssyeongssyeonhssyeonjssyeossttwaebst"
    "twaechttwaegsttwaekkttwaelbttwaelgttwaelhttwaelmttwaelpttwaelsttwaeltttwaengttwa"
    "enhttwaenjttwaessttyaebsttyaechttyaegsttyaekkttyaelbttyaelgttyaelhttyaelmttyaelp"
    "ttyaelsttyaeltttyaengttyaenhttyaenjttyaessttyeobsttyeochttyeogsttyeokkttyeolbtty"
    "eolgttyeolhttyeolmttyeolpttyeolsttyeoltttyeongttyeonhttyeonjttyeossbwaebsbwaechb"
    "waegsbwaekkbwaelbbwaelgbwaelhbwaelmbwaelpbwaelsbwaeltbwaengbwaenhbwaenjbwaessbya"
    "ebsbyaechbyaegsbyaekkbyaelbbyaelgbyaelhbyaelmbyaelpbyaelsbyaeltbyaengbyaenhbyaen"
    "jbyaessbyeobsbyeochbyeogsbyeokkbyeolbbyeolgbyeolhbyeolmbyeolpbyeolsbyeoltbyeongb"
    "yeonhbyeonjbyeosschaebschaechchaegschaekkchaelbchaelgchaelhchaelmchaelpchaelscha"
    "eltchaengchaenhchaenjchaesscheobscheochcheogscheokkcheolbcheolgcheolhcheolmcheol"
    "pcheolscheoltcheongcheonhcheonjcheosscheubscheuchcheugscheukkcheulbcheulgcheulhc"
    "heulmcheulpcheulscheultcheungcheunhcheunjcheusschoebschoechchoegschoekkchoelbcho"
    "elgchoelhchoelmchoelpchoelschoeltchoengchoenhchoenjchoesschuangchuibschuichchuig"
    "schuikkchuilbchuilgchuilhchuilmchuilpchuilschuiltchuingchuinhchuinjchuisschwabsc"
    "hwachchwaedchwaehchwaejchwaemchwaepchwaetchwagschwakkchwalbchwalgchwalhchwalmchw"
    "alpchwalschwaltchwangchwanhchwanjchwasschwebschwechchwegschwekkchwelbchwelgchwel"
    "hchwelmchwelpchwelschweltchwengchwenhchwenjchwesschwibschwichchwigschwikkchwilbc"
    "hwilgchwilhchwilmchwilpchwilschwiltchwingchwinhchwinjchwisschwobschwochchwogschw"
    "okkchwolbchwolgchwolhchwolmchwolpchwolschwoltchwongchwonhchwonjchwosschyabschyac"
    "hchyaedchyaehchyaejchyaemchyaepchyaetchyagschyakkchyalbchyalgchyalhchyalmchyalpc"
    "hyalschyaltchyangchyanhchyanjchyasschyebschyechchyegschyekkchyelbchyelgchyelhchy"
    "elmchyelpchyelschyeltchyengchyenhchyenjchyeodchyeohchyeojchyeomchyeopchyeotchyes"
    "schyobschyochchyogschyokkchyolbchyolgchyolhchyolmchyolpchyolschyoltchyongchyonhc"
    "hyonjchyosschyubschyuchchyugschyukkchyulbchyulgchyulhchyulmchyulpchyulschyultchy"
    "ungchyunhchyunjchyussdwaebsdwaechdwaegsdwaekkdwaelbdwaelgdwaelhdwaelmdwaelpdwael"
    "sdwaeltdwaengdwaenhdwaenjdwaessdyaebsdyaechdyaegsdyaekkdyaelbdyaelgdyaelhdyaelmd"
    "yaelpdyaelsdyaeltdyaengdyaenhdyaenjdyaessdyeobsdyeochdyeogsdyeokkdyeolbdyeolgdye"
    "olhdyeolmdyeolpdyeolsdyeoltdyeongdyeonhdyeonjdyeossgwaebsgwaechgwaegsgwaekkgwael"
    "bgwaelggwaelhgwaelmgwaelpgwaelsgwaeltgwaenggwaenhgwaenjgwaessgyaebsgyaechgyaegsg"
    "yaekkgyaelbgyaelggyaelhgyaelmgyaelpgyaelsgyaeltgyaenggyaenhgyaenjgyaessgyeobsgye"
    "ochgyeogsgyeokkgyeolbgyeolggyeolhgyeolmgyeolpgyeolsgyeoltgyeonggyeonhgyeonjgyeos"
    "sjjaebsjjaechjjaegsjjaekkjjaelbjjaelgjjaelhjjaelmjjaelpjjaelsjjaeltjjaengjjaenhj"
    "jaenjjjaessjjeobsjjeochjjeogsjjeokkjjeolbjjeolgjjeolhjjeolmjjeolpjjeolsjjeoltjje"
    "ongjjeonhjjeonjjjeossjjeubsjjeuchjjeugsjjeukkjjeulbjjeulgjjeulhjjeulmjjeulpjjeul"
    "sjjeultjjeungjjeunhjjeunjjjeussjjoebsjjoechjjoegsjjoekkjjoelbjjoelgjjoelhjjoelmj"
    "joelpjjoelsjjoeltjjoengjjoenhjjoenjjjoessjjuibsjjuichjjuigsjjuikkjjuilbjjuilgjju"
    "ilhjjuilmjjuilpjjuilsjjuiltjjuingjjuinhjjuinjjjuissjjwabsjjwachjjwaedjjwaehjjwae"
    "jjjwaemjjwaepjjwaetjjwagsjjwakkjjwalbjjwalgjjwalhjjwalmjjwalpjjwalsjjwaltjjwangj"
    "jwanhjjwanjjjwassjjwebsjjwechjjwegsjjwekkjjwelbjjwelgjjwelhjjwelmjjwelpjjwelsjjw"
    "eltjjwengjjwenhjjwenjjjwessjjwibsjjwichjjwigsjjwikkjjwilbjjwilgjjwilhjjwilmjjwil"
    "pjjwilsjjwiltjjwingjjwinhjjwinjjjwissjjwobsjjwochjjwogsjjwokkjjwolbjjwolgjjwolhj"
    "jwolmjjwolpjjwolsjjwoltjjwongjjwonhjjwonjjjwossjjyabsjjyachjjyaedjjyaehjjyaejjjy"
    "aemjjyaepjjyaetjjyagsjjyakkjjyalbjjyalgjjyalhjjyalmjjyalpjjyalsjjyaltjjyangjjyan"
    "hjjyanjjjyassjjyebsjjyechjjyegsjjyekkjjyelbjjyelgjjyelhjjyelmjjyelpjjyelsjjyeltj"
    "jyengjjyenhjjyenjjjyeodjjyeohjjyeojjjyeomjjyeopjjyeotjjyessjjyobsjjyochjjyogsjjy"
    "okkjjyolbjjyolgjjyolhjjyolmjjyolpjjyolsjjyoltjjyongjjyonhjjyonjjjyossjjyubsjjyuc"
    "hjjyugsjjyukkjjyulbjjyulgjjyulhjjyulmjjyulpjjyulsjjyultjjyungjjyunhjjyunjjjyussk"
    "kaebskkaechkkaegskkaekkkkaelbkkaelgkkaelhkkaelmkkaelpkkaelskkaeltkkaengkkaenhkka"
    "enjkkaesskkeobskkeochkkeogskkeokkkkeolbkkeolgkkeolhkkeolmkkeolpkkeolskkeoltkkeon"
    "gkkeonhkkeonjkkeosskkeubskkeuchkkeugskkeukkkkeulbkkeulgkkeulhkkeulmkkeulpkkeulsk"
    "keultkkeungkkeunhkkeunjkkeusskkoebskkoechkkoegskkoekkkkoelbkkoelgkkoelhkkoelmkko"
    "elpkkoelskkoeltkkoengkkoenhkkoenjkkoesskkuibskkuichkkuigskkuikkkkuilbkkuilgkkuil"
    "hkkuilmkkuilpkkuilskkuiltkkuingkkuinhkkuinjkkuisskkwabskkwachkkwaedkkwaehkkwaejk"
    "kwaemkkwaepkkwaetkkwagskkwakkkkwalbkkwalgkkwalhkkwalmkkwalpkkwalskkwaltkkwangkkw"
    "anhkkwanjkkwasskkwebskkwechkkwegskkwekkkkwelbkkwelgkkwelhkkwelmkkwelpkkwelskkwel"
    "tkkwengkkwenhkkwenjkkwesskkwibskkwichkkwigskkwikkkkwilbkkwilgkkwilhkkwilmkkwilpk"
    "kwilskkwiltkkwingkkwinhkkwinjkkwisskkwobskkwochkkwogskkwokkkkwolbkkwolgkkwolhkkw"
    "olmkkwolpkkwolskkwoltkkwongkkwonhkkwonjkkwosskkyabskkyachkkyaedkkyaehkkyaejkkyae"
    "mkkyaepkkyaetkkyagskkyakkkkyalbkkyalgkkyalhkkyalmkkyalpkkyalskkyaltkkyangkkyanhk"
    "kyanjkkyasskkyebskkyechkkyegskkyekkkkyelbkkyelgkkyelhkkyelmkkyelpkkyelskkyeltkky"
    "engkkyenhkkyenjkkyeodkkyeohkkyeojkkyeomkkyeopkkyeotkkyesskkyobskkyochkkyogskkyok"
    "kkkyolbkkyolgkkyolhkkyolmkkyolpkkyolskkyoltkkyongkkyonhkkyonjkkyosskkyubskkyuchk"
    "kyugskkyukkkkyulbkkyulgkkyulhkkyulmkkyulpkkyulskkyultkkyungkkyunhkkyunjkkyusslwa"
    "ebslwaechlwaegslwaekklwaelblwaelglwaelhlwaelmlwaelplwaelslwaeltlwaenglwaenhlwaen"
    "jlwaesslyaebslyaechlyaegslyaekklyaelblyaelglyaelhlyaelmlyaelplyaelslyaeltlyaengl"
    "yaenhlyaenjlyaesslyeobslyeochlyeogslyeokklyeolblyeolglyeolhlyeolmlyeolplyeolslye"
    "oltlyeonglyeonhlyeonjlyeossmwaebsmwaechmwaegsmwaekkmwaelbmwaelgmwaelhmwaelmmwael"
    "pmwaelsmwaeltmwaengmwaenhmwaenjmwaessmyaebsmyaechmyaegsmyaekkmyaelbmyaelgmyaelhm"
    "yaelmmyaelpmyaelsmyaeltmyaengmyaenhmyaenjmyaessmyeobsmyeochmyeogsmyeokkmyeolbmye"
    "olgmyeolhmyeolmmyeolpmyeolsmyeoltmyeongmyeonhmyeonjmyeossnwaebsnwaechnwaegsnwaek"
    "knwaelbnwaelgnwaelhnwaelmnwaelpnwaelsnwaeltnwaengnwaenhnwaenjnwaessnyaebsnyaechn"
    "yaegsnyaekknyaelbnyaelgnyaelhnyaelmnyaelpnyaelsnyaeltnyaengnyaenhnyaenjnyaessnye"
    "obsnyeochnyeogsnyeokknyeolbnyeolgnyeolhnyeolmnyeolpnyeolsnyeoltnyeongnyeonhnyeon"
    "jnyeossppaebsppaechppaegsppaekkppaelbppaelgppaelhppaelmppaelpppaelsppaeltppaengp"
    "paenhppaenjppaessppeobsppeochppeogsppeokkppeolbppeolgppeolhppeolmppeolpppeolsppe"
    "oltppeongppeonhppeonjppeossppeubsppeuchppeugsppeukkppeulbppeulgppeulhppeulmppeul"
    "pppeulsppeultppeungppeunhppeunjppeussppoebsppoechppoegsppoekkppoelbppoelgppoelhp"
    "poelmppoelpppoelsppoeltppoengppoenhppoenjppoessppuibsppuichppuigsppuikkppuilbppu"
    "ilgppuilhppuilmppuilpppuilsppuiltppuingppuinhppuinjppuissppwabsppwachppwaedppwae"
    "hppwaejppwaemppwaepppwaetppwagsppwakkppwalbppwalgppwalhppwalmppwalpppwalsppwaltp"
    "pwangppwanhppwanjppwassppwebsppwechppwegsppwekkppwelbppwelgppwelhppwelmppwelpppw"
    "elsppweltppwengppwenhppwenjppwessppwibsppwichppwigsppwikkppwilbppwilgppwilhppwil"
    "mppwilpppwilsppwiltppwingppwinhppwinjppwissppwobsppwochppwogsppwokkppwolbppwolgp"
    "pwolhppwolmppwolpppwolsppwoltppwongppwonhppwonjppwossppyabsppyachppyaedppyaehppy"
    "aejppyaemppyaepppyaetppyagsppyakkppyalbppyalgppyalhppyalmppyalpppyalsppyaltppyan"
    "gppyanhppyanjppyassppyebsppyechppyegsppyekkppyelbppyelgppyelhppyelmppyelpppyelsp"
    "pyeltppyengppyenhppyenjppyeodppyeohppyeojppyeomppyeopppyeotppyessppyobsppyochppy"
    "ogsppyokkppyolbppyolgppyolhppyolmppyolpppyolsppyoltppyongppyonhppyonjppyossppyub"
    "sppyuchppyugsppyukkppyulbppyulgppyulhppyulmppyulpppyulsppyultppyungppyunhppyunjp"
    "pyussshuangssaebsssaechssaegsssaekkssaelbssaelgssaelhssaelmssaelpssaelsssaeltssa"
    "engssaenhssaenjssaesssseobssseochsseogssseokksseolbsseolgsseolhsseolmsseolpsseol"
    "ssseoltsseongsseonhsseonjsseosssseubssseuchsseugssseukksseulbsseulgsseulhsseulms"
    "seulpsseulssseultsseungsseunhsseunjsseussssoebsssoechssoegsssoekkssoelbssoelgsso"
    "elhssoelmssoelpssoelsssoeltssoengssoenhssoenjssoessssuibsssuichssuigsssuikkssuil"
    "bssuilgssuilhssuilmssuilpssuilsssuiltssuingssuinhssuinjssuisssswabssswachsswaeds"
    "swaehsswaejsswaemsswaepsswaetsswagssswakksswalbsswalgsswalhsswalmsswalpsswalsssw"
    "altsswangsswanhsswanjsswasssswebssswechsswegssswekksswelbsswelgsswelhsswelmsswel"
    "psswelsssweltsswengsswenhsswenjsswesssswibssswichsswigssswikksswilbsswilgsswilhs"
    "swilmsswilpsswilssswiltsswingsswinhsswinjsswisssswobssswochsswogssswokksswolbssw"
    "olgsswolhsswolmsswolpsswolssswoltsswongsswonhsswonjsswossssyabsssyachssyaedssyae"
    "hssyaejssyaemssyaepssyaetssyagsssyakkssyalbssyalgssyalhssyalmssyalpssyalsssyalts"
    "syangssyanhssyanjssyassssyebsssyechssyegsssyekkssyelbssyelgssyelhssyelmssyelpssy"
    "elsssyeltssyengssyenhssyenjssyeodssyeohssyeojssyeomssyeopssyeotssyessssyobsssyoc"
    "hssyogsssyokkssyolbssyolgssyolhssyolmssyolpssyolsssyoltssyongssyonhssyonjssyosss"
    "syubsssyuchssyugsssyukkssyulbssyulgssyulhssyulmssyulpssyulsssyultssyungssyunhssy"
    "unjssyussttaebsttaechttaegsttaekkttaelbttaelgttaelhttaelmttaelpttaelsttaeltttaen"
    "gttaenhttaenjttaesstteobstteochtteogstteokktteolbtteolgtteolhtteolmtteolptteolst"
    "teolttteongtteonhtteonjtteosstteubstteuchtteugstteukktteulbtteulgtteulhtteulmtte"
    "ulptteulstteulttteungtteunhtteunjtteussttoebsttoechttoegsttoekkttoelbttoelgttoel"
    "httoelmttoelpttoelsttoeltttoengttoenhttoenjttoessttuibsttuichttuigsttuikkttuilbt"
    "tuilgttuilhttuilmttuilpttuilsttuiltttuingttuinhttuinjttuissttwabsttwachttwaedttw"
    "aehttwaejttwaemttwaepttwaetttwagsttwakkttwalbttwalgttwalhttwalmttwalpttwalsttwal"
    "tttwangttwanhttwanjttwassttwebsttwechttwegsttwekkttwelbttwelgttwelhttwelmttwelpt"
    "twelsttweltttwengttwenhttwenjttwessttwibsttwichttwigsttwikkttwilbttwilgttwilhttw"
    "ilmttwilpttwilsttwiltttwingttwinhttwinjttwissttwobsttwochttwogsttwokkttwolbttwol"
    "gttwolhttwolmttwolpttwolsttwoltttwongttwonhttwonjttwossttyabsttyachttyaedttyaeht"
    "tyaejttyaemttyaepttyaetttyagsttyakkttyalbttyalgttyalhttyalmttyalpttyalsttyalttty"
    "angttyanhttyanjttyassttyebsttyechttyegsttyekkttyelbttyelgttyelhttyelmttyelpttyel"
    "sttyeltttyengttyenhttyenjttyeodttyeohttyeojttyeomttyeopttyeotttyessttyobsttyocht"
    "tyogsttyokkttyolbttyolgttyolhttyolmttyolpttyolsttyoltttyongttyonhttyonjttyosstty"
    "ubsttyuchttyugsttyukkttyulbttyulgttyulhttyulmttyulpttyulsttyultttyungttyunhttyun"
    "jttyusszhuang(cha)baebsbaechbaegsbaekkbaelbbaelgbaelhbaelmbaelpbaelsbaeltbaengba"
    "enhbaenjbaessbeobsbeochbeogsbeokkbeolbbeolgbeolhbeolmbeolpbeolsbeoltbeongbeonhbe"
    "onjbeossbeubsbeuchbeugsbeukkbeulbbeulgbeulhbeulmbeulpbeulsbeultbeungbeunhbeunjbe"
    "ussboebsboechboegsboekkboelbboelgboelhboelmboelpboelsboeltboengboenhboenjboessbu"
    "ibsbuichbuigsbuikkbuilbbuilgbuilhbuilmbuilpbuilsbuiltbuingbuinhbuinjbuissbwabsbw"
    "achbwaedbwaehbwaejbwaembwaepbwaetbwagsbwakkbwalbbwalgbwalhbwalmbwalpbwalsbwaltbw"
    "angbwanhbwanjbwassbwebsbwechbwegsbwekkbwelbbwelgbwelhbwelmbwelpbwelsbweltbwengbw"
    "enhbwenjbwessbwibsbwichbwigsbwikkbwilbbwilgbwilhbwilmbwilpbwilsbwiltbwingbwinhbw"
    "injbwissbwobsbwochbwogsbwokkbwolbbwolgbwolhbwolmbwolpbwolsbwoltbwongbwonhbwonjbw"
    "ossbyabsbyachbyaedbyaehbyaejbyaembyaepbyaetbyagsbyakkbyalbbyalgbyalhbyalmbyalpby"
    "alsbyaltbyangbyanhbyanjbyassbyebsbyechbyegsbyekkbyelbbyelgbyelhbyelmbyelpbyelsby"
    "eltbyengbyenhbyenjbyeodbyeohbyeojbyeombyeopbyeotbyessbyobsbyochbyogsbyokkbyolbby"
    "olgbyolhbyolmbyolpbyolsbyoltbyongbyonhbyonjbyossbyubsbyuchbyugsbyukkbyulbbyulgby"
    "ulhbyulmbyulpbyulsbyultbyungbyunhbyunjbyusschabschachchaedchaehchaejchaemchaepch"
    "aetchagschakkchalbchalgchalhchalmchalpchalschaltchangchanhchanjchasschebschechch"
    "egschekkchelbchelgchelhchelmchelpchelscheltchengchenhchenjcheodcheohcheojcheomch"
    "eopcheotchesscheudcheuhcheujcheumcheupcheutchibschichchigschikkchilbchilgchilhch"
    "ilmchilpchilschiltchingchinhchinjchisschobschochchoedchoehchoejchoemchoepchoetch"
    "ogschokkcholbcholgcholhcholmcholpcholscholtchongchonhchonjchosschuaichubschuchch"
    "ugschuidchuihchuijchuimchuipchuitchukkchulbchulgchulhchulmchulpchulschultchungch"
    "unhchunjchusschwadchwahchwajchwamchwapchwatchwedchwehchwejchwemchwepchwetchwidch"
    "wihchwijchwimchwipchwitchwodchwohchwojchwomchwopchwotchyadchyahchyajchyamchyapch"
    "yatchyedchyehchyejchyemchyepchyetchyodchyohchyojchyomchyopchyotchyudchyuhchyujch"
    "yumchyupchyutdaebsdaechdaegsdaekkdaelbdaelgdaelhdaelmdaelpdaelsdaeltdaengdaenhda"
    "enjdaessdeobsdeochdeogsdeokkdeolbdeolgdeolhdeolmdeolpdeolsdeoltdeongdeonhdeonjde"
    "ossdeubsdeuchdeugsdeukkdeulbdeulgdeulhdeulmdeulpdeulsdeultdeungdeunhdeunjdeussdo"
    "ebsdoechdoegsdoekkdoelbdoelgdoelhdoelmdoelpdoelsdoeltdoengdoenhdoenjdoessduibsdu"
    "ichduigsduikkduilbduilgduilhduilmduilpduilsduiltduingduinhduinjduissdwabsdwachdw"
    "aeddwaehdwaejdwaemdwaepdwaetdwagsdwakkdwalbdwalgdwalhdwalmdwalpdwalsdwaltdwangdw"
    "anhdwanjdwassdwebsdwechdwegsdwekkdwelbdwelgdwelhdwelmdwelpdwelsdweltdwengdwenhdw"
    "enjdwessdwibsdwichdwigsdwikkdwilbdwilgdwilhdwilmdwilpdwilsdwiltdwingdwinhdwinjdw"
    "issdwobsdwochdwogsdwokkdwolbdwolgdwolhdwolmdwolpdwolsdwoltdwongdwonhdwonjdwossdy"
    "absdyachdyaeddyaehdyaejdyaemdyaepdyaetdyagsdyakkdyalbdyalgdyalhdyalmdyalpdyalsdy"
    "altdyangdyanhdyanjdyassdyebsdyechdyegsdyekkdyelbdyelgdyelhdyelmdyelpdyelsdyeltdy"
    "engdyenhdyenjdyeoddyeohdyeojdyeomdyeopdyeotdyessdyobsdyochdyogsdyokkdyolbdyolgdy"
    "olhdyolmdyolpdyolsdyoltdyongdyonhdyonjdyossdyubsdyuchdyugsdyukkdyulbdyulgdyulhdy"
    "ulmdyulpdyulsdyultdyungdyunhdyunjdyussgaebsgaechgaegsgaekkgaelbgaelggaelhgaelmga"
    "elpgaelsgaeltgaenggaenhgaenjgaessgeobsgeochgeogsgeokkgeolbgeolggeolhgeolmgeolpge"
    "olsgeoltgeonggeonhgeonjgeossgeubsgeuchgeugsgeukkgeulbgeulggeulhgeulmgeulpgeulsge"
    "ultgeunggeunhgeunjgeussgoebsgoechgoegsgoekkgoelbgoelggoelhgoelmgoelpgoelsgoeltgo"
    "enggoenhgoenjgoessguangguibsguichguigsguikkguilbguilgguilhguilmguilpguilsguiltgu"
    "ingguinhguinjguissgwabsgwachgwaedgwaehgwaejgwaemgwaepgwaetgwagsgwakkgwalbgwalggw"
    "alhgwalmgwalpgwalsgwaltgwanggwanhgwanjgwassgwebsgwechgwegsgwekkgwelbgwelggwelhgw"
    "elmgwelpgwelsgweltgwenggwenhgwenjgwessgwibsgwichgwigsgwikkgwilbgwilggwilhgwilmgw"
    "ilpgwilsgwiltgwinggwinhgwinjgwissgwobsgwochgwogsgwokkgwolbgwolggwolhgwolmgwolpgw"
    "olsgwoltgwonggwonhgwonjgwossgyabsgyachgyaedgyaehgyaejgyaemgyaepgyaetgyagsgyakkgy"
    "albgyalggyalhgyalmgyalpgyalsgyaltgyanggyanhgyanjgyassgyebsgyechgyegsgyekkgyelbgy"
    "elggyelhgyelmgyelpgyelsgyeltgyenggyenhgyenjgyeodgyeohgyeojgyeomgyeopgyeotgyessgy"
    "obsgyochgyogsgyokkgyolbgyolggyolhgyolmgyolpgyolsgyoltgyonggyonhgyonjgyossgyubsgy"
    "uchgyugsgyukkgyulbgyulggyulhgyulmgyulpgyulsgyultgyunggyunhgyunjgyussjiangjiongjj"
    "absjjachjjaedjjaehjjaejjjaemjjaepjjaetjjagsjjakkjjalbjjalgjjalhjjalmjjalpjjalsjj"
    "altjjangjjanhjjanjjjassjjebsjjechjjegsjjekkjjelbjjelgjjelhjjelmjjelpjjelsjjeltjj"
    "engjjenhjjenjjjeodjjeohjjeojjjeomjjeopjjeotjjessjjeudjjeuhjjeujjjeumjjeupjjeutjj"
    "ibsjjichjjigsjjikkjjilbjjilgjjilhjjilmjjilpjjilsjjiltjjingjjinhjjinjjjissjjobsjj"
    "ochjjoedjjoehjjoejjjoemjjoepjjoetjjogsjjokkjjolbjjolgjjolhjjolmjjolpjjolsjjoltjj"
    "ongjjonhjjonjjjossjjubsjjuchjjugsjjuidjjuihjjuijjjuimjjuipjjuitjjukkjjulbjjulgjj"
    "ulhjjulmjjulpjjulsjjultjjungjjunhjjunjjjussjjwadjjwahjjwajjjwamjjwapjjwatjjwedjj"
    "wehjjwejjjwemjjwepjjwetjjwidjjwihjjwijjjwimjjwipjjwitjjwodjjwohjjwojjjwomjjwopjj"
    "wotjjyadjjyahjjyajjjyamjjyapjjyatjjyedjjyehjjyejjjyemjjyepjjyetjjyodjjyohjjyojjj"
    "yomjjyopjjyotjjyudjjyuhjjyujjjyumjjyupjjyutkkabskkachkkaedkkaehkkaejkkaemkkaepkk"
    "aetkkagskkakkkkalbkkalgkkalhkkalmkkalpkkalskkaltkkangkkanhkkanjkkasskkebskkechkk"
    "egskkekkkkelbkkelgkkelhkkelmkkelpkkelskkeltkkengkkenhkkenjkkeodkkeohkkeojkkeomkk"
    "eopkkeotkkesskkeudkkeuhkkeujkkeumkkeupkkeutkkibskkichkkigskkikkkkilbkkilgkkilhkk"
    "ilmkkilpkkilskkiltkkingkkinhkkinjkkisskkobskkochkkoedkkoehkkoejkkoemkkoepkkoetkk"
    "ogskkokkkkolbkkolgkkolhkkolmkkolpkkolskkoltkkongkkonhkkonjkkosskkubskkuchkkugskk"
    "uidkkuihkkuijkkuimkkuipkkuitkkukkkkulbkkulgkkulhkkulmkkulpkkulskkultkkungkkunhkk"
    "unjkkusskkwadkkwahkkwajkkwamkkwapkkwatkkwedkkwehkkwejkkwemkkwepkkwetkkwidkkwihkk"
    "wijkkwimkkwipkkwitkkwodkkwohkkwojkkwomkkwopkkwotkkyadkkyahkkyajkkyamkkyapkkyatkk"
    "yedkkyehkkyejkkyemkkyepkkyetkkyodkkyohkkyojkkyomkkyopkkyotkkyudkkyuhkkyujkkyumkk"
    "yupkkyutkuanglaebslaechlaegslaekklaelblaelglaelhlaelmlaelplaelslaeltlaenglaenhla"
    "enjlaessleobsleochleogsleokkleolbleolgleolhleolmleolpleolsleoltleongleonhleonjle"
    "ossleubsleuchleugsleukkleulbleulgleulhleulmleulpleulsleultleungleunhleunjleussli"
    "angloebsloechloegsloekkloelbloelgloelhloelmloelploelsloeltloengloenhloenjloesslu"
    "ibsluichluigsluikkluilbluilgluilhluilmluilpluilsluiltluingluinhluinjluisslwabslw"
    "achlwaedlwaehlwaejlwaemlwaeplwaetlwagslwakklwalblwalglwalhlwalmlwalplwalslwaltlw"
    "anglwanhlwanjlwasslwebslwechlwegslwekklwelblwelglwelhlwelmlwelplwelslweltlwenglw"
    "enhlwenjlwesslwibslwichlwigslwikklwilblwilglwilhlwilmlwilplwilslwiltlwinglwinhlw"
    "injlwisslwobslwochlwogslwokklwolblwolglwolhlwolmlwolplwolslwoltlwonglwonhlwonjlw"
    "osslyabslyachlyaedlyaehlyaejlyaemlyaeplyaetlyagslyakklyalblyalglyalhlyalmlyalply"
    "alslyaltlyanglyanhlyanjlyasslyebslyechlyegslyekklyelblyelglyelhlyelmlyelplyelsly"
    "eltlyenglyenhlyenjlyeodlyeohlyeojlyeomlyeoplyeotlyesslyobslyochlyogslyokklyolbly"
    "olglyolhlyolmlyolplyolslyoltlyonglyonhlyonjlyosslyubslyuchlyugslyukklyulblyulgly"
    "ulhlyulmlyulplyulslyultlyunglyunhlyunjlyussmaebsmaechmaegsmaekkmaelbmaelgmaelhma"
    "elmmaelpmaelsmaeltmaengmaenhmaenjmaessmeobsmeochmeogsmeokkmeolbmeolgmeolhmeolmme"
    "olpmeolsmeoltmeongmeonhmeonjmeossmeubsmeuchmeugsmeukkmeulbmeulgmeulhmeulmmeulpme"
    "ulsmeultmeungmeunhmeunjmeussmoebsmoechmoegsmoekkmoelbmoelgmoelhmoelmmoelpmoelsmo"
    "eltmoengmoenhmoenjmoessmuibsmuichmuigsmuikkmuilbmuilgmuilhmuilmmuilpmuilsmuiltmu"
    "ingmuinhmuinjmuissmwabsmwachmwaedmwaehmwaejmwaemmwaepmwaetmwagsmwakkmwalbmwalgmw"
    "alhmwalmmwalpmwalsmwaltmwangmwanhmwanjmwassmwebsmwechmwegsmwekkmwelbmwelgmwelhmw"
    "elmmwelpmwelsmweltmwengmwenhmwenjmwessmwibsmwichmwigsmwikkmwilbmwilgmwilhmwilmmw"
    "ilpmwilsmwiltmwingmwinhmwinjmwissmwobsmwochmwogsmwokkmwolbmwolgmwolhmwolmmwolpmw"
    "olsmwoltmwongmwonhmwonjmwossmyabsmyachmyaedmyaehmyaejmyaemmyaepmyaetmyagsmyakkmy"
    "albmyalgmyalhmyalmmyalpmyalsmyaltmyangmyanhmyanjmyassmyebsmyechmyegsmyekkmyelbmy"
    "elgmyelhmyelmmyelpmyelsmyeltmyengmyenhmyenjmyeodmyeohmyeojmyeommyeopmyeotmyessmy"
    "obsmyochmyogsmyokkmyolbmyolgmyolhmyolmmyolpmyolsmyoltmyongmyonhmyonjmyossmyubsmy"
    "uchmyugsmyukkmyulbmyulgmyulhmyulmmyulpmyulsmyultmyungmyunhmyunjmyussnaebsnaechna"
    "egsnaekknaelbnaelgnaelhnaelmnaelpnaelsnaeltnaengnaenhnaenjnaessneobsneochneogsne"
    "okkneolbneolgneolhneolmneolpneolsneoltneongneonhneonjneossneubsneuchneugsneukkne"
    "ulbneulgneulhneulmneulpneulsneultneungneunhneunjneussniangnoebsnoechnoegsnoekkno"
    "elbnoelgnoelhnoelmnoelpnoelsnoeltnoengnoenhnoenjnoessnuibsnuichnuigsnuikknuilbnu"
    "ilgnuilhnuilmnuilpnuilsnuiltnuingnuinhnuinjnuissnwabsnwachnwaednwaehnwaejnwaemnw"
    "aepnwaetnwagsnwakknwalbnwalgnwalhnwalmnwalpnwalsnwaltnwangnwanhnwanjnwassnwebsnw"
    "echnwegsnwekknwelbnwelgnwelhnwelmnwelpnwelsnweltnwengnwenhnwenjnwessnwibsnwichnw"
    "igsnwikknwilbnwilgnwilhnwilmnwilpnwilsnwiltnwingnwinhnwinjnwissnwobsnwochnwogsnw"
    "okknwolbnwolgnwolhnwolmnwolpnwolsnwoltnwongnwonhnwonjnwossnyabsnyachnyaednyaehny"
    "aejnyaemnyaepnyaetnyagsnyakknyalbnyalgnyalhnyalmnyalpnyalsnyaltnyangnyanhnyanjny"
    "assnyebsnyechnyegsnyekknyelbnyelgnyelhnyelmnyelpnyelsnyeltnyengnyenhnyenjnyeodny"
    "eohnyeojnyeomnyeopnyeotnyessnyobsnyochnyogsnyokknyolbnyolgnyolhnyolmnyolpnyolsny"
    "oltnyongnyonhnyonjnyossnyubsnyuchnyugsnyukknyulbnyulgnyulhnyulmnyulpnyulsnyultny"
    "ungnyunhnyunjnyussppabsppachppaedppaehppaejppaemppaepppaetppagsppakkppalbppalgpp"
    "alhppalmppalpppalsppaltppangppanhppanjppassppebsppechppegsppekkppelbppelgppelhpp"
    "elmppelpppelsppeltppengppenhppenjppeodppeohppeojppeomppeopppeotppessppeudppeuhpp"
    "eujppeumppeupppeutppibsppichppigsppikkppilbppilgppilhppilmppilpppilsppiltppingpp"
    "inhppinjppissppobsppochppoedppoehppoejppoemppoepppoetppogsppokkppolbppolgppolhpp"
    "olmppolpppolsppoltppongpponhpponjppossppubsppuchppugsppuidppuihppuijppuimppuippp"
    "uitppukkppulbppulgppulhppulmppulpppulsppultppungppunhppunjppussppwadppwahppwajpp"
    "wamppwapppwatppwedppwehppwejppwemppwepppwetppwidppwihppwijppwimppwipppwitppwodpp"
    "wohppwojppwomppwopppwotppyadppyahppyajppyamppyapppyatppyedppyehppyejppyemppyeppp"
    "yetppyodppyohppyojppyomppyopppyotppyudppyuhppyujppyumppyupppyutqiangqiongrad/ssh"
    "angshengshmkhshuaissabsssachssaedssaehssaejssaemssaepssaetssagsssakkssalbssalgss"
    "alhssalmssalpssalsssaltssangssanhssanjssassssebsssechssegsssekksselbsselgsselhss"
    "elmsselpsselssseltssengssenhssenjsseodsseohsseojsseomsseopsseotssesssseudsseuhss"
    "eujsseumsseupsseutssibsssichssigsssikkssilbssilgssilhssilmssilpssilsssiltssingss"
    "inhssinjssissssobsssochssoedssoehssoejssoemssoepssoetssogsssokkssolbssolgssolhss"
    "olmssolpssolsssoltssongssonhssonjssossssubsssuchssugsssuidssuihssuijssuimssuipss"
    "uitssukkssulbssulgssulhssulmssulpssulsssultssungssunhssunjssusssswadsswahsswajss"
    "wamsswapsswatsswedsswehsswejsswemsswepsswetsswidsswihsswijsswimsswipsswitsswodss"
    "wohsswojsswomsswopsswotssyadssyahssyajssyamssyapssyatssyedssyehssyejssyemssyepss"
    "yetssyodssyohssyojssyomssyopssyotssyudssyuhssyujssyumssyupssyutttabsttachttaedtt"
    "aehttaejttaemttaepttaetttagsttakkttalbttalgttalhttalmttalpttalsttaltttangttanhtt"
    "anjttassttebsttechttegsttekkttelbttelgttelhttelmttelpttelstteltttengttenhttenjtt"
    "eodtteohtteojtteomtteoptteotttesstteudtteuhtteujtteumtteuptteutttibsttichttigstt"
    "ikkttilbttilgttilhttilmttilpttilsttiltttingttinhttinjttissttobsttochttoedttoehtt"
    "oejttoemttoepttoetttogsttokkttolbttolgttolhttolmttolpttolsttoltttongttonhttonjtt"
    "ossttubsttuchttugsttuidttuihttuijttuimttuipttuitttukkttulbttulgttulhttulmttulptt"
    "ulsttultttungttunhttunjttussttwadttwahttwajttwamttwapttwatttwedttwehttwejttwemtt"
    "wepttwetttwidttwihttwijttwimttwipttwitttwodttwohttwojttwomttwopttwotttyadttyahtt"
    "yajttyamttyapttyatttyedttyehttyejttyemttyepttyetttyodttyohttyojttyomttyopttyottt"
    "yudttyuhttyujttyumttyupttyutxiangxiongzhangzhengzhongzhuai 1/2 1/3 1/4 1/5 1/6 1"
    "/8 2/3 2/5 3/4 3/5 3/8 4/5 5/6 5/8 7/8(10)(11)(12)(13)(14)(15)(16)(17)(18)(19)(2"
    "0)(ba)(ch)(da)(ga)(ha)(ja)(ju)(ka)(la)(ma)(na)(pa)(sa)(ta)1000C/kgVIIIa.m.akbran"
    "g1babsbachbaedbaehbaejbaembaepbaetbagsbakkbalbbalgbalhbalmbalpbalsbaltbangbanhba"
    "njbassbebsbechbegsbekkbelbbelgbelhbelmbelpbelsbeltbengbenhbenjbeodbeohbeojbeombe"
    "opbeotbessbeudbeuhbeujbeumbeupbeutbianbiaobibsbichbigsbikkbilbbilgbilhbilmbilpbi"
    "lsbiltbingbinhbinjbissbkhybobsbochboedboehboejboemboepboetbogsbokkbolbbolgbolhbo"
    "lmbolpbolsboltbongbonhbonjbossbubsbuchbugsbuidbuihbuijbuimbuipbuitbukkbulbbulgbu"
    "lhbulmbulpbulsbultbungbunhbunjbussbwadbwahbwajbwambwapbwatbwedbwehbwejbwembwepbw"
    "etbwidbwihbwijbwimbwipbwitbwodbwohbwojbwombwopbwotbyadbyahbyajbyambyapbyatbyedby"
    "ehbyejbyembyepbyetbyodbyohbyojbyombyopbyotbyudbyuhbyujbyumbyupbyutcangcengchadch"
    "ahchaichajchamchaochapchatchedchehchejchemchepchetchi1chidchihchijchimchipchitch"
    "odchohchojchomchopchotchouchudchuhchujchumchuochupchutcongcuandabsdachdaeddaehda"
    "ejdaemdaepdaetdagsdakkdalbdalgdalhdalmdalpdalsdaltdangdanhdanjdassdebsdechdegsde"
    "kkdelbdelgdelhdelmdelpdelsdeltdengdenhdenjdeoddeohdeojdeomdeopdeotdessdeuddeuhde"
    "ujdeumdeupdeutdiandiaodibsdichdigsdikkdilbdilgdilhdilmdilpdilsdiltdingdinhdinjdi"
    "ssdkhmdobsdochdoeddoehdoejdoemdoepdoetdogsdokkdolbdolgdolhdolmdolpdolsdoltdongdo"
    "nhdonjdossduandubsduchdugsduidduihduijduimduipduitdukkdulbdulgdulhdulmdulpdulsdu"
    "ltdungdunhdunjdussdwaddwahdwajdwamdwapdwatdweddwehdwejdwemdwepdwetdwiddwihdwijdw"
    "imdwipdwitdwoddwohdwojdwomdwopdwotdyaddyahdyajdyamdyapdyatdyeddyehdyejdyemdyepdy"
    "etdyoddyohdyojdyomdyopdyotdyuddyuhdyujdyumdyupdyuteng1fangfengfiaofkhmgabsgachga"
    "edgaehgaejgaemgaepgaetgagsgakkgalbgalggalhgalmgalpgalsgaltgangganhganjgassgebsge"
    "chgegsgekkgelbgelggelhgelmgelpgelsgeltgenggenhgenjgeodgeohgeojgeomgeopgeotgessge"
    "udgeuhgeujgeumgeupgeutghmmghmygibsgichgigsgikkgilbgilggilhgilmgilpgilsgiltginggi"
    "nhginjgissgobsgochgoedgoehgoejgoemgoepgoetgogsgokkgolbgolggolhgolmgolpgolsgoltgo"
    "nggonhgonjgossguaigubsguchgugsguidguihguijguimguipguitgukkgulbgulggulhgulmgulpgu"
    "lsgultgunggunhgunjgussgwadgwahgwajgwamgwapgwatgwedgwehgwejgwemgwepgwetgwidgwihgw"
    "ijgwimgwipgwitgwodgwohgwojgwomgwopgwotgyadgyahgyajgyamgyapgyatgyedgyehgyejgyemgy"
    "epgyetgyodgyohgyojgyomgyopgyotgyudgyuhgyujgyumgyupgyutjiaojjadjjahjjajjjamjjapjj"
    "atjjedjjehjjejjjemjjepjjetjjidjjihjjijjjimjjipjjitjjodjjohjjojjjomjjopjjotjjudjj"
    "uhjjujjjumjjupjjutjuankcalkkadkkahkkajkkamkkapkkatkkedkkehkkejkkemkkepkketkkidkk"
    "ihkkijkkimkkipkkitkkodkkohkkojkkomkkopkkotkkudkkuhkkujkkumkkupkkutkuailabslachla"
    "edlaehlaejlaemlaeplaetlagslakklalblalglalmlalplalslaltlanglanhlanjlasslebslechle"
    "gslekklelblelglelhlelmlelplelsleltlenglenhlenjleodleohleojleomleopleotlessleudle"
    "uhleujleumleupleutliaolibslichligslikklilblilglilhlilmlilplilsliltlinglinhlinjli"
    "sslkhmlobslochloedloehloejloemloeploetlogslokklolblolglolhlolmlolplolsloltlonglo"
    "nhlonjlossluanlubsluchlugsluidluihluijluimluipluitlukklulblulglulhlulmlulplulslu"
    "ltlunglunhlunjlusslwadlwahlwajlwamlwaplwatlwedlwehlwejlwemlweplwetlwidlwihlwijlw"
    "imlwiplwitlwodlwohlwojlwomlwoplwotlyadlyahlyajlyamlyaplyatlyedlyehlyejlyemlyeply"
    "etlyodlyohlyojlyomlyoplyotlyudlyuhlyujlyumlyuplyutmabsmachmaedmaehmaejmaemmaepma"
    "etmagsmakkmalbmalgmalhmalmmalpmalsmaltmangmanhmanjmassmebsmechmegsmekkmelbmelgme"
    "lhmelmmelpmelsmeltmengmenhmenjmeodmeohmeojmeommeopmeotmessmeudmeuhmeujmeummeupme"
    "utmhmdmianmiaomibsmichmigsmikkmilbmilgmilhmilmmilpmilsmiltmingminhminjmissmjkhmk"
    "hjmkhmmkhymobsmochmoedmoehmoejmoemmoepmoetmogsmokkmolbmolgmolhmolmmolpmolsmoltmo"
    "ngmonhmonjmossmubsmuchmugsmuidmuihmuijmuimmuipmuitmukkmulbmulgmulhmulmmulpmulsmu"
    "ltmungmunhmunjmussmwadmwahmwajmwammwapmwatmwedmwehmwejmwemmwepmwetmwidmwihmwijmw"
    "immwipmwitmwodmwohmwojmwommwopmwotmyadmyahmyajmyammyapmyatmyedmyehmyejmyemmyepmy"
    "etmyodmyohmyojmyommyopmyotmyudmyuhmyujmyummyupmyutnabsnachnaednaehnaejnaemnaepna"
    "etnagsnakknalbnalgnalhnalmnalpnalsnaltnangnanhnanjnassnebsnechnegsnekknelbnelgne"
    "lhnelmnelpnelsneltnengnenhnenjneodneohneojneomneopneotnessneudneuhneujneumneupne"
    "utniaonibsnichnigsnikknilbnilgnilhnilmnilpnilsniltningninhninjnissnobsnochnoedno"
    "ehnoejnoemnoepnoetnogsnokknolbnolgnolhnolmnolpnolsnoltnongnonhnonjnossnuannubsnu"
    "chnugsnuidnuihnuijnuimnuipnuitnukknulbnulgnulhnulmnulpnulsnultnungnunhnunjnussnw"
    "adnwahnwajnwamnwapnwatnwednwehnwejnwemnwepnwetnwidnwihnwijnwimnwipnwitnwodnwohnw"
    "ojnwomnwopnwotnyadnyahnyajnyamnyapnyatnyednyehnyejnyemnyepnyetnyodnyohnyojnyomny"
    "opnyotnyudnyuhnyujnyumnyupnyutp.m.pianpiaoppadppahppajppamppapppatppedppehppejpp"
    "emppepppetppidppihppijppimppipppitppodppohppojppomppopppotppudppuhppujppumppuppp"
    "utqiaoqingquanrengrongrswlruanryalshaishaosheishhmshhyshi1shjyshkhshmmshoushuish"
    "unshuoskhyssadssahssajssamssapssatssedssehssejssemssepssetssidssihssijssimssipss"
    "itssodssohssojssomssopssotssudssuhssujssumssupssutsuantiantiaotkhmtkhytmkhttadtt"
    "ahttajttamttapttatttedttehttejttemttepttetttidttihttijttimttipttitttodttohttojtt"
    "omttopttotttudttuhttujttumttupttuttuanviiiwslmxiaoxingxuanyingyuanzangzengzhaizh"
    "aozhi1zhouzhuizhunzhuozongzuan~tsu'om(1)(2)(3)(4)(5)(6)(7)(8)(9)(C)(R)(a)(b)(c)("
    "d)(e)(f)(g)(h)(i)(j)(k)(l)(m)(n)(o)(p)(q)(r)(s)(t)(u)(v)(w)(x)(y)(z)...10.11.12."
    "13.14.15.16.17.18.19.20.::====A/mCo.FAXFr.GHzGPaHAIHEIHOIMHzMPaPPMPtsTELTHzV/mXI"
    "Ia/ca/sai1an1ao1badbahbaibajbambaobapbarbatbedbehbeibejbembepbetbhabhybidbiebihb"
    "ijbimbipbitbodbohbojbombopbotbudbuhbujbumbupbutc/oc/ucaicaoch'ci1coucuicuncuodad"
    "dahdaidajdamdaodapdatdeddehdejdemdepdetdhadhydiddiedihdijdimdipditdiudjidoddohdo"
    "jdomdopdotdoududduhdujdumduodupdutdzueh1ei1en1er1feiffifflfmyfougadgahgaigajgamg"
    "aogapgatgedgehgeigejgemgepgetghaghjghygidgihgijgimgipgitgodgohgojgomgopgotgougud"
    "guhgujgumguogupguthPahmjhoijhajhyjiejiujmhjuek'hkHzkPakaikaokhakhhkkhkmmkoukuola"
    "dlahlailajlamlaolaplatledlehleilejlemlepletlhylidlielihlijlimliplitliuljjljmljyl"
    "mhlmylodlohlojlomloplotlouludlueluhlujlumluoluplutm/smadmahmaimajmammaomapmatmed"
    "mehmeimejmemmepmetmhymidmiemihmijmimmipmitmiumjmmjymodmohmojmommopmotmoumudmuhmu"
    "jmummupmutnadnahnainajnamnaonapnatnednehneinejnemnepnetnhynidnienihnijnimnipnitn"
    "iunjhnjynkhnmynodnohnojnomnopnotnounudnuenuhnujnumnuonupnutou1paipaopeiphapiepou"
    "qieqiuqmhqmmqmyquequnraorhari1rouruirunruosaoshrshysi1sjhsjysmhsmmsousuot'htaita"
    "othathhthjthmthnthrthythztietjmtjytmjtmmts'tuowaiweiwu1xiexiixiuxuexunyaoyhyyi1y"
    "jyykhymmymyyouyu1yueyyazaizaozeizi1zouzuizunzuo~ka~ke~wa~ya~yo~yu i u!!!?'a'l'n("
    "(())),,<<>>?!??AEAUBqC'CECHCrDZDzGBGyHIHPHYIJIUIVIXK'KBKKKML.LJLLLjMBMVMWNJNjNoO"
    "EP'PRPSRHRsRxSSSvT'TLWba1aaaubzc'cccdcmdBdre1evfhfjfufyhvivkAkVkWlnlxlzm1mAmVmWn"
    "AnFnVnWoVp'pApFpHpVpWq'qaqhqlqpqysrtrtzvavevovuwwwyyrywyz||~a~e~i~o~u\"#$%&*+-;@Q"
    "[\\]^_`{}";
#endif // HAVE_ICU

Encoder::Encoder()
        : d(new Encoder::Private())
{
#ifdef HAVE_ICU
    qCInfo(LOG_KBIBTEX_IO) << "Using ICU in Encoder class";
#else // HAVE_ICU
    qCInfo(LOG_KBIBTEX_IO) << "Using built-in translation table in Encoder class";
#endif // HAVE_ICU
}

Encoder::~Encoder()
{
    delete d;
}

const Encoder &Encoder::instance()
{
    static const Encoder self;
    return self;
}

QString Encoder::decode(const QString &text) const
{
    return text;
}

QString Encoder::encode(const QString &text, const TargetEncoding) const
{
    return text;
}

QString Encoder::convertToPlainAscii(const QString &ninput) const
{
    /// Preprocessing where ICU may give unexpected results otherwise
    QString input = ninput;
    input = input.replace(QChar(0x2013), QStringLiteral("--")).replace(QChar(0x2014), QStringLiteral("---"));

#ifdef HAVE_ICU
    static const int uChars_size = 16384;
    static UChar uChars[uChars_size];
    const int inputLen = qMin(input.length(), uChars_size);
    int dataLen = 0;
    /// Make a copy of the input string into an array of UChar, but skip certain characters
    for (int i = 0; i < inputLen; ++i) {
        const auto u = input.at(i).unicode();
        if (u < 32 || u == 127) continue;
        uChars[dataLen++] = u;
    }
    /// Create an ICU-specific unicode string
    icu::UnicodeString uString = icu::UnicodeString(uChars, dataLen);
    /// Perform the actual transliteration, modifying Unicode string
    if (d->translit != nullptr) d->translit->transliterate(uString);
    /// Create regular C++ string from Unicode string
    std::string cppString;
    uString.toUTF8String(cppString);
    /// Convert regular C++ to Qt-specific QString,
    /// should work as cppString contains only ASCII text
    return QString::fromStdString(cppString);
#else // HAVE_ICU
    QString result;
    result.reserve(input.length() + 256);
    for (const QChar &c : input) {
        if (c.isHighSurrogate() || c.isLowSurrogate())
            /// Keep surrogate code points unchanged
            result.append(c);
        else {
            const auto unicode = c.unicode();
            const auto index = unicode < 0xd800 ? unicode : unicode - 2048;
            const auto encodedposlen = Encoder::Private::unidecode_pos[index];
            const auto pos = encodedposlen >> 5;
            const auto len = static_cast<int>(encodedposlen & 31);
            result.append(QString::fromLatin1(Encoder::Private::unidecode_text + pos, len));
        }
    }
    return result;
#endif // HAVE_ICU
}

bool Encoder::containsOnlyAscii(const QString &ntext)
{
    /// Perform Canonical Decomposition followed by Canonical Composition
    const QString text = ntext.normalized(QString::NormalizationForm_C);

    for (const QChar &c : text)
        if (c.unicode() > 127) return false;
    return true;
}
