/***************************************************************************
                          krule.h  -  description
                             -------------------
    begin                : Tue Aug 1 2000
    copyright            : (C) 2000 by Terk Zsolt
    email                : tz124@hszk.bme.hu
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef KRULE_H
#define KRULE_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <qvaluelist.h>
#include <qlist.h>
class QString;
class QStringList;
class KPlace;

extern int substitute(QString &, QString &, QString &, QString &, bool);
extern int translate(QString &, QString &, QString &, QString &);
extern int match(QString &, QString &, QStringList &);

/* structure for special token parameter */
typedef struct
{
  int type; /* 0:string, 1:int, 2:nonterminal, 3:regex, 4:substitution, 5:translation */
  QString * strData;
  QString * strsData[3];
  int intData;
} param_type;


/**Grammatical rule for Kaptain.
  *@author Terk Zsolt
  */

class KRule
{
  friend int yyparse();

  enum Type {Conjunctive, Disjunctive, Special};

  /** The grammatical type of the rule */
  Type type;
  /** The appereance of the rule; KRule::Style constants' mixture. */
  int style;

  /** Left side nonterminal */
  int left;
  /** Tokens on the right side */
  QValueList<int> right;

  /** Consider it a default selection; valid while it is a conjunctive rule */
  bool thisDefault;
  /** The number of the default child in disjunction */
  int myDefault;

  /* pointer to the appropriate place if exists */
  KPlace * place;

  /* if true, this rule should evaluate to an empty string */
  //bool noEval;

  /* the title, tooltip and whatsthis */
  QString title;
  QString toolTip;
  QString whatsThis;

  /* the identifier of special */
  int specType;

  /* the alignment of the widget */
  int align;

  /* parameters for special */
  QList<param_type> * parameters;
  param_type * initValue;

  /* list of transformer units (substitutions, translations) */
  QList<param_type> * transformers;

  /* make conjunctive rule from a special */
  void saveOut();

  /* the boss for a disjunctive rule; the one which determines the choice */
  KRule * bossRule;

 public:
	KRule();
	~KRule();

  enum Style {Framed=1,   Tabbed=2,  Horizontal=4, Wizard=8,
              Reverse=16, NoEval=32, Dialog=64,    Double=128,
              Beside=256, Tree=512,  Detailed=1024 };
	
	/** Inserts the given KRule as a new clause for this rule. Left side of the given rule is discarded.
	 *  This rule becomes a disjunctive rule
	 */
	void addAsNewClause(KRule *);
	
	/* Append new symbol to the end of the rule */
	void append(int);
	
	/* Sets the last clause to be the default */
	void setDefault();
	int getDefault() { return myDefault; }
	
	/* Mutators */
	void setDisjunctive() { type=Disjunctive; }
	void setSpecial(int s) { type=Special; specType=s; }
	
  /* Accessors for grammar data */
  int leftSide() { return left; }
  QValueList<int> rightSide() { return right; }

  /* returns the list of nonterminals on the right side */
  QValueList<int> children();


	/* Accessors for type */
	bool isConjunctive() { return type==Conjunctive; }
	bool isDisjunctive() { return type==Disjunctive; }
	bool isEpsilon() { return right.count()==0 && !isSpecial(); }
	bool isSpecial() { return type==Special; }
	bool isSimple() { return children().count()==0 && !isSpecial() && getTitle().isEmpty(); }
	
	/* Accessors for style */
	bool isReverse() { return style & Reverse; }
	bool isFramed() { return style & Framed; }
	bool isTabbed() { return style & Tabbed; }
	bool isHorizontal() { return style & Horizontal; }
	bool isWizard() { return style & Wizard; }
	bool isDialog() { return style & Dialog; }
  bool isNoEval() { return style & NoEval; }
  bool isDouble() { return style & Double; }
  bool isBeside() { return style & Beside; }
  bool isTree() { return style & Tree; }
  bool isDetailed() { return style & Detailed; }
	
  /* Returns if the rule is the top level (start) rule. Only that can be tabbed */
	bool isTopLevel() { return left==-1; }
	
	void print();
	
	QString getTitle();
	QString getToolTip();
	QString getWhatsThis();
	void setTitle(QString *);
	void setToolTip(QString *);
	void setWhatsThis(QString *);
	
	void setParams(QList<param_type> *);
  void setTransformers(QList<param_type> *);
	void setInitValue(param_type *);

	int getSpecType() { return specType; }
	QList<param_type> * getParams() { return parameters; }
	param_type * getInitValue() { return initValue; }
	
	void setPlace(KPlace * p) { place=p; }
	KPlace * getPlace() { return place; }
	QString evaluate(bool considerNoEval=true);
  QString evalSpecial(bool considerNoEval);

	/* checkbox functionals */
	void correctCheckBox();
	int whichEpsilonChild();
	bool isCheckBox();
	bool isTristate();

	/* constraint features */
	void setConstraint(KRule *);	
  bool hasBoss() { return bossRule; }
  int whichIsSelected();

  /* special widget alignment */
  int getAlign() { return align; }

};

#endif
