/*
    kamefuview.cpp - Kamefu Main View

    Copyright (c) 2005      by Michaël Larouche       <michael.larouche@kdemail.net>

    *************************************************************************
    *                                                                       *
    * This program is free software; you can redistribute it and/or modify  *
    * it under the terms of the GNU General Public License as published by  *
    * the Free Software Foundation; either version 2 of the License, or     *
    * (at your option) any later version.                                   *
    *                                                                       *
    *************************************************************************
*/

#include "kamefuview.h"

// Qt includes
#include <qlayout.h>
#include <qsplitter.h>

// KDE includes
#include <kdebug.h>
#include <klistview.h>
#include <klistviewsearchline.h>
#include <klocale.h>
#include <kio/job.h>
#include <kmessagebox.h>

// Kamefu includes
#include "collectionviewer.h"
#include "filterlist.h"
#include "collectionaccess.h"
#include "kamefupreferences.h"
#include "styledromdisplay.h"
#include "metadataeditdialog.h"

class KamefuView::Private
{
public:
	StyledRomDisplay *display;
	FilterList *filterList;
	CollectionViewer *collectionViewer;
	QSplitter *mainSplitter;
	QSplitter *verticalSplitter;
};

KamefuView::KamefuView(QWidget *parent, const char *name)
 : QWidget(parent, name), d(new Private)
{
	QGridLayout *mainLayout = new QGridLayout(this, 1, 1);
	d->mainSplitter = new QSplitter(QSplitter::Horizontal, this);

	// Create the ROM browser
	unsigned int opts = (DISPLAY_ROMNAME | DISPLAY_SYSTEM | DISPLAY_TITLESCREEN |
		DISPLAY_DEVELOPER | DISPLAY_YEAR | DISPLAY_COUNTRY | DISPLAY_GENRE | DISPLAY_SCREENSHOTS);
	d->display = new StyledRomDisplay(opts, d->mainSplitter, "rom_html_display");

	d->verticalSplitter = new QSplitter(QSplitter::Vertical, d->mainSplitter);

	// Create the filter lists.
	d->filterList = new FilterList(d->verticalSplitter);
	connect(d->filterList, SIGNAL(refreshRoms(const QString&)), this, SLOT(fillRoms(const QString&)));

	d->collectionViewer = new CollectionViewer(d->verticalSplitter);
	connect(d->collectionViewer, SIGNAL(itemExecuted(const QString &)), this, SIGNAL(romExecuted( const QString& )));
	connect(d->collectionViewer, SIGNAL(itemSelected(const QString &)), d->display, SLOT(setRom(const QString& )));
	// Dispatched calls.
	connect(d->collectionViewer, SIGNAL(itemEditMetadata(const KURL::List&)), this, SLOT(editRomsInformation(const KURL::List&)));
	connect(d->collectionViewer, SIGNAL(itemDeleteRoms(const KURL::List&)), this, SLOT(deleteRoms(const KURL::List&)));
	connect(d->collectionViewer, SIGNAL(itemTrashRoms(const KURL::List&)), this, SLOT(trashRoms(const KURL::List&)));

	mainLayout->addWidget(d->mainSplitter, 0, 0);

	// Fill the filters.
	fillFilters();

	// Restore the size from settings.
	readSettings();
}


KamefuView::~KamefuView()
{
	kdDebug() << k_funcinfo << endl;
	delete d;
}

void KamefuView::clearRomList()
{
	d->collectionViewer->clear();
}

void KamefuView::fillFilters()
{
	d->filterList->fillFilterLists();
}

void KamefuView::addRomListViewItem(const Kamefu::RomMetaInformation &romMetaInfo)
{
	d->collectionViewer->addItem(romMetaInfo);
}

void KamefuView::fillRoms(const QString &query)
{
	clearRomList();
	Kamefu::CollectionAccess::self()->retrieveRoms(query);
}

void KamefuView::updateSelectedRomInformation(const QString &romUrl)
{
	d->collectionViewer->updateSelectedRomInformation(romUrl);
	fillFilters();
}

void KamefuView::updateSelectedRomInformation(const KURL::List &romList)
{
	Q_UNUSED(romList);
	d->collectionViewer->updateSelectedRoms();
	fillFilters();
}

void KamefuView::readSettings()
{
	// Restore the size preferences
	if( !KamefuPreferences::self()->viewSplitterSizes().empty() )
		d->verticalSplitter->setSizes( KamefuPreferences::self()->viewSplitterSizes() );

	if( !KamefuPreferences::self()->mainWindowSplitterSizes().empty() )
		d->mainSplitter->setSizes( KamefuPreferences::self()->mainWindowSplitterSizes() );
}

void KamefuView::saveSettings()
{
	// Save size preferences
	KamefuPreferences::self()->setViewSplitterSizes( d->verticalSplitter->sizes() );
	KamefuPreferences::self()->setMainWindowSplitterSizes( d->mainSplitter->sizes() );
	KamefuPreferences::self()->writeConfig();
}

void KamefuView::editRomsInformation(const KURL::List &romList)
{
	kdDebug() << k_funcinfo << endl;

	MetadataEditDialog *editDlg;
	if( romList.count() == 1 )
	{
		KURL url = romList.first();
		QString romUrl = url.path();
		editDlg = new MetadataEditDialog(romUrl, romUrl, this);
	}
	else
	{
		editDlg = new MetadataEditDialog(romList, i18n("", "Information for %n files", romList.count()), this);
	}

	// connect rom modification (Qt disconnects when editDlg is destroyed)
	connect(editDlg, SIGNAL(metadataChanged(const QString&)), this, SLOT(updateSelectedRomInformation(const QString&)));
	connect(editDlg, SIGNAL(metadataChanged(const KURL::List&)), this, SLOT(updateSelectedRomInformation(const KURL::List&)));
	// Update sidebar on modified item.
	connect(editDlg, SIGNAL(metadataChanged(const QString&)), d->display, SLOT(setRom(const QString& )));

	editDlg->show();
}

void KamefuView::deleteRoms(const KURL::List &romList)
{
	if( KMessageBox::questionYesNo(this, i18n("Are you sure you want to delete this file ?", "Are you sure you want to delete %n files.", romList.count()), i18n("Kamefu")) == KMessageBox::Yes )
	{
		KIO::Job *job = KIO::del(romList, false, true); // For first debug, show progress.
		connect(job, SIGNAL(result(KIO::Job*)), this, SLOT(romJobDone(KIO::Job*)));
	}
}

void KamefuView::trashRoms(const KURL::List &romList)
{
	if( KMessageBox::questionYesNo(this, i18n("Are you sure you want to trash this file ?", "Are you sure you want to trash %n files.", romList.count()), i18n("Kamefu")) == KMessageBox::Yes )
	{
		KIO::Job *job = KIO::trash(romList, true); // For first debug, show progress.
		connect(job, SIGNAL(result(KIO::Job*)), this, SLOT(romJobDone(KIO::Job*)));
	}
}

void KamefuView::romJobDone(KIO::Job *job)
{
	if( job->error() )
	{
		KMessageBox::queuedMessageBox( this, KMessageBox::Error, i18n("An error occured while trying to delete or trash ROM file(s).\n%1").arg(job->errorString()), i18n("Kamefu") );
		return;
	}

	KURL::List romList;
	// Determine which type is the resulting job
	// They don't have the same method to retrieve source URLs.
	KIO::DeleteJob *deleteJob = dynamic_cast<KIO::DeleteJob*>(job);
	KIO::CopyJob *copyJob = dynamic_cast<KIO::CopyJob*>(job);

	if(deleteJob)
	{
		kdDebug() << k_funcinfo << "Delete job" << endl;
		romList = deleteJob->urls();
	}
	else if(copyJob)
	{
		kdDebug() << k_funcinfo << "Trash job" << endl;
		romList = copyJob->srcURLs();
	}
	else
	{
		return;
	}

	if( !romList.isEmpty() )
	{
		KURL::List::ConstIterator it, itEnd = romList.constEnd();
		for(it = romList.constBegin(); it != itEnd; ++it)
		{
			KURL temp = *it;
			kdDebug() << k_funcinfo << "Removing from database: " << temp.path() << endl;
			Kamefu::CollectionAccess::self()->removeRomFromDatabase( temp.path() );
		}
		// Remove the items(there are selected) from the list view.
		d->collectionViewer->removeSelectedItems();

		emit romsDeletedOrTrashed();
	}
}

int KamefuView::collectionSize() const
{
	return d->collectionViewer->collectionSize();
}

#include "kamefuview.moc"
