// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Tim Shead (tshead@k-3d.com)
*/

#include "iobject.h"
#include "property.h"
#include "result.h"
#include "transform.h"

#include <sdpxml/sdpxml.h>

#include <iostream>

namespace k3d
{

const vector3 world_position(iunknown& Object)
{
	return object_to_world_matrix(Object) * vector3(0, 0, 0);
}

bool set_position(iunknown& Object, const vector3& Position)
{
	// Try looking for a property with the correct name and type ...
	if(set_property_value(Object, "position", Position))
		return true;

	std::cerr << error << "Couldn't set object position" << std::endl;
	return false;
}

bool set_orientation(iunknown& Object, const angle_axis& Orientation)
{
	// Try looking for a property with the correct name and type ...
	if(set_property_value(Object, "orientation", Orientation))
		return true;

	std::cerr << error << "Couldn't set object orientation" << std::endl;
	return false;
}

bool set_scale(iunknown& Object, const vector3& Scale)
{
	// Try looking for a property with the correct name and type ...
	if(set_property_value(Object, "scale", Scale))
		return true;

	std::cerr << error << "Couldn't set object scale" << std::endl;
	return false;
}

const matrix4 object_to_world_matrix(iunknown& Object)
{
	// First, look for a property with the correct name and type ...
	iproperty* const property = get_typed_property<k3d::matrix4>(Object, "output_matrix");
	if(property)
		return boost::any_cast<k3d::matrix4>(property->value());

	return identity3D();
}

const matrix4 world_to_object_matrix(iunknown& Object)
{
	// Return theinverse of the object-to-world matrix ...
	return object_to_world_matrix(Object).Inverse();
}

const matrix4 object_to_object_matrix(iunknown& From, iunknown& To)
{
	// Return a transformation from one object's coordinate frame into another's ...
	return object_to_world_matrix(From) * world_to_object_matrix(To);
}

} // namespace k3d


