
/*
 * regexp.h -- written for Juice
 *	Copyright (C) 1999, 2000, 2001 Abraham vd Merwe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef DIALOGS_REGEXP_H
#define DIALOGS_REGEXP_H

#include "typedefs.h"

/*
 * ## Match::MatchPattern()
 * ########################
 *
 *  Match the pattern PATTERN against the string TEXT;
 *
 *  returns MATCH_VALID if pattern matches, or an errorcode as follows
 *  otherwise:
 *
 *            MATCH_PATTERN  - bad pattern
 *            MATCH_LITERAL  - match failure on literal mismatch
 *            MATCH_RANGE    - match failure on [..] construct
 *            MATCH_ABORT    - premature end of text string
 *            MATCH_END      - premature end of pattern string
 *            MATCH_VALID    - valid match
 *
 *
 *  A match means the entire string TEXT is used up in matching.
 *
 *  In the pattern string:
 *       *' matches any sequence of characters (zero or more)
 *       ?' matches any character
 *       [SET] matches any character in the specified set,
 *       [!SET] or [^SET] matches any character not in the specified set.
 *
 *  A set is composed of characters or ranges; a range looks like
 *  character hyphen character (as in 0-9 or A-Z).  [0-9a-zA-Z_] is the
 *  minimal set of characters allowed in the [..] pattern construct.
 *  Other characters are allowed (ie. 8 bit characters) if your system
 *  will support them.
 *
 *  To suppress the special syntactic significance of any of []*?!^-\',
 *  and match the character exactly, precede it with a \'.
 * 
 * 
 * ## Match::IsValidPattern()
 * ##########################
 *
 *  Return TRUE if PATTERN has is a well formed regular expression according
 *  to the above syntax
 *
 *  error_type is a return code based on the type of pattern error.  Zero is
 *  returned in error_type if the pattern is a valid one.  error_type return
 *  values are as follows:
 *
 *            PATTERN_VALID - pattern is well formed
 *            PATTERN_ESC   - pattern has invalid escape ('\' at end of pattern)
 *            PATTERN_RANGE - [..] construct has a no end range in a '-' pair (ie [a-])
 *            PATTERN_CLOSE - [..] construct has no end bracket (ie [abc-g )
 *            PATTERN_EMPTY - [..] construct is empty (ie [])
 */

class Match
{
 public:
   /* Match pattern against text */
   bool MatchPattern (const char *text,const char *pattern);
   /* Check if pattern has any special wildcard characters */
   bool IsPattern (const char *pattern);
   /* Check if pattern is a well formed regular expression */
   bool IsValidPattern (const char *pattern,int *error_type);
 protected:
   /* Match pattern against test */
   int match (const char *text,const char *pattern);
 private:
   /* Recursively call match() with final segment of pattern and of text */
   int match_after_star (const char *text,const char *pattern);
};

#endif
