#include <jmp-config.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#ifdef HAVE_ICONV_H
 #include <iconv.h>
#endif

#include <jmp.h>
#include <heap_dump.h>
#ifdef HAVE_WIN32COMPAT_H
 #include <win32compat.h>
#endif

#include <ui_gtk.h>

#ifndef ICONV_INBUF_TYPECAST
 #ifdef __CYGWIN32__
  #define ICONV_INBUF_TYPE const char**
 #else
  #define ICONV_INBUF_TYPE char**
 #endif
 #define ICONV_INBUF_TYPECAST (ICONV_INBUF_TYPE)
#endif

static int string_file_counter = -1;
static char filename[128];

FILE* get_string_dump_file () {
    FILE* f;
    snprintf (filename, 128, "jmp_string_dump-%d.txt", ++string_file_counter);
    f = fopen (filename, "w");
    return f;
}

char* get_current_dump_file () {
    return filename;
}

int write_string (FILE* f, char* text, int count, int used) {
    return fprintf (f, "'%d'\t'%d'\t'%s'\n", used, count, text);    
}

#ifdef WIN32
#define u_int16_t unsigned short
static unsigned char* encode_utf8(unsigned char *buf, unsigned int *leftptr, u_int16_t c) {
    unsigned int left = *leftptr;

    if(c < 0x80) {
	if(left < 1)
	    goto Fail;
	*buf++ = c & 0x7f;
	*leftptr = left - 1;
    } else if(c < 0x800) {
	if(left < 2)
	    goto Fail;
	*buf++ = (c >> 6) | 0xc0;
	*buf++ = (c & 0x3f) | 0x80;
	*leftptr = left - 2;
    } else {
	if(left < 3)
	    goto Fail;
	*buf++ = (c >> 12) | 0xe0;
	*buf++ = ((c >> 6) & 0x3f) | 0x80;
	*buf++ = (c & 0x3f) | 0x80;
	*leftptr = left - 3;
    }

    return buf;

Fail:
    return NULL;
}

static char* jmp_utf16_to_utf8 (const char* text, size_t count, long* items_written, int* err) {
    unsigned char *ptr;
    char* out_buf;
    unsigned int left;
    u_int16_t *charptr;

    charptr = (u_int16_t *)text;
    left = 3 * (count+1);
    out_buf = malloc (left);
    ptr = out_buf;
    while(count > 0) {
	u_int16_t c = *charptr;
	encode_utf8(ptr, &left, c);
	charptr++;
	count--;
	if(c == 0)
	    break;	/* iconv() must use <nul> char */
    }
    return out_buf;
}
#endif

#ifdef HAVE_ICONV_H
static char* jmp_utf16_to_utf8 (const char* text, size_t count, long* items_written, int* err) {
    iconv_t cd; 
    size_t non_rev_count;
    size_t in_bytes_left;
    size_t out_bytes_left;
    char* out_buf;
    const char *in_buf_tmp = text;
    ICONV_INBUF_TYPE in_buf = ICONV_INBUF_TYPECAST &in_buf_tmp;
    cd = iconv_open("utf8", "utf16");
    out_buf = malloc (4 * count);
    non_rev_count = 
	iconv (cd, in_buf, &in_bytes_left, &out_buf, &out_bytes_left);
    iconv_close (cd);
    return out_buf;
}
#endif

void dump_string (obj* o, FILE* f) {
    down_link* dl;
    get_instance_info (o);
    dl = get_last_down_link ();
    while (dl != NULL) {
	long items_written;
	int err = 0;
	char* converted_text;

	switch (dl->type) {
	case JVMPI_GC_PRIM_ARRAY_DUMP:
	    converted_text = jmp_utf16_to_utf8 ((const char*)dl->value.txt, dl->pos, &items_written, &err); 
	    if (err != 0) {
		fprintf (stderr, "Unable to convert text: %s\n", strerror (err));	
		break;
	    }
	    write_string (f, converted_text, 1, dl->pos * 2);
	    free (converted_text);
	    break;
	default:
	    fprintf (stderr, "Odd type when dumping string: %d\n", dl->type);
	}
	dl = dl->next;	
    }
    free_last_down_link ();	
}

void dump_strings () {
    FILE* f;
    f = get_string_dump_file ();
    for_each_string ((string_callback)dump_string, f);
    fclose (f);
}
