/*************************************************************************
 * $Id: mod_timer.c,v 1.4 2000/04/04 23:37:34 dpotter Exp $
 *
 * mod_timer.c -- Timer functions
 *
 * Copyright (C) by Andreas Neuhaus <andy@fasta.fh-dortmund.de>
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "tools.h"
#include "log.h"
#include "mod.h"
#include "mod_timer.h"


/*************************************************************************
 * GLOBALS
 */
time_t mod_timer_sleeptimeout = 0;
time_t mod_timer_sleepfadeout = 0;


/*************************************************************************
 * MODULE INFO
 */
mod_t mod_timer = {
	NULL,			// deinit
	NULL,			// reload
	&blank_fd,		// watchfd
	NULL,			// poll
	mod_timer_update,	// update
	mod_timer_message,	// message
	NULL,			// SIGCHLD handler
};


/*************************************************************************
 * UPDATE
 */
void mod_timer_update (void)
{
	static time_t lastupdate;
	time_t now;
	int stepsleft;

	if (!mod_timer_sleeptimeout)
		return;

	now = time(NULL);
	if (now >= lastupdate+10) {
		lastupdate = now;
		log_printf(LOG_DEBUG, "mod_timer_update(): checking timers\n");

		// check if sleep timer expired
		if (mod_timer_sleeptimeout <= now) {
			log_printf(LOG_NORMAL, "Sleep timer expired\n");
			mod_timer_sleeptimeout = 0;
			mod_timer_sleepfadeout = 0;
			mod_sendmsg(MSGTYPE_INPUT, "stop");
			mod_sendmsg(MSGTYPE_INPUT, "mixer default");
			return;
		}

		// check if volume decrease should be done
		if (now < mod_timer_sleepfadeout)
			return;

		stepsleft = (mod_timer_sleeptimeout - now) / 10;
		log_printf(LOG_VERBOSE, "Sleep timer decreasing volume (%d steps left)\n", stepsleft);
		mod_sendmsgf(MSGTYPE_INPUT, "mixer volume -%d%%", stepsleft ? (100 / stepsleft) : 100);
	}
}


/*************************************************************************
 * RECEIVE MESSAGE
 */
void mod_timer_message (int msgtype, char *msg)
{
	char *c1, *c2, *c3;
	int sleeptime, fadetime;

	// handle input messages
	if (msgtype == MSGTYPE_INPUT) {
		c1 = msg ? strtok(msg, " \t") : NULL;
		c2 = c1 ? strtok(NULL, " \t") : NULL;
		c3 = c2 ? strtok(NULL, " \t") : NULL;

		// process sleep command
		if (c1 && !strcasecmp(c1, "sleep")) {
			sleeptime = c2 ? atoi(c2) : 0;
			fadetime = c3 ? atoi(c3) : 0;
			if (sleeptime) {
				mod_timer_sleeptimeout = time(NULL) + sleeptime * 60;
				mod_timer_sleepfadeout = mod_timer_sleeptimeout - fadetime * 60;
				mod_sendmsgf(MSGTYPE_GENERIC, "sleep %d %d", sleeptime, fadetime);
				log_printf(LOG_NORMAL, "Sleep timer set to %d mins, %d mins fadeout\n", sleeptime, fadetime);
			} else {
				mod_timer_sleeptimeout = 0;
				mod_timer_sleepfadeout = 0;
				mod_sendmsg(MSGTYPE_GENERIC, "sleep 0");
				log_printf(LOG_NORMAL, "Sleep timer off\n");
			}
		}
	}
}


/*************************************************************************
 * MODULE INIT FUNCTION
 */
char *mod_timer_init (void)
{
	log_printf(LOG_DEBUG, "mod_timer_init(): initializing\n");

	// register our module
	mod_register(&mod_timer);

	return NULL;
}


/*************************************************************************
 * EOF
 */
