/*************************************************************************
 * $Id: mod_lcdproc.c,v 1.15 2001/05/10 01:34:46 dpotter Exp $
 *
 * mod_lcdproc.c -- LCDproc display/input module
 *
 * Copyright (C) by Andreas Neuhaus <andy@fasta.fh-dortmund.de>
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <time.h>
#include <errno.h>
#include <netdb.h>
#include <fcntl.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <locale.h>

#include "tools.h"
#include "log.h"
#include "config.h"
#include "mod.h"
#include "mod_lcdproc.h"

/*************************************************************************
 * GLOBALS
 */
int mod_lcdproc_wid = 0;			// display width
int mod_lcdproc_hgt = 0;			// display height
int mod_lcdproc_cellwid = 0;		// cell width
int mod_lcdproc_cellhgt = 0;		// cell height

int mod_lcdproc_mode = 0;			// display mode
char *mod_lcdproc_playmode = "";	// playing mode
int mod_lcdproc_time1 = 0;			// playing time
int mod_lcdproc_time2 = 0;			// "       "
char mod_lcdproc_artist[128] = "";	// artist
char mod_lcdproc_title[128] = "";	// title
int mod_lcdproc_repeat = 0;			// repeat mode
int mod_lcdproc_shuffle = 0;		// shuffle mode
int mod_lcdproc_mute = 0;			// mute mode
time_t mod_lcdproc_tmptimeout = 0;	// timout for tmp status
int	lcdproc_fd=0;
fd_set	mod_lcdproc_fdset;
int mod_lcdproc_clockmode = 0;		// flag for clock/song display
int mod_lcdproc_clocktimer = 0;		// how many seconds of idle before clock starts

/*************************************************************************
 * MODULE INFO
 */
mod_t mod_lcdproc = {
	mod_lcdproc_deinit,	// deinit
	mod_lcdproc_reload,	// reload
	&mod_lcdproc_fdset,	// fd's to watch
	NULL,			// poll
	mod_lcdproc_update,	// update
	mod_lcdproc_message,	// message
	NULL,			// SIGCHLD handler
};


/*************************************************************************
 * DISCONNECT FROM LCDPROC SERVER
 */
void mod_lcdproc_disconnect (void)
{
	if (lcdproc_fd)
		shutdown(lcdproc_fd, 2);
	log_printf(LOG_NOISYDEBUG,"mod_lcdproc_disconnect(): disconnecting from lcdproc.\n");
	lcdproc_fd = 0;
	FD_ZERO(&mod_lcdproc_fdset);
	mod_lcdproc.poll = NULL;
}


/*************************************************************************
 * CONNECT TO LCDPROC SERVER
 */
char *mod_lcdproc_connect (void)
{
	struct sockaddr_in server;
	struct hostent *hostinfo;
	int rc;

	// close if already open
	mod_lcdproc_disconnect();

	// resolve hostname
	server.sin_family = AF_INET;
	server.sin_port = htons(config_getnum("lcdproc_port", 13666));
	hostinfo = gethostbyname(config_getstr("lcdproc_host", "localhost"));
	if (!hostinfo)
		return "LCDproc: Unknown host";
	server.sin_addr = *(struct in_addr *) hostinfo->h_addr;

	// create socket
	lcdproc_fd = socket(PF_INET, SOCK_STREAM, 0);
	if (lcdproc_fd < 0) {
		mod_lcdproc_disconnect();
		return "LCDproc: error creating socket";
	} else
		FD_SET(lcdproc_fd,&mod_lcdproc_fdset);
	log_printf(LOG_DEBUG, "mod_lcdproc_connect(): created socket on fd %d\n", lcdproc_fd);

	// connect
	rc = connect(lcdproc_fd, (struct sockaddr *) &server, sizeof(server));
	if (rc < 0) {
		mod_lcdproc_disconnect();
		return "LCDproc: unable to connect";
	}
	fcntl(lcdproc_fd, F_SETFL, O_NONBLOCK);
	mod_lcdproc.poll = mod_lcdproc_poll;

	return NULL;
}


/*************************************************************************
 * SETUP LCDPROC SCREENS
 */
void mod_lcdproc_setup (void)
{
	if (!lcdproc_fd)
		return;
	log_printf(LOG_DEBUG, "mod_lcdproc_setup(): setting up screens\n");
	sendtext(lcdproc_fd, "hello\n");
	sendtext(lcdproc_fd, "screen_add irmp3\n");
	sendtext(lcdproc_fd, "widget_add irmp3 time string\n");
	sendtext(lcdproc_fd, "widget_add irmp3 artist scroller\n");
	sendtext(lcdproc_fd, "widget_add irmp3 title scroller\n");
	sendtext(lcdproc_fd, "widget_add irmp3 status string\n");
	sendtext(lcdproc_fd, "widget_add irmp3 tmptext string\n");
	sendtext(lcdproc_fd, "widget_add irmp3 tmpbar hbar\n");
	sendtext(lcdproc_fd, "widget_set irmp3 tmpbar 0 0 0\n");
	sendtext(lcdproc_fd, "widget_del irmp3 heartbeat\n");
}


/*************************************************************************
 * SET A DISPLAY WIDGET
 */
void mod_lcdproc_setstring (char *name, int y, char *text, int align)
{
	char buf[128];
	int wid;

	if (!lcdproc_fd || !name)
		return;

	wid = mod_lcdproc_wid;
#if defined(MOD_LCDPROC_PRESERVEUPPERRIGHT) && MOD_LCDPROC_PRESERVEUPPERRIGHT != 0
	if (y == 1 && align == 1)
		wid--;
#endif

	if (!text || !*text)
		sendtext(lcdproc_fd, "widget_set irmp3 %s 0 0 \" \"\n", name);
	else {
		memset(buf, ' ', wid);
		if (align == 1)		// right align
			strcpy(buf+wid-strlen(text), text);
		else if (align == 2)	// center align
			strcpy(buf+((wid-strlen(text))/2), text);
		else			// left align
			strcpy(buf, text);
		sendtext(lcdproc_fd, "widget_set irmp3 %s 1 %d \"%s\"\n", name, y, buf);
	}
}

void mod_lcdproc_setscroller (char *name, int y, char *text)
{
	if (!lcdproc_fd || !name)
		return;

	if (!text || !*text)
		sendtext(lcdproc_fd, "widget_set irmp3 %s 1 1 1 1 h 2 \" \"\n", name);
	else
		sendtext(lcdproc_fd, "widget_set irmp3 %s 1 %d %d %d h 2 \"%s\"\n", name, y, mod_lcdproc_wid, y, text);
}


/*************************************************************************
 * SHOW TEMPORARY STATUS TEXT/BAR
 */
void mod_lcdproc_settmp (char *text, int percent, int centerbar)
{
	int x, wid;
	char buf[128];

	if (!lcdproc_fd)
		return;

	if (!text && !percent && !centerbar) {
		mod_lcdproc_setstring("tmptext", 0, NULL, 0);
		sendtext(lcdproc_fd, "widget_set irmp3 tmpbar 0 0 0\n");
		return;
	}

	mod_lcdproc_tmptimeout = time(NULL) + 2;
	if (text) {
		strcpy(buf, text);
		while (strlen(buf) < mod_lcdproc_wid)
			strcat(buf, " ");
		mod_lcdproc_setstring("tmptext", mod_lcdproc_hgt, buf, 0);
	} else
		mod_lcdproc_setstring("tmptext", 0, NULL, 0);
	if (percent == 0) {
		sendtext(lcdproc_fd, "widget_set irmp3 tmpbar 0 0 0\n");
		return;
	}
	x = text ? strlen(text)+2 : 1;
	if (centerbar) {
		x += (mod_lcdproc_wid-x) / 2;
		percent *= 2;
	}
	wid = (mod_lcdproc_wid - x + 1) * mod_lcdproc_cellwid;
	sendtext(lcdproc_fd, "widget_set irmp3 tmpbar %d %d %d\n", x, mod_lcdproc_hgt, wid*percent/100);
}


/*************************************************************************
 * REFRESH THE DISPLAY
 */
void mod_lcdproc_refreshtime (void)
{
	char buf[128];

	// display remaining time
	if (mod_lcdproc_mode == 1) {
		snprintf(buf, sizeof(buf)-1, "%-5s -%d:%02d:%02d", mod_lcdproc_playmode, mod_lcdproc_time2/3600, mod_lcdproc_time2%3600/60, mod_lcdproc_time2%3600%60);
		mod_lcdproc_setstring("time", 1, buf, 1);

	// display current time
	} else {
		snprintf(buf, sizeof(buf)-1, "%-5s  %d:%02d:%02d", mod_lcdproc_playmode, mod_lcdproc_time1/3600, mod_lcdproc_time1%3600/60, mod_lcdproc_time1%3600%60);
		mod_lcdproc_setstring("time", 1, buf, 1);
	}
}

void mod_lcdproc_refresh (void)
{
	char buf[128];

	log_printf(LOG_DEBUG, "mod_lcdproc_refresh(): refreshing display\n");

	// refresh artist/title
	if (mod_lcdproc_hgt < 4) {
		snprintf(buf, sizeof(buf)-1, "%s - %s", mod_lcdproc_artist, mod_lcdproc_title);
		buf[sizeof(buf)-1] = 0;
		mod_lcdproc_setscroller("artist", 0, NULL);
		mod_lcdproc_setscroller("title", 2, buf);
	} else {
		mod_lcdproc_setscroller("artist", 2, mod_lcdproc_artist);
		mod_lcdproc_setscroller("title", 3, mod_lcdproc_title);
	}

	// refresh status
	if (mod_lcdproc_hgt > 2) {
		snprintf(buf, sizeof(buf)-1, "%s %s %s",
			mod_lcdproc_repeat==0 ? "    " : (mod_lcdproc_repeat==1 ? "Rpt1" : "Rpt+"),
			mod_lcdproc_mute ? "Mute" : "    ",
			mod_lcdproc_shuffle ? "Shfl" : "    ");
		buf[sizeof(buf)-1] = 0;
		mod_lcdproc_setstring("status", mod_lcdproc_hgt, buf, 1);
	}
}


/*************************************************************************
 * POLL INPUT DATA
 */
void mod_lcdproc_poll (int fd)
{
	int rc;
	char s[512], buf[128];
	char *c0, *c1, *c2, *c3;

	// read LCDproc response
	rc = readline(fd, s, sizeof(s));
	if (rc < 0) {
		// close LCDproc connection
		mod_lcdproc_disconnect();
		log_printf(LOG_ERROR, "Lost connection to LCDproc server: %s\n", strerror(errno));
		return;
	}
	if (!*s)
		return;

	// process LCDproc response
	log_printf(LOG_NOISYDEBUG, "mod_lcdproc_poll(): LCDproc response: '%s'\n", s);
	c0 = s ? strdup(s) : NULL;
	c1 = c0 ? strtok(c0, " \t") : NULL;
	c2 = c1 ? strtok(NULL, " \t") : NULL;
	if (c1 && !strcasecmp(c1, "connect")) {
		do {
			c3 = c2 ? strtok(NULL, " \t") : NULL;
			if (c3 && !strcasecmp(c3, "wid")) {
				c3 = c2 ? strtok(NULL, " \t") : NULL;
				mod_lcdproc_wid = c3 ? atoi(c3) : 0;
			} else if (c3 && !strcasecmp(c3, "hgt")) {
				c3 = c2 ? strtok(NULL, " \t") : NULL;
				mod_lcdproc_hgt = c3 ? atoi(c3) : 0;
			} else if (c3 && !strcasecmp(c3, "cellwid")) {
				c3 = c2 ? strtok(NULL, " \t") : NULL;
				mod_lcdproc_cellwid = c3 ? atoi(c3) : 0;
			} else if (c3 && !strcasecmp(c3, "cellhgt")) {
				c3 = c2 ? strtok(NULL, " \t") : NULL;
				mod_lcdproc_cellhgt = c3 ? atoi(c3) : 0;
			}
		} while (c3);
		log_printf(LOG_DEBUG, "mod_lcdproc_poll(): display size %dx%d, cell size %dx%d\n", mod_lcdproc_wid, mod_lcdproc_hgt, mod_lcdproc_cellwid, mod_lcdproc_cellhgt);

	} else if (c1 && !strcasecmp(c1, "listen")) {
		// ignore 'listen' response

	} else if (c1 && !strcasecmp(c1, "ignore")) {
		// ignore 'ignore' response

	} else if (c1 && !strcasecmp(c1, "key") && c2) {
		log_printf(LOG_DEBUG, "mod_lcdproc_poll(): got key '%s'\n", c2);
		snprintf(buf, sizeof(buf)-1, "lcdproc_key_%s", c2);
		c3 = config_getstr(buf, NULL);
		if (c3)
			mod_sendmsg(MSGTYPE_INPUT, c3);

	} else if (c1 && !strcasecmp(c1, "bye")) {
		log_printf(LOG_ERROR, "LCDproc has terminated connection\n");
		mod_lcdproc_disconnect();

	} else {
		log_printf(LOG_DEBUG, "mod_lcdproc_poll(): unknown response: '%s'\n", s);
	}
	free(c0);
}


/*************************************************************************
 * UPDATE
 */
void mod_lcdproc_update (void)
{
	time_t now;

	now = time(NULL);
	if (mod_lcdproc_tmptimeout && now >= mod_lcdproc_tmptimeout) {
		mod_lcdproc_tmptimeout = 0;
		mod_lcdproc_settmp(NULL, 0, 0);
	}
}


/*************************************************************************
 * GO INTO CLOCK MODE
 */
void mod_lcdproc_clockstart()
{

    sendtext(lcdproc_fd, "screen_del irmp3\n");
    sendtext(lcdproc_fd, "screen_add lcdtime\n");
    sendtext(lcdproc_fd, "widget_add lcdtime time string\n");
    sendtext(lcdproc_fd, "widget_add lcdtime date string\n");
	mod_lcdproc_clockmode=1;
	mod_lcdproc_timedisplay();
}

/*************************************************************************
 * DISPLAY THE TIME
 */
void mod_lcdproc_timedisplay(void)
{
	time_t			t;
	struct tm *l_time;
	char			timestr[128];
	char			datestr[128];
	char			x=0,y=0,mx=0,my=0;

	bzero(&timestr,sizeof(timestr));
	t=time(NULL);
								// this code makes the time march
								// around the display (to avoid burn-in)
	mx=mod_lcdproc_wid-10;		
	my=mod_lcdproc_hgt-1;
	if(mx<1) mx=1;				// maximum X starting point
	if(my<1) my=1;				// maximum Y starting point
	x=((t/60)%mx)+1;			// actual X starting point (based on time)
	y=((t/60/mx)%my)+1;			// actual Y starting point 	"	"	"
        
	l_time = localtime(&t);			
	setlocale(LC_TIME,"");		// Get the current locale from environment

	strftime(timestr,sizeof(timestr),config_getstr("lcdproc_timestr","%X"),l_time); 
	strftime(datestr,sizeof(datestr),config_getstr("lcdproc_datestr","%x"),l_time);

	sendtext(lcdproc_fd, "widget_set lcdtime date %d %d \"%s\"\n",x,y,datestr);
	sendtext(lcdproc_fd, "widget_set lcdtime time %d %d \"%s\"\n",x+1,y+1,timestr);
}


	
	
/*************************************************************************
 * GET OUT OF CLOCK MODE
 */
void mod_lcdproc_clockstop()
{
    sendtext(lcdproc_fd, "screen_del lcdtime\n");
	mod_lcdproc_setup();
	mod_lcdproc_clockmode=0;
	mod_lcdproc_refresh();
	mod_lcdproc_refreshtime();
}
	


/*************************************************************************
 * RECEIVE MESSAGE
 */
void mod_lcdproc_message (int msgtype, char *msg)
{
	char *c1, *c2, *c3;
	char buf[128];
	int i, j;

	if (!lcdproc_fd)
		return;


	// pre-process input mesage for handling IDLE messages.
	if (msgtype == MSGTYPE_IDLE) {
		if(mod_lcdproc_clockmode) 	// clock is already started
			mod_lcdproc_timedisplay();
		else if (mod_lcdproc_clocktimer && mod_lcdproc_clocktimer <= strtol(msg,(char **)NULL,10) ) {
			mod_lcdproc_clockstart();
		}
	} else if (mod_lcdproc_clockmode) 
		mod_lcdproc_clockstop();
			

	// process input messages
	if (msgtype == MSGTYPE_INPUT) {
		c1 = msg ? strtok(msg, " \t") : NULL;

		// displaymode
		if (c1 && !strcasecmp(c1, "displaymode")) {
			mod_lcdproc_mode++;
			if (mod_lcdproc_mode > 1)
				mod_lcdproc_mode = 0;
			mod_lcdproc_refreshtime();
		} else if (c1 && !strcasecmp(c1,"loadplaylist")) {

			// display a playlist name here...

			strcpy (buf,&c1[13]);
			mod_lcdproc_settmp(buf,0,0);
		}


	// process player messages
	} else if (msgtype == MSGTYPE_PLAYER) {
		c1 = msg ? strtok(msg, " \t") : NULL;
		c2 = c1 ? strtok(NULL, " \t") : NULL;

		// play
		if (c1 && !strcasecmp(c1, "play")) {
			mod_lcdproc_playmode = "PLAY";
			mod_lcdproc_refreshtime();

		// stop
		} else if (c1 && !strcasecmp(c1, "stop")) {
			mod_lcdproc_playmode = "STOP";
			mod_lcdproc_time1 = 0;
			mod_lcdproc_time2 = 0;
			mod_lcdproc_refreshtime();
			strcpy(mod_lcdproc_artist, "");
			strcpy(mod_lcdproc_title, "");
			mod_lcdproc_refresh();

		// pause
		} else if (c1 && !strcasecmp(c1, "pause")) {
			mod_lcdproc_playmode = "PAUSE";
			mod_lcdproc_refreshtime();

		// song info
		} else if (c1 && !strcasecmp(c1, "info")) {
			c3 = c2 ? strtok(NULL, "") : NULL;
			if (c2 && !strcasecmp(c2, "title"))
				strcpy(mod_lcdproc_title, c3 ? c3 : "");
			else if (c2 && !strcasecmp(c2, "artist"))
				strcpy(mod_lcdproc_artist, c3 ? c3 : "");
			mod_lcdproc_refresh();

		// time info
		} else if (c1 && !strcasecmp(c1, "time")) {
			c3 = c2 ? strtok(NULL, " \t") : NULL;
			mod_lcdproc_time1 = c2 ? atoi(c2) : 0;
			mod_lcdproc_time2 = c3 ? atoi(c3) : 0;
			mod_lcdproc_refreshtime();
		}

	// process mixer messages
	} else if (msgtype == MSGTYPE_MIXER) {
		c1 = msg ? strtok(msg, " \t") : NULL;
		c2 = c1 ? strtok(NULL, " \t") : NULL;

		if (c1 && !strcasecmp(c1, "volume"))
			mod_lcdproc_settmp("Vol", c2 ? atoi(c2) : 0, 0);
		else if (c1 && !strcasecmp(c1, "balance"))
			mod_lcdproc_settmp("Balance", c2 ? atoi(c2) : 0, 1);
		else if (c1 && !strcasecmp(c1, "bass"))
			mod_lcdproc_settmp("Bass", c2 ? atoi(c2) : 0, 0);
		else if (c1 && !strcasecmp(c1, "treble"))
			mod_lcdproc_settmp("Treble", c2 ? atoi(c2) : 0, 0);
		else if (c1 && !strcasecmp(c1, "mute")) {
			mod_lcdproc_mute = c2 ? atoi(c2) : 0;
			mod_lcdproc_refresh();
		}


	// process generic messages
	} else if (msgtype == MSGTYPE_GENERIC) {
		c1 = msg ? strtok(msg, " \t") : NULL;
		c2 = c1 ? strtok(NULL, " \t") : NULL;
		c3 = c2 ? strtok(NULL, " \t") : NULL;

		// repeat mode
		if (c1 && !strcasecmp(c1, "repeat")) {
			mod_lcdproc_repeat = c2 ? atoi(c2) : 0;
			mod_lcdproc_refresh();

		// shuffle mode
		} else if (c1 && !strcasecmp(c1, "shuffle")) {
			mod_lcdproc_shuffle = c2 ? atoi(c2) : 0;
			mod_lcdproc_refresh();

		// sleep timer
		} else if (c1 && !strcasecmp(c1, "sleep")) {
			i = c2 ? atoi(c2) : 0;
			j = c3 ? atoi(c3) : 0;
			if (!i)
				mod_lcdproc_settmp("Sleep timer off", 0, 0);
			else {
				if (j)
					sprintf(buf, "Sleep %d/%d min", i, j);
				else
					sprintf(buf, "Sleep %d min", i);
				mod_lcdproc_settmp(buf, 0, 0);
			}
		} else if (c1 && !strcasecmp(c1,"alarm")) {
			i = c2 ? atoi(c2) : 0;
			if (i) 
				mod_lcdproc_settmp("Alarms enabled",0,0);
			else
				mod_lcdproc_settmp("Alarms disabled",0,0);
		}
	}
}


/*************************************************************************
 * MODULE INIT FUNCTION
 */
char *mod_lcdproc_init (void)
{
	char *s;

	FD_ZERO(&mod_lcdproc_fdset);

	// register our module
	mod_register(&mod_lcdproc);

	// connect to LCDproc server
	s = mod_lcdproc_connect();
	if (s)
		return s;
	log_printf(LOG_DEBUG, "mod_lcdproc_init(): socket connected\n");

	// setup LCDproc
	mod_lcdproc_setup();

	// get clock timer
	mod_lcdproc_clocktimer=60*config_getnum("clock_timer",0);

	return NULL;
}


/*************************************************************************
 * MODULE DEINIT FUNCTION
 */
void mod_lcdproc_deinit (void)
{
	// close LCDproc connection
	mod_lcdproc_disconnect();

	log_printf(LOG_DEBUG, "mod_lcdproc_deinit(): socket closed\n");
}


/*************************************************************************
 * MODULE RELOAD FUNCTION
 */
char *mod_lcdproc_reload (void)
{
	char *s;

	log_printf(LOG_DEBUG, "mod_lcdproc_reload(): reconnecting\n");

	// close and reconnect to LCDproc server
	s = mod_lcdproc_connect();
	if (s)
		return s;

	// setup LCDproc
	mod_lcdproc_setup();

	// refresh display
	mod_lcdproc_refreshtime();
	mod_lcdproc_refresh();

	return NULL;
}


/*************************************************************************
 * EOF
 */
