/*
        IRE game engine

Copyright (c) 2003, IT-HE Software
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

Redistributions of source code must retain the above copyright notice, this
list of conditions and the following disclaimer.

Redistributions in binary form must reproduce the above copyright notice,
this list of conditions and the following disclaimer in the documentation
and/or other materials provided with the distribution.

Neither the name of IT-HE Software nor the names of its contributors may
be used to endorse or promote products derived from this software without
specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED.
IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef _WIN32
	#include <unistd.h>
	#ifdef __DJGPP__
		#include <conio.h>
	#endif
#else
	#include <conio.h>
#endif


#include <malloc.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <sys/stat.h>
#include <time.h>

#include <allegro.h>

#include "core.hpp"
#include "console.h"
#include "media.h"
#include "ithelib.h"
#include "oscli.h"      // Operating System Command Line Interpreter
#include "cookies.h"
#include "gamedata.h"
#include "loadfile.h"
#include "library.hpp"
#include "sound.h"
#include "init.h"
#include "linklist.hpp"
#include "tdgui.h"


/*
 * Defines
 */

/*
 * Variables
 */

BITMAP *gamewin=NULL;        // Projection window
BITMAP *roofwin;        // Projection window
char imgcachedir[1024];
extern char *pevm_context;

/*
 * Functions
 */

void RFS_getescape();
static void read_config();
static void write_config();

static void BookViewer();
static void RunGame();
static int SetVol(void *dp3, int val);

extern void LoadResources(char *filename);
extern void InitVM();
extern void CallVM(char *func);
extern void initgame();
extern void startgame();
extern void LoadMap(int mapno);
extern int data_choose(char *path,char *path_file,int size);

#ifdef __DJGPP__
#include <crt0.h>
int _crt0_startup_flags =_CRT0_FLAG_FILL_SBRK_MEMORY | _CRT0_FLAG_FILL_DEADBEEF | _CRT0_FLAG_LOCK_MEMORY;
#endif

// Code

int main(int argc, char *argv[])
{
char fname[1024];
char homepath[1024];
int ret;

#ifdef FORTIFY
remove("fortify.out");
#endif

curmap = &worldcache[0]; // For multiple map support later
curmap->object=NULL;

allegro_init(); // We need this ASAP

projectname[0]=0;  // Blank the project title
projectdir[0]=0;  // Blank the project title

M_init();         // Init memory systems

party = (OBJECT **)M_get(MAX_MEMBERS,sizeof(OBJECT *));
partyname = (char **)M_get(MAX_MEMBERS,sizeof(char *));

// Cache these linked lists to speed up loading levels
LLcache_register(&MasterList);
LLcache_register(&ActiveList);

#ifdef __SOMEUNIX__
	ihome(homepath);
	sprintf(newbootlog,"%s%s",homepath,"bootlog.txt"); 
	sprintf(oldbootlog,"%s%s",homepath,"bootlog.prv"); 
#else
	strcpy(newbootlog,"bootlog.txt"); // This could be changed later in OSCLI
	strcpy(oldbootlog,"bootlog.prv"); // If I remember to do so..
	strcpy(homepath,"");
#endif

ilog_start(newbootlog,oldbootlog);

// Write the bootlog to both screen and file

ilog_printf("IRE game runtime, Copyright (C) 2004 IT-HE Software\n");
ilog_printf("IRE Kernel version: %s\n",IKV);
ilog_printf("Build date: %s at %s\n",__DATE__,__TIME__);
ilog_printf("IT-HE lib version: %s\n",ithe_get_version());
ilog_printf("Allegro library: %s\n",ALLEGRO_VERSION_STR);
ilog_printf("\n");

// Now get the config settings (this may print stuff if RARfiles are added)

if(argc>=2)
	OSCLI(argc,argv);       // input from the OS commandline interpreter

// Okay, let's look for the user config file
sprintf(fname,"%s%s",homepath,"game.ini");
if(!INI_file(fname))
	INI_file("game.ini"); // Okay, try current directory then

// Now we need to find the main .ini file to tell us what to do.

// Do we have a project directory, passed from the commandline?
ret=0;
if(!projectdir[0])
	{
	// No, assume it's the current directory then
	strcpy(projectdir,"./");
	}

sprintf(fname,"%sgamedata.ini",projectdir);
ret=INI_file(fname);
if(!ret)
	ret=INI_file("gamedata.ini");

if(!ret)
	{
	init_media(1);
	if(!data_choose(projectdir,fname,sizeof(fname)))
		{
		term_media();
		exit(1);
		}
	sprintf(fname,"%sgamedata.ini",projectdir);
	ret=INI_file(fname);
		if(!ret)
			{
			term_media();
			printf("Could not load file '%s'\n",fname);
			exit(1);
			}
	}

// Final checks
if(!projectname[0])
	ithe_safepanic("No project was specified.","There should be a -project <projectname> line in one of the INI files.");

// Use the project directory for all file accesses
ifile_prefix=&projectdir[0];


#ifndef _WIN32
if(!bookview[0]) // Book viewer wants fast startup
	sleep(1);    // Wait a little so the user can see the boot prompt
#endif

init_media(1); // Enter graphics mode now, for graphical popup

ilog_printf("Loading game: %s\n",projectname);
ilog_printf("\n");
#ifndef _WIN32
sleep(1);
#endif

// Now do the map data calculations
VSA=VSW*VSH;		// Viewscreen area
VSMIDX=VSW/2;		// Viewscreen X middle
VSMIDY=VSH/2;		// Viewscreen Y middle
VSW32=VSW*32;		// Viewscreen width in tiles
VSH32=VSH*32;		// Viewscreen height in tiles
VSA32=VSW32*VSH32;	// Viewscreen area in tiles

// Check the mapfile is there before we commit ourselves
ilog_printf("\n");

if(!mapnumber)
	ithe_safepanic("No map file was specified, or invalid map number.","There should be a -map <n> line in one of the INI files.");

// Load in all the fonts, so we can go graphical..
irecon_loadfonts();

init_enginemedia(); // Finish graphics setup for game

// Work out the name of the image cache directory
#ifdef _WIN32
	sprintf(imgcachedir,"cache\\%s\\%s.%d\\",".",projectname,ire_bpp);
#else
	// Get home directory for config files/stuff
	sprintf(imgcachedir,"%s/cache/%s.%d/",homepath,projectname,ire_bpp);
#endif

ilog_printf("Console started\n");
ilog_printf("\n");

// Play some startup music if any
if(fileexists("startup.ogg"))
	StreamSong_start("startup.ogg");

if(!fileexists("resource.txt"))
	if(fileexists("main.txt"))
		ithe_panic("Resource file not found..","Please rename your main.txt file to resource.txt");
	else
		ithe_panic("Resource file not found..","File resource.txt is missing!");

ilog_printf("Load Resources\n");
LoadResources("resource.txt");

// Look for a mouse pointer (optional)
if(loadfile("mouseptr.cel",fname))
	{
	MouseOverPtr=iload_bitmap(fname);
	if(!MouseOverPtr)
		{
		// Stop if the promise couldn't be delivered
		ithe_panic("Could not load mouse pointer",fname);
		}
	}

ilog_printf("Init Random Number Generator\n");
srand(time(0));                 // Initialise rng
qrand_init();

irecon_term();


if(bookview[0])
	BookViewer();
else
	RunGame();

term_media();

if(buggy)
	{
	printf("There were bugs in the VRM code.\n");
	printf("Check 'bootlog.txt' and search for 'BUG:'\n");
	}

return 0;			// Return a value to keep compiler happy
}
END_OF_MAIN();

/*
 *  Run the game
 */

void RunGame()
{
// Ensure we have a valid console for this bit
clear_bitmap(swapscreen);
irecon_init(conx,cony,conwid,conlen);

ilog_printf("Load Sound and Music\n");
S_Load();

read_config();

ilog_printf("VRM fixups\n");
Init_Funcs();                   // Assign the CHlist.func to appropriate VRM

ilog_printf("Starting PEVM microkernel\n");
InitVM();

ilog_quiet("Creating temporary savegame\n");
makesavegamedir(9999);

ilog_quiet("Engine pre-init\n");
initgame();
ire_running=1; // Ensure game is running

ilog_quiet("Calling PE: initproc\n");
pevm_context="Pre-game startup";

// Tear down the old console and build a new one, or not
irecon_term();
clear_bitmap(swapscreen);

if(!SkipInit)
	{
	// Call the init procedure
	CallVM("initproc");
	}
else
	{
	// Do some initialisation ourselves
	LoadBacking(swapscreen,"backings/panel.pcx");
	}

// Create the console anyway.  If we already have, it will be ignored
irecon_init(conx,cony,conwid,conlen);

// If user quit, don't launch the game core
if(!ire_running)
	return;

if(ire_running != -1) // If -1, we loaded an existing game
	{
	ilog_printf("Load Map\n");
	mapnumber = default_mapnumber;
	LoadMap(mapnumber);                   // Load in the game map
	}

LoadBacking(swapscreen,"backings/panel.pcx");

// Okay, start the game up

ilog_printf("Start Game\n");

startgame();                     // Do things
write_config();
ilog_printf("Stop Game\n");
}

/*
 *  Run the engine as a book viewer
 */

void BookViewer()
{
int bret;

irecon_init(conx,cony,conwid,conlen);

// Fake the entries that the engine might use for books
player=OB_Alloc();
victim=OB_Alloc();
person=player;
current_object=victim;

if(bookview2[0])
	bret=talk_to(bookview,bookview2);
else
	bret=talk_to(bookview,NULL);
}




/*
 *      read_config - get previous volume levels from file
 */

void read_config()
{
char string[128];
FILE *config;

// Make filename to store volume settings
ihome(string);
strcat(string,"/volume.cfg");

config = fopen(string,"rb");
if(!config)
	return;

mu_volume = getw(config);
sf_volume = getw(config);

S_MusicVolume(mu_volume);    // Update sound engine
S_SoundVolume(sf_volume);

fclose(config);
}

/*
 *      write_config - write current volume levels to file
 */

void write_config()
{
char string[1024];
FILE *config;

// Make filename to store volume settings
ihome(string);
strcat(string,"/volume.cfg");

config = fopen(string,"wb");
if(!config)
	return;

putw(mu_volume,config);
putw(sf_volume,config);

fclose(config);
}


void RFS_getescape()
{
while((readkey() >> 8) != KEY_ESC);
}


/*
 *      LoadBacking - load a picture into a pre-existing bitmap
 */

void LoadBacking(BITMAP *img,char *fname)
{
char filename[1024];
BITMAP *x;

if(!loadfile(fname,filename))
	{
	Bug("Cannot open image file '%s'\n",fname);
	return;
	}

x = iload_bitmap(filename);
if(!x)
	{
	Bug("Cannot load image file '%s': unsupported format?\n",filename);
	return;
	}
    
draw_sprite(img,x,0,0);
destroy_bitmap(x);
}


//
//  Sound volume callback
//

int SetVol(void *dp3, int val)
{
*(int *)dp3=val;
S_MusicVolume(mu_volume);    // Update sound settings
S_SoundVolume(sf_volume);

return 0;
}

//
//  Soundbar dialogue structure
//

static DIALOG Soundbar_dialog[] =
	{
		/* (dialog proc)        (x)   (y)   (w)   (h)   (fg)  (bg)  (key) (flags)  (d1)  (d2)  (dp)    (dp2)       (dp3)*/
		{ d_billwin_proc,        0,    8,    296,  80,   0,    0,    0,    0,       0,    0,    (void *)"", NULL,  NULL },
		{ d_text_proc,           16,  16,    256,  8,    0,    255,  0,    0,       0,    0,    (void *)"Music volume", NULL,  NULL },
		{ d_billslider_proc,     16,  24,    256,  16,   0,    0,    0,    0,     255,    0,    NULL,  (void *)SetVol, &mu_volume},
		{ d_text_proc,           16,  56,    256,  8,    0,    255,  0,    0,       0,    0,    (void *)"Sound volume", NULL,  NULL },
		{ d_billslider_proc,     16,  64,    256,  16,   0,    0,    0,    0,     255,    0,    NULL,  (void *)SetVol, &sf_volume},
		{ NULL }
	};

//
//  Run teh sound settings dialogue
//

void SoundSettings()
{
// Set defaults
Soundbar_dialog[1].bg = makecol(192,192,192);
Soundbar_dialog[2].d2 = mu_volume;
Soundbar_dialog[3].bg = makecol(192,192,192);
Soundbar_dialog[4].d2 = sf_volume;

// Do the dialog
centre_dialog(Soundbar_dialog);
moveable_do_dialog(Soundbar_dialog, 1);
}


