// -*- C++ -*-
// --------------------------------------------------------------------
// Creating PDF output
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2005  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#ifndef IPEPDFWRITER_H
#define IPEPDFWRITER_H

#include "ipebase.h"
#include "ipepage.h"
#include "ipedoc.h"
#include "ipeimage.h"

class IpeFontPool;

// --------------------------------------------------------------------

class IpePdfPainter : public IpePainter {
public:
  IpePdfPainter(const IpeStyleSheet *style, IpeStream &stream);
  virtual ~IpePdfPainter() { }

  virtual void Rect(const IpeRect &re);

protected:
  virtual void DoPush();
  virtual void DoPop();
  virtual void DoNewPath();
  virtual void DoMoveTo(const IpeVector &v);
  virtual void DoLineTo(const IpeVector &v);
  virtual void DoCurveTo(const IpeVector &v1, const IpeVector &v2,
			 const IpeVector &v3);
  virtual void DoClosePath();
  virtual void DoDrawPath();
  virtual void DoDrawBitmap(IpeBitmap bitmap);
  virtual void DoDrawText(const IpeText *text);

protected:
  void DrawColor(IpeStream &stream, IpeAttribute color,
		 const char *gray, const char *rgb);
  void DrawAttributes();

protected:
  IpeStream &iStream;
  // iActiveState records the attribute settings that have been
  // recorded in the PDF output, to avoid repeating them
  // over and over again.
  StateSeq iActiveState;
};

// --------------------------------------------------------------------

class IpePdfWriter {
public:
  IpePdfWriter(IpeTellStream &stream, const IpeDocument *doc,
	       bool noShading, bool lastView, int compression);
  ~IpePdfWriter();

  void EmbedFonts(const IpeFontPool *pool);
  void CreatePages();
  void CreateBookmarks();
  void CreateXmlStream(IpeString xmldata, bool preCompressed);
  void CreateTrailer(IpeString creator);

private:
  void CreatePageView(int page, int view);
  int StartObject(int objnum = -1);
  void CreateStream(const char *data, int size, bool preCompressed);
  void WriteString(IpeString text);
  void EmbedBitmap(IpeBitmap bitmap);
  IpeRect PaintView(IpeStream &stream, int pno, int view);

private:
  IpeTellStream &iStream;
  const IpeDocument *iDoc;
  //! Save only last view of each page?
  bool iLastView;
  //! Obj id of XML stream.
  int iXmlStreamNum;
  //! Obj id of font resource dictionary.
  int iResourceNum;
  //! Obj id of shading dictionary.
  int iShadingNum;
  //! Obj id of outline dictionary.
  int iBookmarks;
  //! Compression level (0..9).
  int iCompressLevel;
  //! Obj id of font for page numbers.
  int iPageNumberFont;

  std::vector<IpeBitmap> iBitmaps;
  //! Next unused PDF object number.
  int iObjNum;
  //! List of pages, expressed as Pdf object numbers.
  std::vector<int> iPageObjectNumbers;
  //! List of file locations, in object number order (starting with 0).
  std::map<int, long> iXref;
};

// --------------------------------------------------------------------
#endif
