-- statusd_netmon.lua: monitor the speed of a network interface
--
-- Thanx to Tuomo for pointing out a problem in the previous script.
-- 
-- In case this doesn't work for someone, do let me know :)
--
-- Author
-- Sadrul Habib Chowdhury (Adil)
-- imadil at gmail dot com

local defaults = {
    device = "eth0",
    show_avg = 1,       -- show average stat?
    avg_sec = 60,       -- default, shows average of 1 minute
    show_count = 0,     -- show tcp connection count?
    interval = 1*1000   -- update every second
}

local timer = nil       -- the timer
local positions = {}    -- positions where the entries will be
local last = {}         -- the last readings
local history_in = {}   -- history to calculate the average
local history_out = {}
local total_in, total_out = 0, 0
local counter = 0       --

local settings = table.join(statusd.get_config("netmon"), defaults)

--
-- tokenize the string
--
local function tokenize(str)
    local ret = {}
    local i = 0
    local k = nil

    for k in string.gfind(str, '(%w+)') do
        ret[i] = k
        i = i + 1
    end
    return ret
end

-- 
-- get the connection count
--
local function get_connection_count()
    -- local f = io.popen('netstat -st | grep -e "[0-9]\\+ connections established" | sed s/"[ ]\\+\\([0-9]\\+\\) connections established"/\\\\1/', 'r')
    local f = io.popen('netstat -st | grep "connections established"', 'r')
    if not f then return 'q' end

    local ret = f:read('*a')
    ret = string.gsub(ret, '%s*(%d+).+', '%1')
    f:close()
    return tostring(ret)
end

--
-- calculate the average
--
local function calc_avg(lin, lout)
    if counter == settings.avg_sec then
        counter = 0
    end

    total_in = total_in - history_in[counter] + lin
    history_in[counter] = lin

    total_out = total_out - history_out[counter] + lout
    history_out[counter] = lout

    counter = counter + 1

    return string.format("%.1fK/%.1fK", total_in/settings.avg_sec, total_out/settings.avg_sec)
end

--
-- parse the information
--
local function parse_netmon_info()
    local s
    local lin, lout

    for s in io.lines('/proc/net/dev') do
        local f = string.find(s, settings.device)
        if f then
            local t = tokenize(s)
            return t[positions[0]], t[positions[1]]
       end
    end
    return nil, nil
end

--
-- update the netmon monitor
--
local function update_netmon_info()
    local s
    local lin, lout

    lin, lout = parse_netmon_info()
    if not lin or not lout then
        -- you should never reach here
        statusd.inform("netmon", "oops")
        statusd.inform("netmon_hint", "critical")
        return
    end

    last[0], lin = lin, lin - last[0]
    last[1], lout = lout, lout - last[1]

    local output = string.format("%.1fK/%.1fK", lin/1024, lout/1024)

    if settings.show_avg == 1 then
        output = output .. " (" .. calc_avg(lin/1024, lout/1024) .. ")"
    end
    if settings.show_count == 1 then
        output = "[" .. get_connection_count().."] " .. output
    end
    statusd.inform("netmon", output)
    timer:set(settings.interval, update_netmon_info)
end

--
-- is everything ok to begin with?
--
local function sanity_check()
    local f = io.open('/proc/net/dev', 'r')
    local e

    if not f then
        return false
    end

    local s = f:read('*line')
    s = f:read('*line')         -- the second line, which should give
                                -- us the positions of the info we seek

    local t = tokenize(s)
    local n = table.getn(t)
    local i = 0

    for i = 0,n do
        if t[i] == "bytes" then
            positions[0] = i
            break
        end
    end

    i = positions[0] + 1

    for i=i,n do
        if t[i] == "bytes" then
            positions[1] = i
            break
        end
    end

    if not positions[0] or not positions[1] then
        return false
    end

    s = f:read('*a')        -- read the whole file
    if not string.find(s, settings.device) then
        return false        -- the device does not exist
    end
    
    return true
end

--
-- start the timer
-- 
local function init_netmon_monitor()
    if sanity_check() then
        timer = statusd.create_timer()
        last[0], last[1] = parse_netmon_info()
        
        if settings.show_avg == 1 then
            for i=0,settings.avg_sec-1 do
                history_in[i], history_out[i] = 0, 0
            end
        end
        
        statusd.inform("netmon_template", "xxxxxxxxxxxxxxxxxxxxxxx")
        update_netmon_info()
    else
        statusd.inform("netmon", "oops")
        statusd.inform("netmon_hint", "critical")
    end
end

init_netmon_monitor()

