/***************************************************************************
                         modifier.cpp  -  description
                            -------------------
   begin                : Mon Apr 16 2001
   copyright            : (C) 2001 by Jon Anderson
   email                : janderson@onelink.com
***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "modifier.h"
#include "modetoolbar.h"
#include <Commands/commandlib.h>
#include <selectionrecord.h>
#include <objectdb.h>
#include <Entities/vertex.h>
#include <i3d.h>
#include <i3dworkspace.h>
#include <stdlib.h>
#include <stdio.h>
#include <assert.h>

/**
  * Construct a Modifier object.
  */
Modifier::Modifier( int type ) : IControl(), m_selection_center(), m_press(), m_clipboard()
{
   setControlType( MODIFIER );
   m_modify_type = type;

   m_allmenu = new QPopupMenu();

   QPopupMenu * p = new QPopupMenu();
   m_axis_menu = new QPopupMenu();

   p -> insertItem( "Uniform", this, SLOT( slotScaleUniformMode() ) );
   p -> insertItem( "Free", this, SLOT( slotScaleFreeMode() ) );

   m_allmenu -> insertItem( "Select", this, SLOT( slotSelectMode() ) );
   m_allmenu -> insertSeparator();
   m_allmenu -> insertItem( "Rotate", this, SLOT( slotRotateMode() ) );
   m_allmenu -> insertItem( "Move", this, SLOT( slotMoveMode() ) );
   m_allmenu -> insertItem( "Scale", p );
   m_allmenu -> insertSeparator();
   menu_multiselect = m_allmenu -> insertItem( "Multi-Select", this, SLOT( slotToggleMultiSelect() ) );

   menu_x_axis = m_axis_menu -> insertItem( "X Axis", this, SLOT( slotToggleX() ) );
   menu_y_axis = m_axis_menu -> insertItem( "Y Axis", this, SLOT( slotToggleY() ) );
   menu_z_axis = m_axis_menu -> insertItem( "Z Axis", this, SLOT( slotToggleZ() ) );

   m_allmenu -> insertItem( "Toggle Axis...", m_axis_menu );

   m_popup -> insertItem( "Create...", I3D::getInstance() -> getCreateMenu() );
   m_popup -> insertItem( "Modify...", I3D::getInstance() -> getModifyMenu() );

   m_popup -> insertSeparator();

   m_popup -> insertItem( "Transforms...", m_allmenu );



   // get an empty pointer
   m_selection = ObjectDB::getInstance() -> getSelected();

}

Modifier::~Modifier()
{
}


/**
  * Handles any selection events that are passed through from
  * the workspace
  * passed through from a workspace.
  */
void Modifier::filterSelection( int *array, int n, int mode )
{

   SelectionRecord * sr = new SelectionRecord( ( GLuint * ) array, n );

   if ( ModeToolbar::isMultipleSelect() )
      m_selection = sr->getSelected( m_modify_type, mode | SELECT_MULTIPLE );
   else
      m_selection = sr->getSelected( m_modify_type, mode | SELECT_SINGLE );

   /*
   SelectionCenter c;
   std::for_each( m_selection -> begin(), m_selection -> end(), c );
   */
   getCenter( & m_selection_center );

   int ns = ( int ) m_selection->size();

   QString strMessage = "Selected ";

   QString num;
   num.setNum( ns );
   strMessage += num;
   strMessage += " items: ";

   string names;
    
   if(ns == 0)
      names = "(none)";
   else
   {
      for(vector<Selectable *>::iterator iter = m_selection->begin();iter < m_selection->end(); iter++)
      {
         Entity* entity = NULL;
         entity = (Entity *) (*iter);

         if(entity)
         {
            if(entity->isA(Object::TYPE))
            {
               if(iter != m_selection->begin())
                  names += ", ";
               names += entity->getName();

            }
         }	
      }
   }

   selectionEvent();

   strMessage += names.c_str();
   setStatus( ( char * ) strMessage.ascii() );
   updateViews();
}

/**
  * Handles any mouse move events that are
  * passed through from a workspace.
  */
void Modifier::mouseMove ( Vector4 &plane, Vector4 &pt, int )
{
   Vector4 m_old_move;

   m_old_move = m_move;
   m_move = pt;

   int mode = ModeToolbar::getMode();
   //ignore selecting mouse moves.
   if ( mode != ModeToolbar::MODE_ROTATE &&
         mode != ModeToolbar::MODE_MOVE &&
         mode != ModeToolbar::MODE_SCALE_FREE &&
         mode != ModeToolbar::MODE_SCALE_UNIFORM )
      return ;

   if ( m_selection -> size() == 0 )
      return ;

   Vector4 diff;
	 Vector4 sign;
	
   diff = m_move - m_old_move;

   ( diff.x > 0 ) ? sign.x = 1 : sign.x = -1;
   ( diff.y > 0 ) ? sign.y = 1 : sign.y = -1;
   ( diff.z > 0 ) ? sign.z = 1 : sign.z = -1;

   Vector4 workingAxis = ModeToolbar::getAxis();

   //only limit movement to the selected axis for scale an move here.
   //rotate takes care of this differently.		
   if ( mode == ModeToolbar::MODE_SCALE_FREE ||
         mode == ModeToolbar::MODE_SCALE_UNIFORM ||
         mode == ModeToolbar::MODE_MOVE )
   {
     diff.x *= workingAxis.x;
     diff.y *= workingAxis.y;
  	 diff.z *= workingAxis.z;
	}

   // if we are scaling, we need to adjust to a 1 median.
   if ( mode == ModeToolbar::MODE_SCALE_FREE ||
         mode == ModeToolbar::MODE_SCALE_UNIFORM )
   {

      diff.x += 1;
      diff.y += 1;
      diff.z += 1;
   }

   // if we are scaling uniformly, lock all axis's together.
   if ( mode == ModeToolbar::MODE_SCALE_UNIFORM )
   {
      float x = ( diff.x + diff.y + diff.z ) / 3;
      diff.assign( x, x, x, 1 );
   }

		
   if ( mode == ModeToolbar::MODE_ROTATE )
   {
   		//YZ plane
			if( plane == Vector4(1, 0, 0, 1) )
			{
				diff.x = diff.z;
				diff.z = - diff.y;
				diff.y = 0;
			}
			
			//XZ plane			
			if( plane == Vector4(0, 1, 0, 1) )
			{
				diff.y = - diff.x;
				diff.x = diff.z;
				diff.z = 0;
			}
			
			//Xy plane			
			if( plane == Vector4(0, 0, 1, 1) )
			{
				diff.z = - diff.x;
				diff.x = - diff.y;
				diff.y = 0;
			}
			//otherwise, leave alone.
			//limit axis.
     diff.x *= workingAxis.x;
     diff.y *= workingAxis.y;
  	 diff.z *= workingAxis.z;
  	
  	 diff.w = 10;
   }

   transform( diff, m_selection_center );

   updateViews();

}

void Modifier::transform( Vector4 &diff, Vector4 &center )
{
   int mode = ModeToolbar::getMode();
   SelectionIterator it = m_selection -> begin();

   while ( it != m_selection -> end() )
   {

      Selectable * s = *it;

      switch ( mode )
      {
         case ModeToolbar::MODE_ROTATE:
            s -> rotate( diff, m_selection_center );
            break;
         case ModeToolbar::MODE_MOVE:
            s -> move( diff );
            break;
         default:
            s -> scale( diff, m_selection_center );
            break;
      }

      ++it;
   }
}

/**
  * Handles any mouse press events that are
  * passed through from a workspace.
  */
void Modifier::mousePress ( Vector4 &plane, Vector4 &pt, int )
{
   m_press = pt;
   m_move = pt;

   //ignore selecting mouse presses.
   if ( ModeToolbar::getMode() == ModeToolbar::MODE_SELECT )
      return ;

   if ( m_selection -> size() == 0 )
      return ;


   startTransform();

}

/**
  * Handles any mouse release events that are
  * passed through from a workspace.
  */
void Modifier::mouseRelease( Vector4 &plane, Vector4 &pt, int )
{
   m_release = pt;

   //ignore selecting mouse presses.
   if ( ModeToolbar::getMode() == ModeToolbar::MODE_SELECT )
      return ;

   if ( m_selection == 0 || m_selection -> size() == 0 )
      return ;

   endTransform();

   // recalculate the selected center.
   getCenter( & m_selection_center );

   updatePlaceLabel();

}

/**
  * Handles any key input that is passed
  * in from a view.
  */
void Modifier::keyEvent( QKeyEvent * e )
{
   /**Keyboard shortcuts for basic mouse modes. */
   /*
    TODO: Evaluate to config file settings
    switch ( e->key() )
   {
      case Key_A:
         slotSelectMode();
         break;
      case Key_D:
         slotMoveMode();
         break;
      case Key_S:
         slotRotateMode();
         break;
      case Key_F:
         slotScaleFreeMode();
         break;
      case Key_G:
      		slotScaleUniformMode();
      		break;
      default:
         break;
   }*/
}

/**
  * Draws any modifier specific helpers.
  * Default draws nothing.
  */
void Modifier::draw()
{
}


void Modifier::startTransform()
{
   m_cmds.clear();

cerr << "Starting Transform..."<<endl;
   int mode = ModeToolbar::getMode();

   Command *c;

   SelectionIterator it = m_selection -> begin();

	// check for bogus start of transform when creating things.
	if( mode == ModeToolbar::MODE_SELECT )
	{
		cerr << "Bad transform start. Ignoring."<<endl;
		return;
	}

   while ( it != m_selection -> end() )
   {

      Selectable * s = *it;

      if ( mode == ModeToolbar::MODE_ROTATE ||
            mode == ModeToolbar::MODE_MOVE )
      {
         c = new TransformCmd( s );
         static_cast<TransformCmd *>( c ) -> begin();
      }

      else
      {
         c = new CheckPointCmd( static_cast<Entity *>( s ) );
      }

      m_cmds.insert( CommandPair( s, c ) );
      ++it;

   }

}

void Modifier::endTransform()
{
cerr << "Ending transform..."<<endl;
	if( ModeToolbar::getMode() == ModeToolbar::MODE_SELECT || m_cmds.size()==0 )
	{
		cerr << "Bad transform end. Ignoring."<<endl;
		return;
	}
   TransactionCommand * tc = new TransactionCommand();
   int mode = ModeToolbar::getMode();

   Command *c;

   SelectionIterator it = m_selection -> begin();

   while ( it != m_selection -> end() )
   {

      Selectable * s = *it;
      c = m_cmds[ s ];

      assert( c );

      if ( mode == ModeToolbar::MODE_ROTATE ||
            mode == ModeToolbar::MODE_MOVE )
      {
         static_cast<TransformCmd *>( c ) -> end();
      }

      else
      {
         //don't need to tag the end of a checkpoint.
      }

      tc -> addCommand( c );

      ++it;
   }

   //don't need to execute...
   tc -> save();

   //empty out the command map.
   m_cmds.clear();

}

void Modifier::slotSelectMode() { ModeToolbar::setMode( ModeToolbar::MODE_SELECT ); updateViews();}

void Modifier::slotMoveMode() { ModeToolbar::setMode( ModeToolbar::MODE_MOVE ); updateViews();}

void Modifier::slotRotateMode() { ModeToolbar::setMode( ModeToolbar::MODE_ROTATE ); updateViews();}

void Modifier::slotScaleFreeMode() { ModeToolbar::setMode( ModeToolbar::MODE_SCALE_FREE ); updateViews();}

void Modifier::slotScaleUniformMode() { ModeToolbar::setMode( ModeToolbar::MODE_SCALE_UNIFORM ); updateViews();}

void Modifier::slotActivate()
{
   cerr << "Modifier:" << m_name << endl;
   activate();

};

void Modifier::slotToggleMultiSelect()
{
   ModeToolbar::toggleMultipleSelect( );
   updateViews();
}

void Modifier::slotToggleX()
{
   Vector4 p( 1, 0, 0, 0 );
   ModeToolbar::toggleAxis( p );
   updateViews();
}

void Modifier::slotToggleY()
{
   Vector4 p( 0, 1, 0, 0 );
   ModeToolbar::toggleAxis( p );
   updateViews();
}

void Modifier::slotToggleZ()
{
   Vector4 p( 0, 0, 1, 0 );
   ModeToolbar::toggleAxis( p );
   updateViews();
}





bool Modifier::assertSelected( int n )
{

   int i = m_selection -> size();

   if ( i >= n )
   {
      return true;
   }

   else
   {
      warnNoneSelected( n );
      return false;
   }

}

bool Modifier::assertSelectedExactly( int n )
{

   int i = m_selection -> size();

   if ( i == n )
   {
      return true;
   }

   else
   {
      warnExactSelected( n );
      return false;
   }

}

void Modifier::warnNoneSelected( int n )
{
   QString strMessage = "You must select atleast ";
   QString num;
   num.setNum( n );
   strMessage += num;
   strMessage += " items first";

   QMessageBox::information( 0, "I3D", strMessage );

}

void Modifier::warnExactSelected( int n )
{
   QString strMessage = "You must select exactly ";
   QString num;
   num.setNum( n );
   strMessage += num;
   strMessage += " items first";

   QMessageBox::information( 0, "I3D", strMessage );

}

void Modifier::warnNotSameParent()
{
   QMessageBox::information( 0, "I3D", "All selected entities must have the\n same parent object for this operation." );
}

void Modifier::clearSelection()
{
   ObjectDB::getInstance() -> setNoneSelected();
}

/*
 * Sets up an toggle buttons in the menu
 * that might have been changed by the toolbar.
 */
QPopupMenu * Modifier::getPopupMenu()
{

   //setup the multiselect check
   m_allmenu -> setItemChecked( menu_multiselect, ModeToolbar::isMultipleSelect() );

   Vector4 p = ModeToolbar::getAxis();
   m_axis_menu -> setItemChecked( menu_x_axis, ( p.x == 1 ) );
   m_axis_menu -> setItemChecked( menu_y_axis, ( p.y == 1 ) );
   m_axis_menu -> setItemChecked( menu_z_axis, ( p.z == 1 ) );

   return m_popup;
}

void Modifier::getCenter( Vector4 *p )
{
   p -> assign( 0, 0, 0, 0 );
   SelectionIterator it = m_selection -> begin();
   Vector4 tmp;
   int n = 0;

   while ( it != m_selection -> end() )
   {
      Selectable * s = *it;
      s -> getTransformedPosition( &tmp );
      ( *p ) += tmp;
      ++it;
      ++n;
   }

   if ( n > 0 )
      ( *p ) /= n;

}

void Modifier::updatePlaceLabel()
{
   char buffer[ 256 ];
   Vector4 p;

   p = m_selection_center;

   sprintf( buffer, " X:%2.2f Y:%2.2f Z:%2.2f ", p.x, p.y, p.z );
   I3D::setPlaceLabel( buffer );

}

void Modifier::updateViews()
{
   IControl::updateViews();
}


Selectable * Modifier::getFirst()
{
   return * ( m_selection -> begin() );
}

