/***************************************************************************
                         Poly.cpp  -  description
                            -------------------                   
   begin                : Sun Apr 25 1999                    
   copyright            : (C) 1999 by Jonathan E. Anderson            
   email                : ande1514@tc.umn.edu               
***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/

#include "poly.h"
#include "vertex.h"
#include "material.h"
#include "texture.h"
#include "texturematerial.h"
#include "mesh.h"
#include "line.h"
#include "face.h"
#include "polyskin.h"
#include "../objectdb.h"
#include "tesselator.h"
#include <GL/glut.h>

int Poly::TYPE = Typed::getUID();
bool Poly::filled = true;

Poly::Poly( Entity *_parent = 0 ) : Spline( _parent ), tris()
{

   addType( TYPE );
   drawable = false;

}

Poly::Poly( Vector4 &v ) : Spline()
{
   addType( TYPE );
   createVertex( v );
   drawable = false;
}

Poly::~Poly()
{
}

void Poly::extend( Vector4 &v )
{
   createVertex( v );

   if ( verts.size() > 2 )
   {
      drawable = true;
      triangulate();
   }

}


void Poly::triangulate()
{
   Tesselator t;

   vector<Vector4> vlist;

   for ( int i = 0; i < numVerts(); i++ )
   {
      vlist.push_back( getVertex( i ) ->getPosition() );
   }

   tris = t.tesselate( vlist, POLY_CCW );
}

void Poly::drawPolyTris()
{
   if ( !drawable )
      return ;

   glPushAttrib( GL_COLOR_BUFFER_BIT );

   glBegin( GL_TRIANGLES );

   for ( int i = 0; i < ( int ) tris.size(); i++ )
   {
      getVertex( tris[ i ] ) ->draw( VERTEX_COLOR | VERTEX_NORMAL | VERTEX_COORD );
   }

   glEnd();
   glPopAttrib();

}

int Poly::draw( int d_options = 0 )
{
   if ( numVerts() == 1 )
   {
      Vector4 p = verts[ 0 ] ->getPosition();
      glTranslatef( p.x, p.y, p.z );
      glutSolidSphere( .25, 4, 4 );
      return 0;
   }


   drawPolyTris();



   if ( SelectMode::is( Vertex::TYPE ) )       //need to draw verts too?
      drawVerts();

   return 0;

}

/**Reverse the ordering of the vertices.*/
void Poly::reverse()
{
   Spline::reverse();

}

void Poly::closeSpline()
{
   //can't close it
}

void Poly::breakSpline()
{
   //verts.removeRef(verts.last());
   //can't break it!

}


Poly & Poly::operator=( Poly &rhs )
{
   Spline::operator=( rhs );

   tris = rhs.tris;
   drawable = rhs.drawable;
   return *this;

}

Poly & Poly::operator=( Line &rhs )
{
   Spline::operator=( rhs );
   triangulate();
   return *this;
}

void Poly::copyFrom( Entity *rhs )
{
   *this = * static_cast<Poly*>( rhs );
}



Entity * Poly::clone()
{
   Poly * p = new Poly();

   *p = *this;

   return p;

}


/**Sweeps a line along another spline by creating new Lines, and
  *skinning them.
  */
Entity * Poly::sweep( Spline &s )
{
   vector<Spline *> slist;

   VertexList *vlist = s.getVerts();

   slist.reserve( vlist->size() );
   int n = vlist->size();


   Line *l;


   Vector4 curPos;
   Vector4 transPos;
   Vector4 tarPos;
   Vector4 prevPos;


   //the front and back caps.
   Poly *pf = new Poly();
   Poly *pb = new Poly();


   //align the first line with the vector from the 1st to second
   curPos = ( *vlist ) [ 0 ] ->getPosition();
   tarPos = ( *vlist ) [ 1 ] ->getPosition();
   l = getProfile( curPos, curPos, tarPos );
   *pf = *l;
   transPos = curPos - getPosition();
   pf->move( transPos.x, transPos.y, transPos.z );
   slist.push_back( pf );


   for ( int i = 1; i < ( int ) n - 1; i++ )
   {
      curPos = ( *vlist ) [ i ] ->getPosition();
      tarPos = ( *vlist ) [ i + 1 ] ->getPosition();
      prevPos = ( *vlist ) [ i - 1 ] ->getPosition();
      l = getProfile( prevPos, curPos, tarPos );
      transPos = curPos - getPosition();
      l->move( transPos.x, transPos.y, transPos.z );
      slist.push_back( l );
   }

   //align the last line with the vector from last to second to last
   curPos = ( *vlist ) [ n - 2 ] ->getPosition();

   tarPos = ( *vlist ) [ n - 1 ] ->getPosition();

   l = getProfile( curPos, curPos, tarPos );

   *pb = *l;

   transPos = tarPos - getPosition();  //target is the end vector.

   pb->move( transPos.x, transPos.y, transPos.z );

   slist.push_back( pb );


   PolySkin ps;

   ps.setUseQuads( true );

   Mesh *m = ps.getMesh( slist );

   int segs = ( int ) slist.size() - 1;

   VertexList *sverts = ps.getVertArray();

   //set the caps into the mesh
   for ( int i = 0; i < ( int ) tris.size(); i += 3 )
   {
      //reverse one end...
      m -> createFace( sverts[ 0 ][ tris[ i + 2 ] ] ->getParentIndex(),
                       sverts[ 0 ][ tris[ i + 1 ] ] ->getParentIndex(),
                       sverts[ 0 ][ tris[ i ] ] ->getParentIndex() );
      m -> createFace( sverts[ segs ][ tris[ i ] ] ->getParentIndex(),
                       sverts[ segs ][ tris[ i + 1 ] ] ->getParentIndex(),
                       sverts[ segs ][ tris[ i + 2 ] ] ->getParentIndex() );
   }



   return m;

}

/**Revolve the line around the first segment of the given spline.
  *Do this by duplicating the splines, rotating them, and then skinning them.
  */
Entity * Poly::revolve( Spline &s, int degrees, int segments )
{

   //find the vector corresponding to the first segment in the spline.
   VertexList * vlist = s.getVerts();

   Vector4 axis;
   Vector4 sPos;
   axis = ( *vlist ) [ 1 ] ->getPosition() - ( *vlist ) [ 0 ] ->getPosition();
   axis.normalize();

   sPos = ( *vlist ) [ 0 ] ->getPosition();



   vector<Spline *> slist;

   slist.reserve( segments );


   //duplicate the line x segments
   //if 360 degrees, we can just use lines
   if ( degrees == 360 )
   {
      for ( int i = 0; i < segments; i++ )
      {
         Line *l = new Line();
         *l = *this;
         slist.push_back( l );
         //close the line;
         l->closeSpline();
      }
   }

   else
   {
      Poly *p = new Poly();
      *p = *this;

      for ( int i = 1; i < segments - 1; i++ )
      {
         Line *l = new Line();
         *l = *this;
         slist.push_back( l );
         l->closeSpline();
      }

      p = new Poly();
      *p = *this;
      //p->reverse();
      slist.push_back( p );
   }


   //rotate the line segments;
   float segment_theta = ( float ) degrees / segments;

   for ( int i = 0; i < segments; i++ )
   {
      slist[ i ] ->rotate( segment_theta * i, axis.x, axis.y, axis.z, 0, 0, 0 );
      slist[ i ] ->move( sPos.x, sPos.y, sPos.z );
   }

   //close it for 360 degree revoles...
   if ( degrees == 360 )
      slist.push_back( slist[ 0 ] );


   //skin the
   PolySkin ps;

   ps.setUseQuads( true );

   Mesh *m = ps.getMesh( slist );



   return m;


}


Line * Poly::getProfile( Vector4 &prevPos, Vector4 &curPos, Vector4 &tarPos )
{
   Line * l = new Line();
   *l = *this;

   l->closeSpline();

   l->center();

   Vector4 crossResult;
   Vector4 tangent;


   //if line has less than 2 points, use Y as base normalVector, otherwise
   //use the first three points to get a normal vector.
   Vector4 normalVector( 0, 1, 0, 1 );

   VertexList *vlist = getVerts();


   if ( vlist->size() > 2 )
   {
      Vector4 l, r;
      Vector3 norm;

      l = ( *vlist ) [ 0 ] ->getPosition() - ( *vlist ) [ 1 ] ->getPosition();
      r = ( *vlist ) [ 2 ] ->getPosition() - ( *vlist ) [ 1 ] ->getPosition();

      l.normalize();
      r.normalize();

      norm = l.ToVector3() * r.ToVector3();
      normalVector.assign( norm.x, norm.y, norm.z, 1 );

   }

   if ( prevPos == curPos )
   {
      tangent = curPos - tarPos;
      tangent.normalize();
   }

   else
   {
      Vector4 pv, nv;
      nv = curPos - tarPos;
      pv = prevPos - curPos;
      nv.normalize();
      pv.normalize();
      tangent = nv + pv;
      tangent /= 2;
      tangent.normalize();
   }


   float cosAngle = tangent.dot3d( normalVector );

   if ( cosAngle < 0.9999 )
   {
      //use the cross product to determine the vector to rotate about.
      Vector3 r = tangent.ToVector3() * normalVector.ToVector3();
      crossResult.x = r.x;
      crossResult.y = r.y;
      crossResult.z = r.z;
      crossResult.normalize();

      float turnAngle = -acos( ( float ) cosAngle );
      float turnDeg = turnAngle * 180 / M_PI;
      l->rotate( turnDeg, crossResult.x, crossResult.y, crossResult.z, 0, 0, 0 );
   }


   return l;

}















































