;;; initz.el --- Handles the switching of various startup initialization files

;; Copyright (C) 2001-2002 OHASHI Akira <bg66@koka-in.org>

;; Author: OHASHI Akira <bg66@koka-in.org>
;; Keywords: startup, init

;; This file is part of Initz.

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.


;;; Commentary:
;;

;;; Code:

(require 'install)
(require 'product)
(require 'initz-vars)
(require 'initz-globals)
(eval-when-compile (require 'cl))
(eval-and-compile
  (autoload 'initz-error "initz-error" nil t)
  (autoload 'initz-list "initz-list" nil t)
  (autoload 'initz-list-new-file "initz-list" nil t))

(product-provide 'initz
  (product-define "Initz" nil '(0 0 9)))

(defun initz-version (&optional arg)
  "Return Initz version.
If it is called interactively, version string is appeared on minibuffer.
If ARG is specified, don't display code name."
  (interactive "P")
  (let ((product-info (product-string-1 'initz (not arg))))
    (if (interactive-p)
	(message "%s" product-info)
      product-info)))

(defconst initz-done-message-format
  "Loading %s init files for %s...done")

(defun initz-add-to-load-path (paths)
  "Add PATHS to load-path recursively."
  (let ((paths (if (listp paths) paths (list paths))))
    (mapc
     (function (lambda (path)
		 (when (file-directory-p path)
		   (add-to-list 'load-path path)
		   (initz-add-to-load-path
		    ;; Without `.' and `..'.
		    (directory-files
		     path t "^\\([^.].+\\|\\.[^.].+\\|\\.\\..+\\)$")))))
     paths)))

(defun initz-message (mesg)
  "if `initz-verbose' is non-nil, show MESG."
  (when initz-verbose (message mesg)))

(defconst initz-init-alist
  `((misc . (,initz-null-string ,initz-null-string))
    (flavor . ("flavor"
	       ,(concat initz-separator-string
			initz-flavor)))
    (host . ("host"
	     ,(concat initz-separator-string
		      (system-name))))
    (system . ("system"
	       ,(concat initz-separator-string
			(symbol-name system-type))))))

(defun initz-get-init-value (sym type)
  "Return the TYPE's value of SYM from `initz-init-alist'."
  (let ((list (cdr (assq sym initz-init-alist)))
	(count 0))
    (catch 'found
      (mapc
       (function (lambda (temp)
		   (if (eq type temp)
		       (throw 'found (nth count list))
		     (setq count (incf count)))))
       '(dir prefix))
      nil)))

(defun initz-directory (kind)
  "Return the directory of KIND."
  (let ((dir (cond
	      ((eq kind 'startup) "startup")
	      ((eq kind 'flavor) initz-flavor)
	      (t initz-null-string))))
    (expand-file-name dir initz-directory)))

(defun initz-startup-directory (sym)
  "Return the startup directory of SYM."
  (expand-file-name
   (initz-get-init-value sym 'dir)
   (initz-directory 'startup)))

(defun initz-flavor-directory (sym)
  "Return the flavor directory of SYM."
  (expand-file-name
   (initz-get-init-value sym 'dir)
   (initz-directory 'flavor)))

(defun initz-get-kind (file)
  "Return the kind of FILE."
  (catch 'found
    (mapc
     (function (lambda (kind)
		 (when (string-match (initz-directory kind) file)
		   (throw 'found kind))))
     '(startup flavor))
    nil))

(defun initz-get-dir (file)
  "Return dir of the FILE."
  (let ((file (file-name-directory file))
	(directory (initz-directory (initz-get-kind file))))
    (when (string-match "/$" file)
      (setq file (substring file 0 (1- (length file)))))
    (catch 'found
      (if (string= file directory)
	  (throw 'found 'misc)
	(when (string-match (concat directory "\\(.+\\)") file)
	  (let ((dir (substring (match-string 1 file) 1)))
	    (mapc
	     (function (lambda (alist)
			 (when (string= (nth 0 (cdr alist)) dir)
			   (throw 'found (car alist)))))
	     initz-init-alist))))
      nil)))

(defun initz-get-correspondence-file (init-file)
  "Return correspondence file of the INIT-FILE."
  (let* ((file (file-name-nondirectory init-file))
	 (kind (if (eq (initz-get-kind init-file) 'startup)
		   'flavor
		 'startup))
	 (directory (expand-file-name
		     (initz-get-init-value (initz-get-dir init-file) 'dir)
		     (initz-directory kind))))
    (expand-file-name (if (eq kind 'startup)
			  (substring file 0 (1- (length file)))
			(concat file "c"))
		      directory)))

(defun initz-get-base-name (init-file)
  "Return base name of the INIT-FILE."
  (file-name-sans-extension
   (file-name-nondirectory init-file)))

(defun initz-get-module-name (init-file)
  "Return module name of the INIT-FILE."
  (let ((base-name (initz-get-base-name init-file)))
    (cond
     ((string= base-name initz-prefix) initz-prefix)
     ((string-match (concat
		     (regexp-quote (concat initz-prefix
					   initz-separator-string))
		     "\\(" initz-module-regexp "\\)")
		    base-name)
      (match-string 1 base-name))
     (t initz-null-string))))

(defun initz-get-files (kind dir)
  "Return files of the directory made by KIND and DIR."
  (let ((directory (expand-file-name
		    (initz-get-init-value dir 'dir)
		    (initz-directory kind)))
	(prefix (regexp-quote
		 (concat initz-prefix
			 (initz-get-init-value dir 'prefix))))
	(ext (if (eq kind 'startup) "\\.el$" "\\.elc$")))
    (directory-files
     directory t (concat "^\\(" prefix "\\|"
			 prefix initz-module-regexp "\\)" ext))))

(defun initz-features ()
  "Return the Initz features."
  (delq nil
	(mapcar
	 (function (lambda (feature)
		     ;; Don't `initz-get-module-name'?
		     (let ((init-feature (initz-get-module-name
					  (symbol-name feature))))
		       (unless (string= init-feature initz-null-string)
			 (intern init-feature)))))
	 features)))

(defun initz-make-directory (sym)
  "Make SYM's directory."
  (mapc
   (function (lambda (kind)
	       (let ((directory (expand-file-name
				 (initz-get-init-value sym 'dir)
				 (initz-directory kind))))
		 (unless (file-directory-p directory)
		   (make-directory directory t)))))
   '(startup flavor)))

(defun initz-make-directories ()
  "Make initz directories."
  (interactive)
  (mapc
   (function (lambda (alist)
	       (let ((sym (car alist)))
		 (initz-make-directory sym))))
   initz-init-alist))

(defun initz-delete-file (flavor-file)
  "Delete the FLAVOR-FILE when startup-file was deleted."
  (let ((startup-file (initz-get-correspondence-file flavor-file)))
    (unless (file-exists-p startup-file)
      (delete-file flavor-file))))

(defun initz-delete-files (sym)
  "Delete files in the SYM's directory when startup-file was deleted."
  (let ((flavor-files (initz-get-files 'flavor sym)))
    (mapc
     (function (lambda (flavor-file)
		 (initz-delete-file flavor-file)))
     flavor-files)))

(defun initz-delete ()
  "Delete the initz startup files."
  (interactive)
  (initz-make-directories)
  (mapc
   (function (lambda (alist)
	       (let ((sym (car alist)))
		 (initz-delete-files sym))))
   initz-init-alist))

(defun initz-compile-file (startup-file)
  "Compile the STARTUP-FILE."
  (unless (member (initz-get-module-name startup-file)
		  initz-ignore-list)
    (let ((flavor-file (initz-get-correspondence-file startup-file)))
      (when (file-newer-than-file-p startup-file flavor-file)
	(condition-case nil
	    (unless (save-window-excursion
		      (byte-compile-file startup-file))
	      (error nil))
	  (error (add-to-list 'initz-error-compile-files
			      startup-file)
		 nil))))))

(defun initz-compile-files (sym)
  "Compile files in the SYM's directory."
  (let ((startup-files (initz-get-files 'startup sym))
	compiled-files)
    (mapc
     (function (lambda (startup-file)
		 (initz-compile-file startup-file)))
     startup-files)
    (setq compiled-files (directory-files
			  (initz-startup-directory sym) nil "\\.elc$"))
    (install-files compiled-files (initz-startup-directory sym)
		   (initz-flavor-directory sym) t t)))

(defun initz-compile ()
  "Compile the initz startup files."
  (interactive)
  (initz-delete)
  (setq initz-error-compile-files nil)
  (mapc
   (function (lambda (alist)
	       (let ((sym (car alist)))
		 (initz-compile-files sym))))
   initz-init-alist)
  (and initz-error-compile-files (eq initz-verbose 'errors)
       (initz-error)))

(defun initz-load-file (flavor-file &optional unload)
  "Load the FLAVOR-FILE."
  (let* ((module (initz-get-module-name flavor-file))
	 (mesg (format (if unload
			   initz-unload-module-message-format
			 initz-load-module-message-format)
		       module)))
    (if (or (member module initz-ignore-list)
	    (and initz-load-list
		 (not (member module initz-load-list))))
	(initz-message (concat mesg "ignored"))
      (unless (and initz-interactively
		   (not (y-or-n-p
			 (format initz-load-module-ask-message-format
				 module))))
	(initz-message mesg)
	(condition-case nil
	    (let*((base-name (initz-get-base-name flavor-file))
		  (feature (intern base-name)))
	      (if unload
		  (unload-feature feature t)
		(when (memq feature features)
		  (unload-feature feature t))
		(require feature))
	      (initz-message (concat mesg "done")))
	  (error (add-to-list 'initz-error-load-files
			      (initz-get-correspondence-file flavor-file))
		 (initz-message (concat mesg "failed"))
		 nil))))))

(defun initz-load-files (sym)
  "Load files in the SYM's directory."
  (let ((flavor-files (initz-get-files 'flavor sym)))
    (mapc
     (function (lambda (flavor-file)
		 (initz-load-file flavor-file)))
     flavor-files)))

(defun initz-load ()
  "Load the initz startup files."
  (interactive)
  (initz-compile)
  (setq initz-error-load-files nil)
  (initz-add-to-load-path (initz-directory 'flavor))
  (mapc
   (function (lambda (alist)
	       (let ((sym (car alist)))
		 (initz-load-files sym))))
   initz-init-alist)
  (and initz-error-load-files (eq initz-verbose 'errors)
       (initz-error)))

(defun initz-done ()
  "Initz done."
  (initz-message (format initz-done-message-format
			 (initz-version) initz-flavor)))

;;;###autoload
(defun initz-startup ()
  "Initz startup."
  (interactive)
  (unless noninteractive
    (initz-load)
    (initz-done)))

(provide 'initz)

;;; initz.el ends here
