/*
 * Hydrogen
 * Copyright(c) 2002-2004 by Alex >Comix< Cominu [comix@users.sourceforge.net]
 *
 * http://hydrogen.sourceforge.net
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY, without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $Id: Song.cpp,v 1.30 2004/09/06 09:40:37 comix Exp $
 *
 */

#include "Song.h"
#include "config.h"

#include <assert.h>

SequenceList::SequenceList()
 : Object( "SequenceList" )
{
//	infoLog("Init");
}



SequenceList::~SequenceList() {
//	infoLog("destroy");
	for (unsigned int i = 0; i < list.size(); i++) {
		if( list[i] != NULL ){
			delete list[i];
			list[i] = NULL;
		}
	}
}



void SequenceList::add(Sequence* newSequence) {
	list.push_back(newSequence);
}



Sequence* SequenceList::get(int pos) {
	assert( ( pos >= 0 ) && ( pos < (int)list.size() ) );
/*	if ( pos < 0 ) {
		errorLog( "[get] pos < 0 - pos = " + toString( pos ) );
		return NULL;
	}
	if (pos > list.size() ) {
		errorLog( "[get] pos > list.size - pos = " + toString( pos ) + " list.size=" + toString(list.size()) );
		return NULL;
	}*/

	return list[pos];
}



unsigned int SequenceList::getSize() {
	return list.size();
}



SequenceList* SequenceList::copy() {
	SequenceList *newSequenceList = new SequenceList();

	for (uint nSeq = 0; nSeq < this->getSize(); nSeq++) {
		Sequence *oldSequence = this->get(nSeq);
		Sequence *newSequence = oldSequence->copy();
		newSequenceList->add(newSequence);
	}

	return newSequenceList;
}




//:::::::::::::::::::::::




PatternList::PatternList()
 : Object( "PatternList" )
{
//	infoLog("Init");
}



PatternList::~PatternList() {
//	infoLog("destroy");

	// find single patterns. (skip duplicates)
	vector<Pattern*> temp;
	for (unsigned int i = 0; i < list.size(); i++) {
		Pattern *pat = list[i];

		// pat exists in temp?
		bool exists = false;
		for(unsigned int j = 0; j < temp.size(); j++) {
			if (pat == temp[j]) {
				exists = true;
				break;
			}
		}
		if (!exists) {
			temp.push_back(pat);
		}
	}

	// delete patterns
	for (unsigned int i = 0; i < temp.size(); i++) {
		Pattern *pat = temp[i];
		if (pat != NULL) {
			delete pat;
			pat = NULL;
		}
	}
}



void PatternList::add(Pattern* newPattern) {
	list.push_back(newPattern);
}



Pattern* PatternList::get(int nPos) {
	if (nPos > (int)list.size()) {
		errorLog( "[get] Pattern index out of bounds. nPos > list.size() - " + toString( nPos ) + " > " + toString( list.size() ) );
		return NULL;
	}
//	assert( nPos < (int)list.size() );
	return list[ nPos ];
}



unsigned int PatternList::getSize() {
	return list.size();
}



void PatternList::clear() {
	list.clear();
}



/// Replace an existent pattern with another one
void PatternList::replace( Pattern* newPattern, unsigned int pos ) {
	list.insert( list.begin() + pos, newPattern );	// insert the new pattern
	// remove the old pattern
	list.erase( list.begin() + pos + 1 );
}



/// Remove a pattern from the list (every instance in the list), the pattern is not deleted!!!
void PatternList::del(Pattern *pattern) {
	//infoLog("[del] Delete pattern " + pattern->getName());
	// create the new pattern list
	vector<Pattern*> newList;
	for (uint i = 0; i < list.size(); i++) {
		Pattern *pat = list[i];
		if (pat == pattern) {
			continue;
		}
		newList.push_back(pat);
	}

	// copy the new list element over the old vector reference
	list.clear();
	for (uint i = 0; i < newList.size(); i++) {
		list.push_back(newList[i]);
	}
}



/// Remove one pattern from the list, the pattern is not deleted!!!
void PatternList::del(uint index) {
	// create the new pattern list
	vector<Pattern*> newList;
	for (uint i = 0; i < list.size(); i++) {
		if (i == index) {
			continue;
		}
		Pattern *pat = list[i];
		newList.push_back(pat);
	}

	// copy the new list element over the old vector reference
	list.clear();
	for (uint i = 0; i < newList.size(); i++) {
		list.push_back(newList[i]);
	}
}



//::::::::::::::::::



InstrumentList::InstrumentList()
 : Object( "InstrumentList" )
{
//	infoLog("INIT");
}



InstrumentList::~InstrumentList() {
//	infoLog("DESTROY");
	for (unsigned int i = 0; i < list.size(); i++) {
		delete list[i];
	}
}



void InstrumentList::add(Instrument* newInstrument) {
	list.push_back(newInstrument);
	posmap[newInstrument] = list.size() - 1;
}



Instrument* InstrumentList::get(unsigned int pos) {
	if ( pos > list.size() ) {
		errorLog( "[get] pos > list.size(). pos = " + toString(pos) );
		return NULL;
	}
	else if ( pos < 0 ) {
		errorLog( "[get] pos < 0. pos = " + toString(pos) );
		return NULL;
	}
	return list[pos];
}



/// Returns index of instrument in list, if instrument not found, returns -1
int InstrumentList::getPos(Instrument * inst) {
	if (posmap.find(inst) == posmap.end())
		return -1;
	return posmap[inst];
}



unsigned int InstrumentList::getSize() {
	return list.size();
}



//::::::::::::::::::::::::::



Song::Song(string sId, string sName, string sAuthor, float fBpm, float fVolume)
 : Object( "Song     " )
 , m_sId ( sId )
 , m_fBpm( fBpm )
 , m_nResolution( 48 )
 , m_fVolume( fVolume )
 , m_fMetronomeVolume( 0.5 )
 , m_sName( sName )
 , m_sAuthor( sAuthor )
 , m_sNotes( "Song info" )	///\todo: attenzione..questo non verra' tradotto
 , m_pPatternList( NULL )
 , m_pPatternSequence( NULL )
 , m_pInstrumentList( NULL )
 , m_sFilename( "" )
 , m_bIsModified( false )
 , m_bIsLoopEnabled( false )
 , m_bIsHumanizeTimeEnabled( false )
 , m_fHumanizeTimeValue( 0.0 )
 , m_bIsHumanizeVelocityEnabled( false )
 , m_fHumanizeVelocityValue( 0.0 )
 , m_bIsSwingEnabled( false )
 , m_fSwingFactor( 0.0 )
 , m_songMode( PATTERN_MODE )
{
	infoLog( "init \"" + m_sName + "\"" );

	for (uint nFX = 0; nFX < MAX_FX; nFX++) {
		setLadspaFX( nFX, NULL );
	}
}



Song::~Song()
{
	// delete all patterns
	delete m_pPatternList;

	if (m_pPatternSequence) {
		for (uint i = 0; i < m_pPatternSequence->size(); i++) {
			PatternList *pPatternList = (*m_pPatternSequence)[i];
			pPatternList->clear();	// pulisco tutto, i pattern non vanno distrutti qua
			delete pPatternList;
		}
		delete m_pPatternSequence;
	}

	delete m_pInstrumentList;

	for (uint nFX = 0; nFX < MAX_FX; nFX++) {
		delete m_pLadspaFX[nFX];
	}

	infoLog( "destroy \"" + m_sName + "\"" );
}



///Load a song from file
Song* Song::load(string filename) {
	Song *song = NULL;
	LocalFileMng mng;
	song = mng.loadSong(filename);

	return song;
}




/// Save a song to file
void Song::save(string filename) {
	LocalFileMng mng;
	mng.saveSong(this, filename);
}



/// Return an empty song
Song* Song::getEmptySong() {
	string dataDir = DATA_DIR;
	string filename = dataDir + "/DefaultSong.h2song";
	Song *song = Song::load( filename );

	return song;
}





void Song::setSwingFactor( float factor ) {
	if (factor < 0.0) {
		factor = 0.0;
	}
	else if (factor > 1.0) {
		factor = 1.0;
	}

	m_fSwingFactor = factor;
}



//:::::::::::::::::::::::::



Sequence::Sequence() : Object( "Sequence" )
{
//	logger.info(this, "INIT " + name);
//	setName( name );

	for (int i = 0; i < MAX_NOTES; i++) {
		m_noteList[i] = NULL;
	}
}



Sequence::~Sequence(){
	// delete all notes
	for( unsigned int i = 0; i < MAX_NOTES; i++ ){
		delete m_noteList[i];
		m_noteList[i] = NULL;
	}

//	logger.info(this, "DESTROY " + name);

}




Sequence* Sequence::copy() {
	Sequence *newSequence = new Sequence();

	for (uint nNote = 0; nNote < MAX_NOTES; nNote++) {
		Note *oldNote = m_noteList[nNote];
		if (oldNote == NULL) {
			newSequence->m_noteList[nNote] = NULL;
		}
		else {
			Note *newNote = oldNote->copy();
			newSequence->m_noteList[nNote] = newNote;
		}
	}

	return newSequence;
}



//:::::::::::::::::::::::::::



Instrument::Instrument( string sId, string sName, string sAuthor, float fVolume, bool bMuted, float fPan_L, float fPan_R, string sDrumkitName )
 : Object( "Instrument" )
 , m_sId( sId )
 , m_fVolume( fVolume )
 , m_sName( sName )
 , m_sAuthor( sAuthor )
 , m_bMuted( bMuted )
 , m_sDrumkitName( sDrumkitName )
 , m_bActive( true )
{
//	infoLog( "INIT @" + toString( (int)this ) );

	setPan_L( fPan_L );
	setPan_R( fPan_R );
	setPeak_L( 0.0 );
	setPeak_R( 0.0 );
	for (uint nFX = 0; nFX < MAX_FX; nFX++) {
		setFXLevel( nFX, 0.0 );
	}

	for (uint nLayer = 0; nLayer < MAX_LAYERS; nLayer++) {
		m_layers[ nLayer ] = NULL;
	}
}



Instrument::~Instrument(){
//	infoLog( "DESTROY @" + toString( (int)this ) );
	for (uint nLayer = 0; nLayer < MAX_LAYERS; nLayer++) {
		delete m_layers[ nLayer ];
	}
}



void Instrument::setVolume(float volume) {
	if (volume > 1.0) {
		volume = 1.0;
	}
	else if (volume < 0.0) {
		volume = 0.0;
	}
	m_fVolume = volume;
}



void Instrument::setPan_L( float val ) {
	if (val < 0.0) {
		val = 0.0;
		errorLog("setPan_L() val < 0.0");
	}
	else if (val > 1.0) {
		errorLog("setPan_L() val > 1.0");
		val = 1.0;
	}
	m_fPan_L = val;
}



void Instrument::setPan_R( float val ) {
	if (val < 0.0) {
		val = 0.0;
		errorLog("setPan_R() val < 0.0");
	}
	else if (val > 1.0) {
		val = 1.0;
		errorLog("setPan_R() val > 1.0");
	}
	m_fPan_R = val;
}



void Instrument::setPeak_L( float fPeak )
{
	if (fPeak < 0.0f) {
		char tmp[200];
		sprintf( tmp, "[setPeak_L] Error: peak < 0 (peak = %f)", fPeak );
		errorLog( tmp );

		fPeak = 0.0f;
	}
	else if( fPeak > 1.0f) {
		char tmp[200];
		sprintf( tmp, "[setPeak_L] Error: peak > 1 (peak = %f)", fPeak );
		errorLog( tmp );
		fPeak = 1.0f;
	}

	m_fPeak_L = fPeak;
}



void Instrument::setPeak_R( float fPeak )
{
	if (fPeak < 0.0f) {
		char tmp[200];
		sprintf( tmp, "[setPeak_R] Error: peak < 0 (peak = %f)", fPeak );
		errorLog( tmp );

		fPeak = 0.0f;
	}
	else if( fPeak > 1.0f) {
		char tmp[200];
		sprintf( tmp, "[setPeak_R] Error: peak > 1 (peak = %f)", fPeak );
		errorLog( tmp );
		fPeak = 1.0f;
	}

	m_fPeak_R = fPeak;
}



InstrumentLayer* Instrument::getLayer( int nLayer )
{
	if (nLayer < 0 ) {
		errorLog( "[getLayer] nLayer < 0 (nLayer=" + toString(nLayer) + ")" );
		return NULL;
	}
	if (nLayer > MAX_LAYERS ) {
		errorLog( "[getLayer] nLayer > MAX_LAYERS (nLayer=" + toString(nLayer) + ")" );
		return NULL;
	}

	return m_layers[ nLayer ];
}



void Instrument::setLayer( InstrumentLayer* pLayer, uint nLayer )
{
	if (nLayer < MAX_LAYERS) {
		m_layers[ nLayer ] = pLayer;
	}
	else {
		errorLog( "[setLayer] nLayer > MAX_LAYER" );
	}
}




//::::::::::::::::::::::::



Note::Note(	unsigned nPosition,
			float fVelocity,
			float fPan_L,
			float fPan_R,
			int nLength,
			float fPitch
		)
 : Object( "Note" )
 , m_fSamplePosition( 0.0 )
 , m_nPosition( nPosition )
 , m_fVelocity( fVelocity )
 , m_fPan_L( fPan_L )
 , m_fPan_R( fPan_R )
 , m_nLength( nLength )
 , m_fPitch( fPitch )
 , m_pInstrument( NULL )
 , m_nHumanizeDelay( 0 )
{
}



Note::~Note(){
	//infoLog("DESTROY");
}



void Note::setInstrument(Instrument* pInstrument) {
	m_pInstrument = pInstrument;
}



void Note::setVelocity( float val ) {
	if (val < 0.0) {
		char tmp[100];
		sprintf(tmp, "setVelocity(): %f < 0.0", val);
		errorLog(tmp);
		val = 0.0;
	}
	else if (val > 1.0) {
		char tmp[100];
		sprintf(tmp, "setVelocity(): %f > 1.0", val);
		errorLog(tmp);
		val = 1.0;
	}

	m_fVelocity = val;
}



void Note::setPan_L( float val ) {
	if (val < 0.0) {
		char tmp[100];
		sprintf(tmp, "setPan_L(): %f < 0.0", val);
		errorLog(tmp);
		val = 0.0;
	}
	else if (val > 1.0) {
		char tmp[100];
		sprintf(tmp, "setPan_L(): %f > 1.0", val);
		errorLog(tmp);
		val = 1.0;
	}

	m_fPan_L = val;
}



void Note::setPan_R( float val ) {
	if (val < 0.0) {
		char tmp[100];
		sprintf(tmp, "setPan_R(): %f < 0.0", val);
		errorLog(tmp);
		val = 0.0;
	}
	else if (val > 1.0) {
		char tmp[100];
		sprintf(tmp, "setPan_L(): %f > 1.0", val);
		errorLog(tmp);
		val = 1.0;
	}

	m_fPan_R = val;
}



Instrument* Note::getInstrument() {
	return m_pInstrument;
}



Note* Note::copy() {
	Note* newNote = new Note(
		this->getPosition(),
		this->getVelocity(),
		this->getPan_L(),
		this->getPan_R(),
		this->getLength(),
		this->getPitch()
	);
	newNote->setInstrument(this->getInstrument());

	return newNote;
}



//:::::::::::::::::::::



Pattern::Pattern( string name, uint nSize ) : Object( "Pattern" )
{
//	infoLog("init");
	setName( name );
	setSize( nSize );
}



Pattern::~Pattern() {
//	infoLog("destroy");

	// delete all Sequences
	delete m_pSequenceList;
}



void Pattern::setName(string sName) {
	m_sName = sName;
}



string Pattern::getName() {
	return m_sName;
}



/// Returns an empty Pattern
Pattern* Pattern::getEmptyPattern()
{
	SequenceList *sequenceList = new SequenceList();
	for (uint i = 0; i < MAX_INSTRUMENTS; i++) {
		Sequence *trk0 = new Sequence();
		sequenceList->add(trk0);
	}

	Pattern *pat = new Pattern("Empty pattern");
	pat->setSequenceList(sequenceList);

	return pat;
}



Pattern* Pattern::copy()
{
	Pattern *newPat = new Pattern(this->getName());
	newPat->setSize(this->getSize());
	SequenceList *newSeqList = this->getSequenceList()->copy();
	newPat->setSequenceList(newSeqList);

	return newPat;
}



//::::::::::::::::::::



DrumkitInfo::DrumkitInfo()
 : Object( "DrumkitInfo" )
 , instrumentList( NULL )
{
//	infoLog( "INIT @" + toString( (int)this ) );
}



DrumkitInfo::~DrumkitInfo()
{
//	infoLog( "DESTROY @" + toString( (int)this ) );
	delete instrumentList;
}



void DrumkitInfo::dump()
{
	infoLog( "Drumkit dump" );
	infoLog( "\t|- Name = " + name );
	infoLog( "\t|- Author = " + author );
	infoLog( "\t|- Info = " + info );

	infoLog( "\t|- Instrument list" );
	for ( unsigned nInstrument = 0; nInstrument < instrumentList->getSize(); nInstrument++) {
		Instrument *pInstr = instrumentList->get( nInstrument );
		infoLog( "\t\t|- (" + toString( nInstrument ) + " of " + toString( instrumentList->getSize() ) + ") Name = " + pInstr->getName() );
		for ( unsigned nLayer = 0; nLayer < MAX_LAYERS; nLayer++ ) {
			InstrumentLayer *pLayer = pInstr->getLayer( nLayer );
			if ( pLayer ) {
				Sample *pSample = pLayer->getSample();
				if ( pSample ) {
					infoLog( "\t\t   |- " + pSample->getFilename() );
				}
				else {
					infoLog( "\t\t   |- NULL sample" );
				}
			}
			else {
				infoLog( "\t\t   |- NULL Layer" );
			}

		}
//		cout << "\t\t" << i << " - " << instr->getSample()->getFilename() << endl;
	}
}



//:::::::::::::::::::::



InstrumentLayer::InstrumentLayer( Sample *pSample )
 : Object( "InstrumentLayer" )
 , m_pSample( pSample )
 , m_fStartVelocity( 0.0 )
 , m_fEndVelocity( 1.0 )
 , m_fPitch( 0.0 )
 , m_fGain( 1.0 )
{
	//infoLog( "INIT" );
}



InstrumentLayer::~InstrumentLayer()
{
	delete m_pSample;
	//infoLog( "DESTROY" );
}



void InstrumentLayer::setSample( Sample *pSample )
{
//	infoLog( "[setSample]" );
	m_pSample = pSample;
}



void InstrumentLayer::setStartVelocity( float fVel )
{
//	infoLog( "[setStartVelocity] " + toString( fVel ) );

	if (fVel > 1.0) {
		fVel = 1.0;
	}
	else if (fVel < 0.0) {
		fVel = 0.0;
	}

	m_fStartVelocity = fVel;
}



void InstrumentLayer::setEndVelocity( float fVel ) {
//	infoLog( "[setEndVelocity] " + toString( fVel ) );

	if (fVel > 1.0) {
		fVel = 1.0;
	}
	else if (fVel < 0.0) {
		fVel = 0.0;
	}

	m_fEndVelocity = fVel;
}



void InstrumentLayer::setGain( float fGain )
{
	//infoLog( "[setGain] " + toString( fGain ) );
//	if (fGain > 1.0) {
//		fGain = 1.0;
//	}
//	else if (fGain < 0.0) {
	if (fGain < 0.0) {
		fGain = 0.0;
	}
	m_fGain = fGain;
}



void InstrumentLayer::setPitch( float fPitch )
{
//	infoLog( "[setPitch] " + toString( fPitch ) );
	m_fPitch = fPitch;
}


