/*
 * Hydrogen
 * Copyright(c) 2002-2004 by Alex >Comix< Cominu [comix@users.sourceforge.net]
 *
 * http://hydrogen.sourceforge.net
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY, without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $Id: PatternEditorPanel.cpp,v 1.68 2004/09/06 09:41:49 comix Exp $
 *
 */


#include "PatternEditorPanel.h"
#include "MainForm.h"
#include "Button.h"
#include "Fader.h"

#include "lib/Preferences.h"
#include "lib/Hydrogen.h"

#include <cmath>

#include "qlabel.h"
#include "qcombobox.h"


PatternEditorPanel::PatternEditorPanel( QWidget *parent )
  : QWidget(parent)
  , Object("PatEditPanel")
  , m_pPattern( NULL )
{
	//infoLog("INIT");
	m_nPanelWidth = 706;
	m_nPanelHeight = 352;

	int nMaxHeight = 544;
	nMaxHeight = 850;

	setMinimumSize( 200, 200 );
	setMaximumSize( m_nPanelWidth, nMaxHeight );

	resize( m_nPanelWidth, m_nPanelHeight);

	setCaption( trUtf8( "Pattern Editor" ) );
	setIcon( QPixmap( QString(IMG_PATH) + QString( "/img/icon32.png") ) );

	bool bUseBackground = false;

	// Load background image
	string background_path = string(IMG_PATH) + string("/img/patternEditor/patternEditor_background.png");
	bool ok = m_backgroundPixmap.load(background_path.c_str());
	if ( ok == false ) {
		errorLog( "[PatternEditorPanel] Error loading pixmap" );
	}



	// RULER____________________________________

	// Ruler ScrollView
	m_pRulerScrollView = new QScrollView( this );
	m_pRulerScrollView->setFrameShape( QFrame::NoFrame );
	m_pRulerScrollView->setVScrollBarMode( QScrollView::AlwaysOff );
	m_pRulerScrollView->setHScrollBarMode( QScrollView::AlwaysOff );

	// Ruler frame
	QFrame* m_pRulerFrame = new QFrame( m_pRulerScrollView->viewport() );

	// Ruler
	m_pPatternEditorRuler = new PatternEditorRuler( m_pRulerFrame, this );
	m_pPatternEditorRuler->move( 0, 0 );

	m_pRulerFrame->resize( m_pPatternEditorRuler->width(), m_pPatternEditorRuler->height() );
	m_pRulerScrollView->addChild( m_pRulerFrame );
	m_pRulerScrollView->move( 105, 30 );

	//_________________________________________





	// EDITOR _____________________________________

	// Editor scrollview
	m_pEditorScrollView = new QScrollView( this );
	m_pEditorScrollView->setFrameShape( QFrame::NoFrame );
	m_pEditorScrollView->setVScrollBarMode( QScrollView::AlwaysOff );
	m_pEditorScrollView->setHScrollBarMode( QScrollView::AlwaysOff );
	m_pEditorScrollView->move( 105, 45 );

	// Editor frame
	m_pEditorFrame = new QFrame( m_pEditorScrollView->viewport() );

	// Editor
	m_pPatternEditor = new PatternEditor( m_pEditorFrame, this );
	m_pPatternEditor->move( 0, 0 );

	m_pEditorFrame->resize( m_pPatternEditor->width(), m_pPatternEditor->height() );
	m_pEditorScrollView->addChild( m_pEditorFrame );

	QScrollBar *pEditorVScroll = m_pEditorScrollView->verticalScrollBar();
	connect( pEditorVScroll , SIGNAL(valueChanged(int)), this, SLOT( contentsMoving(int) ) );

	//_______________________________________________





	// INSTRUMENT LIST________________________________________

	// Instrument list scrollview
	m_pInstrListScrollView = new QScrollView( this );
	m_pInstrListScrollView->setFrameShape( QFrame::NoFrame );
	m_pInstrListScrollView->setVScrollBarMode( QScrollView::AlwaysOff );
	m_pInstrListScrollView->setHScrollBarMode( QScrollView::AlwaysOff );
	m_pInstrListScrollView->move( 5, 45 );

	// Instrument list frame
	QFrame* m_pInstrListFrame = new QFrame( m_pInstrListScrollView->viewport() );

	// Instrument list
	m_pInstrumentList = new PatternEditorInstrumentList( m_pInstrListFrame, this );
	m_pInstrumentList->move( 0, 0 );

	m_pInstrListFrame->resize( m_pInstrumentList->width(), m_pPatternEditor->height() );
	m_pInstrListScrollView->addChild( m_pInstrListFrame );

	QScrollBar *pInstrListVScroll = m_pInstrListScrollView->verticalScrollBar();
	connect( pInstrListVScroll , SIGNAL(valueChanged(int)), this, SLOT( contentsMoving(int) ) );

	//____________________________________________________________





	// NOTE PROPERTIES RULER______________________________________________

	m_pNotePropertiesScrollView = new QScrollView( this );
	m_pNotePropertiesScrollView->setFrameShape( QFrame::NoFrame );
	m_pNotePropertiesScrollView->setVScrollBarMode( QScrollView::AlwaysOff );
	m_pNotePropertiesScrollView->setHScrollBarMode( QScrollView::AlwaysOff );
	m_pNotePropertiesScrollView->move( 5, 600 );

	QFrame* m_pNotePropertiesFrame = new QFrame( m_pNotePropertiesScrollView->viewport() );

	m_pNotePropertiesRuler = new NotePropertiesRuler( m_pNotePropertiesFrame, this );
	m_pNotePropertiesRuler->move(0,0);

	m_pNotePropertiesFrame->resize( m_pNotePropertiesRuler->width(), m_pNotePropertiesRuler->height() );

	m_pNotePropertiesScrollView->addChild( m_pNotePropertiesFrame );

	//_____________________________________________________________






	// external horizontal scrollbar
	m_pPatternEditorHScrollBar = new QScrollBar( this, "m_pPatternEditorHScrollBar" );
	m_pPatternEditorHScrollBar->setOrientation( Qt::Horizontal );
	connect( m_pPatternEditorHScrollBar, SIGNAL(valueChanged(int)), this, SLOT( syncToExternalHorizontalScrollbar(int) ) );

	// external vertical scrollbar
	m_pPatternEditorVScrollBar = new QScrollBar( this, "m_pPatternEditorVScrollBar" );
	m_pPatternEditorVScrollBar->setOrientation( Qt::Vertical );
	connect( m_pPatternEditorVScrollBar, SIGNAL(valueChanged(int)), this, SLOT( syncToExternalHorizontalScrollbar(int) ) );



	QLabel *gridResolutionLbl = new QLabel(this);
	gridResolutionLbl->setText(trUtf8( "Grid resolution" ));
	gridResolutionLbl->move( 10, 5 );
	gridResolutionLbl->resize( 100, 20 );

	string hearBtn_on_path = string(IMG_PATH).append("/img/patternEditor/hearBtn_on.png");
	string hearBtn_off_path = string(IMG_PATH).append("/img/patternEditor/hearBtn_off.png");
	string hearBtn_over_path = string(IMG_PATH).append("/img/patternEditor/hearBtn_over.png");
	hearNotesBtn = new ToggleButton(this, QSize(20, 20), hearBtn_on_path, hearBtn_off_path, hearBtn_over_path);
	hearNotesBtn->move(200, 5);
	QToolTip::add( hearNotesBtn, trUtf8( "Hear new notes" ) );
	connect( hearNotesBtn, SIGNAL(clicked(Button*)), this, SLOT( hearNotesBtnClick(Button*)));


	// shows the pattern name
	patternNameLbl = new QLabel(this);
	patternNameLbl->setText(trUtf8( "Pattern name") );
	patternNameLbl->move( 284, 5 );
	patternNameLbl->resize( 300, 20 );

	// Grid resolution combo
	gridResolutionCombo = new QComboBox( FALSE, this, "gridResolutionComboBox" );
	gridResolutionCombo->insertItem( trUtf8( "4" ) );
	gridResolutionCombo->insertItem( trUtf8( "8" ) );
	gridResolutionCombo->insertItem( trUtf8( "16" ) );
	gridResolutionCombo->insertItem( trUtf8( "32" ) );
	gridResolutionCombo->insertItem( trUtf8( "64" ) );
	gridResolutionCombo->insertItem( trUtf8( "4T" ) );	// triplets..
	gridResolutionCombo->insertItem( trUtf8( "8T" ) );
	gridResolutionCombo->insertItem( trUtf8( "16T" ) );
	gridResolutionCombo->insertItem( trUtf8( "32T" ) );

	gridResolutionCombo->move( 130, 5 );
	gridResolutionCombo->resize( 60, 20 );
	QToolTip::add( gridResolutionCombo, trUtf8( "Grid resolution" ) );
	connect( gridResolutionCombo, SIGNAL( activated(const QString&) ), this, SLOT( gridResolutionChanged() ) );

	Preferences *pref = ( Preferences::getInstance() );

	// restore grid resolution
	if ( pref->isPatternEditorUsingTriplets() == false ) {
		switch ( pref->getPatternEditorGridResolution() ) {
			case 4:
				gridResolutionCombo->setCurrentItem( 0 );
				break;

			case 8:
				gridResolutionCombo->setCurrentItem( 1 );
				break;

			case 16:
				gridResolutionCombo->setCurrentItem( 2 );
				break;

			case 32:
				gridResolutionCombo->setCurrentItem( 3 );
				break;

			case 64:
				gridResolutionCombo->setCurrentItem( 4 );
				break;

			default:
				gridResolutionCombo->setCurrentItem( 1 );
		}
	}
	else {
		switch ( pref->getPatternEditorGridResolution() ) {
			case 4:
				gridResolutionCombo->setCurrentItem( 5 );
				break;

			case 8:
				gridResolutionCombo->setCurrentItem( 6 );
				break;

			case 16:
				gridResolutionCombo->setCurrentItem( 7 );
				break;

			case 32:
				gridResolutionCombo->setCurrentItem( 8 );
				break;

			default:
				gridResolutionCombo->setCurrentItem( 5 );
		}
	}
	gridResolutionChanged();

	// restore hear new notes button state
	hearNotesBtn->setPressed( pref->getHearNewNotes() );


	// Pattern size combo
	pPatternSizeCombo = new QComboBox( FALSE, this, "pPatternSizeCombo" );
//	pPatternSizeCombo->setEditable( true );
	pPatternSizeCombo->insertItem( trUtf8( "1" ) );
	pPatternSizeCombo->insertItem( trUtf8( "2" ) );
	pPatternSizeCombo->insertItem( trUtf8( "3" ) );
	pPatternSizeCombo->insertItem( trUtf8( "4" ) );
	pPatternSizeCombo->insertItem( trUtf8( "5" ) );
	pPatternSizeCombo->insertItem( trUtf8( "6" ) );
	pPatternSizeCombo->insertItem( trUtf8( "7" ) );
	pPatternSizeCombo->insertItem( trUtf8( "8" ) );
	pPatternSizeCombo->move( 550, 5 );
	pPatternSizeCombo->resize( 80, 20 );
	pPatternSizeCombo->setCurrentItem( 7 );
	QToolTip::add( pPatternSizeCombo, trUtf8( "Pattern size" ) );
	connect( pPatternSizeCombo, SIGNAL( activated(const QString&) ), this, SLOT( patternSizeComboChanged() ) );


	string recBtn_on_path = string(IMG_PATH).append("/img/patternEditor/recBtn_on.png");
	string recBtn_off_path = string(IMG_PATH).append("/img/patternEditor/recBtn_off.png");
	string recBtn_over_path = string(IMG_PATH).append("/img/patternEditor/recBtn_over.png");
	recordEventsBtn = new ToggleButton(this, QSize(20, 20), recBtn_on_path, recBtn_off_path, recBtn_over_path);
	recordEventsBtn->move(230, 5);
	recordEventsBtn->setPressed(pref->getRecordEvents());
	QToolTip::add( recordEventsBtn, trUtf8( "Record keyboard/midi events" ) );
	connect( recordEventsBtn, SIGNAL(clicked(Button*)), this, SLOT( recordEventsBtnClick(Button*)));



	string quantBtn_on_path = string(IMG_PATH).append("/img/patternEditor/quantBtn_on.png");
	string quantBtn_off_path = string(IMG_PATH).append("/img/patternEditor/quantBtn_off.png");
	string quantBtn_over_path = string(IMG_PATH).append("/img/patternEditor/quantBtn_over.png");
	quantizeEventsBtn = new ToggleButton(this, QSize(20, 20), quantBtn_on_path, quantBtn_off_path, quantBtn_over_path);
	quantizeEventsBtn->move(254, 5);
	quantizeEventsBtn->setPressed(pref->getQuantizeEvents());
	QToolTip::add( quantizeEventsBtn, trUtf8( "Quantize keyboard/midi events to grid" ) );
	connect( quantizeEventsBtn, SIGNAL(clicked(Button*)), this, SLOT( quantizeEventsBtnClick(Button*)));


/*
	// show velocity button
	string sVelocityBtn_on_path = string(IMG_PATH).append("/img/patternEditor/quantBtn_on.png");
	string sVelocityBtn_off_path = string(IMG_PATH).append("/img/patternEditor/quantBtn_off.png");
	string sVelocityBtn_over_path = string(IMG_PATH).append("/img/patternEditor/quantBtn_over.png");
	m_pShowVelocityBtn = new ToggleButton(this, QSize(40, 20), sVelocityBtn_on_path, sVelocityBtn_off_path, sVelocityBtn_over_path);
	m_pShowVelocityBtn->move( 10, 600);
	m_pShowVelocityBtn->setPressed( true );
	QToolTip::add( m_pShowVelocityBtn, trUtf8( "Show velocity editor" ) );
	connect( m_pShowVelocityBtn, SIGNAL(clicked(Button*)), this, SLOT( showVelocityBtnClick(Button*)));


	// show pitch button
	string sPitchBtn_on_path = string(IMG_PATH).append("/img/patternEditor/quantBtn_on.png");
	string sPitchBtn_off_path = string(IMG_PATH).append("/img/patternEditor/quantBtn_off.png");
	string sPitchBtn_over_path = string(IMG_PATH).append("/img/patternEditor/quantBtn_over.png");
	m_pShowPitchBtn = new ToggleButton(this, QSize(40, 20), sPitchBtn_on_path, sPitchBtn_off_path, sPitchBtn_over_path);
	m_pShowPitchBtn->move( 10, 630);
	m_pShowPitchBtn->setPressed( false );
	QToolTip::add( m_pShowPitchBtn, trUtf8( "Show pitch editor" ) );
	connect( m_pShowPitchBtn, SIGNAL(clicked(Button*)), this, SLOT( showPitchBtnClick(Button*)));
*/


	if ( bUseBackground ) {
		patternNameLbl->setBackgroundPixmap( m_backgroundPixmap );
		gridResolutionLbl->setBackgroundPixmap( m_backgroundPixmap );
		this->setBackgroundPixmap( m_backgroundPixmap );
	}
	else {
		patternNameLbl->setEraseColor( QColor(131, 149, 180 ) );
		gridResolutionLbl->setEraseColor( QColor(131, 149, 180 ) );
		this->setEraseColor( QColor(131, 149, 180 ) );
	}


	setupPattern();

	// we need to do all this to support the keyboard playing
	// for all the window modes
	pPatternSizeCombo->installEventFilter(HydrogenApp::getInstance()->getMainForm());
	gridResolutionCombo->installEventFilter(HydrogenApp::getInstance()->getMainForm());

	(Hydrogen::getInstance())->addEngineListener(this);

	selectedPatternChanged(); // force an update
}




PatternEditorPanel::~PatternEditorPanel()
{
	(Hydrogen::getInstance())->removeEngineListener(this);
	hide();
	//infoLog("DESTROYING");

	delete m_pPatternEditor;
	delete m_pPatternEditorRuler;
	delete m_pInstrumentList;
	delete m_pNotePropertiesRuler;
	//infoLog("DESTROY");
}



void PatternEditorPanel::setupPattern()
{
	if (m_pPattern) {
		string sCurrentPatternName = m_pPattern->getName();
		this->setCaption( ( trUtf8( "Pattern editor - %1").arg(sCurrentPatternName.c_str()) ) );
		patternNameLbl->setText( trUtf8( "Pattern name: <b>%1</b>").arg(sCurrentPatternName.c_str()) );

		// update pattern size combobox
		int nPatternSize = m_pPattern->getSize();
		int nQuarter = MAX_NOTES / 8;
		if (nPatternSize == nQuarter) {
			pPatternSizeCombo->setCurrentItem( 0 );
		}
		else if (nPatternSize == nQuarter * 2) {
			pPatternSizeCombo->setCurrentItem( 1 );
		}
		else if (nPatternSize == nQuarter * 3) {
			pPatternSizeCombo->setCurrentItem( 2 );
		}
		else if ( nPatternSize == nQuarter * 4 ) {
			pPatternSizeCombo->setCurrentItem( 3 );
		}
		else if ( nPatternSize == nQuarter * 5 ) {
			pPatternSizeCombo->setCurrentItem( 4 );
		}
		else if ( nPatternSize == nQuarter * 6 ) {
			pPatternSizeCombo->setCurrentItem( 5 );
		}
		else if ( nPatternSize == nQuarter * 7 ) {
			pPatternSizeCombo->setCurrentItem( 6 );
		}
		else if ( nPatternSize == nQuarter * 8 ) {
			pPatternSizeCombo->setCurrentItem( 7 );
		}
		else {
			errorLog( "[setupPattern] unhandled case: nPatternSize = " + toString(nPatternSize) );
		}
	}
	else {
		this->setCaption( ( trUtf8( "Pattern editor - %1").arg(QString("NULL")) ) );
		patternNameLbl->setText( trUtf8( "Pattern name: <b>%1</b>").arg(QString("NULL")) );
	}
}



/**
 *
 */
void PatternEditorPanel::syncToExternalHorizontalScrollbar(int)
{
	int nValue;

	// Editor
	nValue = m_pPatternEditorHScrollBar->value();
	QScrollBar *pEditorHScroll = m_pEditorScrollView->horizontalScrollBar();
	if (pEditorHScroll->value() != nValue ) {
		pEditorHScroll->setValue( nValue );
	}

	nValue = m_pPatternEditorVScrollBar->value();
	QScrollBar *pEditorVScroll = m_pEditorScrollView->verticalScrollBar();
	if (pEditorVScroll->value() != nValue) {
		pEditorVScroll->setValue( nValue );
	}


	// Ruler
	nValue = m_pPatternEditorHScrollBar->value();
	QScrollBar *pRulerHScroll = m_pRulerScrollView->horizontalScrollBar();
	if ( pRulerHScroll->value() != nValue ) {
		pRulerHScroll->setValue( nValue );
	}

	// Instrument list
	nValue = m_pPatternEditorVScrollBar->value();
	QScrollBar *pInstrListVScroll = m_pInstrListScrollView->verticalScrollBar();
	if (pInstrListVScroll->value() != nValue ) {
		pInstrListVScroll->setValue( nValue );
	}

	// Velocity ruler
	nValue = m_pPatternEditorHScrollBar->value();
	QScrollBar *pVelocityRulerHScroll = m_pNotePropertiesScrollView->horizontalScrollBar();
	if ( pVelocityRulerHScroll->value() != nValue ) {
		pVelocityRulerHScroll->setValue( nValue );
	}
}




/**
 *
 */
void PatternEditorPanel::gridResolutionChanged()
{

	string selected = (gridResolutionCombo->currentText()).latin1();

	uint nResolution;
	bool bUseTriplets = false;
	if (selected == "4") {
		nResolution = 4;
	}
	else if (selected == "8") {
		nResolution = 8;
	}
	else if (selected == "16") {
		nResolution = 16;
	}
	else if (selected == "32") {
		nResolution = 32;
	}
	else if (selected == "64") {
		nResolution = 64;
	}
	else if (selected == "4T") {
		nResolution = 8;
		bUseTriplets = true;
	}
	else if (selected == "8T") {
		nResolution = 16;
		bUseTriplets = true;
	}
	else if (selected == "16T") {
		nResolution = 32;
		bUseTriplets = true;
	}
	else if (selected == "32T") {
		nResolution = 64;
		bUseTriplets = true;
	}
	m_pPatternEditor->setResolution( nResolution, bUseTriplets );

	( Preferences::getInstance() )->setPatternEditorGridResolution( atoi( selected.c_str() ) );
	( Preferences::getInstance() )->setPatternEditorUsingTriplets( bUseTriplets );
}




///
/// This method will be called only when the engine is not playing. Engine lock/unlock not needed
///
void PatternEditorPanel::patternSizeComboChanged()
{
	string selected = (pPatternSizeCombo->currentText()).latin1();

	uint nQuarter = MAX_NOTES / 8;

	uint nSize;
	if (selected == "1") {
		nSize = nQuarter;
	}
	else if (selected == "2") {
		nSize = nQuarter * 2;
	}
	else if (selected == "3") {
		nSize = nQuarter * 3;
	}
	else if (selected == "4") {
		nSize = nQuarter * 4;
	}
	else if (selected == "5") {
		nSize = nQuarter * 5;
	}
	else if (selected == "6") {
		nSize = nQuarter * 6;
	}
	else if (selected == "7") {
		nSize = nQuarter * 7;
	}
	else if (selected == "8") {
		nSize = nQuarter * 8;
	}
	else {
		cout << "[PatternEditorPanel::patternSizeComboChanged()] Unhandled case " << selected << endl;
	}


	if (m_pPattern != NULL) {
		m_pPattern->setSize( nSize );
	}

	m_pPatternEditorRuler->updateEditor( true );	// redraw all
}






/**
 *
 */
void PatternEditorPanel::updateStart( bool bStart )
{
	m_pPatternEditorRuler->updateStart( bStart );
}




/// This method is called from another thread (audio engine)
void PatternEditorPanel::patternModified()
{
	H2TextEvent *ev = new H2TextEvent("patternModified");
	QApplication::postEvent(this, ev);
}

/// This method is called from another thread (audio engine)
void PatternEditorPanel::selectedPatternChanged()
{
	H2TextEvent *ev = new H2TextEvent("selectedPatternChanged");
	QApplication::postEvent(this, ev);
}



/**
 *
 */
void PatternEditorPanel::customEvent( QCustomEvent *ev )
{
	if ( ev->type() != H2_TEXT_EVENT ) {	// Must be a H2TextEvent
		return;
	}
	QString message = ((H2TextEvent*)ev)->getText();

	if (message == "enablePatternSizeCombo") {
		pPatternSizeCombo->setEnabled( true );
	}
	else if (message == "disablePatternSizeCombo") {
		pPatternSizeCombo->setEnabled( false );
	}
	else if (message == "patternModified") {
		// full update of display
		m_pPatternEditor->updateEditor(true);
	}
	else if (message == "selectedPatternChanged") {
		// full update of display
		m_pPatternEditor->updateEditor(true);
		m_pNotePropertiesRuler->updateEditor();

		Hydrogen *pEngine = Hydrogen::getInstance();
		PatternList *pPatternList = pEngine->getSong()->getPatternList();
		int nSelectedPatternNumber = pEngine->getSelectedPatternNumber();
		if ( (nSelectedPatternNumber != -1) && ( (uint)nSelectedPatternNumber < pPatternList->getSize() ) ) {
			m_pPattern = pPatternList->get( nSelectedPatternNumber );
		}
		else {
			m_pPattern = NULL;
		}
		setupPattern();
	}
}




/**
 *
 */
void PatternEditorPanel::hearNotesBtnClick(Button *ref)
{
	Preferences *pref = ( Preferences::getInstance() );
	pref->setHearNewNotes( ref->isPressed() );

	if (ref->isPressed() ) {
		( HydrogenApp::getInstance() )->setStatusBarMessage( trUtf8( "Hear new notes = On" ), 2000 );
	}
	else {
		( HydrogenApp::getInstance() )->setStatusBarMessage( trUtf8( "Hear new notes = Off" ), 2000 );
	}

}


/**
 *
 */
void PatternEditorPanel::recordEventsBtnClick(Button *ref)
{
	Preferences *pref = ( Preferences::getInstance() );
	pref->setRecordEvents( ref->isPressed() );

	if (ref->isPressed() ) {
		( HydrogenApp::getInstance() )->setStatusBarMessage( trUtf8( "Record keyboard/midi events = On" ), 2000 );
	}
	else {
		( HydrogenApp::getInstance() )->setStatusBarMessage( trUtf8( "Record keyboard/midi events = Off" ), 2000 );
	}

}

/**
 *
 */
void PatternEditorPanel::quantizeEventsBtnClick(Button *ref)
{
	Preferences *pref = ( Preferences::getInstance() );
	pref->setQuantizeEvents( ref->isPressed() );

	if (ref->isPressed() ) {
		( HydrogenApp::getInstance() )->setStatusBarMessage( trUtf8( "Quantize incoming keyboard/midi events = On" ), 2000 );
	}
	else {
		( HydrogenApp::getInstance() )->setStatusBarMessage( trUtf8( "Quantize incoming keyboard/midi events = Off" ), 2000 );
	}
}




/**
 *
 */
void PatternEditorPanel::stateChanged(int state)
{
	if ( state == READY) {
		H2TextEvent *ev = new H2TextEvent("enablePatternSizeCombo");
		QApplication::postEvent(this, ev);
	}
	else {
		H2TextEvent *ev = new H2TextEvent("disablePatternSizeCombo");
		QApplication::postEvent(this, ev);
	}
}



void PatternEditorPanel::resizeEvent( QResizeEvent *ev )
{

	m_nPanelHeight = height();
	m_nPanelWidth = width();

	m_pEditorScrollView->resize( m_nPanelWidth - 120, m_nPanelHeight - 160 );
	m_pRulerScrollView->resize( m_pEditorScrollView->width(), m_pRulerScrollView->height() );
	m_pInstrListScrollView->resize( m_pInstrListScrollView->width(), m_pEditorScrollView->height() );

	// pattern editor Horizontal scroll bar
	m_pPatternEditorHScrollBar->resize( m_pEditorScrollView->width(), 16 );
	m_pPatternEditorHScrollBar->move( 105, m_nPanelHeight - 16 );
	m_pPatternEditorHScrollBar->setMinValue( 0 );
	m_pPatternEditorHScrollBar->setMaxValue( m_pEditorFrame->width() - m_pEditorScrollView->width() );
	m_pPatternEditorHScrollBar->setLineStep(20);
	m_pPatternEditorHScrollBar->setPageStep(300);

	// pattern editor Vertical scroll bar
	m_pPatternEditorVScrollBar->resize( 16, m_pEditorScrollView->height() );
	m_pPatternEditorVScrollBar->move( m_nPanelWidth - 16, m_pEditorScrollView->y() );
	m_pPatternEditorVScrollBar->setMinValue( 0 );
	m_pPatternEditorVScrollBar->setMaxValue( m_pEditorFrame->height() - m_pEditorScrollView->height() );
	m_pPatternEditorVScrollBar->setLineStep(20);
	m_pPatternEditorVScrollBar->setPageStep(300);


	unsigned nNoteEditor_y = m_nPanelHeight - 115;

	// Velocity ruler
	m_pNotePropertiesScrollView->move( 105, nNoteEditor_y );
	m_pNotePropertiesScrollView->resize( m_pEditorScrollView->width(), m_pNotePropertiesRuler->height() );

//	m_pShowVelocityBtn->move( 10, nNoteEditor_y + 10 );
//	m_pShowPitchBtn->move( 10, nNoteEditor_y + 10 + 30);

}




void PatternEditorPanel::showEvent ( QShowEvent *ev )
{
	m_pPatternEditorVScrollBar->setValue( m_pPatternEditorVScrollBar->maxValue() );
}


/// richiamato dall'uso dello scroll del mouse
void PatternEditorPanel::contentsMoving(int dummy)
{
	syncToExternalHorizontalScrollbar(0);
}




void PatternEditorPanel::setSelectedInstrument( int nInstr )
{
	//infoLog( "[setSelectedInstrument]" + toString( nInstr ) );
	Hydrogen::getInstance()->setSelectedInstrumentNumber( nInstr );

	m_pPatternEditor->updateEditor(true);	// force repaint
	m_pInstrumentList->updateEditor();
	m_pNotePropertiesRuler->updateEditor();
}



void PatternEditorPanel::selectedInstrumentChanged()
{
	// aggiorno tutti i componenti
	m_pPatternEditor->updateEditor(true);	// force repaint
	m_pInstrumentList->updateEditor();
	m_pNotePropertiesRuler->updateEditor();
}



void PatternEditorPanel::showVelocityBtnClick(Button *ref)
{
	m_pShowVelocityBtn->setPressed( true );
	m_pShowPitchBtn->setPressed( false );
	m_pNotePropertiesRuler->setMode( NotePropertiesRuler::VELOCITY );
}



void PatternEditorPanel::showPitchBtnClick(Button *ref)
{
	m_pShowPitchBtn->setPressed( true );
	m_pShowVelocityBtn->setPressed( false );
	m_pNotePropertiesRuler->setMode( NotePropertiesRuler::PITCH );
}

