/*
 * Hydrogen
 * Copyright(c) 2002-2004 by Alex >Comix< Cominu [comix@users.sourceforge.net]
 *
 * http://hydrogen.sourceforge.net
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY, without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $Id: InstrumentPropertiesDialog.cpp,v 1.40 2004/08/25 10:46:08 comix Exp $
 *
 */

#include "InstrumentPropertiesDialog.h"
#include "FilePreview.h"

#include "HydrogenApp.h"

#include "qlineedit.h"
#include "qlistbox.h"
#include "qlabel.h"
#include "qfiledialog.h"
#include "qspinbox.h"
#include "qcheckbox.h"
#include "qinputdialog.h"

#include <math.h>

#include "Song.h"
#include "Hydrogen.h"
#include "Globals.h"
#include "config.h"
#include "Mixer.h"
#include "PatternEditorPanel.h"

InstrumentPropertiesDialog::InstrumentPropertiesDialog( QWidget* parent )
 : InstrumentPropertiesDialog_UI(parent)
 , Object( "InstrPropDialog" )
 , m_pMixerLine( NULL )
 , m_pInstrument( NULL )
{
	//infoLog( "INIT" );
	setMinimumSize( width(), height() );	// not resizable
//	setMaximumSize( width(), height() );	// not resizable
	setCaption( trUtf8( "Instrument properties" ) );
	setIcon( QPixmap( QString(IMG_PATH) + QString( "/img/icon32.png") ) );

	QColor backgroundColor(131, 149, 180 );

	this->setEraseColor( backgroundColor );
	textLabel2->setEraseColor( backgroundColor );
	textLabel2_2->setEraseColor( backgroundColor );
	m_pSampleSizeLbl->setEraseColor( backgroundColor );
	textLabel3_2->setEraseColor( backgroundColor );
	textLabel3->setEraseColor( backgroundColor );



	m_pMixerLine = new MixerLine( this );
	m_pMixerLine->move( 5, 5 );

	connect( m_pMixerLine, SIGNAL( noteOnClicked(MixerLine*) ), this, SLOT( noteOnClicked(MixerLine*) ) );
	connect( m_pMixerLine, SIGNAL( noteOffClicked(MixerLine*) ), this, SLOT( noteOffClicked(MixerLine*) ) );
 	connect( m_pMixerLine, SIGNAL( muteBtnClicked(MixerLine*) ), this, SLOT( muteClicked(MixerLine*) ) );
 	connect( m_pMixerLine, SIGNAL( soloBtnClicked(MixerLine*) ), this, SLOT( soloClicked(MixerLine*) ) );
 	connect( m_pMixerLine, SIGNAL( volumeChanged(MixerLine*) ), this, SLOT( volumeChanged(MixerLine*) ) );
 	connect( m_pMixerLine, SIGNAL( instrumentNameClicked(MixerLine*) ), this, SLOT( nameClicked(MixerLine*) ) );
// 	connect( m_pMixerLine, SIGNAL( instrumentNameSelected(MixerLine*) ), this, SLOT( nameSelected(MixerLine*) ) );
 	connect( m_pMixerLine, SIGNAL( panChanged(MixerLine*) ), this, SLOT( panChanged( MixerLine*) ) );
 	connect( m_pMixerLine, SIGNAL( knobChanged(MixerLine*, int) ), this, SLOT( knobChanged( MixerLine*, int) ) );

	m_pLayerListBox->clear();
	for (uint nLayer = 0; nLayer < MAX_LAYERS; nLayer++) {
		m_pLayerListBox->insertItem( trUtf8( "Empty layer" ) );
	}
	m_pLayerListBox->setCurrentItem( 0 );


	(Hydrogen::getInstance())->addEngineListener(this);

	selectedInstrumentChanged(); 	// force an update

	m_pTimer = new QTimer( this );
	connect( m_pTimer, SIGNAL( timeout() ), this, SLOT( updatePeak() ) );

	m_pTimer->start( 50 );
}



InstrumentPropertiesDialog::~InstrumentPropertiesDialog()
{
	m_pTimer->stop();
	//infoLog( "DESTROY" );
	delete m_pMixerLine;
}



void InstrumentPropertiesDialog::selectedInstrumentChanged()
{
	Hydrogen::getInstance()->lockEngine("InstrumentPropertiesDialog::selectedInstrumentChanged" );

	Song *pSong = Hydrogen::getInstance()->getSong();
	if (pSong != NULL) {
		InstrumentList *pInstrList = pSong->getInstrumentList();
		int nInstr = Hydrogen::getInstance()->getSelectedInstrumentNumber();
		if (nInstr == -1) {
			m_pInstrument = NULL;
		}
		else {
			m_pInstrument = pInstrList->get( nInstr );
		}
	}
	else {
		m_pInstrument = NULL;
	}
	Hydrogen::getInstance()->unlockEngine();

	// update layer list
	if (m_pInstrument) {
		for ( int nLayer = 0; nLayer < MAX_LAYERS; nLayer++) {
			QString sText = trUtf8( "Empty layer" );

			InstrumentLayer *pLayer = m_pInstrument->getLayer( nLayer );
			if (pLayer) {
//				Hydrogen::getInstance()->lockEngine("InstrumentPropertiesDialog::selectedInstrumentChanged" );
				// mostro solo il nome del file, senza path
				if ( pLayer->getSample() != NULL ) {
					string sFilename = pLayer->getSample()->getFilename();
					int nPos = sFilename.rfind("/");
					sFilename = sFilename.substr( nPos + 1, sFilename.length() );
					sText = QString( sFilename.c_str() );
				}
//				Hydrogen::getInstance()->unlockEngine();
			}

			if ( m_pLayerListBox->text( nLayer ) != sText ) {
				m_pLayerListBox->changeItem( sText, nLayer );
			}
		}
	}
	m_pLayerListBox->setCurrentItem( 0 );
}



void InstrumentPropertiesDialog::layerSelected()
{
	int nLayerSelected = m_pLayerListBox->currentItem();

//	infoLog( "[layerSelected] " + toString( nLayerSelected ) );
	if (nLayerSelected == -1) {
		return;
	}

	Instrument *pInstr = NULL;

	string sFilename = "Empty layer";
	int nMin = 0;
	int nMax = 0;
	int nGain = 0;
	int nPitch = 0;
	int nBytes = 0;

	bool bSampleFilenameLblEnabled = false;
	bool bMinSpinBoxEnabled = false;
	bool bMaxSpinBoxEnabled = false;
	bool bGainSpinboxEnabled = false;
	bool bPitchSpinboxEnabled = false;
	bool bSampleSizeLblEnabled = false;
	bool bPlayLayerBtnEnabled = false;
	bool bDeleteBtnEnabled = false;


	Hydrogen::getInstance()->lockEngine("InstrumentPropertiesDialog::layerSelected" );


	Song *pSong = Hydrogen::getInstance()->getSong();
	if (pSong != NULL) {
		int nInstr = Hydrogen::getInstance()->getSelectedInstrumentNumber();

		InstrumentLayer *pLayer = NULL;
		if (nInstr != -1 ) {
			pInstr = pSong->getInstrumentList()->get( nInstr );
			if (pInstr) {
				pLayer = pInstr->getLayer( nLayerSelected );
			}
		}

		if ( (pLayer) && (pLayer->getSample() != NULL ) ) {

			// mostro solo il nome del file, senza path
			sFilename = pLayer->getSample()->getFilename();
			int nPos = sFilename.rfind("/");
			sFilename = sFilename.substr( nPos + 1, sFilename.length() );

			nMin = (int)::round( pLayer->getStartVelocity() * 100.0 );
			nMax = (int)::round( pLayer->getEndVelocity() * 100.0 );
			nGain = (int)::round( pLayer->getGain() * 100.0 );
			nPitch = (int)::round( pLayer->getPitch() );
			nBytes = pLayer->getSample()->getNBytes();

			bMinSpinBoxEnabled = true;
			bMaxSpinBoxEnabled = true;
			bGainSpinboxEnabled = true;
			bPitchSpinboxEnabled = true;
			bSampleFilenameLblEnabled = true;
			bSampleSizeLblEnabled = true;
			bPlayLayerBtnEnabled = true;
			bDeleteBtnEnabled = true;
		}
		Hydrogen::getInstance()->unlockEngine();
	}
	else {
		Hydrogen::getInstance()->unlockEngine();
	}

	m_pMinSpinBox->setValue( nMin );
	m_pMaxSpinBox->setValue( nMax );
	m_pGainSpinBox->setValue( nGain );
	m_pPitchSpinBox->setValue( nPitch );
	m_pSampleSizeLbl->setText( QString( "%1 frames" ).arg( nBytes ) );

	m_pMinSpinBox->setEnabled( bMinSpinBoxEnabled );
	m_pMaxSpinBox->setEnabled( bMaxSpinBoxEnabled );
	m_pGainSpinBox->setEnabled( bGainSpinboxEnabled );
	m_pPitchSpinBox->setEnabled( bPitchSpinboxEnabled );
	m_pSampleSizeLbl->setEnabled( bSampleSizeLblEnabled );
	m_pPlayLayerBtn->setEnabled( bPlayLayerBtnEnabled );
	m_pDeleteBtn->setEnabled( bDeleteBtnEnabled );


}



void InstrumentPropertiesDialog::browseBtnClicked()
{
	static QString lastUsedDir = "";

	Hydrogen *engine = Hydrogen::getInstance();
	MainForm *pMainForm = HydrogenApp::getInstance()->getMainForm();

	QFileDialog *fd = new QFileDialog( (QWidget*)pMainForm, "File Dialog", TRUE);
	fd->setMode(QFileDialog::ExistingFile);
	fd->setFilter( trUtf8("Audio files (*.wav *.WAV *.au *.AU *.aiff *.AIFF *.flac *.FLAC)") );
	fd->setCaption( trUtf8("Hydrogen - Load instrument") );
	fd->setDir( lastUsedDir );

	FilePreview *pPreview = new FilePreview();
	fd->setContentsPreviewEnabled( TRUE );
	fd->setContentsPreview( pPreview, pPreview );
	fd->setPreviewMode( QFileDialog::Contents );

	QString filename = "";
	if (fd->exec() == QDialog::Accepted) {
		filename = fd->selectedFile();
	}

	if (filename != "") {
		lastUsedDir = fd->dirPath();
		Sample *newSample = Sample::load( filename.latin1() );

		Instrument *pInstr = NULL;
		int nLayerSelected = m_pLayerListBox->currentItem();

		engine->lockEngine("InstrumentPropertiesDialog::browseBtnClicked");
		Song *song = engine->getSong();
		InstrumentList *instrList = song->getInstrumentList();
		pInstr = instrList->get( engine->getSelectedInstrumentNumber() );

		InstrumentLayer *pLayer = pInstr->getLayer( nLayerSelected );
		if (pLayer != NULL) {
			// delete old sample
			Sample *oldSample = pLayer->getSample();
			delete oldSample;

			// insert new sample from newInstrument
			pLayer->setSample( newSample );
		}
		else {
			pLayer = new InstrumentLayer(newSample);
			pInstr->setLayer( pLayer, nLayerSelected );
		}

		pInstr->setDrumkitName( "" );	// external sample, no drumkit info

		engine->unlockEngine();
	}

	selectedInstrumentChanged();	// update all
}



void InstrumentPropertiesDialog::minValueChanged()
{
	Song *pSong = Hydrogen::getInstance()->getSong();
	if (pSong != NULL) {
		int nInstr = Hydrogen::getInstance()->getSelectedInstrumentNumber();
		Instrument *pInstr = pSong->getInstrumentList()->get( nInstr );

		int nLayerSelected = m_pLayerListBox->currentItem();
		InstrumentLayer *pLayer = pInstr->getLayer( nLayerSelected );
		if (pLayer) {
			float fValue = (float)m_pMinSpinBox->value() / 100.0;
			pLayer->setStartVelocity( fValue );
		}
		else {
			//warningLog( "[minValueChanged] pLayer = NULL" );
		}
	}
}



void InstrumentPropertiesDialog::maxValueChanged()
{
	Song *pSong = Hydrogen::getInstance()->getSong();
	if (pSong != NULL) {
		int nInstr = Hydrogen::getInstance()->getSelectedInstrumentNumber();
		Instrument *pInstr = pSong->getInstrumentList()->get( nInstr );

		int nLayerSelected = m_pLayerListBox->currentItem();
		InstrumentLayer *pLayer = pInstr->getLayer( nLayerSelected );
		if (pLayer) {
			float fValue = (float)m_pMaxSpinBox->value() / 100.0;
			pLayer->setEndVelocity( fValue );
		}
		else {
			//warningLog( "[maxValueChanged] pLayer = NULL" );
		}
	}
}



void InstrumentPropertiesDialog::gainValueChanged()
{
	Song *pSong = Hydrogen::getInstance()->getSong();
	if (pSong != NULL) {
		int nInstr = Hydrogen::getInstance()->getSelectedInstrumentNumber();
		Instrument *pInstr = pSong->getInstrumentList()->get( nInstr );

		int nLayerSelected = m_pLayerListBox->currentItem();
		InstrumentLayer *pLayer = pInstr->getLayer( nLayerSelected );
		if (pLayer) {
			float fValue = (float)m_pGainSpinBox->value() / 100.0;
			pLayer->setGain( fValue );
		}
		else {
			//warningLog( "[gainValueChanged] pLayer = NULL" );
		}
	}
	else {
		warningLog( "[gainValueChanged] pSong = NULL" );
	}
}



void InstrumentPropertiesDialog::pitchValueChanged()
{
	Song *pSong = Hydrogen::getInstance()->getSong();
	if (pSong != NULL) {
		int nInstr = Hydrogen::getInstance()->getSelectedInstrumentNumber();
		Instrument *pInstr = pSong->getInstrumentList()->get( nInstr );

		int nLayerSelected = m_pLayerListBox->currentItem();
		InstrumentLayer *pLayer = pInstr->getLayer( nLayerSelected );
		if (pLayer) {
			float fValue = (float)m_pPitchSpinBox->value();
			pLayer->setPitch( fValue );
		}
		else {
			warningLog( "[pitchValueChanged] pLayer = NULL" );
		}
	}
}



void InstrumentPropertiesDialog::playLayerBtnClicked()
{
	Hydrogen *pEngine = Hydrogen::getInstance();

	Song *song = pEngine->getSong();
	InstrumentList *instrList = song->getInstrumentList();
	int nSelectedInstr = pEngine->getSelectedInstrumentNumber();
	if (nSelectedInstr == -1) {
		return;
	}

	Instrument *pInstr = instrList->get( nSelectedInstr );
	if (pInstr) {

		int nLayerSelected = m_pLayerListBox->currentItem();
		InstrumentLayer *pLayer = pInstr->getLayer( nLayerSelected );
		if ( pLayer) {
//			float fVelocity = ( pLayer->getStartVelocity() + pLayer->getEndVelocity() ) / 2;
			float fVelocity = pLayer->getEndVelocity() - 0.01;

			Note *note = new Note( 0, fVelocity, 1.0f, 1.0f, -1 );
			note->setInstrument( pInstr );

			pEngine->noteOn( note );
		}
	}
}



void InstrumentPropertiesDialog::updatePeak()
{
//	infoLog( "[updatePeak]" );
	if (m_pInstrument) {
		m_pMixerLine->setName( QString( m_pInstrument->getName().c_str() ) );
		m_pMixerLine->setVolume( m_pInstrument->getVolume() );

		// activity
		if ( m_pMixerLine->getActivity() > 0 ) {
			m_pMixerLine->setActivity( m_pMixerLine->getActivity() - 30 );
			m_pMixerLine->setPlayClicked( true );
		}
		else {
			m_pMixerLine->setPlayClicked( false );
		}


		m_pMixerLine->setMuteClicked( m_pInstrument->isMuted() );

		///\todo completare il solo
//		m_pMixerLine->setSoloClicked( m_pMixerLine->isSoloClicked() );



		float panValue = 0.0;
		float pan_L = m_pInstrument->getPan_L();
		float pan_R = m_pInstrument->getPan_R();
		if (pan_R == 1.0) {
			panValue = 1.0 - (pan_L / 2.0);
		}
		else {
			panValue = pan_R / 2.0;
		}
		panValue = panValue * 100.0;
		m_pMixerLine->setPan( (int)panValue );



		Mixer *pMixer = HydrogenApp::getInstance()->getMixer();

		int nSelectedInstrument = Hydrogen::getInstance()->getSelectedInstrumentNumber();
		float fPeak_L;
		float fPeak_R;
		pMixer->getPeaksInMixerLine( nSelectedInstrument, fPeak_L, fPeak_R );


		m_pMixerLine->setPeak_L( fPeak_L );
		m_pMixerLine->setPeak_R( fPeak_R );

		for ( uint nFX = 0; nFX < MAX_FX; nFX++ ) {
			m_pMixerLine->setFXLevel( nFX, m_pInstrument->getFXLevel( nFX ) );
		}
	}
	else {
		m_pMixerLine->setName( QString( "-" ) );
		m_pMixerLine->setVolume( 0.0 );

		m_pMixerLine->setActivity( 0 );
		m_pMixerLine->setMuteClicked( false );
		m_pMixerLine->setSoloClicked( false );
		m_pMixerLine->setPan( 0 );
		m_pMixerLine->setPeak_L( 0 );
		m_pMixerLine->setPeak_R( 0 );

		for ( uint nFX = 0; nFX < MAX_FX; nFX++ ) {
			m_pMixerLine->setFXLevel( nFX, 0.0 );
		}
	}

	m_pMixerLine->updateMixerLine();
}



void InstrumentPropertiesDialog::noteOnClicked(MixerLine* ref)
{
	if (m_pInstrument) {
		Note *note = new Note( 0, 1.0, 1.0f, 1.0f, -1 );
		note->setInstrument( m_pInstrument );
		Hydrogen::getInstance()->noteOn( note );
	}
}



void InstrumentPropertiesDialog::noteOffClicked(MixerLine* ref)
{
	if (m_pInstrument) {
		Note *note = new Note( 0, 1.0, 1.0f, 1.0f, -1 );
		note->setInstrument( m_pInstrument );
		Hydrogen::getInstance()->noteOff( note );
	}
}



void InstrumentPropertiesDialog::muteClicked(MixerLine* ref)
{
	if (m_pInstrument) {
		m_pInstrument->setMuted( ref->isMuteClicked() );
		HydrogenApp::getInstance()->getPatternEditorPanel()->getPatternEditor()->updateEditor( true );
	}
}



void InstrumentPropertiesDialog::soloClicked(MixerLine* ref)
{
	infoLog( "[soloClicked] not implemented yet" );
}



void InstrumentPropertiesDialog::volumeChanged(MixerLine* ref)
{
	if (m_pInstrument) {
		m_pInstrument->setVolume( ref->getVolume() );
	}
}



void InstrumentPropertiesDialog::panChanged(MixerLine* ref)
{
	if (m_pInstrument) {
		float panValue = ref->getPan();

		float pan_L = (100.0 - panValue) / 100.0;
		float pan_R = panValue / 100.0;

		panValue = panValue / 100.0;

		if (panValue >= 0.5) {
			pan_L = (1.0 - panValue) * 2;
			pan_R = 1.0;
		}
		else {
			pan_L = 1.0;
			pan_R = ( 1.0 - ( 1.0 - panValue) ) * 2;
		}

		m_pInstrument->setPan_L( pan_L );
		m_pInstrument->setPan_R( pan_R );
	}
}



void InstrumentPropertiesDialog::knobChanged( MixerLine* ref, int nKnob )
{
	if (m_pInstrument) {
		m_pInstrument->setFXLevel( nKnob, ref->getFXLevel(nKnob) );
		QString sInfo = trUtf8( "Set FX %1 level ").arg( nKnob );
		( HydrogenApp::getInstance() )->setStatusBarMessage( sInfo+ QString( "[%1]" ).arg( ref->getFXLevel(nKnob), 0, 'f', 2 ), 2000 );
	}
}



void InstrumentPropertiesDialog::deleteBtnClicked()
{
	int nLayerSelected = m_pLayerListBox->currentItem();

	Hydrogen *pEngine = Hydrogen::getInstance();
	pEngine->lockEngine( "InstrumentPropertiesDialog::deleteBtnClicked" );

	Song *song = pEngine->getSong();
	InstrumentList *instrList = song->getInstrumentList();
	int nSelectedInstr = pEngine->getSelectedInstrumentNumber();
	if (nSelectedInstr == -1) {
		return;
	}

	Instrument *pInstr = instrList->get( nSelectedInstr );
	if (pInstr) {
		InstrumentLayer *pLayer = pInstr->getLayer( nLayerSelected );
		if (pLayer) {
			pInstr->setLayer( NULL, nLayerSelected );
			delete pLayer;
		}
	}
	pEngine->unlockEngine();

	selectedInstrumentChanged();	// update all
}



void InstrumentPropertiesDialog::nameClicked(MixerLine* ref)
{
	Hydrogen *pEngine = Hydrogen::getInstance();

	Song *song = pEngine->getSong();
	InstrumentList *instrList = song->getInstrumentList();
	int nSelectedInstr = pEngine->getSelectedInstrumentNumber();
	if (nSelectedInstr == -1) {
		return;
	}

	Instrument *pInstr = instrList->get( nSelectedInstr );
	if (pInstr) {
		bool bIsOkPressed;
		QString sOldName = QString( pInstr->getName().c_str() );
		QString text = QInputDialog::getText( "Hydrogen", trUtf8( "Instrument name" ), QLineEdit::Normal, sOldName, &bIsOkPressed, this );
		if ( bIsOkPressed && !text.isEmpty() ) {
			// user entered something and pressed OK
			pEngine->lockEngine( "InstrumentPropertiesDialog::nameClicked" );
			pInstr->setName( text.latin1() );
			pEngine->unlockEngine();

			HydrogenApp::getInstance()->getPatternEditorPanel()->getInstrumentList()->updateEditor();
		}
		else {
			// user entered nothing or pressed Cancel
		}
	}
}
