/*
 * Hydrogen
 * Copyright(c) 2002-2004 by Alex >Comix< Cominu [comix@users.sourceforge.net]
 *
 * http://hydrogen.sourceforge.net
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY, without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $Id: AudioEngineInfoForm.cpp,v 1.35 2004/05/06 12:47:15 comix Exp $
 *
 */


#include "AudioEngineInfoForm.h"

#include "config.h"
#include "qwidget.h"

#include "HydrogenApp.h"

#include "lib/Preferences.h"
#include "lib/Hydrogen.h"
#include "lib/drivers/AlsaMidiDriver.h"

/**
 * Constructor
 */
AudioEngineInfoForm::AudioEngineInfoForm(QWidget* parent) : AudioEngineInfoForm_UI( parent ), Object( "AudioEngineInfoForm" )
{
//	cout << "INIT" << endl;

	setMinimumSize( width(), height() );	// not resizable
	setMaximumSize( width(), height() );	// not resizable

	setCaption( trUtf8( "Audio Engine Info" ) );
	setIcon( QPixmap( QString(IMG_PATH) + QString( "/img/icon32.png") ) );

	updateInfo();
	//currentPatternLbl->setText("NULL pattern");

	timer = new QTimer(this);
	connect(timer, SIGNAL(timeout()), this, SLOT(updateInfo()));

	(Hydrogen::getInstance())->addEngineListener(this);
	updateAudioEngineState();
}



/**
 * Destructor
 */
AudioEngineInfoForm::~AudioEngineInfoForm() {
//	cout << "DESTROY" << endl;
}




/**
 * show event
 */
void AudioEngineInfoForm::showEvent ( QShowEvent *ev ) {
	updateInfo();
	timer->start(200);
}




/**
 * hide event
 */
void AudioEngineInfoForm::hideEvent ( QHideEvent *ev ) {
	timer->stop();
}




void AudioEngineInfoForm::updateInfo() {
	Hydrogen *engine = Hydrogen::getInstance();
	Song *song = engine->getSong();

	// Song position
	QString sSongPos = "N/A";
	if ( engine->getPatternPos() != -1 ) {
		sSongPos = QString::number( engine->getPatternPos() );
	}
	m_pSongPositionLbl->setText( sSongPos );


	// Audio engine Playing notes
	char tmp[100];
	sprintf(tmp, "%03d / %03d", engine->getPlayingNotes(), (Preferences::getInstance())->getMaxNotes() );
	playingNotesLbl->setText( QString(tmp) );

	// Process time
	int perc = 0;
	if ( engine->getMaxProcessTime() != 0.0 ) {
		perc= (int)( engine->getProcessTime() / ( engine->getMaxProcessTime() / 100.0 ) );
	}
	sprintf(tmp, "%#.2f / %#.2f  (%d%%)", engine->getProcessTime(), engine->getMaxProcessTime(), perc );
	processTimeLbl->setText(tmp);

	// Song state
	if (song == NULL) {
		songStateLbl->setText( "NULL song" );
	}
	else {
		if (song->isModified()) {
			songStateLbl->setText( "Modified" );
		}
		else {
			songStateLbl->setText( "Saved" );
		}
	}

	// Number of frames
	sprintf(tmp, "%d", (int)engine->getTotalFrames() );
	nFramesLbl->setText(tmp);

	// tick number
	sprintf(tmp, "%03d", (int)engine->getTickPosition() );
	nTicksLbl->setText(tmp);



	// Audio driver info
	GenericDriver *driver = engine->getAudioDriver();
	if (driver) {
		QString audioDriverName  = (driver->getClassName()).c_str();
		driverLbl->setText(audioDriverName);

		// Audio driver buffer size
		sprintf(tmp, "%d", driver->getBufferSize());
		bufferSizeLbl->setText(QString(tmp));

		// Audio driver sampleRate
		sprintf(tmp, "%d", driver->getSampleRate());
		sampleRateLbl->setText(QString(tmp));
	}
	else {
		driverLbl->setText( "NULL driver" );
		bufferSizeLbl->setText( "N/A" );
		sampleRateLbl->setText( "N/A" );
	}


	// Midi driver info
#ifdef USE_ALSA_SEQ
	AlsaMidiDriver *midiDriver = engine->getMidiDriver();
	QString midiName = (midiDriver->getClassName()).c_str();
	midiDriverName->setText(midiName);
#else
	midiDriverName->setText("No MIDI driver support");
#endif

	int nSelectedPatternNumber = engine->getSelectedPatternNumber();
	if (nSelectedPatternNumber == -1) {
		selectedPatLbl->setText( "N/A");
	}
	else {
		selectedPatLbl->setText( QString("%1").arg(nSelectedPatternNumber) );
	}

	int nSelectedInstrumentNumber = engine->getSelectedInstrumentNumber();
	if (nSelectedInstrumentNumber == -1) {
		m_pSelectedInstrLbl->setText( "N/A" );
	}
	else {
		m_pSelectedInstrLbl->setText( QString("%1").arg(nSelectedInstrumentNumber) );
	}


	string currentPatternName;
	PatternList *pPatternList = (Hydrogen::getInstance())->getCurrentPatternList();
	if (pPatternList) {
		currentPatternLbl->setText( QString::number(pPatternList->getSize()) );
	}
	else {
		currentPatternLbl->setText( "N/A" );
	}

}




/// This method is called from another thread (audio engine)
void AudioEngineInfoForm::stateChanged(int state) {
	H2TextEvent *ev = new H2TextEvent("stateChanged");
	QApplication::postEvent(this, ev);
}




/**
 * This method is called from another thread (audio engine)
 */
void AudioEngineInfoForm::patternChanged() {
	H2TextEvent *ev = new H2TextEvent("patternChanged");
	QApplication::postEvent(this, ev);
}




/**
 *
 */
void AudioEngineInfoForm::customEvent( QCustomEvent *ev ) {
	if ( ev->type() != H2_TEXT_EVENT ) {	// Must be a H2TextEvent
		return;
	}
	QString message = ((H2TextEvent*)ev)->getText();

	if (message == QString( "stateChanged" ) ) {
		updateAudioEngineState();
	}
	else if (message == QString( "patternChanged" ) ) {
		updateAudioEngineState();
	}

}





/**
 * Update engineStateLbl with the current audio engine state
 */
void AudioEngineInfoForm::updateAudioEngineState() {
	// Audio Engine state
	QString stateTxt;
	int state = (Hydrogen::getInstance())->getState();
	switch (state) {
	case UNINITIALIZED:
		stateTxt = "Uninitialized";
		break;

	case INITIALIZED:
		stateTxt = "Initialized";
		break;

	case PREPARED:
		stateTxt = "Prepared";
		break;

	case READY:
		stateTxt = "Ready";
		break;

	case PLAYING:
		stateTxt = "Playing";
		break;

	default:
		stateTxt = "Unknown!?";
		break;
	}
	engineStateLbl->setText(stateTxt);
}



