/* This file is part of MemMXtest
 * Copyright (C) 1999, 2000  J.A. Bezemer
 * This program is licensed under the GNU General Public License,
 * see the top-level README file for details.
 */

/* These are the functions that set cache and refresh modes at the beginning
   of each pass. Change them if you want some other behaviour.

   Current situation:

   Pass  Cache  Refresh
   0     Half   BIOS default (usually 15 ms)
   1     On     BIOS default (usually 15 ms)
   2     Half   Extended (150 ms)
   3     On     Extended (150 ms)
   4     Half   Normal (15 ms)
   5     On     Normal (15 ms)
   6 see 2
 */

#include <asm/io.h>
#include "defines.h"
#include "mtest.h"

/* Inline function definitions */
#define SET_REFRESH(low,high)						\
	outb (0x74, 0x43);	/* 0111 0100				\
					   +- 16-bits binary counter	\
					+++-- rate generator		\
				     ++------ r/w bits 0-7 followed by 8-15  \
				   ++-------- select counter #1 (refresh) */ \
									\
	outb ((low), 0x41);	/* Write bits 0-7 */			\
	outb ((high), 0x41)	/* Write bits 8-15 */

/* You can define other refresh rates here */
#define SET_REFRESH_15MS   SET_REFRESH(0x12, 0x00)	/* 18 / 1.2 = 15ms */
#define SET_REFRESH_150MS  SET_REFRESH(0xb4, 0x00)	/* 180 / 1.2 = 150ms */
#define SET_REFRESH_500MS  SET_REFRESH(0x58, 0x02)	/* 600 / 1.2 = 500ms */


/* Set cache mode at the beginning of each pass */
void
set_cache ()
{
  switch (cache_mode)
    {
    case 0:
      /* Alternate cache half/on */
      switch (pass % 2)
	{
	case 0:
	  sub_cachemode = P_CACHEMODE_HALF;
	  cprint ("Cache: Half");
	  println ();
	  break;
	case 1:
	  sub_cachemode = P_CACHEMODE_ON;
	  cprint ("Cache: On");
	  println ();
	  break;
	default:
	  /* Not used */
	  sub_cachemode = P_CACHEMODE_OFF;
	  cprint ("Cache: Off");
	  println ();
	}
      break;

    case 1:
      sub_cachemode = P_CACHEMODE_ON;
      cprint ("Cache: OnForced");
      println ();
      break;
    case 2:
      sub_cachemode = P_CACHEMODE_HALF;
      cprint ("Cache: HalfForced");
      println ();
      break;
    case 3:
      sub_cachemode = P_CACHEMODE_OFF;
      cprint ("Cache: OffForced");
      println ();
      break;
    }
}

/* Set refresh rate at the beginning of each pass */
void
set_refresh ()
{
  switch (refresh_mode)
    {
    case 0:
      if (pass < 2)
	{
	  /* Keep original refresh rate. This is (or could be) set
	     in the BIOS setup menu, so we don't know what it is. */
	  cprint ("Refresh: BIOSDefault (unknown)");
	  println ();
	  return;
	}

      /* Starting from the 2nd pass, alternate every 2 passes between short
         and long refresh rates */
      if ((pass % 4) >= 2)
	{
	  /* Set refresh to 150ms */
	  SET_REFRESH_150MS;
	  cprint ("Refresh: Extended (150ms)");
	  println ();
	}
      else
	{
	  /* Set refresh to 15ms */
	  SET_REFRESH_15MS;
	  cprint ("Refresh: Normal (15ms)");
	  println ();
	}
      break;

    case 1:
      /* Set refresh to 15ms */
      SET_REFRESH_15MS;
      cprint ("Refresh: NormalForced (15ms)");
      println ();
      break;
    case 2:
      /* Set refresh to 150ms */
      SET_REFRESH_150MS;
      cprint ("Refresh: ExtendedForced (150ms)");
      println ();
      break;
    case 3:
      /* Set refresh to 500ms */
      /* If the chips don't get errors here, they're really good. */
      SET_REFRESH_500MS;
      cprint ("Refresh: XLongForced (500ms)");
      println ();
      break;
    default:
      /* KeepThis */
      cprint ("Refresh: Unchanged (unknown)");
      println ();
    }
}
