// ExTransport.h

/* Copyright (C) 2000-2001 Hewlett-Packard Company
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 */

/* Original author: David Paschal */

#ifndef EXTRANSPORT_H
#define EXTRANSPORT_H

#include <bp/ex/ExMgr.h>
#ifndef EX_TRANSPORT_UNIX_PORT
#include <vxWorks.h>
#include <util/Queue.h>
#include <errno.h>
#include <bp/ex/service/ExService.h>
#include <bp/ex/ExPhysicalPort.h>
#include <bp/ex/ExMsg.h>
#include <bp/ex/ExBdr.h>
#include <bp/ex/ExWatchdogTimer.h>
#include <bp/ex/ExTknob.h>
#include <nettest/nettest.h>
#endif

/* Macros to access 16-bit words stored in big-endian in a character array. */
#define BEND_WORD_MSB(x)    (((unsigned char *)(x))[0])
#define BEND_WORD_LSB(x)    (((unsigned char *)(x))[1])
#define BEND_GET_WORD(x)    ((BEND_WORD_MSB(x)<<8)|(BEND_WORD_LSB(x)))
#define BEND_SET_WORD(x,a)  (BEND_WORD_MSB(x)=((a)>>8),BEND_WORD_LSB(x)=(a))

#define SETTHIS(x) (this->x=x)

/* Optimized for 1284.4 header "control" field values. */
#define EX_TRANSPORT_HEADER_FLAG_OUT_OF_BAND	0x01
#define EX_TRANSPORT_HEADER_FLAG_END_OF_MESSAGE	0x02

class ExTransport;
typedef class ExTransportChannel *TCD;

class ExLookup: public QueueEntry {
    public:
	static const int MAX_SERVICE_NAME_LEN=40;
	static const int MAX_SERVICE_NAME_ALLOC=(MAX_SERVICE_NAME_LEN+1);

	static const int LAST_SET_NOTHING=0;
	static const int LAST_SET_SERVICE_NAME=1;
	static const int LAST_SET_SOCKET_ID=2;

    protected:
	ExService *pService;
	char serviceName[MAX_SERVICE_NAME_ALLOC];
	int socketID;
	int lastSet;
	int status;

    public:
	// Constructor, destructor, dump:
	ExLookup(ExService *pService);
	~ExLookup(void);
#ifdef JD_DEBUGLITE
	void dump(void);
#endif

	ExService *getService(void) {
		/* The caller panics if the service pointer is null. */
		return pService;
	}
	void setService(ExService *pService) {
		this->pService=pService;
	}

	int getServiceName(char *dest,int maxlen);
	char *getServiceName(void) {
		return serviceName;
	}
	int setServiceName(char *src,int len=0,int preserveLookupType=0);

	int getSocketID(void) {
		return socketID;
	}
	void setSocketID(int src,int preserveLookupType=0) {
		socketID=src;
		if (!preserveLookupType) lastSet=LAST_SET_SOCKET_ID;
	}

	int getLastSet(void) {
		return lastSet;
	}

	int getStatus(void) {
		return status;
	}
	void setStatus(int src) {
		status=src;
	}
};

class ExLookupQueue: public Queue {
    public:
	ExLookupQueue(void): Queue("ExLookupQueue") { }
	virtual ~ExLookupQueue(void) { }

	void add(ExLookup *pLookupRequest) {
		base_add(pLookupRequest);
	}
	ExLookup *peek(void) {
		return (ExLookup *)base_peek();
	}
	ExLookup *pop(void) {
		return (ExLookup *)base_pop();
	}
};

class ExCounter {
    protected:
	int count;

    public:
	int *pointToCount(void) { return &count; }

    public:
	ExCounter(void) { reset(); }
	int get(void) { return count; }
	void reset(void) { count=0; }
	void increment(void) { if (count!=-1) count++; }
};

class ExDebugCounter: public ExCounter {
    public:
	void increment(void) {
#ifdef JD_DEBUGLITE
		ExCounter::increment();
#else
		count=1;
#endif
	}
};

class ExCreditCounter {
    protected:
	int current;
	int initial;
	int low;
	int high;
	int max;

    public:
	int *pointToCurrent(void) { return &current; }

    public:
	ExCreditCounter(int initial=0) { resetMax(initial); }

	int isValid(int value) { return (value>=0 && value<=max); }

	int isEmpty(void) { return (!current); }
	int get(void) { return current; }
	int getInitial(void) { return initial; }
	int getLow(void) { return low; }
	int getHigh(void) { return high; }
	int getMax(void) { return max; }

	void setMax(int newval) { max=newval; }
	void preset(int newval) { initial=low=high=newval; }
	void reset(void) { reset(initial); }
	void reset(int newval) { preset(newval); current=newval; }
	void resetMax(int newval) { reset(newval); setMax(newval); }
	STATUS grab(int delta=1) {
		if (max<=0) return OK;
		return set(current-delta);
	}
	STATUS increment(int delta=1) {
		if (max<=0) return OK;
		return set(current+delta);
	}

	int set(int newval) {
		if (!isValid(newval)) return ERROR;
		current=newval;
#ifdef JD_DEBUGLITE
		if (current<low) low=current;
		if (current>high) high=current;
#endif
		return OK;
	}

#ifdef JD_DEBUGLITE
	void dump(void) {
		printf("current=%d, initial=%d, low=%d, high=%d, max=%d\n",
			current,initial,low,high,max);
	}
#endif
};

class ExTransportChannel: public ExMsgHandler {
    public:	// const
	static const int FLAG_ALLOCATED=1;
	static const int FLAG_OPEN=2;
	static const int FLAG_DEACTIVATING=4;

    protected:	// const
	static const int openSetFlags=FLAG_ALLOCATED|FLAG_OPEN;
	static const int openClearFlags=0;
	static const int closedSetFlags=FLAG_ALLOCATED;
	static const int closedClearFlags=FLAG_OPEN;

	enum {
		RESET_STARTUP,
		RESET_ACTIVATE,
		RESET_OPEN,
		RESET_CLOSE,
		RESET_START_DEACTIVATE,
		RESET_FINISH_DEACTIVATE
	};

	enum {
		MSG_EX_TRANSPORT_CHANNEL_LAST=0
	};

    protected:
	// Set only by the constructor:
	int port;
	int channel;
	int localSocket;
	ExTransport *pTransport;
	ExMgr *pMgr;
	ExPhysicalPort *pPhysicalPort;
	int openingAnySetFlags;
	int closingAnySetFlags;

	// Initialized at activation, possibly set elsewhere:
	int flags;
	ExDebugCounter countOpen;
	ExDebugCounter countOpenFailure;
	ExService *pService;
	SCD scd;
	int forwardDataPriority;
	int minBuffersRequired;
	int benefitOfMoreBuffers;
	int reverseDataBufferSize;
	int bufferCount;
	// Set by setRemoteSocket():
	int remoteSocket;

	// Initialized by open*():
	int maxForwardDatalen;
	int maxReverseDatalen;
	ExDebugCounter countForwardData;
	ExDebugCounter countReverseData;
	ExDebugCounter countReverseBuffers;
	ExDebugCounter countReverseBufferReturns;
#ifdef JD_DEBUGLITE
	ExBdr *lastReverseBuffer;
#endif

	// Used by forwardDataAvailable():
	ExBdr *currentForwardBuffer;
	int currentGrabbedCredit;
	int currentGrabbedTransaction;

	// Constructor, destructor, reset, dump, handleMsg:
    public:
	ExTransportChannel(ExTransport *pTransport,int channel,int localSocket,
		ExMgr *pMgr,ExPhysicalPort *pPhysicalPort,int final=1);
	virtual ~ExTransportChannel(void);
    protected:
	virtual void reset(int rtype);
	virtual void registerStuff(char *title);
    public:
#ifdef JD_DEBUGLITE
	virtual void dump(void);
#endif
    protected:
	virtual void handleMsg(ExMsg *pMsg);

	// Get/set flags:
    protected:
	int getFlags(void) {
		return flags;
	}
	int testFlagsSetClear(int set,int clear) {
		return ((flags&(set|clear))==set);
	}
	void setFlags(int value) {
		flags=value;
	}
	void setClearFlags(int set,int clear) {
		setFlags((flags|set)&(~clear));
	}

	// Allocate/deallocate channel:
    public:
	int isAllocated(void) {
		return testFlagsSetClear(FLAG_ALLOCATED,0);
	}
	virtual void allocate(ExService *pService,SCD scd,
	    int forwardDataPriority,
	    int minBuffersRequired,int benefitOfMoreBuffers,
	    int reverseDataBufferSize);
	void deactivate(void) { reset(RESET_START_DEACTIVATE); }
	void deallocate(void) { reset(RESET_FINISH_DEACTIVATE); }

	// Routines for dividing up buffers:
    public:
	int getMinBuffersRequired(void) {
		return minBuffersRequired;
	}
	int getBenefitOfMoreBuffers(void) {
		return benefitOfMoreBuffers;
	}
	virtual void grantBuffers(int count);
	int getBufferCount(void) {
		return bufferCount;
	}

	// Get/set local/remote socket:
    public:
	int getLocalSocket(void) { return localSocket; }
	int getRemoteSocket(void) { return remoteSocket; }
	void setRemoteSocket(int remoteSocket,
	    int maxForwardDatalen,int maxReverseDatalen);
    protected:
	virtual void setRemoteSocket_ts(int remoteSocket,
	    int maxForwardDatalen,int maxReverseDatalen);
    public:
	void sendRemoteSocketSetResponse(int status);

	// Open:
    public:
	int isOpening(void) {
		return (isClosed() && (getFlags()&openingAnySetFlags));
	}
	int isOpen(void) {
		return testFlagsSetClear(openSetFlags,openClearFlags);
	}
	void open(int maxForwardDatalen,int maxReverseDatalen);
    protected:
	virtual void open_ts(int maxForwardDatalen,int maxReverseDatalen);
	void setOpen(void) {
		setClearFlags(FLAG_OPEN,0);
	}
	void sendChannelOpenResponse(int status);

	// Close:
    public:
	int isClosing(void) {
		return (isOpen() && (getFlags()&closingAnySetFlags));
	}
	int isClosed(void) {
		return testFlagsSetClear(closedSetFlags,closedClearFlags);
	}
	void close(void);
    protected:
	virtual void close_ts(void);
	void setClosed(void) {
		reset(RESET_CLOSE);
	}
	void sendChannelCloseResponse(int status);

	// BDR management for forward/reverse data flow:
    protected:
	void markBDR(ExBdr *pBdr,ExCounter *pCounter=0);
	static void measureBDR(ExBdr *pBdr,int *pDatalen,int *pBdrCount);

	// Forward data flow:
    public:
	int getMaxForwardDatalen(void) { return maxForwardDatalen; }
	int getForwardDataPriority(void) { return forwardDataPriority; }
	void setForwardDataPriority(int priority) {
		forwardDataPriority=priority;
	}
	void forwardDataAvailable(void);
	virtual void noMoreForwardData(void);
    protected:
	virtual ExBdr *servicePullForwardData(void);
	virtual int grabForwardCredit(ExBdr *pBdr);
	virtual STATUS addOverhead(ExBdr **ppBdr);
    public:
	void forwardDataResponse(ExBdr *pBdr,int status);
    protected:
	virtual void removeOverhead(ExBdr **ppBdr);
	virtual void serviceForwardDataResponse(ExBdr *pBdr,int status);

	// Reverse data flow:
    public:
	int getMaxReverseDatalen(void) { return maxReverseDatalen; }
	void reverseDataReceived(ExBdr *pBdr,int status) {
		countReverseData.increment();
		reverseDataReceived_ts(pBdr,status);
	}
	virtual void reverseDataReceived_ts(ExBdr *pBdr,int status);
    protected:
	virtual void serviceReverseDataReceived(ExBdr *pBdr,int status,
	    unsigned char *data,int datalen);
    public:
	virtual void returnBufferNotification(ExBdr *pBdr);
};

class ExTransport: public ExMsgHandler {
    public:	// const
	static const int GRC_FLAG_END_OF_TRANSACTION=1;
	static const int GRC_FLAG_RECEIVE_DATA_BEFORE_NEXT_REQUEST=2;
	static const int GRC_FLAG_PARTIAL_DATA_OK=4;	// Also excess data OK.
	static const int GRC_FLAG_APPEND_TO_PREVIOUS_BUFFER=8;
	static const int GRC_FLAG_ALIGN_NEXT_DATA=16;

    protected:	// const
	static const int DEFAULT_FORWARD_DATA_TIMEOUT=WAIT_FOREVER;

	enum {
		RESET_STARTUP,
		RESET_ACTIVATE,
		RESET_START_DEACTIVATE,
		RESET_FINISH_DEACTIVATE
	};

	enum {
		MSG_FORWARD_DATA_TIMEOUT=0,
		MSG_EX_TRANSPORT_LAST
	};

	enum {
		REASON_UNUSED=EX_CLOSE_REASON_TRANSPORT_BASE,
		REASON_FORWARD_DATA_TIMEOUT,
		REASON_EX_TRANSPORT_LAST
	};

    protected:
	// Initialized by the constructor:
	int port;
	ExMgr *pMgr;
	ExPhysicalPort *pPhysicalPort;
	ExCreditCounter forwardTransactionCounter;   // Also reset elsewhere.
	int channelCount;
	ExTransportChannel **channelArray;
	int overheadBufferCount;
#ifdef JD_DEBUGLITE
	ExTransport *pNextTransport;
#endif

	ExCountingWatchdogTimer *pForwardDataTimer;
	ExMsg *pForwardDataTimeoutMsg;

	// Initialized at activation:
	int reverseDataBufferCount;
	int reverseDataBufferSize;
	int maxForwardBdrsPerTransaction;	/* TODO: Use this properly! */
	int nextChannelToAllocate;

	// Set at runtime:
	int nextBlockedChannel;
	ExLookupQueue lookupQueue;

	// Constructor, destructor, reset, dump/debug, handleMsg:
    public:
	ExTransport(ExMgr *pMgr,ExPhysicalPort *pPhysicalPort,
	    int forwardTransactionCount,int channelCount,int final=1);
    protected:
	virtual void initChannelArray(void);
	virtual ExTransportChannel *newChannel(int channel);
    public:
	virtual ~ExTransport(void);
    protected:
	virtual void reset(int rtype);
	virtual void registerStuff(char *title);
    public:
#ifdef JD_DEBUGLITE
	virtual void dump(void);
	void dumpall(void);
	ExTransportChannel *locateTransportChannel(int channel);
	ExTransport *getNextTransport(void) { return pNextTransport; }
#endif
	virtual void handleMsg(ExMsg *pMsg);

	// Activate:
    public:
	int getOverheadBufferCount(void) { return overheadBufferCount; }
	void activate(int reverseDataBufferCount,int reverseDataBufferSize,
	    int maxForwardBdrsPerTransaction);
	void transportResetComplete(void);
    protected:
	virtual void transportResetComplete_ts(void);
	void sendWaitingForInitialRequests(/*int status*/);
    public:
	TCD allocateChannel(ExService *pService,SCD scd,
	    int forwardDataPriority,
	    int minBuffersRequired,int benefitOfMoreBuffers);
	void initialRequestsComplete(void);
    protected:
	void sendActivateResponse(int status);

	// Deactivate:
    public:
	void deactivate(int reason);
	void physicalPortNotActive(void);
    protected:
	void sendDeactivateResponse(int status);

	// Remote socket lookup:
    public:
	void lookupRemoteSocket(ExLookup *pLookupRequest);
    protected:
	void lookupNextRemoteSocket(void);
	virtual void lookupRemoteSocket_ts(ExLookup *pLookupRequest);
	void flushLookupQueue(void);
	void sendRemoteSocketLookupResponse(ExLookup *pLookupRequest);

	// Forward data flow:
    protected:
	int getBlockedChannel(void) {
		return nextBlockedChannel;
	}
    public:
	void setBlockedChannel(int channel) {
		if (nextBlockedChannel==ERROR) nextBlockedChannel=channel;
	}
    protected:
	int clearBlockedChannel(void) {
		int channel=nextBlockedChannel;
		nextBlockedChannel=ERROR;
		return channel;
	}
    public:
	STATUS grabForwardTransaction(void) {
		int r=forwardTransactionCounter.grab();
		if (r!=ERROR) pForwardDataTimer->start();
		return r;
	}
	void returnForwardTransaction(void);

	// Reverse data flow:
    public:
	virtual void resetReverseCount(void);
	virtual STATUS getReverseCount(char *lastData,int lastLength,
	    int *pCount,int *pFlags);
	void reverseDataReceived(ExBdr *pBdr,int status);
	static void unmarkBDR(ExBdr *pBdr);
    protected:
	virtual ExTransportChannel *lookupChannel(ExBdr *pBdr);
	void returnReverseBuffer(ExBdr *pBdr);
    public:
	void returnBufferNotification(ExBdr *pBdr);

#ifndef EX_TRANSPORT_UNIX_PORT
	void deallocateChannel(TCD tcd) {
		tcd->deallocate();
	}
	void setRemoteSocket(TCD tcd,int remoteSocket,
	    int maxForwardDatalen,int maxReverseDatalen) {
		tcd->setRemoteSocket(remoteSocket,
			maxForwardDatalen,maxReverseDatalen);
	}
	void openChannel(TCD tcd,int maxForwardDatalen,int maxReverseDatalen) {
		tcd->open(maxForwardDatalen,maxReverseDatalen);
	}
	void closeChannel(TCD tcd) {
		tcd->close();
	}
	void forwardDataAvailable(TCD tcd) {
		tcd->forwardDataAvailable();
	}
	void forwardDataResponse(ExBdr *pBdr,int status) {
		pBdr->getTCD()->forwardDataResponse(pBdr,status);
	}
	void setForwardDataPriority(TCD tcd,int priority) {
		tcd->setForwardDataPriority(priority);
	}
#endif
};

#endif
