// ExTransport.cpp

/* Copyright (C) 2000-2001 Hewlett-Packard Company
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 */

/* Original author: David Paschal */

#include <string.h>

#include <stdio.h>
#ifndef EX_TRANSPORT_UNIX_PORT
#include <bp/ex/ExFileNum.h>
#define cFileNum CFILENUM(cModExio,cFileNum_transport_ExTransport)
#include <lm/LoggingManager.h>
#include <lm/lmc.h>
#endif

#include <ExTransport.h>

/*****************************************************************************\
|* class ExLookup
\*****************************************************************************/

/*---------------------------------------------------------------------------*\
|* static const member definitions:
\*---------------------------------------------------------------------------*/

const int ExLookup::MAX_SERVICE_NAME_LEN;
const int ExLookup::MAX_SERVICE_NAME_ALLOC;
const int ExLookup::LAST_SET_NOTHING;
const int ExLookup::LAST_SET_SERVICE_NAME;
const int ExLookup::LAST_SET_SOCKET_ID;

/*---------------------------------------------------------------------------*\
|* Constructor, destructor, dump:
\*---------------------------------------------------------------------------*/

ExLookup::ExLookup(ExService *pService) {
	int i;

	LOG_ENTRY(cEXTP,0,"ExLookup constructor called, "
		"this=0x%8.8X, pService=0x%8.8X.\n",this,pService);

	SETTHIS(pService);
	for (i=0;i<MAX_SERVICE_NAME_ALLOC;i++) serviceName[i]=0;
	socketID=0;
	lastSet=LAST_SET_NOTHING;
	status=ERROR;
}

ExLookup::~ExLookup(void) {
	LOG_ENTRY(cEXTP,0,"ExLookup destructor called, this=0x%8.8X.\n",
		this);
}

#ifdef JD_DEBUGLITE
void ExLookup::dump(void) {
	printf("sizeof(ExLookup)=%d, this=0x%8.8X\n",
		sizeof(ExLookup),(int)this);
	printf("pService=0x%8.8X\n",
		(int)pService);
	printf("serviceName='%s' (len=%d)\n",
		serviceName,strlen(serviceName));
	printf("socketID=%d\n",
		socketID);
	printf("lastSet=%d\n",
		lastSet);
	printf("status=%d\n",
		status);
}
void ExLookup__dump(ExLookup *_this) {
	return _this->dump();
}
#endif

/*---------------------------------------------------------------------------*\
|* Get/set service name:
\*---------------------------------------------------------------------------*/

int ExLookup::getServiceName(char *dest,int maxlen) {
	LOG_ASSERT(dest && maxlen>0,cEXTP,0,cCauseBadParm,
		"ExLookup::getServiceName: "
		"invalid dest=0x%8.8X or maxlen=%d!\n",
		dest,maxlen);

	dest[maxlen-1]=0;
	strncpy(dest,serviceName,maxlen);
	if (dest[maxlen-1]) {
		dest[maxlen-1]=0;
		LOG_ERROR(cEXTP,0,cCauseNoMem,
			"ExLookup::getServiceName: truncated string!\n");
		return maxlen;
	}

	return strlen(dest);
}

int ExLookup::setServiceName(char *src,int len,int preserveLookupType) {
	LOG_ASSERT(src,cEXTP,0,cCauseBadParm,
		"ExLookup::setServiceName: invalid src=0x%8.8X!\n",src);

	if (!len) len=strlen(src);
	if (len>MAX_SERVICE_NAME_LEN) {
		LOG_ERROR(cEXTP,0,cCauseNoMem,
			"ExLookup::setServiceName: truncated string!\n");
		len=MAX_SERVICE_NAME_LEN;
	}

	memcpy(serviceName,src,len);
	serviceName[len]=0;

	if (!preserveLookupType) lastSet=LAST_SET_SERVICE_NAME;

	return len;
}

/*****************************************************************************\
|* class ExTransportChannel
\*****************************************************************************/

/*---------------------------------------------------------------------------*\
|* static const member definitions:
\*---------------------------------------------------------------------------*/

const int ExTransportChannel::FLAG_ALLOCATED;
const int ExTransportChannel::FLAG_OPEN;
const int ExTransportChannel::FLAG_DEACTIVATING;
const int ExTransportChannel::openSetFlags;
const int ExTransportChannel::openClearFlags;
const int ExTransportChannel::closedSetFlags;
const int ExTransportChannel::closedClearFlags;

/*---------------------------------------------------------------------------*\
|* Constructor, destructor, dump, reset, handleMsg:
\*---------------------------------------------------------------------------*/

ExTransportChannel::ExTransportChannel(ExTransport *pTransport,int channel,
    int localSocket,ExMgr *pMgr,ExPhysicalPort *pPhysicalPort,int final=1) {
	port=pMgr->getPortNumber();

	SETTHIS(channel);
	SETTHIS(localSocket);
	SETTHIS(pTransport);
	SETTHIS(pMgr);
	SETTHIS(pPhysicalPort);

	openingAnySetFlags=0;
	closingAnySetFlags=FLAG_DEACTIVATING;

	if (final) {
		reset(RESET_STARTUP);
		registerStuff("ExTransportChannel");
	}
}

ExTransportChannel::~ExTransportChannel(void) {
	/* Does nothing. */
}

void ExTransportChannel::reset(int rtype) {
	if (rtype==RESET_START_DEACTIVATE) {
		setClearFlags(FLAG_DEACTIVATING,0);
	} else {
		setClearFlags(0,~FLAG_ALLOCATED);
		if (rtype!=RESET_OPEN && rtype!=RESET_CLOSE) {
			setFlags(0);
		}
	}

	if (rtype!=RESET_OPEN &&
	    rtype!=RESET_CLOSE &&
	    rtype!=RESET_START_DEACTIVATE &&
	    rtype!=RESET_FINISH_DEACTIVATE) {
		countOpen.reset();
		countOpenFailure.reset();
		pService=0;
		scd=0;
		forwardDataPriority=0;
		minBuffersRequired=0;
		benefitOfMoreBuffers=0;
		reverseDataBufferSize=0;
		bufferCount=0;

		remoteSocket=ERROR;
	}

	if (rtype!=RESET_CLOSE &&
	    rtype!=RESET_START_DEACTIVATE &&
	    rtype!=RESET_FINISH_DEACTIVATE) {
		maxForwardDatalen=0;
		maxReverseDatalen=0;
		countForwardData.reset();
		countReverseData.reset();
		if (rtype!=RESET_OPEN) {
			countReverseBuffers.reset();
			countReverseBufferReturns.reset();
#ifdef JD_DEBUGLITE
			lastReverseBuffer=0;
#endif
		}
	}

	if (currentForwardBuffer &&
	    (rtype==RESET_CLOSE ||
	     rtype==RESET_START_DEACTIVATE ||
	     rtype==RESET_FINISH_DEACTIVATE)) {
		serviceForwardDataResponse(currentForwardBuffer,
			S_hlio_BP_CHAN_CLOSING);
	}
	currentForwardBuffer=0;
	currentGrabbedCredit=ERROR;
	currentGrabbedTransaction=ERROR;
}

void ExTransportChannel::registerStuff(char *title) {
	nettestRegister(NETTEST_TYPE_INT32,
		&localSocket,
		"%s[%d,%d].localSocket",
		title,port,channel);
	nettestRegister(NETTEST_TYPE_INT32,
		&flags,
		"%s[%d,%d].flags",
		title,port,channel);
	nettestRegister(NETTEST_TYPE_INT32,
		countOpen.pointToCount(),
		"%s[%d,%d].countOpen",
		title,port,channel);
	nettestRegister(NETTEST_TYPE_INT32,
		countOpenFailure.pointToCount(),
		"%s[%d,%d].countOpenFailure",
		title,port,channel);
#if 0
	nettestRegister(NETTEST_TYPE_INT32,
		&forwardDataPriority,
		"%s[%d,%d].forwardDataPriority",
		title,port,channel);
	nettestRegister(NETTEST_TYPE_INT32,
		&minBuffersRequired,
		"%s[%d,%d].minBuffersRequired",
		title,port,channel);
	nettestRegister(NETTEST_TYPE_INT32,
		&benefitOfMoreBuffers,
		"%s[%d,%d].benefitOfMoreBuffers",
		title,port,channel);
	nettestRegister(NETTEST_TYPE_INT32,
		&reverseDataBufferSize,
		"%s[%d,%d].reverseDataBufferSize",
		title,port,channel);
#endif
	nettestRegister(NETTEST_TYPE_INT32,
		&bufferCount,
		"%s[%d,%d].bufferCount",
		title,port,channel);
	nettestRegister(NETTEST_TYPE_INT32,
		&remoteSocket,
		"%s[%d,%d].remoteSocket",
		title,port,channel);
	nettestRegister(NETTEST_TYPE_INT32,
		&maxForwardDatalen,
		"%s[%d,%d].maxForwardDatalen",
		title,port,channel);
	nettestRegister(NETTEST_TYPE_INT32,
		&maxReverseDatalen,
		"%s[%d,%d].maxReverseDatalen",
		title,port,channel);
	nettestRegister(NETTEST_TYPE_INT32,
		countForwardData.pointToCount(),
		"%s[%d,%d].countForwardData",
		title,port,channel);
	nettestRegister(NETTEST_TYPE_INT32,
		countReverseData.pointToCount(),
		"%s[%d,%d].countReverseData",
		title,port,channel);
	nettestRegister(NETTEST_TYPE_INT32,
		countReverseBuffers.pointToCount(),
		"%s[%d,%d].countReverseBuffers",
		title,port,channel);
	nettestRegister(NETTEST_TYPE_INT32,
		countReverseBufferReturns.pointToCount(),
		"%s[%d,%d].countReverseBufferReturns",
		title,port,channel);
#ifdef JD_DEBUGLITE
	nettestRegister(NETTEST_TYPE_INT32,
		&lastReverseBuffer,
		"%s[%d,%d].lastReverseBuffer",
		title,port,channel);
#endif
	nettestRegister(NETTEST_TYPE_INT32,
		&currentForwardBuffer,
		"%s[%d,%d].currentForwardBuffer",
		title,port,channel);
	nettestRegister(NETTEST_TYPE_INT32,
		&currentGrabbedCredit,
		"%s[%d,%d].currentGrabbedCredit",
		title,port,channel);
	nettestRegister(NETTEST_TYPE_INT32,
		&currentGrabbedTransaction,
		"%s[%d,%d].currentGrabbedTransaction",
		title,port,channel);
}

#ifdef JD_DEBUGLITE
void ExTransportChannel::dump(void) {
	printf("this=0x%8.8X\n",
		(int)this);
	printf("port=%d\n",
		port);
	printf("channel=%d\n",
		channel);
	printf("localSocket=%d\n",
		localSocket);
	printf("pTransport=0x%8.8X\n",
		(int)pTransport);
	printf("pMgr=0x%8.8X\n",
		(int)pMgr);
	printf("pPhysicalPort=0x%8.8X\n",
		(int)pPhysicalPort);
	printf("openingAnySetFlags=0x%4.4X\n",
		openingAnySetFlags);
	printf("closingAnySetFlags=0x%4.4X\n",
		closingAnySetFlags);
	printf("flags=0x%4.4X\n",
		flags);
	printf("countOpen=%d\n",
		countOpen.get());
	printf("countOpenFailure=%d\n",
		countOpenFailure.get());
	printf("pService=0x%8.8X\n",
		(int)pService);
	printf("scd=0x%8.8X\n",
		(int)scd);
	printf("forwardDataPriority=%d\n",
		forwardDataPriority);
	printf("minBuffersRequired=%d\n",
		minBuffersRequired);
	printf("benefitOfMoreBuffers=%d\n",
		benefitOfMoreBuffers);
	printf("reverseDataBufferSize=%d\n",
		reverseDataBufferSize);
	printf("bufferCount=%d\n",
		bufferCount);
	printf("remoteSocket=%d\n",
		remoteSocket);
	printf("maxForwardDatalen=%d\n",
		maxForwardDatalen);
	printf("maxReverseDatalen=%d\n",
		maxReverseDatalen);
	printf("countForwardData=%d\n",
		countForwardData.get());
	printf("countReverseData=%d\n",
		countReverseData.get());
	printf("countReverseBuffers=%d\n",
		countReverseBuffers.get());
	printf("countReverseBufferReturns=%d\n",
		countReverseBufferReturns.get());
#ifdef JD_DEBUGLITE	// Redundant ifdef, I know.
	printf("lastReverseBuffer=0x%8.8X\n",
		(int)lastReverseBuffer);
#endif
	printf("currentForwardBuffer=0x%8.8X\n",
		(int)currentForwardBuffer);
	printf("currentGrabbedCredit=%d\n",
		currentGrabbedCredit);
	printf("currentGrabbedTransaction=%d\n",
		currentGrabbedTransaction);
}
#endif

void ExTransportChannel::handleMsg(ExMsg *pMsg) {
	LOG_ERROR_FATAL(cEXTP,0,cCauseBadParm,"");
}

/*---------------------------------------------------------------------------*\
|* Allocate/deallocate channel:
\*---------------------------------------------------------------------------*/

void ExTransportChannel::allocate(ExService *pService,SCD scd,
    int forwardDataPriority,int minBuffersRequired,int benefitOfMoreBuffers,
    int reverseDataBufferSize) {
	LOG_ENTRY(cEXTP,0,
		"allocate(port=%d,channel=%d,pService=0x%8.8X,scd=0x%8.8X,"
		"forwardDataPriority=%d,minBuffersRequired=%d,"
		"benefitOfMoreBuffers=%d,reverseDataBufferSize=%d): "
		"tcd=0x%8.8X.\n",
		port,channel,pService,scd,
		forwardDataPriority,minBuffersRequired,
		benefitOfMoreBuffers,reverseDataBufferSize);

	LOGD_ASSERT(!isAllocated(),cEXTP,0,cCauseNoHandles,
		"allocate(port=%d,channel=%d): already allocated!\n",
		port,channel);

	reset(RESET_ACTIVATE);

	setFlags(FLAG_ALLOCATED);
	SETTHIS(pService);
	SETTHIS(scd);
	SETTHIS(forwardDataPriority);
	SETTHIS(minBuffersRequired);
	SETTHIS(benefitOfMoreBuffers);
	SETTHIS(reverseDataBufferSize);
}

void ExTransportChannel::grantBuffers(int count) {
	bufferCount+=count;
}

/*---------------------------------------------------------------------------*\
|* Set remote socket:
\*---------------------------------------------------------------------------*/

void ExTransportChannel::setRemoteSocket(int remoteSocket,
    int maxForwardDatalen,int maxReverseDatalen) {
	LOG_ENTRY(cEXTP,0,
		"setRemoteSocket(port=%d,channel=%d,remoteSocket=%d,"
		"maxForwardDatalen=%d,maxReverseDatalen=%d)\n",
		port,channel,remoteSocket,maxForwardDatalen,maxReverseDatalen);

	LOG_ASSERT(isClosed(),cEXTP,0,cCauseBadState,"");

	SETTHIS(remoteSocket);
	// There's no need to set these here:
	// SETTHIS(maxForwardDatalen);
	// SETTHIS(maxReverseDatalen);

	setRemoteSocket_ts(remoteSocket,maxForwardDatalen,maxReverseDatalen);
}

void ExTransportChannel::setRemoteSocket_ts(int remoteSocket,
    int maxForwardDatalen,int maxReverseDatalen) {
	sendRemoteSocketSetResponse(OK);
}

void ExTransportChannel::sendRemoteSocketSetResponse(int status) {
	LOGD_ASSERT(isClosed(),cEXTP,0,cCauseBadState,"");

	if (status!=OK) remoteSocket=ERROR;

	ExMsg *pMsg=pMgr->getFreeMsg();

	pMsg->setType(eEXMSG_REMOTE_SOCK_RESPONSE);
	pMsg->setParams((int)scd,status);
	pMsg->send(pService);
}

/*---------------------------------------------------------------------------*\
|* Open:
\*---------------------------------------------------------------------------*/

void ExTransportChannel::open(int maxForwardDatalen,int maxReverseDatalen) {
	LOG_ENTRY(cEXTP,0,"open(port=%d,channel=%d,"
		"maxForwardDatalen=%d,maxReverseDatalen=%d)\n",
		port,channel,maxForwardDatalen,maxReverseDatalen);

	LOG_ASSERT(isClosed(),cEXTP,0,cCauseBadState,"");
	LOG_ASSERT(getRemoteSocket()!=ERROR,cEXTP,0,cCauseBadState,"");

	reset(RESET_OPEN);

	open_ts(maxForwardDatalen,maxReverseDatalen);
}

void ExTransportChannel::open_ts(int maxForwardDatalen,int maxReverseDatalen) {
	SETTHIS(maxForwardDatalen);
	SETTHIS(maxReverseDatalen);

	setOpen();
	sendChannelOpenResponse(OK);
}

void ExTransportChannel::sendChannelOpenResponse(int status) {
	ExDebugCounter *pOpenCounter=&countOpen;
	if (status!=OK) pOpenCounter=&countOpenFailure;
	pOpenCounter->increment();

	ExMsg *pMsg=pMgr->getFreeMsg();

	pMsg->setType(eEXMSG_OPEN_CHAN_RESPONSE);
	pMsg->setParams((int)scd,maxForwardDatalen,maxReverseDatalen,status);
	pMsg->send(pService);
}

/*---------------------------------------------------------------------------*\
|* Close:
\*---------------------------------------------------------------------------*/

void ExTransportChannel::close(void) {
	LOG_ENTRY(cEXTP,0,"close(port=%d,channel=%d)\n",port,channel);

	LOG_ASSERT(isOpen(),cEXTP,0,cCauseBadState,"");

	close_ts();
}

void ExTransportChannel::close_ts(void) {
	setClosed();
	sendChannelCloseResponse(S_hlio_BP_CHAN_CLOSE_HAD_NO_EFFECT);
}

void ExTransportChannel::sendChannelCloseResponse(int status) {
	ExMsg *pMsg=pMgr->getFreeMsg();

	pMsg->setType(eEXMSG_CLOSE_CHAN_RESPONSE);
	pMsg->setParams((int)scd,status);
	pMsg->send(pService);
}

/*---------------------------------------------------------------------------*\
|* BDR management for forward/reverse data flow:
\*---------------------------------------------------------------------------*/

void ExTransportChannel::markBDR(ExBdr *pBdr,ExCounter *pCounter=0) {
	while (pBdr) {
#ifdef JD_DEBUGLITE
		if (pCounter) pCounter->increment();
#endif
		pBdr->setTransportNotification(pTransport);
		pBdr->setTCD(this);
		pBdr=pBdr->getNext();
	}
}

void ExTransportChannel::measureBDR(ExBdr *pBdr,int *pDatalen,int *pBdrCount) {
	*pDatalen=0;
	*pBdrCount=0;

	while (pBdr) {
		*pDatalen+=pBdr->getDataLength();
		*pBdrCount+=1;
		pBdr=pBdr->getNext();
	}
}

/*---------------------------------------------------------------------------*\
|* Forward data flow:
\*---------------------------------------------------------------------------*/

/* TODO: Verify we don't exceed maxForwardBdrsPerTransaction? */
void ExTransportChannel::forwardDataAvailable(void) {
	ExBdr *pBdr;
#ifdef JD_DEBUGLITE
	char *msg=0;
	int loopCount=0;
#endif

	if (!isOpen() || isClosing()) {
		LOG_INFO(cEXTP,0,
			"forwardDataAvailable(port=%d,channel=%d): "
			"not open.\n",port,channel);
		return;
	}

	while (42) {
		if (!currentForwardBuffer) {
			currentGrabbedCredit=ERROR;
			currentGrabbedTransaction=ERROR;

			/* TODO: If maxForwardDatalen is zero, then return
			 * buffer right back to service with an appropriate
			 * status code.  Make sure service handles a zero
			 * datalen appropriately! */
			currentForwardBuffer=servicePullForwardData();
			if (!currentForwardBuffer) {
				noMoreForwardData();
#ifdef JD_DEBUGLITE
				msg="no more data.";
#endif
				break;
			}
		}

#ifdef JD_DEBUGLITE
		loopCount++;
#endif

		if (currentGrabbedCredit==ERROR) {
			currentGrabbedCredit=
				grabForwardCredit(currentForwardBuffer);
			if (currentGrabbedCredit==ERROR) {
#ifdef JD_DEBUGLITE
				msg="out of credit!";
#endif
				break;
			}
		}

		if (currentGrabbedTransaction==ERROR) {
			currentGrabbedTransaction=
				pTransport->grabForwardTransaction();
			if (currentGrabbedTransaction==ERROR) {
				pTransport->setBlockedChannel(channel);
#ifdef JD_DEBUGLITE
				msg="out of forward transactions!";
#endif
				break;
			}
		}
		/* CAUTION -- Don't block the channel after this point,
		 * because grabForwardTransaction() starts a timer! */

		if (addOverhead(&currentForwardBuffer)==ERROR) {
			LOG_ERROR_FATAL(cEXTP,0,cCauseBadParm,
				"forwardDataAvailable(port=%d,channel=%d): "
				"addOverhead failed!\n",port,channel);
		}

		markBDR(currentForwardBuffer);	/* For forwardDataResponse() */

		LOG_ENTRY(cEXTP,0,"sendForwardData(port=%d,channel=%d,"
			"pBdr=0x%8.8X,priority=%d)\n",port,channel,
			currentForwardBuffer,forwardDataPriority);
		countForwardData.increment();

		pBdr=currentForwardBuffer;
		currentForwardBuffer=0;
		pPhysicalPort->sendForwardData(pBdr,forwardDataPriority);
	}

#ifdef JD_DEBUGLITE
	if (msg) {
		LOG_INFO(cEXTP,0,
			"forwardDataAvailable(port=%d,channel=%d): "
			"loopCount=%d, %s\n",
			port,channel,loopCount,msg);
	}
	/* TODO: Use loopCount to measure average efficiency? */
#endif
}

void ExTransportChannel::noMoreForwardData(void) {
	/* Does nothing. */
}

ExBdr *ExTransportChannel::servicePullForwardData(void) {
	return pService->pullForwardData(scd,maxForwardDatalen);
}

int ExTransportChannel::grabForwardCredit(ExBdr *pBdr) {
	return 0;
}

STATUS ExTransportChannel::addOverhead(ExBdr **ppBdr) {
	return OK;
}

void ExTransportChannel::forwardDataResponse(ExBdr *pBdr,int status) {
	LOG_ENTRY(cEXTP,0,"forwardDataResponse(port=%d,channel=%d,"
		"pBdr=0x%8.8X,status=%d)\n",port,channel,pBdr,status);

	removeOverhead(&pBdr);

	serviceForwardDataResponse(pBdr,status);

	pTransport->returnForwardTransaction();
}

void ExTransportChannel::removeOverhead(ExBdr **ppBdr) {
	/* Does nothing. */
}

void ExTransportChannel::serviceForwardDataResponse(ExBdr *pBdr,int status) {
	pService->forwardDataResponse(scd,pBdr,status);
}

/*---------------------------------------------------------------------------*\
|* Reverse data flow:
\*---------------------------------------------------------------------------*/

void ExTransportChannel::reverseDataReceived_ts(ExBdr *pBdr,int status) {
	/* We're passing zeroes for the last two parameters, since
	 * only the MLC/1284.4 command channel needs them. */
	serviceReverseDataReceived(pBdr,status,0,0);
}

void ExTransportChannel::serviceReverseDataReceived(ExBdr *pBdr,int status,
    unsigned char *data,int datalen) {
	/* TODO: Filter out empty packets, unless we decide not to later. */

#ifdef JD_DEBUGLITE
	lastReverseBuffer=pBdr;
#endif

	pService->reverseDataReceived(scd,pBdr,status);
}

void ExTransportChannel::returnBufferNotification(ExBdr *pBdr) {
	/* Does nothing.  The caller is responsible for actually returning
	 * the buffer to the physical port. */
}

/*****************************************************************************\
|* class ExTransport
\*****************************************************************************/

/*---------------------------------------------------------------------------*\
|* static const member definitions:
\*---------------------------------------------------------------------------*/

const int ExTransport::GRC_FLAG_END_OF_TRANSACTION;
const int ExTransport::GRC_FLAG_RECEIVE_DATA_BEFORE_NEXT_REQUEST;
const int ExTransport::GRC_FLAG_PARTIAL_DATA_OK;
const int ExTransport::GRC_FLAG_APPEND_TO_PREVIOUS_BUFFER;
const int ExTransport::GRC_FLAG_ALIGN_NEXT_DATA;
const int ExTransport::DEFAULT_FORWARD_DATA_TIMEOUT;

/*---------------------------------------------------------------------------*\
|* Constructor, destructor, reset, dump/debug, handleMsg:
\*---------------------------------------------------------------------------*/

#ifdef JD_DEBUGLITE
static ExTransport *pFirstTransport=0;
void ExTransport__dumpall(void);
#endif

static int ExTransportRegisteredWithLoggingManager=0;

void ExTransport__log(void) {
	lmcSetFilter("ExTransport","entry,exit,info,",0);
	lmcStartConsole();
}
void ExTransport__nolog(void) {
	lmcSetFilter(",",",",",");
	lmcStartConsole();
}

ExTransport::ExTransport(ExMgr *pMgr,ExPhysicalPort *pPhysicalPort,
    int forwardTransactionCount,int channelCount,int final=1):
      forwardTransactionCounter(forwardTransactionCount) {
	if (!ExTransportRegisteredWithLoggingManager) {
#ifdef JD_DEBUGLITE
		LOG_MODULE_REGISTER(cEXTP,"ExTransport",ExTransport__dumpall,0);
#else
		LOG_MODULE_REGISTER(cEXTP,"ExTransport",0,0);
#endif
		ExTransportRegisteredWithLoggingManager=1;
	}

	port=pMgr->getPortNumber();
	SETTHIS(pMgr);
	SETTHIS(pPhysicalPort);
	forwardTransactionCounter.resetMax(forwardTransactionCount);
	SETTHIS(channelCount);
	channelArray=0;
	overheadBufferCount=0;

	pForwardDataTimeoutMsg=pMgr->getFreeMsg();
	pForwardDataTimeoutMsg->setType((ExMsgType)MSG_FORWARD_DATA_TIMEOUT);
	pForwardDataTimer=new ExCountingWatchdogTimer(this,
		pForwardDataTimeoutMsg,
		DEBUG_STRING("pForwardDataTimer"));

#ifdef JD_DEBUGLITE
	pNextTransport=0;
	if (!pFirstTransport) {
		pFirstTransport=this;
	} else {
		ExTransport *pTransport=pFirstTransport,*pNextTransport;
		while (42) {
			pNextTransport=pTransport->getNextTransport();
			if (!pNextTransport) {
				pTransport->pNextTransport=this;
				break;
			}
			pTransport=pNextTransport;
		}
	}
#endif

	if (final) {
		reset(RESET_STARTUP);
		registerStuff("ExTransport");
	}
}

void ExTransport::initChannelArray(void) {
	LOG_ASSERT(channelCount>0,cEXTP,0,cCauseBadParm,
		"initChannelArray(port=%d): invalid channelCount=%d!\n",
		port,channelCount);
	LOG_ASSERT(!channelArray,cEXTP,0,cCauseBadState,"");
	channelArray=new TCD[channelCount];
	LOG_ASSERT(channelArray,cEXTP,0,cCauseNoMem,
		"ExTransport(port=%d): failed to malloc channelArray!\n",
		port);

	for (int channel=0;channel<channelCount;channel++) {
		channelArray[channel]=newChannel(channel);
		LOG_ASSERT(channelArray[channel],cEXTP,0,cCauseNoMem,
			"initChannelArray(port=%d): "
			"failed to malloc channel=%d!\n",
			port,channel);
	}
}

ExTransportChannel *ExTransport::newChannel(int channel) {
	return new ExTransportChannel(this,channel,channel,pMgr,pPhysicalPort);
}

ExTransport::~ExTransport(void) {
	for (int channel=0;channel<channelCount;channel++) {
		delete channelArray[channel];
	}
	delete[] channelArray;

#ifdef JD_DEBUGLITE
	pFirstTransport=0;
#endif
}

void ExTransport::reset(int rtype) {
	if (rtype==RESET_STARTUP) {
		initChannelArray();
	}

	pForwardDataTimer->reset();

	if (rtype!=RESET_START_DEACTIVATE &&
	    rtype!=RESET_FINISH_DEACTIVATE) {
		forwardTransactionCounter.reset();

		reverseDataBufferCount=0;
		reverseDataBufferSize=0;
		maxForwardBdrsPerTransaction=0;
		nextChannelToAllocate=0;

		clearBlockedChannel();

		resetReverseCount();
	}

	flushLookupQueue();
}

void ExTransport::registerStuff(char *title) {
	nettestRegister(NETTEST_TYPE_INT32,
		forwardTransactionCounter.pointToCurrent(),
		"%s[%d].forwardTransactionCounter",
		title,port);
	nettestRegister(NETTEST_TYPE_INT32,
		&channelCount,
		"%s[%d].channelCount",
		title,port);
#if 0
	nettestRegister(NETTEST_TYPE_INT32,
		&overheadBufferCount,
		"%s[%d].overheadBufferCount",
		title,port);
#endif
	nettestRegister(NETTEST_TYPE_INT32,
		pForwardDataTimer->pointToDelay(),
		"%s[%d].forwardDataTimer.delay",
		title,port);
	nettestRegister(NETTEST_TYPE_INT32,
		pForwardDataTimer->pointToCount(),
		"%s[%d].forwardDataTimer.count",
		title,port);
	nettestRegister(NETTEST_TYPE_INT32,
		&reverseDataBufferCount,
		"%s[%d].reverseDataBufferCount",
		title,port);
	nettestRegister(NETTEST_TYPE_INT32,
		&reverseDataBufferSize,
		"%s[%d].reverseDataBufferSize",
		title,port);
#if 0
	nettestRegister(NETTEST_TYPE_INT32,
		&maxForwardBdrsPerTransaction,
		"%s[%d].maxForwardBdrsPerTransaction",
		title,port);
#endif
	nettestRegister(NETTEST_TYPE_INT32,
		&nextChannelToAllocate,
		"%s[%d].nextChannelToAllocate",
		title,port);
	nettestRegister(NETTEST_TYPE_INT32,
		&nextBlockedChannel,
		"%s[%d].nextBlockedChannel",
		title,port);
}

#ifdef JD_DEBUGLITE

void ExTransport::dump(void) {
	int channel;

	printf("this=0x%8.8X\n",
		(int)this);
	printf("port=%d\n",
		port);
	printf("pMgr=0x%8.8X\n",
		(int)pMgr);
	printf("pPhysicalPort=0x%8.8X\n",
		(int)pPhysicalPort);
	printf("forwardTransactionCounter: ");
		forwardTransactionCounter.dump();
	printf("channelCount=%d\n",
		channelCount);
	printf("channelArray=0x%8.8X\n",
		(int)channelArray);
    for (channel=0;channel<channelCount;channel++) {
	printf("channelArray[%d]=0x%8.8X\n",
		channel,(int)channelArray[channel]);
    }
	printf("overheadBufferCount=%d\n",
		overheadBufferCount);
	printf("pNextTransport=0x%8.8X\n",
		(int)pNextTransport);
	printf("pForwardDataTimer=0x%8.8X (count=%d)\n",
		(int)pForwardDataTimer,pForwardDataTimer->getCount());
	printf("pForwardDataTimeoutMsg=0x%8.8X\n",
		(int)pForwardDataTimeoutMsg);
	printf("reverseDataBufferCount=%d\n",
		reverseDataBufferCount);
	printf("reverseDataBufferSize=%d\n",
		reverseDataBufferSize);
	printf("maxForwardBdrsPerTransaction=%d\n",
		maxForwardBdrsPerTransaction);
	printf("nextChannelToAllocate=%d\n",
		nextChannelToAllocate);
	printf("nextBlockedChannel=%d\n",
		nextBlockedChannel);
    int depth=lookupQueue.depth();
    ExLookup *pLookupRequest=lookupQueue.peek();
	printf("lookupQueue: depth=%d, peek=0x%8.8X\n",
		depth,(int)pLookupRequest);
    if (pLookupRequest) {
	pLookupRequest->dump();
    }
}

void ExTransport::dumpall(void) {
	int channel=0;
	TCD tcd;

	dump();
	printf("\n\n");

	for (channel=0;channel<channelCount;channel++) {
		tcd=channelArray[channel];
		if (tcd->isAllocated()) {
			tcd->dump();
			printf("\n\n");
		}
	}
}

ExTransport *ExTransport__locate(int index) {
	ExTransport *pTransport=pFirstTransport;

	while (index>0 && pTransport) {
		index--;
		pTransport=pTransport->getNextTransport();
	}

	return pTransport;
}

STATUS ExTransport__dump(int index) {
	ExTransport *pTransport=ExTransport__locate(index);
	if (!pTransport) return ERROR;
	pTransport->dump();
	return OK;
}

void ExTransport__dumpall(void) {
	int index=0;
	ExTransport *pTransport;

	while (42) {
		pTransport=ExTransport__locate(index);
		if (!pTransport) break;
		pTransport->dumpall();
		index++;
	}
}

ExTransportChannel *ExTransport::locateTransportChannel(int channel) {
	if (channel<0 || channel>=channelCount) return 0;
	return channelArray[channel];
}

ExTransportChannel *ExTransportChannel__locate(int index,int channel) {
	ExTransport *pTransport=ExTransport__locate(index);
	if (!pTransport) return 0;
	return pTransport->locateTransportChannel(channel);
}

STATUS ExTransportChannel__dump(int index,int channel) {
	ExTransportChannel *pTransportChannel=
		ExTransportChannel__locate(index,channel);
	if (!pTransportChannel) return ERROR;
	pTransportChannel->dump();
	return OK;
}

#endif

void ExTransport::handleMsg(ExMsg *pMsg) {
	switch (pMsg->getType()) {
	   case MSG_FORWARD_DATA_TIMEOUT:
		if (!pForwardDataTimer->isCancelled()) {
			LOG_ERROR(cEXTP,0,cCausePeriphError,
				"Forward data timeout on port=%d, count=%d!\n",
				port,pForwardDataTimer->getCount());
			pMgr->exClose(REASON_FORWARD_DATA_TIMEOUT);
		}
		pForwardDataTimer->setMsg(pForwardDataTimeoutMsg);
		break;

	   default:
		LOG_ERROR_FATAL(cEXTP,0,cCauseBadParm,"");
	}
}

/*---------------------------------------------------------------------------*\
|* Activate:
\*---------------------------------------------------------------------------*/

void ExTransport::activate(int reverseDataBufferCount,
    int reverseDataBufferSize,int maxForwardBdrsPerTransaction) {
	LOG_ENTRY(cEXTP,0,
		"activate(port=%d,reverseDataBufferCount=%d,"
		"reverseDataBufferSize=%d,maxForwardBdrsPerTransaction=%d)\n",
		port,reverseDataBufferCount,
		reverseDataBufferSize,maxForwardBdrsPerTransaction);

	reset(RESET_ACTIVATE);

	SETTHIS(reverseDataBufferCount);
	SETTHIS(reverseDataBufferSize);
	SETTHIS(maxForwardBdrsPerTransaction);

	/* Tell physical port who we are, to start an appropriate reset,
	 * to drain reverse data, and to start sending us reverse data. */
	pPhysicalPort->resetTransport(this);
}

void ExTransport::transportResetComplete(void) {
	int timeout=DEFAULT_FORWARD_DATA_TIMEOUT;

	tknobGetWorkingValue(port,EX_KNOB_TRANSPORT_FORWARD_DATA_TIMEOUT,
		&timeout);
	pForwardDataTimer->setDelay(timeout);

	transportResetComplete_ts();
}

void ExTransport::transportResetComplete_ts(void) {
	sendWaitingForInitialRequests();
}

void ExTransport::sendWaitingForInitialRequests(/*int status*/) {
	ExMsg *pMsg=pMgr->getFreeMsg();

	pMsg->setType(eEXMSG_ACTIVATE_WAIT);
	pMsg->setParams(eEXCLASS_TRANSPORT,(int)this);
	pMsg->send(pMgr);
}

TCD ExTransport::allocateChannel(ExService *pService,SCD scd,
    int forwardDataPriority,int minBuffersRequired,int benefitOfMoreBuffers) {
	TCD tcd;

	LOG_ASSERT(nextChannelToAllocate<channelCount,
		cEXTP,0,cCauseNoHandles,
		"allocateChannel(port=%d): "
		"no available channels!\n",port);

	tcd=channelArray[nextChannelToAllocate];
	tcd->allocate(pService,scd,forwardDataPriority,
		minBuffersRequired,benefitOfMoreBuffers,reverseDataBufferSize);
	nextChannelToAllocate++;

	return tcd;
}

void ExTransport::initialRequestsComplete(void) {
	TCD tcd;
	int channel,bufferCount,minBuffers,benefit;
	int accumMinBuffers=0,accumBenefit=0;
	int highestBenefit=1,highestBenefitCount=0;
	int extraBuffers,moreBuffers,accumGrantedBuffers=0;

#define GRANT_BUFFERS(x,phase) \
	do { \
		LOG_INFO(cEXTP,0,"initialRequestsComplete(port=%d): " \
			"granting %d buffers to channel=%d, phase=%d.\n", \
			port,x,channel,phase); \
		tcd->grantBuffers(x); \
		accumGrantedBuffers+=x; \
	} while (0)

	/* First pass through the channel list to determine buffer needs. */
	for (channel=0;channel<channelCount;channel++) {
		tcd=channelArray[channel];
		if (!tcd->isAllocated()) continue;

		/* Account for buffers channels have already granted
		 * themselves, such as the MLC command channel. */
		bufferCount=tcd->getBufferCount();
		if (bufferCount<=0) {
			bufferCount=0;
		} else {
			accumGrantedBuffers+=bufferCount;
			LOG_INFO(cEXTP,0,"initialRequestsComplete(port=%d): "
				"already granted %d buffers to channel=%d.\n",
				port,bufferCount,channel);
		}

		/* Read and accumulate minBuffers and benefit values. */
		minBuffers=tcd->getMinBuffersRequired();
		if (minBuffers<0) minBuffers=0;
		accumMinBuffers+=minBuffers;
		benefit=tcd->getBenefitOfMoreBuffers();
		if (benefit<0) benefit=0;
		accumBenefit+=benefit;

		/* Keep track of the highest benefitOfMoreBuffers value
		 * and the number of times it occurs. */
		if (benefit==highestBenefit) {
			highestBenefitCount++;
		} else if (benefit>highestBenefit) {
			highestBenefit=benefit;
			highestBenefitCount=1;
		}
	}

	/* Calculate number of extra buffers available. */
	extraBuffers=reverseDataBufferCount-accumGrantedBuffers-accumMinBuffers;
	LOG_ASSERT(extraBuffers>=0,cEXTP,0,cCauseBadParm,
		"waitingForInitialRequests(port=%d): extraBuffers=%d #1!\n",
		port,extraBuffers);

	/* For each channel, grant minimum and extra buffers,
	 * proportional to its relative benefit parameter.
	 * Channels with a zero benefit value will not get extra buffers. */
	for (channel=0;channel<channelCount;channel++) {
		tcd=channelArray[channel];
		if (!tcd->isAllocated()) continue;

		minBuffers=tcd->getMinBuffersRequired();
		if (minBuffers<0) minBuffers=0;
		benefit=tcd->getBenefitOfMoreBuffers();
		if (benefit<0) benefit=0;

		GRANT_BUFFERS(minBuffers,0);
		if (accumBenefit>0) {
			moreBuffers=(extraBuffers*benefit)/accumBenefit;
			GRANT_BUFFERS(moreBuffers,1);
		}
	}

	/* Make sure we didn't give out too many buffers. */
	extraBuffers=reverseDataBufferCount-accumGrantedBuffers;
	LOG_ASSERT(extraBuffers>=0,cEXTP,0,cCauseBadParm,
		"waitingForInitialRequests(port=%d): extraBuffers=%d #2!\n",
		port,extraBuffers);

	/* If there are still available buffers (due to division roundoff),
	 * then divide them among the channel(s) with the highest benefit. */
	if (extraBuffers>0 && highestBenefitCount>0) {
		moreBuffers=extraBuffers/highestBenefitCount;
		extraBuffers%=highestBenefitCount;
		for (channel=0;channel<channelCount &&
		      (moreBuffers || extraBuffers);channel++) {
			tcd=channelArray[channel];
			if (!tcd->isAllocated()) continue;

			benefit=tcd->getBenefitOfMoreBuffers();
			// if (benefit<0) benefit=0;
			if (benefit!=highestBenefit) continue;

			GRANT_BUFFERS(moreBuffers,2);
			if (extraBuffers>0) {
				GRANT_BUFFERS(1,3);
				extraBuffers--;
			}
		}
	}
#ifdef JD_DEBUGLITE
	if (extraBuffers) {
		LOG_INFO(cEXTP,0,"waitingForInitialRequests(port=%d): "
		"extraBuffers=%d #3!\n",port,extraBuffers);
	}
#endif

	/* Make sure we didn't give out too many buffers. */
	LOG_ASSERT(accumGrantedBuffers<=reverseDataBufferCount,
		cEXTP,0,cCauseBadParm,
		"waitingForInitialRequests(port=%d): "
		"accumGrantedBuffers=%d > reverseDataBufferCount=%d!\n",
		accumGrantedBuffers,reverseDataBufferCount);

#undef GRANT_BUFFERS

	sendActivateResponse(OK);
}

void ExTransport::sendActivateResponse(int status) {
	ExMsg *pMsg=pMgr->getFreeMsg();

	pMsg->setType(eEXMSG_ACTIVATE_RESPONSE);
	pMsg->setParams(eEXCLASS_TRANSPORT,(int)this,status);
	pMsg->send(pMgr);
}

/*---------------------------------------------------------------------------*\
|* Deactivate:
\*---------------------------------------------------------------------------*/

void ExTransport::deactivate(int reason) {
	LOG_ENTRY(cEXTP,0,"deactivate(port=%d,reason=%d)\n",port,reason);

	reset(RESET_START_DEACTIVATE);

	for (int channel=0;channel<channelCount;channel++) {
		channelArray[channel]->deactivate();
	}
}

void ExTransport::physicalPortNotActive(void) {
	LOG_ENTRY(cEXTP,0,"physicalPortNotActive(port=%d)\n",port);

	reset(RESET_FINISH_DEACTIVATE);

	for (int channel=0;channel<channelCount;channel++) {
		channelArray[channel]->deallocate();
	}

	sendDeactivateResponse(OK);
}

void ExTransport::sendDeactivateResponse(int status) {
	ExMsg *pMsg=pMgr->getFreeMsg();

	pMsg->setType(eEXMSG_DEACTIVATE_RESPONSE);
	pMsg->setParams(eEXCLASS_TRANSPORT,(int)this,status);
	pMsg->send(pMgr);
}

/*---------------------------------------------------------------------------*\
|* Remote socket lookup:
\*---------------------------------------------------------------------------*/

void ExTransport::lookupRemoteSocket(ExLookup *pLookupRequest) {
	LOG_ENTRY(cEXTP,0,
		"lookupRemoteSocket(port=%d,pLookupRequest=0x%8.8X,"
		"lastSet=%d,serviceName=<%s>,socketID=%d)\n",
		port,pLookupRequest,pLookupRequest->getLastSet(),
		(pLookupRequest->getLastSet()==ExLookup::LAST_SET_SERVICE_NAME?
		 pLookupRequest->getServiceName():""),
		pLookupRequest->getSocketID());

	lookupQueue.add(pLookupRequest);
	if (lookupQueue.depth()==1) {
		lookupNextRemoteSocket();
	}
}

void ExTransport::lookupNextRemoteSocket(void) {
	ExLookup *pLookupRequest=lookupQueue.peek();
	if (pLookupRequest) {
		lookupRemoteSocket_ts(pLookupRequest);
	}
}

void ExTransport::lookupRemoteSocket_ts(ExLookup *_pLookupRequest) {
	static struct {
		char socketID;
		char *serviceName;
	} legacyServiceLookupTable[]={
		{ 1,"HP-MESSAGE"},
		{ 2,"PRINT"},
		{ 4,"HP-SCAN"},
		{ 6,"ECHO"},
		{ 7,"HP-FAX-SEND"},
		{ 8,"HP-FAX-RECEIVE"},
		{ 9,"HP-DIAGNOSTICS"},
		{10,"HP-RESERVED"},
		{11,"HP-IMAGE-DOWNLOAD"},
		{12,"HP-HOST-DATA-STORE-UPLOAD"},
		{13,"HP-HOST-DATA-STORE-DOWNLOAD"},
		{14,"HP-CONFIGURATION-UPLOAD"},
		{15,"HP-CONFIGURATION-DOWNLOAD"},
		{ 0,0}
	};

	int i,status=S_hlio_BP_SERVICE_LOOKUP_FAILED;

	ExLookup *pLookupRequest=lookupQueue.pop();
	LOG_ASSERT(pLookupRequest==_pLookupRequest,cEXTP,0,cCauseBadParm,"");
	int lastSet=pLookupRequest->getLastSet();
	int socketID=pLookupRequest->getSocketID();
	char *serviceName=pLookupRequest->getServiceName();

	for (i=0;legacyServiceLookupTable[i].serviceName;i++) {
		if (lastSet==ExLookup::LAST_SET_SERVICE_NAME) {
			if (!strcmp(legacyServiceLookupTable[i].serviceName,
			     serviceName)) {
				status=OK;
				pLookupRequest->setSocketID(
					legacyServiceLookupTable[i].socketID,
					1);
				break;
			}

		} else if (lastSet==ExLookup::LAST_SET_SOCKET_ID) {
			if (legacyServiceLookupTable[i].socketID==socketID) {
				status=OK;
				pLookupRequest->setServiceName(
					legacyServiceLookupTable[i].serviceName,
					0,1);
				break;
			}

		} else {
			LOG_ERROR_FATAL(cEXTP,0,cCauseBadParm,"");
		}
	}

	pLookupRequest->setStatus(status);
	sendRemoteSocketLookupResponse(pLookupRequest);
}

void ExTransport::flushLookupQueue(void) {
	ExLookup *pLookupRequest;

	while (42) {
		pLookupRequest=lookupQueue.pop();
		if (!pLookupRequest) break;

		pLookupRequest->setStatus(S_hlio_BP_LOOKUP_NOT_SUPPORTED);
		sendRemoteSocketLookupResponse(pLookupRequest);
	}
}

void ExTransport::sendRemoteSocketLookupResponse(ExLookup *pLookupRequest) {
	ExService *pService=pLookupRequest->getService();
	LOG_ASSERT(pService,cEXTP,0,cCauseBadParm,"");

	LOG_ENTRY(cEXTP,0,
		"sendRemoteSocketLookupResponse(port=%d,"
		"pLookupRequest=0x%8.8X): pService=0x%8.8X, status=%d.\n",
		port,pLookupRequest,pService,pLookupRequest->getStatus());

	ExMsg *pMsg=pMgr->getFreeMsg();

	pMsg->setType(eEXMSG_LOOKUP_RESPONSE);
#ifndef EX_TRANSPORT_UNIX_PORT
	pMsg->setParams((int)pLookupRequest);
#else
	pMsg->setParams((void *)pLookupRequest);
#endif
	pMsg->send(pService);

	lookupNextRemoteSocket();
}

/*---------------------------------------------------------------------------*\
|* Forward data flow:
\*---------------------------------------------------------------------------*/

void ExTransport::returnForwardTransaction(void) {
	int wasEmpty,r;

	wasEmpty=forwardTransactionCounter.isEmpty();
	r=forwardTransactionCounter.increment();
	LOG_ASSERT(r!=ERROR,cEXTP,0,cCauseBadState,"");
	pForwardDataTimer->stop();

	/* TODO: two separate functions? */
	int blockedChannel=clearBlockedChannel();
	if (blockedChannel!=ERROR) {
		LOGD_ASSERT(wasEmpty,cEXTP,0,cCauseBadState,"");

		int channel=blockedChannel;
		do {
			LOG_INFO(cEXTP,0,
				"returnForwardTransaction(port=%d): "
				"trying to unblock channel=%d.\n",
				port,channel);

			TCD tcd=channelArray[channel];

			if (tcd->isOpen()) tcd->forwardDataAvailable();

			channel++;
			if (channel>=nextChannelToAllocate) channel=0;
		} while (channel!=blockedChannel && getBlockedChannel()==ERROR);
	}
}

/*---------------------------------------------------------------------------*\
|* Reverse data flow:
\*---------------------------------------------------------------------------*/

void ExTransport::resetReverseCount(void) {
	/* Do nothing. */
}

STATUS ExTransport::getReverseCount(char *lastData,int lastLength,
    int *pCount,int *pFlags) {
	TCD tcd=channelArray[0];

	if (!tcd->isOpen()) {
		*pCount=reverseDataBufferSize;
	} else {
		*pCount=tcd->getMaxReverseDatalen();
	}
	*pFlags=(GRC_FLAG_END_OF_TRANSACTION|GRC_FLAG_PARTIAL_DATA_OK);
	return OK;
}

void ExTransport::reverseDataReceived(ExBdr *pBdr,int status) {
#if 0
	if (status!=OK) {
		/* TODO: How do we handle this? */
	}
#endif

	unmarkBDR(pBdr);
	ExTransportChannel *tcd=lookupChannel(pBdr);
	if (!tcd) {
		LOGD_ERROR(cEXTP,0,cCauseBadParm,
			"reverseDataReceived(pBdr=0x%8.8X): null TCD!\n",
			pBdr);
		returnReverseBuffer(pBdr);
	} else {
		tcd->reverseDataReceived(pBdr,status);
	}
}

void ExTransport::unmarkBDR(ExBdr *pBdr) {
	while (pBdr) {
		pBdr->setTransportNotification((ExTransport *)0);
		pBdr->setTCD(0);
		pBdr=pBdr->getNext();
	}
}

ExTransportChannel *ExTransport::lookupChannel(ExBdr *pBdr) {
	TCD tcd=channelArray[0];
	LOG_ASSERT(tcd,cEXTP,0,cCauseBadState,"");
	return tcd;
}

void ExTransport::returnReverseBuffer(ExBdr *pBdr) {
	/* TODO: Make sure this works correctly! */
	exReturnPhysicalPortBuffer(pBdr);
}

void ExTransport::returnBufferNotification(ExBdr *pBdr) {
	TCD tcd=pBdr->getTCD();
	if (tcd) {
		tcd->returnBufferNotification(pBdr);
	} else {
		LOGD_ERROR(cEXTP,0,cCauseBadParm,
			"returnBufferNotification(pBdr=0x%8.8X): null TCD!\n",
			pBdr);
	}
}
