#include <bognor/br-queue.h>

#include <nbtk/nbtk.h>

#include "hrn.h"
#include "hrn-view.h"
#include "hrn-tiler.h"
#include "hrn-ui-controls.h"
#include "hrn-pin-manager.h"
#include "hrn-content-area.h"
#include "hrn-source-manager.h"
#include "hrn-theatre-ui.h"

enum {
    PROP_0,
    PROP_PIN_MANAGER,
    PROP_QUEUE,
    PROP_SOURCE_MANAGER,
    PROP_THEATRE_UI,
};

enum {
    SOURCE_CHANGED,
    QUERY_CHANGED,
    FILTER_CHANGED,
    ACTIVATE_THEATRE,
    LAST_SIGNAL
};

struct _HrnContentAreaPrivate {
    HrnView *view;
    ClutterActor *scroll;

    HrnTheatreUi *theatre_ui;
    HrnPinManager *pin_manager;
    BrQueue *local_queue;
    HrnUiControls *controls;

    HrnSource *current_source;

    HrnSourceManager *source_manager;

    guint32 search_id;

    BklItemType filter;
};

/* FIXME: These definitions should be stored somewhere global */
#define SIDEBAR_WIDTH 200.0

#define GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), HRN_TYPE_CONTENT_AREA, HrnContentAreaPrivate))
G_DEFINE_TYPE (HrnContentArea, hrn_content_area, CLUTTER_TYPE_GROUP);
static guint32 signals[LAST_SIGNAL] = {0, };

static void
hrn_content_area_finalize (GObject *object)
{
    G_OBJECT_CLASS (hrn_content_area_parent_class)->finalize (object);
}

static void
hrn_content_area_dispose (GObject *object)
{
    G_OBJECT_CLASS (hrn_content_area_parent_class)->dispose (object);
}

static void
source_manager_ready (HrnSourceManager *manager,
                      HrnContentArea   *area)
{
}

static void
source_added_cb (HrnSourceManager *manager,
                 HrnSource        *source,
                 HrnContentArea   *area)
{
    HrnContentAreaPrivate *priv = area->priv;
    const char *source_path;
    HrnSidebar *sidebar;
    gboolean is_local;

    source_path = hrn_source_get_object_path (source);
    is_local = g_str_equal (source_path, BKL_LOCAL_SOURCE_PATH);
    sidebar = hrn_ui_controls_get_sidebar (priv->controls);
    hrn_sidebar_add_source (sidebar, source, is_local);
}

static void
source_removed_cb (HrnSourceManager *manager,
                   HrnSource        *source,
                   HrnContentArea   *area)
{
    HrnContentAreaPrivate *priv = area->priv;
    HrnSidebar *sidebar;

    if (priv->current_source == source) {
        HrnToolbar *tb;

        tb = hrn_ui_controls_get_toolbar (priv->controls);
        hrn_toolbar_set_pinned (tb, FALSE);

        priv->filter = BKL_ITEM_TYPE_AUDIO;
        hrn_content_area_set_source
            (area, hrn_source_manager_get_local_source (priv->source_manager));
    }

    sidebar = hrn_ui_controls_get_sidebar (priv->controls);
    hrn_sidebar_remove_source (sidebar, source);
}

static void
hrn_content_area_set_property (GObject      *object,
                               guint         prop_id,
                               const GValue *value,
                               GParamSpec   *pspec)
{
    HrnContentArea *self = (HrnContentArea *) object;
    HrnContentAreaPrivate *priv = self->priv;

    switch (prop_id) {
    case PROP_PIN_MANAGER:
        priv->pin_manager = g_value_dup_object (value);
        break;

    case PROP_QUEUE:
        priv->local_queue = g_value_dup_object (value);
        break;

    case PROP_SOURCE_MANAGER:
        priv->source_manager = g_value_dup_object (value);
        g_signal_connect (priv->source_manager, "ready",
                          G_CALLBACK (source_manager_ready), self);
        g_signal_connect (priv->source_manager, "source-added",
                          G_CALLBACK (source_added_cb), self);
        g_signal_connect (priv->source_manager, "source-removed",
                          G_CALLBACK (source_removed_cb), self);
        break;

    case PROP_THEATRE_UI:
        priv->theatre_ui = g_value_dup_object (value);
        break;

    default:
        break;
    }
}

static void
hrn_content_area_get_property (GObject    *object,
                               guint       prop_id,
                               GValue     *value,
                               GParamSpec *pspec)
{
    switch (prop_id) {

    default:
        break;
    }
}

static void
hrn_content_area_allocate (ClutterActor          *actor,
                           const ClutterActorBox *box,
                           ClutterAllocationFlags flags)
{
    HrnContentArea *area = (HrnContentArea *) actor;
    HrnContentAreaPrivate *priv = area->priv;
    float width, height;

    CLUTTER_ACTOR_CLASS (hrn_content_area_parent_class)->allocate
        (actor, box, flags);

    width = box->x2 - box->x1;
    height = box->y2 - box->y1;

    clutter_actor_set_size ((ClutterActor *) priv->scroll,
                            width - SIDEBAR_WIDTH, height - TOOLBAR_HEIGHT);

    clutter_actor_set_size ((ClutterActor *) priv->controls, width, height);
}

/* FIXME: Should this be moved up to toplevel, and HCA just fires another
   signal? */
static void
quit_requested_cb (HrnUiControls  *controls,
                   HrnContentArea *area)
{
    hrn_quit ();
}

static void
reload_view (HrnContentArea *area)
{
    HrnContentAreaPrivate *priv = area->priv;
    HrnClusterTree *tree;
    HrnClusterNode *node;

    tree = hrn_source_get_cluster_tree (priv->current_source);

    switch (priv->filter) {
    case BKL_ITEM_TYPE_AUDIO:
        node = tree->audio;
        break;

    case BKL_ITEM_TYPE_IMAGE:
        node = tree->image;
        break;

    case BKL_ITEM_TYPE_VIDEO:
        node = tree->video;
        break;

    default:
        node = tree->audio;
        break;
    }

    hrn_view_clear (priv->view);
    hrn_tiler_add_items (HRN_TILER (priv->view), node);
}

static void
source_changed_cb (HrnUiControls  *controls,
                   HrnSource      *source,
                   HrnContentArea *area)
{
    hrn_content_area_set_source (area, source);
    g_signal_emit (area, signals[SOURCE_CHANGED], 0, source);
}

static void
filter_changed_cb (HrnUiControls  *controls,
                   int             filter,
                   HrnContentArea *area)
{
    HrnContentAreaPrivate *priv = area->priv;

    priv->filter = filter;
    reload_view (area);

    g_signal_emit (area, signals[FILTER_CHANGED], 0, filter);
}

static gboolean
deferred_search (gpointer data)
{
    HrnContentArea *area = (HrnContentArea *) data;
    HrnContentAreaPrivate *priv = area->priv;
    HrnToolbar *toolbar = hrn_ui_controls_get_toolbar (priv->controls);
    const char *search;

    priv->search_id = 0;

    search = hrn_toolbar_get_query (toolbar);
    if (search == NULL || *search == '\0') {
        hrn_view_reset (priv->view);
    }

    hrn_view_close_shown_item (priv->view);

    hrn_source_filter (priv->current_source, search);
    g_signal_emit (area, signals[QUERY_CHANGED], 0, search);

    return FALSE;
}

static void
query_changed_cb (HrnUiControls  *controls,
                  const char     *query,
                  HrnContentArea *area)
{
    HrnContentAreaPrivate *priv = area->priv;

    if (priv->search_id) {
        g_source_remove (priv->search_id);
        priv->search_id = 0;
    }

    if (query == NULL || *query == '\0') {
        deferred_search (area);
        return;
    }

    priv->search_id = g_timeout_add (250, deferred_search, area);
}

static void
pinned_cb (HrnUiControls  *controls,
           gboolean        pinned,
           HrnContentArea *area)
{
    HrnContentAreaPrivate *priv = area->priv;

    if (pinned) {
        HrnToolbar *toolbar = hrn_ui_controls_get_toolbar (priv->controls);
        const char *search;

        search = hrn_toolbar_get_query (toolbar);
        hrn_view_close_shown_item (priv->view);
        hrn_pin_manager_add (priv->pin_manager, search, search,
                             hrn_source_get_object_path (priv->current_source),
                             priv->filter, 0);
    } else {
        const char *pin;

        pin = hrn_pin_manager_get_selected_pin (priv->pin_manager);
        hrn_pin_manager_remove (priv->pin_manager, pin);
    }
}

static void
show_image_node (HrnContentArea *area,
                 HrnClusterNode *node)
{
    HrnContentAreaPrivate *priv = area->priv;

    g_signal_emit (area, signals[ACTIVATE_THEATRE], 0, 1);
    hrn_theatre_ui_show_node (priv->theatre_ui, node);
}

static void
show_video_node (HrnContentArea *area,
                 HrnClusterNode *node)
{
    HrnContentAreaPrivate *priv = area->priv;

    g_signal_emit (area, signals[ACTIVATE_THEATRE], 0, 1);
    hrn_theatre_ui_show_node (priv->theatre_ui, node);
}

static void
play_track (HrnContentArea *area,
            HrnClusterNode *node)
{
    HrnContentAreaPrivate *priv = area->priv;
    TrackCluster *track = (TrackCluster *) node->data;
    BklItem *item = track->item;

    br_queue_play_uri (priv->local_queue,
                       bkl_item_get_uri (item), bkl_item_get_mimetype (item));
}

static void
play_node (HrnContentArea *area,
           HrnClusterNode *node)
{
    HrnContentAreaPrivate *priv = area->priv;
    GList *flat_tree = NULL, *f;

    hrn_cluster_node_flatten_reversed (node, &flat_tree);
    flat_tree = g_list_reverse (flat_tree);
    for (f = flat_tree; f; f = f->next) {
        HrnClusterNode *child = f->data;
        TrackCluster *track = child->data;
        BklItem *item = track->item;

        if (child->hidden) {
            continue;
        }

        /* FIXME: Being able to pass a list to BR would help with the
           DBus traffic level. */
        br_queue_add_uri (priv->local_queue,
                          bkl_item_get_uri (item),
                          bkl_item_get_mimetype (item));
    }
    g_list_free (flat_tree);

    /* Start playing if its not already playing */
    br_queue_play (priv->local_queue);
}

static void
action_play (HrnContentArea *area,
             HrnClusterNode *node)
{
    HrnContentAreaPrivate *priv = area->priv;

    switch (node->type) {
    case HRN_CLUSTER_NODE_TYPE_AUDIO_ROOT:
    case HRN_CLUSTER_NODE_TYPE_IMAGE_ROOT:
    case HRN_CLUSTER_NODE_TYPE_VIDEO_ROOT:
        /* How did we get here? */
        break;

    case HRN_CLUSTER_NODE_TYPE_ARTIST:
        play_node (area, node);
        break;

    case HRN_CLUSTER_NODE_TYPE_ALBUM:
        play_node (area, node);
        break;

    case HRN_CLUSTER_NODE_TYPE_TRACK:
        play_track (area, node);
        break;

    case HRN_CLUSTER_NODE_TYPE_YEAR:
    case HRN_CLUSTER_NODE_TYPE_MONTH:
        show_image_node (area, node);
        hrn_theatre_ui_set_playing (priv->theatre_ui, TRUE);
        break;

    case HRN_CLUSTER_NODE_TYPE_IMAGE:
        show_image_node (area, node);
        break;

    case HRN_CLUSTER_NODE_TYPE_VIDEO:
        show_video_node (area, node);
        hrn_theatre_ui_set_playing (priv->theatre_ui, TRUE);
        break;

    default:
        break;
    }
}

static char *
get_node_string (HrnClusterNode *node)
{
    if (node->type == HRN_CLUSTER_NODE_TYPE_AUDIO_ROOT ||
        node->type == HRN_CLUSTER_NODE_TYPE_IMAGE_ROOT ||
        node->type == HRN_CLUSTER_NODE_TYPE_VIDEO_ROOT) {
        return g_strdup ("");
    }

    if (node->parent == NULL) {
        return g_strdup (node->name);
    }

    if (node->parent->type == HRN_CLUSTER_NODE_TYPE_AUDIO_ROOT ||
        node->parent->type == HRN_CLUSTER_NODE_TYPE_IMAGE_ROOT ||
        node->parent->type == HRN_CLUSTER_NODE_TYPE_VIDEO_ROOT) {
        return g_strdup (node->name);
    }

    return g_strdup_printf ("%s ➜ %s", node->parent->name, node->name);
}

static void
action_expanded (HrnContentArea *area,
                 HrnClusterNode *node)
{
    HrnContentAreaPrivate *priv = area->priv;
    HrnToolbar *toolbar;
    char *text_str;

    toolbar = hrn_ui_controls_get_toolbar (priv->controls);

    switch (node->type) {
    case HRN_CLUSTER_NODE_TYPE_AUDIO_ROOT:
    case HRN_CLUSTER_NODE_TYPE_IMAGE_ROOT:
    case HRN_CLUSTER_NODE_TYPE_VIDEO_ROOT:
        /* How did we get here? */
        break;

    case HRN_CLUSTER_NODE_TYPE_YEAR:
    case HRN_CLUSTER_NODE_TYPE_ARTIST:
        hrn_toolbar_set_query (toolbar, node->name);
        break;

    case HRN_CLUSTER_NODE_TYPE_ALBUM:
    case HRN_CLUSTER_NODE_TYPE_MONTH:
        text_str = get_node_string (node);
        hrn_toolbar_set_query (toolbar, text_str);
        g_free (text_str);
        break;

    case HRN_CLUSTER_NODE_TYPE_TRACK:
    case HRN_CLUSTER_NODE_TYPE_IMAGE:
    case HRN_CLUSTER_NODE_TYPE_VIDEO:
        /* How did we get here? */
        break;

    default:
        break;
    }
}

static void
view_activated_cb (HrnView          *view,
                   HrnTileableAction action,
                   gpointer          payload,
                   HrnContentArea   *area)
{
    HrnContentAreaPrivate *priv = area->priv;
    HrnToolbar *toolbar;
    char *text_str;

    switch (action) {
    case HRN_TILEABLE_ACTION_PLAY:
        action_play (area, (HrnClusterNode *) payload);
        break;

    case HRN_TILEABLE_ACTION_EXPANDED:
        action_expanded (area, (HrnClusterNode *) payload);
        break;

    case HRN_TILEABLE_ACTION_SEARCH:
        if (priv->search_id > 0) {
            g_source_remove (priv->search_id);
            priv->search_id = 0;
        }

        hrn_view_close_shown_item (priv->view);

        toolbar = hrn_ui_controls_get_toolbar (priv->controls);
        hrn_toolbar_set_query (toolbar, (char *) payload);

        hrn_source_filter (priv->current_source, (char *) payload);
        g_signal_emit (area, signals[QUERY_CHANGED], 0, (char *) payload);
        break;

    case HRN_TILEABLE_ACTION_LEVEL_CHANGE:
        toolbar = hrn_ui_controls_get_toolbar (priv->controls);

        text_str = get_node_string ((HrnClusterNode *) payload);
        hrn_toolbar_set_query (toolbar, text_str);
        g_free (text_str);
        break;

    default:
        break;
    }
}

static GObject *
hrn_content_area_constructor (GType                  type,
                              guint                  n_params,
                              GObjectConstructParam *params)
{
    GObject *object;
    HrnContentArea *area;
    HrnContentAreaPrivate *priv;

    object = G_OBJECT_CLASS (hrn_content_area_parent_class)->constructor
        (type, n_params, params);

    area = HRN_CONTENT_AREA (object);
    priv = area->priv;

    priv->scroll = (ClutterActor *) nbtk_scroll_view_new ();
    clutter_actor_set_position (priv->scroll, SIDEBAR_WIDTH, TOOLBAR_HEIGHT);
    clutter_container_add_actor ((ClutterContainer *) area, priv->scroll);

    priv->view = g_object_new (HRN_TYPE_VIEW, NULL);
    clutter_container_add_actor ((ClutterContainer *) priv->scroll,
                                 (ClutterActor *) priv->view);
    g_signal_connect (priv->view, "activated",
                      G_CALLBACK (view_activated_cb), area);

    priv->controls = g_object_new (HRN_TYPE_UI_CONTROLS,
                                   "pin-manager", priv->pin_manager,
                                   "local-queue", priv->local_queue,
                                   "source-manager", priv->source_manager,
                                   NULL);

    g_signal_connect (priv->controls, "quit-requested",
                      G_CALLBACK (quit_requested_cb), area);
    g_signal_connect (priv->controls, "source-changed",
                      G_CALLBACK (source_changed_cb), area);
    g_signal_connect (priv->controls, "query-changed",
                      G_CALLBACK (query_changed_cb), area);
    g_signal_connect (priv->controls, "filter-changed",
                      G_CALLBACK (filter_changed_cb), area);
    g_signal_connect (priv->controls, "pinned",
                      G_CALLBACK (pinned_cb), area);

    clutter_container_add_actor ((ClutterContainer *) area,
                                 (ClutterActor *) priv->controls);

    return object;
}

static void
hrn_content_area_class_init (HrnContentAreaClass *klass)
{
    GObjectClass *o_class = (GObjectClass *) klass;
    ClutterActorClass *a_class = (ClutterActorClass *) klass;

    o_class->dispose = hrn_content_area_dispose;
    o_class->finalize = hrn_content_area_finalize;
    o_class->set_property = hrn_content_area_set_property;
    o_class->get_property = hrn_content_area_get_property;
    o_class->constructor = hrn_content_area_constructor;

    a_class->allocate = hrn_content_area_allocate;

    g_type_class_add_private (klass, sizeof (HrnContentAreaPrivate));

    g_object_class_install_property (o_class, PROP_PIN_MANAGER,
                                     g_param_spec_object ("pin-manager", "", "",
                                                          HRN_TYPE_PIN_MANAGER,
                                                          G_PARAM_WRITABLE |
                                                          G_PARAM_CONSTRUCT_ONLY |
                                                          G_PARAM_STATIC_STRINGS));
    g_object_class_install_property (o_class, PROP_QUEUE,
                                     g_param_spec_object ("local-queue", "", "",
                                                          BR_TYPE_QUEUE,
                                                          G_PARAM_WRITABLE |
                                                          G_PARAM_CONSTRUCT_ONLY |
                                                          G_PARAM_STATIC_STRINGS));
    g_object_class_install_property (o_class, PROP_SOURCE_MANAGER,
                                     g_param_spec_object ("source-manager", "", "",
                                                          HRN_TYPE_SOURCE_MANAGER,
                                                          G_PARAM_WRITABLE |
                                                          G_PARAM_CONSTRUCT_ONLY |
                                                          G_PARAM_STATIC_STRINGS));
    g_object_class_install_property (o_class, PROP_THEATRE_UI,
                                     g_param_spec_object ("theatre-ui", "", "",
                                                          HRN_TYPE_THEATRE_UI,
                                                          G_PARAM_WRITABLE |
                                                          G_PARAM_CONSTRUCT_ONLY |
                                                          G_PARAM_STATIC_STRINGS));

    signals[SOURCE_CHANGED] = g_signal_new ("source-changed",
                                            G_TYPE_FROM_CLASS (klass),
                                            G_SIGNAL_RUN_FIRST |
                                            G_SIGNAL_NO_RECURSE, 0, NULL, NULL,
                                            g_cclosure_marshal_VOID__OBJECT,
                                            G_TYPE_NONE, 1, HRN_TYPE_SOURCE);
    signals[QUERY_CHANGED] = g_signal_new ("query-changed",
                                           G_TYPE_FROM_CLASS (klass),
                                           G_SIGNAL_RUN_FIRST |
                                           G_SIGNAL_NO_RECURSE, 0, NULL, NULL,
                                           g_cclosure_marshal_VOID__STRING,
                                           G_TYPE_NONE, 1, G_TYPE_STRING);
    signals[FILTER_CHANGED] = g_signal_new ("filter-changed",
                                            G_TYPE_FROM_CLASS (klass),
                                            G_SIGNAL_RUN_FIRST |
                                            G_SIGNAL_NO_RECURSE, 0, NULL, NULL,
                                            g_cclosure_marshal_VOID__INT,
                                            G_TYPE_NONE, 1, G_TYPE_INT);
    signals[ACTIVATE_THEATRE] = g_signal_new ("activate-theatre",
                                              G_TYPE_FROM_CLASS (klass),
                                              G_SIGNAL_RUN_FIRST |
                                              G_SIGNAL_NO_RECURSE, 0, NULL, NULL,
                                              g_cclosure_marshal_VOID__BOOLEAN,
                                              G_TYPE_NONE, 1, G_TYPE_BOOLEAN);
}

static void
hrn_content_area_init (HrnContentArea *self)
{
    HrnContentAreaPrivate *priv;

    self->priv = GET_PRIVATE (self);
    priv = self->priv;

    priv->filter = BKL_ITEM_TYPE_AUDIO;
}

void
hrn_content_area_show (HrnContentArea *area)
{
    HrnContentAreaPrivate *priv = area->priv;

    hrn_ui_controls_show (priv->controls);

    clutter_actor_show ((ClutterActor *) area);
#if 0
    clutter_actor_animate ((ClutterActor *) priv->view, CLUTTER_LINEAR,
                           HRN_FROM_THEATRE_DURATION,
                           "opacity", 0xFF,
                           NULL);
#endif
}

void
hrn_content_area_hide (HrnContentArea *area)
{
    HrnContentAreaPrivate *priv = area->priv;

    hrn_ui_controls_hide (priv->controls);
#if 0
    clutter_actor_animate ((ClutterActor *) area,
                           CLUTTER_EASE_IN_OUT_CUBIC,
                           SIDEBAR_OUT_DURATION,
                           "depth", 0.0,
                           "x", 0.0,
                           NULL);
#endif
}

void
hrn_content_area_restore_state (HrnContentArea  *area,
                                HrnStateManager *state_manager)
{
    HrnContentAreaPrivate *priv = area->priv;
    HrnToolbar *tb;

    priv->filter = hrn_state_manager_get_filter (state_manager);

    tb = hrn_ui_controls_get_toolbar (priv->controls);
    hrn_toolbar_set_query (tb, hrn_state_manager_get_query (state_manager));

    hrn_source_filter (priv->current_source,
                       hrn_state_manager_get_query (state_manager));
}

void
hrn_content_area_set_enabled (HrnContentArea *area,
                              gboolean        enabled)
{
    HrnContentAreaPrivate *priv = area->priv;

    hrn_ui_controls_set_enabled (priv->controls, enabled);
}

void
hrn_content_area_set_zoom (HrnContentArea *area,
                           HrnZoomLevel    zoom_level)
{
    g_warning ("FIXME: hrn_content_area_set_zoom");
}

/* This function is a hack because the items "quicksearch" doesn't emit
   signals, but just tries to set the query directly */
void
hrn_content_area_set_query (HrnContentArea *area,
                            const char     *query)
{
    HrnContentAreaPrivate *priv = area->priv;

    hrn_ui_controls_set_query (priv->controls, query);
}

void
hrn_content_area_set_source (HrnContentArea *area,
                             HrnSource      *source)
{
    HrnContentAreaPrivate *priv = area->priv;

    if (priv->current_source == source) {
        return;
    }
    priv->current_source = source;

    reload_view (area);
}
