/*
 *  JLib - Jacob's Library.
 *  Copyright (C) 2003, 2004  Juan Carlos Seijo Prez
 * 
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 * 
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 * 
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 *  Juan Carlos Seijo Prez
 *  jacob@mainreactor.net
 */

/** Clase base para threads.
 * @file    JThread.h
 * @author  Juan Carlos Seijo Prez
 * @date    18/04/2004
 * @version 0.0.1 - 18/04/2004 - Primera versin.
 */

#ifndef _JTHREAD_INCLUDED
#define _JTHREAD_INCLUDED

#include <JLib/Util/JTypes.h>
#include <SDL/SDL.h>
#include <SDL/SDL_thread.h>

class JThread
{
 protected:
	bool valid;                 /**< Flag de validez del thread. */
	bool paused;                /**< Flag de pausa de la ejecucin del thread. */
	bool terminate;             /**< Flag de terminacin del thread. */
	SDL_Thread *thread;         /**< Thread SDL de este thread. */
	u32 id;                     /**< Identificador de este thread. */

 public:
	/** Crea un nuevo thread para ejecutar la funcin especificada.
	 * Se puede implementar una funcin en la clase hija que sea interna
	 * si se define como 'static int Func(void *data)' o cualquier otro
	 * nombre en lugar de Func y llamando a JThread(&Func, data).
	 * Un ejemplo de la forma en que se debera implementar la funcin func es:
	 * <pre>
	 * while (!terminate)
	 * {
	 *   while (paused) 
	 *     sleep(0);
	 *
	 *   // Cuerpo de la ejecucin del thread
	 * }
	 * </pre>
	 * @param  func Fucnin a ejecutar. Si es nula no se lanza el thread.
	 * @param  data Datos adicionales a pasar a la funcin.
	 */
	JThread(int (* func)(void *), void *data = 0);

	/** Devuelve el identificador de este thread.
	 * @return Identificador de este thread.
	 */
	u32 GetId() {return id;}

	/** Determina si este thread debe terminar.
	 * @return <b>true</b> si debe terminar, <b>false</b> si no.
	 */
	bool MustTerminate() {return terminate;}

	/** Sealiza este thread para terminar. Esto no garantiza que termine,
	 * puesto que depende de que en la implementacin se use MustTerminate()
	 * como indicador de terminacin.
	 */
	void Terminate() {terminate = true; paused = false; valid = false; SDL_WaitThread(thread, 0);}

	/** Determina si este thread est sealizado en pausa.
	 * @return <b>true</b> si est en pausa, <b>false</b> si no.
	 */
	bool Paused() {return paused;}

	/** Activa el flag de pausa del thread.
	 */
	void Pause() {paused = true;}

	/** Desactiva el flag de pausa del thread.
	 */
	void Continue() {paused = false;}

	/** Mata el thread. Debe evitarse esta aproximacin, es mejor implementar
	 * en la clase hija un mecanismo de salida tipo:
	 * <pre>
	 * while (!terminate)
	 * {
	 *   while (paused) 
	 *     sleep(0);
	 *
	 *   // Cuerpo de la ejecucin del thread
	 * }
	 * </pre>
	 */
	void Kill() {SDL_KillThread(thread);}

	/** Determina si este thread es vlido.
	 * @return <b>true</b> si lo es, <b>false</b> si no.
	 */
	bool Valid() {return valid;}
};

#endif // _JTHREAD_INCLUDED
