/*
 *  JLib - Jacob's Library.
 *  Copyright (C) 2003, 2004  Juan Carlos Seijo Prez
 * 
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 * 
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 * 
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 *  Juan Carlos Seijo Prez
 *  jacob@mainreactor.net
 */

/** Sistema de ficheros virtual.
 * @file    JFS.cpp
 * @author  Juan Carlos Seijo Prez
 * @date    23/12/2003
 * @version 0.0.1 - 23/12/2003 - Primera versin.
 */

#include <JLib/Util/JFS.h>

u32 JResource::idCount;

// Crea y devuelve un nuevo dato de recurso adecuado a partir del tipo pasado.
// En el caso de un bloque de recursos, lo crea con nombre vaco y padre = 0.
void * JFSNewFromType(u32 type)
{
	switch (type)
	{
		// Bloque de recursos
		case JRES_RESOURCEBLOCK:
			return (new JResourceBlock("", 0));
			break;

		// Imagen
		case JRES_IMAGE:
			return (new JImage());
			break;

		// Sprite de imgenes
		case JRES_IMAGESPRITE:
			return (new JImageSprite());
			break;

		// TODO: Aadir aqu JRES_JSCENEDxxx cuando se integren JLib y
		// sus herramientas (JSprited, JScened, etc.)
		default:
			return 0;
	}
}

// Carga la cabecera
u32 JResourceHeader::Load(JFile &f)
{
  if (0 != f.Read(&dataOffset, sizeof(dataOffset)) &&
      0 != f.Read(&nextOffset, sizeof(nextOffset)) &&
      0 != f.Read(&type, sizeof(type)) &&
      0 == name.Load(f))
  {
    return 0;
  }

  return 1;
}

// Salva la cebecera
u32 JResourceHeader::Save(JFile &f)
{
  if (0 != f.Write(&dataOffset, sizeof(dataOffset)) &&
      0 != f.Write(&nextOffset, sizeof(nextOffset)) &&
      0 != f.Write(&type, sizeof(type)) &&
      0 == name.Save(f))
  {
    return 0;
  }

  return 1;
}

// Comienza la grabacin del recurso. Escribe la cabecera.
u32 JResource::BeginSave(JFile &f)
{
  u32 oldPos, resp = 0;

  // Salva la posicin inicial
  oldPos = f.Pos();
  if (0 == (resp |= header.Save(f)))
  {
    // Actualiza el offset de datos
    header.dataOffset = f.Pos();
    f.Pos(oldPos);

    if (0 == f.Write(&header.dataOffset, sizeof(header.dataOffset)))
    {
      return 1;
    }

    // Guarda temporalmente la posicin de nextOffset para EndSave()
    header.nextOffset = f.Pos();
    f.Pos(header.dataOffset);
  }

  return resp;
}

// Finaliza la grabacin del recurso. Actualiza los offsets.
u32 JResource::EndSave(JFile &f)
{
  u32 oldPos, resp = 0;
  oldPos = f.Pos();
  
  // Recuperamos la posicin que guardamos en BeginSave()
  f.Pos(header.nextOffset);
  if (0 == f.Write(&oldPos, sizeof(oldPos)))
    resp = 1;
  
  f.Pos(oldPos);

  return resp;
}

// Carga slo la cabecera. Para cargar el recurso actual
// hay que usar Load(JFile, JLoadSave*)
u32 JResource::Load(JFile &f)
{
	u32 ret;
	if (0 != (ret = BeginLoad(f)))
	{
		EndLoad();
		return 0;
	}
	
	return ret;
}

// Carga el recurso.
u32 JResource::Load(JFile &f, JLoadSave *where)
{
  u32 resp = 0;

  if (0 == (resp |= where->Load(f)))
  {
    loaded = true;
    data = where;
  }

  return resp;
}

// Salva el recurso
u32 JResource::Save(JFile &f)
{
  if (data)
  {
    u32 resp = 0;
    if (0 == (resp |= BeginSave(f)))
    {
      if (0 == (resp |= data->Save(f)))
      {
        resp |= EndSave(f);
      }
    }

    return resp;
  }

  // No hay datos que salvar
  return 2;
}

// Destructor
JResourceBlock::~JResourceBlock()
{
  JResIterator it;
  for (it = block.begin(); it != block.end(); ++it)
  {
    delete *it;
  }
}

// Carga el bloque de recursos y los bloques que contenga.
// Slo lee las cabeceras de los recursos contenidos en l.
u32 JResourceBlock::Load(JFile &f)
{
  u32 sz;
  if (0 == BeginLoad(f) && 
      header.type == JRES_RESOURCEBLOCK &&
      f.Read(&sz, sizeof(sz)))
  {
    JResource *res;
    u32 resp = 0;

    for (u32 i = 0; i < sz && resp == 0; ++i)
    {
      // Recupera el tipo correcto a partir de la cabecera
      res = new JResource();
			if (0 == (resp = res->Load(f)))
			{
				// Aade el recurso
				block.push_back(res);
      
				// Si es un bloque, lo carga
				if (res->Type() == JRES_RESOURCEBLOCK)
				{
					resp = res->Load(f);
				}
      
				// Se sita en el siguiente recurso
				f.Pos(res->NextOffset());
			}
			else
			{
				delete res;
			}
    }

    // Error, vuelve atrs
    if (resp != 0)
    {
      JResIterator it;
      for (it = block.begin(); it != block.end(); ++it)
      {
        delete *it;
      }

      block.clear();
  
      return 2;                 // Error al crear el recurso
    }

    EndLoad();                  // Todo OK
    
    return 0;
  }

  return 1;
}

// Carga el recurso con el nombre dado.
u32 JResourceBlock::Load(const JString &name, JFile &f, JLoadSave *where)
{
  JResIterator it;
  JResource *res = 0;
  bool found = false;
  
  for (it = block.begin(); !found && it != block.end(); ++it)
  {
    res = ((JResource *)*it);
    if (name == res->Name())
    {
      found = true;
    }
  }

  if (found)
  {
    f.Pos(res->DataOffset());
    
    return res->Load(f, where);
  }

  return 2;
}

// Carga el recurso dado.
u32 JResourceBlock::Load(JResource *res, JFile &f, JLoadSave *where)
{
  f.Pos(res->DataOffset());
  
  return res->Load(f, where);
}

// Salva el bloque de recursos. El fichero debe estar en modo
// "w" (nunca "a")
u32 JResourceBlock::Save(JFile &f)
{
  u32 sz = (u32)block.size();      // Nmero de elementos

  if (0 == BeginSave(f) &&
      f.Write(&sz, sizeof(sz)))
  {
    JResIterator it;
    for (it = block.begin(); it != block.end(); ++it)
    {
      ((JResource *)*it)->Save(f);
    }

    return EndSave(f);
  }

  return 1;
}

// Carga el fichero de recursos
u32 JFS::Load()
{
  if (!root.Loaded())
  {
    if (f.Open(name, "r"))
    {
      return root.Load(f);
    }
  }

  return 1;
}

// Salva el fichero de recursos
u32 JFS::Save()
{
  if (f.Open(name, "w"))
  {
    return root.Save(f);
  }
  
  return 1;
}

// Cambia al bloque con el nombre dado
bool JFS::CD(JString &_name)
{
  JResource *res;
  bool found = false;
  for (it = cur->List().begin(); !found && it != cur->List().end(); ++it)
  {
    res = ((JResource *)*it);
    if (res->Type() == JRES_RESOURCEBLOCK && _name == res->Name())
    {
      found = true;
      cur = (JResourceBlock *)res;
    }
  }

  return found;
}

// Carga el recurso dado.
u32 JFS::Load(JString &_name, JLoadSave *where)
{
  return cur->Load(_name, f, where);
}

// Devuelve el recurso pedido
JResource * JFS::Get(JString &_name)
{
  JResource *res;
  bool found = false;
  for (it = cur->List().begin(); !found && it != cur->List().end(); ++it)
  {
    res = ((JResource *)*it);
    if (res->Type() != JRES_RESOURCEBLOCK && _name == res->Name())
    {
      found = true;
    }
  }

  if (found)
  {
    if (!res->Loaded())
    {
      if (0 != Load(res))
        return 0;
    }

    return res;
  }

  return 0;
}

// Borra el recurso dado
bool JFS::Delete(JString &_name)
{
  JResource *res;
  bool found = false;
  for (it = cur->List().begin(); !found && it != cur->List().end(); ++it)
  {
    res = ((JResource *)*it);
    if (res->Type() != JRES_RESOURCEBLOCK && _name == res->Name())
    {
      found = true;
      cur->List().erase(it);
    }
  }

  if (found)
  {
    return true;
  }

  return false;
}
