/*
 *  JLib - Jacob's Library.
 *  Copyright (C) 2003, 2004  Juan Carlos Seijo Prez
 * 
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 * 
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 * 
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 *  Juan Carlos Seijo Prez
 *  jacob@mainreactor.net
 */

/** Clase para gestin de una ventana grfica.
 * @file    JApp.h
 * @author  Juan Carlos Seijo Prez
 * @date    01/04/2003
 * @version 0.0.1 - Primera versin - 01/04/2003
 */

#ifndef _JAPP_INCLUDED
#define _JAPP_INCLUDED

#include <JLib/Util/JTypes.h>
#include <JLib/Util/JString.h>
#include <JLib/Util/JTimer.h>
#include <JLib/Graphics/JFont.h>
#include <JLib/Sound/JMixer.h>
#include <SDL/SDL.h>
#include <stdio.h>
#include <stdlib.h>

#ifdef _WIN32
#else
#include <unistd.h>
#define _sleep sleep
#endif

/** Clase base de aplicacin. Permite mostrar una ventana
 * y actualizar su contenido frame a frame. Tras construir el
 * objeto es necesario llamar a Init() y despus a MainLoop(), donde
 * se quedar la ejecucin hasta que haya un error o se cierre la ventana.
 */
class JApp
{
protected:
  bool active;                    /**< Determina si est en primer plano la aplicacin */
  bool paused;                    /**< Determina si est en pausa la aplicacin */
  bool endLoop;                   /**< Determina si debe terminar la aplicacin */
  JString title;                  /**< Ttulo de la ventana */
  s32 width;                      /**< Anchura de ventana */
  s32 height;                     /**< Altura de ventana */
  s32 depth;                      /**< Profundidad de color en bytes */
  bool fullScreen;                /**< Indicador de pantalla completa */
  bool doInput;                   /**< Indicador de captura de eventos (true captura, flase no) */
  u32 flags;                      /**< Flags de SDL adicionales */
  u32 appPauseTime;               /**< Tiempo en que se pausa la aplicacin (ms) */
	u8 *keys;                       /**< Estado del teclado. */
	SDLMod keyMods;                 /**< Modificadores de teclado. */
	s32 numKeys;                    /**< Nmero de teclas del teclado. */
	s32 mouseX;                     /**< Posicin x del ratn. */
	s32 mouseY;                     /**< Posicin y del ratn. */
	u8 mouseBt;                     /**< Mscara de botones del ratn. */
	s32 mouseRelX;                  /**< Incremento x del ratn. */
	s32 mouseRelY;                  /**< Incremento y del ratn. */
	JTimer timer;                   /**< Timer de la aplicacin. */
	s32 fps;                        /**< Frames por segundo de reproduccin. */
	bool dumpVideoInfo;             /**< Determina si debe mostrar informacin del vdeo al inicio. */

	JMixer mixer;                   /**< Mixer de audio. */
	bool soundEnabled;              /**< Flag de reproduccin de sonido. */

	char *iconName;                 /**< Icono de la aplicacin. */
	
	static JApp* thisApp;           /**< Puntero a esta aplicacin. */

  void (*__OnActive)(bool active, s32 state);
  void (*__OnKeyUp )(SDL_keysym key);
  void (*__OnKeyDown)(SDL_keysym key);
  void (*__OnMouseMove)(s32 x, s32 y, s32 bt, s32 xRel, s32 yRel);
  void (*__OnMouseDown)(s32 bt, s32 x, s32 y);
  void (*__OnMouseUp)(s32 bt, s32 x, s32 y);
  void (*__OnJoyMove)(s32 joyNum, s32 axis, s32 value);
  void (*__OnJoyBall)(s32 joyNum, s32 ballIndex, s32 xRel, s32 yRel);
  void (*__OnJoyHat)(s32 joyNum, s32 hatIndex, s32 value);
  void (*__OnJoyUp)(s32 joyNum, s32 btIndex);
  void (*__OnJoyDown)(s32 joyNum, s32 btIndex);
  void (*__OnQuit)();
  void (*__OnSysWM)();
  void (*__OnResize)(s32 w, s32 h);
  void (*__OnExpose)();
  void (*__OnUser)(s32 code, void *data1, void *data2);

  SDL_Surface *screen;            /**< Framebuffer */

public:
  /** Construye el objeto. El mezclador de audio se debe inicializar en la clase hija.
   * @param  strTitle Ttulo de la ventana.
   * @param  w Anchura de la ventana
   * @param  h Altura de la ventana
   * @param  fullScr Indicador de pantalla completa
   * @param  _depth Profundidad de color en bits
   * @param  otherFlags Flags adicionales de SDL
   */
  JApp(const JString& strTitle, s32 w = 1024, s32 h = 768, bool fullScr = true, s32 _depth = 16, u32 otherFlags = 0);

  /** Comprueba los eventos.
   * @return <b>true</b> si todo va bien, <b>false</b> en caso de no poder recibir eventos.
   */
  virtual bool UpdateEvents();

  /** Inicializa la aplicacin. Crea la ventana con los parmetros dados en construccin.
   * @return <b>true</b> si todo fue bien <b>false</b> si no se pudo inicializar la aplicacin.
   */
  virtual bool Init();

  /** Actualiza los objetos del juego (Grficos, sonido, IA, etc.). Debe ser implementado
   * en la clase hija. JApp llamar a este mtodo antes de Draw() en la clase hija.
   * @return <b>true</b> si todo va bien, <b>false</b> si hay algn error y debe cerrarse la aplicacin.
   */
  virtual bool Update() = 0;

  /** Presenta el juego en pantalla. Debe implementarse en la clase hija.
   * @return <b>true</b> si todo va bien, <b>false</b> si hay algn error y debe cerrarse la aplicacin.
   */
  virtual bool Draw() = 0;

	/** Bucle principal del juego. La implementacin por defecto llama a Update() y
   * en caso de que devuelva <b>true</b> llama a Draw(). La aplicacin permanece en
   * bucle hasta que es finaliza con Exit() o con el fallo de Update() o Draw().
   * @return Cdigo de salida de la aplicacin.
   */
	virtual s32 MainLoop();

  /** Devuelve la anchura de la ventana.
   * @return Anchura de la ventana.
   */
  s32 Width() {return width;}

  /** Devuelve la altura de la ventana.
   * @return Altura de la ventana.
   */
  s32 Height() {return height;}

  /** Cambia el tamao de la ventana
   * @param  w Nueva anchura de la ventana
   * @param  h Nueva altura de la ventana
   * @param  fullScreen <b>true</b> si se quiere pantalla completa, <b>false</b> para
   * ventana.
   */
  void Resize(s32 w, s32 h, bool fullScreen);

  /** Determina si est en pantalla completa.
   * @return <b>true</b> si est en pantalla completa, <b>false</b> si est en ventana.
   */
  bool IsFullscreen() {return fullScreen;}
  
  /** Devuelve la profundidad de color
   * @return Profudidad de color en bits. Tpicamente ser 16, 24  32.
   */
  s32 Depth() {return depth;}

  /** Termina la aplicacin.
   */
  void Exit() {endLoop = true;}

  /** Determina si est en primer plano la aplicacin
   * @return <b>true</b> si est en primer plano <b>false</b> si no.
   */
  bool IsActive() {return active;}
  
  /** Determina si est en pausa la aplicacin
   * @return <b>true</b> si est en pausa <b>false</b> si no.
   */
  bool IsPaused() {return paused;}
  
  /** Pausa la aplicacin
   * @param  doPause <b>true</b> pausa la aplciacin, <b>false</b> sale de pausa.
   * @return si entra en modo pausa (doPause es <b>true</b>), devuelve el nmero
   * de milisegundos transcurridos desde el inicio de la aplicacin. Si sale
   * del modo pausa (doPause <b>false</b>), devuelve el nmero de milisegundos
   * transcurridos mientras estaba en modo pausa.
   */
  u32 Pause(bool doPause);

  /** Devuelve el ttulo de la ventana
   * @return Ttulo de la ventana.
   */
  const JString &Title() {return title;}

  /** Establece el ttulo de la ventana
   * @param  newTitle Nuevo ttulo de la ventana.
   */
  void Title(JString &newTitle) {title = newTitle; SDL_WM_SetCaption(title, 0);}

  /** Devuelve el tiempo de ejecucin de aplicacin en milisegundos.
   * @return Tiempo de ejecucin de aplicacin en milisegundos.
   */
  u32 AppTime() {return SDL_GetTicks();}

  /** Devuelve la superficie del framebuffer. la superficie se crea tras la llamada a 
   * Init(). En caso de que falle Init() este mtodo podra devolver 0.
   * @return Devuelve la superficie del framebuffer.
   */
  SDL_Surface * Screen() {return screen;}

  /** Establece si se deben capturar eventos
   * @param  b <b>true</b> si se deben capturar <b>false</b> si no.
   * @return 
   */
  void DoInput(bool b) {doInput = b;}

  /** Devuelve si se deben recoger eventos o no
   * @return <b>true</b> si se deben capturar <b>false</b> si no.
   */
  bool DoInput() {return doInput;}

  /** Establece la callback ante el evento de actividad de la ventana.
	 * @param _OnActive Callback.
	 */
  void SetOnActive    (void (*_OnActive)(bool active, s32 state)) 
	{__OnActive = _OnActive;}

  /** Establece la callback ante el evento de soltar una tecla.
	 * @param  _OnKeyUp Callback.
	 */
  void SetOnKeyUp     (void (*_OnKeyUp )(SDL_keysym key)) 
	{__OnKeyUp  = _OnKeyUp ;}

  /** Establece la callback ante el evento de pulsar una tecla.
	 * @param  _OnKeyDown Callback.
	 */
  void SetOnKeyDown   (void (*_OnKeyDown)(SDL_keysym key)) 
	{__OnKeyDown = _OnKeyDown;}

  /** Establece la callback ante el evento de mover el ratn.
	 * @param  _OnMouseMove Callback.
	 */
  void SetOnMouseMove (void (*_OnMouseMove)(s32 x, s32 y, s32 bt, s32 xRel, s32 yRel)) 
	{__OnMouseMove = _OnMouseMove;}

  /** Establece la callback ante el evento de pulsar botn del ratn.
	 * @param  _OnMouseDown Callback.
	 */
  void SetOnMouseDown (void (*_OnMouseDown)(s32 bt, s32 x, s32 y)) 
	{__OnMouseDown = _OnMouseDown;}

  /** Establece la callback ante el evento de soltar botn del ratn.
	 * @param  _OnMouseUp Callback.
	 */
  void SetOnMouseUp   (void (*_OnMouseUp)(s32 bt, s32 x, s32 y)) 
	{__OnMouseUp = _OnMouseUp;}

  /** Establece la callback ante el evento de movimiento de joystick.
	 * @param  _OnJoyMove Callback.
	 */
  void SetOnJoyMove   (void (*_OnJoyMove)(s32 joyNum, s32 axis, s32 value)) 
	{__OnJoyMove = _OnJoyMove;}

  /** Establece la callback ante el evento de bola de joystick movida.
	 * @param  _OnJoyBall Callback.
	 */
  void SetOnJoyBall   (void (*_OnJoyBall)(s32 joyNum, s32 ballIndex, s32 xRel, s32 yRel)) 
	{__OnJoyBall = _OnJoyBall;}

  /** Establece la callback ante el evento de botn superior de joystick pulsado.
	 * @param  _OnJoyHat Callback.
	 */
  void SetOnJoyHat    (void (*_OnJoyHat)(s32 joyNum, s32 hatIndex, s32 value)) 
	{__OnJoyHat = _OnJoyHat;}

  /** Establece la callback ante el evento de botn de joystick soltado.
	 * @param  _OnJoyUp Callback.
	 */
  void SetOnJoyUp     (void (*_OnJoyUp)(s32 joyNum, s32 btIndex)) 
	{__OnJoyUp = _OnJoyUp;}

  /** Establece la callback ante el evento de botn de joystick pulsado.
	 * @param  _OnJoyDown Callback.
	 */
  void SetOnJoyDown   (void (*_OnJoyDown)(s32 joyNum, s32 btIndex)) 
	{__OnJoyDown = _OnJoyDown;}

  /** Establece la callback ante el evento de salida.
	 * @param  _OnQuit Callback.
	 */
  void SetOnQuit      (void (*_OnQuit)()) 
	{__OnQuit = _OnQuit;}

  /** Establece la callback ante el evento de gestor de ventanas.
	 * @param  _OnSysWM Callback.
	 */
  void SetOnSysWM     (void (*_OnSysWM)()) 
	{__OnSysWM = _OnSysWM;}

  /** Establece la callback ante el evento de cambio de tamao.
	 * @param  _OnResize Callback.
	 */
  void SetOnResize    (void (*_OnResize)(s32 w, s32 h)) 
	{__OnResize = _OnResize;}

  /** Establece la callback ante el evento de exposicin.
	 * @param  _OnExpose Callback.
	 */
  void SetOnExpose    (void (*_OnExpose)()) 
	{__OnExpose = _OnExpose;}

  /** Establece la callback ante el evento de usuario.
	 * @param  _OnUser Callback.
	 */
  void SetOnUser(void (*_OnUser)(s32 code, void *data1, void *data2)) 
	{__OnUser = _OnUser;}

	/** Devuelve la coordenada x del ratn.
	 * @return Coordenada x del ratn.
	 */
	s32 MouseX() {return mouseX;}

	/** Devuelve la coordenada y del ratn.
	 * @return Coordenada y del ratn.
	 */
	s32 MouseY() {return mouseY;}

	/** Devuelve la coordenada relativa x del ratn.
	 * @return Coordenada relativa x del ratn.
	 */
	s32 MouseRelX() {return mouseRelX;}

	/** Devuelve la coordenada relativa y del ratn.
	 * @return Coordenada relativa y del ratn.
	 */
	s32 MouseRelY() {return mouseRelY;}

	/** Devuelve la mscara de estado de los botones del ratn.
	 * @return Mscara de estado de los botones del ratn.
	 */
	u8 MouseBt() {return mouseBt;}

	/** Devuelve el array de estado de teclas. Se determina el estado de una tecla
	 * haciendo arr[SDLK_XXX], con arr el array del estado de teclas y SDLK_XXX
	 * la tecla a consultar. Si el valor es 0, no est pulsada, si no, s lo est.
	 * @return Array de estado de teclas.
	 */
	u8 * Keys() {return keys;}

	/** Devuelve la mscara de bits de modificadores de teclas. Los 
	 * bits de que se compone tienen la forma KMOD_XXX, con XXX=LCTRL, etc.
	 * @return Mscara de bits de modificadores de teclas.
	 */
	SDLMod KeyMods() {return keyMods;}

	/** Actualiza la region dada de la ventana principal. Si se pasa cero a todos 
	 * los parmetros se actualiza toda la ventana.
	 * @param  x Coordenada x de la regin.
	 * @param  y Coordenada y de la regin.
	 * @param  w Anchura de la regin.
	 * @param  h Altura de la regin.
	 */
	void UpdateRect(s32 x, s32 y, s32 w, s32 h) {SDL_UpdateRect(screen, x, y, w, h);}

	/** Actualiza las regiones dadas de la ventana principal.
	 * @param  num Nmero de regiones a actualizar.
	 * @param  rc Array de regiones a actualizar.
	 */
	void UpdateRects(s32 num, SDL_Rect *rc) {SDL_UpdateRects(screen, num, rc);}

	/** Intercambia el back-buffer con el front-buffer en caso de soportar doble buffer
	 * o bien hace el equivalente a UpdateRect(0, 0, 0, 0).
	 * @return <b>true</b> si todo va bien, <b>false</b> si hubo algn error.
	 */
	bool Flip() {return -1 != SDL_Flip(screen);}

  /** Muestra una ventana con una imagen fija.
   * @param  strTitle Ttulo de la ventana de mensaje.
   * @param  content Imagen a mostrar.
   */
  static void ShowWindow(const s8 *strTitle, JImage *content);

	/** Devuelve el mezclador de audio. Se debe inicializar en la clase hija.
	 * @return Mezclador de audio.
	 */
	JMixer & Mixer() {return mixer;}

	/** Devuelve esta aplicacin.
	 * @return Puntero a esta aplicacin.
	 */
	static JApp * App() {return thisApp;}

	/** Ajusta el nmero de frames por segundo de reproduccin.
	 * @param  newFPS Nuevo nmero de frames por segundo de reproduccin.
	 */
	void FPS(s32 newFPS);

	/** Devuelve el nmero de frames por segundo de reproduccin.
	 * @return nmero de frames por segundo de reproduccin.
	 */
	s32 FPS() {return fps;}
	
	/** Devuelve un puntero de solo lectura a la informacin del modo actual de video.
	 * Si se llama antes de Init o Resize contiene las mejores capacidades del video.
	 * @return Informacin sobre el modo actual o sobre el vdeo en general.
	 */
	const SDL_VideoInfo *VideoInfo() {return SDL_GetVideoInfo();}

	/** Muestra en la salida estndar una descripcin de la informacin de VideoInfo().
	 */
	void DumpVideoInfo();

	/** Parses argument.
	 * @param  args Command line arguments.
	 * @param  argc Arguments left.
	 * @return Number of parameters used.
	 */
	virtual int ParseArg(char *args[], int argc);
	
	/** Shows the usage string.
	 */
	virtual void PrintUsage(char *program);

	/** Parses the application arguments.
	 * @param  argc Command line argument count.
	 * @param  argv Command line arguments.
	 */
	void ParseArgs(s32 argc, char **argv);
	
	/** Establece el flag de sonido activo.
	 */
	void SoundEnabled(bool enable) {soundEnabled = (enable && mixer.Valid());}

	/** Establece el modo de visibilidad del cursor del ratn.
	 * @param b <b>true</b> hace que se muestre, <b>false</b> lo oculta.
	 */
	void MouseCursor(bool b) {SDL_ShowCursor(b ? SDL_ENABLE : SDL_DISABLE);}
	
	/** Devuelve el modo de visibilidad del cursor del ratn.
	 * @param b <b>true</b> si se muestra, <b>false</b> si no.
	 */
	bool MouseCursor() {return SDL_ENABLE == SDL_ShowCursor(SDL_QUERY);}
 
	/** Establece el nombre del fichero de imagen con el icono. Debe ser llamada antes de Init().
	 * En Windows, el fichero debe tener un tamao de 32x32.
	 * @param  filename Nombre del fichero con el icono, si es cero, no pone ningn icono.
	 */
	void Icon(const char *filename);
	
	/** Devuelve si el sonido est activo o no. Que est activo quiere decir que el mixer se
	 * inicializ correctamente y adems que se quiere escuchar sonido. Que est inactivo puede
	 * deberse a que solicitaron no escuchar sonidos o bien que el mixer no se inicializ correctamente.
	 * @return <b>true</b> si est activo, <b>false</b> si no.
	 * @see JMixer::Valid(), Mixer()
	 */
	bool SoundEnabled() {return soundEnabled;}

  /** Destruye el objeto.
   */
  void Destroy();

  /** Destruye el objeto
   */
  virtual ~JApp();
};

#endif  // JAPP_INCLUDED
