/*
 * Hexxagon board game.
 * Copyright (C) 2001 Erik Jonsson.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 * Email erik@nesqi.homeip.net
 * 
 */

#include "render.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

guchar *board_pic;
guchar *thinking_pic;
guchar *font_pic;
guchar *rgbbuf;

struct point lookup[61];

void apply_thinking()
{
	int i, j;

	for(i = 0; i < THINK_HEIGHT; i++)
	{
		for(j = 0; j < THINK_WIDTH; j++)
		{
			memcpy(rgbbuf + (j + 212 + (i + 200) * IMAGE_WIDTH) * 3,
				   thinking_pic + (j + i * THINK_WIDTH) * 3,
				   3);
		}
	}
}

void renderDigit(int x, int y, int number)
{
	int offset = number * FONT_CHAR_W;
	int i, j;

	for(i = 0; i < FONT_CHAR_H; i++)
	{
		for(j = 0; j < FONT_CHAR_W; j++)
		{
			memcpy(rgbbuf + (j + x + (i + y) * IMAGE_WIDTH) * 3,
				   font_pic + (j + i * FONT_WIDTH + offset) * 3, 
				   3);
		}
	}
}

void renderScore(HexxagonBoard board, int turn)
{
	int left, right;

	left = board.countBricks(1 + turn);
	right = board.countBricks(2 - turn);
	/* left */
	if(left > 9)
	{
		renderDigit(20, 20, (left - (left % 10)) / 10);
		renderDigit(60, 20, left % 10);
	} else {
		renderDigit(20, 20, left);
	}

	/* right */
	if(right > 9)
		renderDigit(IMAGE_WIDTH - 2 * FONT_CHAR_W - 20, 
					20, (right - (right % 10)) / 10);
	renderDigit(IMAGE_WIDTH - 2 * FONT_CHAR_W + 20, 20, right % 10);
}

void fill_cell(int no, int color)
{
	guchar r = 0, g = 0, b = 0, col;
	guchar *pos1, *pos2;
	int offs;

	if(color == 1) /* pl 1.*/
	{
		r = 165; g = 23; b = 209;
	}
	else if(color == 2) /* pl. 2. */
	{
		r = 204; g = 30; b = 30;		
	}
	else if(color == 3) /* clone */
	{
		r = 117; g = 183; b = 117;		
	}
	else if(color == 4) /* jump */
	{
		r = 85; g = 150; b = 85;		
	}

	pos1 = board_pic + (lookup[no].x + lookup[no].y * IMAGE_WIDTH + 5) * 3;
	col = pos1[2];

	while(pos1[2] == col)
	{
		pos2 = pos1;
		
		while(*(pos2 - 1) == col)
			pos2 -= 3;

		while(pos2[2] == col)
		{
			offs = pos2 - board_pic;
			(rgbbuf + offs)[0] = r;
			(rgbbuf + offs)[1] = g;
			(rgbbuf + offs)[2] = b;
			pos2 += 3;		
		}

		pos1 += IMAGE_WIDTH * 3;
	}
		
}

void render_board(HexxagonBoard board, int turn)
{
	BitBoard64 b, c;
	int i;

	b = board.getBitBoardBoard();
	c = board.getBitBoardColor();

	memcpy(rgbbuf, board_pic, IMAGE_WIDTH * IMAGE_HEIGHT * 3);

	for(i = 0; i < 61; i++)
	{
		if(b.getBit(i))
		{
			if(c.getBit(i))
				fill_cell(i, turn + 1);
			else
				fill_cell(i, 2 - turn);
		}
	}

	renderScore(board, turn);
}

guchar *loadRAW(char *filename, int size)
{
	FILE *file;
	guchar *buffer;

	file = fopen(filename, "rb");
	if(!file)
	{
		printf("Could not open image.\n");
		return 0;
	}

	buffer = (guchar*) malloc(size);
	
	fread(buffer, size, 1, file);

	fclose(file);

	return buffer;
}

int initPictures()
{
	rgbbuf = (guchar*) malloc(IMAGE_WIDTH * IMAGE_HEIGHT * 3);

	if(!(board_pic = loadRAW("data/board.raw", IMAGE_WIDTH * IMAGE_HEIGHT * 3)))
		return 1;
	if(!(thinking_pic = loadRAW("data/thinking.raw", THINK_WIDTH * THINK_HEIGHT * 3)))
		return 1;
	if(!(font_pic = loadRAW("data/font.raw", IMAGE_WIDTH * IMAGE_HEIGHT * 3)))
		return 1;

	return 0;
}

int get_cell_from_cord(int x, int y)
{
	int color;

	if(board_pic[(x + y * IMAGE_WIDTH) * 3] != 79
	   || board_pic[(x + y * IMAGE_WIDTH) * 3 + 1] != 90)
		return -1;

	color = board_pic[(x + y * IMAGE_WIDTH) * 3 + 2];

	if(color >= 158 && color <= 218)
		return color - 158;
	else 
		return -1;
}

void init_lookup()
{
	memset(lookup, 0, sizeof(lookup));

	int i, j, c;

	for(i = 0; i < IMAGE_HEIGHT; i++)
	{
		for(j = 0; j < IMAGE_WIDTH; j++)
		{
			if((c = get_cell_from_cord(j, i)) >= 0)
			{
				if(lookup[c].x == 0 && lookup[c].y == 0)
				{
					lookup[c].x = j;
					lookup[c].y = i;
				}
			}
		}
	}
}

