/* -*- C++ -*- */

#ifndef _SUPERBLOCKHEAP_H_
#define _SUPERBLOCKHEAP_H_

#include <assert.h>
#include <stddef.h>
#include <new.h>


template <int SuperblockSize>
class SuperblockHeapBase {
public:

  SuperblockHeapBase (void)
    : _freeList (NULL),
      _numAvailable (0),
      _numAllocated (0)
  {}

  virtual ~SuperblockHeapBase (void) {}

  static inline size_t getSize (void * ptr) {
    return ObjectHeader::getSize(ptr);
  }

  // Define an abbreviation for this object's type.
  typedef SuperblockHeapBase<SuperblockSize> thisType;

  thisType * getNext (void) { return _next; }
  thisType * getPrev (void) { return _prev; }
  void setNext (thisType * n) { _next = n; }
  void setPrev (thisType * p) { _prev = p; }

  // Return the heap for a given object.
  static SuperblockHeapBase<SuperblockSize> * getHeap (void * ptr) {
    return ObjectHeader::getHeap (ptr);
  }


protected:

  class ObjectHeader {
  public:

    ObjectHeader (SuperblockHeapBase<SuperblockSize> * mh)
      : myHeap (mh),
	next (NULL)
    {}

    // Get the next object in the list.
    ObjectHeader * getNext (void) { return next; }

    // Set the next object.
    void setNext (ObjectHeader * ptr) { next = ptr; }

    // Return the heap for a given object.
    static SuperblockHeapBase<SuperblockSize> * getHeap (void * ptr) {
      return ((ObjectHeader *) ptr - 1)->myHeap;
    }

    static size_t getSize (void * ptr) {
      return ((ObjectHeader *) ptr - 1)->objsize;
    }

    void setSize (size_t sz) { objsize = sz; }

  private:

    // The order of the fields should NOT be changed.

    union {
      size_t objsize;
      ObjectHeader * next;
    };
    SuperblockHeapBase<SuperblockSize> * myHeap;
  };


  // Get one block.
  ObjectHeader * getBlock (void) {
    ObjectHeader * ptr = _freeList;
    if (ptr != NULL) {
      _freeList = ptr->getNext();
    }
    return ptr;
  }

  // Put one back.
  void putBlock (ObjectHeader * obj) {
    obj->setNext (_freeList);
    _freeList = obj;
  }


  int		_numAvailable;	// The number of available blocks.
  int		_numAllocated;  // The number of blocks "allocated".
  ObjectHeader * _freeList;	// A pointer to the first free block.
  thisType *  	_next;		// The next superblock in the list.
  thisType *  	_prev;		// The previous superblock in the list.

};

template <int SuperblockSize, int ObjectSize>
class SuperblockHeap : public SuperblockHeapBase<SuperblockSize> {
public:

  SuperblockHeap (void)
    : SuperblockHeapBase<SuperblockSize>()
  {}


  ~SuperblockHeap (void) {}

  inline void * malloc (size_t sz) {
    // printf ("heap malloc.\n");
    assert (sz <= ObjectSize);
    // Return NULL if there are no blocks available.
    ObjectHeader * p = getBlock();
    if (p == NULL) {
      assert (_numAvailable == 0);
      // Have we allocated all of the blocks yet?
      if (_numAllocated == getNumObjects()) {
	// We have.
	return NULL;
      }
      // Allocate another one and return it.
      ObjectHeader * thisObject = getObject(_numAllocated);
      _numAllocated++;
      new ((void *) thisObject) ObjectHeader (this);
      p = thisObject;
    } else {
      _numAvailable--;
    }
    p->setSize (ObjectSize);
    return (void *) (p + 1);
  }

  inline void free (void * ptr) {
    if (ptr == NULL)
      return;
    // Turn ptr into an object header.
    ObjectHeader * obj = ((ObjectHeader *) ptr - 1);
    // Check to make sure that the freed block actually came from this heap.
    if (this != obj->getHeap(ptr)) {
      assert (0);
      return;
    } else {
      assert (_numAvailable < _numAllocated);
      _numAvailable++;
      putBlock (obj);
    }
  }

  inline int isFree (void) {
    // printf ("navail = %d, nalloc = %d\n", _numAvailable, _numAllocated);
    return (_numAvailable == _numAllocated);
  }

  inline static size_t getSize (void *) {
    return ObjectSize;
  }

private:

  inline static int getNumObjects (void) {
    return (SuperblockSize - sizeof(thisType)) / (ObjectSize + sizeof(ObjectHeader));
  }

  ObjectHeader * getObject (int i) {
    assert (i >= 0);
    assert (i < getNumObjects());
    return (ObjectHeader *) ((char *) (this + 1) + (sizeof(ObjectHeader) + ObjectSize) * i);
  }

  inline size_t getObjectSize (void) {
    return ObjectSize;
  }

  // Define an abbreviation for this object's type.
  typedef SuperblockHeap<SuperblockSize, ObjectSize> thisType;
};


#endif
