// -*- C++ -*-

#ifndef _EMPTYCLASS_H_
#define _EMPTYCLASS_H_

#include "check.h"
#include "array.h"
#include "hldefines.h"

/**
 * @class EmptyClass
 * @brief Maintains superblocks organized by emptiness.
 */

template <class SuperblockType_,
	  int EmptinessClasses,
	  size_t SuperblockSize>
class EmptyClass {
public:

  typedef SuperblockType_ SuperblockType;

  EmptyClass (void)
  {
    for (int i = 0; i <= EmptinessClasses + 1; i++) {
      _available(i) = NULL;
    }
  }

  void dumpStats (void) {
    for (int i = 0; i <= EmptinessClasses + 1; i++) {
      SuperblockType * s = _available(i);
      if (s) {
	fprintf (stderr, "EmptyClass: emptiness class = %d\n", i);
	while (s) {
	  s->dumpStats();
	  s = s->getNext();
	}
      }
    }
  }


  SuperblockType * getEmpty (void) {
    Check<EmptyClass, MyChecker> check (this);
    SuperblockType * s = _available(0);
    if (s && 
	(s->getObjectsFree() == s->getTotalObjects())) {
      // Got an empty one. Remove it.
      _available(0) = s->getNext();
      if (_available(0)) {
	_available(0)->setPrev (NULL);
      }
      s->setPrev (NULL);
      s->setNext (NULL);
      return s;
    }
    return NULL;
  }

  SuperblockType * get (void) {
    Check<EmptyClass, MyChecker> check (this);
    // Return as empty a superblock as possible
    // by iterating from the emptiest to the fullest available class.
    for (int n = 0; n < EmptinessClasses + 1; n++) {
      SuperblockType * s = _available(n);
      while (s) {
	assert (s->isValidSuperblock());
	// Got one. Remove it.
	_available(n) = s->getNext();
	if (_available(n)) {
	  _available(n)->setPrev (NULL);
	}
	s->setPrev (NULL);
	s->setNext (NULL);

#ifndef NDEBUG
	// Verify that this superblock is *gone* from the lists.
	for (int z = 0; z < EmptinessClasses + 1; z++) {
	  SuperblockType * p = _available(z);
	  while (p) {
	    assert (p != s);
	    p = p->getNext();
	  }
	}
#endif

	// Ensure that we return a superblock that is as free as
	// possible.
	int cl = getFullness (s);
	if (cl > n) {
	  put (s);
	  SuperblockType * sNew = _available(n);
	  assert (s != sNew);
	  s = sNew;
	} else {
	  return s;
	}
      }
    }
    return NULL;
  }

  void put (SuperblockType * s) {
    Check<EmptyClass, MyChecker> check (this);

#ifndef NDEBUG
    // Check to verify that this superblock is not already on one of the lists.
    for (int n = 0; n < EmptinessClasses + 1; n++) {
      SuperblockType * p = _available(n);
      while (p) {
	assert (p != s);
	p = p->getNext();
      }
    }
#endif

    // Put on the appropriate available list.
    int cl = getFullness (s);

    //    printf ("put %x, cl = %d\n", s, cl);
    s->setPrev (NULL);
    s->setNext (_available(cl));
    if (_available(cl)) {
      _available(cl)->setPrev (s);
    }
    _available(cl) = s;
  }

  MALLOC_FUNCTION void * malloc (size_t sz);

  INLINE void free (void * ptr) {
    Check<EmptyClass, MyChecker> check (this);
    SuperblockType * s = getSuperblock (ptr);
    int oldCl = getFullness (s);
    s->free (ptr);
    int newCl = getFullness (s);

    if (oldCl != newCl) {
      // Transfer.
      assert (s->getObjectsFree() > 0);
      SuperblockType * prev = s->getPrev();
      SuperblockType * next = s->getNext();
      if (prev) { prev->setNext (next); }
      if (next) { next->setPrev (prev); }
      if (s == _available(oldCl)) {
	assert (prev == NULL);
	_available(oldCl) = next;
      }
      s->setNext (_available(newCl));
      s->setPrev (NULL);
      if (_available(newCl)) { _available(newCl)->setPrev (s); }
      _available(newCl) = s;
    }
  }

  /// Find the superblock (by bit-masking) that holds a given pointer.
  static INLINE SuperblockType * getSuperblock (void * ptr) {
    SuperblockType * s = reinterpret_cast<SuperblockType *>(reinterpret_cast<size_t>(ptr) & ~(SuperblockSize-1));
    assert (s->isValidSuperblock());
    return s;
  }

private:

  static INLINE int getFullness (SuperblockType * s) {
    // Completely full = EmptinessClasses + 1
    // Completely empty (all available) = 0
    int total = s->getTotalObjects();
    int free = s->getObjectsFree();
    if (total == free) {
      return 0;
    } else {
      return 1 + (EmptinessClasses * (total - free)) / total;
    }
  }

  /// Forward declarations for the sanity checker.
  /// @sa Check
  class MyChecker;
  friend class MyChecker;

  /// Precondition and postcondition checking.
  class MyChecker {
  public:
#ifndef NDEBUG
    static void precondition (EmptyClass * e) {
      e->sanityCheckPre();
    }
    static void postcondition (EmptyClass * e) {
      e->sanityCheck();
    }
#else
    static void precondition (EmptyClass *) {}
    static void postcondition (EmptyClass *) {}
#endif
  };

  void sanityCheckPre (void) { sanityCheck(); }

  void sanityCheck (void) {
    for (int i = 0; i <= EmptinessClasses + 1; i++) {
      SuperblockType * s = _available(i);
      while (s) {
	assert (getFullness(s) == i);
	s = s->getNext();
      }
    }
  }

  /// The bins of superblocks, by emptiness class.
  /// @note index 0 = completely empty, EmptinessClasses + 1 = full
  HL::Array<EmptinessClasses + 2, SuperblockType *> _available;

};



#endif
