/*  -*-objc-*-
 *  IconsViewer.m: Implementation of the IconsViewer Class 
 *  of the GNUstep GWorkspace application
 *
 *  Copyright (c) 2001 Enrico Sersale <enrico@imago.ro>
 *  
 *  Author: Enrico Sersale
 *  Date: August 2001
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#import <Foundation/Foundation.h>
#import <AppKit/AppKit.h>
#include "IconsViewer.h"
#include "IconsPath.h"
#include "PathIcon.h"
#include "IconsPanel.h"
#include "GWScrollView.h"
#include "Functions.h"
#include "GNUstep.h"

/* Notifications */
#ifndef GNUSTEP
	NSString *GWFileSystemWillChangeNotification = @"GWFileSystemWillChangeNotification";
	NSString *GWFileSystemDidChangeNotification = @"GWFileSystemDidChangeNotification";
	NSString *GWDidSetFileAttributesNotification = @"GWDidSetFileAttributesNotification";
	NSString *GWSortTypeDidChangeNotification = @"GWSortTypeDidChangeNotification";
#else
	extern NSString *GWFileSystemWillChangeNotification;
	extern NSString *GWFileSystemDidChangeNotification;
	extern NSString *GWDidSetFileAttributesNotification;
	extern NSString *GWSortTypeDidChangeNotification;
#endif

/* File Operations */
#ifndef GNUSTEP
	NSString *GWorkspaceRenameOperation = @"GWorkspaceRenameOperation";
	NSString *GWorkspaceCreateFileOperation = @"GWorkspaceCreateFileOperation";
	NSString *GWorkspaceCreateDirOperation = @"GWorkspaceCreateDirOperation";
	NSString *GWorkspaceRecycleOutOperation = @"GWorkspaceRecycleOutOperation";
	NSString *GWorkspaceEmptyRecyclerOperation = @"GWorkspaceEmptyRecyclerOperation";
#else
	extern NSString *GWorkspaceRenameOperation;
	extern NSString *GWorkspaceCreateFileOperation;
	extern NSString *GWorkspaceCreateDirOperation;
	extern NSString *GWorkspaceRecycleOutOperation;
	extern NSString *GWorkspaceEmptyRecyclerOperation;
#endif

/* File Watcher Strings */
#ifndef GNUSTEP
	NSString *GWFileWatcherFileDidChangeNotification = @"GWFileWatcherFileDidChangeNotification";
	NSString *GWWatchedDirectoryDeleted = @"GWWatchedDirectoryDeleted";
	NSString *GWFileDeletedInWatchedDirectory = @"GWFileDeletedInWatchedDirectory";
	NSString *GWFileCreatedInWatchedDirectory = @"GWFileCreatedInWatchedDirectory";
#else
	extern NSString *GWFileWatcherFileDidChangeNotification;
	extern NSString *GWWatchedDirectoryDeleted;
	extern NSString *GWFileDeletedInWatchedDirectory;
	extern NSString *GWFileCreatedInWatchedDirectory;
#endif

@implementation IconsViewer

- (void)dealloc
{
  [[NSNotificationCenter defaultCenter] removeObserver: self];
  TEST_RELEASE (rootPath);
  TEST_RELEASE (selectedPaths);
  TEST_RELEASE (watchedPaths);
  TEST_RELEASE (iconsPath);
  TEST_RELEASE (pathsScroll);
  TEST_RELEASE (panelScroll);  
  TEST_RELEASE (panel);
  [super dealloc];
}

- (id)init
{
	self = [super initWithFrame: NSZeroRect];
  rootPath = nil;
  selectedPaths = nil;
  watchedPaths = nil;
  iconsPath = nil;
  pathsScroll = nil;
  panelScroll = nil;  
  panel = nil;	
	return self;
}

- (void)setPath:(NSString *)path 
       delegate:(id)adelegate
       viewApps:(BOOL)canview
{
	int colswidth, winwidth;

  fm = [NSFileManager defaultManager];

	[self setDelegate: adelegate];
  ASSIGN (rootPath, path);
  viewsapps = canview;
  [self setAutoresizingMask: (NSViewWidthSizable | NSViewHeightSizable)];

	colswidth = [delegate getBrowserColumnsWidth];
	resizeIncrement = colswidth;			
	winwidth = [delegate getWindowFrameWidth];			
	columns = (int)winwidth / resizeIncrement;      
	columnsWidth = (winwidth - 16) / columns;		
  autoSynchronize = YES;

	TEST_RELEASE (iconsPath);

	if (pathsScroll != nil) {
		[pathsScroll removeFromSuperview];
		RELEASE (pathsScroll);
	}
	
	pathsScroll = [GWScrollView new];
  [pathsScroll setHasHorizontalScroller: YES];
  [pathsScroll setHasVerticalScroller: NO]; 
  [pathsScroll setLineScroll: columnsWidth];
  [pathsScroll setAutoresizingMask: NSViewWidthSizable];      
  [pathsScroll setDelegate: self];

  iconsPath = [[IconsPath alloc] initWithRootAtPath: rootPath
                    			columnsWidth: columnsWidth delegate: self];

  [pathsScroll setDocumentView: iconsPath];	
  [self addSubview: pathsScroll]; 

	TEST_RELEASE (panel);

	if (panelScroll != nil) {
		[panelScroll removeFromSuperview];
		RELEASE (panelScroll);
	}

  panelScroll = [NSScrollView new];
  [panelScroll setHasHorizontalScroller: YES];
  [panelScroll setHasVerticalScroller: YES]; 
  [panelScroll setAutoresizingMask: (NSViewWidthSizable | NSViewHeightSizable)];      

  panel = [[IconsPanel alloc] initAtPath: rootPath delegate: self];

  [panelScroll setDocumentView: panel];	
  [self addSubview: panelScroll]; 

	[[NSNotificationCenter defaultCenter] removeObserver: self];

  [[NSNotificationCenter defaultCenter] addObserver: self 
                				selector: @selector(fileSystemWillChange:) 
                					  name: GWFileSystemWillChangeNotification
                					object: nil];

  [[NSNotificationCenter defaultCenter] addObserver: self 
                				selector: @selector(fileSystemDidChange:) 
                					  name: GWFileSystemDidChangeNotification
                					object: nil];

  [[NSNotificationCenter defaultCenter] addObserver: self 
                				selector: @selector(sortTypeDidChange:) 
                					  name: GWSortTypeDidChangeNotification
                					object: nil];

	if (watchedPaths != nil) {
		[self unsetWatchers];
		RELEASE (watchedPaths);
		watchedPaths = nil;
	}

  [self setSelectedPaths: [NSArray arrayWithObject: rootPath]]; 
}

- (NSString *)menuName
{
	return @"Icon";
}

- (NSString *)shortCut
{
	return @"i";
}

- (BOOL)usesShelf
{
	return YES;
}

- (NSImage *)preview
{
	NSBundle *bundle = [NSBundle bundleForClass: [self class]];
	NSString *imgpath = [bundle pathForResource: @"preview" ofType: @"tiff"];
	NSImage *img = [[NSImage alloc] initWithContentsOfFile: imgpath];	
	return AUTORELEASE (img);
}

- (NSImage *)miniicon
{
	NSBundle *bundle = [NSBundle bundleForClass: [self class]];
	NSString *imgpath = [bundle pathForResource: @"miniwindow" ofType: @"tiff"];
	NSImage *img = [[NSImage alloc] initWithContentsOfFile: imgpath];	
	return AUTORELEASE (img);
}

- (void)validateRootPathAfterOperation:(NSDictionary *)opdict
{
  if ([rootPath isEqualToString: @"/"] == YES) {
    return;

  } else {
    NSString *operation = [opdict objectForKey: @"operation"];
    NSString *source = [opdict objectForKey: @"source"];
    NSArray *files = [opdict objectForKey: @"files"];
    int i;
    
    if ([operation isEqualToString: NSWorkspaceMoveOperation] 
        || [operation isEqualToString: NSWorkspaceDestroyOperation]
        || [operation isEqualToString: GWorkspaceRenameOperation]
				|| [operation isEqualToString: NSWorkspaceRecycleOperation]
				|| [operation isEqualToString: GWorkspaceRecycleOutOperation]
				|| [operation isEqualToString: GWorkspaceEmptyRecyclerOperation]) { 

      if ([operation isEqualToString: GWorkspaceRenameOperation]) {      
        files = [NSArray arrayWithObject: [source lastPathComponent]]; 
        source = [source stringByDeletingLastPathComponent];            
      } 
      
      for (i = 0; i < [files count]; i++) {
        NSString *fpath = [source stringByAppendingPathComponent: [files objectAtIndex: i]];

        if ((subPath(fpath, rootPath) == YES) 
                        || ([fpath isEqualToString: rootPath] == YES)) {  
          [self closeNicely];      
          break;
        }
      }
      
    }
  }  
}

- (void)fileSystemWillChange:(NSNotification *)notification
{
  NSDictionary *dict;
  NSString *operation, *source, *destination;
  NSArray *files;
     
  dict = (NSDictionary *)[notification object];
  operation = [dict objectForKey: @"operation"];
  source = [dict objectForKey: @"source"];
  destination = [dict objectForKey: @"destination"];
  files = [dict objectForKey: @"files"];

  if ([operation isEqualToString: NSWorkspaceMoveOperation] 
      || [operation isEqualToString: NSWorkspaceDestroyOperation]
			|| [operation isEqualToString: NSWorkspaceRecycleOperation]
			|| [operation isEqualToString: GWorkspaceRecycleOutOperation]
			|| [operation isEqualToString: GWorkspaceEmptyRecyclerOperation]) { 
    
    if ([[panel currentPath] isEqualToString: source] == YES) {
      [panel removeIconsWithNames: files];
      [self setCurrentSelection: [NSArray arrayWithObject: source]];
      [self synchronize];
      
    } else {
      if (subPath(source, rootPath) == NO) {
        NSArray *components = [[panel currentPath] pathComponents];  
        NSMutableArray *paths = [NSMutableArray arrayWithCapacity: 1];  
        NSString *s = [NSString string];        
        int i;

        for (i = 0; i < [components count]; i++) {
          s = [s stringByAppendingPathComponent: [components objectAtIndex: i]];
          if (subPath(s, rootPath) == NO) {
            [paths addObject: s];  
          }         
        }  

        for (i = 0; i < [files count]; i++) {  
          NSString *fname = [files objectAtIndex: i];
          NSString *fpath = [source stringByAppendingPathComponent: fname];

          if ([paths containsObject: fpath] == YES) {
            fpath = [fpath stringByDeletingLastPathComponent];
            [self setCurrentSelection: [NSArray arrayWithObject: fpath]];
            [panel removeIconsWithNames: files];
            [self synchronize];            
            break;
          }
        }
      }
    }
  }
  
  [self validateRootPathAfterOperation: dict];
}

- (void)fileSystemDidChange:(NSNotification *)notification
{
  NSDictionary *dict;
  NSString *operation, *source, *destination;
  NSArray *files;
	  
  dict = (NSDictionary *)[notification object];    
  operation = [dict objectForKey: @"operation"];
  source = [dict objectForKey: @"source"];
  destination = [dict objectForKey: @"destination"];
  files = [dict objectForKey: @"files"];
    
  if ([operation isEqualToString: GWorkspaceRenameOperation]) {
    destination = [destination stringByDeletingLastPathComponent];
  }  

  if (subPath(destination, rootPath) == NO) {
    if ([[panel currentPath] isEqualToString: destination] == YES) {
      [panel makeFileIcons];
      [self setCurrentSelection: [NSArray arrayWithObject: destination]];
    }

    if ([operation isEqualToString: GWorkspaceCreateFileOperation] 
          || [operation isEqualToString: GWorkspaceCreateDirOperation]) { 
      NSString *filePath = [destination stringByAppendingPathComponent: [files objectAtIndex: 0]];
      [panel selectIconsWithPaths: [NSArray arrayWithObject: filePath]];
      [self synchronize];
    }
  }    
  
  [self validateRootPathAfterOperation: dict];  
  [delegate updateTheInfoString];
}

- (void)sortTypeDidChange:(NSNotification *)notification
{
	NSString *notifPath = [notification object];

	if (notifPath != nil) {
		if ([rootPath isEqualToString: notifPath]) {
			[panel makeFileIcons];
		}
	} else {
		[panel makeFileIcons];
	}
}

- (void)watcherNotification:(NSNotification *)notification
{
  NSDictionary *notifdict = (NSDictionary *)[notification object];
  NSString *path = [notifdict objectForKey: @"path"];
	
  if ([watchedPaths containsObject: path] == NO) {
    return;    

  } else {
    NSString *event = [notifdict objectForKey: @"event"];
    int i;
  
    if (event == GWWatchedDirectoryDeleted) {
      if ((subPath(path, rootPath)) || ([path isEqualToString: rootPath])) {  
        [self closeNicely];      
        return;
      } else {
        NSString *s = [path stringByDeletingLastPathComponent];
        [self unsetWatcherForPath: path];        
        [self setCurrentSelection: [NSArray arrayWithObject: s]];
        return;
      }
    }

    if (event == GWFileDeletedInWatchedDirectory) {
      if ([path isEqualToString: [panel currentPath]] == NO) {
        return;       
      } else {
				NSArray *files;
				int count;
			
				files = [notifdict objectForKey: @"files"];
        [panel removeIconsWithNames: files];
				
				count = 0;
				if (selectedPaths != nil) {
					count = [selectedPaths count];
				}
				
				if (count > 0) {
					NSMutableArray *selpaths = [selectedPaths mutableCopy];
								
					for (i = 0; i < count; i++) {
						NSString *fname = [[selpaths objectAtIndex: i] lastPathComponent];

						if ([files containsObject: fname] == YES) {
  						[selpaths removeObjectAtIndex: i];
  						count--;
  						i--;
						}
					}
				
					if ([selpaths count] > 0) {
						[self setCurrentSelection: selpaths];
					} else {
						[self setCurrentSelection: [NSArray arrayWithObject: [panel currentPath]]];
					}
				
					RELEASE (selpaths);
				} else {
					[self setCurrentSelection: [NSArray arrayWithObject: [panel currentPath]]];
				}
				
        return;
      }
    }
    
    if (event == GWFileCreatedInWatchedDirectory) {
      if ([path isEqualToString: [panel currentPath]] == NO) {
        return; 
      } else {
        [panel makeFileIcons];
      }
    }

  }
  
  [delegate updateTheInfoString];
}

- (void)setWatchersFromTimer:(id)sender
{
  [self setWatchers];
}

- (void)setWatchers
{
  int i;
  
  for (i = 0; i < [watchedPaths count]; i++) {
    [self setWatcherForPath: [watchedPaths objectAtIndex: i]];  
  }

  [[NSNotificationCenter defaultCenter] addObserver: self 
                		selector: @selector(watcherNotification:) 
                				name: GWFileWatcherFileDidChangeNotification
                			object: nil];
}

- (void)setWatcherForPath:(NSString *)path
{
	[delegate addWatcherForPath: path];
}

- (void)unsetWatchers
{
  int i;

  [[NSNotificationCenter defaultCenter] removeObserver: self 
                name: GWFileWatcherFileDidChangeNotification object: nil];

  for (i = 0; i < [watchedPaths count]; i++) {
    [self unsetWatcherForPath: [watchedPaths objectAtIndex: i]];  
  }
}

- (void)unsetWatcherForPath:(NSString *)path
{
	[delegate removeWatcherForPath: path];
}

- (void)resizeWithOldSuperviewSize:(NSSize)oldFrameSize
{
  NSRect r = [self frame];
  int winwidth;
	 
  [pathsScroll setFrame: NSMakeRect(0, r.size.height - 98, r.size.width, 98)]; 
  [panelScroll setFrame: NSMakeRect(0, 0, r.size.width, r.size.height - 104)];
  [panel setFrame: [[panelScroll contentView] frame]];
  [panel resizeWithOldSuperviewSize: [panel frame].size];
  
	winwidth = [delegate getWindowFrameWidth];
  columns = (int)winwidth / resizeIncrement;
  columnsWidth = (winwidth - 16) / columns;
  [self synchronize];
}

- (void)clickOnIcon:(PathIcon *)icon
{
  [self setCurrentSelection: [icon paths]];
}

- (void)doubleClickOnIcon:(PathIcon *)icon newViewer:(BOOL)isnew
{
  [self setCurrentSelection: [icon paths]];
  [self openCurrentSelection: [icon paths] newViewer: isnew];
}

- (void)synchronize
{
  NSClipView *clip;
  float x, y;
  int nicons;

  if (autoSynchronize == NO) {
    return;
  } 
    
  [iconsPath setIconsForSelection: selectedPaths]; 
  nicons = [iconsPath numberOfIcons];
  clip = [pathsScroll contentView];
  x = [clip bounds].origin.x;
  y = [clip bounds].origin.y;

  if (nicons > columns) {    
    x += columnsWidth * (nicons - columns);
    [clip scrollToPoint: NSMakePoint(x, y)];
  }
}

- (void)setSelectedPaths:(NSArray *)paths
{
  NSString *fullPath;
  NSArray *components;
  NSMutableArray *wpaths;
  NSString *s;
  BOOL isDir;
  int i, j;

  if (([paths count] == 0) || ([paths isEqualToArray: selectedPaths] == YES)) {
    return;
  }

  ASSIGN (selectedPaths, paths);
	[delegate setTheSelectedPaths: paths];   
  
  [self synchronize];     
  
  fullPath = [paths objectAtIndex: 0]; 
  [fm fileExistsAtPath: fullPath isDirectory: &isDir];
  if ((isDir == NO) || ([paths count] > 1)) {
    fullPath = [fullPath stringByDeletingLastPathComponent];
  }

  components = [fullPath pathComponents];  
  wpaths = [NSMutableArray arrayWithCapacity: 1];  
  s = [NSString string];        
   
  for (i = 0; i < [components count]; i++) {  
    s = [s stringByAppendingPathComponent: [components objectAtIndex: i]];
    [wpaths addObject: s];            
  }  
  
  if (watchedPaths == nil) {          
    ASSIGN (watchedPaths, wpaths);  
    [self setWatchers];

  } else {  
    int count = [wpaths count];
  
    for (i = 0; i < [watchedPaths count]; i++) {
      NSString *s1, *s2;

      s1 = [watchedPaths objectAtIndex: i];

      if (count > i) {
        s2 = [wpaths objectAtIndex: i];  
      } else {
        i = count;
        break;
      }

      if ([s1 isEqualToString: s2] == NO) {
        break;
      }    
    }

    for (j = i; j < [watchedPaths count]; j++) {  
      [self unsetWatcherForPath: [watchedPaths objectAtIndex: j]];
    }

    for (j = i; j < [wpaths count]; j++) {  
      [self setWatcherForPath: [wpaths objectAtIndex: j]];
    }

    ASSIGN (watchedPaths, wpaths);
  }
}

- (NSArray *)selectedPaths
{
  return selectedPaths;
}

- (NSPoint)locationOfIconForPath:(NSString *)path
{
	NSString *name = [path lastPathComponent];
	NSPoint p = [panel locationOfIconWithName: name];
	
	if ((p.x == 0) && (p.y == 0)) {
		return p;
		
	} else { 
		NSView *sview = [[self window] contentView];
		NSRect r = [self visibleRect];
		NSPoint lim1 = r.origin;
		NSPoint lim2 = NSMakePoint(lim1.x + r.size.width, lim1.y + r.size.height);

		p = [sview convertPoint: p fromView: panel];
		lim1 = [sview convertPoint: lim1 fromView: self];
		lim2 = [sview convertPoint: lim2 fromView: self];

		if (p.x < lim1.x) p.x = lim1.x;
		if (p.y < lim1.y) p.y = lim1.y;

		if (p.x >= lim2.x) p.x = lim2.x - 60;
		if (p.y >= lim2.y) p.y = lim2.y - 60;

		if (p.x == 0) p.x = 1;
		if (p.y == 0) p.y = 1;
	}
	
	return p;
}

- (void)setCurrentSelection:(NSArray *)paths
{
  NSString *path, *defApp, *type;
  NSArray *selection;
	
  if (([paths count] == 0) || ([paths isEqualToArray: selectedPaths] == YES)) {
    return;
  }
  
  [self setSelectedPaths: paths];
  
  path = [paths objectAtIndex: 0];
  
  if ([paths count] > 1) {
    path = [path stringByDeletingLastPathComponent];
    selection = [NSArray arrayWithArray: selectedPaths];
  } else {
    BOOL isdir;
  
    [fm fileExistsAtPath: path isDirectory: &isdir];
  
    if (isdir == NO) {
      path = [path stringByDeletingLastPathComponent];
      selection = [NSArray arrayWithArray: selectedPaths];
    } else {
			[[NSWorkspace sharedWorkspace] getInfoForFile: path application: &defApp type: &type]; 
			if ((type == NSApplicationFileType) && (viewsapps == NO)) {
				path = [path stringByDeletingLastPathComponent];
			}
      selection = [NSArray array];
    }
  }
  
  [panel setPath: path];  
  [panel selectIconsWithPaths: selection];
  [delegate updateTheInfoString];
}

- (void)openCurrentSelection:(NSArray *)paths newViewer:(BOOL)newv
{
  NSString *defApp, *type;
  NSMutableArray *files, *dirs, *apps;
  int i;
  
  [self setSelectedPaths: paths];
      
  files = [NSMutableArray arrayWithCapacity: 1]; 
  apps = [NSMutableArray arrayWithCapacity: 1]; 
  dirs = [NSMutableArray arrayWithCapacity: 1]; 
      
  for (i = 0; i < [paths count]; i++) {
    NSString *path = [paths objectAtIndex: i];
    
    [[NSWorkspace sharedWorkspace] getInfoForFile: path application: &defApp type: &type];     

    if ([type isEqualToString: NSPlainFileType] == YES) {
      [files addObject: path];  
    } else if ([type isEqualToString: NSApplicationFileType] == YES) {
      [apps addObject: path];      
    } else if (([type isEqualToString: NSDirectoryFileType] == YES)
                || ([type isEqualToString: NSFilesystemFileType] == YES)) { 
      [dirs addObject: path];      
    }
  }
        
  for (i = 0; i < [files count]; i++) {		
		[delegate openFile: [files objectAtIndex: i]];
  }

  for (i = 0; i < [apps count]; i++) {
    NSString *path = [apps objectAtIndex: i];
    
    if (newv == YES) {
			[delegate newViewerAtPath: path canViewApps: YES];
    } else {
      [[NSWorkspace sharedWorkspace] launchApplication: path];
    }
  }
    
  if ([dirs count] == 1) {  
    if (newv == YES) {    		
			[delegate newViewerAtPath: [dirs objectAtIndex: 0] canViewApps: NO];
    } else {
      [panel setPath: [dirs objectAtIndex: 0]];
    }
  }
}

- (void)setResizeIncrement:(int)increment
{
  resizeIncrement = increment;
}

- (void)setAutoSynchronize:(BOOL)value
{
  autoSynchronize = value;
}

- (id)viewerView
{
  return panel;
}

- (BOOL)viewsApps
{
  return viewsapps;
}

- (void)selectAll
{
  [panel selectAllIcons];
}

- (id)delegate
{
  return delegate;
}

- (void)setDelegate:(id)anObject
{	
  delegate = anObject;
}

- (void)closeNicely
{
  NSTimer *t;
  
  [self unsetWatchers]; 
  [[NSNotificationCenter defaultCenter] removeObserver: self];

  t = [NSTimer timerWithTimeInterval: 2 target: self 
          selector: @selector(close:) userInfo: nil repeats: NO];                                             
  [[NSRunLoop currentRunLoop] addTimer: t forMode: NSDefaultRunLoopMode];
}

- (void)close:(id)sender
{
  [[self window] performClose: nil];
}

//
// scrollview delegate methods
//
- (void)gwscrollView:(GWScrollView *)sender 
  scrollViewScrolled:(NSClipView *)clip
             hitPart:(NSScrollerPart)hitpart
{
  if (autoSynchronize == NO) {
    return;
  } else {
    int x = (int)[clip bounds].origin.x;
    int y = (int)[clip bounds].origin.y;
    int rem = x % (int)columnsWidth;
    
    if (rem != 0) {
      if (rem <= columnsWidth / 2) {
        x -= rem;
      } else {
        x += columnsWidth - rem;
      }
      
      [clip scrollToPoint: NSMakePoint(x, y)];    
      [sender reflectScrolledClipView: clip];
    }
  }
}

@end

//
// IconsPanel delegate methods
//

@implementation IconsViewer (IconsPanelDelegateMethods)

- (void)setTheSelectedPaths:(id)paths
{
	[delegate addPathToHistory: paths];
	[self setSelectedPaths: paths];
}

- (void)openTheCurrentSelection:(id)paths newViewer:(BOOL)newv
{
	[self openCurrentSelection: paths newViewer: newv];
}

- (void)performFileOperationWithDictionary:(id)opdict
{
	[delegate performFileOperationWithDictionary: opdict];
}

- (int)sortTypeForDirectoryAtPath:(id)path
{
	return [delegate sortTypeForDirectoryAtPath: path];
}

- (NSString *)getTrashPath
{
	return [delegate getTrashPath];
}

- (int)getIconViewCellsWidth
{	
	return [delegate getIconViewCellsWidth];
}

@end

//
// IconsPath delegate methods
//

@implementation IconsViewer (IconsPathDelegateMethods)

- (void)clickedIcon:(id)anicon
{
	[self clickOnIcon: anicon];
}

- (void)doubleClickedIcon:(id)anicon newViewer:(BOOL)isnew
{
	[self doubleClickOnIcon: anicon newViewer: isnew];
}

- (void)performFileOperationWithDictionary:(id)opdict
{
	[delegate performFileOperationWithDictionary: opdict];
}

- (NSString *)getTrashPath
{
	return [delegate getTrashPath];
}

@end
