/*
Gwenview - A simple image viewer for KDE
Copyright (C) 2000-2002 Aurlien Gteau

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/
// Qt includes
#include <qdir.h>
#include <qdragobject.h>
#include <qheader.h>
#include <qpopupmenu.h>
#include <qtimer.h>

// KDE includes
#include <kdebug.h>
#include <kdirlister.h>
#include <kiconloader.h>
#include <kio/global.h>
#include <kio/job.h>
#include <klocale.h>
#include <klineeditdlg.h>
#include <kmessagebox.h>
#include <kpropsdlg.h>
#include <kurldrag.h>

// Our includes
#include "dirview.moc"

#include "dirviewitem.h"


const int AUTO_OPEN_DELAY=1000;


/**
 * Constructor
 * Create the root node and the popup menu
 */
DirView::DirView(QWidget* parent) : KListView(parent), mDropTarget(0L) {
	DirViewItem* item;
	
	mPopupMenu=new QPopupMenu(this);
	mPopupMenu->insertItem(SmallIcon("folder_new"),i18n("New folder"),this,SLOT(makeDir()));
	mPopupMenu->insertSeparator();
	mPopupMenu->insertItem(i18n("Rename..."),this,SLOT(renameDir()));
	mPopupMenu->insertItem(SmallIcon("editdelete"),i18n("Delete"),this,SLOT(removeDir()));
	mPopupMenu->insertSeparator();
	mPopupMenu->insertItem(i18n("Properties"),this,SLOT(showPropertiesDialog()));

	mDirLister=new KDirLister;
	mDirLister->setDirOnlyMode(true);

	connect(this,SIGNAL(selectionChanged(QListViewItem*)),
		this,SLOT(onSelectionChanged(QListViewItem*)));
	
	connect(mDirLister, SIGNAL(newItems(const KFileItemList&)),
		this,SLOT(onNewItems(const KFileItemList&)));
	connect(mDirLister, SIGNAL(deleteItem(KFileItem*)),
		this,SLOT(onDeleteItem(KFileItem*)));
	connect(mDirLister, SIGNAL(refreshItems(const KFileItemList&)),
		this,SLOT(onRefreshItems(const KFileItemList&)));

	mNewDirTimer=new QTimer(this);
	connect(mNewDirTimer, SIGNAL(timeout()),
		this,SLOT(processNewDirQueue()));

	mAutoOpenTimer=new QTimer(this);
	connect(mAutoOpenTimer, SIGNAL(timeout()),
		this,SLOT(autoOpenDropTarget()));

	connect(this,SIGNAL(contextMenu(KListView*,QListViewItem*,const QPoint&)),
		this,SLOT(onContextMenu(KListView*,QListViewItem*,const QPoint&)));



// Header
	addColumn(i18n("Folders"));
	header()->hide();
	setAllColumnsShowFocus(true);

// DnD
	setDragEnabled(true);
	setDropVisualizer(true);
	setDropHighlighter(true);
	setAcceptDrops(true);

// Root
	KURL root;
	root.setPath("/");
	item=new RootDirViewItem(this,"/");
	item->setOpen(true);
}


DirView::~DirView() {
	delete mDirLister;
}


/**
 * Emit our folderChanged signal
 */
void DirView::onSelectionChanged(QListViewItem* paramItem) {
	if (!paramItem) return;
	DirViewItem* item=static_cast<DirViewItem*>(paramItem);

	mURL.setPath(item->folder());
	emit folderChanged(item->folder());
}


/**
 * Init mURL and start the dir lister on it
 */
void DirView::setFolder(QString newFolder) {

	KURL url;
	url.setPath(newFolder);

	if (mURL.cmp(url,true)) {
		return;
	}
	mURL=url;


	QStringList folderParts;
	QStringList::Iterator folderIter,endFolderIter;
	QString folder="/";
	DirViewItem* viewItem;
	DirViewItem* nextViewItem;

	folderParts=QStringList::split('/',newFolder);
	folderIter=folderParts.begin();
	endFolderIter=folderParts.end();
	viewItem=static_cast<DirViewItem*>(firstChild());

// Finds the deepest existing view item
	for(;folderIter!=endFolderIter;++folderIter) {

		nextViewItem=viewItem->find(*folderIter);
		if (nextViewItem) {
			folder+=*folderIter + "/";
			viewItem=nextViewItem;
		} else {
			break;
		}
	}

	url.setPath(folder);
	mDirLister->openURL(url,true,true);

}


//-Dir lister slots-------------------------------------------------------
/**
 * Filters hidden items, appends items to the new-dir queue and starts
 * the new-dir "thread" if it has not started yet
 */
void DirView::onNewItems(const KFileItemList& items) {
	KFileItem* item;
	QListIterator<KFileItem> it(items);
	for(;it.current();++it) {
		item=*it;
		if (item->name()[0]!='.') {
			mNewDirQueue.append(item);
		}
	}
	if (!mNewDirTimer->isActive() && !mNewDirQueue.isEmpty()) {
		mNewDirTimer->start(0,false);
	}
}


void DirView::onDeleteItem(KFileItem* fileItem) {
	if (mItemMap.contains(fileItem)) {
		DirViewItem* viewItem=mItemMap[fileItem];

		if (viewItem==currentItem()) {
			setCurrentItem(viewItem->parent());
		}

		removeViewItemFromItemMap(viewItem);
		delete viewItem;
    } else {
		return;
	}

	mNewDirQueue.removeRef(fileItem);
}



/**
 * Remove view item from item map recursively
 */
void DirView::removeViewItemFromItemMap(QListViewItem* viewItem) {
	QListViewItem* child;

	for (child=viewItem->firstChild(); child; child=child->nextSibling() ) {
		removeViewItemFromItemMap(child);
	}

	mItemMap.remove( static_cast<SubDirViewItem*>(viewItem)->fileItem() );
}



void  DirView::onRefreshItems(const KFileItemList& items) {
	KFileItem* fileItem;
	SubDirViewItem* viewItem;
	QListIterator<KFileItem> it(items);
	for(;it.current();++it) {
		fileItem=*it;

		viewItem=static_cast<SubDirViewItem*>(mItemMap[fileItem]);
		if (!viewItem) {
			continue;
		}

		viewItem->refreshLabel();

		if (viewItem==currentItem()) {
			emit folderChanged(viewItem->folder());
		}

	// Remove children if any and close the item
		if (viewItem->childCount()>0) {
			QListViewItem* child;
			for (child=viewItem->firstChild(); child; child=child->nextSibling() ) {
				removeViewItemFromItemMap(child);
				delete child;
			}
			viewItem->setOpen(false);
		}

	}
}


//-Private-----------------------------------------------------------------
void DirView::processNewDirQueue() {
	if (mNewDirQueue.isEmpty()) {
		mNewDirTimer->stop();
		return;
	}
	KFileItem* item=mNewDirQueue.getFirst();
	mNewDirQueue.removeFirst();
	addDir(item);
}


/**
 * Adds a new dir to the tree if it does not already exist
 */
void DirView::addDir(KFileItem* fileItem) {
	QStringList folderParts;
	QStringList::Iterator folderIter,endFolderIter;
	QString folder="/";
	DirViewItem* viewItem;
	DirViewItem* nextViewItem;

	folderParts=QStringList::split('/',fileItem->url().path());
	folderIter=folderParts.begin();
	endFolderIter=folderParts.end();
	viewItem=static_cast<DirViewItem*>(firstChild());

// Finds the deepest existing view item
	for(;folderIter!=endFolderIter;++folderIter) {
		nextViewItem=viewItem->find(*folderIter);
		if (nextViewItem) {
			folder+=*folderIter + "/";
			viewItem=nextViewItem;
		} else {
			break;
		}
	}

// Already exists, strange...
	if (folderIter==endFolderIter) {
		kdWarning() << "  dir '" << fileItem->url().path() << "' already exists. Should not happen\n";
		return;
	}

// Creates missing view item
	folder+=*folderIter + "/";
	viewItem=new SubDirViewItem(this,viewItem,*folderIter,fileItem);
	mItemMap[fileItem]=viewItem;

	KURL url;
	url.setPath(folder);

	if ( url.cmp(mURL,true)) {
		viewItem->setOpen(true);
		blockSignals(true);
		setSelected(viewItem,true);
		blockSignals(false);
		ensureItemVisible(viewItem);
	} else if (url.isParentOf(mURL)) {
		viewItem->setOpen(true);
	} else {
		viewItem->initExpandable();
	}

}




DirViewItem* DirView::find(const KURL& url) const {
	QStringList folderParts;
	QStringList::Iterator folderIter,endFolderIter;
	DirViewItem* viewItem;

	folderParts=QStringList::split('/',url.path());
	endFolderIter=folderParts.end();

	viewItem=static_cast<DirViewItem*>(firstChild());
	for(folderIter=folderParts.begin();
		folderIter!=endFolderIter;
		++folderIter) {

		viewItem=viewItem->find(*folderIter);
		if (!viewItem) {
			return 0L;
		}
	}

	return viewItem;
}


//-Drag'n drop------------------------------------------------------
void DirView::contentsDragMoveEvent(QDragMoveEvent* event) {
	if (!QUriDrag::canDecode(event)) {
		event->ignore();
		return;
	}

	QPoint point(0,event->pos().y());
	DirViewItem* newDropTarget=static_cast<DirViewItem*>( itemAt(contentsToViewport(point)) );

	if (newDropTarget) {
		event->accept();
	} else {
		event->ignore();
	}

	if (newDropTarget==mDropTarget) return;

	mAutoOpenTimer->stop();

// Remove highlight on previous drop target
	if (mDropTarget) {
		mDropTarget->setIsDropTarget(false);
		mDropTarget->repaint();
	}

// Highlight new drop target, if any
	mDropTarget=newDropTarget;
	if (mDropTarget) {
		mDropTarget->setIsDropTarget(true);
		mDropTarget->repaint();
		mAutoOpenTimer->start(AUTO_OPEN_DELAY,true);
	}

}


void DirView::contentsDragLeaveEvent(QDragLeaveEvent*) {
	if (mDropTarget) {
		mDropTarget->setIsDropTarget(false);
		mDropTarget->repaint();
	}
	mAutoOpenTimer->stop();
	mDropTarget=0L;
}


void DirView::contentsDropEvent(QDropEvent* event) {
	mAutoOpenTimer->stop();

// Quit if no valid target
	if (!mDropTarget) return;

// Get data from drop (do it before showing menu to avoid mDropTarget changes)
	KURL dest=mDropTarget->url();

	KURL::List urls;
	KURLDrag::decode(event,urls);


// Show popup
	QPopupMenu menu(this);
	int copyItemID = menu.insertItem( i18n("&Copy here") );
	int moveItemID = menu.insertItem( i18n("&Move here") );

	menu.setMouseTracking(true);
	int id = menu.exec(QCursor::pos());

// Handle menu choice
	if (id!=-1) {
		if (id ==copyItemID) {
			KIO::copy(urls, dest, true);
		} else if (id==moveItemID) {
			KIO::move(urls, dest, true);
		}
	}

// Remove target highlight
	if (mDropTarget) {
		mDropTarget->setIsDropTarget(false);
		mDropTarget->repaint();
		mDropTarget=0L;
	}
}


void DirView::startDrag() {
	if (!currentItem()) return;

// Create a list of the URLs that we want to drag
	KURL::List urls;
	urls.append( static_cast<DirViewItem*>(currentItem())->url() );

// drag object
	QDragObject* dragObject=KURLDrag::newDrag( urls, this );
	dragObject->setPixmap( *(currentItem()->pixmap(0)) );
	dragObject->dragCopy();
}


void DirView::autoOpenDropTarget() {
	if (mDropTarget) {
		mDropTarget->setOpen(true);
	}
}


//- Popup ----------------------------------------------------------
void DirView::onContextMenu(KListView*,QListViewItem*,const QPoint& pos) {
	mPopupMenu->popup(pos);
}


void DirView::makeDir() {
	KIO::Job* job;
	if (!currentItem()) return;

	bool ok;
	QString newDir=KLineEditDlg::getText(i18n("Enter the name of the new folder :"),"",&ok,this);
	if (!ok) return;
	
	KURL newURL(mURL);
	newURL.addPath(newDir);
	job=KIO::mkdir(newURL);
	
	connect(job,SIGNAL(result(KIO::Job*)),
		this,SLOT(onDirMade(KIO::Job*)));
}


void DirView::onDirMade(KIO::Job* job) {
	if (job->error()) {
		job->showErrorDialog(this);
		return;
	}
	if (!currentItem()) return;
	currentItem()->setOpen(true);
}


void DirView::renameDir() {
	KIO::Job* job;
	if (!currentItem()) return;
	
	bool ok;
	QString newDir=KLineEditDlg::getText(i18n("Rename this folder to :"),mURL.filename(),&ok,this);
	if (!ok) return;

	KURL newURL=mURL.upURL();
	
	newURL.addPath(newDir);
	job=KIO::rename(mURL,newURL,false);
	mURL=newURL;

	connect(job,SIGNAL(result(KIO::Job*)),
		this,SLOT(onDirRenamed(KIO::Job*)));
}


void DirView::onDirRenamed(KIO::Job* job) {
	if (job->error()) {
		job->showErrorDialog(this);
		return;
	}
}


void DirView::removeDir() {
	KIO::Job* job;
	if (!currentItem()) return;

	int response=KMessageBox::questionYesNo(this,
		i18n("Are you sure you want to delete the folder <b>%1</b> ?").arg(mURL.path() ));
	if (response==KMessageBox::No) return;

	job=KIO::del(mURL);
	connect(job,SIGNAL(result(KIO::Job*)),
		this,SLOT(onDirRemoved(KIO::Job*)));
}


void DirView::onDirRemoved(KIO::Job* job) {
	if (job->error()) {
		job->showErrorDialog(this);
	}
}


void DirView::showPropertiesDialog() {
	(void)new KPropertiesDialog(mURL);
}

