// ****************************************************************************
//  Project:        GUYMAGER
// ****************************************************************************
//  Programmer:     Guy Voncken
//                  Police Grand-Ducale
//                  Service de Police Judiciaire
//                  Section Nouvelles Technologies
// ****************************************************************************
//  Module:         Thread for writing data.
// ****************************************************************************

#include "common.h"
#include "compileinfo.h"

#include <errno.h>
#include <zlib.h>
#include <fcntl.h>

#include <QtCore>

#include "toolconstants.h"
#include "sysinfo/toolsysinfo.h"

#include "util.h"
#include "config.h"
#include "device.h"
#include "main.h"
#include "aaff.h"
#include "threadwrite.h"


const unsigned long THREADWRITE_WAIT_FOR_HANDLE_GRANULARITY =   100; // ms
const unsigned long THREADWRITE_WAIT_FOR_HANDLE             = 30000; // ms
const unsigned long THREADWRITE_SLOWDOWN_SLEEP              =   700; // ms

const int           THREADWRITE_AFF_MODE                    =  0666; // file mode flags

class t_OutputFile
{
   public:
      virtual        ~t_OutputFile  (void) {};
      virtual APIRET  Open          (t_pDevice pDevice)       = 0;
      virtual APIRET  Write         (t_pFifoBlock pFifoBlock) = 0;
      virtual APIRET  Close         (void)                    = 0;
      virtual bool    Opened        (void)                    = 0;
      virtual void *  GetFileHandle (void)                    = 0;
};

class t_OutputFileDD: public t_OutputFile
{
   public:
      t_OutputFileDD (void) :
         poFile (NULL)
      {
      } //lint -esym(613, t_OutputFileDD::poFile)  Possible use of NULL pointer
        //lint -esym(668, fclose, fwrite)  Possibly passing NULL pointer

      ~t_OutputFileDD (void)
      {
         if (t_OutputFileDD::Opened())
            (void) t_OutputFileDD::Close();
      } //lint !e1740

      APIRET Open (t_pDevice pDevice)
      {
         QString Extension;
         QString FileName;

         if (CONFIG (WriteToDevNull))
         {
            FileName = "/dev/null";
         }
         else
         {
            CHK (t_File::GetFormatExtension (pDevice->Acquisition.Format, &Extension))
            FileName = pDevice->Acquisition.ImagePath + pDevice->Acquisition.ImageFilename + Extension;
         }

         poFile = fopen64 (QSTR_TO_PSZ (FileName), "w");
         if (poFile == NULL)
         {
            LOG_ERROR ("fopen on %s failed, errno=%d '%s'", QSTR_TO_PSZ (FileName), errno, ToolErrorTranslateErrno (errno))
            return ERROR_THREADWRITE_OPEN_FAILED;
         }
         return NO_ERROR;
      }

      APIRET Write (t_pFifoBlock pFifoBlock)
      {
         unsigned int Written = fwrite (pFifoBlock->Buffer, pFifoBlock->DataSize, 1, poFile);
         if (Written != 1)
         {
            LOG_ERROR ("fwrite failed, errno=%d '%s'", errno, ToolErrorTranslateErrno (errno))
            return ERROR_THREADWRITE_WRITE_FAILED;
         }
         return NO_ERROR;
      }

      APIRET Close (void)
      {
         int Res;

         if (poFile == NULL)
            CHK (ERROR_THREADWRITE_NOT_OPENED)

         Res = fflush (poFile);
         if (Res)
         {
            (void) fclose (poFile);
            LOG_ERROR ("fflush failed, errno=%d '%s'", errno, ToolErrorTranslateErrno (errno))
            return ERROR_THREADWRITE_CLOSE_FAILED;
         }

         Res = fclose (poFile);
         poFile = NULL;
         if (Res)
         {
            LOG_ERROR ("fclose failed, errno=%d '%s'", errno, ToolErrorTranslateErrno (errno))
            return ERROR_THREADWRITE_CLOSE_FAILED;
         }
         return NO_ERROR;
      }

      void * GetFileHandle (void)
      {
         return poFile;
      }

      bool Opened (void)
      {
         return (poFile != NULL);
      }

   private:
      FILE *poFile;
};

#define CHK_LIBEWF(Fn)                                                 \
{                                                                      \
   int rclibewf = (Fn);                                                \
   if (rclibewf != 1)                                                  \
   {                                                                   \
      LOG_ERROR ("Error in libewf function: %s, rc=%d", #Fn, rclibewf) \
      return ERROR_THREADWRITE_LIBEWF_FAILED;                          \
   }                                                                   \
}

class t_OutputFileEWF: public t_OutputFile
{
   public:
      t_OutputFileEWF (void)
      {
         poFile   = NULL;
         poDevice = NULL;   //lint -esym(613,t_OutputFileEWF::poDevice)  Prevent lint from telling us about possible null pointers in the following code
          oHasCompressionThreads = false;
      }

      ~t_OutputFileEWF (void)
      {
         if (t_OutputFileEWF::Opened())
            (void) t_OutputFileEWF::Close();
      } //lint !e1740

      APIRET Open (t_pDevice pDevice)
      {
         QString         Uname;
         QString         GuymagerVersion;
         LIBEWF_HANDLE *pFile;
         QByteArray      AsciiFileName = (pDevice->Acquisition.ImagePath + pDevice->Acquisition.ImageFilename).toAscii();
         char          *pAsciiFileName = AsciiFileName.data();

         oHasCompressionThreads = pDevice->HasCompressionThreads();
         poDevice               = pDevice;

         pFile = libewf_open (&pAsciiFileName, 1, LIBEWF_OPEN_WRITE);
         if (pFile == NULL)
            return ERROR_THREADWRITE_OPEN_FAILED;


         #define STR_AND_LEN(QStr) QStr.toAscii().data(), strlen(QStr.toAscii().data())


         libewf_set_notify_values(stderr, 1);

         CHK_LIBEWF (libewf_set_sectors_per_chunk  (pFile, 64))
         CHK_LIBEWF (libewf_set_bytes_per_sector   (pFile, pDevice->SectorSize))
         CHK_LIBEWF (libewf_set_segment_file_size  (pFile, (unsigned int) (CONFIG (EwfSegmentSize) * BYTES_PER_MEGABYTE)))
         CHK_LIBEWF (libewf_set_compression_values (pFile, CONFIG (EwfCompression), 1))
         CHK_LIBEWF (libewf_set_media_type         (pFile, pDevice->Removable ? LIBEWF_MEDIA_TYPE_REMOVABLE : LIBEWF_MEDIA_TYPE_FIXED))
         CHK_LIBEWF (libewf_set_volume_type        (pFile, LIBEWF_VOLUME_TYPE_PHYSICAL))
         CHK_LIBEWF (libewf_set_format             (pFile, (uint8_t) CONFIG (EwfFormat)))
         CHK_LIBEWF (libewf_set_media_size         (pFile, pDevice->Size))

         CHK_LIBEWF (libewf_set_header_value (pFile, (char *)"case_number"    , STR_AND_LEN(pDevice->Acquisition.CaseNumber    )))
         CHK_LIBEWF (libewf_set_header_value (pFile, (char *)"description"    , STR_AND_LEN(pDevice->Acquisition.Description   )))
         CHK_LIBEWF (libewf_set_header_value (pFile, (char *)"examiner_name"  , STR_AND_LEN(pDevice->Acquisition.Examiner      )))
         CHK_LIBEWF (libewf_set_header_value (pFile, (char *)"evidence_number", STR_AND_LEN(pDevice->Acquisition.EvidenceNumber)))
         CHK_LIBEWF (libewf_set_header_value (pFile, (char *)"notes"          , STR_AND_LEN(pDevice->Acquisition.Notes         )))

         CHK (ToolSysInfoUname (Uname))
         GuymagerVersion = QString("guymager ") + QString(pCompileInfoVersion);
         CHK_LIBEWF (libewf_set_header_value (pFile, (char *)"acquiry_operating_system", STR_AND_LEN(Uname)))
         CHK_LIBEWF (libewf_set_header_value (pFile, (char *)"acquiry_software_version", STR_AND_LEN(GuymagerVersion)))

         #undef STR_AND_LEN

         poFile = pFile; // Only set poFile at the very end, so the CompressionThreads won't use it until everything is initialised
         return NO_ERROR;
      }

      APIRET Write (t_pFifoBlock pFifoBlock)
      {
         int Written;
         int Size;

         if (oHasCompressionThreads != pFifoBlock->EwfPreprocessed)
            CHK (ERROR_THREADWRITE_WRONG_BLOCK)

         if (oHasCompressionThreads)
         {
            Size = pFifoBlock->EwfDataSize;
            Written = libewf_raw_write_buffer (poFile, pFifoBlock->Buffer, Size, pFifoBlock->DataSize,
                                                                                 pFifoBlock->EwfCompressionUsed,
                                                                                 pFifoBlock->EwfChunkCRC,
                                                                                 pFifoBlock->EwfWriteCRC);
         }
         else
         {
            Size = pFifoBlock->DataSize;
            Written = libewf_write_buffer (poFile, pFifoBlock->Buffer, Size);
         }
         if (Written != Size)
         {
            LOG_ERROR ("Written %d/%d bytes", Written, Size)
            return ERROR_THREADWRITE_WRITE_FAILED;
         }

         return NO_ERROR;
      }

      APIRET Close (void)
      {
         int rc;

         if (poFile == NULL)
            CHK (ERROR_THREADWRITE_NOT_OPENED)

         if (poDevice->HasCompressionThreads())
            CHK_LIBEWF (libewf_set_md5_hash (poFile, (uint8_t*)&poDevice->MD5Digest, HASH_MD5_DIGEST_LENGTH))

         rc = libewf_close (poFile);
         if (rc != 0)
         {
            LOG_ERROR ("Error in libewf function: libewf_close, rc=%d", rc)
            return ERROR_THREADWRITE_LIBEWF_FAILED;
         }

         poFile = NULL;
         return NO_ERROR;
      }

      void * GetFileHandle (void)
      {
         return poFile;
      }

      bool Opened (void)
      {
         return (poFile != NULL);
      }

   private:
      LIBEWF_HANDLE *poFile;
      t_pDevice      poDevice;
      bool            oHasCompressionThreads;
};


class t_OutputFileAFF: public t_OutputFile
{
   public:
      t_OutputFileAFF (void) :
         poFile   (NULL),
         poDevice (NULL)
      {
      }

      ~t_OutputFileAFF (void)
      {
         if (t_OutputFileAFF::Opened())
            (void) t_OutputFileAFF::Close();
      } //lint !e1740

      APIRET Open (t_pDevice pDevice)
      {
         QString               Extension;
         QString               FileName;
         QString               Mac;
         QString               DateTimeStr;
         t_ToolSysInfoMacAddr  MacAddr;
         char                *pCommandLine;
         t_pAaff              pFile;
         APIRET                rc;

//         printf ("\nCalling AaffOpen");
         poDevice = pDevice;
         oHasCompressionThreads = pDevice->HasCompressionThreads();


         CHK (t_File::GetFormatExtension (pDevice->Acquisition.Format, &Extension))
         FileName = pDevice->Acquisition.ImagePath + pDevice->Acquisition.ImageFilename + Extension;


         CHK (AaffOpen (&pFile, QSTR_TO_PSZ(FileName), pDevice->Size, pDevice->SectorSizePhys, pDevice->FifoBlockSize))

         rc = ToolSysInfoGetMacAddr (&MacAddr);
         if (rc == TOOLSYSINFO_ERROR_NO_ADDR)
         {
            Mac = "none";
         }
         else
         {
            CHK (rc)
            Mac = &MacAddr.AddrStr[0];
         }

         DateTimeStr  = poDevice->StartTimestamp.toString ("yyyy-MM-dd hh:mm:ss");
         DateTimeStr += " localtime";
         CHK (MainGetCommandLine (&pCommandLine))

         CHK (AaffWriteSegmentStr (pFile, AAFF_SEGNAME_COMMAND_LINE, 0, pCommandLine))
         CHK (AaffWriteSegmentStr (pFile, AAFF_SEGNAME_MACADDR     , 0, QSTR_TO_PSZ(Mac                                 )))
         CHK (AaffWriteSegmentStr (pFile, AAFF_SEGNAME_DATE        , 0, QSTR_TO_PSZ(DateTimeStr                         )))
         CHK (AaffWriteSegmentStr (pFile, AAFF_SEGNAME_DEVICE      , 0, QSTR_TO_PSZ(poDevice->LinuxDevice               )))
         CHK (AaffWriteSegmentStr (pFile, AAFF_SEGNAME_MODEL       , 0, QSTR_TO_PSZ(poDevice->Model                     )))
         CHK (AaffWriteSegmentStr (pFile, AAFF_SEGNAME_SN          , 0, QSTR_TO_PSZ(poDevice->SerialNumber              )))
         CHK (AaffWriteSegmentStr (pFile, "CaseNumber"             , 0, QSTR_TO_PSZ(poDevice->Acquisition.CaseNumber    )))
         CHK (AaffWriteSegmentStr (pFile, "EvidenceNumber"         , 0, QSTR_TO_PSZ(poDevice->Acquisition.EvidenceNumber)))
         CHK (AaffWriteSegmentStr (pFile, "Examiner"               , 0, QSTR_TO_PSZ(poDevice->Acquisition.Examiner      )))
         CHK (AaffWriteSegmentStr (pFile, "Description"            , 0, QSTR_TO_PSZ(poDevice->Acquisition.Description   )))
         CHK (AaffWriteSegmentStr (pFile, "Notes"                  , 0, QSTR_TO_PSZ(poDevice->Acquisition.Notes         )))

//         printf ("\nAaffOpen finished");
         poFile = pFile; // Only set poFile at the very end, so the CompressionThreads won't use it until everything is initialised
         return NO_ERROR;
      }

      APIRET Write (t_pFifoBlock pFifoBlock)
      {
         if (!oHasCompressionThreads) // Do preprocessing if not already done in compression threads
         {
            t_pFifoBlock pPreprocessBlock;

            CHK_EXIT (t_Fifo::CreateCompressionOptimised (pPreprocessBlock, poDevice->FifoBlockSize))
            pPreprocessBlock->DataSize = pFifoBlock->DataSize;
            CHK_EXIT (AaffPreprocess (&pPreprocessBlock->pAaffPreprocess, pFifoBlock->Buffer, pFifoBlock->DataSize, pPreprocessBlock->Buffer, pPreprocessBlock->BufferSize))
            if (pPreprocessBlock->pAaffPreprocess->Compressed)
            {
               CHK_EXIT (t_Fifo::Destroy (pFifoBlock))
               pFifoBlock = pPreprocessBlock;
            }
            else
            {
               pFifoBlock->pAaffPreprocess = pPreprocessBlock->pAaffPreprocess;
               CHK_EXIT (t_Fifo::Destroy (pPreprocessBlock))
            }
         }

         CHK (AaffWrite (poFile, pFifoBlock->pAaffPreprocess, pFifoBlock->Buffer, pFifoBlock->DataSize))

         return NO_ERROR;
      }

      APIRET Close (void)
      {
         QList<quint64> BadSectors;
         int            Seconds;

         if (poFile == NULL)
            CHK (ERROR_THREADWRITE_NOT_OPENED)

         CHK (poDevice->GetBadSectors (BadSectors, false))
         Seconds  = poDevice->StartTimestamp.secsTo (QDateTime::currentDateTime());

         CHK (AaffClose (poFile, BadSectors.count(), (unsigned char*)&poDevice->MD5Digest, (unsigned char*)&poDevice->SHA256Digest, Seconds))

         poFile = NULL;

         return NO_ERROR;
      }

      void *GetFileHandle (void)
      {
         return poFile;
      }

      bool Opened (void)
      {
         return (poFile != NULL);
      }

   private:
      t_pAaff    poFile;
      t_pDevice  poDevice;
      bool        oHasCompressionThreads;
};


class t_ThreadWriteLocal
{
   public:
      t_ThreadWriteLocal (void) :  pOutputFile(NULL), pDevice(NULL) {}

   public:
      t_OutputFile *pOutputFile;
      bool         *pSlowDownRequest;
      t_pDevice     pDevice;
};


t_ThreadWrite::t_ThreadWrite(void)
{
   CHK_EXIT (ERROR_THREADWRITE_CONSTRUCTOR_NOT_SUPPORTED)
} //lint !e1401 not initialised

t_ThreadWrite::t_ThreadWrite (t_pDevice pDevice, bool *pSlowDownRequest)
   :pOwn(NULL)
{
   static bool Initialised = false;

   if (!Initialised)
   {
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_THREADWRITE_OPEN_FAILED              ))
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_THREADWRITE_WRITE_FAILED             ))
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_THREADWRITE_CLOSE_FAILED             ))
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_THREADWRITE_NOT_OPENED               ))
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_THREADWRITE_INVALID_FORMAT           ))
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_THREADWRITE_CONSTRUCTOR_NOT_SUPPORTED))
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_THREADWRITE_WRONG_BLOCK              ))
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_THREADWRITE_OUT_OF_SEQUENCE_BLOCK    ))
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_THREADWRITE_HANDLE_NOT_YET_AVAILABLE ))
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_THREADWRITE_HANDLE_TIMEOUT           ))
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_THREADWRITE_LIBEWF_FAILED            ))

      Initialised = true;
   }

   pOwn = new t_ThreadWriteLocal;
   pOwn->pDevice          = pDevice;
   pOwn->pOutputFile      = NULL;
   pOwn->pSlowDownRequest = pSlowDownRequest;

   CHK_QT_EXIT (connect (this, SIGNAL(finished()), this, SLOT(SlotFinished())))
}

t_ThreadWrite::~t_ThreadWrite (void)
{
   delete pOwn;
}


APIRET ThreadWriteCheckData (t_pFifoBlock pFifoBlock)
{
   unsigned char *pBufferUncompressed;
   uLongf          UncompressedSize;
   int             zrc=99;
   uint32_t        CRC1;
   uint32_t        CRC2;
   bool            Error;

   Error = !pFifoBlock->EwfPreprocessed;

   // Get the original data
   // ---------------------
   if (pFifoBlock->EwfCompressionUsed)
   {
      UncompressedSize = pFifoBlock->DataSize + 4096; // DataSize should be enough for the uncompression buffer, as the uncompressed data originally had that
                                                      // size, but as this fn is used for debugging and we do not know what the bug might be, we add 4K for safety
      pBufferUncompressed = (unsigned char *) UTIL_MEM_ALLOC (UncompressedSize);
      if (pBufferUncompressed == NULL)
      {
         LOG_ERROR ("malloc returned NULL")
         CHK_EXIT (1967)
      }
      zrc = uncompress ((Bytef*)pBufferUncompressed, &UncompressedSize, pFifoBlock->Buffer, pFifoBlock->EwfDataSize);
      Error = Error || (zrc != Z_OK);
      Error = Error || (UncompressedSize != (unsigned) pFifoBlock->DataSize);

      ((char *)&CRC1)[0] = ((char *)&pFifoBlock->EwfChunkCRC)[3];
      ((char *)&CRC1)[1] = ((char *)&pFifoBlock->EwfChunkCRC)[2];
      ((char *)&CRC1)[2] = ((char *)&pFifoBlock->EwfChunkCRC)[1];
      ((char *)&CRC1)[3] = ((char *)&pFifoBlock->EwfChunkCRC)[0];
   }
   else
   {
      pBufferUncompressed = pFifoBlock->Buffer;
      UncompressedSize = pFifoBlock->DataSize;
      CRC1 = pFifoBlock->EwfChunkCRC;
   }

   // Get the CRC
   // -----------
   CRC2 = adler32 (1, (Bytef*)pBufferUncompressed, UncompressedSize);
   Error = Error || (CRC1 != CRC2);

   // Clean up
   // --------
   if (pFifoBlock->EwfCompressionUsed)
   {
      UTIL_MEM_FREE (pBufferUncompressed);  //lint !e673 Possibly inappropriate deallocation
   }

   if (Error)
   {
      LOG_ERROR ("zrc=%d CRC1=%08X CRC2=%08X Size1=%d Size2=%lu EwfPreprocessed=%d, EwfCompressionUsed=%d EwfWriteCRC=%d ",
                  zrc, CRC1, CRC2, pFifoBlock->DataSize, UncompressedSize,
                                   pFifoBlock->EwfPreprocessed?1:0,
                                   pFifoBlock->EwfCompressionUsed,
                                   pFifoBlock->EwfWriteCRC)
   }
   return NO_ERROR;
}

void t_ThreadWrite::run (void)
{
   t_pDevice     pDevice;
   t_pFifoBlock  pFifoBlock;
   t_OutputFile *pOutputFile= NULL;
   bool           Finished  = false;
   quint64        Blocks    = 0;
   APIRET         rc;

   LOG_INFO ("Acquisition of %s: Writing thread started", QSTR_TO_PSZ (pOwn->pDevice->LinuxDevice))

   pDevice = pOwn->pDevice;
   pDevice->SetCurrentWritePos (0LL);

   LOG_INFO ("Deleting existing image files of the same name")
   CHK_EXIT (DeleteImageFiles (false))

   switch (pDevice->Acquisition.Format)
   {
      case t_File::DD:  pOutputFile = new t_OutputFileDD;  break;
      case t_File::EWF: pOutputFile = new t_OutputFileEWF; break;
      case t_File::AFF: pOutputFile = new t_OutputFileAFF; break;
      default: CHK_EXIT (ERROR_THREADWRITE_INVALID_FORMAT)
   }

   rc = pOutputFile->Open (pDevice);
   if (rc == ERROR_THREADWRITE_OPEN_FAILED)
   {
      LOG_INFO ("Could not open destination file %s", QSTR_TO_PSZ (pOwn->pDevice->Acquisition.ImageFilename))
      pDevice->AbortReason  = t_Device::ThreadWriteFileError;
      pDevice->AbortRequest = true;
   }
   else
   {
      CHK_EXIT (rc)
   }

   pOwn->pOutputFile = pOutputFile;  // Only set pOwn->pOutputFile here, after initialisation completed successfully. The reason is,
                                     // that other threads should remain blocked in t_ThreadWrite::GetpFileHandle until this point.

   while (!Finished && !pDevice->AbortRequest)
   {
      if (*(pOwn->pSlowDownRequest))
         msleep (THREADWRITE_SLOWDOWN_SLEEP);

      CHK_EXIT (pDevice->pFifoWrite->Get (pFifoBlock))
      if (pFifoBlock)
      {
         if (pFifoBlock->Nr != Blocks)
         {
            LOG_ERROR ("Fifo block number out of sequence. Expected: %Ld Received: %Ld", Blocks, pFifoBlock->Nr)
            CHK_EXIT (ERROR_THREADWRITE_OUT_OF_SEQUENCE_BLOCK)
         }
         Blocks++;
         rc = pOwn->pOutputFile->Write (pFifoBlock);

         pDevice->IncCurrentWritePos (pFifoBlock->DataSize);
         if (rc == ERROR_THREADWRITE_WRITE_FAILED)
         {
            LOG_ERROR ("Could not write to destination file %s", QSTR_TO_PSZ (pOwn->pDevice->Acquisition.ImageFilename))
            LOG_INFO ("Last block sizes: %d - %d - %zd ", pFifoBlock->BufferSize, pFifoBlock->DataSize, pFifoBlock->EwfDataSize)
            pDevice->AbortReason  = t_Device::ThreadWriteFileError;
            pDevice->AbortRequest = true;
         }
         else
         {
            CHK_EXIT (rc)
         }
         if (pDevice->HasCompressionThreads() && (pDevice->Acquisition.Format == t_File::EWF) && CONFIG(CheckEwfData))
            CHK_EXIT (ThreadWriteCheckData (pFifoBlock))
         CHK_EXIT (t_Fifo::Destroy (pFifoBlock))
      }
      else
      {
         LOG_INFO ("Dummy block")
         Finished = true;
      }
   }

   LOG_INFO ("Waiting for all other threads using the file handle to finish")
   while ((pDevice->pThreadRead != NULL) ||
          (pDevice->pThreadHash != NULL) ||                // Wait until all threads that might use the file handle have finished
          !pDevice->ThreadCompressList.isEmpty())
   {
      msleep (THREADWRITE_WAIT_FOR_HANDLE_GRANULARITY);
   }

   LOG_INFO ("Closing output file")
   if (pOwn->pOutputFile->Opened())
   {
      rc = pOwn->pOutputFile->Close ();
      if (rc == ERROR_THREADWRITE_CLOSE_FAILED)
      {
         pDevice->AbortReason  = t_Device::ThreadWriteFileError;
         pDevice->AbortRequest = true;
      }
   }
   delete pOwn->pOutputFile;
   pOwn->pOutputFile = NULL;
   pDevice->State = t_Device::Cleanup;

   if (pDevice->DeleteAfterAbort)
      CHK_EXIT (DeleteImageFiles (true))

   pDevice->StopTimestamp = QDateTime::currentDateTime();

   LOG_INFO ("Writing thread exits now (device %s, %Ld blocks processed, %Ld bytes written to output file)", QSTR_TO_PSZ (pOwn->pDevice->LinuxDevice), Blocks, pDevice->GetCurrentWritePos ())
}

static APIRET DeleteImageFiles0 (t_pDevice pDevice, const QDir &Dir, const QStringList &NameFilter)
{
   QFileInfoList  FileInfoList;
   QFileInfo      FileInfo;
   QString        Info;
   bool           Success;

   FileInfoList = Dir.entryInfoList (NameFilter, QDir::Files, QDir::Name);
   while (!FileInfoList.isEmpty())
   {
      FileInfo = FileInfoList.takeFirst();
      CHK (pDevice->SetMessage (QObject::tr("Deleting %1") .arg(FileInfo.fileName())))

      Info = "Deleting " + FileInfo.absoluteFilePath() + " - ";
      Success = QFile::remove (FileInfo.absoluteFilePath());
      if (Success)
           Info += "successfull";
      else Info += "could not be deleted";
      LOG_INFO ("%s", QSTR_TO_PSZ(Info))
   }

   return NO_ERROR;
}

APIRET t_ThreadWrite::DeleteImageFiles (bool AlsoDeleteInfoFile)
{
   t_pDevice pDevice = pOwn->pDevice;
   QDir       DirImage (pDevice->Acquisition.ImagePath);
   QDir       DirInfo  (pDevice->Acquisition.InfoPath );
   QString    ExtensionImage;

   CHK (t_File::GetFormatExtension (pDevice->Acquisition.Format, &ExtensionImage))
   CHK (DeleteImageFiles0 (pDevice, DirImage, QStringList(pDevice->Acquisition.ImageFilename + ExtensionImage)))

   if (AlsoDeleteInfoFile)
      CHK (DeleteImageFiles0 (pDevice, DirImage, QStringList(pDevice->Acquisition.InfoFilename + t_File::pExtensionInfo)))

   CHK (pDevice->SetMessage (QString()))

   return NO_ERROR;
}

void t_ThreadWrite::SlotFinished (void)
{
   emit SignalEnded (pOwn->pDevice);
}

APIRET t_ThreadWrite::GetpFileHandle0 (void **ppHandle)
{
   *ppHandle = NULL;

   if (!isRunning())
      return ERROR_THREADWRITE_HANDLE_NOT_YET_AVAILABLE;

   if (!pOwn->pOutputFile)
      return ERROR_THREADWRITE_HANDLE_NOT_YET_AVAILABLE;

   *ppHandle = pOwn->pOutputFile->GetFileHandle();

   if (!*ppHandle)
      return ERROR_THREADWRITE_HANDLE_NOT_YET_AVAILABLE;

   return NO_ERROR;
}


APIRET t_ThreadWrite::GetpFileHandle (void **ppHandle)
{
   unsigned int Wait=0;
   APIRET       rc;

   *ppHandle = NULL;
   do
   {
      rc = GetpFileHandle0 (ppHandle);
      if (rc != ERROR_THREADWRITE_HANDLE_NOT_YET_AVAILABLE)
         CHK (rc)

      if (pOwn->pDevice->AbortRequest)  // May happen, for example, if the write thread wasn't able to open the destination file.
         break;

      Wait += THREADWRITE_WAIT_FOR_HANDLE_GRANULARITY;
      if (Wait > THREADWRITE_WAIT_FOR_HANDLE)
         CHK_EXIT (ERROR_THREADWRITE_HANDLE_TIMEOUT)

      msleep (THREADWRITE_WAIT_FOR_HANDLE_GRANULARITY);
   } while (*ppHandle == NULL);

   return NO_ERROR;
}

