/* upsfetch - library for UPS communications from client programs

   Copyright (C) 1999  Russell Kroll <rkroll@exploits.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              
*/


#include <errno.h>
#include <netdb.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <netinet/in.h>
#include <sys/time.h>
#include <sys/socket.h>
#include "nut_config.h"
#include "nut_version.h"
#include "upsfetch.h"

	int	upserror, syserrno;

char *upsstrerror (int errnum)
{
	char	buf[256];

	switch (errnum) {
		case UPSF_UNKNOWN:
			return (strdup("Unknown error"));
			break;
		case UPSF_NOSUCHVAR: 
			return (strdup("No such variable"));
			break;
		case UPSF_NOSUCHHOST:
			return (strdup("No such host"));
			break;
		case UPSF_SENDFAILURE:
			snprintf (buf, sizeof(buf), "Send failure: %s",
			         strerror(syserrno));
			return (strdup(buf));
			break;
		case UPSF_RECVFAILURE:
			snprintf (buf, sizeof(buf), "Receive failure: %s",
			         strerror(syserrno));
			return (strdup(buf));
			break;
		case UPSF_SOCKFAILURE:
			snprintf (buf, sizeof(buf), "Socket failure: %s",
			         strerror(syserrno));
			return (strdup(buf));
			break;
		case UPSF_BINDFAILURE:
			snprintf (buf, sizeof(buf), "bind failure: %s",
			         strerror(syserrno));
			return (strdup(buf));
			break;
		case UPSF_RECVTIMEOUT:
			return (strdup("Receive timeout"));
			break;
		case UPSF_NOCOMM1:
			return (strdup("Data source is stale"));
			break;
		case UPSF_UNKNOWNUPS:
			return (strdup("Unknown ups name"));
			break;
		case UPSF_ACCESSDENIED:
			return (strdup("Access denied"));
			break;
		case UPSF_CONNFAILURE:
			snprintf (buf, sizeof(buf), "Connection failure: %s",
			         strerror(syserrno));
			return (strdup(buf));
			break;
		case UPSF_READERROR:
			snprintf (buf, sizeof(buf), "Read error: %s",
			         strerror(syserrno));
			return (strdup(buf));
			break;
		case UPSF_WRITEERROR:
			snprintf (buf, sizeof(buf), "Write error: %s",
			         strerror(syserrno));
			return (strdup(buf));
			break;
		case UPSF_INVALIDPW:
			return (strdup("Invalid password"));
			break;
		case UPSF_BADPASSWD:
			return (strdup("Password incorrect"));
			break;
		case UPSF_WRONGVAR:
			return (strdup("Received wrong variable"));
			break;
		case UPSF_UNKNOWNLVL:
			return (strdup("Unknown level"));
			break;
		case UPSF_UNKNOWNCMD:
			return (strdup("Unknown command"));
			break;
		default:
			snprintf (buf, sizeof(buf), "Unknown error 0x%04x", errnum);
			return (strdup(buf));
			break;
	}

	/* notreached */
	return (NULL);
}

/* common error condition handler */
int errcheck (char *buf)
{
	if (!strncmp(buf, "ERR", 3)) {
		upserror = UPSF_UNKNOWN;		/* default error */

		if (!strncmp(buf, "ERR ACCESS-DENIED", 17)) 
			upserror = UPSF_ACCESSDENIED;

		if (!strncmp(buf, "ERR UNKNOWN-UPS", 15))
			upserror = UPSF_UNKNOWNUPS;

		if (!strncmp(buf, "ERR INVALID-PASSWORD", 20))
			upserror = UPSF_INVALIDPW;

		if (!strncmp(buf, "ERR PASSWORD-INCORRECT", 22))
			upserror = UPSF_INVALIDPW;

		return (-1);
	}

	if (!strncmp(buf, "Unknown command", 15)) {
		upserror = UPSF_UNKNOWNCMD;
		return (-1);
	}

	return 0;
}

/* send <req> to <host> and put the answer in <buf> */

int fetch (char *hostin, char *req, char *buf, int buflen)
{
	int	res, udpport = UDPPORT, fd, fromlen, err;
	struct	sockaddr_in xmit, dest, from;
	struct	hostent *serv;
	char	sbuf[512], host[512], *cpos;
	fd_set	rfd;
	struct	timeval tv;

	upserror = UPSF_UNKNOWN;

	snprintf (host, sizeof(host), "%s", hostin);
	cpos = strstr (host, ":");

	if (cpos != NULL) {
		cpos[0] = 0;
		udpport = atoi(++cpos);
	}

	if ((serv = gethostbyname(host)) == (struct hostent *) NULL) {
		upserror = UPSF_NOSUCHHOST;
		return (-1);
	}

	fd = socket (AF_INET, SOCK_DGRAM, 0);
	if (fd < 0) {
		upserror = UPSF_SOCKFAILURE;
		syserrno = errno;
		return (-1);
	}

	memset (&xmit, '\0', sizeof(xmit));
	xmit.sin_family = AF_INET;

	res = bind (fd, (struct sockaddr *) &xmit, sizeof(xmit));
	if (res < 0) {
		upserror = UPSF_BINDFAILURE;
		syserrno = errno;
		close (fd);
		return (-1);
	}

	memset (&dest, '\0', sizeof(dest));
	memcpy (&dest.sin_addr, serv->h_addr, serv->h_length);
	dest.sin_family = AF_INET;
	dest.sin_port = htons(udpport);

	snprintf (sbuf, sizeof(sbuf), "%s", req);
	res = sendto (fd, sbuf, strlen(sbuf), 0, 
	             (struct sockaddr *) &dest, sizeof(dest));

	if (res < 0) {
		close (fd);
		syserrno = errno;
		upserror = UPSF_SENDFAILURE;
		return (-1);	/* failure */
	}

	FD_ZERO (&rfd);
	FD_SET (fd, &rfd);
	tv.tv_sec = 2;	/* 2.0 seconds allowed for reply */
	tv.tv_usec = 0;

	res = select (fd + 1, &rfd, NULL, NULL, &tv);

	if (res > 0) {
		memset (buf, '\0', buflen);
		fromlen = sizeof(from);
		res = recvfrom (fd, buf, buflen, 0, (struct sockaddr *) &from,
		                &fromlen);
	}
	else {
		close (fd);
		syserrno = errno;
		upserror = UPSF_RECVTIMEOUT;
		return (-1);	/* failure */
	}

	if (res < 0) {
		close (fd);
		syserrno = errno;
		upserror = UPSF_RECVFAILURE;
		return (-1);	/* failure */
	}

	close (fd);

	if ((err = errcheck (buf)) != 0)
		return (err);

	upserror = 0;
	return (1);	/* success */
}

int getupsvar (char *hostin, char *varname, char *buf, int buflen)
{
	char	cmd[256], tmp[512], *ptr, *host, *upsname, hosttmp[512],
		vartmp[256];
	int	ret;

	/* host = ups-1@localhost */

	snprintf (hosttmp, sizeof(hosttmp), "%s", hostin);
	ptr = strstr (hosttmp, "@");

	if (ptr != NULL) {
		ptr[0] = 0;
		upsname = hosttmp;
		host = ptr + 1;
		snprintf (cmd, sizeof(cmd), "REQ %s@%s", varname, upsname);
	}
	else {
		upsname = NULL;
		host = hosttmp;
		snprintf (cmd, sizeof(cmd), "REQ %s", varname);
	}

	ret = fetch (host, cmd, tmp, sizeof(tmp));

	if (ret != 1) {
		return (ret);
	}

	tmp[strlen(tmp) - 1] = 0;

	/* ANS <varname> <answer> */

	/* check variable name for sanity */
	snprintf (vartmp, sizeof(vartmp), "%s", &tmp[4]);
	vartmp [strlen(varname)] = 0;

	if (strcasecmp(vartmp, varname) != 0) {
		upserror = UPSF_WRONGVAR;
		return (-1);		/* failure */
	}

	if (upsname == NULL)		/* skip "ANS <varname> " */
		ptr = tmp + strlen(varname) + 5;
	else				/* skip "ANS <varname>@<upsname> " */
		ptr = tmp + strlen(varname) + 5 + strlen(upsname) + 1;

	if (!strcmp(ptr, "NOT-SUPPORTED")) {
		upserror = UPSF_NOSUCHVAR;
		return (-1);	/* failure */
	}

	if (!strcmp(ptr, "DATA-STALE")) {
		upserror = UPSF_NOCOMM1;
		return (-1);	/* failure */
	}

	if (!strcmp(ptr, "UNKNOWN-UPS")) {
		upserror = UPSF_UNKNOWNUPS;
		return (-1);	/* failure */
	}

	if (!strcmp(ptr, "UNKNOWN")) {
		upserror = UPSF_NOSUCHVAR;
		return (-1);	/* failure */
	}

	memset (buf, '\0', buflen);
	strncpy (buf, ptr, buflen);

	return (ret);
}

int getupsvarlist (char *hostin, char *buf, int buflen)
{
	char	tmp[512], *ptr, *upsname, *host, cmd[128], hosttmp[512];
	int	ret;

	/* host = ups-1@localhost */

	snprintf (hosttmp, sizeof(hosttmp), "%s", hostin);
	ptr = strstr (hosttmp, "@");

	if (ptr != NULL) {
		ptr[0] = 0;
		host = ptr + 1;
		upsname = hosttmp;
		snprintf (cmd, sizeof(cmd), "LISTVARS %s", upsname);
	}
	else {
		upsname = NULL;
		host = hosttmp;
		snprintf (cmd, sizeof(cmd), "LISTVARS");
	}

	ret = fetch (host, cmd, tmp, sizeof(tmp));

	if (ret != 1)
		return (ret);

	/* VARS <varlist> */
	tmp[strlen(tmp) - 1] = 0;

	if (upsname == NULL)
		ptr = tmp + 5;	/* skip "VARS " */
	else
		ptr = tmp + 5 + strlen(upsname) + 2;  /* skip "VARS <name> " */

	memset (buf, '\0', buflen);
	strncpy (buf, ptr, buflen);

	return (ret);
}

int upsconnect (char *hostin)
{
	struct	sockaddr_in local, server;
	struct	hostent *serv;
	int	fd, port = UDPPORT;
	char	*cpos, *host;

	host = strdup (hostin);
	cpos = strstr (host, ":");
	if (cpos != NULL) {
		cpos[0] = 0;
		port = atoi(++cpos);
	}

	if ((serv = gethostbyname(host)) == (struct hostent *) NULL) {
		upserror = UPSF_NOSUCHHOST;
		return (-1);
	}

	if ((fd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
		upserror = UPSF_SOCKFAILURE;
		syserrno = errno;
		return (-1);
	}		

	memset (&local, '\0', sizeof(struct sockaddr_in));
	local.sin_family = AF_INET;
	local.sin_port  = htons (INADDR_ANY);

	memset (&server, '\0', sizeof(struct sockaddr_in));
	server.sin_family = AF_INET;

	server.sin_port = htons (port);
	memcpy (&server.sin_addr, serv->h_addr, serv->h_length);

	if (bind (fd, (struct sockaddr *) &local, sizeof(struct sockaddr_in)) == -1) {
		upserror = UPSF_BINDFAILURE;
		syserrno = errno;
		close (fd);
		return (-1);
	}

	if (connect (fd, (struct sockaddr *) &server, sizeof(struct sockaddr_in)) == -1) {
		upserror = UPSF_CONNFAILURE;
		syserrno = errno;
		close (fd);
		return (-1);
	}

	return (fd);
}

void closeupsfd (int fd)
{
	shutdown (fd, 2);
	close (fd);
}

/* read from <fd> until a LF */
int ups_readline (int fd, char *tmp, int buflen)
{
	int	tmpptr = 0, ret;
	char	ch;

	for (;;) {
		ret = read (fd, &ch, 1);

		if (ret < 1) {
			upserror = UPSF_READERROR;
			syserrno = errno;
			return (ret);
		}

		tmp[tmpptr++] = ch;
		
		if ((ch == 10) || (tmpptr >= buflen))
			return 0;
	}

	return 0;
}

int getupsvarlistfd (int fd, char *upsname, char *buf, int buflen)
{
	char	tmp[512], *ptr, cmd[128];
	int	ret, err;

	if (upsname == NULL)
		snprintf (cmd, sizeof(cmd), "LISTVARS\n");
	else
		snprintf (cmd, sizeof(cmd), "LISTVARS %s\n", upsname);

	ret = write (fd, cmd, strlen(cmd));

	if (ret < 1) {
		upserror = UPSF_WRITEERROR;
		syserrno = errno;
		return (-1);
	}

	memset (tmp, '\0', sizeof(tmp));

	if ((err = ups_readline (fd, tmp, buflen)) != 0)
		return (err);

	if ((err = errcheck (tmp)) != 0)
		return (err);

	/* VARS <varlist> */
	tmp[strlen(tmp) - 1] = 0;

	if (upsname == NULL)
		ptr = tmp + 5;	/* skip "VARS " */
	else
		ptr = tmp + 5 + strlen(upsname) + 2;  /* skip "VARS <name> " */

	memset (buf, '\0', buflen);
	strncpy (buf, ptr, buflen);

	return (ret);
}

int getupsvarfd (int fd, char *upsname, char *varname, char *buf, int buflen)
{
	char	cmd[256], tmp[512], *ptr;
	int	ret, err;

	if (upsname == NULL)
		snprintf (cmd, sizeof(cmd), "REQ %s\n", varname);
	else
		snprintf (cmd, sizeof(cmd), "REQ %s@%s\n", varname, upsname);

	ret = write (fd, cmd, strlen(cmd));

	if (ret < 1) {
		upserror = UPSF_WRITEERROR;
		syserrno = errno;
		return (-1);
	}

	memset (tmp, '\0', sizeof(tmp));

	if ((err = ups_readline (fd, tmp, buflen)) != 0)
		return (err);

	if ((err = errcheck (tmp)) != 0)
		return (err);

	tmp[strlen(tmp) - 1] = 0;
	/* ANS <varname> <answer> */

	if (upsname == NULL)		/* skip "ANS <varname> " */
		ptr = tmp + strlen(varname) + 5;
	else				/* skip "ANS <varname>@<upsname> " */
		ptr = tmp + strlen(varname) + 5 + strlen(upsname) + 1;

	if (!strcmp(ptr, "NOT-SUPPORTED")) {
		upserror = UPSF_NOSUCHVAR;
		return (-1);	/* failure */
	}

	if (!strcmp(ptr, "DATA-STALE")) {
		upserror = UPSF_NOCOMM1;
		return (-1);	/* failure */
	}

	if (!strcmp(ptr, "UNKNOWN-UPS")) {
		upserror = UPSF_UNKNOWNUPS;
		return (-1);	/* failure */
	}

	memset (buf, '\0', buflen);
	strncpy (buf, ptr, buflen);

	return (ret);
}

int upslogin (int fd, char *upsname, char *password)
{
	char	cmd[256], tmp[256];
	int	err, ret;

	/* send the password */

	if ((password == NULL) || (strlen(password) == 0))
		snprintf (cmd, sizeof(cmd), "PASSWORD\n");
	else
		snprintf (cmd, sizeof(cmd), "PASSWORD %s\n", password);

	ret = write (fd, cmd, strlen(cmd));

	if (ret < 1) {
		upserror = UPSF_WRITEERROR;
		syserrno = errno;
		return (-1);
	}

	memset (tmp, '\0', sizeof(tmp));

	if ((err = ups_readline (fd, tmp, sizeof(tmp))) != 0)
		return (err);

	if ((err = errcheck (tmp)) != 0)
		return (err);

	tmp[strlen(tmp) - 1] = 0;

	if (strcmp(tmp, "OK") != 0) {
		/* something happened but it's not OK or ERR */
		upserror = UPSF_UNKNOWN;
		return (-1);	/* failure */
	}

	/* now attempt login */

	if ((upsname == NULL) || (strlen(upsname) == 0))
		snprintf (cmd, sizeof(cmd), "LOGIN\n");
	else
		snprintf (cmd, sizeof(cmd), "LOGIN %s\n", upsname);

	ret = write (fd, cmd, strlen(cmd));

	if (ret < 1) {
		upserror = UPSF_WRITEERROR;
		syserrno = errno;
		return (-1);
	}

	memset (tmp, '\0', sizeof(tmp));

	if ((err = ups_readline (fd, tmp, sizeof(tmp))) != 0)
		return (err);

	if ((err = errcheck (tmp)) != 0)
		return (err);

	tmp[strlen(tmp) - 1] = 0;

	if (!strcmp(tmp, "OK"))
		return (0);	/* success */

	/* something happened but it's not OK or ERR */
	upserror = UPSF_UNKNOWN;
	return (-1);	/* failure */
}

int upsgetprivs (int fd, int level)
{
	char	cmd[256], tmp[256];
	int	err, ret;

	upserror = UPSF_UNKNOWN;

	switch (level) {
		case UPSPRIV_MASTER:
			snprintf (cmd, sizeof(cmd), "MASTER\n");
			break;

		default:
			upserror = UPSF_UNKNOWNLVL;
			return (-1);
	}

	ret = write (fd, cmd, strlen(cmd));

	if (ret < 1) {
		upserror = UPSF_WRITEERROR;
		syserrno = errno;
		return (-1);
	}

	memset (tmp, '\0', sizeof(tmp));

	if ((err = ups_readline (fd, tmp, sizeof(tmp))) != 0)
		return (err);

	if ((err = errcheck (tmp)) != 0)
		return (err);

	tmp[strlen(tmp) - 1] = 0;

	if (strcmp(tmp, "OK") != 0) {
		/* something happened but it's not OK or ERR */
		upserror = UPSF_UNKNOWN;
		return (-1);	/* failure */
	}

	return (1);	/* success */
}

int upssendraw (int fd, char *cmd)
{
	int	ret;

	upserror = UPSF_UNKNOWN;

	ret = write (fd, cmd, strlen(cmd));

	if (ret < 1) {
		upserror = UPSF_WRITEERROR;
		syserrno = errno;
		return (-1);
	}

	upserror = 0;
	return (1);		/* success */
}

int upsreadraw (int fd, char *buf, int buflen)
{
	char	tmp[256];
	int	err;

	memset (tmp, '\0', sizeof(tmp));

	if ((err = ups_readline (fd, tmp, sizeof(tmp))) != 0)
		return (err);

	if ((err = errcheck (tmp)) != 0)
		return (err);

	snprintf (buf, buflen, tmp);

	upserror = 0;
	return (1);	/* success */
}

