/* Guile SimpleSQL - the basic functions
   
   Copyright (C) 2001, 2004 David J. Lambert <d.j.lambert@sms.ed.ac.uk>
   Copyright (C) 1999 forcer <forcer@mindless.com>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
   USA */

#include "guile-simplesql.h"

#include <guile/gh.h>
#include <libguile.h>
#include <libguile/error.h>
#include <string.h>
#include <stdio.h>

#if TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# if HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif

#ifdef HAVE_MYSQL
#include "simplesql-mysql.h"
#endif 

#ifdef HAVE_POSTGRESQL
#include "simplesql-postgresql.h"
#endif 

struct sq_db_procs {
        char *api_name;
        struct sq_db_procs *next;
        sql_open_func open;
        sql_query_func query;
        sql_close_func close;
};


struct sq_db_smob
{
    void *database_connection;
    struct sq_db_procs *procs;
    int closed;
};

/* A list of all registered api's */
static struct sq_db_procs *api_list;

/* The smob tag for our db smob */
static long sq_db_smob_tag;

static struct sq_db_procs *sq_lookup_api(char *api);

static size_t sq_db_smob_free (SCM db);
static int sq_db_smob_print (SCM, SCM, scm_print_state *);

static struct sq_db_smob *sq_scm2sq_db_smob(SCM db);
static SCM sq_make_db_smob_scm (void *, const char *, struct sq_db_procs *);

static char *sq_string_join (SCM lis, int *reslen, const char *proc, int argn);


#define SIMPLESQL_SMOB_P(obj) ((SCM_NIMP(obj)) && (SCM_TYP16(obj)==sq_db_smob_tag))



#ifndef HAVE_STRDUP
/* let's define it */
char *strdup(const char *s);

char *strdup(const char *s) {
        char *dst;
        dst = malloc(strlen(s)+1);
        if(dst == NULL)
                return NULL;
        do {
                *dst++ = *s;
        } while(*s++);
        return dst;
}

#endif /* HAVE_STRDUP */

/*
 * Function to register new backends. Can be used at runtime to add
 * new backends.
 * api is the name for the API
 * open is the function to be called for the database to be opened,
 *      and should return a pointer that the API can use to refer to
 *      this connection lateron
 * query is a function that performs the actualy query and returns the result
 * close is a function that just closes the database connection again and
 *       frees related memory. It's called for garbage collection too!
 */
int
sql_register_api(const char *api, sql_open_func open, 
                 sql_query_func query, sql_close_func close)
{
        struct sq_db_procs *apil;
        char *newapi = strdup(api);

        if(newapi == NULL)
                return 0;

        if(api_list == NULL){
                api_list = malloc(sizeof(struct sq_db_procs));
                if(api_list == NULL){
                       	free(newapi);
                        return 0;
                }
                apil = api_list;
        }else{
                apil = api_list;
                while(apil->next != NULL)
                        apil = apil->next;
                apil->next = malloc(sizeof(struct sq_db_procs));
                if(apil->next == NULL){
                        free(newapi);
                        return 0;
                }
                apil = apil->next;
        }
        apil->api_name = newapi;
        apil->next = NULL;
        apil->open = open;
        apil->query = query;
        apil->close = close;
        return 1;
}

/*
 * Just returns the struct with the associated functions for the
 * specified API
 */
static struct sq_db_procs *
sq_lookup_api(char *api)
{
        struct sq_db_procs *apil;
        for(apil = api_list; apil; apil = apil->next){
                if(strcasecmp(apil->api_name, api) == 0)
                        return apil;
        }
        return NULL;
}

/*
 * The free function for the db smob
 * Call the close function for the db if and only if it hasn't been
 * called before
 */
static size_t 
sq_db_smob_free (SCM obj)
{
    struct sq_db_smob *db = sq_scm2sq_db_smob(obj);
    if(!db->closed)
	db->procs->close(db->database_connection);
    scm_must_free(db);
    return sizeof(struct sq_db_smob);
}


/* Print function for the db smob.  */
static int 
sq_db_smob_print (SCM obj, SCM port, scm_print_state *state)
{
    char buf[3*sizeof(void*)];
    struct sq_db_smob *db = sq_scm2sq_db_smob (obj);
    
    scm_puts ("#<simplesql-database ", port);
    sprintf (buf, "%p ", db);
    scm_puts (buf, port);
    if (db->closed)
	scm_puts ("closed", port);
    else 
    {
	scm_puts ("\"", port);
	scm_puts ((char *) (db+1), port);
	scm_puts ("\"", port);
    }
    scm_puts (">", port);
    return 1;
}


/*
 * Conversion function. Check type and convert from SCM to the db smob.
 */
static struct sq_db_smob *
sq_scm2sq_db_smob (SCM db)
{
    return (struct sq_db_smob *)SCM_SMOB_DATA(db);
}


/* Make a Scheme object for the database connection.  */
static SCM
sq_make_db_smob_scm (void *database_connection, const char *database_name, struct sq_db_procs *vprocs)
{
    struct sq_db_smob *db;
    int name_len = strlen (database_name);

    db = scm_must_malloc (sizeof (struct sq_db_smob) + name_len + 1,
			  "SimpleSQL database SMOB");
    strcpy ((char *) (db + 1), database_name);
    printf ("\"%s\", \"%s\"", database_name, (char *) (db+1));
    db->closed = 0;
    db->database_connection = database_connection;
    db->procs = vprocs;
    SCM_RETURN_NEWSMOB(sq_db_smob_tag, db);
}


/*
 * Function to convert a list of strings|u8v's to a char*
 * FIXME: handle u8 vectors!
 */
static char *
sq_string_join(SCM lis, int *reslen, const char *proc, int argn)
{
        int len = 0;
        SCM cur, s;
        char *result, *dst;

        for(cur=lis; !SCM_NULLP(cur); cur = SCM_CDR(cur)){
                SCM_ASSERT(SCM_CONSP(cur), lis, argn, proc);
                s = SCM_CAR(cur);
                SCM_ASSERT(SCM_NIMP(s)
                           && SCM_STRINGP(s),
                           lis, argn, proc);
                len += SCM_STRING_LENGTH(s);
        }
        result = malloc(len);
        *reslen = len;
        dst = result;
        for(cur=lis; !SCM_NULLP(cur); cur = SCM_CDR(cur)){
                s = SCM_CAR(cur);
                memcpy(dst, SCM_STRING_CHARS(s),
                       SCM_STRING_LENGTH(s));
                dst += SCM_STRING_LENGTH(s);
        }
        return result;
}

/*
 * Function to grab a LATIN 1 string
 * FIXME: convert from latin1!
 */
SCM
sq_latin1_string(char *str, int len)
{
    return scm_mem2string (str, len);
}

/*
 * Function to grab a byte vector
 * FIXME: actually use u8vec
 */
SCM
sq_binary_vector(char *str, int len)
{
    return scm_mem2string (str, len);
}

/*
 * The actual sql-open function
 * Check types, convert them to C values, and just pass them to the
 * API function
 */
SCM_DEFINE (simplesql_open, "%simplesql-open", 6, 0, 0,
	    (SCM api, SCM database, SCM host, SCM port_or_socket, SCM user, SCM password),
	    "")
#define FUNC_NAME s_simplesql_open
{
        struct sq_db_procs *api_procs;
        char *database_buf, *host_buf, *user_buf, *password_buf, *socket;
	int port;
        void *db;
        SCM err;

        SCM_ASSERT(SCM_NIMP(api)&&SCM_SYMBOLP(api), 
		   api, SCM_ARG1, FUNC_NAME);
        SCM_ASSERT(SCM_NIMP(database)&&SCM_STRINGP(database), 
		   database, SCM_ARG2, FUNC_NAME);
        SCM_ASSERT((SCM_NIMP(host)&&SCM_STRINGP(host)) || SCM_FALSEP (host),
		   host, SCM_ARG3, FUNC_NAME);
        SCM_ASSERT((SCM_NIMP(port_or_socket)&&SCM_STRINGP(port_or_socket)) ||
		   (SCM_INUMP(port_or_socket)) || SCM_FALSEP (port_or_socket),
		   port_or_socket, SCM_ARG4, FUNC_NAME);
	SCM_ASSERT((SCM_NIMP(user)&&SCM_STRINGP(user)) || SCM_FALSEP (user),
		   user, SCM_ARG5, FUNC_NAME);
        SCM_ASSERT((SCM_NIMP(password)&&SCM_STRINGP(password)) || SCM_FALSEP (password),
		   password, SCM_ARG6, FUNC_NAME);

        api_procs = sq_lookup_api (SCM_SYMBOL_CHARS (api));
        if(api_procs == NULL)
                scm_misc_error(FUNC_NAME, "SQL API ~A not registered",
                               scm_cons(api, SCM_EOL));

        database_buf = SCM_STRING_CHARS (database);
        host_buf = SCM_FALSEP (host) ? NULL : SCM_STRING_CHARS (host);
	user_buf = SCM_FALSEP (user) ? NULL : SCM_STRING_CHARS (user);
        password_buf = SCM_FALSEP (password) ? NULL : SCM_STRING_CHARS (password);

	socket = (SCM_NIMP(port_or_socket) && SCM_STRINGP (port_or_socket))
	    ? SCM_STRING_CHARS (port_or_socket) : NULL;
	port = SCM_INUMP (port_or_socket) ? SCM_INUM (port_or_socket) : 0;

        db = api_procs->open(database_buf, host_buf, port, socket, user_buf, password_buf, &err);
        if(db == NULL)
	    scm_misc_error(FUNC_NAME,
			   "Connection to ~A server failed: ~A",
			   scm_cons (api, scm_cons (err, SCM_EOL)));

        return sq_make_db_smob_scm (db, database_buf, api_procs);
}
#undef FUNC_NAME

/*
 * The actual sql-query function
 * query is a list of queries which have to be concatanated,
 * with strings being converted to latin1 and u8vectors left as they
 * are
 */


SCM_DEFINE (simplesql_query, "simplesql-query", 1, 0, 1,
	    (SCM sdb, SCM query),
	    "")
#define FUNC_NAME s_simplesql_query
{
    SCM lis;
    char *q;
    int qlen;
    struct sq_db_smob *db;

    SCM_ASSERT (SIMPLESQL_SMOB_P (sdb), sdb, SCM_ARG1, FUNC_NAME);
    
    if (SCM_NULLP (query))
	scm_wrong_num_args(scm_makfrom0str (FUNC_NAME));
    
    db = sq_scm2sq_db_smob(sdb);
    if (db->closed)
	scm_wrong_type_arg(FUNC_NAME, SCM_ARG1, sdb);
    
    q = sq_string_join(query, &qlen, FUNC_NAME, SCM_ARG2);
    lis = db->procs->query(db->database_connection, q, qlen);
    free (q);
    return lis;
}
#undef FUNC_NAME


/*
 * The actual sql-close function
 * Close the connection if it's not closed yet, using the API function,
 * but don't free the actual smob - it'll be garbage collected
 */
SCM_DEFINE (simplesql_close, "simplesql-close", 1, 0, 0,
	    (SCM sdb),
	    "")
#define FUNC_NAME s_simplesql_close
{
    struct sq_db_smob *db;

    SCM_ASSERT (SIMPLESQL_SMOB_P (sdb), sdb, SCM_ARG1, FUNC_NAME);
    
    db = sq_scm2sq_db_smob(sdb);

    if(!db->closed){
	db->closed = 1;
	db->procs->close(db->database_connection);
    }
    return SCM_UNSPECIFIED;
}
#undef FUNC_NAME

/*
 * Predicate to determine wether the object is one of our
 * db smobs
 */

SCM_DEFINE (simplesql_database_p, "simplesql-database?", 1, 0, 0,
	    (SCM obj),
	    "")
#define FUNC_NAME s_simplesql_database_p
{
    return (SCM_NIMP(obj)&&(SCM_TYP16(obj)==sq_db_smob_tag))
	? SCM_BOOL_T
	: SCM_BOOL_F;
}
#undef FUNC_NAME

/*
 * Escape a string suitable for SQL
 * To be escaped: \0 \n \r \\ \' \"
 * Takes both u8vectors and strings, and returns the type it was given,
 * just escaped
 * 
 * Because we want to be compatible, we should use latin1 for strings.
 * Bytevectors have to be escaped too, though (blobs).
 * Returns the type it was given, just escaped, and strings in a way they'd
 * show up as latin1
 *
 * FIXME: use multibytes
 */
SCM_DEFINE (simplesql_escape, "simplesql-escape", 1, 0, 0,
	    (SCM str),
	    "")
#define FUNC_NAME s_simplesql_escape
{
        int len, count;
        char *orig, *src, *dst;

        SCM_ASSERT(SCM_NIMP(str)&&SCM_STRING_CHARS(str), str,
                   SCM_ARG1, FUNC_NAME);

        count = len = SCM_STRING_LENGTH (str);
        src = SCM_STRING_CHARS(str);
        orig = dst = scm_must_malloc ((len*2)+1, FUNC_NAME);

        while(len--){
                switch(*src){
                    case '\0':
                        *dst++ = '\\';
                        *dst++ = '0';
                        break;
                    case '\n':
                        *dst++ = '\\';
                        *dst++ = 'n';
                        break;
                    case '\r':
                        *dst++ = '\\';
                        *dst++ = 'r';
                        break;
                    case '\\':
                        *dst++ = '\\';
                        *dst++ = '\\';
                        break;
                    case '\'':
                        *dst++ = '\\';
                        *dst++ = '\'';
                        break;
                    case '\"':
                        *dst++ = '\\';
                        *dst++ = '\"';
                        break;
                    default:
                        *dst++ = *src;
                }
                src++;
        }
        return scm_take_str(orig, dst-orig);
}
#undef FUNC_NAME


/* Borrowed from libguile/stime.c.  */
SCM
_simplesql_filltime (struct tm *bd_time, int zoff, char *zname)
{
    SCM result = scm_make_vector (SCM_MAKINUM (11), SCM_UNDEFINED);

    SCM_VECTOR_SET (result, 0, SCM_MAKINUM (bd_time->tm_sec));
    SCM_VECTOR_SET (result, 1, SCM_MAKINUM (bd_time->tm_min));
    SCM_VECTOR_SET (result, 2, SCM_MAKINUM (bd_time->tm_hour));
    SCM_VECTOR_SET (result, 3, SCM_MAKINUM (bd_time->tm_mday));
    SCM_VECTOR_SET (result, 4, SCM_MAKINUM (bd_time->tm_mon));
    SCM_VECTOR_SET (result, 5, SCM_MAKINUM (bd_time->tm_year));
    SCM_VECTOR_SET (result, 6, SCM_MAKINUM (bd_time->tm_wday));
    SCM_VECTOR_SET (result, 7, SCM_MAKINUM (bd_time->tm_yday));
    SCM_VECTOR_SET (result, 8, SCM_MAKINUM (bd_time->tm_isdst));
    SCM_VECTOR_SET (result, 9, SCM_MAKINUM (zoff));
    SCM_VECTOR_SET (result, 10, zname ? scm_makfrom0str (zname) : SCM_BOOL_F);
    return result;
}


/* Initialisation function to be called when Guile loads this
   extension. */
void
simplesql_extension_init ()
{
    api_list = NULL;
    
    sq_db_smob_tag = scm_make_smob_type ("simplesql-db", sizeof(struct sq_db_smob));
    scm_set_smob_mark (sq_db_smob_tag, NULL);
    scm_set_smob_free (sq_db_smob_tag, sq_db_smob_free);
    scm_set_smob_print (sq_db_smob_tag, sq_db_smob_print);
    scm_set_smob_equalp (sq_db_smob_tag, NULL);

#ifndef SCM_MAGIC_SNARFER
#include "guile-simplesql.x"
#endif

#ifdef HAVE_MYSQL
        sq_mysql_init();
#endif
#ifdef HAVE_POSTGRESQL
        sq_pgsql_init();
#endif
}

