/* handling functions for cdrom drives installed in Gnome Toaster */

#include <gtk/gtk.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>

#include "config.h"
#include "int.h"

#include "cddrives.h"
#include "datatrack.h"
#include "audiotrack.h"
#include "streams.h"
#include "preferences.h"
#include "varman.h"
#include "varmanwidgets.h"
#include "tracks.h"
#include "record.h"
#include "cdromlow.h"
#include "updatehandlers.h"
#include "configfile.h"
#include "dialog.h"
#include "forms.h"
#include "main.h"
#include "calc.h"
#include "piping.h"
#include "stdfiletrack.h"

#include "recorder.xpm"

#ifdef HAVE_PTHREADS
# include <pthread.h>
#endif

/* uncomment for debugging */
// #define DEBUG

GdkPixmap *cddrives_recorder_pixmap;
GdkBitmap *cddrives_recorder_mask;

cddrives_drives_def cddrives_drives;
volatile int cddrives_update=1;
volatile int cddrives_updatethreadexit=0;
int cddrives_updatethreadrunning=0;

void cddrives_updatedrivelist();
int cddrives_checkcdchange(gpointer);

/* see,if update process is active. wait for another second if this was the
 * case to make sure the cdrom drive has regained control of itself */
#ifdef HAVE_PTHREADS
# define LOCKDB pthread_mutex_lock(&cddrives_drives.locked)
# define UNLOCKDB pthread_mutex_unlock(&cddrives_drives.locked)
#else
# define LOCKDB
# define UNLOCKDB
#endif

/* this function will usually be executed within a separate thread,
 * called by the cddrives_mthreadhandler().
 * However, if pthread support has been disabled in the configure script,
 * it's gonna get executed by the drive update timeout handler */
void cddrives_updatemediachangeflags()
{
   int x;
   cddrives_cdinfo *d;

   if (cddrives_update>0)
     {
#ifdef DEBUG
	printf("cddrives_updatemediachangeflags: updating media change flags\n");
#endif
	LOCKDB;
	for (x=0;x<cddrives_drives.drivecount;x++)
	  {
	     gtoaster_handle_t tochandle=0;
#ifdef DEBUG
	     printf ("cddrives_updatemediachangeflags: checking drive #%i\n",x);
#endif
	     d=gtk_clist_get_row_data(GTK_CLIST(cddrives_drives.drivelist),x);
	     tochandle=cdromlow_gettochandle(d->device);
	     if ((!d->mediachange)&&(tochandle)&&
		 (d->es.cddb_key!=
		  cdromlow_cddbnumber(tochandle)))
	       d->mediachange=1;
	     if (tochandle)
	       cdromlow_closetochandle(tochandle);
	  }
	;
	UNLOCKDB;
     };
}
;

#ifdef HAVE_PTHREADS
void *cddrives_mthreadhandler(void *dummy)
{
   while (1)
     {
	usleep(3000000);
	/* if the thread should exit */
	if (cddrives_updatethreadexit)
	  {
# ifdef DEBUG
	     printf ("cddrives_mthreadhandler: exiting on request from the main app\n");
# endif
	     pthread_exit(NULL);
	  };
	cddrives_updatemediachangeflags();
     };
};
#endif

void cddrives_enablecdchangedetection()
{

/* spawn cd change detect in a secondary thread to prevent a laggy cdrom
 * from blocking gnometoaster's user interface */
#ifdef HAVE_PTHREADS
   pthread_t *thread;

   cddrives_update=1;
   cddrives_updatethreadexit=0;
   thread=(pthread_t*)malloc(sizeof(pthread_t));

# ifdef DEBUG
   printf ("cddrives_enablecdchangedetection: spawning media change thread.\n");
# endif
   cddrives_updatethreadrunning=
     (pthread_create(thread,NULL,cddrives_mthreadhandler,NULL)==0);
#else
   cddrives_updatethreadrunning=0;
#endif

   gtk_timeout_add(5000,
		   cddrives_checkcdchange,
		   NULL);
}
;

void cddrives_disablecdchangedetection()
{
   /* stop all update handlers */
   cddrives_updatethreadexit=1;
}
;

void cddrives_destroy()
{
   /* make the cd change detect thread exit */
   cddrives_disablecdchangedetection();
};

char *cddrives_dltitles[]=
{
   "",
     N_("Drive Type"),
   N_("Manufacturer Name"),
   N_("Device Name"),
   N_("Scsi ID"),
   N_("Mountpoint")
};

/* this loads the drive list out of a given position of the config file */
void cddrives_loaddrivelist(char *filename,char *section)
{
   FILE *f;
   char name[256];
   char value[256]; /* set real vars depending on name
		     * to make the config file entries extensible */
   int  nextent;    /* this var is set true to indicate that one registry
		     * entry has been read completely and that we're now
		     * ready to proceed to the next one */
   int  endoflist=0;
   cddrives_cdinfo *e;

   f=fopen(filename,"r");
   if (f!=NULL)
     {
	if (configfile_seeksection(f,section))
	  {
	     do
	       {
		  /* create dummy entry */
		  e=cddrives_cdinfo_create("",
					   "",
					   "",
					   "",
					   "",
					   0,0);
		  nextent=0;
		  do
		    {
		       configfile_getnextentry(f,name,value);
		       if (!strcasecmp(name,"model"))
			 strcpy(e->model,value);
		       if (!strcasecmp(name,"manufacturer"))
			 strcpy(e->manufacturer,value);
		       if (!strcasecmp(name,"device"))
			 strcpy(e->device,value);
		       if (!strcasecmp(name,"scsiid"))
			 strcpy(e->scsiid,value);
		       if (!strcasecmp(name,"mountpoint"))
			 strcpy(e->mountpoint,value);
		       if (!strcasecmp(name,"dae_usesg"))
			 e->dae_usesg=(!strcasecmp(value,"1"));
		       if (!strcasecmp(name,"is_active"))
			 e->is_active=(!strcasecmp(value,"1"));		       
		       if (!strcasecmp(name,"is_recorder"))
			 {
			    e->is_recorder=
			      (!strcasecmp(value,"1"));
			    /* synchronize to this entry */
			    nextent=1;
			 };
		    }
		  while ((!nextent)&&(strlen(name)>0));
		  if (strlen(name)>0)
		    cddrives_register_drive(e);
		  else
		    {
		       cddrives_cdinfo_destroy(e);
		       endoflist=1;
		    };
	       }
	     while (!endoflist);
	  };
	/* FIXME: if no drives could be loaded from the config file,
	 * issue a warning and perhaps bring the user into the drive setup,
	 * maybe even perform a bus scan */
	fclose(f);
     };
   /* FIXME: If the config file couldn't be found,
    * issue a warning and perhaps bring the user into the drive setup,
    * maybe even perform a bus scan */
};

void cddrives_storedrivelist(FILE *fd,gpointer data)
{
   int x;
   cddrives_cdinfo *i;

   for (x=0;x<cddrives_drives.drivecount;x++)
     {
	i=(cddrives_cdinfo*)gtk_clist_get_row_data(GTK_CLIST(cddrives_drives.drivelist),x);
	fprintf(fd,"model=%s\n",i->model);
	fprintf(fd,"manufacturer=%s\n",i->manufacturer);
	fprintf(fd,"device=%s\n",i->device);
	fprintf(fd,"scsiid=%s\n",i->scsiid);
	fprintf(fd,"mountpoint=%s\n",i->mountpoint);
	fprintf(fd,"dae_usesg=%i\n",i->dae_usesg);
	fprintf(fd,"is_active=%i\n",i->is_active);
	fprintf(fd,"is_recorder=%i\n",i->is_recorder);
     };
};

#define CDDRIVES_SCANBUS_MAXOUTPUT 16384

#define BUSMAX 3
#define IDMAX 7
#define LUNMAX 0

/* get mountpoint from lookup file (usually /etc/fstab) */
void cddrives_getmountpoint(char *device,char *mountpoint)
{
   char *func=NULL;
   int fd;
   char lmp[255];

   if (strlen(device)>0)
     {
	fd=open(varman_getvar(global_defs,"rec_inqdrive_getmountpoint_file"),O_RDONLY);
	if (fd!=-1)
	  {
	     OUTBUF_LOCK;
	     /* FIXME: maybe read file line by line for very large ones */
	     read(fd,outbuf,CDDRIVES_SCANBUS_MAXOUTPUT);
	     close(fd);

	     func=varman_getvar_copy(global_defs,"rec_inqdrive_getmountpoint_exp");
	     varman_replacestring(func,"$device",device);
	     calc_calculate(func,
			    lmp);
	     strcpy(mountpoint,&lmp[1]);
	     if (strchr(mountpoint,'"')!=NULL)
	       *strchr(mountpoint,'"')=0;
	     free(func);
	     OUTBUF_UNLOCK;
	  }
	else
	  {
	     int num;

	     num=device[strlen(device)-1]-(int)'0';
	     if (num>0)
	       sprintf(mountpoint,"/mnt/cdrom%i",num);
	     else
	       strcpy(mountpoint,"/mnt/cdrom");
	  };
     }
   else
     strcpy(mountpoint,"");
};

/* This function returns a list of scsi ids that have been found to
 * belong to a cdrom/cd-recording device. To obtain this list,
 * this implementation will traditionally call 'cdrecord -scanbus'
 * and parse it's output */
void cddrives_getscanrange(GList **scanlist)
{
   if (scanlist)
     {
	int fd=-1;
	piping_create(varman_getvar_copy(global_defs,"rec_scanbus"),NULL,&fd,NULL);
	if (fd!=-1)
	  {
	     FILE *tfd = fdopen(fd,"r");
	     if (tfd!=NULL)
	       {
		  while (!feof(tfd))
		    {
		       char scsiid[256];
		       scsiid[0]=0;

		       OUTBUF_LOCK;
		       fgets(outbuf,CDDRIVES_SCANBUS_MAXOUTPUT,tfd);

		       calc_calculate(varman_getvar(global_defs,"rec_sbgetid"),
				      scsiid);
		       // Cut quotation marks first
		       if (scsiid[0]=='\"')
			 strcpy(scsiid,&scsiid[1]);
		       if (strchr(scsiid,'\"'))
			 *strchr(scsiid,'\"')=0;
		       if (strlen(scsiid))
			 {
			    // Remember our ID
			    *scanlist=g_list_prepend(*scanlist,strdup(scsiid));
			 };
#ifdef DEBUG
		       printf("Parsed '%s', got '%s'\n", outbuf,scsiid);
#endif

		       OUTBUF_UNLOCK;
		    };
		  fclose(tfd);
		  // Sort our SCSI ids alphabetically to be able
		  // to guess the corresponding device special files
		  *scanlist = g_list_sort(*scanlist,(GCompareFunc)strcmp);
	       };
	  };
     };
};

void cddrives_scanbus()
{
   char *call;
   char scsiid[6];
   char vendor[256];
   char model[256];
   char type[256];
   char drivetype[256];
   int devicenum=0;
   char device[63];
   /* FIXME: get mountpoint from fstab */
   char mountpoint[255];
   cddrives_cdinfo *i;

   GList *scanids = NULL;
   GList *current = NULL;

   cddrives_getscanrange(&scanids);

   /* scan scsi system for cdrom drives */
   for (current=scanids;current!=NULL;current=current->next)
     {
	strncpy(scsiid,(const char*)current->data,sizeof(scsiid));
	OUTBUF_LOCK;
	call=varman_getvar_copy(global_defs,"rec_inqdrive");
	varman_replacevars(global_defs,call);
	varman_replacestring(call,"$scsiid",scsiid);
	piping_create_getoutput(call,outbuf,CDDRIVES_SCANBUS_MAXOUTPUT,
				PIPING_WATCHALL);
	free(call);
	calc_calculate(varman_getvar(global_defs,"rec_inqdrive_getvendor"),
		       vendor);
	calc_calculate(varman_getvar(global_defs,"rec_inqdrive_getmodel"),
		       model);
	calc_calculate(varman_getvar(global_defs,"rec_inqdrive_gettype"),
		       type);
	calc_calculate(varman_getvar(global_defs,"rec_inqdrive_getdrivetype"),
		       drivetype);
	OUTBUF_UNLOCK;
	    /* if entry is valid */
	if ((strlen(vendor)>0)&&
	    (strlen(model)>0)&&
	    (strlen(type)>0)&&
	    (strstr(drivetype,"CD-ROM")!=NULL))
	  {
	     int is_recorder=0;

	     if (strchr(&vendor[1],'"')!=NULL)
	       *strchr(&vendor[1],'"')=0;
	     if (strchr(&model[1],'"')!=NULL)
	       *strchr(&model[1],'"')=0;
	     if ((strstr(type,"CD-R.")!=NULL)||
		 (strstr(type,"CD-RW.")!=NULL)||
		 (strstr(type,"DVD-R.")!=NULL)||
		 (strstr(type,"DVD-RW.")!=NULL)||
		 (strstr(type,"CD-R50S")!=NULL))
	       is_recorder=1;
	     sprintf(device,"/dev/scd%i",devicenum);
		 /* get mountpoint from lookup file (usually /etc/fstab) */
	     cddrives_getmountpoint(device,mountpoint);
	     if (cddrives_getrombyscsiid(scsiid)!=NULL)
	       {
		  i=cddrives_getrombyscsiid(scsiid);
		  strcpy(i->model,&model[1]);
		  strcpy(i->manufacturer,&vendor[1]);
		  i->is_recorder=is_recorder;
		  updatehandlers_call(cddrives_drives.updatehandlers);
	       }
	     else
	       {
		  cddrives_register_drive(cddrives_cdinfo_create(&model[1],
								 &vendor[1],
								 device,
								 scsiid,
								 mountpoint,
								 is_recorder,
								     /* enable dae extraction via scsi bus only if we got a scsi id for this drive */
								 ((scsiid!=NULL)?1:0)
								 ));
	       };
	     devicenum++;
	  };
     };

};

/* this updates a clist line with the informations in *drive  */
void cddrives_updateclist(cddrives_cdinfo*drive,int row)
{
#ifdef DEBUG
   printf ("cddrives_updateclist: updating drive information display\n");
#endif
   if (drive->is_recorder)
     gtk_clist_set_pixmap(GTK_CLIST(cddrives_drives.drivelist),
			  row,0,
			  cddrives_recorder_pixmap,
			  cddrives_recorder_mask);
   else
     gtk_clist_set_text(GTK_CLIST(cddrives_drives.drivelist),
			row,0,"");
   gtk_clist_set_text(GTK_CLIST(cddrives_drives.drivelist),
		      row,1,
		      drive->model);
   gtk_clist_set_text(GTK_CLIST(cddrives_drives.drivelist),
		      row,2,
		      drive->manufacturer);
   gtk_clist_set_text(GTK_CLIST(cddrives_drives.drivelist),
		      row,3,
		      drive->device);
   gtk_clist_set_text(GTK_CLIST(cddrives_drives.drivelist),
		      row,4,
		      drive->scsiid);
   gtk_clist_set_text(GTK_CLIST(cddrives_drives.drivelist),
		      row,5,
		      drive->mountpoint);
};

void cddrives_displayform(cddrives_cdinfo *i)
{
   forms_create(_("Please enter drive data"),
		11,
		&cddrives_drives.updatehandlers,
		FORMS_ENTRY,
		_("Model"),
		i->model,
		FORMS_ENTRY,
		_("Manufacturer"),
		i->manufacturer,
		FORMS_NEWLINE,
		FORMS_ENTRY,
		_("Device File"),
		i->device,
		FORMS_ENTRY,
		_("Scsi ID"),
		i->scsiid,
		FORMS_NEWLINE,
		FORMS_ENTRY,
		_("Mountpoint"),
		i->mountpoint,
		FORMS_NEWLINE,
		FORMS_CHECKBOX,
		_("This Drive is a CD writer"),
		&i->is_recorder,
		FORMS_NEWLINE,
		FORMS_CHECKBOX,
		_("Use SCSI Interface for DAE"),
		&i->dae_usesg
		);
};

void cddrives_addbutton(GtkWidget *w,
			gpointer data)
{
   cddrives_cdinfo *i;

   i=cddrives_cdinfo_create(_("(Model)"),
			    _("(Manufacturer)"),
			    _("/dev/scd0"),
			    _("0,0"),
			    _("/mnt/cdrom"),
			    0,
			    0);
   cddrives_register_drive(i);

   cddrives_displayform(i);
};

void cddrives_removebutton(GtkWidget *w,
			   gpointer data)
{
   int sl;
   cddrives_cdinfo *i;
   if (GTK_CLIST(cddrives_drives.drivelist)->selection!=NULL)
     {
	sl=(int)(GTK_CLIST(cddrives_drives.drivelist)->selection)->data;
	i=gtk_clist_get_row_data(GTK_CLIST(cddrives_drives.drivelist),sl);

	LOCKDB;
	cddrives_drives.drivecount--;
	gtk_clist_remove(GTK_CLIST(cddrives_drives.drivelist),sl);
	cddrives_cdinfo_destroy(i);

	updatehandlers_call(cddrives_drives.updatehandlers);
	UNLOCKDB;
     };
};

void cddrives_editbutton(GtkWidget *w,
			 gpointer data)
{
   int sl;
   cddrives_cdinfo *i;
   if (GTK_CLIST(cddrives_drives.drivelist)->selection!=NULL)
     {
	sl=(int)(GTK_CLIST(cddrives_drives.drivelist)->selection)->data;
	i=gtk_clist_get_row_data(GTK_CLIST(cddrives_drives.drivelist),sl);

	cddrives_displayform(i);
     };
};

void cddrives_scanbusbutton(GtkWidget *w,
			    gpointer data)
{
   cddrives_scanbus();
};

/* updatehandler realizing changes in the drive list */
void cddrives_update_list(gpointer data)
{
   int x;

   for (x=0;x<cddrives_drives.drivecount;x++)
     cddrives_updateclist((cddrives_cdinfo*)gtk_clist_get_row_data(GTK_CLIST(cddrives_drives.drivelist),x),
			  x);
};

static int cddrives_drivelist_resize(GtkWidget *w,
				     GtkAllocation *a,
				     gpointer data)
{
   gtk_clist_set_column_width(GTK_CLIST(w),0,16);
   gtk_clist_set_column_width(GTK_CLIST(w),1,
			      (int)((float)(a->width-16)*0.25));
   gtk_clist_set_column_width(GTK_CLIST(w),2,
			      (int)((float)(a->width-16)*0.25));
   gtk_clist_set_column_width(GTK_CLIST(w),3,
			      (int)((float)(a->width-16)*0.12));
   gtk_clist_set_column_width(GTK_CLIST(w),4,
			      (int)((float)(a->width-16)*0.06));
   gtk_clist_set_column_width(GTK_CLIST(w),5,
			      (int)((float)(a->width-16)*0.25));
   return TRUE;
};

/* install a preferences page in setup to enter a list of valid cd drives */
void cddrives_init()
{
   GtkWidget *cddrives_prefs;
   GtkWidget *label;
   GtkWidget *dlscroll;
   GtkWidget *dledit;
   GtkWidget *editbutton;
   GtkStyle *style;
   varmanwidgets_widget *switch_autoscan;

   style=gtk_widget_get_style(window);
   cddrives_recorder_pixmap=gdk_pixmap_create_from_xpm_d(window->window,
							 &cddrives_recorder_mask,
							 &style->bg[GTK_STATE_NORMAL],
							 recorder_xpm);
   cddrives_prefs=gtk_vbox_new(0,0);
   gtk_widget_show(cddrives_prefs);
   label=gtk_label_new(_("CDROM and Recorder Setup"));
   gtk_widget_show(label);
   preferences_append_page(cddrives_prefs,label);

   dlscroll=gtk_scrolled_window_new(NULL,NULL);
   gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(dlscroll),
				  GTK_POLICY_AUTOMATIC,
				  GTK_POLICY_AUTOMATIC);
   cddrives_drives.drivelist=gtk_clist_new_with_titles(6,
						       helpings_translatestringlist(cddrives_dltitles));
   gtk_clist_set_selection_mode(GTK_CLIST(cddrives_drives.drivelist),
				GTK_SELECTION_SINGLE);
   gtk_signal_connect(GTK_OBJECT(cddrives_drives.drivelist),"size_allocate",
		      GTK_SIGNAL_FUNC(cddrives_drivelist_resize),NULL);
   gtk_container_add(GTK_CONTAINER(dlscroll),
		     cddrives_drives.drivelist);
   gtk_widget_show(cddrives_drives.drivelist);
   gtk_box_pack_start(GTK_BOX(cddrives_prefs),dlscroll,1,1,1);
   gtk_widget_show(dlscroll);

   /* edit functions for the drive list */
   dledit=gtk_hbox_new(0,0);
   editbutton=gtk_button_new_with_label(_("Add"));
   gtk_box_pack_start(GTK_BOX(dledit),editbutton,0,0,0);
   gtk_signal_connect(GTK_OBJECT(editbutton),"clicked",
		      GTK_SIGNAL_FUNC(cddrives_addbutton),NULL);
   gtk_widget_show(editbutton);
   editbutton=gtk_button_new_with_label(_("Remove"));
   gtk_box_pack_start(GTK_BOX(dledit),editbutton,0,0,0);
   gtk_signal_connect(GTK_OBJECT(editbutton),"clicked",
		      GTK_SIGNAL_FUNC(cddrives_removebutton),NULL);
   gtk_widget_show(editbutton);
   editbutton=gtk_button_new_with_label(_("Edit"));
   gtk_box_pack_start(GTK_BOX(dledit),editbutton,0,0,0);
   gtk_signal_connect(GTK_OBJECT(editbutton),"clicked",
		      GTK_SIGNAL_FUNC(cddrives_editbutton),NULL);
   gtk_widget_show(editbutton);
   editbutton=gtk_button_new_with_label(_("Scan Bus"));
   gtk_box_pack_start(GTK_BOX(dledit),editbutton,0,0,0);
   gtk_signal_connect(GTK_OBJECT(editbutton),"clicked",
		      GTK_SIGNAL_FUNC(cddrives_scanbusbutton),NULL);
   gtk_widget_show(editbutton);
   gtk_widget_show(dledit);
   gtk_box_pack_start(GTK_BOX(cddrives_prefs),dledit,0,1,0);

   switch_autoscan=varmanwidgets_checkbox_new(_("Perform bus scan at startup"),
					      "rec_autoscan",
					      global_defs,
					      APPLYMODE_BUTTON,160,
					      "true","false");
   gtk_box_pack_start(GTK_BOX(cddrives_prefs),switch_autoscan->visual,0,0,0);

   /* initialize the drive database structure */
   cddrives_drives.drivecount=0;
   cddrives_drives.contentchange=NULL;
   cddrives_drives.updatehandlers=NULL;
   /* even the drivelist itself is served by this updatehandler */
   updatehandlers_register(&cddrives_drives.updatehandlers,
			   cddrives_update_list,
			   NULL);

   /* try to load drive list from config file */
   cddrives_loaddrivelist(varman_replacevars_copy(dynamic_defs,
						  "$HOME/.gtoasterrc"),
			  "[drives]");
   /* this registers a config file storage handler */
   configfile_registersection("[drives]",
			      (configfile_store)cddrives_storedrivelist,
			      NULL);

#ifdef HAVE_PTHREADS
   /* init the drive database lock by initializing a mutex for it
    * use default type "fast", which doesn't perform any error checking.
    * I think I'll find any deadlocks soon enough ;-) */
   pthread_mutex_init(&cddrives_drives.locked,NULL);
#endif

   if ((varman_getvar(global_defs,"rec_autoscan"))&&
       (!strcasecmp(varman_getvar(global_defs,"rec_autoscan"),"true")))
     /* perform busscan at startup ? */
     cddrives_scanbus();

   cddrives_enablecdchangedetection();

}
;

void cddrives_cdinfo_update(cddrives_cdinfo *info)
{
   gtoaster_handle_t tochandle=cdromlow_gettochandle(info->device);

   int tracksoncd;
   int trackcount;

   /* media change detect during the content update could stale-lock
    * some cd drives */
   LOCKDB;

#ifdef DEBUG
   printf ("cddrives_cdinfo_update: reading TOC...");
   fflush(NULL);
#endif

   info->tracks=0;
   if (tochandle)
     {
	tracksoncd=cdromlow_tracks(tochandle);

	/* calculate essential data for cddb lookup */
	info->es.cddb_key=cdromlow_cddbnumber(tochandle);

	/* Return playing time, rounded up to the next second */
	info->es.playingtime=
	  ((150+cdromlow_trackoffset(tochandle,tracksoncd+1))/75);
	info->es.tracks=tracksoncd;

	/* offsets are being filled out in this loop */
	if (tracksoncd>0)
	  {
	     /* Set CD-Text informations for the whole disc */
	     const char *disctitle=cdromlow_getdisctitle(tochandle);
	     const char *discperformer=cdromlow_getdiscperformer(tochandle);
	     if (disctitle)
	       strcpy(info->disctitle,disctitle);
	     else
	       info->disctitle[0]=0;
	     if (discperformer)
	       strcpy(info->discperformer,discperformer);
	     else
	       info->discperformer[0]=0;

	     /* calculate cddb essentials */

	     /* add 150 sectors to the sector offset, for the lead-in */
	     for (trackcount=0;trackcount<tracksoncd;trackcount++)
	       info->es.offsets[trackcount]=cdromlow_trackoffset(tochandle,trackcount+1)+150;

	     for (trackcount=info->tracks;trackcount<tracksoncd;trackcount++)
	       {
		  if (cdromlow_gettracktype(tochandle,
					    trackcount+1)==TRACKTYPE_AUDIO)
		    {
		       char *title=cdromlow_gettitle(tochandle,trackcount+1);
		       char *performer=cdromlow_getperformer(tochandle,trackcount+1);
		       info->track[trackcount]=audiotrack_create(info->device,
								 trackcount+1,
								 title,
								 performer,
								 &info->es);
		       if (title)
			 free(title);
		       if (performer)
			 free(performer);
		    }
		  else
		    info->track[trackcount]=datatrack_create(info->device,trackcount+1);
		  /* if the track is located on the current cd writer
		   * force precaching */
		  if (info->is_recorder)
		    info->track[trackcount]->precacherequired=atadd;
		  info->tracks++;
	       }
	     ;
	  }
	;
	cdromlow_closetochandle(tochandle);
     };

#ifdef DEBUG
   printf ("done.\n");
#endif

   UNLOCKDB;
}
;

char *nothing[]=
{
   NULL,NULL,NULL,NULL,NULL
};

void cddrives_register_drive(cddrives_cdinfo*drive)
{
   LOCKDB;
   gtk_clist_append(GTK_CLIST(cddrives_drives.drivelist),nothing);
   cddrives_updateclist(drive,cddrives_drives.drivecount);
   gtk_clist_set_row_data(GTK_CLIST(cddrives_drives.drivelist),
			  cddrives_drives.drivecount,
			  (gpointer)drive);
   cddrives_drives.drivecount++;
   UNLOCKDB;
   updatehandlers_call(cddrives_drives.updatehandlers);
};

/* wrapper creating a new device */
cddrives_cdinfo *cddrives_cdinfo_create(char *model,
					char *manufacturer,
					char *device,
					char *scsiid,
					char *mountpoint,
					int  is_recorder,
					int  dae_usesg)
{
   cddrives_cdinfo *info;

   info=(cddrives_cdinfo*)malloc(sizeof(cddrives_cdinfo));
   strcpy(info->model,model);
   strcpy(info->manufacturer,manufacturer);
   strcpy(info->device,device);
   strcpy(info->scsiid,scsiid);
   strcpy(info->mountpoint,mountpoint);
   info->is_recorder=is_recorder;
   info->dae_usesg=dae_usesg;

   /* initialize with default values for runtime data */
   info->is_active=0;

   info->tracks=0;
   info->mediachange=0;

   info->disctitle[0]=0;
   info->discperformer[0]=0;

   return info;
}
;

void cddrives_cdinfo_destroycontent(cddrives_cdinfo *info)
{
   int x;

   for (x=0;x<info->tracks;x++)
     {
	tracks_unclaim(info->track[x]);
     }
   ;
}
;

void cddrives_cdinfo_destroy(cddrives_cdinfo *info)
{
   cddrives_cdinfo_destroycontent(info);
   free(info);
}
;

/* this function is running within a 5 sec. Timeout handler context */
int cddrives_checkcdchange(gpointer dp)
{
   int x;
   cddrives_cdinfo *d;

   /* update flags here if for some reason the media change thread
    * couldn't be spawned,e.g. if there's no support for phreads on your
    * system */
   if ((!cddrives_updatethreadrunning)&&(cddrives_update>0))
     cddrives_updatemediachangeflags();

   if (cddrives_update>0)
     {

#ifdef DEBUG
	printf ("cddrives_checkcdchange: checking for a cd change\n");
#endif
	for (x=0;x<cddrives_drives.drivecount;x++)
	  {
#ifdef DEBUG
	     printf("cddrives_checkcdchange: checking drive #%i\n",x);
#endif
	     d=gtk_clist_get_row_data(GTK_CLIST(cddrives_drives.drivelist),x);
	     if ((d!=NULL)&&(d->mediachange))
	       {
		  cddrives_cdinfo_destroycontent(d);
		  cddrives_cdinfo_update(d);
		  /* mark this particular media change as processed */
		  d->mediachange=0;
		  /* process eventual update handlers */
		  updatehandlers_call(cddrives_drives.contentchange);
	       }
	     ;
	  }
	;
     }
   ;
   if (cddrives_updatethreadexit)
     return 0;
   else
     return 1; /* keep the timeout running unless termination has been
		* requested by the main app */
}
;

/* return rom containing cd with cddb code */
cddrives_cdinfo *cddrives_getrombycddb(int cddb)
{
   int x;
   cddrives_cdinfo *f=NULL;
   cddrives_cdinfo *d;

   for (x=0;(x<cddrives_drives.drivecount)&&(f==NULL);x++)
     {
	d=gtk_clist_get_row_data(GTK_CLIST(cddrives_drives.drivelist),x);
	if (d->es.cddb_key==cddb)
	  f=d;
     }
   ;
   return f;
}
;

/* return the first cd recorder in the list */
cddrives_cdinfo *cddrives_getrecorder()
{
   int num_recorders=0;
   int last_recorder=0;
   int x;
   cddrives_cdinfo *f=NULL;
   cddrives_cdinfo *d;

   /* FIXME: return *active* recorder here */
   for (x=0;(x<cddrives_drives.drivecount)&&(f==NULL);x++)
     {
	d=gtk_clist_get_row_data(GTK_CLIST(cddrives_drives.drivelist),x);
	if (d->is_recorder)
	  {
	     ++num_recorders;
	     last_recorder=x;
	     if (d->is_active)
	       f=d;
	  };
     }
   ;
   /* It doesn't make sense to even try to determine an active recorder
    * if there's no recorder at all */
   if ((!f)&&(num_recorders>0))
     {
	if (num_recorders==1)
	  {
	     /* Only one recorder is present on the system
	      * but it is not selected as the active recorder.
	      * Change that and return it... */
	     d=gtk_clist_get_row_data(GTK_CLIST(cddrives_drives.drivelist),last_recorder);
	     d->is_active=1;
	     f=d;
	  }
	else
	  /* Popup the recorder selection dialog */
	  cddrives_selectrecorder();	  
     };
   return f;
}
;

/* return rom with given device name */
cddrives_cdinfo *cddrives_getrombydevicename(const char *device)
{
   int x;
   cddrives_cdinfo *f=NULL;
   cddrives_cdinfo *d;
#ifdef DEBUG
   printf ("cddrives_getrombydevicename: seeking %s in (",
	   device);
#endif

   for (x=0;(x<cddrives_drives.drivecount)&&(f==NULL);x++)
     {
	d=gtk_clist_get_row_data(GTK_CLIST(cddrives_drives.drivelist),x);
#ifdef DEBUG
	printf ("%s ",d->device);
#endif
	if (!strcmp(d->device,device))
	  f=d;
     }
   ;
#ifdef DEBUG
   printf(").\n");
#endif
   return f;
}
;

cddrives_cdinfo *cddrives_getdrivenum(int num)
{
   return (cddrives_cdinfo*)gtk_clist_get_row_data(GTK_CLIST(cddrives_drives.drivelist),num);
};   

/* return rom by scsi id, used by the autoscan function */
cddrives_cdinfo *cddrives_getrombyscsiid(const char *scsiid)
{
   int x;
   cddrives_cdinfo *f=NULL;
   cddrives_cdinfo *d;
#ifdef DEBUG
   printf ("cddrives_getrombydevicename: seeking %s in (",
	   scsiid);
#endif

   for (x=0;(x<cddrives_drives.drivecount)&&(f==NULL);x++)
     {
	d=gtk_clist_get_row_data(GTK_CLIST(cddrives_drives.drivelist),x);
#ifdef DEBUG
	printf ("%s ",d->scsiid);
#endif
	if (!strcmp(d->scsiid,scsiid))
	  f=d;
     }
   ;
#ifdef DEBUG
   printf(").\n");
#endif
   return f;
}
;

void cddrives_updatedisable()
{
   LOCKDB;
   cddrives_update--;
#ifdef DEBUG
   printf("cddrives_updatedisable: new count is #%i\n",cddrives_update);
#endif
   UNLOCKDB;
};

void cddrives_updateenable()
{
   cddrives_update++;
#ifdef DEBUG
   printf("cddrives_updateenable: new count is #%i\n",cddrives_update);
#endif
};

int cddrives_getnumdrives()
{
   return cddrives_drives.drivecount;
};

void cddrives_recorderselect_done(gint reply, gpointer data)
{
   GtkCList *recorderlist = GTK_CLIST(data);
   if (recorderlist&&(reply==DIALOG_YES))
     {
	/* First, reset all recorder to not active */
	int i = 0;
	GList *current = NULL;
	for (i=0;i<recorderlist->rows;++i)
	  {
	     cddrives_cdinfo *info=gtk_clist_get_row_data(recorderlist,i);
	     if (info)
	       info->is_active=0;
	  };
	/* Now activate all selected recorders */
	for (current=recorderlist->selection;current!=NULL;current=current->next)
	  {
	     cddrives_cdinfo *info=gtk_clist_get_row_data(recorderlist,(int)current->data);
	     if (info)
	       info->is_active=1;
	  };
     };
};

char *cddrives_recseltitles[]=
{
   N_("Model"),
   N_("Manufacturer")
};

void cddrives_selectrecorder()
{
   GtkWidget *recorderlist = gtk_clist_new_with_titles(2,helpings_translatestringlist(cddrives_recseltitles));
   int i=0;
   gtk_clist_set_column_auto_resize(GTK_CLIST(recorderlist),0,1);
   for (i=0;i<cddrives_drives.drivecount;++i)
     {	
	cddrives_cdinfo *currentdrive = cddrives_getdrivenum(i);
	if (currentdrive&&currentdrive->is_recorder)
	  {	     
	     gchar *newrow[2];	
	     int position= -1;
	     newrow[0]=currentdrive->model;
	     newrow[1]=currentdrive->manufacturer;
	     position = gtk_clist_append(GTK_CLIST(recorderlist),newrow);
	     gtk_clist_set_row_data(GTK_CLIST(recorderlist),position,(gpointer)currentdrive);
	     // Select the first entry or the active recorder...
	     if (currentdrive->is_active||(position==0))
	       gtk_clist_select_row(GTK_CLIST(recorderlist),position,0);
	  };
     };
   /* apply a certain minimum size to the list */
   gtk_widget_set_usize(recorderlist,320,100);
   dialog_showuserwidget(_("Choose recorder"),recorderlist,
			 DIALOG_USERWIDGET_OKCANCEL,
			 cddrives_recorderselect_done,
			 GTK_CLIST(recorderlist));
};

