/* some functions to do lowlevel access to cdrom drives */

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <signal.h>
#include <unistd.h>
#if !defined( __FreeBSD__) && !defined(__svr4__ )
# include <linux/cdrom.h>
#else
# include <sys/cdio.h>
# include <sys/param.h>
# include <sys/mount.h>
#endif

#include "main.h"
#include "piping.h"
#include "cdromlow.h"
#include "preferences.h"
#include "varman.h"
#include "helpings.h"
#include "varman.h"
#include "calc.h"
#include "stdfiletrack.h"
#include "cddrives.h"

#ifdef HAVE_LINUX_UCDROM_H /* Test after we've included config.h */
# include <linux/ucdrom.h>
#endif

/* uncomment for debugging */
// #define DEBUG

/* we provide some macro mappings here. FreeBSD structs are a bit
 * different to those used by Linux so we will use macros to compensate that */
#ifdef __FreeBSD__

/* Needed by FreeBSD */
# define CDROM_DATA_TRACK 4
# define CDROM_LEADOUT 0xaa
# define CD_FRAMESIZE 2048
# define CD_FRAMESIZE_RAW 2352
# define CDROM_LBA CD_LBA_FORMAT
# define CDROMREADTOCHDR CDIOREADTOCHEADER
# define CDROMREADTOCENTRY CDIOREADTOCENTRY

# define cdromlow_definecdromheader	struct ioc_toc_header
# define cdromlow_definecdromhentry	struct ioc_read_toc_single_entry

/* referencing the TOC header */
# define cdromlow_tochdr_firstt starting_track
# define cdromlow_tochdr_lastt  ending_track

/* referencing TOC entries */
# define cdromlow_tocent_track  track
# define cdromlow_tocent_format address_format
# define cdromlow_tocent_addr   entry.addr
#elif __svr4__
# define CD_FRAMESIZE 2048
# define CD_FRAMESIZE_RAW 2352

# define cdromlow_definecdromheader	struct cdrom_tochdr
# define cdromlow_definecdromhentry	struct cdrom_tocentry

/* referencing the TOC header */
# define cdromlow_tochdr_firstt cdth_trk0
# define cdromlow_tochdr_lastt  cdth_trk1

/* referencing TOC entries */
# define cdromlow_tocent_track  cdte_track
# define cdromlow_tocent_format cdte_format
# define cdromlow_tocent_addr   cdte_addr
#else

/* those are the mapping macros for Linux systems */

# define cdromlow_definecdromheader	struct cdrom_tochdr
# define cdromlow_definecdromhentry	struct cdrom_tocentry

/* referencing the TOC header */
# define cdromlow_tochdr_firstt cdth_trk0
# define cdromlow_tochdr_lastt  cdth_trk1

/* referencing TOC entries */
# define cdromlow_tocent_track  cdte_track
# define cdromlow_tocent_format cdte_format
# define cdromlow_tocent_addr   cdte_addr

#endif

#define CDROMLOW_MAXCDTEXTOUT 16384

typedef struct
{
   char *cdda2wavoutput;
   char *devicename;
   int  filedescriptor;
}
cdromlow_tochandle;

int cdromlow_gettochandle(const char *device)
{
   cdromlow_tochandle *handle=(cdromlow_tochandle*)malloc(sizeof(cdromlow_tochandle));
   /* no cd text output is generated until needed */
   handle->cdda2wavoutput=NULL;
   handle->devicename=strdup(device);
   handle->filedescriptor=open(device,O_RDONLY|O_NONBLOCK);
#ifdef DEBUG
   printf("cdromlow_gettochandle result is %i\n",(int)handle);
#endif
   return (int)handle;
};

void cdromlow_closetochandle(int handle)
{
   cdromlow_tochandle *info=(cdromlow_tochandle*)handle;
   /* free cd text output if it was generated */
   if (info->cdda2wavoutput)
     free (info->cdda2wavoutput);
   if (info->filedescriptor!=-1)
     close(info->filedescriptor);
   free(info->devicename);
   free(info);
};

/* reimplemented using direct cdrom access:
 * does the current cd have one or more data tracks ? *
 * returns -1 or data track of cd */
int cdromlow_hasdatatrack(int handle)
{
   cdromlow_tochandle *info=(cdromlow_tochandle*)handle;
   cdromlow_definecdromheader header;
   cdromlow_definecdromhentry entry;

   int x;
   int hasdatatrack=-1;

   if (info->filedescriptor!=-1)
     {
	if (ioctl(info->filedescriptor,CDROMREADTOCHDR,&header)!=-1)
	  for (x=header.cdromlow_tochdr_firstt;x<=header.cdromlow_tochdr_lastt;x++)
	    {
	       entry.cdromlow_tocent_track=x;
	       entry.cdromlow_tocent_format=CDROM_LBA;
	       ioctl(info->filedescriptor,CDROMREADTOCENTRY,&entry);
#ifndef __FreeBSD__
	       if (entry.cdte_ctrl&CDROM_DATA_TRACK)
#else
		 if (entry.entry.control & CDROM_DATA_TRACK)
#endif
		   hasdatatrack=x;
	    }
	else
	  {
	     perror("cdromlow_hasdatatrack: error asking cdrom drive");
	     hasdatatrack=-1;
	  };
     }
   ;
   return hasdatatrack;
}
;

/* return number of tracks on cd,reimplemented using ioctl()s */
int cdromlow_tracks(int handle)
{
   cdromlow_tochandle *info=(cdromlow_tochandle*)handle;
   cdromlow_definecdromheader header;
   int tracknum;

   tracknum=0;
   if (info->filedescriptor!=-1)
     {
	if (ioctl(info->filedescriptor,CDROMREADTOCHDR,&header)!=-1)
	  tracknum = 1 + header.cdromlow_tochdr_lastt - header.cdromlow_tochdr_firstt;
	else
	  tracknum=0;
     }
   ;
   return (tracknum);
}
;

/* returns the offset of a track's starting position in LBA format */
int cdromlow_trackoffset(int handle,int num)
{
   cdromlow_tochandle *info=(cdromlow_tochandle*)handle;
   cdromlow_definecdromhentry entry;
   int offset=-1;

   if (info->filedescriptor!=-1)
     {
	/* Return offset of leadout if track > last track is requested */
	if (num>cdromlow_tracks(handle))
	  entry.cdromlow_tocent_track=CDROM_LEADOUT;
	else
	  entry.cdromlow_tocent_track=num;
	entry.cdromlow_tocent_format=CDROM_LBA;
	if (ioctl(info->filedescriptor,CDROMREADTOCENTRY,&entry)!=-1)
	  {

	     offset=entry.cdromlow_tocent_addr.lba;
#ifdef DEBUG
	     printf ("cdromlow_trackoffset: offset #%i is %i\n",
		     num,offset);
#endif
	  };
     }
   ;
   return offset;
}
;

/* return tracksize of track num in sectors (LBA) */
int cdromlow_tracksize_sectors(int handle,int num)
{
   cdromlow_tochandle *info=(cdromlow_tochandle*)handle;
   cdromlow_definecdromhentry track;
   cdromlow_definecdromhentry tracknext;
   int tracksize;
   int ok=1;

   tracksize=0;
   if (info->filedescriptor!=-1)
     {
	track.cdromlow_tocent_track=num;
	track.cdromlow_tocent_format=CDROM_LBA;
	if (ioctl(info->filedescriptor,CDROMREADTOCENTRY,&track)==-1)
	  ok=0;

	if (num==cdromlow_tracks(handle))
	  tracknext.cdromlow_tocent_track=CDROM_LEADOUT;
	else
	  tracknext.cdromlow_tocent_track=num+1;
	tracknext.cdromlow_tocent_format=CDROM_LBA;
	if (ioctl(info->filedescriptor,CDROMREADTOCENTRY,&tracknext)==-1)
	  ok=0;

	if (ok)
	  tracksize=(tracknext.cdromlow_tocent_addr.lba-track.cdromlow_tocent_addr.lba);
	else
	  tracksize=0;
     }
   ;
   return tracksize;
}
;

/* return tracksize for audiotrack n in bytes,reimplemented using ioctl()s */
int cdromlow_tracksize(int handle,int num)
{
   return cdromlow_tracksize_sectors(handle,num)*CD_FRAMESIZE_RAW;
}
;

/* return tracksize of datatrack in bytes,reimplemented using ioctl()s */
int cdromlow_datatracksize(int handle,int tracknum)
{
   if (cdromlow_hasdatatrack(handle)!=-1)
     return cdromlow_tracksize_sectors(handle,
				       tracknum
				       )*CD_FRAMESIZE;
   else
     return 0;
}
;

/* return cddb number of cdrom */
int cdromlow_cddbnumber(int handle)
{
   cdromlow_tochandle *info=(cdromlow_tochandle*)handle;

   cdromlow_definecdromheader header;
   cdromlow_definecdromhentry entry;
   int x,secs;
   int cddbnum=0;

   cddbnum=0;
   if (info->filedescriptor!=-1)
     {
	if (ioctl(info->filedescriptor,CDROMREADTOCHDR,&header)!=-1)
	  {
	     entry.cdromlow_tocent_track=CDROM_LEADOUT;
	     entry.cdromlow_tocent_format=CDROM_LBA;
	     ioctl(info->filedescriptor,CDROMREADTOCENTRY,&entry);
	     secs=(entry.cdromlow_tocent_addr.lba/75);

	     for (x=header.cdromlow_tochdr_firstt;x<=header.cdromlow_tochdr_lastt;x++)
	       {
		  entry.cdromlow_tocent_track=x;
		  entry.cdromlow_tocent_format=CDROM_LBA;
		  ioctl(info->filedescriptor,CDROMREADTOCENTRY,&entry);
		  cddbnum+=helpings_dsum((entry.cdromlow_tocent_addr.lba/75)+2);
		  if (x==header.cdromlow_tochdr_firstt) secs-=(entry.cdromlow_tocent_addr.lba/75);
	       }
	     ;

	     cddbnum<<=24;
	     cddbnum|=(secs<<8);
	     cddbnum|=(1+header.cdromlow_tochdr_lastt-header.cdromlow_tochdr_firstt);
	  }
	;
     }
   ;
   return cddbnum;
}
;

/* return the tracktype of specified track */
cdromlow_tracktype cdromlow_gettracktype(int handle,int num)
{
   cdromlow_tochandle *info=(cdromlow_tochandle*)handle;
   cdromlow_definecdromhentry track;
   int ok=1;
   cdromlow_tracktype result=TRACKTYPE_UNKNOWN;

   if (info->filedescriptor!=-1)
     {
	track.cdromlow_tocent_track=num;
	track.cdromlow_tocent_format=CDROM_LBA;
	if (ioctl(info->filedescriptor,CDROMREADTOCENTRY,&track)==-1)
	  ok=0;
#ifndef __FreeBSD__
	if (track.cdte_ctrl&CDROM_DATA_TRACK)
#else
	  if (track.entry.control & CDROM_DATA_TRACK)
#endif
	    result=TRACKTYPE_DATA;
	else
	  result=TRACKTYPE_AUDIO;
     }
   ;
   return result;
}
;

void cdromlow_createcdtextoutput(int handle)
{
   cdromlow_tochandle *info=(cdromlow_tochandle*)handle;
   if (!info->cdda2wavoutput)
     {
	char *call=varman_getvar_copy(global_defs,
				      "audiotrack_readcdtext");
	cddrives_cdinfo *drive=cddrives_getrombydevicename(info->devicename);

	OUTBUF_LOCK;
	info->cdda2wavoutput=(char*)malloc(CDROMLOW_MAXCDTEXTOUT);

	varman_replacestring(call,"$scsiid",drive->scsiid);
	piping_create_getoutput(call,outbuf,CDROMLOW_MAXCDTEXTOUT,PIPING_WATCHALL);
	free(call);
	memcpy(info->cdda2wavoutput,outbuf,CDROMLOW_MAXCDTEXTOUT);
#ifdef DEBUG
	printf("cdromlow_createcdtextoutput: got output:\n%s\n",info->cdda2wavoutput);
#endif
	OUTBUF_UNLOCK;
	
     };
};

char *cdromlow_getcdtext(int handle, int num,const char *parse)
{
   cdromlow_tochandle *info=(cdromlow_tochandle*)handle;
   char buffer[256];
   char *func=(char*)malloc(((parse)?strlen(parse):0)+11);

   cdromlow_createcdtextoutput(handle);
     
   strcpy(buffer,"");
   strcpy(func,((parse)?parse:""));
   OUTBUF_LOCK;
   memcpy(outbuf,info->cdda2wavoutput,CDROMLOW_MAXCDTEXTOUT);

   varman_replacestringbyfloat(func,"$tracknum",num);
   calc_calculate(func,buffer);
   OUTBUF_UNLOCK;
#ifdef DEBUG
   printf("cdromlow_getcdtext: result of CD-Text query is '%s', query was %s\n",buffer,parse);
#endif
   free(func);
   if (buffer[0]=='"')
     strcpy(buffer,&buffer[1]);
   if (strchr(buffer,'"'))
     *strchr(buffer,'"')=0;
   
   return (strlen(buffer)?strdup(buffer):NULL);
};

char *cdromlow_gettitle(int handle,int num)
{
   return cdromlow_getcdtext(handle,num,varman_getvar(global_defs,"audiotrack_parsecdtext_title"));
};

char *cdromlow_getperformer(int handle,int num)
{
   return cdromlow_getcdtext(handle,num,varman_getvar(global_defs,"audiotrack_parsecdtext_performer"));
};

char *cdromlow_getdisctitle(int handle)
{
   return cdromlow_getcdtext(handle,0,varman_getvar(global_defs,"audiotrack_parsecdtext_cdtitle"));
};

char *cdromlow_getdiscperformer(int handle)
{
   return cdromlow_getcdtext(handle,0,varman_getvar(global_defs,"audiotrack_parsecdtext_cdperformer"));
};

