/*
 * gtkoptions.c: Options dialog functions
 *
 * This file is part of GTick
 *
 * Copyright (c) 1999, Alex Roberts
 * Copyright (c) 2003, Roland Stigge <stigge@antcom.de>
 *
 * GTick is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * GTick is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GTick; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

/* GNU headers */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* own headers */
#include "globals.h"
#include "gtkoptions.h"
#include "options.h"
#include "option.h"
#include "util.h"

/*
 * handler for apply function in options dialog
 * returns 0 on success, -1 otherwise
 */
static int response_apply(GtkDialog *dialog, options_t* options) {
  GSList* group;
  GtkWidget* radio_button;
  const char* samplename;
  const char* sounddevice;
  const char* command;
  const char* value;
  
  /*
   * Sample filename setup
   */
  group = g_object_get_data(G_OBJECT(dialog), "radiobuttons");
  while (group &&
    !gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(group->data)))
  {
    group = group->next;
  }
  radio_button = GTK_WIDGET(group->data);
  if (!strcmp(g_object_get_data(G_OBJECT(radio_button), "choice"),
	      "<default>"))
  {
    samplename = "<default>";
  } else if (!strcmp(g_object_get_data(G_OBJECT(radio_button), "choice"),
		     "<sine>"))
  {
    samplename = "<sine>";
  } else { /* other */
#ifdef WITH_SNDFILE
    samplename = gtk_entry_get_text(
	GTK_ENTRY(g_object_get_data(G_OBJECT(dialog), "sample_name_entry")));
#else
    fprintf(stderr, "Warning: Unhandled samplename case: \"%s\".\n",
	    (char*) g_object_get_data(G_OBJECT(radio_button), "choice"));
    samplename = "";
#endif /* WITH_SNDFILE */
  }

  option_set(options->option_list, "SampleFilename", samplename);

  /*
   * Sound device setup
   */
  sounddevice = gtk_entry_get_text(
      GTK_ENTRY(g_object_get_data(G_OBJECT(dialog), "sound_device_entry")));
  option_set(options->option_list, "SoundDevice", sounddevice);

  /*
   * External commands setup
   */
  command = gtk_entry_get_text(
      GTK_ENTRY(g_object_get_data(G_OBJECT(dialog), "onstart_entry")));
  option_set(options->option_list, "CommandOnStart", command);
  command = gtk_entry_get_text(
      GTK_ENTRY(g_object_get_data(G_OBJECT(dialog), "onstop_entry")));
  option_set(options->option_list, "CommandOnStop", command);

  /*
   * BPM bounds
   */
  value = dtostr_c(gtk_spin_button_get_value(
      GTK_SPIN_BUTTON(g_object_get_data(G_OBJECT(dialog), "min_bpm_entry"))),1);
  option_set(options->option_list, "MinBPM", value);
  value = dtostr_c(gtk_spin_button_get_value(
      GTK_SPIN_BUTTON(g_object_get_data(G_OBJECT(dialog), "max_bpm_entry"))),1);
  option_set(options->option_list, "MaxBPM", value);

  return 0;
}

/*
 * callback invoked by response signal in dialog
 */
static void response_cb(GtkDialog *dialog, gint response, options_t* options) {
  switch (response) {
    case GTK_RESPONSE_APPLY:
      response_apply(dialog, options);
      break;
    case GTK_RESPONSE_OK:
      response_apply(dialog, options);
      gtk_widget_destroy(GTK_WIDGET(dialog));
      break;
    case GTK_RESPONSE_CANCEL:
      gtk_widget_destroy(GTK_WIDGET(dialog));
      break;
    case GTK_RESPONSE_DELETE_EVENT: /* ignore, window will just be closed */
      break;
    default:
      fprintf(stderr, "Warning: Unhandled response to dialog.\n");
  }
}

/*
 * callback invoked by ok button in file selection dialog
 */
#ifdef WITH_SNDFILE
static void choose_cb_store(GtkWidget *button) {
  const gchar* filename;
  GtkWidget* file_selection;
  GtkWidget* entry;

  file_selection = g_object_get_data(G_OBJECT(button), "file_selection");
  filename =
    gtk_file_selection_get_filename(GTK_FILE_SELECTION(file_selection));
  entry = g_object_get_data(G_OBJECT(button), "sample_name_entry");
  gtk_entry_set_text(GTK_ENTRY(entry), filename);
}
#endif /* WITH_SNDFILE */

/*
 * callback invoked by filename choose button
 */
#ifdef WITH_SNDFILE
static void choose_cb(GtkWidget *button) {
  GtkWidget* file_selection =
    gtk_file_selection_new(_("Please choose a sound file."));
  gtk_window_set_modal(GTK_WINDOW(file_selection), TRUE);
  
  g_object_set_data(G_OBJECT(GTK_FILE_SELECTION(file_selection)->ok_button),
                    "sample_name_entry",
		    g_object_get_data(G_OBJECT(button), "sample_name_entry"));
  g_object_set_data(G_OBJECT(GTK_FILE_SELECTION(file_selection)->ok_button),
                    "file_selection",
		    file_selection);
  g_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(file_selection)->ok_button),
                   "clicked",
		   G_CALLBACK(choose_cb_store),
		   NULL);
    
  /* destroy file selection dialog if ok or cancel button has been clicked */
  g_signal_connect_swapped(
      GTK_OBJECT(GTK_FILE_SELECTION(file_selection)->ok_button),
      "clicked",
      G_CALLBACK(gtk_widget_destroy), 
      (gpointer) file_selection); 
  g_signal_connect_swapped(
      GTK_OBJECT(GTK_FILE_SELECTION(file_selection)->cancel_button),
      "clicked",
      G_CALLBACK(gtk_widget_destroy),
      (gpointer) file_selection); 
   
  gtk_widget_show(file_selection);
}
#endif /* WITH_SNDFILE */

/*
 * callback invoked on toggled radiobutton for filename
 */
#ifdef WITH_SNDFILE
static void radio_toggled_cb(GtkWidget *button, GtkWidget* activewidget) {
  gtk_widget_set_sensitive(activewidget,
      gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button)));
}
#endif /* WITH_SNDFILE */

/*
 * return options dialog
 */
GtkWidget* gtk_options_dialog_new(GtkWidget* parent, options_t* options) {
  GtkWidget* window = gtk_dialog_new_with_buttons(_("Properties"),
      GTK_WINDOW(parent),
      GTK_DIALOG_DESTROY_WITH_PARENT,
      GTK_STOCK_OK, GTK_RESPONSE_OK,
      GTK_STOCK_APPLY, GTK_RESPONSE_APPLY,
      GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
      NULL);
  GtkWidget* vbox;
  GtkWidget* frame;
  GtkWidget* soundvbox;
  GtkWidget* radiobutton;
  GtkWidget* hbox;
  GtkWidget* entry;
  GtkWidget* spinbutton;
  GtkWidget* table;
#ifdef WITH_SNDFILE
  GtkWidget* button;
  GtkWidget* activehbox;
#endif /* WITH_SNDFILE */
  GtkWidget* label;
  const char* samplename;

  if (!(samplename = option_get(options->option_list, "SampleFilename"))) {
    fprintf(stderr, "gtk_options_dialog_new: "
	    "Couldn't get SampleFilename from options.\n");
    exit(1);
  }

  gtk_box_set_spacing(GTK_BOX(GTK_DIALOG(window)->vbox), 8);
  
  vbox = gtk_vbox_new(FALSE, 8);
  gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(window)->vbox), vbox);
  gtk_container_set_border_width(GTK_CONTAINER(vbox), 8);
  gtk_widget_show(vbox);
  
  /*
   * Sound effect setting
   */
  frame = gtk_frame_new(_("Sound"));
  gtk_box_pack_start(GTK_BOX(vbox), frame, TRUE, TRUE, 0);
  gtk_widget_show(frame);

  soundvbox = gtk_vbox_new(FALSE, 5);
  gtk_container_add(GTK_CONTAINER(frame), soundvbox);
  gtk_container_set_border_width(GTK_CONTAINER(soundvbox), 8);
  gtk_widget_show(soundvbox);

  radiobutton = gtk_radio_button_new_with_label(NULL, _("Default"));
  if (!strcmp(samplename, "<default>"))
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radiobutton), TRUE);
  gtk_box_pack_start_defaults(GTK_BOX(soundvbox), radiobutton);
  g_object_set_data(G_OBJECT(radiobutton), "choice", "<default>");
  gtk_widget_show(radiobutton);

  radiobutton =
    gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(radiobutton),
	                                        _("Sine"));
  if (!strcmp(samplename, "<sine>"))
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radiobutton), TRUE);
  gtk_box_pack_start_defaults(GTK_BOX(soundvbox), radiobutton);
  g_object_set_data(G_OBJECT(radiobutton), "choice", "<sine>");
  gtk_widget_show(radiobutton);

#ifdef WITH_SNDFILE
  hbox = gtk_hbox_new(FALSE, 5);
  gtk_box_pack_start(GTK_BOX(soundvbox), hbox, TRUE, FALSE, 0);
  gtk_widget_show(hbox);

  radiobutton =
    gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(radiobutton),
	                                        _("Other:"));
  g_object_set_data(G_OBJECT(radiobutton), "choice", "<other>");
  gtk_box_pack_start(GTK_BOX(hbox), radiobutton, FALSE, FALSE, 0);
  gtk_widget_show(radiobutton);

  activehbox = gtk_hbox_new(FALSE, 5);
  gtk_box_pack_start_defaults(GTK_BOX(hbox), activehbox);
  gtk_widget_show(activehbox);
  g_signal_connect(G_OBJECT(radiobutton),
                   "toggled",
		   G_CALLBACK(radio_toggled_cb),
		   activehbox);
 
  if (samplename[0] != '<') {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radiobutton), TRUE);
  } else {
    gtk_widget_set_sensitive(activehbox, FALSE);
  }

  entry = gtk_entry_new();
  if (samplename[0] != '<')
    gtk_entry_set_text(GTK_ENTRY(entry), samplename);
  gtk_box_pack_start_defaults(GTK_BOX(activehbox), entry);
  gtk_widget_show(entry);

  button = gtk_button_new_with_label(_("Choose"));
  gtk_box_pack_start(GTK_BOX(activehbox), button, FALSE, FALSE, 0);
  gtk_widget_show(button);
#endif /* WITH_SNDFILE */
  
  /* save entry and radiobutton group at window to retrieve value on apply */
#ifdef WITH_SNDFILE
  g_object_set_data(G_OBJECT(window), "sample_name_entry", entry);
#endif /* WITH_SNDFILE */
  g_object_set_data(G_OBJECT(window),
                    "radiobuttons",
                    gtk_radio_button_get_group(GTK_RADIO_BUTTON(radiobutton)));

  /* save entry at button to retrieve value in file selection dialog */
#ifdef WITH_SNDFILE
  g_object_set_data(G_OBJECT(button), "sample_name_entry", entry);

  g_signal_connect(G_OBJECT(button),
                   "clicked",
		   G_CALLBACK(choose_cb),
		   options);
#endif /* WITH_SNDFILE */

  /*
   * Sound device setting
   */
  frame = gtk_frame_new(_("Sound Device"));
  gtk_box_pack_start(GTK_BOX(vbox), frame, TRUE, TRUE, 0);
  gtk_widget_show(frame);

  hbox = gtk_hbox_new(FALSE, 5);
  gtk_container_add(GTK_CONTAINER(frame), hbox);
  gtk_container_set_border_width(GTK_CONTAINER(hbox), 8);
  gtk_widget_show(hbox);

  label = gtk_label_new(_("Device filename:"));
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
  gtk_widget_show(label);

  entry = gtk_entry_new();
  gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
  gtk_entry_set_text(GTK_ENTRY(entry),
                     option_get(options->option_list, "SoundDevice"));
  gtk_widget_show(entry);

  /* save entry at window to retrieve value on apply */
  g_object_set_data(G_OBJECT(window), "sound_device_entry", entry);

  /*
   * External commands
   */
  frame = gtk_frame_new(_("External Commands"));
  gtk_box_pack_start(GTK_BOX(vbox), frame, TRUE, TRUE, 0);
  gtk_widget_show(frame);

  table = gtk_table_new(2, 2, FALSE);
  gtk_container_add(GTK_CONTAINER(frame), table);
  gtk_table_set_row_spacings(GTK_TABLE(table), 8);
  gtk_table_set_col_spacings(GTK_TABLE(table), 5);
  gtk_widget_show(table);

  label = gtk_label_new(_("Execute on start:"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1,
      0, GTK_EXPAND, 0, 0);
  gtk_container_set_border_width(GTK_CONTAINER(table), 8);
  gtk_widget_show(label);

  entry = gtk_entry_new();
  gtk_entry_set_text(GTK_ENTRY(entry),
                     option_get(options->option_list, "CommandOnStart"));
  gtk_table_attach(GTK_TABLE(table), entry, 1, 2, 0, 1,
      GTK_EXPAND | GTK_FILL, GTK_EXPAND, 0, 0);
  gtk_widget_show(entry);
  g_object_set_data(G_OBJECT(window), "onstart_entry", entry);
  
  label = gtk_label_new(_("Execute on stop:"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2,
      0, GTK_EXPAND, 0, 0);
  gtk_widget_show(label);

  entry = gtk_entry_new();
  gtk_entry_set_text(GTK_ENTRY(entry),
                     option_get(options->option_list, "CommandOnStop"));
  gtk_table_attach(GTK_TABLE(table), entry, 1, 2, 1, 2,
      GTK_EXPAND | GTK_FILL, GTK_EXPAND, 0, 0);
  gtk_widget_show(entry);
  g_object_set_data(G_OBJECT(window), "onstop_entry", entry);

  /*
   * BPM bounds
   */
  frame = gtk_frame_new(_("Speed Range"));
  gtk_box_pack_start(GTK_BOX(vbox), frame, TRUE, TRUE, 0);
  gtk_widget_show(frame);

  table = gtk_table_new(2, 2, FALSE);
  gtk_container_add(GTK_CONTAINER(frame), table);
  gtk_table_set_row_spacings(GTK_TABLE(table), 8);
  gtk_table_set_col_spacings(GTK_TABLE(table), 5);
  gtk_widget_show(table);

  label = gtk_label_new(_("Minimum BPM:"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1,
      0, GTK_EXPAND, 0, 0);
  gtk_container_set_border_width(GTK_CONTAINER(table), 8);
  gtk_widget_show(label);

  spinbutton = gtk_spin_button_new_with_range(MIN_BPM, MAX_BPM, 10.0);
  gtk_spin_button_set_digits(GTK_SPIN_BUTTON(spinbutton), 1);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(spinbutton),
                          strtod_c(option_get(options->option_list, "MinBPM")));
  gtk_table_attach(GTK_TABLE(table), spinbutton, 1, 2, 0, 1,
      0, GTK_EXPAND, 0, 0);
  gtk_widget_show(spinbutton);
  g_object_set_data(G_OBJECT(window), "min_bpm_entry", spinbutton);
  
  label = gtk_label_new(_("Maximum BPM:"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2,
      0, GTK_EXPAND, 0, 0);
  gtk_widget_show(label);

  spinbutton = gtk_spin_button_new_with_range(MIN_BPM, MAX_BPM, 10.0);
  gtk_spin_button_set_digits(GTK_SPIN_BUTTON(spinbutton), 1);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(spinbutton),
                          strtod_c(option_get(options->option_list, "MaxBPM")));
  gtk_table_attach(GTK_TABLE(table), spinbutton, 1, 2, 1, 2,
      0, GTK_EXPAND, 0, 0);
  gtk_widget_show(spinbutton);
  g_object_set_data(G_OBJECT(window), "max_bpm_entry", spinbutton);
  
  g_signal_connect(G_OBJECT(window),
                   "response",
		   G_CALLBACK(response_cb),
		   options);
  return window;
}

